/*******************************************************************************
 *  Imixs Workflow Technology
 *  Copyright (C) 2001, 2008 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika
 *******************************************************************************/
package org.imixs.archive.signature.ca;

import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.KeyPair;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.SignatureException;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Optional;
import java.util.logging.Logger;

import javax.ejb.Singleton;
import javax.inject.Inject;

import org.bouncycastle.operator.OperatorCreationException;
import org.eclipse.microprofile.config.inject.ConfigProperty;
import org.imixs.archive.signature.KeystoreService;
import org.imixs.archive.signature.pdf.SigningService;
import org.imixs.workflow.ItemCollection;

/**
 * The CAService provides methods to managed X509Certificates stored in a
 * keystore. The certificates are used for digital Signature only. Certificates
 * are singed by an existing root or intermediate Certificate stored in a
 * keystore.
 * <p>
 * Certificates generated by this service have empty passwords and protected by
 * the keystore. For that reason a certificate managed by this service should
 * never be published and used for internal digital signatures only
 * <p>
 * The service is implemented as a singleton to avoid concurrent access from
 * different clients.
 * 
 * @see X509CertificateGenerator
 * @author rsoika
 * @version 1.0
 */
@Singleton
public class CAService {

    @Inject
    @ConfigProperty(name = SigningService.ENV_SIGNATURE_ROOTCERT_ALIAS)
    Optional<String> rootCertAlias;

    @Inject
    @ConfigProperty(name = SigningService.ENV_SIGNATURE_ROOTCERT_PASSWORD)
    Optional<String> rootCertPassword;

    @Inject
    KeystoreService keystoreService;

    private static Logger logger = Logger.getLogger(CAService.class.getName());

    /**
     * Test if a certificate chain for a given alias exists in the keyStore.
     * 
     * @param alias
     * @return true in case a certificate chain was found.
     */
    public boolean existsCertificate(String alias) {

        // test if a certificate exits....
        Certificate[] certificateChain = keystoreService.loadCertificate(alias);
        if (certificateChain == null || certificateChain.length == 0) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * This method generates a new X509 signed certificate based on an optional
     * given profile ItemCollection. Existing items with the prefix X509 will be
     * used to build the X509 naming.
     * <p>
     * The generated certificate will be signed with the root certificated referred
     * by the SIGNATURE_ROOTCERT_ALIAS
     * 
     * @param alias   - the alias the new certificate will be stored into the
     *                keystore
     * @param profile - optional itemCollection containing X509 attributes
     * @throws NoSuchProviderException
     * @throws NoSuchAlgorithmException
     * @throws KeyStoreException
     * @throws UnrecoverableKeyException
     * @throws SignatureException
     * @throws CertificateException
     * @throws OperatorCreationException
     * @throws InvalidKeyException
     * @throws IOException
     * @throws Exception
     */
    @SuppressWarnings("unchecked")
    public void createCertificate(String alias, ItemCollection profile)
            throws UnrecoverableKeyException, KeyStoreException, NoSuchAlgorithmException, NoSuchProviderException,
            InvalidKeyException, OperatorCreationException, CertificateException, SignatureException, IOException {
        Certificate[] certificateChain = null;

        logger.info("...generating new X509Certificate for alias '" + alias + "'...");
        X509CertificateGenerator x509CertificateGenerator = new X509CertificateGenerator();

        // load the root cert and private key...
        Certificate[] rootCertChain = keystoreService.loadCertificate(rootCertAlias.get());
        X509Certificate rootCert = (X509Certificate) rootCertChain[0];
        PrivateKey rootPrivKey = keystoreService.loadPrivateKey(rootCertAlias.get(), rootCertPassword.get());

        // create a new issuer keypair...
        KeyPair issueKeyPair = x509CertificateGenerator.generateKeyPair();
        // extract x509 attributes form optional profile
        if (profile != null) {
            String cn = profile.getItemValueString("x509.cn");
            if (cn.isEmpty()) {
                cn = alias;
            }
            // create the certificate...
            certificateChain = x509CertificateGenerator.generateSignedCertificate(rootCert, rootPrivKey, issueKeyPair,
                    cn,profile.getItemValueString("x509.o"), profile.getItemValue("x509.ou"), 
                    profile.getItemValueString("x509.city"), profile.getItemValueString("x509.state"),
                    profile.getItemValueString("x509.country"));
        } else {
            // create simple certificate without attributes
            certificateChain = x509CertificateGenerator.generateSignedCertificate(rootCert, rootPrivKey, issueKeyPair,
                    alias, null, null, null, null, null);

        }

        // store results to the keystore
        keystoreService.storeCertificate(certificateChain, issueKeyPair.getPrivate(), "", alias);

    }

}
