/*******************************************************************************
 *  Imixs Workflow Technology
 *  Copyright (C) 2001, 2008 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika
 *******************************************************************************/
package org.imixs.signature.pdf;

import java.awt.Color;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.List;
import java.util.Optional;
import java.util.logging.Logger;

import javax.ejb.LocalBean;
import javax.ejb.Stateless;
import javax.inject.Inject;

import org.apache.pdfbox.cos.COSName;
import org.apache.pdfbox.io.IOUtils;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.PDPageContentStream;
import org.apache.pdfbox.pdmodel.PDResources;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.apache.pdfbox.pdmodel.common.PDStream;
import org.apache.pdfbox.pdmodel.font.PDFont;
import org.apache.pdfbox.pdmodel.font.PDType1Font;
import org.apache.pdfbox.pdmodel.graphics.form.PDFormXObject;
import org.apache.pdfbox.pdmodel.graphics.image.PDImageXObject;
import org.apache.pdfbox.pdmodel.interactive.annotation.PDAnnotationWidget;
import org.apache.pdfbox.pdmodel.interactive.annotation.PDAppearanceDictionary;
import org.apache.pdfbox.pdmodel.interactive.annotation.PDAppearanceStream;
import org.apache.pdfbox.pdmodel.interactive.digitalsignature.ExternalSigningSupport;
import org.apache.pdfbox.pdmodel.interactive.digitalsignature.PDSignature;
import org.apache.pdfbox.pdmodel.interactive.digitalsignature.SignatureOptions;
import org.apache.pdfbox.pdmodel.interactive.form.PDAcroForm;
import org.apache.pdfbox.pdmodel.interactive.form.PDField;
import org.apache.pdfbox.pdmodel.interactive.form.PDSignatureField;
import org.apache.pdfbox.util.Matrix;
import org.bouncycastle.asn1.x500.RDN;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.x500.style.IETFUtils;
import org.eclipse.microprofile.config.inject.ConfigProperty;
import org.imixs.signature.pdf.cert.CertificateVerificationException;
import org.imixs.signature.pdf.cert.SigningException;
import org.imixs.signature.pdf.util.SigUtils;
import org.imixs.signature.service.KeystoreService;

/**
 * The SignatureService provides methods to sign a PDF document on a X509
 * certificate. The service adds a digital signature and also a visual
 * signature. The method 'signPDF' expects a Imixs FileData object containing
 * the content of the PDF file to be signed.
 * <p>
 * The service expects the existence of a valid X509 certificate stored in the
 * keystore.
 * <p>
 * The service supports the following environment variables:
 * <ul>
 * <li>SIGNATURE_KEYSTORE_PATH - path from which the keystore is loaded</li>
 * <li>SIGNATURE_KEYSTORE_PASSWORD - the password used to check the integrity of
 * the keystore, the password used to unlock the keystore</li>
 * <li>SIGNATURE_ROOTCERT_ALIAS - the root cert alias</li>
 * <li>SIGNATURE_ROOTCERT_PASSWORD - the root cert password (optional)</li>
 * </ul>
 * 
 * 
 * <p>
 * See also here: https://jvmfy.com/2018/11/17/how-to-digitally-sign-pdf-files/
 * https://github.com/apache/pdfbox/blob/trunk/examples/src/main/java/org/apache/pdfbox/examples/signature/CreateVisibleSignature2.java
 * https://ordina-jworks.github.io/security/2019/08/14/Using-Lets-Encrypt-Certificates-In-Java.html
 * 
 * @author rsoika
 * @version 1.0
 */
@Stateless
@LocalBean
public class SigningService {

    public final static String ENV_SIGNATURE_TSA_URL = "signature.tsa.url";
    public final static String ENV_SIGNATURE_ROOTCERT_ALIAS = "signature.rootcert.alias";
    public final static String ENV_SIGNATURE_ROOTCERT_PASSWORD = "signature.rootcert.password";

    @Inject
    KeystoreService keystoreService;

    @Inject
    @ConfigProperty(name = ENV_SIGNATURE_TSA_URL)
    Optional<String> tsaURL;

    private static Logger logger = Logger.getLogger(SigningService.class.getName());

    /**
     * Method Opens the keystore with the given password and creates a new signed
     * PDF file based on the given PDF File and a signature image.
     * <p>
     * generate pkcs12-keystore-file with
     * <p>
     * {@code
      keytool -storepass 123456 -storetype PKCS12 -keystore file.p12 -genkey -alias client -keyalg RSA
      }
     *
     * @param inputFileData   A byte array containing the source PDF document.
     * @param certAlias       Certificate alias name to be used for signing
     * @param certPassword    optional private key password
     * @param externalSigning optional boolean flag to trigger an external signing
     *                        process
     * @return A byte array containing the singed PDF document
     * 
     * @throws SigningException
     * @throws CertificateVerificationException
     * 
     * 
     * 
     */
    public byte[] signPDF(byte[] inputFileData, String certAlias, String certPassword, boolean externalSigning)
            throws CertificateVerificationException, SigningException {

        byte[] signedFileData = signPDF(inputFileData, certAlias, certPassword, externalSigning, null, 0, "Signature1",
                null, null);

        return signedFileData;

    }

    /**
     * Sign pdf file and create new file that ends with "_signed.pdf".
     *
     * @param inputFileData      A byte array containing the source PDF document.
     * @param certAlias          Certificate alias name to be used for signing
     * @param certPassword       optional private key password
     * @param externalSigning    optional boolean flag to trigger an external
     *                           signing process
     * @param humanRect          rectangle from a human viewpoint (coordinates start
     *                           at top left)
     * @param page               page number (beginning with 1) to place the visual
     *                           signature
     * @param tsaUrl             optional TSA url
     * @param signatureFieldName optional name of an existing (unsigned) signature
     *                           field
     * @param imageFile          optional image file
     * @param reason             workflow status
     * @return A byte array containing the singed PDF document
     * @throws CertificateVerificationException
     * @throws SigningException
     */
    public byte[] signPDF(byte[] inputFileData, String certAlias, String certPassword, boolean externalSigning,
            Rectangle2D humanRect, int page, String signatureFieldName, byte[] imageFile, String reason)
            throws CertificateVerificationException, SigningException {

        SignatureOptions signatureOptions = null;
        byte[] signedContent = null;

        if (inputFileData == null || inputFileData.length == 0) {
            throw new SigningException("empty file data");
        }

        // = Loader.loadPDF(inputFile)) {
        // ByteArrayOutputStream
        // try (FileOutputStream fos = new FileOutputStream(signedFile); PDDocument doc
        // = PDDocument.load(inputFileData.getContent())) {
        try (ByteArrayOutputStream bos = new ByteArrayOutputStream(); PDDocument doc = PDDocument.load(inputFileData)) {
            int accessPermissions = SigUtils.getMDPPermission(doc);
            if (accessPermissions == 1) {
                throw new SigningException(
                        "No changes to the document are permitted due to DocMDP transform parameters dictionary");
            }
            // Note that PDFBox has a bug that visual signing on certified files with
            // permission 2
            // doesn't work properly, see PDFBOX-3699. As long as this issue is open, you
            // may want to
            // be careful with such files.

            PDSignature pdSignature = null;
            PDAcroForm acroForm = doc.getDocumentCatalog().getAcroForm();
            PDRectangle rect = null;

            // If the PDF contains an existing empty signature, as created by the
            // CreateEmptySignatureForm example we can reuse it here
            if (acroForm != null) {
                try {
                    pdSignature = findExistingSignature(acroForm, signatureFieldName);
                    if (pdSignature != null) {
                        rect = acroForm.getField(signatureFieldName).getWidgets().get(0).getRectangle();
                    }
                } catch (IllegalStateException ise) {
                    // we can not use this signature field
                    logger.warning("signature " + signatureFieldName + " already exists: " + ise.getMessage());
                    signatureFieldName = signatureFieldName + ".1";
                }
            }

            if (pdSignature == null) {
                // create signature dictionary
                pdSignature = new PDSignature();
            }

            if (rect == null && humanRect != null) {
                rect = createSignatureRectangle(doc, humanRect);
            }

            // Optional: certify
            // can be done only if version is at least 1.5 and if not already set
            // doing this on a PDF/A-1b file fails validation by Adobe preflight
            // (PDFBOX-3821)
            // PDF/A-1b requires PDF version 1.4 max, so don't increase the version on such
            // files.
            if (doc.getVersion() >= 1.5f && accessPermissions == 0) {
                SigUtils.setMDPPermission(doc, pdSignature, 2);
            }

            if (acroForm != null && acroForm.getNeedAppearances()) {
                // PDFBOX-3738 NeedAppearances true results in visible signature becoming
                // invisible
                // with Adobe Reader
                if (acroForm.getFields().isEmpty()) {
                    // we can safely delete it if there are no fields
                    acroForm.getCOSObject().removeItem(COSName.NEED_APPEARANCES);
                    // note that if you've set MDP permissions, the removal of this item
                    // may result in Adobe Reader claiming that the document has been changed.
                    // and/or that field content won't be displayed properly.
                    // ==> decide what you prefer and adjust your code accordingly.
                } else {
                    logger.warning("NeedAppearances is set, signature may be ignored by Adobe Reader");
                }
            }

            // default filter
            pdSignature.setFilter(PDSignature.FILTER_ADOBE_PPKLITE);

            // subfilter for basic and PAdES Part 2 signatures
            pdSignature.setSubFilter(PDSignature.SUBFILTER_ADBE_PKCS7_DETACHED);

            // pdSignature.setName("Name");
            // pdSignature.setLocation("Location");
            if (reason != null && !reason.isEmpty()) {
                pdSignature.setReason(reason);
            }

            // the signing date, needed for valid signature
            pdSignature.setSignDate(Calendar.getInstance());

            // do not set SignatureInterface instance, if external signing used
            // Signature sig = new Signature(certificateChain, privateKey);

            Signature signature = null;
            Certificate[] certificateChain = null;

            certificateChain = keystoreService.loadCertificate(certAlias);
            if (certificateChain == null || certificateChain.length == 0) {
                throw new CertificateVerificationException(
                        "...certificate alias '" + certAlias + "' not found in keystore");
            }

            // create the Signature for signing.....
            try {
                // test if a TSA URL was injected....
                String sTsaUrl = null;
                if (tsaURL.isPresent() && !tsaURL.get().isEmpty()) {
                    sTsaUrl = tsaURL.get();
                }
                // load the corresponding private key from the keystore...
                PrivateKey privateKey = keystoreService.loadPrivateKey(certAlias, certPassword);

                // create a signature object..
                signature = new Signature(certificateChain, privateKey, sTsaUrl);
            } catch (UnrecoverableKeyException | CertificateNotYetValidException | CertificateExpiredException
                    | KeyStoreException | NoSuchAlgorithmException | IOException e) {
                throw new SigningException("Failed to create signature - " + e.getMessage(), e);

            }

            // register signature dictionary and sign interface
            signatureOptions = new SignatureOptions();

            // create visual signature if a signing rect object exists....
            if (rect != null) {
                // we adjust the page as the template uses index 0 for the first page
                if (page > 0) {
                    page--;
                }
                signatureOptions.setVisualSignature(
                        createVisualSignatureTemplate(doc, page, rect, pdSignature, imageFile, certificateChain));
            } else {
                logger.info("...Signature Image not provided, no VisualSignature will be added!");
            }

            // we place the signatureOpens on the given page
            signatureOptions.setPage(page);
            doc.addSignature(pdSignature, signature, signatureOptions);

            if (externalSigning) {
                ExternalSigningSupport externalSigningSupport = doc.saveIncrementalForExternalSigning(bos);
                // invoke external signature service
                byte[] cmsSignature = signature.sign(externalSigningSupport.getContent());

                // set signature bytes received from the service and save the file
                externalSigningSupport.setSignature(cmsSignature);

            } else {
                // write incremental (only for signing purpose)
                doc.saveIncremental(bos);
                signedContent = bos.toByteArray();
            }

        } catch (IOException e) {
            throw new SigningException("Failed to create signature - " + e.getMessage(), e);
        } finally {
            // Do not close signatureOptions before saving, because some COSStream objects
            // within
            // are transferred to the signed document.
            // Do not allow signatureOptions get out of scope before saving, because then
            // the COSDocument
            // in signature options might by closed by gc, which would close COSStream
            // objects prematurely.
            // See https://issues.apache.org/jira/browse/PDFBOX-3743
            IOUtils.closeQuietly(signatureOptions);
        }

        // return the new singed content
        return signedContent;
    }

    private PDRectangle createSignatureRectangle(PDDocument doc, Rectangle2D humanRect) {
        float x = (float) humanRect.getX();
        float y = (float) humanRect.getY();
        float width = (float) humanRect.getWidth();
        float height = (float) humanRect.getHeight();
        PDPage page = doc.getPage(0);
        PDRectangle pageRect = page.getCropBox();
        PDRectangle rect = new PDRectangle();
        // signing should be at the same position regardless of page rotation.
        switch (page.getRotation()) {
        case 90:
            rect.setLowerLeftY(x);
            rect.setUpperRightY(x + width);
            rect.setLowerLeftX(y);
            rect.setUpperRightX(y + height);
            break;
        case 180:
            rect.setUpperRightX(pageRect.getWidth() - x);
            rect.setLowerLeftX(pageRect.getWidth() - x - width);
            rect.setLowerLeftY(y);
            rect.setUpperRightY(y + height);
            break;
        case 270:
            rect.setLowerLeftY(pageRect.getHeight() - x - width);
            rect.setUpperRightY(pageRect.getHeight() - x);
            rect.setLowerLeftX(pageRect.getWidth() - y - height);
            rect.setUpperRightX(pageRect.getWidth() - y);
            break;
        case 0:
        default:
            rect.setLowerLeftX(x);
            rect.setUpperRightX(x + width);
            rect.setLowerLeftY(pageRect.getHeight() - y - height);
            rect.setUpperRightY(pageRect.getHeight() - y);
            break;
        }
        return rect;
    }

    // create a template PDF document with empty signature and return it as a
    // stream.
    private InputStream createVisualSignatureTemplate(PDDocument srcDoc, int pageNum, PDRectangle rect,
            PDSignature signature, byte[] imageFile, Certificate[] certificateChain) throws IOException {

        final int SIGNATURE_DETAILS_OFSET = 40;

        try (PDDocument doc = new PDDocument()) {

            PDPage page = new PDPage(srcDoc.getPage(pageNum).getMediaBox());
            doc.addPage(page);
            PDAcroForm acroForm = new PDAcroForm(doc);
            doc.getDocumentCatalog().setAcroForm(acroForm);
            PDSignatureField signatureField = new PDSignatureField(acroForm);
            PDAnnotationWidget widget = signatureField.getWidgets().get(0);
            List<PDField> acroFormFields = acroForm.getFields();
            acroForm.setSignaturesExist(true);
            acroForm.setAppendOnly(true);
            acroForm.getCOSObject().setDirect(true);
            acroFormFields.add(signatureField);

            widget.setRectangle(rect);

            // from PDVisualSigBuilder.createHolderForm()
            PDStream stream = new PDStream(doc);
            PDFormXObject form = new PDFormXObject(stream);
            PDResources res = new PDResources();
            form.setResources(res);
            form.setFormType(1);
            PDRectangle bbox = new PDRectangle(rect.getWidth(), rect.getHeight());
            float height = bbox.getHeight();
            float width = bbox.getWidth();
            Matrix initialScale = null;
            switch (srcDoc.getPage(pageNum).getRotation()) {
            case 90:
                form.setMatrix(AffineTransform.getQuadrantRotateInstance(1));
                initialScale = Matrix.getScaleInstance(bbox.getWidth() / bbox.getHeight(),
                        bbox.getHeight() / bbox.getWidth());
                height = bbox.getWidth();
                break;
            case 180:
                form.setMatrix(AffineTransform.getQuadrantRotateInstance(2));
                break;
            case 270:
                form.setMatrix(AffineTransform.getQuadrantRotateInstance(3));
                initialScale = Matrix.getScaleInstance(bbox.getWidth() / bbox.getHeight(),
                        bbox.getHeight() / bbox.getWidth());
                height = bbox.getWidth();
                break;
            case 0:
            default:
                break;
            }
            form.setBBox(bbox);
            PDFont fontNormal = PDType1Font.HELVETICA;// .HELVETICA_BOLD;
            PDFont fontBold = PDType1Font.HELVETICA_BOLD;

            // from PDVisualSigBuilder.createAppearanceDictionary()
            PDAppearanceDictionary appearance = new PDAppearanceDictionary();
            appearance.getCOSObject().setDirect(true);
            PDAppearanceStream appearanceStream = new PDAppearanceStream(form.getCOSObject());
            appearance.setNormalAppearance(appearanceStream);
            widget.setAppearance(appearance);

            try (PDPageContentStream cs = new PDPageContentStream(doc, appearanceStream)) {
                // for 90Â° and 270Â° scale ratio of width / height
                // not really sure about this
                // why does scale have no effect when done in the form matrix???
                if (initialScale != null) {
                    cs.transform(initialScale);
                }
                cs.addRect(-5000, -5000, 10000, 10000);

                // helper border
//                cs.setStrokingColor(Color.GRAY);
//                cs.moveTo(0, 0); // half height - offset
//                cs.lineTo(0, height);
//                cs.lineTo(width, height);
//                cs.lineTo(width, 0);
//                cs.lineTo(0, 0);
//                cs.stroke();

                // **********************************
                // * draw signature image
                // **********************************
                if (imageFile != null) {
                    // save and restore graphics if the image is too large and needs to be scaled
                    cs.saveGraphicsState();
                    // in the following we scale the content stream so that the
                    // signing image fits into the upper half of the rectangle.
                    PDImageXObject img = PDImageXObject.createFromByteArray(doc, imageFile, null);
                    
                    float imageMaxHeight=height-SIGNATURE_DETAILS_OFSET;
                    float  scaleFactorHeight=imageMaxHeight/ img.getHeight();
                    float  scaleFactorWidth=width/ img.getWidth();
                    // find the best fit (width vs. height)
                    float scaleFactor=scaleFactorHeight;
                    if (scaleFactorWidth<scaleFactorHeight) {
                        scaleFactor=scaleFactorWidth;
                    }
                    cs.transform(Matrix.getScaleInstance(scaleFactor, scaleFactor));
                    // Place the image above the SIGNATURE_DETAILS_OFSET
                    cs.drawImage(img, 0, SIGNATURE_DETAILS_OFSET/scaleFactor);
                    cs.restoreGraphicsState();
                }

                // **********************************
                // * draw signature information
                // **********************************

                // first draw signature line
                cs.setStrokingColor(Color.BLACK);
                cs.moveTo(0, SIGNATURE_DETAILS_OFSET); // height - offset
                cs.lineTo(width, SIGNATURE_DETAILS_OFSET);
                cs.stroke();

                // set font
                float fontSize = 8;
                float leading = fontSize * 1.3f;
                cs.setFont(fontBold, fontSize);
                cs.setLeading(leading);

                // begin text below the signature line
                cs.beginText();
                cs.newLineAtOffset(fontSize, SIGNATURE_DETAILS_OFSET - leading); // first line

                X509Certificate cert = (X509Certificate) certificateChain[0];
                X500Name x500Name = new X500Name(cert.getSubjectX500Principal().getName());
                RDN cn = x500Name.getRDNs(BCStyle.CN)[0];
                String name = IETFUtils.valueToString(cn.getFirst().getValue());
                SimpleDateFormat dateFormat = new SimpleDateFormat("EEE, dd. MMM yyyy HH:mm:ss");
                String reason = signature.getReason();

                cs.showText("Signer: ");
                cs.setFont(fontNormal, fontSize);
                cs.showText(name);
                cs.newLine();
                cs.setFont(fontBold, fontSize);
                cs.showText("Date: ");
                cs.setFont(fontNormal, fontSize);
                cs.showText(dateFormat.format(signature.getSignDate().getTime()));
                if (reason != null && !reason.isEmpty()) {
                    cs.newLine();
                    cs.setFont(fontBold, fontSize);
                    cs.showText("Reason: ");
                    cs.setFont(fontNormal, fontSize);
                    cs.showText(reason);
                }
                cs.endText();
            }

            // no need to set annotations and /P entry
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            doc.save(baos);
            return new ByteArrayInputStream(baos.toByteArray());
        }
    }

    /**
     * This method verifies if for a given sigFieldName a signature already exists.
     * If so, the method throws a IllegalStateException. In that case, the
     * signatureField can not be used for another signature and a new empty
     * signatureField have to be created.
     * 
     * @see singPDF
     * @param acroForm
     * @param sigFieldName
     * @return a PDSignature if exits
     */
    private PDSignature findExistingSignature(PDAcroForm acroForm, String sigFieldName) {
        PDSignature signature = null;
        PDSignatureField signatureField;
        if (acroForm != null) {
            signatureField = (PDSignatureField) acroForm.getField(sigFieldName);
            if (signatureField != null) {
                // retrieve signature dictionary
                signature = signatureField.getSignature();
                if (signature == null) {
                    signature = new PDSignature();
                    // after solving PDFBOX-3524
                    // signatureField.setValue(signature)
                    // until then:
                    signatureField.getCOSObject().setItem(COSName.V, signature);
                } else {
                    throw new IllegalStateException("The signature field " + sigFieldName + " is already signed.");
                }
            }
        }
        return signature;
    }

}
