/*******************************************************************************
 *  Imixs IX Workflow Technology
 *  Copyright (C) 2001, 2008 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika
 *******************************************************************************/

package org.imixs.workflow;

import java.text.DateFormat;
import java.util.Date;
import java.util.Vector;

/**
 * The Workflowkernel is the core component of this Framework to control the
 * processing of a workitem. A <code>Workflowmanager</code> loads an instance of
 * a Workflowkernel and hand over a <code>Model</code> and register
 * <code>Plugins</code> for processing one or many workitems.
 * 
 * @author Ralph Soika
 * @version 1.1
 * @see org.imixs.workflow.WorkflowManager
 */

public class WorkflowKernel {

	/** Debug **/
	protected int iDebugLevel = -1;
	public static final int DEBUG_OFF = 0;
	public static final int DEBUG_ERRORS = 1;
	public static final int DEBUG_VERBOSE = 2;

	/** Plugin objects **/
	private Vector vectorPluginsClassNames = null;
	private Vector vectorPlugins = null;
	private WorkflowContext ctx = null;
	private ItemCollection documentContext = null;
	private ItemCollection documentActivity = null;
	private ItemCollection documentProcess = null;
	private Vector vectorEdgeHistory = new Vector();

	

	

	/**
	 * Constructor initialize the contextobject and plugin vectors
	 */
	public WorkflowKernel(WorkflowContext actx) {
		ctx = actx;

		iDebugLevel = ctx.getDebugLevel();
		if (iDebugLevel > DEBUG_OFF) {
			System.out.println("[imixs.org WorkflowKernel (V5.0926) }");
			if (iDebugLevel == DEBUG_ERRORS)
				System.out.println("[WorkflowKernel] DebugLevel=SHOW_ERRORS ");
			else
				System.out.println("[WorkflowKernel] DebugLevel=VERBOSE ");
		}

		vectorPluginsClassNames = new Vector();
		vectorPlugins = new Vector();
	}

	public void registerPlugin(String aClassName) throws Exception {
		vectorPluginsClassNames.addElement(aClassName);
		if (iDebugLevel == DEBUG_VERBOSE)
			System.out.println("[WorkflowKernel] register Plugin: "
					+ aClassName);
	}

	/**
	 * 
	 * @param document
	 * @throws Exception
	 */
	public void process(ItemCollection document) throws Exception {
		try {
			vectorEdgeHistory = new Vector();
			documentContext = document;
			// validateDocumentContext();

			while (hasMoreActivities())
				processActivity();
		} catch (Exception e2) {
			// Kein Result gefunden
			System.out.println("[WorkflowKernel]  error process() ");
			System.out.println(e2.toString());

			// Error Log schreiben
			String sLog = DateFormat.getDateTimeInstance(DateFormat.MEDIUM,
					DateFormat.MEDIUM).format(new Date());

			// 23.6.2005 15:40:41|1000.90
			sLog += "|" + documentContext.getItemValueInteger("$ProcessID")
					+ "." + documentContext.getItemValueInteger("$ActivityID");
			documentContext.replaceItemValue("txtworkflowactivityerror", sLog);

			throw e2;
		} finally {
			// $ActivityID und $ActivityIDList auf 0 setzen
			documentContext.replaceItemValue("$activityid", new Integer(0));
			documentContext.replaceItemValue("$activityidlist", new Integer(0));
		}

	}

	/**********************************************
	 * This method controls the Process-Chain The method returns true if a) a
	 * valid $activityid exists b) the attribute $activityidlist has more valid
	 * ActivityIDs
	 ***/
	private boolean hasMoreActivities() throws Exception {
		int integerID;
		try {
			// is $activityid provided?
			try {
				integerID = documentContext.getItemValueInteger("$activityid");

				if ((integerID > 0))
					return true;
				else
					throw new Exception("no $activityid found");
			} catch (Exception e) {				
				// test if there is a activityID List defined in property $ActivityIDList
				Vector v = (Vector) documentContext
						.getItemValue("$activityidlist");
				
				// remove 0 values if contained!
				while (v.indexOf(new Integer(0))>-1) {
					v.remove(v.indexOf(new Integer(0)));
				}
				
				if ((v != null) && (v.size() > 0)) {
					// yes - load next ID from activityID List
					int iNextID = 0;
					Object oA = v.firstElement();
					if (oA instanceof Integer)
						iNextID = ((Integer) oA).intValue();
					if (oA instanceof Double)
						iNextID = ((Double) oA).intValue();

					if (iNextID > 0) {
						// load activity 
						if (iDebugLevel == DEBUG_VERBOSE)
							System.out
									.println("[WorkflowKernel] loading next entry from $activityidlist="
											+ iNextID);
						v.removeElementAt(0);
						documentContext.replaceItemValue("$activityid",
								new Integer(iNextID));
						documentContext.replaceItemValue("$activityidlist", v);
						return true;
					}
				}
				// no more Activities defined 
				return false;
			}
		} catch (Exception e2) {
			// general error - terminate
			System.out.println("[WorkflowKernel]  error hasMoreActivities() ");
			System.out.println(e2.toString());
			throw e2;
		}
	}

	private void checkWorkItemID(ItemCollection doc) throws Exception {
		// Check if $WorkItemID is available
		if ("".equals(doc.getItemValueString("$WorkItemID"))) {
			String sIDPart1 = Long.toHexString(System.currentTimeMillis());
			double d = Math.random() * 900000000;
			int i = new Double(d).intValue();
			String sIDPart2 = Integer.toHexString(i);
			String workID = sIDPart1 + "-" + sIDPart2;
			doc.replaceItemValue("$WorkItemID", workID);
		}

	}

	/**
	 * This method process a activity instance by loading and running all
	 * plugins. 
	 * 
	 * If an FollowUp Activity is defined (keyFollowUp="1" &
	 * numNextActivityID>0) it will be attached at the $ActiviyIDList.
	 * 
	 * @throws Exception
	 */
	private void processActivity() throws Exception {
		try {
			// Check if $WorkItemID is available
			checkWorkItemID(documentContext);

			loadActivity();
			loadPlugins();

			int iStatus = runPlugins();
			closePlugins(iStatus);

			if (iStatus == Plugin.PLUGIN_ERROR) {
				throw new Exception(
						"[WorkflowKernel] Error in Plugin detected.");
			}

			writeLog();

			// put current edge in history
			vectorEdgeHistory.addElement(documentActivity
					.getItemValueInteger("numprocessid")
					+ "."
					+ documentActivity.getItemValueInteger("numactivityid"));

			/*** Next Task ermitteln **/
			int iNewProcessID = documentActivity
					.getItemValueInteger("numnextprocessid");
			if (iDebugLevel == DEBUG_VERBOSE)
				System.out.println("[WorkflowKernel] new ProcessID="
						+ iNewProcessID + "");

			// determine model version
			String sModelVersion = documentContext
					.getItemValueString("$modelversion");
			// depending of the provided Workflow Context call different methods
			if (ctx instanceof ExtendedWorkflowContext)
				documentProcess = ((ExtendedWorkflowContext) ctx)
						.getExtendedModel().getProcessEntityByVersion(
								iNewProcessID, sModelVersion);
			else
				documentProcess = ctx.getModel()
						.getProcessEntity(iNewProcessID);

			/*******************************************
			 * NextProcessID nur dann setzen wenn NextTask>0
			 * *****************************************/
			if (iNewProcessID > 0) {
				documentContext.replaceItemValue("$processid", new Integer(
						iNewProcessID));
			}

			/***
			 * ckear ActivityID and create new  workflowActivity Instance
			 */
			documentContext.replaceItemValue("$activityid", new Integer(0));
			if (iDebugLevel == DEBUG_VERBOSE)
				System.out.println("[WorkflowKernel] set newProcessID: "
						+ iNewProcessID + " sucessfull");

			/** FollowUp Activty ? **/
			String sFollowUp = documentActivity
					.getItemValueString("keyFollowUp");
			int iNextActivityID = documentActivity
					.getItemValueInteger("numNextActivityID");
			if ("1".equals(sFollowUp) && iNextActivityID > 0) {
				this.appendActivityID(iNextActivityID);
			}

		} catch (Exception e2) {
			// Kein Result gefunden
			System.out.println("[WorkflowKernel]  error processActivity() ");
			System.out.println(e2.toString());
			throw e2;
		}
	}

	/**
	 * This method adds a new ActivityID into the current activityList ($ActivityIDList)
	 * The activity list may not contain 0 values.
	 * 
	 */
	private void appendActivityID(int aID) {
		try {
			// check if activityidlist is available
			Vector v = documentContext.getItemValue("$ActivityIDList");
			if (v == null)
				v = new Vector();
			// clear list?
			if ((v.size() == 1) && ("".equals(v.elementAt(0).toString())))
				v = new Vector();

			v.addElement(new Integer(aID));
			
			// remove 0 values if contained!
			while (v.indexOf(new Integer(0))>-1) {
				v.remove(v.indexOf(new Integer(0)));
			}
			
			documentContext.replaceItemValue("$ActivityIDList", v);
			if (iDebugLevel == DEBUG_VERBOSE)
				System.out
						.println("[WorkflowKernel]  append new Activity ID=" + aID);
			
			
			
			
		} catch (Exception e) {
			// no Opperation
			System.out.println("[WorkflowKernel] ERROR - appendActivityID : "
					+ aID);
		}
	}

	/**
	 * This method is responsible for the internal workflow log. The attribute
	 * txtworkflowactivitylog logs the work-flow from one process to another.
	 * 
	 * @throws Exception
	 */
	private void writeLog() throws Exception {
		try {
			String sLog = DateFormat.getDateTimeInstance(DateFormat.MEDIUM,
					DateFormat.MEDIUM).format(new Date());

			// 22.9.2004 13:50:41|1000.90:=1000
			sLog = sLog + "|"
					+ documentActivity.getItemValueInteger("numprocessid")
					+ "."
					+ documentActivity.getItemValueInteger("numactivityid")
					+ ":="
					+ documentActivity.getItemValueInteger("numnextprocessid");

			Vector vLog = (Vector) documentContext
					.getItemValue("txtworkflowactivitylog");
			if (vLog == null)
				vLog = new Vector();

			vLog.addElement(sLog);
			documentContext.replaceItemValue("txtworkflowactivitylog", vLog);
			documentContext.replaceItemValue("numlastactivityid", new Integer(
					documentActivity.getItemValueInteger("numactivityid")));

		} catch (Exception e) {
			System.out.println("[WorkflowKernel] Error writeLog() "
					+ e.toString());
			throw e;

		}

	}

	/***
	 * This Method loads the current Activity Entity from the provides Model
	 * 
	 * The method also verifies the activity to be valid
	 */
	private void loadActivity() throws Exception {
		try {
			int aProcessID = documentContext.getItemValueInteger("$processid");
			int aActivityID = documentContext
					.getItemValueInteger("$activityid");

			// determine model version
			String sModelVersion = documentContext
					.getItemValueString("$modelversion");

			// depending of the provided Workflow Context call different methods
			if (ctx instanceof ExtendedWorkflowContext)
				documentActivity = ((ExtendedWorkflowContext) ctx)
						.getExtendedModel().getActivityEntityByVersion(
								aProcessID, aActivityID, sModelVersion);
			else
				documentActivity = ctx.getModel().getActivityEntity(aProcessID,
						aActivityID);

			if (documentActivity == null)
				throw new Exception("[WorkflowKernel] model entry "
						+ aProcessID + "." + aActivityID + " not found");

			if (iDebugLevel == DEBUG_VERBOSE)
				System.out.println("[WorkflowKernel] loadActivity: "
						+ aProcessID + "." + aActivityID + " successfully");

			// Check for loop in edge history
			if (vectorEdgeHistory != null) {
				if (vectorEdgeHistory.indexOf((aProcessID + "." + aActivityID)) != -1)
					throw new Exception("[WorkflowKernel] loop detected "
							+ aProcessID + "." + aActivityID + ","
							+ vectorEdgeHistory.toString());
			}

		} catch (Exception e) {
			System.out.println("[WorkflowKernel] Error loadActivity()");
			throw e;

		}

	}

	private void loadPlugins() throws Exception {
		try {
			// Warning if no plugins registerd!
			if (vectorPluginsClassNames.size() == 0
					&& iDebugLevel >= DEBUG_VERBOSE)
				System.out
						.println("[WorkflowKernel] Warning loadPlugins: no plugins defined!");

			for (int i = 0; i < vectorPluginsClassNames.size(); i++) {
				/**** Pluglin laden ***/
				String sPluginClass = vectorPluginsClassNames.elementAt(i)
						.toString();

				if ((sPluginClass != null) && (!"".equals(sPluginClass))) {
					if (iDebugLevel == DEBUG_VERBOSE) {
						System.out.println("[WorkflowKernel] loading Plugin "
								+ sPluginClass + "...");
					}
					Class clazz = null;
					try {
						clazz = Class.forName(sPluginClass);
						Plugin plugin = (Plugin) clazz.newInstance();
						plugin.init(ctx);
						vectorPlugins.add(plugin);

					} catch (Exception e) {
						throw new Exception(
								"[WorkflowKernel] Could not create Plugin: "
										+ sPluginClass + " Reason: "
										+ e.toString());
					}
				}

			}
		} catch (Exception e) {
			System.out.println("[WorkflowKernel] Error loadPlugins()");
			throw e;

		}
	}

	/**
	 * This method runs all registered plugins until the run method of a plugin
	 * breaks with an error In this case the method stops The attribute
	 * txtWorkflowPluginLog will store the list of process plugins with a
	 * timestamp
	 */
	private int runPlugins() throws Exception {
		int iStatus;
		Vector vectorPluginLog;
		try {

			vectorPluginLog = documentContext
					.getItemValue("txtWorkflowPluginLog");
			for (int i = 0; i < vectorPlugins.size(); i++) {
				Plugin plugin = (Plugin) vectorPlugins.elementAt(i);
				if (iDebugLevel == DEBUG_VERBOSE)
					System.out.println("[WorkflowKernel] running Plugin: "
							+ plugin.getClass().getName() + "...");
				iStatus = plugin.run(documentContext, documentActivity);

				// write PluginLog
				String sLog = DateFormat.getDateTimeInstance(DateFormat.LONG,
						DateFormat.MEDIUM).format(new Date());
				vectorPluginLog.addElement(sLog + " "
						+ plugin.getClass().getName() + "=" + iStatus);

				if (iStatus == Plugin.PLUGIN_ERROR) {
					// write PluginLog into workitem
					documentContext.replaceItemValue("txtWorkflowPluginLog",
							vectorPluginLog);
					return Plugin.PLUGIN_ERROR;
				}
			}
			// write PluginLog into workitem
			documentContext.replaceItemValue("txtWorkflowPluginLog",
					vectorPluginLog);
			return Plugin.PLUGIN_OK;

		} catch (Exception e) {
			System.out.println("[WorkflowKernel] Error runPlugins()");
			throw e;

		}

	}

	private void closePlugins(int astatus) throws Exception {
		try {
			for (int i = 0; i < vectorPlugins.size(); i++) {
				Plugin plugin = (Plugin) vectorPlugins.elementAt(i);
				if (iDebugLevel == DEBUG_VERBOSE)
					System.out.println("[WorkflowKernel] closing Plugin: "
							+ plugin.getClass().getName() + "...");
				plugin.close(astatus);
			}
			// reset Plugins
			vectorPlugins = new Vector();
		} catch (Exception e) {
			System.out.println("[WorkflowKernel] Error closePlugins()");
			throw e;

		}

	}

}
