/*******************************************************************************
 *  Imixs IX Workflow Technology
 *  Copyright (C) 2001, 2008 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika
 *******************************************************************************/

package org.imixs.workflow;

import java.util.Collection;

import org.imixs.workflow.exceptions.AccessDeniedException;
import org.imixs.workflow.exceptions.InvalidWorkitemException;
import org.imixs.workflow.exceptions.ProcessingErrorException;

/**
 * The WorkflowManager is the general interface for a concrete implementation of
 * a workflow management system. The Interface defines the basic methods for
 * processing and encountering a workitem. The Workflowmanger instantiate a
 * WorkflowKernel, an supports the platform dependet enviroment for concrete
 * Workitems and Workfmodels.
 * 
 * @author Ralph Soika
 * @version 1.0
 * @see org.imixs.workflow.WorkflowKernel
 */

public interface WorkflowManager {

	/**
	 * This Method process a Workitem. The workitem provided by the method
	 * caller needs at least containing the valid attributes $ProcessID and
	 * $ActivityID (integer) to identify the current processEntity the worktiem
	 * belongs to and the concrete activtyEntity which should be processed by
	 * the wokflowManager implementation. If the workitem is new the method
	 * should create a new instance.
	 * 
	 * The method is responsible to persist the worktiem after successfull
	 * processing. The method returns the workitem with additional workflow
	 * informations defined by the workfowManager Implementation.
	 * 
	 * The Method throws an InvalidWorkitemException if the provided Workitem is
	 * invalid or the provided attributes $ProcessID and $ActivityID (integer)
	 * did not match an valid modelEntity the workitem can be processed to.
	 * 
	 * @param workitem
	 *            a workitem instance which should be processed
	 * @return the Workitem instance after successful processing
	 * @throws InvalidWorkitemException
	 */
	public ItemCollection processWorkItem(ItemCollection workitem)
			throws AccessDeniedException,InvalidWorkitemException,ProcessingErrorException;

	/**
	 * The method removes the provide Workitem form the persistence unit managed
	 * by the WorkflowManager implementation.
	 * 
	 * The Method throws an InvalidWorkitemException if the provided Workitem is
	 * invalid.
	 * 
	 * @param uniqueid
	 *            of the WorkItem to be removed
	 * @throws InvalidWorkitemException
	 */
	public void removeWorkItem(ItemCollection workitem)
			throws AccessDeniedException,InvalidWorkitemException;

	/**
	 * Finds a Workitem by a defined uniuqeID ($uniqueID)
	 * 
	 * @param uniqueid
	 * @return WorkItem
	 * @throws Exception
	 */
	public ItemCollection getWorkItem(String uniqueid) throws Exception;

	/**
	 * The method returns all workitems assigned to a specified username or role
	 * 
	 * @param name
	 *            of user or group
	 * @return List of workitems
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkList(String name) throws Exception;

	/**
	 * This method returns all workitems assigned to another workitem by the
	 * Attribute $uniqueidRef. A Worktitem which holds a reference to another
	 * workitem is a called subprocess or childprocess to the process it
	 * references to. This process which is referenced is called a
	 * parentProcess. By defining references process instances can be structured
	 * in hierarchy.
	 * 
	 * @param aref
	 *            A unique reference to another workitem inside a database
	 * @return List of workitems
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkListByRef(String aref)
			throws Exception;

}
