/*******************************************************************************
 *  Imixs IX Workflow Technology
 *  Copyright (C) 2001, 2008 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika
 *******************************************************************************/

package org.imixs.workflow.plugins;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import org.imixs.workflow.ItemCollection;
import org.imixs.workflow.Plugin;
import org.imixs.workflow.WorkflowContext;

/**
 * This abstract class implements different helper methods used by subclasses
 * 
 * @author Ralph Soika
 * @version 1.1
 * @see org.imixs.workflow.WorkflowManager
 * 
 */

public abstract class AbstractPlugin implements Plugin {

	public WorkflowContext ctx;

	public void init(WorkflowContext actx) throws Exception {
		ctx = actx;
	}

	/**
	 * this method replaces a string with place holders
	 * <p>
	 * 
	 * <code><field></code> defines a field of a workitem wich will be replaced
	 * by its value
	 * <p>
	 * inside a field tag it is possible to provide an additional format string
	 * to be used to format a object value. example:
	 * <code>...<field>datStart[mm:ss]</field>.....</code> this will format a
	 * date object with the format mm:ss
	 * 
	 */
	public String replaceDynamicValues(String aString,
			ItemCollection documentContext) throws Exception {
		int iStartPos;

		if (aString == null)
			return "";

		try {
			// replace <Field> Tag - compute start and end position
			while ((iStartPos = aString.toLowerCase().indexOf(
					"<FIELD>".toLowerCase())) != -1) {
				int iEndPos = aString.toLowerCase().indexOf(
						"</FIELD>".toLowerCase(), iStartPos);
				
				// if no end tag found return string unchanged...
				if (iEndPos==-1) 
					return aString;
				
				// replace substring
				String sFieldName = aString.substring(iStartPos
						+ "<FIELD>".toLowerCase().length(), iEndPos);

				// check if fieldname contains a Format String
				String sFormat = null;
				if (sFieldName.indexOf("[") > -1 && sFieldName.indexOf("]") > -1) {
						sFormat = sFieldName.substring(
								sFieldName.indexOf("[") + 1, sFieldName
										.indexOf("]"));
						sFieldName = sFieldName.substring(0, sFieldName
								.indexOf("["));
				}

				String sFeldWert = formatItemValues(sFieldName, documentContext
						.getItemValue(sFieldName), ", ", sFormat);
				aString = aString.substring(0, iStartPos)
						+ sFeldWert
						+ aString.substring(iEndPos
								+ "</FIELD>".toLowerCase().length());
			}

			return aString;

		} catch (Exception e) {
			System.out
					.println("[AbstractPlugin] Error: replaceDynamicValues() "
							+ e.toString());
			throw e;
		}
	}

	/**
	 * this method formate a string object depending of an attribute type
	 */
	public String formatItemValues(String sFeldname, Object aItem,
			String sDelimeter, String sFormat) throws Exception {

		String sString = "";
		if (aItem == null)
			return "";
		// look for Collections
		if (aItem instanceof Collection) {
			Collection col = (Collection) aItem;
			Iterator it = col.iterator();
			while (it.hasNext()) {

				Object o = it.next();
				String aValue = formatObjectValue(o, sFormat);

				// Convert namFields?
				if (sFeldname.toLowerCase().startsWith("nam")) {
					aValue = AbstractPlugin.getAbbreviateName(aValue);
				}

				sString += aValue;

				// last entry?
				if (it.hasNext()) // no append delimiter
					sString += sDelimeter;
			}

			return sString;

		} else {
			sString = formatObjectValue(aItem, sFormat);
			// Convert namFields?
			if (sFeldname.toLowerCase().startsWith("nam")) {
				sString = AbstractPlugin.getAbbreviateName(sString);
			}

			return sString;
		}
	}

	/**
	 * This helper method test the type of an object provided by a
	 * itemcollection and formats the object into a string value.
	 * 
	 * Only Date Objects will be formated into a modified representation. other
	 * objects will be returned using the toString() method.
	 * 
	 * if an optional format is provided this will be used to format date
	 * objects.
	 * 
	 * @param o
	 * @return
	 */
	private String formatObjectValue(Object o, String format) {

		Date dateValue = null;

		// now test the objct type to date
		if (o instanceof Date) {
			dateValue = (Date) o;
		}

		if (o instanceof Calendar) {
			Calendar cal = (Calendar) o;
			dateValue = cal.getTime();
		}

		// format date string?
		if (dateValue != null) {
			String singleValue = "";
			if (format != null) {
				// format date with provided formater
				SimpleDateFormat formatter = new SimpleDateFormat(format);
				singleValue = formatter.format(dateValue);
			} else
				// use standard formate short/short
				singleValue = DateFormat.getDateTimeInstance(DateFormat.SHORT,
						DateFormat.SHORT).format(dateValue);

			return singleValue;
		}

		return o.toString();
	}

	/**
	 * returns the common name component (CN=) of a hierarchical name.
	 * 
	 * @param aName
	 *            (e.g "CN=Ralph Soika/O=IMIXS")
	 * @return ("Ralph Soika")
	 * 
	 */
	public static String getCommonName(String aName) {
		if (aName == null || "".equals(aName) || aName.indexOf("=") == -1)
			return aName;
		try {
			String sResult = aName.substring(aName.indexOf("=") + 1);
			int iPos = sResult.indexOf("/");
			if (iPos > -1)
				return sResult.substring(0, iPos);
			else
				return sResult;
		} catch (Exception in) {
			// no op;
		}
		return aName;
	}

	/**
	 * returns the abbreviate name of a hierarchical name.
	 * 
	 * @param aName
	 *            (e.g "CN=Ralph Soika/O=IMIXS")
	 * @return ("Ralph Soika/IMIXS")
	 * 
	 */
	public static String getAbbreviateName(String aName) {
		if (aName == null || "".equals(aName) || aName.indexOf("=") == -1)
			return aName;
		try {
			String sResult = aName.substring(aName.indexOf("=") + 1);
			// Ralph Soika/O=IMIXS/OU=DE

			while (sResult.indexOf("=") > -1) {
				int iPos1 = sResult.indexOf("/");
				int iPos2 = sResult.indexOf("=");
				sResult = sResult.substring(0, iPos1 + 1)
						+ sResult.substring(iPos2 + 1);
			}
			return sResult;
		} catch (Exception in) {
			// no op;
		}
		return aName;
	}

}