/*******************************************************************************
 *  Imixs IX Workflow Technology
 *  Copyright (C) 2001, 2008 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika
 *******************************************************************************/

package org.imixs.workflow;

import java.util.Date;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;

/**
 * This Class defines a ValueObject to be used to exchange Datastructures used
 * by the org.imixs.workflow Framework. Every component of this framework should
 * use this wrapper class to easy transport workflowrelevant data between the
 * different workflow modules. ValueObjects, particular in J2EE Applications,
 * have the advantage to improve perfomance of remote methode calls and also
 * enables a flexibly datastructure. A ItemCollection contains various Items
 * (attributes). Every Item exist of a Name and a Value. Internal every Value is
 * stored inside a Vector Class. So it is also possible to store
 * multivalueobjects. The ItemCollection wraps the
 * <code>java.util.Hashtable</code> Class and implements the
 * <code>java.io.Serializable</code> Interface, so the ValeOject can also be
 * serialised inside a remote methode call.
 * 
 * 
 * @author Ralph Soika
 * @version 1.0
 * @see org.imixs.workflow.WorkflowManager
 */

public class ItemCollection implements java.io.Serializable {

	private Hashtable hash = new Hashtable();

	/**
	 * Creates a empty ItemCollection
	 * 
	 * @throws Exception
	 */
	public ItemCollection() { // throws Exception
		super();
	}

	/**
	 * Creates a new ItemCollection and transfers all Objects of map as new
	 * values.
	 * 
	 * @param map
	 * @throws Exception
	 */
	public ItemCollection(Map map) throws Exception {
		Iterator it = map.entrySet().iterator();
		while (it.hasNext()) {
			Map.Entry entry = (Map.Entry) it.next();
			this.replaceItemValue(entry.getKey().toString(), entry.getValue());
		}
	}

	public boolean equals(Object o) {
		if (!(o instanceof ItemCollection))
			return false;
		return hash.equals(((ItemCollection) o).getAllItems());
	}

	/**
	 * returns the Value of a single Item inside the ItemCollection. If the item
	 * has no value, this method returns an empty vector. If no item with the
	 * specified name exists, this method returns an empty vector. It does not
	 * throw an exception. The ItemName is not case sensitive. Use hasItem to
	 * verify the existence of an item.
	 * 
	 * @param aName
	 *            The name of an item.
	 * @return The value or values contained in the item. The data type of the
	 *         value depends on the data type of the item.
	 * @throws Exception
	 */
	public Vector getItemValue(String aName) {
		aName = aName.toLowerCase();
		Object o = hash.get(aName);
		if (o == null)
			return new Vector();
		else {
			Vector v=(Vector) o;
			// scan vector for null values
			for (int i=0;i<v.size();i++){ 
				if ( v.get( i ) ==null )   
				    v.remove( i );  
			}
			return v;
		}
	}

	
	/**
	 * Returns the value of an item with a single text value. If the item has no
	 * value or the value is numeric or non text, this method returns an empty
	 * String. If no item with the specified name exists, this method returns an
	 * empty String. It does not throw an exception. If the item has multiple
	 * values, this method returns the first value. The ItemName is not case
	 * sensitive. Use hasItem to verify the existence of an item.
	 * 
	 * @param aName
	 *            The name of an item.
	 * @return The value of the item
	 * @throws Exception
	 */
	public String getItemValueString(String aName) {
		try {
			aName = aName.toLowerCase();
			Vector v = (Vector) getItemValue(aName);
			if (v.size() == 0)
				return "";
			else {
				// verify if value is null
				Object o=v.firstElement();
				if (o==null)
					return "";
				else 
					return (String)o;
			}
		} catch (ClassCastException e) {
			return "";
		}

	}

	/**
	 * Returns the value of an item with a single numeric value. If the item has
	 * no value or the value is no Integer, or empty, this method returns 0. If
	 * no item with the specified name exists, this method returns 0. It does
	 * not throw an exception. If the item has multiple values, this method
	 * returns the first value. This method throws a Exception if the stored
	 * Item has another item type The ItemName is not case sensitive. Use
	 * hasItem to verify the existence of an item.
	 * 
	 * @param aName
	 * @return integer value
	 * @throws Exception
	 */
	public int getItemValueInteger(String aName) {
		try {
			aName = aName.toLowerCase();
			Vector v = (Vector) getItemValue(aName);
			if (v.size() == 0)
				return 0;

			String sValue = v.firstElement().toString();
			return new Double(sValue).intValue();
		} catch (NumberFormatException e) {
			return 0;
		} catch (ClassCastException e) {
			return 0;
		}
	}
	
	/**
	 * Returns the value of an item with a single Date value. If the item has
	 * no value or the value is no Date, or empty, this method returns null. If
	 * no item with the specified name exists, this method returns null. It does
	 * not throw an exception. If the item has multiple values, this method
	 * returns the first value. This method throws a Exception if the stored
	 * Item has another item type The ItemName is not case sensitive. Use
	 * hasItem to verify the existence of an item.
	 * 
	 * @param aName
	 * @return integer value
	 * @throws Exception
	 */
	public Date getItemValueDate(String aName) {
		try {
			aName = aName.toLowerCase();
			Vector v = (Vector) getItemValue(aName);
			if (v.size() == 0)
				return null;

			Object o=v.firstElement();
			if (!(o instanceof Date))
				return null;
			
			return (Date)o;		
		} catch (ClassCastException e) {
			return null;
		}
	}

	/**
	 * Returns the value of an item with a single numeric value. If the item has
	 * no value or the value is no Double, or empty, this method returns 0.0. If
	 * no item with the specified name exists, this method returns 0.0. It does
	 * not throw an exception. If the item has multiple values, this method
	 * returns the first value. The Itemname is not case sensetive. Use hasItem
	 * to verify the existence of an item.
	 * 
	 * @param aName
	 * @return double value
	 * @throws Exception
	 */
	public double getItemValueDouble(String aName) {
		try {
			aName = aName.toLowerCase();
			Vector v = (Vector) getItemValue(aName);
			if (v.size() == 0)
				return 0.0;
			else
				return ((Double) v.firstElement()).doubleValue();
		} catch (ClassCastException e) {
			return 0.0;
		}
	}

	/**
	 * Returns the boolean value of an item. If the item has
	 * no value or the value is no boolean, or empty, this method returns false. If
	 * no item with the specified name exists, this method returns false. It does
	 * not throw an exception. If the item has multiple values, this method
	 * returns the first value. This method throws a Exception if the stored
	 * Item has another item type. The Itemname is not case sensitive. Use
	 * hasItem to verify the existence of an item.
	 * 
	 * @param aName
	 * @return boolean value
	 * @throws Exception
	 */
	public boolean getItemValueBoolean(String aName) {
		try {
			aName = aName.toLowerCase();
			Vector v = (Vector) getItemValue(aName);
			if (v.size() == 0)
				return false;
			String sValue = v.firstElement().toString();
			return new Boolean(sValue).booleanValue();
		} catch (ClassCastException e) {
			return false;
		}
	}
	
	
	
	/**
	 * Indicates whether an item exists in the document.
	 * 
	 * @param aName
	 *            The name of an item.
	 * @return true if an item with name exists in the document, false if no
	 *         item with name exists in the document
	 * @throws Exception
	 */
	public boolean hasItem(String aName) {
		aName = aName.toLowerCase();
		return (hash.get(aName) != null);
	}

	/**
	 * returns all Items of the Collection as a Map
	 * 
	 * @return Map with all Items
	 */
	public Map getAllItems() {
		return hash;

	}

	/**
	 * Replaces a item of the specified name with one new item, which is
	 * assigned the specified value. If the ItemCollection does not contain an
	 * item with the specified name, this method creates a new item and adds it
	 * to the ItemCollection. The ItemName is not case sensitive. Use hasItem to
	 * verify the existence of an item.
	 * The ItemName will be lower cased. 
	 * 
	 * @param aName
	 *            The name of the item or items you want to replace.
	 * @param o
	 *            The value of the new item. The data type of the item depends
	 *            upon the data type of value, and does not need to match the
	 *            data type of the old item.
	 * @throws Exception
	 */
	public void replaceItemValue(String aName, Object o) throws Exception {
		if (o == null || aName == null)
			return;
		if (!(o instanceof java.io.Serializable)) {
			throw new Exception(
					"[ItemCollection] replaceItemValue() Object no Serializable!");
		}
		aName = aName.toLowerCase();
		if (!(o instanceof Vector)) {
			Vector v = new Vector();
			v.addElement(o);
			hash.put(aName, v);
		} else {
			Vector v=(Vector)o;
			// scan vector for null values
			for (int i=0;i<v.size();i++){ 
				if ( v.get( i ) ==null )   
				    v.remove( i );  
			}
			hash.put(aName, v);
		}
	}

	/**
	 * Replaces all items specified in the map with new items, which are
	 * assigned to the specified values inside the map
	 * 
	 * @param map
	 * @throws Exception
	 */
	public void replaceAllItems(Map map) throws Exception {
		Iterator it = map.entrySet().iterator();
		while (it.hasNext()) {
			Map.Entry entry = (Map.Entry) it.next();
			replaceItemValue(entry.getKey().toString(), entry.getValue());
		}

	}

	/**
	 * removes a attribute from the item collection
	 * @param name
	 */
	public void removeItem(String name) {
		name = name.toLowerCase();
		this.getAllItems().remove(name);
	}
}
