package org.imixs.workflow.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import org.imixs.workflow.ItemCollection;

/**
 * This Class adapts the Imixs ItemCollection Class for usage in a JSF
 * Application. Using the ItemCollectionAdapter allows simplified access to
 * attributes from a JSP/JSF page
 * 
 * Each attribute accessed by the ItemCollectionAdapter will be added
 * automatically to the corresponding ItemCollection Class
 * 
 * Examples:
 * 
 * #{myControllerMB.item['numpersonen']}
 * 
 * #{myControllerMB.itemList['teamlist']}
 * 
 * @author rsoika
 * 
 */
public class ItemCollectionAdapter {
	ItemCollection itemCollection;

	ItemAdapter itemAdapter;
	ItemListAdapter itemListAdapter;
	ItemListArrayAdapter itemListArrayAdapter;

	public ItemCollectionAdapter() {
		itemCollection = new ItemCollection();
		itemAdapter = new ItemAdapter(itemCollection);
		itemListAdapter = new ItemListAdapter(itemCollection);
		itemListArrayAdapter = new ItemListArrayAdapter(itemCollection);
	}

	public ItemCollectionAdapter(ItemCollection acol) {
		itemCollection = acol;
		itemAdapter = new ItemAdapter(itemCollection);
		itemListAdapter = new ItemListAdapter(itemCollection);
		itemListArrayAdapter = new ItemListArrayAdapter(itemCollection);
	}

	public ItemCollection getItemCollection() {
		return itemCollection;
	}

	public void setItemCollection(ItemCollection acol) {
		itemCollection = acol;
		itemAdapter.setItemCollection(itemCollection);
		itemListAdapter.setItemCollection(itemCollection);
		itemListArrayAdapter.setItemCollection(itemCollection);
	}
	
	
	
	/**
	 * This method adds a single file to the ItemCollection. files will be
	 * stored into the property $file.
	 * 
	 * @param data
	 *            - byte array with file data
	 * @param fileName
	 *            - name of the file attachment
	 * @param contentType
	 *            - the contenttype (e.g. 'Text/HTML')
	 * @throws Exception
	 */
	public void addFile(byte[] data, String fileName, String contentType)
			throws Exception {
		if (data != null) {
			Vector<Object> vectorFileInfo = null;

			// IE includes '\' characters! so remove all these characters....
			if (fileName.indexOf('\\') > -1)
				fileName = fileName.substring(fileName.lastIndexOf('\\') + 1);
			if (fileName.indexOf('/') > -1)
				fileName = fileName.substring(fileName.lastIndexOf('/') + 1);

			if (contentType == null || "".equals(contentType))
				contentType = "application/unknown";

			// Store files using a hashmap....
			HashMap mapFiles = null;
			Vector vFiles = getItemCollection().getItemValue("$file");
			if (vFiles != null && vFiles.size() > 0)
				mapFiles = (HashMap) vFiles.elementAt(0);
			else
				mapFiles = new HashMap();

			// existing file will be overridden!
			vectorFileInfo = (Vector) mapFiles.get(fileName);
			vectorFileInfo = new Vector<Object>();
			// put file in a vector containing the byte array and also the
			// content type
			vectorFileInfo.add(contentType);
			vectorFileInfo.add(data);
			mapFiles.put(fileName, vectorFileInfo);
			getItemCollection().replaceItemValue("$file", mapFiles);
		}
	}

	/**
	 * This method removes a single file attachment from the BlobWorkitem
	 * 
	 * @throws Exception
	 */
	public void removeFile(String aFilename) throws Exception {

		if (getItemCollection() != null) {
			/* delete attachment */
			String[] files = new String[0];
			HashMap mapFiles = null;
			try {
				Vector vFiles = getItemCollection().getItemValue("$file");
				if (vFiles != null && vFiles.size() > 0) {
					mapFiles = (HashMap) vFiles.elementAt(0);
					mapFiles.remove(aFilename);
					getItemCollection().replaceItemValue("$file", mapFiles);
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	
	/**
	 * Returns a list of file names attached to the current BlobWorkitem. File
	 * Attachments can be added using the method addFile().
	 * 
	 * @return
	 */
	public String[] getFiles() {
		// File attachments...
		String[] files = new String[0];
		if (getItemCollection() != null) {
			HashMap mapFiles = null;
			Vector vFiles = getItemCollection().getItemValue("$file");
			if (vFiles != null && vFiles.size() > 0) {
				mapFiles = (HashMap) vFiles.elementAt(0);
				files = new String[mapFiles.entrySet().size()];
				Iterator iter = mapFiles.entrySet().iterator();
				int iFileCount = 0;
				while (iter.hasNext()) {
					Map.Entry mapEntry = (Map.Entry) iter.next();
					String aFileName = mapEntry.getKey().toString();
					files[iFileCount] = aFileName;
					iFileCount++;
				}
			}
		}
		return files;
	}

	
	

	public Map getItem() {
		return itemAdapter;
	}

	public Map getItemList() {
		return itemListAdapter;
	}

	public Map getItemListArray() {
		return itemListArrayAdapter;
	}

	/**
	 * This class helps to addapt the behavior of a singel value item to be used
	 * in a jsf page using a expression language like this:
	 * 
	 * #{mybean.item['txtMyItem']}
	 * 
	 * 
	 * @author rsoika
	 * 
	 */
	class ItemAdapter implements Map {
		ItemCollection itemCollection;

		public ItemAdapter() {
			itemCollection = new ItemCollection();
		}

		public ItemAdapter(ItemCollection acol) {
			itemCollection = acol;
		}

		public void setItemCollection(ItemCollection acol) {
			itemCollection = acol;
		}

		/**
		 * returns a single value out of the ItemCollection if the key dos not
		 * exist the method will create a value automatical
		 */
		@SuppressWarnings("unchecked")
		public Object get(Object key) {
			// check if a value for this key is available...
			// if not create a new empty value
			try {
				if (!itemCollection.hasItem(key.toString()))
					itemCollection.replaceItemValue(key.toString(), "");
			} catch (Exception e) {

				e.printStackTrace();
			}

			// return first value from vector if size >0
			Vector v = itemCollection.getItemValue(key.toString());
			if (v.size() > 0)
				return v.firstElement();
			else
				// otherwise return null
				return null;
		}

		/**
		 * puts a single value into the ItemCollection
		 */
		public Object put(Object key, Object value) {
			try {
				if (key==null)
					return null;
				itemCollection.replaceItemValue(key.toString(), value);
			} catch (Exception e) {
				e.printStackTrace();
			}
			return value;
		}

		/* ############### Default methods ################# */

		public void clear() {
			itemCollection.getAllItems().clear();
		}

		public boolean containsKey(Object key) {
			return itemCollection.getAllItems().containsKey(key);
		}

		public boolean containsValue(Object value) {
			return itemCollection.getAllItems().containsValue(value);
		}

		public Set entrySet() {
			return itemCollection.getAllItems().entrySet();
		}

		public boolean isEmpty() {
			return itemCollection.getAllItems().isEmpty();
		}

		public Set keySet() {
			return itemCollection.getAllItems().keySet();
		}

		public void putAll(Map m) {
			itemCollection.getAllItems().putAll(m);

		}

		public Object remove(Object key) {
			return itemCollection.getAllItems().remove(key);
		}

		public int size() {
			return itemCollection.getAllItems().size();
		}

		public Collection values() {
			return itemCollection.getAllItems().values();
		}

	}

	/**
	 * This class helps to addapt the behavior of a multivalue item to be used
	 * in a jsf page using a expression language like this:
	 * 
	 * #{mybean.item['txtMyList']}
	 * 
	 * 
	 * @author rsoika
	 * 
	 */
	class ItemListAdapter extends ItemAdapter {

		public ItemListAdapter(ItemCollection acol) {
			itemCollection = acol;
		}

		/**
		 * returns a multi value out of the ItemCollection if the key dos not
		 * exist the method will create a value automatical
		 */
		public Object get(Object key) {
			// check if a value for this key is available...
			// if not create a new empty value
			try {
				if (!itemCollection.hasItem(key.toString()))
					itemCollection.replaceItemValue(key.toString(), "");
			} catch (Exception e) {

				e.printStackTrace();
			}
			// return vector
			return itemCollection.getItemValue(key.toString());
		}

	}

	class ItemListArrayAdapter extends ItemAdapter {

		public ItemListArrayAdapter(ItemCollection acol) {
			itemCollection = acol;
		}

		/**
		 * returns a multi value out of the ItemCollection if the key dos not
		 * exist the method will create a value automatical
		 */
		public Object get(Object key) {
			// check if a value for this key is available...
			// if not create a new empty value
			try {
				if (!itemCollection.hasItem(key.toString()))
					itemCollection.replaceItemValue(key.toString(), "");
			} catch (Exception e) {
				e.printStackTrace();
			}
			// return new ArrayList Object containing values from vector
			ArrayList<Object> aList = new ArrayList<Object>();
			Collection col = itemCollection.getItemValue(key.toString());
			for (Object aEntryValue : col) {
				aList.add(aEntryValue);
			}
			return aList;

		}

		/**
		 * puts a arraylist value into the ItemCollection
		 */
		public Object put(Object key, Object value) {
			
			if (key==null)
				return null;
			
			try {
				// skipp null values
				if (value==null) {
					itemCollection.replaceItemValue(key.toString(), new Vector());
					return null;
				}
				// convert List into Vector object
				if (value instanceof List || value instanceof Object[]) {
					Vector v = new Vector();
					// check type of list (array and list are supported but need
					// to be read in different ways
					if (value instanceof List)
						for (Object aEntryValue : (List) value) {
							v.add(aEntryValue);
						}
					else if (value instanceof Object[])
						for (Object aEntryValue : (Object[]) value) {
							v.add(aEntryValue);
						}
					itemCollection.replaceItemValue(key.toString(), v);
				} else
					// non convertable object!
					itemCollection.replaceItemValue(key.toString(), value);

			} catch (Exception e) {
				e.printStackTrace();
			}
			return value;
		}
	}
}