/*******************************************************************************
 *  Imixs Workflow 
 *  Copyright (C) 2001, 2011 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Project: 
 *  	http://www.imixs.org
 *  	http://java.net/projects/imixs-workflow
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika - Software Developer
 *******************************************************************************/

package org.imixs.workflow.jee.ejb;

import java.util.ArrayList;
import java.util.Collection;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.annotation.security.DeclareRoles;
import javax.annotation.security.RolesAllowed;
import javax.ejb.EJB;
import javax.ejb.LocalBean;
import javax.ejb.SessionContext;
import javax.ejb.Stateless;

import org.imixs.workflow.ExtendedModel;
import org.imixs.workflow.ExtendedWorkflowContext;
import org.imixs.workflow.ItemCollection;
import org.imixs.workflow.Model;
import org.imixs.workflow.WorkflowKernel;
import org.imixs.workflow.exceptions.AccessDeniedException;
import org.imixs.workflow.exceptions.InvalidWorkitemException;
import org.imixs.workflow.exceptions.ProcessingErrorException;
import org.imixs.workflow.jee.jpa.EntityIndex;


/**
 * The WorkflowService is the JEE Implementation for the IX Workflow API. This
 * interface acts as a service facade and supports basic methods to create,
 * process and access workitems. The Interface extends the core api interface
 * org.imixs.workflow.WorkflowManager with getter methods to fetch collections
 * of workitems. The ModelManager is independent form the IX JEE Entity EJBs and
 * uses the standard IntemCollection Object as a data transfer object to
 * comunitcate with clients.
 * 
 * @author rsoika
 * 
 */

@DeclareRoles( { "org.imixs.ACCESSLEVEL.NOACCESS",
		"org.imixs.ACCESSLEVEL.READERACCESS",
		"org.imixs.ACCESSLEVEL.AUTHORACCESS",
		"org.imixs.ACCESSLEVEL.EDITORACCESS",
		"org.imixs.ACCESSLEVEL.MANAGERACCESS" })
@RolesAllowed( { "org.imixs.ACCESSLEVEL.NOACCESS",
		"org.imixs.ACCESSLEVEL.READERACCESS",
		"org.imixs.ACCESSLEVEL.AUTHORACCESS",
		"org.imixs.ACCESSLEVEL.EDITORACCESS",
		"org.imixs.ACCESSLEVEL.MANAGERACCESS" })
@Stateless
@LocalBean
public class WorkflowService implements ExtendedWorkflowContext {
	
	public static final int SORT_ORDER_CREATED_DESC = 0;
	public static final int SORT_ORDER_CREATED_ASC = 1;
	public static final int SORT_ORDER_MODIFIED_DESC = 2;
	public static final int SORT_ORDER_MODIFIED_ASC = 3;


	private ItemCollection profil = null;
	private int DebugLevel = WorkflowKernel.DEBUG_VERBOSE;

	@EJB
	EntityService entityService;

	@EJB
	ModelService modelService;

	@Resource
	SessionContext ctx;

	/**
	 * create default index properties
	 * 
	 * @throws Exception
	 */
	@SuppressWarnings("unused")
	@PostConstruct
	private void initIndexProperties() throws Exception {
		entityService.addIndex("namCreator", EntityIndex.TYP_TEXT);
		entityService.addIndex("txtWorkflowGroup", EntityIndex.TYP_TEXT);
		entityService.addIndex("$ProcessID", EntityIndex.TYP_INT);
		entityService.addIndex("$workitemid", EntityIndex.TYP_TEXT);
		entityService.addIndex("$uniqueidref", EntityIndex.TYP_TEXT);
		entityService.addIndex("txtname", EntityIndex.TYP_TEXT);

	}

	/**
	 * This method loads a Workitem with the corresponding uniqueid
	 */
	public ItemCollection getWorkItem(String uniqueid) throws Exception {
		return entityService.load(uniqueid);
	}

	/**
	 * Returns a collection of workitems belonging to a specified name. The name
	 * is a username or role contained in the $WriteAccess attribute of the
	 * workitem.
	 * 
	 * 
	 * The method returns only workitems the call has sufficient read access
	 * for.
	 * 
	 * @param name
	 *            = username or role contained in $writeAccess - if null current
	 *            username will be used
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkList(String name) throws Exception {
		return getWorkList(name, 0, -1, null,0);
	}

	public Collection<ItemCollection> getWorkList(String name, int startpos,
			int count, String type, int sortorder) throws Exception {

		if (name == null || "".equals(name))
			name = ctx.getCallerPrincipal().getName();

		String sQuery = null;
		sQuery = "SELECT";
		sQuery += " wi FROM Entity as wi " + " JOIN wi.writeAccessList as wa"
				+ " JOIN wi.textItems as s " + "WHERE ";

		if (type != null && !"".equals(type))
			sQuery += " wi.type='" + type + "' AND ";

		sQuery += " wa.value = '" + name + "'"
				+ " AND s.itemName = '$workitemid' ORDER BY wi.created desc";
		return entityService.findAllEntities(sQuery, startpos, count);
	}

	/**
	 * Returns a collection of workitems created by a specified user
	 * (namCreator). The behaivor is simmilar to the method getWorkList.
	 * 
	 * 
	 * @param name
	 *            = username for property namCreator - if null current username
	 *            will be used
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkListByCreator(String name,
			int startpos, int count, String type,int sortorder) throws Exception {

		if (name == null || "".equals(name))
			name = ctx.getCallerPrincipal().getName();

		String sQuery = null;
		sQuery = "SELECT";
		sQuery += " wi FROM Entity as wi"
				+ " JOIN wi.textItems as t JOIN wi.textItems as s " + "WHERE ";
		if (type != null && !"".equals(type))
			sQuery += " wi.type='" + type + "' AND ";

		sQuery += " t.itemName = 'namcreator' and t.itemValue = '" + name + "'"
				+ " AND s.itemName = '$workitemid' "+createSortOrderClause(sortorder);

		return entityService.findAllEntities(sQuery, startpos, count);
	}

	

	/**
	 * Returns a collection of workitems containing a namOwner property
	 * belonging to a specified username. The namOwner property is typical
	 * controled by the OwnerPlugin
	 * 
	 * @param name
	 *            = username for property namOwner - if null current username
	 *            will be used
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkListByOwner(String name,
			int startpos, int count, String type,int sortorder) throws Exception {

		if (name == null || "".equals(name))
			name = ctx.getCallerPrincipal().getName();

		String sQuery = null;
		sQuery = "SELECT";
		sQuery += " wi FROM Entity as wi"
				+ " JOIN wi.textItems as t JOIN wi.textItems as s " + "WHERE ";
		if (type != null && !"".equals(type))
			sQuery += " wi.type='" + type + "' AND ";

		sQuery += " t.itemName = 'namowner' and t.itemValue = '" + name + "'"
				+ " AND s.itemName = '$workitemid' "+createSortOrderClause(sortorder);

		return entityService.findAllEntities(sQuery, startpos, count);
	}

	/**
	 * Returns a collection of workitems where the current user has a
	 * writeAccess. This means the either the username or one of the userroles
	 * is contained in the $writeaccess property
	 * 
	 * 
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkListByWriteAccess(int startpos,
			int count, String type,int sortorder) throws Exception {
		String nameList = "";

		String name = ctx.getCallerPrincipal().getName();

		// construct nameList. Begin with empty string '' and username
		nameList = "'" + name + "'";
		// now construct role list

		String accessRoles = entityService.getAccessRoles();

		String roleList = "org.imixs.ACCESSLEVEL.READERACCESS,org.imixs.ACCESSLEVEL.AUTHORACCESS,org.imixs.ACCESSLEVEL.EDITORACCESS,"
				+ accessRoles;
		// add each role the user is in to the name list
		StringTokenizer roleListTokens = new StringTokenizer(roleList, ",");
		while (roleListTokens.hasMoreTokens()) {
			String testRole = roleListTokens.nextToken().trim();
			if (!"".equals(testRole) && ctx.isCallerInRole(testRole))
				nameList += ",'" + testRole + "'";
		}

		String sQuery = "SELECT wi FROM Entity as wi "
				+ " JOIN wi.writeAccessList wa " + " WHERE ";
		if (type != null && !"".equals(type))
			sQuery += " wi.type='" + type + "' ";

		sQuery += " AND wa.value IN (" + nameList + ")"
				+ createSortOrderClause(sortorder);

		return entityService.findAllEntities(sQuery, startpos, count);
	}

	public Collection<ItemCollection> getWorkListByGroup(String name,
			int startpos, int count, String type,int sortorder) throws Exception {

		String sQuery = null;
		sQuery = "SELECT";
		sQuery += " wi FROM Entity as wi " + " JOIN wi.textItems as t "
				+ " JOIN wi.textItems as s " + "WHERE ";

		if (type != null && !"".equals(type))
			sQuery += " wi.type='" + type + "' AND ";

		sQuery += " t.itemName = 'txtworkflowgroup' and t.itemValue = '" + name
				+ "'"
				+ " AND s.itemName = '$workitemid' "+createSortOrderClause(sortorder);
		return entityService.findAllEntities(sQuery, startpos, count);
	}

	/**
	 * Returns a collection of workitems belonging to a specified $processID
	 * defined by the workflow model. The behaivor is simmilar to the method
	 * getWorkList.
	 * 
	 * @param aID
	 *            = $ProcessID for the workitems to be returned.
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkListByProcessID(int aid,
			int startpos, int count, String type,int sortorder) throws Exception {

		String sQuery = null;
		sQuery = "SELECT";
		sQuery += " wi FROM Entity as wi "
				+ " JOIN wi.integerItems as t JOIN wi.textItems as s "
				+ "WHERE ";

		if (type != null && !"".equals(type))
			sQuery += " wi.type='" + type + "' AND ";

		sQuery += " t.itemName = '$processid' and t.itemValue = '" + aid + "'"
				+ " AND s.itemName = '$workitemid' "+createSortOrderClause(sortorder);

		return entityService.findAllEntities(sQuery, startpos, count);
	}

	/**
	 * Returns a collection of workitems belonging to a specified workitem
	 * identified by the attribute $UniqueIDRef.
	 * 
	 * The behaivor of this Mehtod is simmilar to the method getWorkList.
	 * 
	 * @param aref
	 *            A unique reference to another workitem inside a database *
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkListByRef(String aref)
			throws Exception {
		return getWorkListByRef(aref, 0, -1,null,0);
	}

	/**
	 * Returns a collection of workitems belonging to a specified workitem
	 * identified by the attribute $UniqueIDRef.
	 * 
	 * The behaivor of this Mehtod is simmilar to the method getWorkList.
	 * 
	 * @param aref
	 *            A unique reference to another workitem inside a database *
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkListByRef(String aref,
			int startpos, int count, String type,int sortorder) throws Exception {

		String sQuery = null;
		sQuery = "SELECT";
		sQuery += " wi FROM Entity as wi "
				+ " JOIN wi.textItems as t JOIN wi.textItems as s " + "WHERE ";
		if (type != null && !"".equals(type))
			sQuery += " wi.type='" + type + "' AND ";

		sQuery += " t.itemName = '$uniqueidref' and t.itemValue = '" + aref
				+ "'"
				+ " and s.itemName = '$workitemid' "+createSortOrderClause(sortorder);

		return entityService.findAllEntities(sQuery, startpos, count);
	}

	/**
	 * generates a sort order clause depending on a sororder id
	 * 
	 * @param asortorder
	 * @return
	 */
	private String createSortOrderClause(int asortorder) {
		switch (asortorder) {

		case WorkflowService.SORT_ORDER_CREATED_ASC: {
			return " ORDER BY wi.created asc";
		}
		case WorkflowService.SORT_ORDER_MODIFIED_ASC: {
			return " ORDER BY wi.modified asc";
		}
		case WorkflowService.SORT_ORDER_MODIFIED_DESC: {
			return " ORDER BY wi.modified desc";
		}
		default:
			return " ORDER BY wi.created desc";
		}

	}

	/**
	 * processes a workitem with the provided Activity ID
	 */
	public ItemCollection processWorkItem(ItemCollection itemCollection)
			throws AccessDeniedException, InvalidWorkitemException,
			ProcessingErrorException {

		int aActivityInstanceID = itemCollection
				.getItemValueInteger("$activityid");
		if (aActivityInstanceID <= 0)
			throw new InvalidWorkitemException("$activityid not defined");

		String modelversion = itemCollection
				.getItemValueString("$modelversion");

		/*
		 * get the current Profile Entity for this version. The method will
		 * automatically default to the highest Version number if the provided
		 * version is not available by the ModelManager
		 */
		try {
			modelversion = updateProfileEntity(modelversion);
		} catch (Exception e) {
			throw new InvalidWorkitemException("Invalid ModelVersion");
		}
		WorkflowKernel workflowkernel = new WorkflowKernel(this);

		// register plugins defined in the environment.profile ....
		Vector vPlugins = profil.getItemValue("txtPlugins");

		for (int i = 0; i < vPlugins.size(); i++) {
			String sPlugin = (String) vPlugins.elementAt(i);
			try {
				workflowkernel.registerPlugin(sPlugin);
			} catch (Exception e) {
				throw new ProcessingErrorException(
						"Unable to register Plugin - " + sPlugin);
			}
		}

		try {
			// update Model Version for the Workitem - this is the version the
			// workitem will be processed now
			itemCollection.replaceItemValue("$modelversion", modelversion);

			// identify Caller and update CurrentEditor
			String name;
			name = ctx.getCallerPrincipal().getName();

			// add namCreator if new workitem
			if ("".equals(itemCollection.getItemValueString("namCreator")))
				itemCollection.replaceItemValue("namCreator", name);

			// update curreneditor
			itemCollection.replaceItemValue("namlasteditor", itemCollection
					.getItemValueString("namcurrenteditor"));
			itemCollection.replaceItemValue("namcurrenteditor", name);

			// now process the workitem
			workflowkernel.process(itemCollection);

		} catch (Exception e) {
			throw new ProcessingErrorException("Unable to process workitem: "
					+ e.getLocalizedMessage());
		}
		if (this.getDebugLevel() == WorkflowKernel.DEBUG_VERBOSE)
			System.out
					.println("[WorkflowManager] workitem processed sucessfull");

		return entityService.save(itemCollection);

	}

	public void removeWorkItem(ItemCollection aworkitem)
			throws AccessDeniedException, InvalidWorkitemException {
		entityService.remove(aworkitem);
	}

	/***************************************************************************
	 * Workflow Context
	 */
	public int getDebugLevel() {
		return DebugLevel;
	}

	/**
	 * This Method returns the modelManager Instance. The current ModelVersion
	 * is automatically updated during the Method updateProfileEntity which is
	 * called from the processWorktiem method.
	 * 
	 */
	public Model getModel() {
		return modelService;
	}

	public ExtendedModel getExtendedModel() {
		return modelService;
	}

	public Object getSessionContext() {
		return ctx;
	}

	/**
	 * This method lookups the environment.profile form the model to determine
	 * the debuglevel and current Plugin definition
	 * 
	 * The Method expects the modelVersion to be used by the WorkflowManager to
	 * process a workitem.
	 * 
	 * If no Model to the provided ModelVersion is found the method fetches
	 * automatically the highest ModelVersion proivided by the ModelManager
	 * 
	 * The mehtod returns the selected modelVersion. The Version can be
	 * different from the provided version if no coresponding model was found.
	 * So the higherst verion number defaults
	 * 
	 * 
	 * @throws Exception
	 * 
	 */
	private String updateProfileEntity(String modelversion) throws Exception {
		DebugLevel = WorkflowKernel.DEBUG_VERBOSE;

		// if no modelversion is provided default to newest version
		if (modelversion == null || "".equals(modelversion)) {
			modelversion = modelService.getLatestVersion();
		}

		// try to get Profile for current version.....
		Collection<ItemCollection> col;
		String sQuery = null;
		sQuery = "SELECT";
		sQuery += " environment FROM Entity AS environment"
				+ " JOIN environment.textItems as n "
				+ " JOIN environment.textItems as v"
				+ " WHERE environment.type = 'WorkflowEnvironmentEntity'"
				+ " AND n.itemName = 'txtname' AND n.itemValue = 'environment.profile'"
				+ " AND v.itemName = '$modelversion' AND v.itemValue = '"
				+ modelversion + "'";
		col = entityService.findAllEntities(sQuery, 0, 1);
		// if no model for that ModelVersion is found - default to the higherst
		// Version
		if (col.size() == 0) {
			// reset model version to get the latest versionnummer...
			modelversion = modelService.getLatestVersion();

			sQuery = "SELECT";
			sQuery += " environment FROM Entity AS environment"
					+ " JOIN environment.textItems as n "
					+ " JOIN environment.textItems as v"
					+ " WHERE environment.type = 'WorkflowEnvironmentEntity'"
					+ " AND n.itemName = 'txtname' AND n.itemValue = 'environment.profile'"
					+ " AND v.itemName = '$modelversion' AND v.itemValue = '"
					+ modelversion + "'";
			// no try to get current profile again....
			col = entityService.findAllEntities(sQuery, 0, 1);
		}

		if (col.size() > 0) {
			// ItemCollection wi = col.iterator().next();
			profil = col.iterator().next();

			// determine Debuglevel....
			String sDebug = profil.getItemValueString("keyDebugLevel");
			try {
				int idebug = Integer.parseInt(sDebug);
				DebugLevel = idebug;
			} catch (Exception e) {
				DebugLevel = WorkflowKernel.DEBUG_VERBOSE;
			}
			return modelversion;
		} else
			throw new Exception(
					"[WorkflowManagerImplemenation] fatal error - now valid model version '"
							+ modelversion + "' found! Verify WorkflowModels.");

	}

	/**
	 * This method returns an instance of the Imixs JEE EntityService used by
	 * the WorkflowManager Implementation. The method can be used to access the
	 * EntityService during a Plugin call.
	 * 
	 * @return EntityService
	 * @throws Exception
	 */
	public EntityService getEntityService() throws Exception {
		return entityService;
	}
	
	/**
	 * This method returns an instance of the Imixs JEE ModelService used by
	 * the WorkflowManager Implementation. The method can be used to access the
	 * ModelService during a Plugin call.
	 * 
	 * @return EntityService
	 * @throws Exception
	 */
	public ModelService getModelService() throws Exception {
		return modelService;
	}

	/**
	 * Obtain the java.security.Principal that identifies the caller and returns
	 * the name of this principal.
	 * 
	 * @return the user name
	 */
	public String getUserName() throws Exception {
		return ctx.getCallerPrincipal().getName();

	}

	/**
	 * Test if the caller has a given security role.
	 * 
	 * @param rolename
	 * @return true if user is in role
	 */
	public boolean isUserInRole(String rolename) throws Exception {
		try {
			return ctx.isCallerInRole(rolename);
		} catch (Exception e) {
			// avoid a exception for a role request which is not defined
			return false;
		}
		
	}

	/**
	 * Returns a String array containing all user roles the caller has given
	 * 
	 * @return String array with the users access roles
	 * @throws Exception
	 */
	public String[] getUserRoles() throws Exception {
		ArrayList<String> roleList = new ArrayList<String>();
		// create default role list
		String roles = EntityService.ACCESSLEVEL_READERACCESS + ","
				+ EntityService.ACCESSLEVEL_AUTHORACCESS + ","
				+ EntityService.ACCESSLEVEL_EDITORACCESS + ","
				+ EntityService.ACCESSLEVEL_MANAGERACCESS;
		// get the individual access roles provided by the EntityService EJB
		String accessRoles = this.getEntityService().getAccessRoles();
		if (accessRoles != null && !"".equals(accessRoles))
			roles += "," + accessRoles;
		// add each role the user is in to the String array list
		StringTokenizer roleListTokens = new StringTokenizer(roles, ",");
		while (roleListTokens.hasMoreTokens()) {
			String testRole = roleListTokens.nextToken().trim();
			if (!"".equals(testRole) && isUserInRole(testRole))
				roleList.add(testRole);
		}

		// convert to string array
		String[] result = new String[roleList.size()];
		for (int i=0;i<roleList.size();i++)
			result[i]=roleList.get(i);
			
		
		return result;
	}
}
