/*******************************************************************************
 *  Imixs Workflow 
 *  Copyright (C) 2001, 2011 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Project: 
 *  	http://www.imixs.org
 *  	http://java.net/projects/imixs-workflow
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika - Software Developer
 *******************************************************************************/

package org.imixs.workflow.plugins;

import java.util.Enumeration;
import java.util.Vector;

import org.imixs.workflow.ItemCollection;
import org.imixs.workflow.Plugin;
import org.imixs.workflow.WorkflowContext;
import org.imixs.workflow.WorkflowKernel;

/**
 * This Plugin module implements a generic Access management controled throught the 
 * configuration in a Workflowactivity.
 * The Plugin updates the attriubtes $ReadAccess and $WriteAccess depending on the 
 * configuration in a ActivityEntiy.
 * <p>
 * These attributes are: 
 * <ul>
 * <li> keyaccessmode (Vector): '1'=update '0'=renew
 * <li> namaddreadaccess (Vector): Names & Groups to be added /replaced
 * <li> namaddwriteaccess (Vector): Names & Groups to be added/replaced
 * <li> keyaddreadroles (Vector): Roles to be added/replaced
 * <li> keyaddwriteroles (Vector): Roles to added/replaced
 * <li> keyaddreadfields (Vector): Attributes of the processd workitem to add there
 * values 
 * <li> keyaddwritefields (Vector): Attributes of the processd workitem to add therevalues
 * 
 * @author Ralph Soika
 * @version 2.0
 * @see org.imixs.workflow.WorkflowManager
 */

public class AccessPlugin implements Plugin {
	ItemCollection documentContext;
	ItemCollection documentActivity;
	Vector itemReadRollback, itemWriteRollback;
	WorkflowContext workflowContext;

	public void init(WorkflowContext actx) throws Exception {
		workflowContext = actx;
	}

	/**
	 * changes the $readAccess and $writeAccess attribues
	 * depending to the activityentity
	 */
	public int run(ItemCollection adocumentContext,
			ItemCollection adocumentActivity) throws Exception {
		Vector itemRead;
		Vector itemWrite;
		Vector vectorAccess;
		try {
			documentContext = adocumentContext;
			documentActivity = adocumentActivity;

			// Validate Activity and Workitem
			validate();

			itemRead = (Vector) documentContext
					.getItemValue("$readAccess");

			// save Attribute for roleback
			itemReadRollback = (Vector) documentContext
					.getItemValue("$readAccess");

			// neuen ReadAccess hinzuf�gen
			if ("1"
					.equals(documentActivity
							.getItemValueString("keyaccessmode")))
				vectorAccess = itemRead;
			else
				vectorAccess = new Vector();
			if (workflowContext.getDebugLevel() == WorkflowKernel.DEBUG_VERBOSE)
				System.out.println("[AccessPlugin] AccessMode: '"
						+ documentActivity.getItemValueString("keyaccessmode")
						+ "'");

			if (vectorAccess == null)
				vectorAccess = new Vector();

			// **1** AllowAccess add names
			mergeVectors(vectorAccess, documentActivity
					.getItemValue("namaddreadaccess"));
			// **2** AllowAccess add roles
			mergeVectors(vectorAccess, documentActivity
					.getItemValue("keyaddreadroles"));
			// **3** AllowAccess add Mapped Fields
			mergeMappedFieldValues(vectorAccess, documentActivity
					.getItemValue("keyaddreadfields"));

			// clean Vector
			vectorAccess=uniqueVector(vectorAccess);

			// save Vector
			documentContext.replaceItemValue("$readAccess",
					vectorAccess);
			if ((workflowContext.getDebugLevel() == WorkflowKernel.DEBUG_VERBOSE)
					&& (vectorAccess.size() > 0)) {
				System.out.println("[AccessPlugin] ReadAccess:");
				for (int j = 0; j < vectorAccess.size(); j++)
					System.out.println("              "
							+ (String) vectorAccess.elementAt(j));
			}

			/**** now process write access ***/

			// check for $writeAccess
			itemWrite = documentContext.getItemValue("$writeAccess");

			// save Attribute for roleback
			itemWriteRollback = documentContext
					.getItemValue("$writeAccess");

			// add new WriteAccess �gen

			if ("1"
					.equals(documentActivity
							.getItemValueString("keyaccessmode")))
				vectorAccess = itemWrite;
			else
				vectorAccess = new Vector();

			if (vectorAccess == null)
				vectorAccess = new Vector();

			// **1** AllowAccess add Names
			mergeVectors(vectorAccess, documentActivity
					.getItemValue("namaddwriteaccess"));
			// **2** AllowAccess add Rolles
			mergeVectors(vectorAccess, documentActivity
					.getItemValue("keyaddwriteroles"));
			// **3** AllowAccess add Mapped Fields �gen
			mergeMappedFieldValues(vectorAccess, documentActivity
					.getItemValue("keyaddwritefields"));

			// clean Vector
			vectorAccess=uniqueVector(vectorAccess);

			// save Vector
			documentContext.replaceItemValue("$writeAccess",
					vectorAccess);
			if ((workflowContext.getDebugLevel() == WorkflowKernel.DEBUG_VERBOSE)
					&& (vectorAccess.size() > 0)) {
				System.out.println("[AccessPlugin] WriteAccess:");
				for (int j = 0; j < vectorAccess.size(); j++)
					System.out.println("               "
							+ (String) vectorAccess.elementAt(j));
			}

		} catch (Exception e) {
			System.out.println("[AccessPlugin] Error run()" + e.toString());
			return Plugin.PLUGIN_ERROR;
		}

		return Plugin.PLUGIN_OK;
	}

	public void close(int status) throws Exception {
		try {

			// restore changes?
			if (status == Plugin.PLUGIN_ERROR) {
				documentContext.replaceItemValue("$writeAccess",
						itemWriteRollback);
				documentContext.replaceItemValue("$readAccess",
						itemReadRollback);
			}
		} catch (Exception e) {
			System.out.println("[AccessPlugin] Error close() " + e.toString());
			throw e;
		}
	}

	// =========================================================
	// Erg�nzt vectorDest um die Werte aus VectorSource
	// unter ber�cksichtigung von Unique-Werten
	// =========================================================
	private void mergeVectors(Vector p_VectorDestination, Vector p_VectorSource) {

		// System.out.println("mergeVectors ...");
		if ((p_VectorSource != null) && (p_VectorSource.size() > 0)) {

			Enumeration enumSource = p_VectorSource.elements();
			while (enumSource.hasMoreElements()) {
				Object o = enumSource.nextElement();
				if (p_VectorDestination.indexOf(o) == -1)
					p_VectorDestination.addElement(o);
			}
		}
	}

	// ===========================================================
	// == Diese Methode f�gt einem �bergebenen Vector die Inhalte
	// == der Felder aus dem Formular hinzu welche im �bergebenen
	// == vector p_VectorFieldList stehen
	// == Werte werden auf Unique gepr�ft!
	// ===========================================================
	private void mergeMappedFieldValues(Vector p_VectorDestination,
			Vector p_VectorFieldList) {
		try {
			if ((p_VectorFieldList != null) && (p_VectorFieldList.size() > 0)) {
				// Feldliste durchgehen
				Enumeration enumFields = p_VectorFieldList.elements();
				while (enumFields.hasMoreElements()) {
					// Feldwerte f�r jedes Feld aus Formular holen
					String sFeldName = (String) enumFields.nextElement();
					Vector vValues = documentContext.getItemValue(sFeldName);
					// jetzt die Inhalte des Vectors in den p_VectorDestination
					// �bertragen
					if ((vValues != null) && (vValues.size() > 0)) {
						Enumeration enumValues = vValues.elements();
						while (enumValues.hasMoreElements()) {
							Object o = enumValues.nextElement();

							// nur anh�ngen wenn noch nicht vorhanden
							if (p_VectorDestination.indexOf(o) == -1)
								p_VectorDestination.addElement(o);
						}
					}
				}
			}
		} catch (Exception e) {
			System.out
					.println("[AccessPlugin] Error addMappedFieldValuesToVector() "
							+ e.toString());
		}
	}

	/**
	 * Ensures that the workitem and activityentity has a valid set of
	 * attributes to be process by this plugin.
	 */
	private void validate() throws Exception {
		try {
			// validate activity
			if (!documentActivity.hasItem("keyaccessmode"))
				documentActivity.replaceItemValue("keyaccessmode", "");

			if (!documentActivity.hasItem("namaddreadaccess"))
				documentActivity.replaceItemValue("namaddreadaccess", "");

			if (!documentActivity.hasItem("keyaddreadroles"))
				documentActivity.replaceItemValue("keyaddreadroles", "");

			if (!documentActivity.hasItem("keyaddreadfields"))
				documentActivity.replaceItemValue("keyaddreadfields", "");

			if (!documentActivity.hasItem("namaddwriteaccess"))
				documentActivity.replaceItemValue("namaddwriteaccess", "");

			if (!documentActivity.hasItem("keyaddwriteroles"))
				documentActivity.replaceItemValue("keyaddwriteroles", "");

			if (!documentActivity.hasItem("keyaddreadfields"))
				documentActivity.replaceItemValue("keyaddreadfields", "");

			// validate document
			if (!documentContext.hasItem("$readAccess"))
				documentContext.replaceItemValue("$readAccess", "");

			if (!documentContext.hasItem("$writeAccess"))
				documentContext.replaceItemValue("$writeAccess", "");

		} catch (Exception e) {
			System.out.println("[AccessPlugin] Error validate()() "
					+ e.toString());
			throw e;
		}
	}

	/**
	 * this method removes duplicate and null values from a vector object The
	 * method is called by the run method after build new read and write access
	 * elements.
	 * 
	 * @param p_Vector
	 */
	private Vector uniqueVector(Vector p_Vector) {
		int iVectorSize = p_Vector.size();
		Vector cleanedVector=new Vector();
		
		for (int i=0;i<iVectorSize;i++) {
			Object o=p_Vector.elementAt(i);
			if (o==null || cleanedVector.indexOf(o)>-1 || "".equals(o.toString()))
				continue;
			
			// add unique object
			cleanedVector.add(o);
		}
		p_Vector=cleanedVector;
		// do not work with empty vectors....
		if (p_Vector.size() == 0)
			p_Vector.addElement("");
		
		return p_Vector;
	}	
}
