/*******************************************************************************
 *  Imixs Workflow 
 *  Copyright (C) 2001, 2011 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Project: 
 *  	http://www.imixs.org
 *  	http://java.net/projects/imixs-workflow
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika - Software Developer
 *******************************************************************************/

package org.imixs.workflow.plugins;

import java.util.Enumeration;
import java.util.Vector;

import org.imixs.workflow.ItemCollection;
import org.imixs.workflow.Plugin;
import org.imixs.workflow.WorkflowContext;
import org.imixs.workflow.WorkflowKernel;

/**
 * This Pluginmodul implements a Ownership Control for Workflowactivitys. The
 * Plugin manages the modifications of the field namOwner of a Workitem inside a
 * activity by setting this Attribute. A Workflowmanager can use this attributes
 * to reflect there settings to the ownership inside a certain databasesystem.
 * The Plugin checks a set of activity attributes to manage the new settings of
 * the ownership defined inside the activity entity These attributes are:
 * 
 * o keyOwnershipMode (Vector): '1'=modify access '0'=renew access
 * 
 * o namOwnershipNames (Vector): Names & Groups to add to the namOwner (Vector):
 * 
 * o keyaddwriteroles (Vector): Roles to add to the namOwner attribute
 * 
 * @author Ralph Soika
 * @version 1.0
 * @see org.imixs.workflow.WorkflowManager
 */

public class OwnerPlugin implements Plugin {
	ItemCollection documentContext;
	ItemCollection documentActivity;
	Vector itemOwnerRollback;
	WorkflowContext workflowContext;

	public void init(WorkflowContext actx) throws Exception {
		workflowContext = actx;
	}

	/**
	 * changes the namworkflowreadaccess and namworkflowwriteaccess attribues
	 * depending to the activityentity
	 */
	public int run(ItemCollection adocumentContext,
			ItemCollection adocumentActivity) throws Exception {
		Vector itemOwner;
		Vector vectorAccess;
		try {
			documentContext = adocumentContext;
			documentActivity = adocumentActivity;

			// Validate Activity and Workitem
			validate();

			itemOwner = (Vector) documentContext
					.getItemValue("namowner");

			// save Attribute for roleback
			itemOwnerRollback = (Vector) documentContext
					.getItemValue("namOwners");

			// add new ownership
			if ("1"
					.equals(documentActivity
							.getItemValueString("keyOwnershipMode")))
				vectorAccess = itemOwner;
			else
				vectorAccess = new Vector();
			if (workflowContext.getDebugLevel() == WorkflowKernel.DEBUG_VERBOSE)
				System.out.println("[OwnerPlugin] AccessMode: '"
						+ documentActivity.getItemValueString("keyOwnershipMode")
						+ "'");

			if (vectorAccess == null)
				vectorAccess = new Vector();

			// **1** AllowAccess add names
			mergeVectors(vectorAccess, documentActivity
					.getItemValue("namOwnershipNames"));
			
			// **3** AllowAccess add Mapped Fields
			mergeMappedFieldValues(vectorAccess, documentActivity
					.getItemValue("keyOwnershipFields"));

			// clean Vector
			vectorAccess = uniqueVector(vectorAccess);

			// save Vector
			documentContext.replaceItemValue("namOwner",
					vectorAccess);
			if ((workflowContext.getDebugLevel() == WorkflowKernel.DEBUG_VERBOSE)
					&& (vectorAccess.size() > 0)) {
				System.out.println("[OwnerPlugin] Owner:");
				for (int j = 0; j < vectorAccess.size(); j++)
					System.out.println("              "
							+ (String) vectorAccess.elementAt(j));
			}

			

		} catch (Exception e) {
			System.out.println("[AccessPlugin] Error run()" + e.toString());
			return Plugin.PLUGIN_ERROR;
		}

		return Plugin.PLUGIN_OK;
	}

	public void close(int status) throws Exception {
		try {

			// restore changes?
			if (status == Plugin.PLUGIN_ERROR) {
				documentContext.replaceItemValue("namOwner",
						itemOwnerRollback);
				
			}
		} catch (Exception e) {
			System.out.println("[AccessPlugin] Error close() " + e.toString());
			throw e;
		}
	}

	// =========================================================
	// Erg�nzt vectorDest um die Werte aus VectorSource
	// unter ber�cksichtigung von Unique-Werten
	// =========================================================
	private void mergeVectors(Vector p_VectorDestination, Vector p_VectorSource) {

		// System.out.println("mergeVectors ...");
		if ((p_VectorSource != null) && (p_VectorSource.size() > 0)) {

			Enumeration enumSource = p_VectorSource.elements();
			while (enumSource.hasMoreElements()) {
				Object o = enumSource.nextElement();
				if (p_VectorDestination.indexOf(o) == -1)
					p_VectorDestination.addElement(o);
			}
		}
	}

	// ===========================================================
	// == Diese Methode f�gt einem �bergebenen Vector die Inhalte
	// == der Felder aus dem Formular hinzu welche im �bergebenen
	// == vector p_VectorFieldList stehen
	// == Werte werden auf Unique gepr�ft!
	// ===========================================================
	private void mergeMappedFieldValues(Vector p_VectorDestination,
			Vector p_VectorFieldList) {
		try {
			if ((p_VectorFieldList != null) && (p_VectorFieldList.size() > 0)) {
				// Feldliste durchgehen
				Enumeration enumFields = p_VectorFieldList.elements();
				while (enumFields.hasMoreElements()) {
					// Feldwerte f�r jedes Feld aus Formular holen
					String sFeldName = (String) enumFields.nextElement();
					Vector vValues = documentContext.getItemValue(sFeldName);
					// jetzt die Inhalte des Vectors in den p_VectorDestination
					// �bertragen
					if ((vValues != null) && (vValues.size() > 0)) {
						Enumeration enumValues = vValues.elements();
						while (enumValues.hasMoreElements()) {
							Object o = enumValues.nextElement();

							// nur anh�ngen wenn noch nicht vorhanden
							if (p_VectorDestination.indexOf(o) == -1)
								p_VectorDestination.addElement(o);
						}
					}
				}
			}
		} catch (Exception e) {
			System.out
					.println("[AccessPlugin] Error addMappedFieldValuesToVector() "
							+ e.toString());
		}
	}

	/**
	 * Ensures that the workitem and activityentity has a valid set of
	 * attributes to be process by this plugin.
	 */
	private void validate() throws Exception {
		try {
			// validate activity
			if (!documentActivity.hasItem("keyOwnershipMode"))
				documentActivity.replaceItemValue("keyOwnershipMode", "");

			if (!documentActivity.hasItem("namOwnershipNames"))
				documentActivity.replaceItemValue("namOwnershipNames", "");

			
			if (!documentActivity.hasItem("keyOwnershipFields"))
				documentActivity.replaceItemValue("keyOwnershipFields", "");

			
			// validate document
			if (!documentContext.hasItem("namOwner"))
				documentContext.replaceItemValue("namOwner", "");

		} catch (Exception e) {
			System.out.println("[OwnerPlugin] Error validate()() "
					+ e.toString());
			throw e;
		}
	}

	/**
	 * this method removes duplicate and null values from a vector object The
	 * method is called by the run method after build new read and write access
	 * elements.
	 * 
	 * @param p_Vector
	 */
	private Vector uniqueVector(Vector p_Vector) {
		int iVectorSize = p_Vector.size();
		Vector cleanedVector = new Vector();

		for (int i = 0; i < iVectorSize; i++) {
			Object o = p_Vector.elementAt(i);
			if (o == null || cleanedVector.indexOf(o) > -1
					|| "".equals(o.toString()))
				continue;

			// add unique object
			cleanedVector.add(o);
		}
		p_Vector = cleanedVector;
		// do not work with empty vectors....
		if (p_Vector.size() == 0)
			p_Vector.addElement("");

		return p_Vector;
	}
}
