/*******************************************************************************
 *  Imixs Workflow 
 *  Copyright (C) 2001, 2011 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Project: 
 *  	http://www.imixs.org
 *  	http://java.net/projects/imixs-workflow
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika - Software Developer
 *******************************************************************************/

package org.imixs.workflow.plugins;

import java.util.Vector;

import org.imixs.workflow.ItemCollection;
import org.imixs.workflow.Plugin;

/**
 * This Pluginmodul generates a result message. The Plugin checks if the activiy
 * document has an "txtActivityResult" attribute. The value will be parsed for
 * "<field>attribute</field>" tags and these tags will be replaced with the
 * values of the corresponding attributes of the workitem.
 * 
 * The final result will be stored into the attribute
 * "txtworkflowresultmessage".
 * 
 * This field could be used by an application to display individual messages
 * (e.g HTML Code) or return result Strings (e.g. JSF Action Results)
 * 
 * @author Ralph Soika
 * @version 1.0
 * @see org.imixs.workflow.WorkflowManager
 * 
 */
public class ResultPlugin extends AbstractPlugin {
	ItemCollection documentContext;
	String sActivityResult;

	public int run(ItemCollection adocumentContext,
			ItemCollection adocumentActivity) throws Exception {
		try {

			documentContext = adocumentContext;

			// get ResultMessage
			sActivityResult = adocumentActivity
					.getItemValueString("txtActivityResult");
			sActivityResult = replaceDynamicValues(sActivityResult,
					adocumentContext);

			// evaluate new items....
			evaluate(sActivityResult, adocumentContext);

		} catch (Exception e) {
			System.out.println("[ResultPlugin] Error run(): " + e.toString());
			return Plugin.PLUGIN_ERROR;
		}

		return Plugin.PLUGIN_OK;
	}

	public void close(int status) {
		try {
			// restore changes?
			if (status < Plugin.PLUGIN_ERROR) {
				documentContext.replaceItemValue("txtworkflowresultmessage",
						sActivityResult);
			}
		} catch (Exception e) {
			System.out.println("[ResultPlugin] Error close(): " + e.toString());

		}
	}

	/**
	 * This method parses a string for xml tag <item>. Those tags will result in
	 * new workitem properties.
	 * 
	 * <code>
	 *   
	 *   <item name="txtTitle">new Title</item>
	 *   
	 *   
	 * </code>
	 * 
	 * Additional the attribute 'type' can be provided to specify a field type 'boolean', 'string', 'integer'
	 * 
	 * The item will be added into the workitem documentContext
	 * 
	 * 
	 */
	public static void evaluate(String aString, ItemCollection documentContext)
			throws Exception {
		int iTagStartPos;
		int iTagEndPos;

		int iContentStartPos;
		int iContentEndPos;

		int iNameStartPos;
		int iNameEndPos;

		int iTypeStartPos;
		int iTypeEndPos;

		String sName = "";
		String sType = " ";
		String sItemValue;

		if (aString == null)
			return;

		try {
			// test if a <value> tag exists...
			while ((iTagStartPos = aString.toLowerCase().indexOf("<item")) != -1) {

				iTagEndPos = aString.toLowerCase().indexOf("</item>",
						iTagStartPos);

				// if no end tag found return string unchanged...
				if (iTagEndPos == -1)
					return;

				// reset pos vars
				iContentStartPos = 0;
				iContentEndPos = 0;
				iNameStartPos = 0;
				iNameEndPos = 0;
				iTypeStartPos = 0;
				iTypeEndPos = 0;
				sName = "";
				sType = " ";
				sItemValue = "";

				// so we now search the beginning of the tag content
				iContentEndPos = iTagEndPos;
				// start pos is the last > before the iContentEndPos
				String sTestString = aString.substring(0, iContentEndPos);
				iContentStartPos = sTestString.lastIndexOf('>') + 1;

				// if no end tag found return string unchanged...
				if (iContentStartPos >= iContentEndPos)
					return;

				iTagEndPos = iTagEndPos + "</item>".length();

				// now we have the start and end position of a tag and also the
				// start and end pos of the value

				// next we check if the start tag contains a 'name' attribute
				iNameStartPos = aString.toLowerCase().indexOf("name=",
						iTagStartPos);
				// extract format string if available
				if (iNameStartPos > -1 && iNameStartPos < iContentStartPos) {
					iNameStartPos = aString.indexOf("\"", iNameStartPos) + 1;
					iNameEndPos = aString.indexOf("\"", iNameStartPos + 1);
					sName = aString.substring(iNameStartPos, iNameEndPos);
				}
				
				// next we check if the start tag contains a 'type'
				// attribute
				iTypeStartPos = aString.toLowerCase().indexOf(
						"type=", iTagStartPos);
				// extract format string if available
				if (iTypeStartPos > -1
						&& iTypeStartPos < iContentStartPos) {
					iTypeStartPos = aString.indexOf("\"",
							iTypeStartPos) + 1;
					iTypeEndPos = aString.indexOf("\"",
							iTypeStartPos + 1);
					sType = aString.substring(iTypeStartPos,
							iTypeEndPos);
				}

				// extract Item Value
				sItemValue = aString
						.substring(iContentStartPos, iContentEndPos);

				// replace the item value
				if (sName!=null && !"".equals(sName)) {
					// convert to type...
					Object oValue=sItemValue;
					
					if ("boolean".equalsIgnoreCase(sType))
						oValue=new Boolean(sItemValue);
					
					if ("integer".equalsIgnoreCase(sType))
						oValue=new Integer(sItemValue);
					
					documentContext.replaceItemValue(sName, oValue);
				}
				
				// now cut the tag form the string
				aString = aString.substring(0, iTagStartPos) + ""
						+ aString.substring(iTagEndPos);
			}

		} catch (Exception e) {
			System.out.println("[ResultPlugin] Error: evaluate() "
					+ e.toString());
			throw e;
		}
	}

}
