package org.imixs.workflow.jee.ejb;

import java.util.Collection;
import java.util.List;

import javax.ejb.Remote;

import org.imixs.workflow.ItemCollection;
import org.imixs.workflow.Model;
import org.imixs.workflow.exceptions.AccessDeniedException;
import org.imixs.workflow.exceptions.ProcessingErrorException;

@Remote
public interface WorkflowServiceRemote {

	/**
	 * This method loads a Workitem with the corresponding uniqueid
	 */
	public abstract ItemCollection getWorkItem(String uniqueid);

	/**
	 * Returns a collection of workItems belonging to current user. The method returns only 
	 * workitems where the current user has read access. 
	 * 
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 */
	public abstract Collection<ItemCollection> getWorkList(int startpos, int count, String type, int sortorder);

	public abstract Collection<ItemCollection> getWorkList();

	/**
	 * Returns a collection of workItems belonging to a specified username. The name
	 * is a username or role contained in the $WriteAccess attribute of the
	 * workItem.
	 * 
	 * The method returns only workitems the call has sufficient read access
	 * for.
	 * 
	 * @param name
	 *            = username or role contained in $writeAccess - if null current
	 *            username will be used
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 */
	public abstract Collection<ItemCollection> getWorkListByAuthor(String name, int startpos,
			int count, String type, int sortorder);
	
	/**
	 * Returns the worklist by author for the current user
	 * @param name
	 * @return
	 */
	public abstract Collection<ItemCollection> getWorkListByAuthor(String name);


	/**
	 * Returns a collection of workitems created by a specified user
	 * (namCreator). The behaivor is simmilar to the method getWorkList.
	 * 
	 * 
	 * @param name
	 *            = username for property namCreator - if null current username
	 *            will be used
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 */
	public abstract Collection<ItemCollection> getWorkListByCreator(
			String name, int startpos, int count, String type, int sortorder);

	/**
	 * Returns a collection of workitems containing a namOwner property
	 * belonging to a specified username. The namOwner property is typical
	 * controled by the OwnerPlugin
	 * 
	 * @param name
	 *            = username for property namOwner - if null current username
	 *            will be used
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 */
	public abstract Collection<ItemCollection> getWorkListByOwner(String name,
			int startpos, int count, String type, int sortorder);

	/**
	 * Returns a collection of workitems where the current user has a
	 * writeAccess. This means the either the username or one of the userroles
	 * is contained in the $writeaccess property
	 * 
	 * 
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 */
	public abstract Collection<ItemCollection> getWorkListByWriteAccess(
			int startpos, int count, String type, int sortorder);

	public abstract Collection<ItemCollection> getWorkListByGroup(String name,
			int startpos, int count, String type, int sortorder);

	/**
	 * Returns a collection of workitems belonging to a specified $processID
	 * defined by the workflow model. The behaivor is simmilar to the method
	 * getWorkList.
	 * 
	 * @param aID
	 *            = $ProcessID for the workitems to be returned.
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 */
	public abstract Collection<ItemCollection> getWorkListByProcessID(int aid,
			int startpos, int count, String type, int sortorder);

	/**
	 * Returns a collection of workitems belonging to a specified workitem
	 * identified by the attribute $UniqueIDRef.
	 * 
	 * The behaivor of this Mehtod is simmilar to the method getWorkList.
	 * 
	 * @param aref
	 *            A unique reference to another workitem inside a database *
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 */
	public abstract Collection<ItemCollection> getWorkListByRef(String aref);

	/**
	 * Returns a collection of workitems belonging to a specified workitem
	 * identified by the attribute $UniqueIDRef.
	 * 
	 * The behaivor of this Mehtod is simmilar to the method getWorkList.
	 * 
	 * @param aref
	 *            A unique reference to another workitem inside a database *
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 */
	public abstract Collection<ItemCollection> getWorkListByRef(String aref,
			int startpos, int count, String type, int sortorder);

	/**
	 * processes a workItem. The workitem have to provide the properties
	 * '$modelversion', '$processid' and '$activityid'
	 * 
	 * The method try to load the current instance of the given workitem and
	 * compares the property $processID. If it is not equal the method throws an
	 * ProcessingErrorException.
	 * 
	 * @param workitem
	 *            - the workItem to be processed
	 * @return updated version of the processed workItem
	 * @throws InvalidItemValueException
	 *             if $modelVersion did not exists or the workItem has an
	 *             invalid status
	 * @throws AccessDeniedException
	 *             if call has no write access for the given workItem
	 * @throws ProcessingErrorException
	 *             if processing by a plugin fails
	 */
	public abstract ItemCollection processWorkItem(ItemCollection workitem)
			throws AccessDeniedException, ProcessingErrorException;

	public abstract void removeWorkItem(ItemCollection aworkitem)
			throws AccessDeniedException;

	/***************************************************************************
	 * Workflow Context
	 */
	public abstract int getLogLevel();

	/**
	 * This Method returns the modelManager Instance. The current ModelVersion
	 * is automatically updated during the Method updateProfileEntity which is
	 * called from the processWorktiem method.
	 * 
	 */
	public abstract Model getModel();

	/**
	 * This method returns an instance of the Imixs JEE ModelService used by the
	 * WorkflowManager Implementation. The method can be used to access the
	 * ModelService during a Plugin call.
	 * 
	 * @return EntityService
	 * @throws Exception
	 */
	public abstract ModelService getModelService();

	/**
	 * Obtain the java.security.Principal that identifies the caller and returns
	 * the name of this principal.
	 * 
	 * @return the user name
	 */
	public abstract String getUserName();

	/**
	 * Test if the caller has a given security role.
	 * 
	 * @param rolename
	 * @return true if user is in role
	 */
	public abstract boolean isUserInRole(String rolename);

	/**
	 * This method returns a list of user names, roles and application groups
	 * the caller belongs to.
	 * 
	 * @return
	 */
	public abstract List<String> getUserNameList();

}