/*******************************************************************************
 *  Imixs Workflow 
 *  Copyright (C) 2001, 2011 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Project: 
 *  	http://www.imixs.org
 *  	http://java.net/projects/imixs-workflow
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika - Software Developer
 *******************************************************************************/

package org.imixs.workflow.plugins;

import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;

import org.imixs.workflow.ItemCollection;
import org.imixs.workflow.Plugin;
import org.imixs.workflow.WorkflowContext;
import org.imixs.workflow.exceptions.PluginException;

/**
 * This Plugin evalueats business rules
 * 
 * @author Ralph Soika
 * @version 1.0
 * 
 */

public class RulePlugin extends AbstractPlugin {

	private static ScriptEngine engine = null;
	private static Logger logger = Logger.getLogger("org.imixs.workflow");

	/**
	 * initalizes the script engine
	 */
	@Override
	public void init(WorkflowContext actx) throws PluginException {
		super.init(actx);

		ScriptEngineManager manager = new ScriptEngineManager();

		engine = manager.getEngineByName("javascript");
	}

	/**
	 * The run method evaluates if the action is valid
	 * 
	 * <code>
	 *   isValid==true
	 * </code>
	 * 
	 * 
	 * Next the method evaluates follow up activities
	 * 
	 */
	public int run(ItemCollection adocumentContext,
			ItemCollection adocumentActivity) throws PluginException {

		if (evaluateBusinessRule(adocumentContext, adocumentActivity)) {

			Boolean isValidActivity = (Boolean) engine.get("isValid");
			if (isValidActivity != null && !isValidActivity)
				throw new PluginException(
						"BusinessRule: activity is not valid for this process instance!");

			// now test the followup activity

			Object o = engine.get("followUp");
			if (o != null) {
				// try to get double value...
				Double d = Double.valueOf(o.toString());
				Long followUpActivity = d.longValue();
				if (followUpActivity != null && followUpActivity > 0) {
					adocumentActivity.replaceItemValue("keyFollowUp", "1");
					adocumentActivity.replaceItemValue("numNextActivityID",
							followUpActivity);

				}
			}

			return Plugin.PLUGIN_OK;

		} else
			// no business rule is defined
			return Plugin.PLUGIN_OK; // nothing to do

	}

	/**
	 * nothing to do
	 */
	public void close(int status) {

	}

	/**
	 * returns true if the documentContext is valid for the current activity
	 * 
	 * @param adocumentContext
	 * @param adocumentActivity
	 * @return
	 */
	public static boolean isValid(ItemCollection documentContext,
			ItemCollection activity) {

		if (evaluateBusinessRule(documentContext, activity) == true) {
			Boolean isValidActivity = (Boolean) engine.get("isValid");
			if (isValidActivity != null)
				return isValidActivity;

		}
		// no rule defined
		return true;
	}

	/**
	 * evaluates the business rule defined by the provided activity. The method
	 * returns true if a rule evaluation was successful. The method returns
	 * false if no rule was defined.
	 * 
	 * @param adocumentContext
	 * @param adocumentActivity
	 * @return
	 * @throws PluginException
	 */
	private static boolean evaluateBusinessRule(ItemCollection documentContext,
			ItemCollection activity) throws PluginException {

		// test if a business rule is defined
		String script = activity.getItemValueString("txtBusinessRule");
		if ("".equals(script.trim()))
			return false; // nothing to do

		// initialize the script engine...
		ScriptEngineManager manager = new ScriptEngineManager();
		String sEngineType = activity
				.getItemValueString("txtBusinessRuleEngine");
		// set default engine to javascript if no engine is specified
		if ("".equals(sEngineType))
			sEngineType = "javascript";

		engine = manager.getEngineByName(sEngineType);

		// setup document data...
		Map<String, Object> itemList = documentContext.getAllItems();
		for (Map.Entry<String, Object> entry : itemList.entrySet()) {
			String key = entry.getKey();
			List value = (List) entry.getValue();
			if (value.size() > 0)
				engine.put(key.toLowerCase(), value.toArray());
		}

		logger.info("SCRIPT:" + script);
		try {
			engine.eval(script);

		} catch (ScriptException e) {
			// script not valid
			throw new PluginException("BusinessRule invalid script!", e);
		}

		return true;
	}

}
