/*******************************************************************************
 *  Imixs Workflow 
 *  Copyright (C) 2001, 2011 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Project: 
 *  	http://www.imixs.org
 *  	http://java.net/projects/imixs-workflow
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika - Software Developer
 *******************************************************************************/

package org.imixs.workflow.plugins.jee.extended;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.logging.Logger;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.Term;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.util.Version;
import org.imixs.workflow.ItemCollection;
import org.imixs.workflow.Plugin;
import org.imixs.workflow.WorkflowContext;
import org.imixs.workflow.exceptions.PluginException;
import org.imixs.workflow.jee.ejb.EntityService;
import org.imixs.workflow.jee.ejb.WorkflowService;
import org.imixs.workflow.plugins.AbstractPlugin;

/**
 * This Plugin add workitems to a lucene search index. The Plugin provides a set
 * of static methods which can be used also outside the workflowManager to index
 * single workitems or collections of workitems.
 * 
 * With the method addWorkitem() a ItemCollection can be added to a lucene
 * search index. The Plugin reads the property file 'imixs-search.properties'
 * from the current classpath to determine the configuration.
 * 
 * <ul>
 * <li>The property "IndexDir" defines the location of the lucene index
 * <li>The property "FulltextFieldList" lists all fields which should be
 * searchable after a workitem was updated
 * <li>The property "IndexFieldList" lists all fields which should be indexed as
 * keywords by the lucene search engine
 * 
 * If the plugin is used as worflow pugin in the model definition, the plugin
 * should be run last to be sure that newly computed values like the worklfow
 * status or the wokflowSummary are indexed correctly
 * 
 * @author rsoika
 * 
 */
public class LucenePlugin extends AbstractPlugin {
	// Properties properties = null;
	IndexWriter writer = null;
	static List<String> searchFieldList = null;
	static List<String> indexFieldListAnalyse = null;
	static List<String> indexFieldListNoAnalyse = null;
	private static Logger logger = Logger.getLogger("org.imixs.workflow");

	@Override
	public void init(WorkflowContext actx) throws PluginException {
		super.init(actx);
	}

	/**
	 * This method adds the current workitem to the search index by calling the
	 * method addWorkitem. The method computes temporarily the field $processid
	 * based on the numnextprocessid from teh activty entity. This will ensure
	 * that the workitem is indexed correctly on the $processid the workitem
	 * will hold after the process step is completed.
	 * 
	 * If and how the workitem will be added to the search index is fully
	 * controlled by the method addWorkitem.
	 */
	public int run(ItemCollection documentContext, ItemCollection activity)
			throws PluginException {

		// compute next $processid to be added correctly into the search index
		int nextProcessID = activity.getItemValueInteger("numnextprocessid");
		int currentProcessID = documentContext
				.getItemValueInteger("$processid");
		// temporarily replace the $processid
		try {
			documentContext.replaceItemValue("$processid", nextProcessID);

			// add the current Worktitem to the search index
			addWorkitem(documentContext);
			// restore $processid
			documentContext.replaceItemValue("$processid", currentProcessID);

		} catch (Exception e) {
			throw new PluginException(e.getMessage());
		}
		return Plugin.PLUGIN_OK;
	}

	public void close(int status) throws PluginException {

	}

	/**
	 * This method adds a single workitem into the search index. The adds the
	 * workitem into a empty Collection and calls teh method addWorklist.
	 * 
	 * @param documentContext
	 * @return
	 * @throws Exception
	 */
	public static boolean addWorkitem(ItemCollection documentContext)
			throws Exception {
		List<ItemCollection> workitems = new ArrayList<ItemCollection>();

		workitems.add(documentContext);

		addWorklist(workitems);

		return true;
	}

	/**
	 * This method adds a collection of workitems into the search index. The
	 * method loads the imixs-search.properties file from the classpath. If no
	 * properties are defined the method terminates. For each workitem the
	 * method test if it did match the conditions to be added into the search
	 * index. If the workitem did not match the conditions the workitem will be
	 * removed from the index.
	 * 
	 * 
	 * @param worklist
	 * @return
	 * @throws Exception
	 */
	public static boolean addWorklist(Collection<ItemCollection> worklist)
			throws Exception {
		// try loading imixs-search properties
		Properties prop = loadProperties();
		if (prop.isEmpty())
			return false;

		IndexWriter awriter = createIndexWriter(prop);

		// add workitem to search index....
		try {

			for (ItemCollection workitem : worklist) {
				// create term
				Term term = new Term("$uniqueid",
						workitem.getItemValueString("$uniqueid"));
				// test if document should be indexed or not
				if (matchConditions(prop, workitem))
					awriter.updateDocument(term, createDocument(prop, workitem));
				else
					awriter.deleteDocuments(term);

			}
		} catch (Exception luceneEx) {
			// close writer!
			logger.warning(" Lucene Exception : " + luceneEx.getMessage());
			throw luceneEx;
		} finally {
			logger.fine(" close writer");
			awriter.optimize();
			awriter.close();
		}

		logger.fine(" update worklist successfull");
		return true;
	}

	/**
	 * test if the workitem matches the conditions to be added into the search
	 * index. The Property keys MatchingType and MatchingProcessID can provide
	 * regular expressions
	 * 
	 * @param aworktiem
	 * @return
	 */
	public static boolean matchConditions(Properties prop,
			ItemCollection aworktiem) {

		String typePattern = prop.getProperty("MatchingType");
		String processIDPattern = prop.getProperty("MatchingProcessID");

		String type = aworktiem.getItemValueString("Type");
		String sPid = aworktiem.getItemValueInteger("$Processid") + "";

		// test type pattern
		if (typePattern != null && !"".equals(typePattern)
				&& !type.matches(typePattern))
			return false;

		// test $processid pattern
		if (processIDPattern != null && !"".equals(processIDPattern)
				&& !sPid.matches(processIDPattern))
			return false;

		return true;
	}

	/**
	 * Returns a ItemCollection List matching the provided search term. The
	 * provided search team will we extended with a users roles to test the read
	 * access level of each workitem matching the search term. The usernames and
	 * user roles will be search lowercase!
	 * 
	 * @param sSearchTerm
	 * @param workflowService
	 * @return collection of search result
	 * @throws Exception
	 */
	public static List<ItemCollection> search(String sSearchTerm,
			WorkflowService workflowService) throws Exception {

		ArrayList<ItemCollection> workitems = new ArrayList<ItemCollection>();

		// test if searchtem is provided
		if (sSearchTerm == null || "".equals(sSearchTerm))
			return workitems;

		long ltime = System.currentTimeMillis();
		Properties prop = loadProperties();
		if (prop.isEmpty())
			return workitems;

		String sIndexDir = prop.get("IndexDir") + "";
		Directory directory;
		try {
			directory = FSDirectory.open(new File(sIndexDir));

			IndexSearcher searcher = new IndexSearcher(directory, true);

			// Analyzer analyzer = new StandardAnalyzer(Version.LUCENE_31);
			Analyzer analyzer = new org.apache.lucene.analysis.KeywordAnalyzer();
			QueryParser parser = new QueryParser(Version.LUCENE_31, "content",
					analyzer);

			// extend the Search Term
			if (!workflowService
					.isUserInRole(EntityService.ACCESSLEVEL_MANAGERACCESS)) {
				// get user names list
				List<String> userNameList = workflowService.getUserNameList();
				// create search term
				String sAccessTerm = "($readaccess:ANONYMOUS";
				for (String aRole : userNameList) {
					if (!"".equals(aRole))
						sAccessTerm += " $readaccess:\"" + aRole + "\"";
				}
				sAccessTerm += ") AND ";
				sSearchTerm = sAccessTerm + sSearchTerm;
			}
			logger.info("  lucene search:" + sSearchTerm);

			if (!"".equals(sSearchTerm)) {
				parser.setAllowLeadingWildcard(true);
				// parser.setDefaultOperator(Operator.AND);
				TopDocs topDocs = searcher.search(parser.parse(sSearchTerm),
						1000);

				logger.fine("  total hits=" + topDocs.totalHits);

				// Get an array of references to matched documents
				ScoreDoc[] scoreDosArray = topDocs.scoreDocs;
				for (ScoreDoc scoredoc : scoreDosArray) {
					// Retrieve the matched document and show relevant details
					Document doc = searcher.doc(scoredoc.doc);

					String sID = doc.get("$uniqueid");
					logger.fine("  lucene $uniqueid=" + sID);
					ItemCollection itemCol = workflowService.getEntityService()
							.load(sID);
					if (itemCol != null) {
						workitems.add(itemCol);
					}
				}

			}

			searcher.close();
			directory.close();

			logger.info(" lucene serach: "
					+ (System.currentTimeMillis() - ltime) + " ms");
		} catch (Exception e) {
			logger.warning("  lucene error!");
			e.printStackTrace();
		}

		return workitems;
	}

	/**
	 * This method creates a lucene document based on a ItemCollection. The
	 * Method creates for each field specified in the FieldList a separate index
	 * field for the lucene document.
	 * 
	 * The property 'AnalyzeIndexFields' defines if a indexfield value should by
	 * analyzed by the Lucene Analyzer (default=false)
	 * 
	 * @param aworkitem
	 * @return
	 */
	public static Document createDocument(Properties prop,
			ItemCollection aworkitem) {
		String sValue = null;
		Document doc = new Document();

		// combine all search fields from the search field list into one field
		// ('content')
		// for the lucene document
		String sContent = "";
		for (String aFieldname : searchFieldList) {
			sValue = "";
			// check value list - skip empty fields
			List vValues = aworkitem.getItemValue(aFieldname);
			if (vValues.size() == 0)
				continue;
			// get all values of a value list field
			for (Object o : vValues) {
				if (o == null)
					// skip null values
					continue;

				if (o instanceof Calendar || o instanceof Date) {
					SimpleDateFormat dateformat = new SimpleDateFormat(
							"yyyyMMddHHmm");
					// convert calendar to string
					String sDateValue;
					if (o instanceof Calendar)
						sDateValue = dateformat
								.format(((Calendar) o).getTime());
					else
						sDateValue = dateformat.format((Date) o);
					sValue += sDateValue + ",";

				} else
					// simple string representation
					sValue += o.toString() + ",";
			}
			if (sValue != null) {
				logger.fine("  add SearchField: " + aFieldname + " = " + sValue);
				sContent += sValue + ",";
			}
		}
		logger.fine("  content = " + sContent);
		doc.add(new Field("content", sContent, Field.Store.NO,
				Field.Index.ANALYZED));

		// add each field from the indexFieldList into the lucene document
		for (String aFieldname : indexFieldListAnalyse) {
			addFieldValue(doc, aworkitem, aFieldname, true);
		}

		for (String aFieldname : indexFieldListNoAnalyse) {
			addFieldValue(doc, aworkitem, aFieldname, false);
		}

		// add default value $uniqueid
		doc.add(new Field("$uniqueid", aworkitem
				.getItemValueString("$uniqueid"), Field.Store.YES,
				Field.Index.NOT_ANALYZED));

		// add default values $readAccess
		List<String> vReadAccess = aworkitem.getItemValue("$readAccess");
		if (vReadAccess.size() == 0
				|| (vReadAccess.size() == 1 && "".equals(vReadAccess.get(0)
						.toString()))) {
			sValue = "ANONYMOUS";
			doc.add(new Field("$readaccess", sValue, Field.Store.NO,
					Field.Index.NOT_ANALYZED_NO_NORMS));
		} else {
			sValue = "";
			// add each role / username as a single field value
			for (String sReader : vReadAccess)
				doc.add(new Field("$readaccess", sReader, Field.Store.NO,
						Field.Index.NOT_ANALYZED_NO_NORMS));

		}
		return doc;
	}

	/**
	 * adds a field value into a lucene document
	 * 
	 * @param doc
	 *            an existing lucene document
	 * @param aworkitem
	 *            the workitem containg the values
	 * @param aFieldname
	 *            the Fieldname inside the workitem
	 * @param analyzeValue
	 *            indicates if the value should be parsed by the analyzer
	 */
	private static void addFieldValue(Document doc, ItemCollection aworkitem,
			String aFieldname, boolean analyzeValue) {
		String sValue = null;
		List vValues = aworkitem.getItemValue(aFieldname);
		if (vValues.size() == 0)
			return;
		if (vValues.get(0) == null)
			return;

		for (Object singleValue : vValues) {

			// Object o = vValues.firstElement();
			if (singleValue instanceof Calendar || singleValue instanceof Date) {
				SimpleDateFormat dateformat = new SimpleDateFormat(
						"yyyyMMddHHmm");

				// convert calendar to string
				String sDateValue;
				if (singleValue instanceof Calendar)
					sDateValue = dateformat.format(((Calendar) singleValue)
							.getTime());
				else
					sDateValue = dateformat.format((Date) singleValue);
				sValue = sDateValue;

			} else
				// simple string representation
				sValue = singleValue.toString();

			logger.fine("  add IndexField (analyse=" + analyzeValue + "): "
					+ aFieldname + " = " + sValue);
			if (analyzeValue)
				doc.add(new Field(aFieldname, sValue, Field.Store.NO,
						Field.Index.ANALYZED));
			else
				// do not nalyse content of index fields!
				doc.add(new Field(aFieldname, sValue, Field.Store.NO,
						Field.Index.NOT_ANALYZED));

		}

	}

	/**
	 * This method creates a new instance of a lucene IndexWriter. The timeout
	 * to wait for a write lock is set to 10 seconds.
	 * 
	 * @return
	 * @throws Exception
	 */
	public static IndexWriter createIndexWriter(Properties prop)
			throws Exception {

		String sIndexDir = prop.get("IndexDir") + "";
		String sFulltextFieldList = prop.get("FulltextFieldList") + "";
		String sIndexFieldListAnalyse = prop.get("IndexFieldListAnalyze") + "";
		String sIndexFieldListNoAnalyse = prop.get("IndexFieldListNoAnalyze")
				+ "";

		logger.fine("IndexDir:" + sIndexDir);
		logger.fine("FulltextFieldList:" + sFulltextFieldList);
		logger.fine("IndexFieldListAnalyse:" + sIndexFieldListAnalyse);
		logger.fine("IndexFieldListNoAnalyse:" + sIndexFieldListNoAnalyse);
		// compute search field list
		StringTokenizer st = new StringTokenizer(sFulltextFieldList, ",");
		searchFieldList = new ArrayList<String>();
		while (st.hasMoreElements()) {
			String sName = st.nextToken().toLowerCase();
			// do not add internal fields
			if (!"$uniqueid".equals(sName) && !"$readaccess".equals(sName))
				searchFieldList.add(sName);
		}

		// compute Index field list (Analyze)
		st = new StringTokenizer(sIndexFieldListAnalyse, ",");
		indexFieldListAnalyse = new ArrayList<String>();
		while (st.hasMoreElements()) {
			String sName = st.nextToken().toLowerCase();
			// do not add internal fields
			if (!"$uniqueid".equals(sName) && !"$readaccess".equals(sName))
				indexFieldListAnalyse.add(sName);
		}

		// compute Index field list (Analyze)
		st = new StringTokenizer(sIndexFieldListNoAnalyse, ",");
		indexFieldListNoAnalyse = new ArrayList<String>();
		while (st.hasMoreElements()) {
			String sName = st.nextToken().toLowerCase();
			// do not add internal fields
			if (!"$uniqueid".equals(sName) && !"$readaccess".equals(sName))
				indexFieldListNoAnalyse.add(sName);
		}

		// initialize lucene index writer
		// Directory indexDir = new SimpleFSDirectory(new File(sIndexDir));
		Directory indexDir = FSDirectory.open(new File(sIndexDir));

		// KeywordAnalyzer StandardAnalyzer
		Analyzer analyzer = new StandardAnalyzer(Version.LUCENE_31);
		IndexWriterConfig iwc = new IndexWriterConfig(Version.LUCENE_31,
				analyzer);
		// set the WriteLockTimeout to wait for a write lock (in milliseconds)
		// for this instance.
		// 10 seconds!
		iwc.setWriteLockTimeout(10000);

		// there is no need to unlock the index if we set the timeout to 10
		// seconds
		// if (IndexWriter.isLocked(indexDir)) {
		// logger.warning("Lucene IndexWriter was locked! - try to unlock....");
		// IndexWriter.unlock(indexDir);
		// }
		return new IndexWriter(indexDir, iwc);

	}

	public static IndexSearcher createIndexSearcher() throws Exception {
		Directory directory;

		Properties properties = LucenePlugin.loadProperties();
		String sIndexDir = properties.get("IndexDir") + "";

		directory = FSDirectory.open(new File(sIndexDir));

		return new IndexSearcher(directory, true);
	}

	/**
	 * loads a imixs-search.property file
	 * 
	 * @return
	 * @throws Exception
	 */
	public static Properties loadProperties() throws Exception {
		// try loading imixs-search properties
		Properties prop = new Properties();
		try {
			prop.load(Thread.currentThread().getContextClassLoader()
					.getResource("imixs-search.properties").openStream());
		} catch (Exception ep) {
			// no properties found
		}
		return prop;
	}

}
