/*******************************************************************************
 *  Imixs Workflow 
 *  Copyright (C) 2001, 2011 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Project: 
 *  	http://www.imixs.org
 *  	http://java.net/projects/imixs-workflow
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika - Software Developer
 *******************************************************************************/

package org.imixs.workflow.jee.faces;

import java.util.Collection;
import java.util.List;
import java.util.Vector;

import javax.ejb.EJB;

import org.imixs.workflow.ItemCollection;

/**
 * This BLOBWorkitemController is used to store large objects into a single
 * ItemCollection mapped to a EntityBean. The BLOBWorkitemController supports
 * also the management of file attachment inside an ItemCollection. So the
 * BLOBWorkitemController can be used to save large data objects or files into a
 * Workitem (called BlobWorkitems). The BlobWorkitem is always bounded to a
 * parent workitem by its referrer id ($uniqueidRef). An application can
 * implement a lazy loading for BLOBWorkitems. The read- and write access
 * settings of BLOBWorkitems are always synchronized to the settings of the
 * parent workitem. Before the BlobWorkitem can be accessed the workitem needs
 * to be loaded by the load() method. The Data can be accessed by the embedded
 * Itemcollection through the method getWorkitem(). The BlobWorkitem can be
 * saved by calling the save() method. Both - the load() and the save() method
 * expect the Parent ItemCollection where the BlobWorkitem should be bound to.
 * 
 * @version 0.0.1
 * @author rsoika
 * 
 */
public class BLOBWorkitemController {

	@EJB
	org.imixs.workflow.jee.ejb.EntityService entityService;

	private ItemCollection blobWorkitem = null;

	public BLOBWorkitemController() {
		super();

		// set the workitemAdapter
		clear();
	}

	/**
	 * update the workitemAdapter and the blobworkitem reference
	 * 
	 * @param aItemcol
	 */
	public void setWorkitem(ItemCollection aItemcol) {
		blobWorkitem = aItemcol;

	}

	/**
	 * returns the ItemCollection for the curren BlobWorkitem object.
	 * 
	 * @return
	 */
	public ItemCollection getWorkitem() {
		return blobWorkitem;
	}

	/**
	 * Returns a list of file names attached to the current BlobWorkitem. File
	 * Attachments can be added using the method addFile().
	 * 
	 * @return
	 */
	public String[] getFiles() {
		List<String> files= blobWorkitem.getFileNames();
		return (String[]) files.toArray();

	}

	/**
	 * Returns a Vector of file names attached to the current BlobWorkitem.
	 * 
	 * @return
	 */
	public Vector<String> getFileList() {
		Vector<String> vFileList = new Vector<String>();
		String[] sFileList = getFiles();
		for (String aName : sFileList)
			vFileList.add(aName);

		return vFileList;
	}

	/**
	 * Removes the connection to the parend workitem and clear the
	 * itemCollection
	 */
	public void clear() {
		blobWorkitem = new ItemCollection();
		try {
			blobWorkitem.replaceItemValue("type", "workitemlob");
		} catch (Exception e) {
			e.printStackTrace();
		}
		setWorkitem(blobWorkitem);
	}

	/**
	 * This method saves the current BlobWorkitem. Therefore the method copies
	 * the read- and write access list from the given parent workitem into the
	 * BlobWorkitem before save.
	 * 
	 * So this method should be called after a WorkflowProcessing step to update
	 * the read- and write access identically to the parentWorkitem
	 * <p>
	 * The method did not save the blobWorkitem if the parent workitem has no
	 * $unqiueID!
	 * <p>
	 * 
	 * @throws Exception
	 */
	public void save(ItemCollection parentWorkitem) throws Exception {

		if (blobWorkitem != null && parentWorkitem != null) {

			// verifiy if a uniqueid is still defined. If not return without
			// saving!
			if ("".equals(parentWorkitem.getItemValueString("$uniqueID")))
				return;

			// Update Read and write access list from parent workitem
			List vAccess = parentWorkitem.getItemValue("$ReadAccess");
			blobWorkitem.replaceItemValue("$ReadAccess", vAccess);

			vAccess = parentWorkitem.getItemValue("$WriteAccess");
			blobWorkitem.replaceItemValue("$WriteAccess", vAccess);

			blobWorkitem.replaceItemValue("$uniqueidRef",
					parentWorkitem.getItemValueString("$uniqueID"));
			blobWorkitem.replaceItemValue("type", "workitemlob");
			// Update BlobWorkitem
			blobWorkitem = entityService.save(blobWorkitem);
			// update adapter
			setWorkitem(blobWorkitem);

		}
	}

	/**
	 * This method adds a single file to the ItemCollection. files will be
	 * stored into the property $file.
	 * 
	 * @param data
	 *            - byte array with file data
	 * @param fileName
	 *            - name of the file attachment
	 * @param contentType
	 *            - the contenttype (e.g. 'Text/HTML')
	 * @throws Exception
	 */
	public void addFile(byte[] data, String fileName, String contentType)
			throws Exception {

		blobWorkitem.addFile(data, fileName, contentType);

	}

	/**
	 * This method removes a single file attachment from the BlobWorkitem
	 * 
	 * @throws Exception
	 */
	public void removeFile(String aFilename) throws Exception {
		blobWorkitem.removeFile(aFilename);
	}

	/**
	 * Loads the BlobWorkitem of a given parent Workitem. The BlobWorkitem is
	 * identified by the $unqiueidRef. If no BlobWorkitem still exists the
	 * method creates a new empty BlobWorkitem which can be saved later.
	 * 
	 * @param itemCol
	 *            - parent workitem where the BlobWorkitem will be attached to
	 * @throws Exception
	 */
	public void load(ItemCollection itemCol) throws Exception {
		ItemCollection ablobWorkitem = null;

		// test if the blobworkitem could be loaded
		String sUniqueID = itemCol.getItemValueString("$uniqueid");

		if (!"".equals(sUniqueID)) {
			// search entity...
			String sQuery = " SELECT lobitem FROM Entity as lobitem"
					+ " join lobitem.textItems as t2"
					+ " WHERE lobitem.type = 'workitemlob'"
					+ " AND t2.itemName = '$uniqueidref'"
					+ " AND t2.itemValue = '" + sUniqueID + "'";

			Collection<ItemCollection> itemcol = entityService.findAllEntities(
					sQuery, 0, 1);
			// if workitem found then update blobWorkitem...
			if (itemcol != null && itemcol.size() > 0) {
				ablobWorkitem = itemcol.iterator().next();
				// update workitem
				setWorkitem(ablobWorkitem);
			}

		}
		// if no blobWorkitem was found, create a empty itemCollection..
		if (ablobWorkitem == null) {
			clear();
		}

	}

	/*
	 * public Map getItem() throws Exception { return blobWorkitem.getItem(); }
	 * 
	 * 
	 * public Map getItemList() throws Exception { return
	 * blobWorkitem.getItemList(); }
	 * 
	 * 
	 * public Map getItemListArray() throws Exception { return
	 * blobWorkitem.getItemListArray(); }
	 */

}
