/*******************************************************************************
 *  Imixs Workflow 
 *  Copyright (C) 2001, 2011 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Project: 
 *  	http://www.imixs.org
 *  	http://java.net/projects/imixs-workflow
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika - Software Developer
 *******************************************************************************/

package org.imixs.workflow.jaxrs.plugins;

import java.io.ByteArrayOutputStream;
import java.io.FileOutputStream;
import java.io.StringWriter;
import java.util.Collection;
import java.util.List;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.logging.Logger;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;

import org.imixs.workflow.ItemCollection;
import org.imixs.workflow.engine.WorkflowService;
import org.imixs.workflow.engine.plugins.AbstractPlugin;
import org.imixs.workflow.exceptions.PluginException;
import org.imixs.workflow.exceptions.QueryException;
import org.imixs.workflow.xml.DocumentCollection;
import org.imixs.workflow.xml.XMLItemCollectionAdapter;

/**
 * This plugin handles the creation of a Imixs Reoprt. The coresponding
 * activityEntity provide the following properties:
 * <p>
 * <ul>
 * <li>txtReportName=Name of the Report to be processed
 * <li>txtReportFilePath= filename or filepath the result will be saved
 * <li>txtReportTarget = where the result is saved (0=workitem, 1=blobWorkitem,
 * 2= disk)
 * 
 * 
 * 
 * 
 * 
 * @author Ralph Soika
 * @version 1.0
 */

public class ReportPlugin extends AbstractPlugin {

	public static final String INVALID_CONTEXT = "INVALID_CONTEXT";
	public static final String REPORT_UNDEFINED = "REPORT_UNDEFINED";
	public static final String INVALID_REPORT_DEFINITION = "INVALID_REPORT_DEFINITION";

	private String reportName = null;
	private String reportFilePath = null;
	private String reportTarget = null;
	private String sEQL;
	private String sXSL;
	private String sContentType;
	private String sEncoding;
	private ItemCollection blobWorkitem = null;

	private static Logger logger = Logger.getLogger(ReportPlugin.class.getName());

	

	/**
	 * creates report defined by the txtReportName.
	 * <p>
	 * The method runs the EQL Statement defined in the corresponding Report
	 * definition to compute the collection of entities to be processed by a
	 * template.
	 * <p>
	 * As the current Workitem can not be identically included by the resultset
	 * of the EQL Statement (because the documentContext is yet not saved) the
	 * method tests if the resultset includes the current workitem. In this case
	 * the 'old' result will be replaced with the new currently processed
	 * workitem.
	 * 
	 * The xml generated by the Jax-B Marshaller will be modified afterwards to
	 * avoid escaped content of fields with a name starting with 'xml' or
	 * 'html'.
	 * 
	 * 
	 */
	@SuppressWarnings("unchecked")
	public ItemCollection run(ItemCollection adocumentContext, ItemCollection adocumentActivity) throws PluginException {

		reportName = adocumentActivity.getItemValueString("txtReportName");
		reportFilePath = adocumentActivity.getItemValueString("txtReportFilePath");
		if ("".equals(reportFilePath))
			reportFilePath = reportName;

		// replace dynamic field values
		reportFilePath = this.replaceDynamicValues(reportFilePath, adocumentContext);

		reportTarget = adocumentActivity.getItemValueString("txtReportTarget");

		if ("".equals(reportName))
			return adocumentContext;

		// autocomplete file extention if missing
		if (!reportName.endsWith(".ixr"))
			reportName = reportName + ".ixr";

		ItemCollection itemCol = getWorkflowService().getReportService().getReport(reportName);
		if (itemCol == null) {
			// report undefined
			throw new PluginException(ReportPlugin.class.getSimpleName(), REPORT_UNDEFINED,
					"Report '" + reportName + " is undefined", new Object[] { reportName });

		}

		// get Query and output format
		sEQL = itemCol.getItemValueString("txtquery");

		// compute dynamical params
		String sParamString = adocumentActivity.getItemValueString("txtReportParams");
		// replace field values...
		sParamString = this.replaceDynamicValues(sParamString, adocumentContext);

		// compute jpql statement
		sEQL = computeEQLParams(sEQL, sParamString);
		logger.info("ReportPlugin JPQL=" + sEQL);
		sXSL = itemCol.getItemValueString("txtXSL").trim();
		// if no XSL is provided return
		if ("".equals(sXSL))
			return adocumentContext;

		sContentType = itemCol.getItemValueString("txtcontenttype");
		if ("".equals(sContentType))
			sContentType = "text/html";

		List<String> vAttributList = (List<String>) itemCol.getItemValue("txtAttributeList");
		if (vAttributList.size() == 0) {
			vAttributList = null;
		}

		sEncoding = itemCol.getItemValueString("txtencoding");
		// no encoding defined so take a default encoding
		// (UTF-8)
		if ("".equals(sEncoding))
			sEncoding = "UTF-8";

		// query result ....
		Collection<ItemCollection> col;
		try {
			col = getWorkflowService().getDocumentService().find(sEQL, 0, -1);
		} catch (QueryException e1) {
			throw new PluginException(e1.getErrorContext(), e1.getErrorCode(), e1.getMessage(), e1);
		}
		// now add the current Workitem into the collection if an older
		// version is
		// included in the result
		String sUnqiueID = adocumentContext.getItemValueString("$uniqueID");
		if (!"".equals(sUnqiueID)) {
			Collection<ItemCollection> colNew = new Vector<ItemCollection>();
			for (ItemCollection aitemCol : col) {
				if (sUnqiueID.equals(aitemCol.getItemValueString("$uniqueid"))) {

					ItemCollection itemTemp = new ItemCollection(adocumentContext.getAllItems());
					itemTemp.replaceItemValue("$temp", "true");
					colNew.add(itemTemp);
					logger.info(" ReportPlugin - relaced deprecated workitem from collection");
				} else
					colNew.add(aitemCol);
			}
			col = colNew;
		} else {
			// seems that we are currently processing a new workitem - so
			// include it into the resultset
			ItemCollection itemTemp = new ItemCollection(adocumentContext.getAllItems());
			itemTemp.replaceItemValue("$temp", "true");
			col.add(itemTemp);
			logger.info(" ReportPlugin - add current workitem into collection");

		}
		try {
			// Transform XML per XSL and generate output
			DocumentCollection xmlCol = XMLItemCollectionAdapter.putCollection(col, vAttributList);

			StringWriter xmlWriter = new StringWriter();

			JAXBContext context = JAXBContext.newInstance(DocumentCollection.class);

			Marshaller m = context.createMarshaller();
			m.setProperty("jaxb.encoding", sEncoding);
			m.marshal(xmlCol, xmlWriter);

			// test if FOP Tranformation
			ByteArrayOutputStream outputStream = new ByteArrayOutputStream();

			// this is the code where we unescape content if field name start
			// with xml or html
			String xmlContent = xmlWriter.toString();

			// unescape xml and html fields which were escaped by the Marshaller
			String xmlContentExtended = unescapeXMLContent(xmlContent, vAttributList, col);

			// String path = "~/input.xml";
			// Files.write(Paths.get(path), xmlContentExtended.getBytes(),
			// StandardOpenOption.CREATE);

			try {
				if ("application/pdf".equals(sContentType.toLowerCase()))
					org.imixs.workflow.jaxrs.ReportRestService.fopTranformation(xmlContentExtended, sXSL, sEncoding,
							outputStream);
				else
					org.imixs.workflow.jaxrs.ReportRestService.xslTranformation(xmlContentExtended, sXSL, sEncoding,
							outputStream);
			} finally {
				outputStream.close();
			}

			// write to workitem
			if ("0".equals(reportTarget)) {

				adocumentContext.addFile(outputStream.toByteArray(), reportFilePath, sContentType);
			}
			// write to blob
			if ("1".equals(reportTarget)) {
				loadBlobWorkItem(adocumentContext);

				blobWorkitem.addFile(outputStream.toByteArray(), reportFilePath, sContentType);
				saveBlobWorkitem(adocumentContext);
				// add the file name (with empty data) into the
				// parentWorkitem.
				byte[] empty = { 0 };
				adocumentContext.addFile(empty, reportFilePath, "");
			}
			// write to filesystem
			if ("2".equals(reportTarget)) {
				FileOutputStream fos = null;
				try {
					fos = new FileOutputStream(reportFilePath);
					fos.write(outputStream.toByteArray());
					fos.flush();
				} finally {
					if (fos != null) {
						fos.close();
					}
				}
			}

			return adocumentContext;
		} catch (Exception e) {
			// report undefined
			throw new PluginException(ReportPlugin.class.getSimpleName(), INVALID_REPORT_DEFINITION,
					"Unable to process report '" + reportName + "' ", new Object[] { reportName });
		}
	}

	/**
	 * The method replaces the value tags of xml and html elements (starting
	 * with xml and html)
	 * 
	 * 
	 * Example
	 * 
	 * <code>
	    <item><name>htmlanswer</name>
	          <value xsi:type="xs:string">&lt;p&gt;Some conent&lt;/p&gt;</value>
	    </item>
	 * 
	 * </code>
	 * 
	 * We need to iterate over the EntityCollection to replace each entiy tag
	 * with the corresponding original values
	 * 
	 * @param aContent
	 *            - xml conent
	 * @param vAttributList
	 *            - list of items to replace content in unescaped format
	 * @param xmlCol
	 *            - xml EntityCollection containing the original values
	 */
	private String unescapeXMLContent(String aContent, List<String> vAttributList, Collection<ItemCollection> col) {

		int entityPos = aContent.indexOf("<entity>");
		// iterate over all entities ...
		for (ItemCollection entity : col) {

			for (String fieldname : vAttributList) {

				if (fieldname.toLowerCase().startsWith("html") || fieldname.toLowerCase().startsWith("xml")) {

					// find <name>field</name>
					String tag = "<name>" + fieldname + "</name>";
					String sOriginValue = entity.getItemValueString(fieldname);
					// process only if not an empty value
					if (!sOriginValue.isEmpty()) {
						int iPos = aContent.indexOf(tag, entityPos);
						if (iPos > -1) {
							// find value start pos and end pos
							int start = aContent.indexOf('>', iPos + tag.length());
							if (start > -1) {
								// if empty tag it ends with /> instead of
								// </value>. But we skipt empty values before.
								// So we need not to care about this
								int end = aContent.indexOf("</value>", start);

								if (end > -1) {
									// replace conent with origiaal value...

									aContent = aContent.substring(0, start + 1) + sOriginValue
											+ aContent.substring(end);
								}

							}
						}
					}
				}
			}

			// now we need to forward to the next <entity> element in the xml
			// structure...
			entityPos = aContent.indexOf("<entity>", entityPos + 1);
		}

		return aContent;
	}

	public void close(int status) throws PluginException {

	}

	/**
	 * This method parses the query Params of a Request URL and adds params to a
	 * given EQL Query.
	 * 
	 * The Query params are provided in the attribute txtReportParams from the
	 * Activity Entity in the following format<br>
	 * 
	 * <code>
	      param1=xxx&param2=xxx&param3=xxx
	 * </code>
	 * 
	 * @param uriInfo
	 * @return
	 */
	private String computeEQLParams(String aQuery, String sParamString) {

		// cut prafix ? or & if available
		if (sParamString.startsWith("?") || sParamString.startsWith("&"))
			sParamString = sParamString.substring(1);

		// split params
		StringTokenizer tokenizer = new StringTokenizer(sParamString, "&");

		while (tokenizer.hasMoreTokens()) {

			String aToken = tokenizer.nextToken();
			// if no '=' contained - continue...
			if (aToken.indexOf('=') == -1)
				continue;

			String sKeyName = aToken.substring(0, aToken.indexOf('='));
			String sParamValue = aToken.substring(aToken.indexOf('=') + 1);
			// test if key is contained in query
			if (aQuery.indexOf("?" + sKeyName) > -1)
				aQuery = aQuery.replace("?" + sKeyName, sParamValue);

		}
		return aQuery;
	}

	/**
	 * Loads the BlobWorkitem of a given parent Workitem. The BlobWorkitem is
	 * identified by the $unqiueidRef. If no BlobWorkitem still exists the
	 * method creates a new empty BlobWorkitem which can be saved later.
	 * 
	 * @param itemCol
	 *            - parent workitem where the BlobWorkitem will be attached to
	 * @throws Exception
	 */
	private void loadBlobWorkItem(ItemCollection itemCol) throws Exception {

		String sUniqueID = itemCol.getItemValueString("$uniqueid");

		// search entity...
		String searchTerm = "(type:\"workitemlob\" AND $uniqueidref:\"" + sUniqueID + "\")";
		Collection<ItemCollection> itemcol = getWorkflowService().getDocumentService().find(searchTerm, 0, 1);
		if (itemcol != null && itemcol.size() > 0) {

			blobWorkitem = itemcol.iterator().next();
		} else {
			blobWorkitem = new ItemCollection();
			blobWorkitem.replaceItemValue("$uniqueidRef", sUniqueID);
			blobWorkitem.replaceItemValue("type", "workitemlob");
		}

	}

	/**
	 * This method saves the current BlobWorkitem. Therefore the method copies
	 * the read- and write access list from the given parent workitem into the
	 * BlobWorkitem before save.
	 * 
	 * So this method should be called after a WorkflowProcessing step to update
	 * the read- and write access identically to the parentWorkitem
	 * 
	 * @throws Exception
	 */
	private void saveBlobWorkitem(ItemCollection parentWorkitem) throws Exception {

		if (blobWorkitem != null && parentWorkitem != null) {

			// Update Read and write access list from parent workitem
			List<?> vAccess = parentWorkitem.getItemValue("$ReadAccess");
			blobWorkitem.replaceItemValue("$ReadAccess", vAccess);

			vAccess = parentWorkitem.getItemValue("$WriteAccess");
			blobWorkitem.replaceItemValue("$WriteAccess", vAccess);

			blobWorkitem.replaceItemValue("$uniqueidRef", parentWorkitem.getItemValueString("$uniqueID"));
			blobWorkitem.replaceItemValue("type", "workitemlob");
			// Update BlobWorkitem
			// blobWorkitem = entityService.save(blobWorkitem);
			// new transaction....
			blobWorkitem = this.getWorkflowService().getSessionContext().getBusinessObject(WorkflowService.class).getDocumentService()
					.saveByNewTransaction(blobWorkitem);

		}
	}

}