/*******************************************************************************
 *  Imixs IX Workflow Technology
 *  Copyright (C) 2001, 2008 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika
 *******************************************************************************/
package org.imixs.workflow.jee.ejb;

import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.annotation.Resource;
import javax.annotation.security.DeclareRoles;
import javax.annotation.security.RolesAllowed;
import javax.ejb.Local;
import javax.ejb.Remote;
import javax.ejb.SessionContext;
import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.imixs.workflow.ItemCollection;
import org.imixs.workflow.exceptions.AccessDeniedException;
import org.imixs.workflow.exceptions.InvalidWorkitemException;
import org.imixs.workflow.jee.jpa.CalendarItem;
import org.imixs.workflow.jee.jpa.DoubleItem;
import org.imixs.workflow.jee.jpa.Entity;
import org.imixs.workflow.jee.jpa.EntityIndex;
import org.imixs.workflow.jee.jpa.IntegerItem;
import org.imixs.workflow.jee.jpa.ReadAccessEntity;
import org.imixs.workflow.jee.jpa.TextItem;
import org.imixs.workflow.jee.jpa.WriteAccessEntity;

/**
 * The EntityServiceBean is the a stateless session EJB implementation of the
 * Interface org.imixs.workflow.jee.ejb.EntityService
 * <p>
 * This session bean is used to persist and find instances of the generic value
 * transfer object org.imixs.workflow.ItemCollection. The Bean holds an instance
 * of an EntityPersistenceManager for the persistence unit
 * 'org.imixs.workflow.jee.ejb' to manage the following Entity EJBs:
 * <ul>
 * <li>Entity,
 * <li>EntityIndex,
 * <li>TextIndex,
 * <li>IntegerIndex,
 * <li>DoubleIndex,
 * <li>CalendarIndex,
 * <li>ReadAccessEntity
 * <li>WriteAccessEntity
 * </ul>
 * < These Entity EJBs are used to store the attributes of a ItemCollection into
 * the connected database.
 * <p>
 * The save() method persists any instance of an ItemCollection. If a
 * ItemCollection is saved the first time the EntityServiceBean generates the
 * attribute $uniqueid which will be included in the ItemCollection returned by
 * this method. If a ItemCollection was saved before the method updates the
 * corresponding Entity Object.
 * <p>
 * The load() and findAllEntities() methods are used to read ItemCollections
 * from the database. The remove() Method deletes a saved ItemCollection from
 * the database.
 * <p>
 * All methods expect and return Instances of the object
 * org.imixs.workflow.ItemCollection which is no entity EJB. So these objects
 * are not managed by any instance of an EntityPersistenceManager.
 * <p>
 * A collection of ItemCollections can be read using the findAllEntites() method
 * using EQL syntax.
 * <p>
 * This EntityServiceBean has methods to brand out different Attributes of a
 * ItemCollection into external properties (TextIndex, IntegerIndex,
 * DoubleIndex, CalendarIndex) With these functionality a client can query a set
 * of ItemCollections later with EJB Query Language (EQL).
 * <p>
 * To define which attributes should be expended into external properties the
 * Session EJB supports the method addIndex(). This method creates an index
 * entry and ensures that every Entity saved before will be updated and future
 * calls of the save() method will care about the new index to separate
 * attributes into the Index Properties.
 * <p>
 * So each call of the save method automatically disassemble a ItemCollection
 * into predefined Index properties and stores attributes which have an Index
 * into a corresponding IndexData Object (TextIndex, IntegerIndex, DoubleIndex,
 * CalendarIndex). On the other hand the load() method reassembles the
 * ItemCollection including all attributes form the data property an any
 * external index property. The EntiyService did not allow to access a managed
 * Entity EJB directly. This simplifies the usage as a client works only with
 * ItemCollections and so there is no need to handle managed and detached entity
 * EJB instances.
 * <p>
 * Additional to the basic functionality to save and load instances of the
 * object org.imixs.workflow.ItemCollection the method also manages the read-
 * and writeAccess for each instance of an ItemCollection. Therefore the save()
 * method scans an ItemCollection for the attributes '$ReadAccess' and
 * '$WriteAccess' and saves these informations into the Entity EJBs
 * ReadAccessEntity and WriteAccessEntity controlled by each instance of the EJB
 * Entity. The EntityServiceBean implementation verifies in each call of the
 * save() load(), remove() and findAllEntities() methods if the current
 * callerPrincipal is granted to the affected entities. If an ItemCollection was
 * saved with read- or writeAccess the access to an Instance of a saved
 * ItemCollection will be protected for a callerPrincipal with missing read- or
 * writeAccess. The default Read- and WriteAccess attributes '$ReadAccess' and
 * '$WriteAccess' can be overwritten by the environment settings
 * 'READ_ACCESS_FIELDS' and 'WRITE_ACCESS_FIELDS'
 * <p>
 * Some useful links about the oneToMany Relationship
 * http://www.avaje.org/manydatatypes.html
 * http://thomas-schuett.de/2008/11/14/ordered-lists-in-jpa-do-it-yourself/
 * http:
 * //javaehrsson.blogspot.com/2005/10/ejb3-onetomany-and-orderby-set-versus.html
 * http://forums.java.net/jive/thread.jspa?threadID=30869&start=0&tstart=0
 * 
 * @see org.imixs.workflow.jee.ejb.EntityPersistenceManager
 * @author rsoika
 * @version 1.0
 * 
 */

@DeclareRoles( { "org.imixs.ACCESSLEVEL.NOACCESS",
		"org.imixs.ACCESSLEVEL.READERACCESS",
		"org.imixs.ACCESSLEVEL.AUTHORACCESS",
		"org.imixs.ACCESSLEVEL.EDITORACCESS",
		"org.imixs.ACCESSLEVEL.MANAGERACCESS" })
@RolesAllowed( { "org.imixs.ACCESSLEVEL.NOACCESS",
		"org.imixs.ACCESSLEVEL.READERACCESS",
		"org.imixs.ACCESSLEVEL.AUTHORACCESS",
		"org.imixs.ACCESSLEVEL.EDITORACCESS",
		"org.imixs.ACCESSLEVEL.MANAGERACCESS" })
@Stateless
@Local(org.imixs.workflow.jee.ejb.EntityService.class)
@Remote(org.imixs.workflow.jee.ejb.EntityServiceRemote.class)
public class EntityServiceBean implements EntityService {

	@Resource
	SessionContext ctx;

	@Resource(name = "READ_ACCESS_FIELDS")
	private String readAccessFields = "";
	@Resource(name = "WRITE_ACCESS_FIELDS")
	private String writeAccessFields = "";
	@Resource(name = "ACCESS_ROLES")
	private String accessRoles = "";

	@PersistenceContext(unitName = "org.imixs.workflow.jee.jpa")
	private EntityManager manager;

	private Entity activeEntity;

	private EntityIndex activeEntityIndex = null;

	private Collection<EntityIndex> entityIndexCache = null;

	
	
	
	public String getAccessRoles() {
		return accessRoles;
	}

	public String getReadAccessFields() {
		return readAccessFields;
	}

	public String getWriteAccessFields() {
		return writeAccessFields;
	}

	/**
	 * returns a collection of all defined entityIndex fields.
	 */
	public Collection<EntityIndex> findAllEntityIndices() {
		// Caching
		if (entityIndexCache != null)
			return entityIndexCache;
		Query q = manager
				.createQuery("SELECT entityindex FROM EntityIndex entityindex");
		return q.getResultList();
	}

	/**
	 * returns a collection of ItemCollections defined by a query. The method
	 * returns detached instances of new ItemCollection objects.
	 * <p>
	 * Unfortunately we need to call the manager.refresh() method to ensure that
	 * the entity will be synchronized with database. I saw situations with the
	 * remote session ejb where the entityManager returns old relationships :-(.
	 * The reason for this behavior is the second level cache of the TopLink
	 * driver.
	 * <p>
	 * 
	 * @see http
	 *      ://weblogs.java.net/blog/guruwons/archive/2006/09/understanding_t
	 *      .html
	 */
	public Collection<ItemCollection> findAllEntities(String query,
			int startpos, int count) {
		Vector<ItemCollection> vectorResult = new Vector<ItemCollection>();

		try {
			// optimize query....
			query = optimizeQuery(query);

			Query q = manager.createQuery(query);
			if (startpos >= 0)
				q.setFirstResult(startpos);
			if (count > 0)
				q.setMaxResults(count);

			Collection<Entity> entityList = q.getResultList();
			// detach objects because implodeEntity will modify values
			// no longer necessary - hopefully manager.clear();

			if (entityList == null)
				return vectorResult;
			for (Entity aEntity : entityList) {
				activeEntity = aEntity;
				// Unfortunately we need a refresh here
				try {
					manager.refresh(activeEntity);
				} catch (EntityNotFoundException ex) {
					ex.printStackTrace();
					return null;
				}
				// implode the ItemCollection object and add it to the resultset
				vectorResult.add(implodeEntity());
			}
			return vectorResult;

		} catch (Exception e) {
			e.printStackTrace();
		}
		return vectorResult;
	}

	/**
	 * This method is used to optimize a query by adding a where clause which
	 * ensures that the query will only include entities accessible by the
	 * current user. For this reason the method adds a clause which test the
	 * readAccess property for predefined roles and usernames.
	 * <p>
	 * This method is most important to secure each findAllEntiteis method call.
	 * <p>
	 * Example
	 * <p>
	 * Here are some Examples:
	 * <ul>
	 * <li>SELECT entity FROM Entity entity
	 * <li>convert into
	 * <li>SELECT entity FROM Entity entity, entity.readAccess access WHERE
	 * access.value IN ('xxx')
	 * </ul>
	 * 
	 * <ul>
	 * <li>SELECT entity FROM Entity entity WHERE entity.type='workitem'
	 * <li>convert into
	 * <li>SELECT entity FROM Entity entity, entity.readAccess access WHERE
	 * access.value IN ('rsoika') AND entity.type='workitem'
	 * </ul>
	 * <p>
	 * The Method also verifies if a DISTINCT clause is used in the Query. If
	 * not the method will add a distinct clause to avoid duplicates in the
	 * returned result set. duplicates can be returned by complex queries with
	 * multiple joins.
	 * 
	 * @param aQuery
	 * @return
	 * @throws Exception
	 */
	private String optimizeQuery(String aQuery) throws Exception {
		String nameList = "";
		aQuery = aQuery.trim();
		StringTokenizer st = new StringTokenizer(aQuery);
		// find identifier for Entity
		if (st.countTokens() < 5)
			throw new Exception("Invalid query : " + aQuery);

		// test if DISTINCT clause is included
		st.nextToken();
		String sDistinct = st.nextToken();
		if (!"distinct".equals(sDistinct.toLowerCase().trim())) {
			// add distinct.
			int iDisPos = aQuery.toLowerCase().indexOf("select") + 6;
			aQuery = aQuery.substring(0, iDisPos) + " DISTINCT"
					+ aQuery.substring(iDisPos);
		}

		// don't optimize for manager...
		/* temporaly disabled */
		if (ctx.isCallerInRole("org.imixs.ACCESSLEVEL.MANAGERACCESS"))
			return aQuery;

		// construct nameList. Begin with empty string '' and username
		nameList = "'','" + ctx.getCallerPrincipal().getName().toString() + "'";
		// now construct role list
		String roleList = "org.imixs.ACCESSLEVEL.READERACCESS,org.imixs.ACCESSLEVEL.AUTHORACCESS,org.imixs.ACCESSLEVEL.EDITORACCESS,"
				+ accessRoles;
		// add each role the user is in to tht name list
		StringTokenizer roleListTokens = new StringTokenizer(roleList, ",");
		while (roleListTokens.hasMoreTokens()) {
			String testRole = roleListTokens.nextToken().trim();
			if (!"".equals(testRole) && ctx.isCallerInRole(testRole))
				nameList += ",'" + testRole + "'";
		}

		// System.out.println("Normal Query:");
		// System.out.println(aQuery);

		// now select identifier - this is the last word before a 'WHERE',
		// 'ORDER BY' or 'JOIN'
		int iPos = 0;
		// we begin with join
		iPos = aQuery.toLowerCase().indexOf("join");
		if (iPos == -1)
			// test for where
			iPos = aQuery.toLowerCase().indexOf("where");
		if (iPos == -1)
			// and end to test for order by clause
			iPos = aQuery.toLowerCase().indexOf("order by");

		String firstPart;
		if (iPos == -1)
			firstPart = aQuery;
		else
			firstPart = aQuery.substring(0, iPos - 1);

		firstPart = firstPart.trim();

		// System.out.println("FirstPart=" + firstPart);

		iPos = firstPart.length();
		// System.out.println("iPos=" + iPos);
		// go back to first ' '
		String identifier = null;
		identifier = firstPart.substring(firstPart.lastIndexOf(' ')).trim();

		// test for ,
		if (identifier.endsWith(","))
			identifier = identifier.substring(0, identifier.length() - 1);
		// System.out.println("Identifier='" + identifier + "'");

		// add access JOIN (should be a unique token name)
		String aNewQuery = aQuery.substring(0, iPos);
		aNewQuery += " JOIN " + identifier + ".readAccess access807 ";
		aNewQuery += aQuery.substring(iPos);

		aQuery = aNewQuery.trim();
		// test if WHERE clause is available
		int iWherePos = aQuery.toLowerCase().indexOf("where");
		if (iWherePos > -1) {
			// insert access check
			aNewQuery = aQuery.substring(0, iWherePos + 5)
					+ " (access807.value IS NULL OR access807.value IN ("
					+ nameList + ")) AND " + aQuery.substring(iWherePos + 6);
			aQuery = aNewQuery;

		} else {
			// no WHERE clause - so add a new one
			int iOrderPos = aQuery.toLowerCase().indexOf("order by");
			if (iOrderPos > -1)
				aNewQuery = aQuery.substring(0, iOrderPos - 1)
						+ " WHERE (access807.value IS NULL OR access807.value IN ("
						+ nameList + ")) " + aQuery.substring(iOrderPos);
			else
				aNewQuery = aQuery
						+ " WHERE (access807.value IS NULL OR access807.value IN("
						+ nameList + ")) ";

			aQuery = aNewQuery;
		}

		// System.out.println("Optimized Query="+aQuery);
		return aQuery;
	}

	/**
	 * This method loads a entity by its id and assembles all existing index
	 * attributes which were breaded out to subdata objects into one single
	 * ItemCollection. Notice: CalendarItems will be automatically converted
	 * into Date Objects.
	 * 
	 * The method checks also if the callerprincipal has read access to the
	 * specific entity. if not the method returns null. The method dose not
	 * throw an exception if the user is not allowed to read the entity to
	 * prevent a aggressor with informations about the existence of that
	 * specific workitem.
	 * 
	 * The returned entity also includes a temporaly attribute "isauthor" to
	 * indecate if the caller has write access to this entity.
	 * 
	 * @param id
	 *            of the entity to be loaded
	 * @return a detached Entity object or null if the entity dose not exist or
	 *         the callerpincipal hat insufficient readaccess.
	 * 
	 */
	public ItemCollection load(String id) {
		activeEntity = null;
		activeEntity = manager.find(Entity.class, id);
		// detach objects because implodeEntity will modify values
		// no longer necessary - hopefully manager.clear();

		// implode the ItemCollection object
		if (activeEntity != null && isCallerReader()) {
			// Unfortunately we need a refresh here
			try {
				manager.refresh(activeEntity);
			} catch (EntityNotFoundException ex) {
				ex.printStackTrace();
				return null;
			}

			return implodeEntity();
		} else
			return null;
	}

	/**
	 * removes an ItemCollection from the Database. The method throws an
	 * Exception if the CallerPrincipal has less write Access
	 */
	public void remove(ItemCollection itemcol) throws AccessDeniedException,
			InvalidWorkitemException {
		String sID = itemcol.getItemValueString("$uniqueid");
		activeEntity = manager.find(Entity.class, sID);

		if (activeEntity != null) {
			if (!isCallerReader())
				throw new AccessDeniedException(
						"EntityPersistanceManager: You are not allowed to perform this operation");
			manager.remove(activeEntity);
			// manager.flush();
		} else
			throw new InvalidWorkitemException();
	}

	/**
	 * This method adds a new index field to the current index list and updates
	 * existing entities. If the index already exists no changes where made.
	 * Notice that the index field name will be lowercased! Each EQL statement
	 * should use lower cased fieldnames!
	 * 
	 * The method checks if the Caller is in Role "MANGER". Since the caller is
	 * no Manager there is no guarantee that the caller can update all existing
	 * Entities as maybe he can not read all entities.
	 * 
	 * All existing Entities will be reorderd by this method call. So the method
	 * can take a long time to proceed
	 * 
	 * @param stitel
	 *            - will be automatical transformed to lowercase!
	 * @param ityp
	 *            - Type of EntityIndex
	 * @throws Exception
	 */
	public void addIndex(String stitel, int ityp) throws Exception {
		// lower case title
		stitel = stitel.toLowerCase();

		// check if index already exists?
		activeEntityIndex = manager.find(EntityIndex.class, stitel);
		if (activeEntityIndex != null)
			return;

		// Check security roles
		// only MANAGERACCESS is allowed to add index and rebuild existing
		// entities!
		if (ctx.isCallerInRole("org.imixs.ACCESSLEVEL.MANAGERACCESS") == false)
			throw new AccessDeniedException(
					"EntityPersistanceManager: You are not allowed to add index fields");

		// add new index
		activeEntityIndex = new EntityIndex(stitel, ityp);
		manager.persist(activeEntityIndex);

		// Update IndexCache
		Query queryIndex = manager
				.createQuery("SELECT wii FROM EntityIndex wii");
		entityIndexCache = queryIndex.getResultList();

		// 1.) find all existing entities
		Collection<Entity> entityList = null;
		Query q = manager.createQuery("SELECT wi FROM Entity wi");
		entityList = q.getResultList();
		updateAllEntityIndexFields(entityList);

	}

	/**
	 * This method removes an existing index from the current indexlist and
	 * updates existing entities. Notice that the index field name will be
	 * lowercased! Each EQL statement should use lower cased fieldnames!
	 * 
	 * The method checks if the Caller is in Role "MANGER". Since the caller is
	 * no Manager there is no guarantee that the caller can update all existing
	 * Entities as maybe he can not read all entities.
	 * 
	 * All existing Entities will be reorderd by this method call. So the
	 * methode can take a long time to proceed
	 * 
	 * @param stitel
	 *            - will be automatical lowercased!
	 * @throws Exception
	 */
	public void removeIndex(String stitel) throws Exception {
		// lower case title
		stitel = stitel.toLowerCase();

		// check if index already exists?
		activeEntityIndex = manager.find(EntityIndex.class, stitel);
		if (activeEntityIndex == null)
			return;

		// Check security roles
		if (ctx.isCallerInRole("org.imixs.ACCESSLEVEL.MANAGERACCESS") == false)
			throw new AccessDeniedException(
					"EntityPersistanceManager: You are not allowed to add index fields");

		// remove index
		manager.remove(activeEntityIndex);

		// Update IndexCache
		Query queryIndex = manager
				.createQuery("SELECT wii FROM EntityIndex wii");
		entityIndexCache = queryIndex.getResultList();

		// now update all entity index fields
		// 1.) find all existing entities
		Collection<Entity> entityList = null;
		Query q = manager.createQuery("SELECT wi FROM Entity wi");
		// TODO how can we preselect worktiems which are really affected ?
		entityList = q.getResultList();
		updateAllEntityIndexFields(entityList);

	}

	/**
	 * This Method saves an ItemCollection as an Entity Bean Instance using the
	 * JPA EntityManager. The method verifies if the corresponding Entity was
	 * persisted before by doing a search with the entity id ($uniqueid). If an
	 * entity instance with the corresponding id still exists this entity will
	 * be updated with the properties provided by the ItemColleciton.
	 * <p>
	 * If the ItemCollection dose not contain the property $uniqueid then the
	 * method will create a new Entity Bean instance calling the persist()
	 * method of the EntityManager. If the ItemColleciton contains a property
	 * $uniqueid but no Entity Bean instance with this ID exists in the database
	 * than a new Entity will be created using the provided $uniqueid as its
	 * primary key (id).
	 * <p>
	 * The method looks for a property "type" in the provided ItemCollection and
	 * maps this property to the corresponding Bean property. If no property
	 * with the name "type" is specified the method defaults this property to
	 * the value 'Entity'. The type property is used by the Entity Bean to
	 * categorize entities. @see org.imixs.workflow.jee.jpa.Entity
	 * <p>
	 * After persisting the provided ItemCollection the method returns a new
	 * updated instance of an ItemCollection containing the properties
	 * '$uniqueid' '$modified', '$created' and 'type'
	 * 
	 */
	public ItemCollection save(ItemCollection itemcol)
			throws AccessDeniedException {

		// check if a $uniqueid is available
		String sID = itemcol.getItemValueString("$uniqueid");
		if (!"".equals(sID)) {
			// try to find the Entity by its primary key
			activeEntity = manager.find(Entity.class, sID);
			if (activeEntity != null) {
				// Unfortunately we need a refresh here
				try {
					manager.refresh(activeEntity);
				} catch (EntityNotFoundException ex) {
					ex.printStackTrace();
					activeEntity = null;
				}
			}
			if (activeEntity == null) {
				// workitem no found! so create new instance using the provided
				// id
				// but first check if user is allowd to create Entities....
				if (!(ctx.isCallerInRole("org.imixs.ACCESSLEVEL.MANAGERACCESS")
						|| ctx
								.isCallerInRole("org.imixs.ACCESSLEVEL.EDITORACCESS") || ctx
						.isCallerInRole("org.imixs.ACCESSLEVEL.AUTHORACCESS"))) {
					throw new AccessDeniedException(
							"EntityServiceBean: You are not allowed to perform this operation");
				}
				// create new one with the provided id
				activeEntity = new Entity(sID);
				// if $Created is provided than overtake this information
				if (itemcol.hasItem("$Created")) {
					Date datCreated = itemcol.getItemValueDate("$Created");
					Calendar cal = Calendar.getInstance();
					cal.setTime(datCreated);
					// Overwrite Creation Date
					activeEntity.setCreated(cal);
				}
				// convert to managed entity
				manager.persist(activeEntity);
				// throw new IllegalArgumentException("Unknow $uniqueid: " +
				// sID);

			} else {
				// activeEntity is now managed
				// verify if current user has write- and readaccess to this
				// entity
				if (!isCallerAuthor() || !isCallerReader()) {
					throw new AccessDeniedException(
							"EntityServiceBean: You are not allowed to perform this operation");
				}

			}
		} else {
			// create a new entity object
			// but first check if user is allowd to create Entities....
			if (!(ctx.isCallerInRole("org.imixs.ACCESSLEVEL.MANAGERACCESS")
					|| ctx.isCallerInRole("org.imixs.ACCESSLEVEL.EDITORACCESS") || ctx
					.isCallerInRole("org.imixs.ACCESSLEVEL.AUTHORACCESS"))) {
				throw new AccessDeniedException(
						"EntityServiceBean: You are not allowed to perform this operation");
			}
			// create new one
			activeEntity = new Entity();
			// convert to managed entity
			manager.persist(activeEntity);
		}

		// remove property $isauthor
		itemcol.removeItem("$isauthor");

		// update Data and type property
		activeEntity.getData().setItemCollection(itemcol);
		String aType = itemcol.getItemValueString("type");
		if ("".equals(aType))
			aType = "Entity";
		activeEntity.setType(aType);

		// update the standard attributes $modified $created and $uniqueid
		Calendar cal = Calendar.getInstance();
		try {
			activeEntity.getData().getItemCollection().replaceItemValue(
					"$uniqueid", activeEntity.getId());
			activeEntity.getData().getItemCollection().replaceItemValue(
					"$modified", cal.getTime());
			activeEntity.getData().getItemCollection().replaceItemValue(
					"$created", activeEntity.getCreated().getTime());
		} catch (Exception e) {
			e.printStackTrace();
		}

		// now update the index fields...
		explodeEntity();

		// update read- and writeAccess List
		updateReadAccessList();
		updateWriteAccessList();

		/*
		 * Finally I merge the activeEntity to synchronize the relationships
		 * added by the explodeEntity and updateAccessList methods. This is
		 * necessary as these methods did not do any persist or merge calls. So
		 * the activeEntiy will not be synchronized. Especially in long
		 * transactions (like the WebService update methods from the
		 * modelService) I recognized situations where a SQL exception was
		 * thrown. Could be that in some vendor specific implementations this
		 * merge is not necessary or throws an exception. But I am currently not
		 * sure about this. If the merge call leads into an exception we need to
		 * surround this call with a try-catch. The merge() call works and was
		 * tested for Glassfish V2 Server.
		 */
		activeEntity = manager.merge(activeEntity);

		/*
		 * finally return a new detached version of the saved ItemCollection.
		 * This guarantees that the caller will receive a copy of the
		 * ItemCollection instance actually saved. Index Properties will contain
		 * Unique values!
		 */
		return implodeEntity();

	}

	/**
	 * This method updates the internal ReadAccessList. Therefore the method
	 * verifies if the data element (itemCollection) contains ReadAccess
	 * properties. The default property name is '$readaccess'. Additional
	 * property names can be provided by using the resource "READ_ACCESS_FIELDS"
	 * in the deployment descriptor. The Values of the corresponding Items in
	 * the ItemCollection will not be modified or removed by this method.
	 * <p>
	 * If no read restriction is defined the method will generate an empty entry
	 */
	private void updateReadAccessList() {
		// clear current read access list
		Collection<ReadAccessEntity> colAccessEntities = activeEntity
				.getReadAccess();
		activeEntity.setReadAccess(null);
		for (ReadAccessEntity aItem : colAccessEntities)
			manager.remove(aItem);

		// generate property name list for readAccess
		Vector<String> vAccessFieldList = new Vector<String>();
		// add default fieldname
		vAccessFieldList.add("$readAccess");

		// test if the Resource READ_ACCESS_FIELDS is provided
		if (readAccessFields != null && !"".equals(readAccessFields)) {
			String[] stringArrayList = readAccessFields.split(",");
			for (String aentry : stringArrayList) {
				vAccessFieldList.add(aentry);
			}
		}
		ItemCollection itemCol = activeEntity.getData().getItemCollection();
		// process all reader attributes
		Vector vValueList = new Vector();
		for (String aentry : vAccessFieldList) {
			// add values. But do not add empty string entries here!
			Vector vEntryList = itemCol.getItemValue(aentry);
			Iterator iterTest = vEntryList.iterator();
			while (iterTest.hasNext()) {
				String aEntry = (String) iterTest.next();
				;
				if (aEntry != null && !"".equals(aEntry))
					vValueList.add(aEntry);
			}
		}

		// test if the complete valuelist is empty. If so add one empty string
		// element
		if (vValueList.size() == 0)
			vValueList.add("");
		Iterator iter = vValueList.iterator();
		while (iter.hasNext()) {
			addReadAccessEntity(activeEntity.getReadAccess(),
					new ReadAccessEntity(iter.next().toString()));
		}
	}

	/**
	 * This method updates the internal WriteAccessList. Therefore the method
	 * verifies if the itemCollection contains WriteAccess properties. The
	 * default property name is '$writeaccess'. Additional property names can be
	 * provided by using the resource "WRITE_ACCESS_FIELDS" in the deployment
	 * descriptor. The Values of the corresponding Items in the ItemCollection
	 * will not be modified or removed by this method.
	 */
	private void updateWriteAccessList() {
		// clear current write access list
		Collection<WriteAccessEntity> colAccessEntities = activeEntity
				.getWriteAccess();
		activeEntity.setWriteAccess(null);
		for (WriteAccessEntity aItem : colAccessEntities)
			manager.remove(aItem);

		Vector<String> vAccessFieldList = new Vector<String>();
		// add default fieldname
		vAccessFieldList.add("$writeAccess");

		// test if the Resource WRITE_ACCESS_FIELDS is provided
		if (writeAccessFields != null && !"".equals(writeAccessFields)) {
			String[] stringArrayList = writeAccessFields.split(",");
			for (String aentry : stringArrayList) {
				vAccessFieldList.add(aentry);
			}
		}
		ItemCollection itemCol = activeEntity.getData().getItemCollection();

		// process all attributes
		Vector vValueList = new Vector();
		for (String aentry : vAccessFieldList) {
			// add values. But do not add empty string entries here!
			Vector vEntryList = itemCol.getItemValue(aentry);
			Iterator iterTest = vEntryList.iterator();
			while (iterTest.hasNext()) {
				String aEntry = (String) iterTest.next();
				;
				if (aEntry != null && !"".equals(aEntry))
					vValueList.add(aEntry);
			}
		}

		// test if valuelist is empty
		if (vValueList.size() == 0)
			vValueList.add("");
		Iterator iter = vValueList.iterator();
		while (iter.hasNext()) {
			addWriteAccessEntity(activeEntity.getWriteAccess(),
					new WriteAccessEntity(iter.next().toString()));
		}
	}

	/**
	 * Explodes an ItemCollection into a Entity with its index fields. This
	 * method extracts all items with corresponding Indices into the subtables.
	 */
	private void explodeEntity() {
		// clear first all existing index entities to make sure that the
		// collections are empty
		/*
		 * activeEntity.getCalendarItems().clear();
		 * activeEntity.getDoubleItems().clear();
		 * activeEntity.getIntegerItems().clear();
		 * activeEntity.getTextItems().clear();
		 */
		removeIndexProperties();

		// now update the entity object
		entityIndexCache = findAllEntityIndices();
		Iterator<EntityIndex> it = entityIndexCache.iterator();
		while (it.hasNext()) {
			activeEntityIndex = it.next();
			String name = activeEntityIndex.name;
			int typ = activeEntityIndex.typ;
			explodeItem(name, typ);
		}
	}

	/**
	 * This method will remove all existing Items attached to the current
	 * Entity.
	 */
	private void removeIndexProperties() {
		// remove all Items...
		Collection<TextItem> colText = activeEntity.getTextItems();
		for (TextItem aItem : colText)
			manager.remove(aItem);
		activeEntity.getTextItems().clear();
		activeEntity.setTextItems(null);

		Collection<IntegerItem> colInteger = activeEntity.getIntegerItems();
		for (IntegerItem aItem : colInteger)
			manager.remove(aItem);
		activeEntity.getIntegerItems().clear();
		activeEntity.setIntegerItems(null);

		Collection<DoubleItem> colDouble = activeEntity.getDoubleItems();
		for (DoubleItem aItem : colDouble)
			manager.remove(aItem);
		activeEntity.getDoubleItems().clear();
		activeEntity.setDoubleItems(null);

		Collection<CalendarItem> colCalendar = activeEntity.getCalendarItems();
		for (CalendarItem aItem : colCalendar)
			manager.remove(aItem);
		activeEntity.getCalendarItems().clear();
		activeEntity.setCalendarItems(null);
	}

	/**
	 * This method extracts an single property of the ItemCollection (data
	 * object of the activeEntity) into the corresponding index properties
	 * (TextItem, IntegerItem DoubleItem, CalendarItem).
	 * <p>
	 * The ItemName will be transformed to lowercase!
	 * <p>
	 * If an instance of a Date Object is provided in the ItemCollection the
	 * values will be converted from the Date object into a Calendar Object.
	 * This is because the ItemCollection works typical with Date Objects
	 * instead of Calendar Objects.
	 * <p>
	 * After the method has moved an attribute with its values to the
	 * corresponding Index Property the method removes the attribute form the
	 * itemCollection.
	 * <p>
	 * If a value of a property did not match the indexProperty Type the item
	 * value will not be exploded. But the full item will be removed from the
	 * ItemCollection. The Method prints an exception to the log file. As a
	 * result of this implementation the invalid value will be lost!
	 * 
	 * 
	 * @see implodeEntity()
	 * @param name
	 */
	private void explodeItem(String name, int typ) {
		// transform itemname into lowercase
		name = name.toLowerCase();

		// get values
		Vector vValue = activeEntity.getData().getItemCollection()
				.getItemValue(name);

		for (int i = 0; i < vValue.size(); i++) {
			try {
				Object o = vValue.elementAt(i);
				if (typ == EntityIndex.TYP_TEXT)
					addTextItem(activeEntity.getTextItems(), new TextItem(name,
							o.toString()));

				if (typ == EntityIndex.TYP_INT)
					addIntegerItem(activeEntity.getIntegerItems(),
							new IntegerItem(name, (Integer) o));

				if (typ == EntityIndex.TYP_DOUBLE)
					addDoubleItem(activeEntity.getDoubleItems(),
							new DoubleItem(name, (Double) o));

				// Date Objects will be converted into Calendar Objects
				if (typ == EntityIndex.TYP_CALENDAR) {
					if (o instanceof java.util.Date) {
						Calendar cal = Calendar.getInstance();
						cal.setTime((java.util.Date) o);
						o = cal;
					}
					addCalendarItem(activeEntity.getCalendarItems(),
							new CalendarItem(name, (Calendar) o));

				}

			} catch (ClassCastException e) {
				// if a ClassCastExcepiton was thrown then one of the values did
				// not match the propertyIndex Type.
				// We print this exception to the stack trace but did continue
				// work this results into a situation where invalid values will
				// be removed from the ItemCollection value list. But it
				// guarantees that the entity exists in a valid structure and
				// matches the data model
				e.printStackTrace();
			}
		}

		// now remove the attribute from ItemCollection
		activeEntity.getData().getItemCollection().removeItem(name);

	}

	/**
	 * This method returns an detached new Instance of an ItemCollection from
	 * the activtEntity object. Therefore the method builds a new ItemCollection
	 * instance containing all values of the activeEntity. Properties which
	 * where separated out into the index properties (TextItems, IntegerItems,
	 * DoubleItems, CalendarItems) will be moved back into the ItemCollection
	 * Object returned by this method. So the returned ItemCollection will
	 * contain all attributes managed currently by the activeEntity.
	 * <p>
	 * The method takes care about the fact that the values of the activeEntiy
	 * are typical managed by the PersistenceManger. For this reason the method
	 * did not copy the Vector objects into the new detached Instance of the new
	 * ItemCollection but copies only the values of each vector. This mechanism
	 * guarantees that the reconstruction of indexProperties did not affect the
	 * managed activeEntity and its binded objects. So it is not necessary that
	 * the activeEntity was detached before this method is called as the method
	 * will not! modify property values which would be written back to the
	 * database.
	 * <p>
	 * Calendar Objects will be converted into Date Objects. ItemCollection
	 * works typical with Date Objects instead of Calendar Objects.
	 * <p>
	 * The method verifies if the callerPricipal has author access and adds the
	 * attribute $isAuthor into the ItemCollection to indicate the author access
	 * of this entity for the current user.
	 * 
	 * @see explodeEntity()
	 * @return
	 */
	@SuppressWarnings("unchecked")
	private ItemCollection implodeEntity() {
		// create new empty ItemCollection
		ItemCollection detachedItemCollection = new ItemCollection();

		// verify author access and add property '$isauthor'
		try {
			detachedItemCollection.replaceItemValue("$isauthor",
					isCallerAuthor());
		} catch (Exception e1) {
		}

		/*
		 * Now we first copy for each IndexProperty the values into our new
		 * detached ItemColleciton Object. The mechanism guarantees that the
		 * values of the indexProperty will be added into a new separated
		 * instance of an empty Vector. This vector is not managed by the
		 * persitenceManger and will not reflect back to the persistenceContext.
		 * This is important here because we may not modify the activeEntity as
		 * this is a managed object!
		 * 
		 * The call of
		 * 
		 * detachedItemCollection .getItemValue(ti.itemName);
		 * 
		 * will return an empty vector the fist time!
		 */

		// so now copy all separated TextItems
		Collection<TextItem> textitems = activeEntity.getTextItems();
		Iterator<TextItem> iterText = textitems.iterator();
		while (iterText.hasNext()) {
			TextItem ti = iterText.next();
			try {
				Vector vValue = detachedItemCollection
						.getItemValue(ti.itemName);
				vValue.addElement(ti.itemValue);
				detachedItemCollection.replaceItemValue(ti.itemName, vValue);
			} catch (Exception e) {
			}
		}

		// copy all separated IntegerItems
		Collection<IntegerItem> integeritems = activeEntity.getIntegerItems();
		Iterator<IntegerItem> iterInteger = integeritems.iterator();
		while (iterInteger.hasNext()) {
			IntegerItem ii = iterInteger.next();
			try {
				Vector vValue = detachedItemCollection
						.getItemValue(ii.itemName);
				vValue.addElement(ii.itemValue);
				detachedItemCollection.replaceItemValue(ii.itemName, vValue);
			} catch (Exception e) {
			}
		}

		// copy all separated DoubleItems
		Collection<DoubleItem> doubleitems = activeEntity.getDoubleItems();
		Iterator<DoubleItem> iterDouble = doubleitems.iterator();
		while (iterDouble.hasNext()) {
			DoubleItem di = iterDouble.next();
			try {
				Vector vValue = detachedItemCollection
						.getItemValue(di.itemName);
				vValue.addElement(di.itemValue);
				detachedItemCollection.replaceItemValue(di.itemName, vValue);
			} catch (Exception e) {
			}
		}

		// copy all separated CalendarItems
		Collection<CalendarItem> calendaritems = activeEntity
				.getCalendarItems();
		Iterator<CalendarItem> iterCalendar = calendaritems.iterator();
		while (iterCalendar.hasNext()) {
			CalendarItem ci = iterCalendar.next();
			try {
				Vector vValue = detachedItemCollection
						.getItemValue(ci.itemName);
				// Calendar Objects will be converted into Date Objects
				vValue.addElement(ci.itemValue.getTime());
				detachedItemCollection.replaceItemValue(ci.itemName, vValue);
			} catch (Exception e) {
			}
		}

		/*
		 * After we added all indexProperties to our empty ItemCollection we can
		 * now add the rest of the properties contained in the EntityData
		 * Object. But we need to make sure that we do not overwrite an new
		 * created index property. So we do not simply call the putAll() method.
		 * But we verify each property name and value. If its new than we will
		 * copy the values - again not the Vector! as we want to avoid to
		 * manipulate or return managed objects. So we do create new Vector()
		 * instances for each property
		 */

		// ! do not use :
		// detachedItemCollection.getAllItems().putAll(activeEntity.getData().getItemCollection().getAllItems());
		Map activeMap = activeEntity.getData().getItemCollection()
				.getAllItems();
		if (activeMap != null) {
			// iterate over all properties
			Iterator iter = activeMap.entrySet().iterator();
			while (iter.hasNext()) {
				Map.Entry mapEntry = (Map.Entry) iter.next();
				// extract name and value
				String activePropertyName = mapEntry.getKey().toString();
				if (!detachedItemCollection.hasItem(activePropertyName)) {
					// if we did not have created this property from the
					// indexProperties before we add the new property....
					Vector activePropertyValue = (Vector) mapEntry.getValue();
					// create new Vector
					Vector detachePropertyValue = new Vector();
					// copy objects
					detachePropertyValue.addAll(activePropertyValue);
					// add property
					try {
						detachedItemCollection.replaceItemValue(
								activePropertyName, detachePropertyValue);
					} catch (Exception e) {
						// unexpected exception - but we will continue work
						e.printStackTrace();
					}
				}
				// now we are sure that we have a new instance of an vector
				// for each property
			}
		}

		return detachedItemCollection;
	}

	/**
	 * This method returns an detached new Instance of an ItemCollection from
	 * the activtEntity object. Therefore the method builds a new ItemCollection
	 * instance containing all values of the activeEntity. Properties which
	 * where separated out into the index properties (TextItems, IntegerItems,
	 * DoubleItems, CalendarItems) will be moved back into the ItemCollection
	 * Object returned by this method. So the returned ItemCollection will
	 * contain all attributes managed currently by the activeEntity.
	 * <p>
	 * It is important that the activeEntity is detached before this method is
	 * called as the method will modify the property values which would be
	 * written back to the database if the Entity was still managed by the
	 * PersistenceManager! This is the reason why the save, find and load method
	 * did a clear() before it calls this method
	 * <p>
	 * Calendar Objects will be converted into Date Objects. ItemCollection
	 * works typical with Date Objects instead of Calendar Objects.
	 * <p>
	 * The method verifies if the callerPricipal has author access and adds the
	 * attribute $isAuthor into the ItemCollection to indicate the author access
	 * of this entity for the current user.
	 * 
	 * @see explodeEntity()
	 * @return
	 */
	@Deprecated
	@SuppressWarnings("unchecked")
	private ItemCollection oldimplodeEntity() {
		// create new ItemCollection
		ItemCollection detachedItemCollection = new ItemCollection();

		// verify author access and add property '$isauthor'
		try {
			detachedItemCollection.replaceItemValue("$isauthor",
					isCallerAuthor());
		} catch (Exception e1) {
		}

		// copy all values into the new object....
		detachedItemCollection.getAllItems().putAll(
				activeEntity.getData().getItemCollection().getAllItems());
		// copy all separated TextItems
		Collection<TextItem> textitems = activeEntity.getTextItems();
		Iterator<TextItem> iterText = textitems.iterator();
		while (iterText.hasNext()) {
			TextItem ti = iterText.next();
			try {
				Vector vValue = detachedItemCollection
						.getItemValue(ti.itemName);
				vValue.addElement(ti.itemValue);
				detachedItemCollection.replaceItemValue(ti.itemName, vValue);
			} catch (Exception e) {
			}
		}

		// copy all separated IntegerItems
		Collection<IntegerItem> integeritems = activeEntity.getIntegerItems();
		Iterator<IntegerItem> iterInteger = integeritems.iterator();
		while (iterInteger.hasNext()) {
			IntegerItem ii = iterInteger.next();
			try {
				Vector vValue = detachedItemCollection
						.getItemValue(ii.itemName);
				vValue.addElement(ii.itemValue);
				detachedItemCollection.replaceItemValue(ii.itemName, vValue);
			} catch (Exception e) {
			}
		}

		// copy all separated DoubleItems
		Collection<DoubleItem> doubleitems = activeEntity.getDoubleItems();
		Iterator<DoubleItem> iterDouble = doubleitems.iterator();
		while (iterDouble.hasNext()) {
			DoubleItem di = iterDouble.next();
			try {
				Vector vValue = detachedItemCollection
						.getItemValue(di.itemName);
				vValue.addElement(di.itemValue);
				detachedItemCollection.replaceItemValue(di.itemName, vValue);
			} catch (Exception e) {
			}
		}

		// copy all separated CalendarItems
		Collection<CalendarItem> calendaritems = activeEntity
				.getCalendarItems();
		Iterator<CalendarItem> iterCalendar = calendaritems.iterator();
		while (iterCalendar.hasNext()) {
			CalendarItem ci = iterCalendar.next();
			try {
				Vector vValue = detachedItemCollection
						.getItemValue(ci.itemName);
				// Calendar Objects will be converted into Date Objects
				vValue.addElement(ci.itemValue.getTime());
				detachedItemCollection.replaceItemValue(ci.itemName, vValue);
			} catch (Exception e) {
			}
		}

		return detachedItemCollection;
	}

	/**
	 * This method checks if the Caller Principal has read access for the
	 * activeEntity.
	 * 
	 * @return true if user has readaccess
	 */
	private boolean isCallerReader() {

		Collection<ReadAccessEntity> readAccessList = activeEntity
				.getReadAccess();

		/**
		 * 1.) org.imixs.ACCESSLEVEL.NOACCESS
		 * 
		 * always = false -> no access
		 */

		if (ctx.isCallerInRole("org.imixs.ACCESSLEVEL.NOACCESS"))
			return false;

		/**
		 * 2.) org.imixs.ACCESSLEVEL.MANAGERACCESS
		 * 
		 * always = true -> grant access.
		 */

		if (ctx.isCallerInRole("org.imixs.ACCESSLEVEL.MANAGERACCESS"))
			return true;

		/**
		 * 2.) org.imixs.ACCESSLEVEL.EDITOR org.imixs.ACCESSLEVEL.AUTHOR
		 * ACCESSLEVEL.READER
		 * 
		 * check read access
		 */
		if (readAccessList == null || readAccessList.size() == 0) {
			// no restriction found
			return true;
		}

		boolean notemptyfield = false;

		// check each write access
		Iterator<ReadAccessEntity> iteratorACL = readAccessList.iterator();
		while (iteratorACL.hasNext()) {
			ReadAccessEntity readAccessEntry = iteratorACL.next();
			String sAccessName = readAccessEntry.getValue();
			if (sAccessName != null && !"".equals(sAccessName)) {
				notemptyfield = true;
				try {
					if (ctx.getCallerPrincipal().getName().equals(sAccessName)
							|| ctx.isCallerInRole(sAccessName))
						return true; // known role - grant access
				} catch (Exception e) {
					// no operation - Role simple not defined
					// this could be an model issue
				}

			}
		}

		if (!notemptyfield)
			return true;

		return false;
	}

	/**
	 * Verifies if the caller has write access to the current entity object.
	 * 
	 * @return
	 */
	private boolean isCallerAuthor() {
		Collection<WriteAccessEntity> writeAccessList = activeEntity
				.getWriteAccess();

		/**
		 * 1.) org.imixs.ACCESSLEVEL.NOACCESS allways false - now write access!
		 */
		if (ctx.isCallerInRole("org.imixs.ACCESSLEVEL.NOACCESS"))
			return false;

		/**
		 * 2.) org.imixs.ACCESSLEVEL.MANAGERACCESS or
		 * org.imixs.ACCESSLEVEL.EDITOR Always true - grant writeaccess.
		 */
		if (ctx.isCallerInRole("org.imixs.ACCESSLEVEL.MANAGERACCESS")
				|| ctx.isCallerInRole("org.imixs.ACCESSLEVEL.EDITORACCESS"))
			return true;

		/**
		 * 2.) org.imixs.ACCESSLEVEL.AUTHOR
		 * 
		 * check write access in detail
		 */
		if (ctx.isCallerInRole("org.imixs.ACCESSLEVEL.AUTHORACCESS")) {
			if (writeAccessList == null || writeAccessList.size() == 0) {
				// now wirte access
				return false;
			}

			// check each write access
			Iterator<WriteAccessEntity> iteratorACL = writeAccessList
					.iterator();
			while (iteratorACL.hasNext()) {
				WriteAccessEntity writeAccessEntry = iteratorACL.next();
				String sAccessName = writeAccessEntry.getValue();
				if (sAccessName != null && !"".equals(sAccessName)) {
					try {
						if (ctx.getCallerPrincipal().getName().equals(
								sAccessName)
								|| ctx.isCallerInRole(sAccessName))
							return true; // Role known
					} catch (Exception e) {
						// no operation - Role simply not defined
						// this could be an model issue and need not to be
						// handled as an error
					}
				}
			}
		}
		return false;
	}

	/**
	 * This Method updates the index properties from a collection of entities.
	 * The Method first implode the Entity to get all external properties and
	 * than calls an explode to recreate the indexproperties. The merge() call
	 * after the explode is necessary for sychronize the persitence context for
	 * the SQL updates from the relationships.
	 */
	private void updateAllEntityIndexFields(Collection<Entity> entityList)
			throws Exception {
		Iterator<Entity> iter = entityList.iterator();
		while (iter.hasNext()) {
			activeEntity = iter.next();
			// Unfortunately we need a refresh here - see comment for method
			// findAllEntities
			try {
				manager.refresh(activeEntity);
			} catch (EntityNotFoundException ex) {
				ex.printStackTrace();
				throw ex;
			}

			// implode each Entity into its ItemCollection
			ItemCollection itemCol = implodeEntity();
			activeEntity.getData().setItemCollection(itemCol);
			// explode entity with new Index properties
			explodeEntity();

			/*
			 * Finally I merge the activeEntity to synchronize the relationships
			 * added by the explodeEntity method. This is necessary as this
			 * method did not do any persist or merge calls. So the activeEntiy
			 * will not be synchronized. It could be that in some vendor
			 * specific implementations this merge is not necessary or throws an
			 * exception. But I am currently not sure about this. If the merge
			 * call leads into an exception we need to surround this call with a
			 * try-catch. The merge() call works and was tested for Glassfish V2
			 * Server.
			 * 
			 * @see also the implementation of the save() method.
			 */
			manager.merge(activeEntity);

		}
	}

	/**
	 * this method adds a textitem to an existing list of textitems. The method
	 * verifies is the new textitem is not still included in the collection. If
	 * so the method did not add the new item. So the collection will include
	 * unique entries.
	 * 
	 * @param col
	 * @param newItem
	 */
	private void addTextItem(Collection<TextItem> col, TextItem newItem) {
		Iterator<TextItem> iter = col.iterator();
		while (iter.hasNext()) {
			TextItem aItem = iter.next();
			// compare - if included return
			if (aItem.itemName.equals(newItem.itemName)
					&& aItem.itemValue.equals(newItem.itemValue))
				return;
		}
		// now add new Item
		col.add(newItem);
	}

	/**
	 * @see addTextItem
	 * @param col
	 * @param newItem
	 */
	private void addReadAccessEntity(Collection<ReadAccessEntity> col,
			ReadAccessEntity newItem) {
		Iterator<ReadAccessEntity> iter = col.iterator();
		while (iter.hasNext()) {
			ReadAccessEntity aItem = iter.next();
			// compare - if included return
			if (aItem.getValue().equals(newItem.getValue()))
				return;
		}
		// now add new Item
		col.add(newItem);
	}

	/**
	 * @see addTextItem
	 * @param col
	 * @param newItem
	 */
	private void addWriteAccessEntity(Collection<WriteAccessEntity> col,
			WriteAccessEntity newItem) {
		Iterator<WriteAccessEntity> iter = col.iterator();
		while (iter.hasNext()) {
			WriteAccessEntity aItem = iter.next();
			// compare - if included return
			if (aItem.getValue().equals(newItem.getValue()))
				return;
		}
		// now add new Item
		col.add(newItem);
	}

	/**
	 * @see addTextItem
	 * @param col
	 * @param newItem
	 */
	private void addCalendarItem(Collection<CalendarItem> col,
			CalendarItem newItem) {
		Iterator<CalendarItem> iter = col.iterator();
		while (iter.hasNext()) {
			CalendarItem aItem = iter.next();
			// compare - if included return
			if (aItem.itemName.equals(newItem.itemName)
					&& aItem.itemValue.equals(newItem.itemValue))
				return;
		}
		// now add new Item
		col.add(newItem);
	}

	/**
	 * @see addTextItem
	 * @param col
	 * @param newItem
	 */
	private void addDoubleItem(Collection<DoubleItem> col, DoubleItem newItem) {
		Iterator<DoubleItem> iter = col.iterator();
		while (iter.hasNext()) {
			DoubleItem aItem = iter.next();
			// compare - if included return
			if (aItem.itemName.equals(newItem.itemName)
					&& aItem.itemValue.equals(newItem.itemValue))
				return;
		}
		// now add new Item
		col.add(newItem);
	}

	/**
	 * @see addTextItem
	 * @param col
	 * @param newItem
	 */
	private void addIntegerItem(Collection<IntegerItem> col, IntegerItem newItem) {
		Iterator<IntegerItem> iter = col.iterator();
		while (iter.hasNext()) {
			IntegerItem aItem = iter.next();
			// compare - if included return
			if (aItem.itemName.equals(newItem.itemName)
					&& aItem.itemValue.equals(newItem.itemValue))
				return;
		}
		// now add new Item
		col.add(newItem);
	}

}
