/*******************************************************************************
 *  Imixs IX Workflow Technology
 *  Copyright (C) 2001, 2008 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika
 *******************************************************************************/
package org.imixs.workflow.jee.ejb;

import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Vector;

import javax.annotation.Resource;
import javax.annotation.security.DeclareRoles;
import javax.annotation.security.RunAs;
import javax.ejb.EJB;
import javax.ejb.Local;
import javax.ejb.Remote;
import javax.ejb.SessionContext;
import javax.ejb.Stateless;
import javax.ejb.Timeout;
import javax.ejb.Timer;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;

import org.imixs.workflow.ItemCollection;

/**
 * This is the implementation of a scheduled workflow service. The EJB
 * implementation can be used as a Timer Service to process scheduled workflow
 * activities.
 * 
 * A scheduled workflow activity can be defined using the IX Workflow modeller.
 * The service verifies if a workitem should be processed scheduled and
 * processes such a workitem automatically.
 * 
 * The TimerService can be started using the method scheduleWorkflow. The
 * Methods findTimerDescription and findAllTimerDescriptions are used to lookup
 * enabled and running service instances.
 * 
 * Each Method expects or generates a TimerDescription Object. This object is an
 * instance of a ItemCollection. To create a new timer the ItemCollection should
 * contain the following attributes:
 * 
 * datstart - Date Object
 * 
 * datstop - Date Object
 * 
 * numInterval - Integer Object (interval in seconds)
 * 
 * id - String - unique identifier for the schedule Service.
 * 
 * $modelversion - String - identifies the model which schould be managed by the
 * service
 * 
 * the following additional attributes are generated by the finder methods and
 * can be used by an application to verfiy the status of a running instance:
 * 
 * nextTimeout - Next Timeout - pint of time when the service will be scheduled
 * 
 * timeRemaining - Timeout in milliseconds
 * 
 * statusmessage - text message
 * 
 * How a worklist will be processed:
 * 
 * If the ActivityEntity has defined a EQL statement (attribute
 * txtscheduledview) then the method selects the workitems by this query.
 * Otherwise the method use the standard method getWorklistByProcessID()
 * 
 * @author rsoika
 * 
 */
@DeclareRoles( { "org.imixs.ACCESSLEVEL.MANAGERACCESS" })
@Stateless
@RunAs("org.imixs.ACCESSLEVEL.MANAGERACCESS")
@Local(org.imixs.workflow.jee.ejb.WorkflowScheduler.class)
@Remote(org.imixs.workflow.jee.ejb.WorkflowSchedulerRemote.class)
public class WorkflowSchedulerBean implements
WorkflowScheduler {

	private Date startDate, endDate;
	private int interval;
	private String id;
	private int iProcessWorkItems = 0;
	private int iScheduledWorkItems = 0;

	private Collection<ItemCollection> colScheduledActivities = null;

	@Resource
	SessionContext ctx;

	@EJB
	WorkflowService workflowService;

	@EJB
	EntityService entityService;

	@EJB
	ModelService modelService;

	@Resource
	javax.ejb.TimerService timerService;

	/**
	 * This Method starts a new TimerService.
	 * 
	 * The method expects an ItemCollection (timerdescription) with the
	 * following informations:
	 * 
	 * datstart - Date Object
	 * 
	 * datstop - Date Object
	 * 
	 * numInterval - Integer Object (interval in seconds)
	 * 
	 * id - String - unique identifier for the schedule Service.
	 * 
	 * The param 'id' should contain a unique identifier (e.g. the EJB Name) as
	 * only one scheduled Workflow should run inside a WorkflowInstance. If a
	 * timer with the id is already running the method stops this timer object
	 * first and reschedules the timer.
	 * 
	 * The method throws an exception if the timerdescription contains invalid
	 * attributes or values.
	 */
	public void scheduleWorkflow(ItemCollection timerdescription)
			throws Exception {

		validateTimerDescription(timerdescription);

		// try to cancel an existing timer for this workflowinstance
		cancelScheduleWorkflow(timerdescription.getItemValueString("id"));

		// set statusmessage
		Calendar calNow = Calendar.getInstance();
		String msg = "Started: " + calNow.getTime() + " by "
				+ ctx.getCallerPrincipal().getName();
		timerdescription.replaceItemValue("statusmessage", msg);
		timerdescription.replaceItemValue("$created", calNow.getTime());
		timerService.createTimer(startDate, interval, timerdescription);
	}

	/**
	 * This Method lookups a TimerService. The method returns a ItemCollection
	 * with the TimerDescirpiton for the specific timer object. If no Timer was
	 * found the method returns an empty timer Description.
	 * 
	 * @param id
	 *            identifies the timer object
	 * @return ItemCollection containing the TimerDescription
	 * @throws Exception
	 */
	public ItemCollection findTimerDescription(String id) throws Exception {
		Timer timer = this.getTimer(id);
		if (timer != null) {
			ItemCollection adescription = (ItemCollection) timer.getInfo();
			// add timer info...
			adescription
					.replaceItemValue("nextTimeout", timer.getNextTimeout());
			adescription.replaceItemValue("timeRemaining", timer
					.getTimeRemaining());
			return adescription;

		}
		// return empty timerdescription
		ItemCollection adescription = new ItemCollection();
		adescription.replaceItemValue("id", id);
		return adescription;
	}

	/**
	 * This method returns a collection of running timerServices.
	 * 
	 */
	public Collection<ItemCollection> findAllTimerDescriptions()
			throws Exception {
		Vector v = new Vector();
		for (Object obj : timerService.getTimers()) {
			Timer timer = (javax.ejb.Timer) obj;

			ItemCollection adescription = null;
			if (timer.getInfo() instanceof ItemCollection) {
				adescription = (ItemCollection) timer.getInfo();
				adescription.replaceItemValue("nextTimeout", timer
						.getNextTimeout());
				adescription.replaceItemValue("timeRemaining", timer
						.getTimeRemaining());
				v.add(adescription);
			}
		}
		return v;
	}

	/**
	 * Cancels a running timer instance. After cancel a timer the corresponding
	 * timerDescripton (ItemCollection) is no longer valid
	 * 
	 */
	public void cancelScheduleWorkflow(String id) throws Exception {
		Timer timer = this.getTimer(id);
		if (timer != null) {
			timer.cancel();
			System.out
					.println("[WorkflowScheduler] cancelScheduleWorkflow: "
							+ id);
		}
	}

	/**
	 * This method returns a timer for a corresponding id if such a timer object
	 * exists.
	 * 
	 * @param id
	 * @return Timer
	 * @throws Exception
	 */
	private Timer getTimer(String id) throws Exception {
		for (Object obj : timerService.getTimers()) {
			Timer timer = (javax.ejb.Timer) obj;
			if (timer.getInfo() instanceof ItemCollection) {
				ItemCollection adescription = (ItemCollection) timer.getInfo();
				if (id.equals(adescription.getItemValueString("id"))) {
					return timer;
				}
			}
		}
		return null;
	}

	/**
	 * this method verifies the attributes in a timer description
	 * 
	 * @param atimerDesc
	 * @throws Exception
	 */
	private void validateTimerDescription(ItemCollection atimerDesc)
			throws Exception {
		boolean valid = true;
		try {
			startDate = (Date) atimerDesc.getItemValue("datstart")
					.firstElement();
			endDate = (Date) atimerDesc.getItemValue("datstop").firstElement();
			interval = atimerDesc.getItemValueInteger("numinterval");
			id = atimerDesc.getItemValueString("id");
			if ("".equals(id))
				valid = false;
		} catch (Exception e) {
			valid = false;
		}

		if (!valid)
			throw new Exception(
					"TimerDescirption is not valid! datstart(Date), datstop(Date), numinterval(Integer), id(String) need to be sported!");
	}

	/**
	 * This is the mehtod which processed scheuduled workitems when the timer is
	 * called.
	 * 
	 * @param timer
	 */
	@Timeout
	public void processWorkItems(javax.ejb.Timer timer) {
		String sModelVersion = null;
		String sTimerID=null;
		try {
			ItemCollection adescription = (ItemCollection) timer.getInfo();
			
			sTimerID=adescription.getItemValueString("id");
			System.out.println("[WorkflowScheduler]  Processing : "
					+ sTimerID);

			sModelVersion = adescription.getItemValueString("$modelversion");
			if ("".equals(sModelVersion))
				sModelVersion = modelService.getLatestVersion();
			System.out.println("[WorkflowScheduler] ModelVersion="
					+ sModelVersion);

			System.out
					.println("[WorkflowScheduler] start process Workitems...");
			iProcessWorkItems = 0;
			iScheduledWorkItems = 0;

			// find scheduled Activities

			colScheduledActivities = findScheduledActivities(sModelVersion);
			System.out.println("[WorkflowScheduler]  "
					+ colScheduledActivities.size()
					+ " scheduled activityEntities found");
			// process all workitems for coresponding activities
			for (ItemCollection aactivityEntity : colScheduledActivities) {
				processWorkList(aactivityEntity);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

		System.out.println("[WorkflowScheduler]  finished successfull");
		System.out.println("[WorkflowScheduler] " + iProcessWorkItems
				+ " workitems checked");
		System.out.println("[WorkflowScheduler] " + iScheduledWorkItems
				+ " workitems processed");

		/*
		 * Check if Timer should be canceld now?
		 */
		if (endDate != null) {
			Calendar calNow = Calendar.getInstance();

			if (calNow.getTime().after(endDate)) {
				timer.cancel();
				System.out
						.println("[WorkflowScheduler] cancelScheduleWorkflow: "
								+ sTimerID);
			}
		}
	}

	/*
	 * 
	 * Helper Methods to process scheduled workitems
	 */

	/**
	 * collects all scheduled workflow activities. An scheduled worklfow
	 * activity is identified by the attribute keyScheduledActivity="1"
	 * 
	 * The method goes through the latest or a specific Model Version
	 * 
	 */
	private Collection<ItemCollection> findScheduledActivities(
			String aModelVersion) throws Exception {
		Vector<ItemCollection> vectorActivities = new Vector<ItemCollection>();
		Collection<ItemCollection> colProcessList = null;

		// get a complete list of process entities...
		if (aModelVersion != null)
			colProcessList = modelService.getProcessEntityListByVersion(aModelVersion);
		else
			colProcessList = modelService.getProcessEntityList();
		for (ItemCollection aprocessentity : colProcessList) {
			// select all activities for this process entity...
			int processid = aprocessentity.getItemValueInteger("numprocessid");
			// System.out.println("Analyse processentity '" + processid+ "'");
			Collection<ItemCollection> aActivityList = modelService
					.getActivityEntityListByVersion(processid, aModelVersion);

			for (ItemCollection aactivityEntity : aActivityList) {
				// System.out.println("Analyse acitity '" + aactivityEntity
				// .getItemValueString("txtname") + "'");

				// check if activity is scheduled
				if ("1".equals(aactivityEntity
						.getItemValueString("keyScheduledActivity")))
					vectorActivities.add(aactivityEntity);
			}
		}
		return vectorActivities;
	}

	/**
	 * This method processes all workitems for a specific processID. the
	 * processID is idenfied by the activityEntity Object (numprocessid)
	 * 
	 * If the ActivityEntity has defined a EQL statement (attribute
	 * txtscheduledview) then the method selects the workitems by this query.
	 * Otherwise the method use the standard method getWorklistByProcessID()
	 * 
	 * @param aProcessID
	 * @throws Exception
	 */
	private void processWorkList(ItemCollection activityEntity)
			throws Exception {
		// get processID
		int iProcessID = activityEntity.getItemValueInteger("numprocessid");
		// get Modelversion
		String sModelVersion = activityEntity
				.getItemValueString("$modelversion");

		// if a query is defined in the activityEntity then use the EQL
		// statement
		// to query the items. Otherwise use standard method
		// getWorklistByProcessID()
		String sQuery = activityEntity.getItemValueString("txtscheduledview");

		// get all workitems...
		Collection<ItemCollection> worklist = null;
		if (sQuery != null && !"".equals(sQuery)) {
			//System.out.println("[WorkflowScheduler] Query=" + sQuery);
			worklist = entityService.findAllEntities(sQuery, 0, -1);
		} else {
			//System.out
			//		.println("[WorkflowScheduler] getWorkListByProcessID.."
			//				+ sQuery);
			worklist = workflowService.getWorkListByProcessID(iProcessID, 0, -1,null,0);
		}
		iProcessWorkItems += worklist.size();
		for (ItemCollection workitem : worklist) {
			// verify processID
			if (iProcessID == workitem.getItemValueInteger("$processid")) {
				// verify modelversion
				if (sModelVersion.equals(workitem
						.getItemValueString("$modelversion"))) {
					// verify due date
					if (workItemInDue(workitem, activityEntity)) {

						int iActivityID = activityEntity
								.getItemValueInteger("numActivityID");
						workitem.replaceItemValue("$activityid",iActivityID);
						processWorkitem(workitem);
						iScheduledWorkItems++;

					}
				}
			}
		}
	}

	/**
	 * start new Transaction for each process step
	 * 
	 * @param aWorkitem
	 * @param aID
	 */
	@TransactionAttribute(value = TransactionAttributeType.REQUIRES_NEW)
	private void processWorkitem(ItemCollection aWorkitem) {
		try {
			workflowService.processWorkItem(aWorkitem);
		} catch (Exception e) {

			e.printStackTrace();
		}
	}

	/**
	 * This method checks if a workitem (doc) is in due. There are 4 different
	 * cases which will be compared: The case is determined by the
	 * keyScheduledBaseObject of the activity entity
	 * 
	 * Basis : keyScheduledBaseObject "last process"=1, "last Modification"=2
	 * "Creation"=3 "Field"=4
	 * 
	 * The logic is not the best one but it works. So we are open for any kind
	 * of improvements
	 * 
	 * @return true if workitem is is due
	 */
	public boolean workItemInDue(ItemCollection doc, ItemCollection docActivity) {
		try {
			int iCompareType = -1;
			int iDelayUnit = -1;

			Date dateTimeCompare = null;
			// int iRepeatTime = 0,
			int iActivityDelay = 0;

			String suniqueid = doc.getItemValueString("$uniqueid");

			String sDelayUnit = docActivity
					.getItemValueString("keyActivityDelayUnit");
			iDelayUnit = Integer.parseInt(sDelayUnit); // min | 1; hours | 2;
			// days | 3
			// iRepeatTime =
			// docActivity.getItemValueInteger("numActivityMinOffset");
			iActivityDelay = docActivity
					.getItemValueInteger("numActivityDelay");
			if (true) {
				if ("1".equals(sDelayUnit))
					sDelayUnit = "minutes";
				if ("2".equals(sDelayUnit))
					sDelayUnit = "hours";
				if ("3".equals(sDelayUnit))
					sDelayUnit = "days";
				
				//System.out.println("[WorkflowScheduler] "
				//		+ suniqueid + " delay =" + iActivityDelay + " "
				//		+ sDelayUnit);
				
			}
			// Delay in sekunden umrechnen
			if (iDelayUnit == 1) {
				iActivityDelay *= 60; // min->sec
			} else {
				if (iDelayUnit == 2) {
					iActivityDelay *= 3600; // hour->sec
				} else {
					if (iDelayUnit == 3) {
						iActivityDelay *= 3600 * 24; // day->sec
					}
				}
			}

			iCompareType = Integer.parseInt(docActivity
					.getItemValueString("keyScheduledBaseObject"));

			// get current time for compare....
			Date dateTimeNow = Calendar.getInstance().getTime();

			switch (iCompareType) {
			// last process -
			case 1: {
				//System.out.println("[WorkflowScheduler] "
				//		+ suniqueid + ": CompareType = last process");

				if (!doc.hasItem("timWorkflowLastAccess"))
					return false;

				dateTimeCompare = (Date) doc.getItemValue(
						"timWorkflowLastAccess").firstElement();
				System.out.println("[WorkflowScheduler] "
						+ suniqueid + ": timWorkflowLastAccess="
						+ dateTimeCompare);

				// scheduled time
				dateTimeCompare = this.adjustSecond(dateTimeCompare,
						iActivityDelay);

				return dateTimeCompare.before(dateTimeNow);
			}

				// last modification - es erfolgt kein Vergleich mit last
				// Event, da dieses ja selbst der auslöser der Zeit ist
			case 2: {
				//System.out.println("[WorkflowScheduler] "
				//		+ suniqueid + ": CompareType = last modify");

				dateTimeCompare = (Date) doc.getItemValue("$modified")
						.firstElement();

				System.out.println("[WorkflowScheduler] "
						+ suniqueid + ": modified=" + dateTimeCompare);

				dateTimeCompare = adjustSecond(dateTimeCompare, iActivityDelay);

				return dateTimeCompare.before(dateTimeNow);
			}

				// creation
			case 3: {
				//System.out.println("[WorkflowScheduler] "
				//		+ suniqueid + ": CompareType = creation");

				dateTimeCompare = (Date) doc.getItemValue("$created")
						.firstElement();
				System.out.println("[WorkflowScheduler] "
						+ suniqueid + ": doc.getCreated() =" + dateTimeCompare);

				// Nein -> Creation date ist masstab
				dateTimeCompare = this.adjustSecond(dateTimeCompare,
						iActivityDelay);

				return dateTimeCompare.before(dateTimeNow);
			}

				// field
			case 4: {
				String sNameOfField = docActivity
						.getItemValueString("keyTimeCompareField");
				//System.out.println("[WorkflowScheduler] "
				//		+ suniqueid + ": CompareType = field: '" + sNameOfField
				//		+ "'");

				if (!doc.hasItem(sNameOfField)) {
					System.out.println("[WorkflowScheduler] "
							+ suniqueid + ": CompareType =" + sNameOfField
							+ " no value found!");
					return false;
				}

				dateTimeCompare = (Date) doc.getItemValue(sNameOfField)
						.firstElement();

				System.out.println("[WorkflowScheduler] "
						+ suniqueid + ": " + sNameOfField + "="
						+ dateTimeCompare);

				dateTimeCompare = adjustSecond(dateTimeCompare, iActivityDelay);

				if (true) {
					System.out.println("[WorkflowScheduler] "
							+ suniqueid + ": Compare " + dateTimeCompare + " <-> " + dateTimeNow);
				}
				return dateTimeCompare.before(dateTimeNow);
			}
			default:
				return false;
			}

		} catch (Exception e) {

			e.printStackTrace();
			return false;
		}

	}

	/**
	 * This method add seconds to a given date object
	 * 
	 * @param adate
	 *            to be adjusted
	 * @param seconds
	 *            to be added (can be <0)
	 * @return new date object
	 */
	private Date adjustSecond(Date adate, int seconds) {
		Calendar calTimeCompare = Calendar.getInstance();
		calTimeCompare.setTime(adate);
		calTimeCompare.add(Calendar.SECOND, seconds);
		return calTimeCompare.getTime();
	}

}
