/*******************************************************************************
 *  Imixs IX Workflow Technology
 *  Copyright (C) 2001, 2008 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika
 *******************************************************************************/

package org.imixs.workflow.jee.ejb;

import java.util.Collection;

import org.imixs.workflow.ItemCollection;
import org.imixs.workflow.exceptions.AccessDeniedException;
import org.imixs.workflow.exceptions.InvalidWorkitemException;
import org.imixs.workflow.jee.jpa.EntityIndex;

/**
 * The EntityService is used to save and load instances of ItemCollection into a
 * Database. The EntityService throws an AccessDeniedException if the
 * CallerPrincipal is not allowed to save or read a specific ItemCollection from
 * the database. So the EntityService can be used to save business objects into
 * a database with individual read- or writeAccess restrictions.
 * 
 * @author rsoika
 * @version 1.0
 */
public interface EntityService {

	public static final int TYP_TEXT = 0;

	public static final int TYP_INT = 1;

	public static final int TYP_DOUBLE = 2;

	public static final int TYP_CALENDAR = 3;

	public static final int TYP_READER = 4;

	public static final int TYP_AUTHOR = 5;

	/**
	 * Returns additional AccessRoles defined for the EJB instance
	 * 
	 * @return
	 */
	public String getAccessRoles();

	/**
	 * Returns additional ReadAccessFields defined for the EJB instance.
	 * Default=$ReadAccess
	 * 
	 * @return
	 */
	public String getReadAccessFields();

	/**
	 * Returns additional WriteAccessFields defined for the EJB instance.
	 * Default=$WriteAccess
	 * 
	 * @return
	 */
	public String getWriteAccessFields();

	/**
	 * This Method saves an ItemCollection into a database. If the
	 * ItemCollection is saved the first time the method generates a uniqueID
	 * ('$uniqueid') which can be used to identify the ItemCollection by its ID.
	 * If the ItemCollection was saved before the method updates the
	 * ItemCollection stored in the database. The Method returns an updated
	 * instance of the ItemCollection containing the attributes $modified,
	 * $created, and $uniqueid
	 * <p>
	 * The method throws an AccessDeniedException if the CallerPrincipal is not
	 * allowed to save or update the ItemCollection in the database. The
	 * CallerPrincipial should have at least the access Role
	 * org.imixs.ACCESSLEVEL.AUTHORACCESS
	 * 
	 * @param ItemCollection
	 *            to be saved
	 * @return updated ItemCollection
	 */
	public ItemCollection save(ItemCollection itemcol)
			throws AccessDeniedException;

	/**
	 * This method loads an ItemCollection from the Database. The method expects
	 * a valid $unqiueID to identify the ItemCollection saved before into the
	 * database. The method returns null if no ItemCollection with the
	 * corresponding ID exists.
	 * <p>
	 * The method checks also if the CallerPrincipal has read access to
	 * ItemCollection stored in the database. If not the method returns null.
	 * The method dose not throw an AccessDeniedException if the user is not
	 * allowed to read the entity to prevent a aggressor with informations about
	 * the existence of that specific ItemCollection.
	 * <p>
	 * CallerPrincipial should have at least the access Role
	 * org.imixs.ACCESSLEVEL.READACCESS
	 * 
	 * @param id
	 *            the $unqiueid of the ItemCollection to be loaded
	 * @return ItemCollection object or null if the ItemColleciton dose not
	 *         exist or the CallerPrincipal hat insufficient read access.
	 * 
	 */
	public ItemCollection load(String id);

	/**
	 * This method removes an ItemCollection from the database. If the
	 * CallerPrincipal is not allowed to access the ItemColleciton the method
	 * throws an AccessDeniedException.
	 * <p>
	 * The CallerPrincipial should have at least the access Role
	 * org.imixs.ACCESSLEVEL.AUTHORACCESS
	 * 
	 * @param ItemCollection
	 *            to be removed
	 * @throws Exception
	 */
	public void remove(ItemCollection itemcol) throws AccessDeniedException,
			InvalidWorkitemException;

	/**
	 * Adds an EntityIndex to the current list of external properties. A
	 * EntityIndex defines a Index to a specific Attribute provided by
	 * ItemCollections saved to the database. The method throws an
	 * AccessDeniedException if the CallerPrinciapal is not in the role
	 * org.imixs.ACCESSLEVEL.MANAGERACCESS.
	 * 
	 * @param stitel
	 * @param ityp
	 *            - Type of EntityIndex
	 * @throws Exception
	 */
	public void addIndex(String stitel, int ityp) throws Exception;

	/**
	 * Removes an EntityIndex from the current list of external properties. The
	 * method throws an AccessDeniedException if the CallerPrinciapal is not in
	 * the role org.imixs.ACCESSLEVEL.MANAGERACCESS.
	 * 
	 * @param stitel
	 * @throws Exception
	 */
	public void removeIndex(String stitel) throws Exception;

	/**
	 * The method returns a list of EntityIndex which defines external
	 * properties
	 * 
	 * @return
	 */
	public Collection<EntityIndex> findAllEntityIndices();

	/**
	 * The method returns a collection of ItemCollections. The method expects an
	 * valid EQL statement. The method returns only ItemCollections which are
	 * readable by the CallerPrincipal. With the startpos and count parameters
	 * it is possible to read chunks of entities.
	 * 
	 * @param query
	 * @param startpos
	 * @param count
	 * @return
	 */
	public Collection<ItemCollection> findAllEntities(String query,
			int startpos, int count);

	/**
	 * Indecates if the calerPrincipal has author rights for the coressponding
	 * Entity.
	 * 
	 * @param id
	 * @return
	 */
	// public boolean isAuthor(String id);
}
