package org.imixs.workflow.jee.jsf.util;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;

import javax.ejb.EJB;

import org.imixs.workflow.ItemCollection;
import org.imixs.workflow.util.ItemCollectionAdapter;

/**
 * This BLOBWorkitemController is used to store large objects into a single
 * ItemCollection mapped to a EntityBean. The BLOBWorkitemController supports
 * also the management of file attachment inside an ItemCollection. So the
 * BLOBWorkitemController can be used to save large data objects into a Workitem
 * (called BlobWorkitems). The BlobWorkitem is always bounded to a parent
 * workitem by its referrer id ($uniqueidRef). So an application can implement a
 * lazy loading for BLOBWorkitems. The read- and write access settings of
 * BLOBWorkitems are always synchronized to the settings of the parent workitem.
 * Before the BlobWorkitem can be accessed the workitem needs to be loaded by
 * the load() method. The Data can be accessed by the embedded Itemcollection
 * through the method getWorkitem(). The BlobWorkitem can be saved by calling
 * the save() method. Both - the load() and the save() method expect the Parent
 * ItemCollection where the BlobWorkitem should be bound.
 * 
 * @version 0.0.1
 * @author rsoika
 * 
 */
public class BLOBWorkitemController {

	@EJB
	org.imixs.workflow.jee.ejb.EntityService entityService;

	private ItemCollection blobWorkitem = null;
	protected ItemCollectionAdapter workitemAdapter = null;

	public BLOBWorkitemController() {
		super();

		blobWorkitem = new ItemCollection();
		workitemAdapter = new ItemCollectionAdapter(blobWorkitem);
	}

	/**
	 * returns the ItemCollection for the curren BlobWorkitem object.
	 * 
	 * @return
	 */
	public ItemCollection getWorkitem() {
		return blobWorkitem;
	}

	/**
	 * Returns a list of file names attached to the current BlobWorkitem. File
	 * Attachments can be added using the method addFile().
	 * 
	 * @return
	 */
	public String[] getFiles() {
		// File attachments...
		String[] files = new String[0];
		if (blobWorkitem != null) {
			HashMap mapFiles = null;
			Vector vFiles = blobWorkitem.getItemValue("$file");
			if (vFiles != null && vFiles.size() > 0) {
				mapFiles = (HashMap) vFiles.elementAt(0);
				files = new String[mapFiles.entrySet().size()];
				Iterator iter = mapFiles.entrySet().iterator();
				int iFileCount = 0;
				while (iter.hasNext()) {
					Map.Entry mapEntry = (Map.Entry) iter.next();
					String aFileName = mapEntry.getKey().toString();
					files[iFileCount] = aFileName;
					iFileCount++;
				}
			}
		}
		return files;
	}

	/**
	 * Removes the connection to the parend workitem and clear the
	 * itemCollection
	 */
	public void clear() {
		blobWorkitem = new ItemCollection();
	}

	/**
	 * This method saves the current BlobWorkitem. Therefore the method copies
	 * the read- and write access list from the given parent workitem into the
	 * BlobWorkitem before save.
	 * 
	 * So this method should be called after a WorkflowProcessing step to update
	 * the read- and write access identically to the parentWorkitem
	 * 
	 * @throws Exception
	 */
	public void save(ItemCollection parentWorkitem) throws Exception {

		if (blobWorkitem != null && parentWorkitem != null) {

			// Update Read and write access list from parent workitem
			Vector vAccess = parentWorkitem
					.getItemValue("$ReadAccess");
			blobWorkitem.replaceItemValue("$ReadAccess",
					vAccess);

			vAccess = parentWorkitem.getItemValue("$WriteAccess");
			blobWorkitem.replaceItemValue("$WriteAccess",
					vAccess);

			blobWorkitem.replaceItemValue("$uniqueidRef", parentWorkitem
					.getItemValueString("$uniqueID"));
			blobWorkitem.replaceItemValue("type", "workitemlob");
			// Update BlobWorkitem
			entityService.save(blobWorkitem);

		}
	}

	/**
	 * This method adds a single file to the ItemCollection. files will be
	 * stored into the property $file.
	 * 
	 * @param data
	 *            - byte array with file data
	 * @param fileName
	 *            - name of the file attachment
	 * @param contentType
	 *            - the contenttype (e.g. 'Text/HTML')
	 * @throws Exception
	 */
	public void addFile(byte[] data, String fileName, String contentType)
			throws Exception {
		if (data != null) {
			Vector<Object> vectorFileInfo = null;

			// IE includes '\' characters! so remove all these characters....
			if (fileName.indexOf('\\') > -1)
				fileName = fileName.substring(fileName.lastIndexOf('\\') + 1);
			if (fileName.indexOf('/') > -1)
				fileName = fileName.substring(fileName.lastIndexOf('/') + 1);

			if (contentType == null || "".equals(contentType))
				contentType = "application/unknown";

			// Store files using a hashmap....
			HashMap mapFiles = null;
			Vector vFiles = blobWorkitem.getItemValue("$file");
			if (vFiles != null && vFiles.size() > 0)
				mapFiles = (HashMap) vFiles.elementAt(0);
			else
				mapFiles = new HashMap();

			// existing file will be overridden!
			vectorFileInfo = (Vector) mapFiles.get(fileName);
			vectorFileInfo = new Vector<Object>();
			// put file in a vector containing the byte array and also the
			// content type
			vectorFileInfo.add(contentType);
			vectorFileInfo.add(data);
			mapFiles.put(fileName, vectorFileInfo);
			blobWorkitem.replaceItemValue("$file", mapFiles);
		}
	}

	/**
	 * This method removes a single file attachment from the BlobWorkitem
	 * 
	 * @throws Exception
	 */
	public void removeFile(String aFilename) throws Exception {

		if (blobWorkitem != null) {
			/* delete attachment */
			String[] files = new String[0];
			HashMap mapFiles = null;
			try {
				Vector vFiles = blobWorkitem.getItemValue("$file");
				if (vFiles != null && vFiles.size() > 0) {
					mapFiles = (HashMap) vFiles.elementAt(0);
					mapFiles.remove(aFilename);
					blobWorkitem.replaceItemValue("$file", mapFiles);
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * Loads the BlobWorkitem of a given parent Workitem. The BlobWorkitem is
	 * identified by the $unqiueidRef. If no BlobWorkitem still exists the
	 * method creates a new empty BlobWorkitem which can be saved later.
	 * 
	 * @param itemCol
	 *            - parent workitem where the BlobWorkitem will be attached to
	 * @throws Exception
	 */
	public void load(ItemCollection itemCol) throws Exception {

		String sUniqueID = itemCol.getItemValueString("$uniqueid");

		// search entity...
		String sQuery = " SELECT lobitem FROM Entity as lobitem"
				+ " join lobitem.textItems as t1"
				+ " join lobitem.textItems as t2"
				+ " WHERE t1.itemName = 'type'"
				+ " AND t1.itemValue = 'workitemlob'"
				+ " AND t2.itemName = '$uniqueidref'" + " AND t2.itemValue = '"
				+ sUniqueID + "'";

		Collection<ItemCollection> itemcol = entityService.findAllEntities(
				sQuery, 0, 1);
		if (itemcol != null && itemcol.size() > 0) {

			blobWorkitem = itemcol.iterator().next();
		} else {
			blobWorkitem = new ItemCollection();
			blobWorkitem.replaceItemValue("$uniqueidRef", sUniqueID);
			blobWorkitem.replaceItemValue("type", "workitemlob");
		}

		// update adapter
		workitemAdapter = new ItemCollectionAdapter(blobWorkitem);
	}

	/**
	 * returns a workitemAdapter Item field
	 * 
	 * @return
	 * @throws Exception
	 */
	public Map getItem() throws Exception {
		return workitemAdapter.getItem();
	}

	/**
	 * returns a workitemAdapter ItemList
	 * 
	 * @return
	 * @throws Exception
	 */
	public Map getItemList() throws Exception {
		return workitemAdapter.getItemList();
	}

	/**
	 * returns a workitemAdapter ItemList Array
	 * 
	 * @return
	 * @throws Exception
	 */
	public Map getItemListArray() throws Exception {
		return workitemAdapter.getItemListArray();
	}

}
