package org.imixs.workflow.jee.jsf.util;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;

import javax.ejb.EJB;
import javax.faces.application.FacesMessage;
import javax.faces.component.UIComponent;
import javax.faces.component.UIData;
import javax.faces.component.UIParameter;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.faces.event.ActionEvent;

import org.imixs.workflow.ItemCollection;
import org.imixs.workflow.jee.ejb.WorkflowService;
import org.imixs.workflow.util.ItemCollectionAdapter;

/**
 * This Class is an abstract controller used in JSF Application A JSF Web Modul
 * should subclass the AbstractWorkflowController for implementing a project
 * specific Imixs Workflow BackingBean
 * 
 * 
 * @author rsoika
 * @version 0.0.1
 */
public abstract class AbstractWorkflowController {
	protected ItemCollection workitemItemCollection = null;
	protected ItemCollectionAdapter workitemAdapter = null;
	private String type;

	/* Workflow Model */
	private ArrayList<ItemCollectionAdapter> activityList;
	private ArrayList<ItemCollectionAdapter> startProcessList = null;
	private String modelVersion = null;

	/* Worklist and Search */
	private ArrayList<ItemCollectionAdapter> workitems = null;
	private int maxSearchResult = 10;
	private int row = 0;
	private boolean endOfList = false;
	private int queryType = 0;
	private String searchQuery = null;
	final int QUERY_WORKLIST_BY_OWNER = 0;
	final int QUERY_WORKLIST_BY_CREATOR = 1;
	final int QUERY_WORKLIST_BY_AUTHOR = 2;
	final int QUERY_WORKLIST_ALL = 3;
	final int QUERY_WORKLIST_ARCHIVE = 4;
	final int QUERY_SEARCH = 5;
	final int QUERY_WORKLIST_BY_WRITEACCESS = 6;
	private int sortOrder = WorkflowService.SORT_ORDER_CREATED_DESC;
	// private String accessRoles = null;

	// EJBs

	@EJB
	private org.imixs.workflow.jee.ejb.EntityService entityService;

	@EJB
	private org.imixs.workflow.jee.ejb.ModelService modelService;

	@EJB
	private org.imixs.workflow.jee.ejb.WorkflowService workflowService;

	public AbstractWorkflowController() {
		super();
		setType("workitem");
		workitemItemCollection = new ItemCollection();
		workitemAdapter = new ItemCollectionAdapter(workitemItemCollection);
	}

	/**
	 * returns an instance of the EntityService EJB
	 * 
	 * @return
	 */
	public org.imixs.workflow.jee.ejb.EntityService getEntityService() {
		return entityService;
	}

	/**
	 * returns an instance of the ModelService EJB
	 * 
	 * @return
	 */
	public org.imixs.workflow.jee.ejb.ModelService getModelService() {
		return modelService;
	}

	/**
	 * returns an instance of the WorkflowService EJB
	 * 
	 * @return
	 */
	public org.imixs.workflow.jee.ejb.WorkflowService getWorkflowService() {
		return workflowService;
	}

	/**
	 * returns the $uniqueID of the current workitem
	 * 
	 * @return
	 */
	public String getID() {
		if (workitemItemCollection == null)
			return null;
		else
			return workitemItemCollection.getItemValueString("$uniqueid");
	}

	/**
	 * set the value for the attribute 'type' of a workitem to be generated or
	 * search by this controller
	 */
	public String getType() {
		return type;
	}

	/**
	 * defines the type attribute of a workitem to be generated or search by
	 * this controller
	 * 
	 * Subclasses may overwrite the type
	 * 
	 * @param type
	 */
	public void setType(String type) {
		this.type = type;
	}

	/**
	 * defines the default sortOrder for finder methods
	 * 
	 * @return
	 */
	public int getSortOrder() {
		return sortOrder;
	}

	public void setSortOrder(int sortOrder) {
		this.sortOrder = sortOrder;
	}

	/**
	 * returns the current ModelVersion. If no ModelVersion is defined the
	 * method returns the latest modelversion defined
	 * 
	 * @return
	 */
	public String getModelVersion() {
		if (modelVersion == null) {
			try {
				modelVersion = modelService.getLatestVersion();
			} catch (Exception e) {
				e.printStackTrace();
				return null;
			}
		}
		return modelVersion;
	}

	public void setModelVersion(String modelVersion) {
		this.modelVersion = modelVersion;
	}

	/**
	 * returns the maximum size of a search result
	 * 
	 * @return
	 */
	public int getMaxSearchResult() {
		return maxSearchResult;
	}

	/**
	 * set the maximum size of a search result
	 * 
	 * @param searchCount
	 */
	public void setMaxSearchResult(int searchCount) {
		this.maxSearchResult = searchCount;
	}

	/**
	 * defines the searchQuery to be use for a EQL search by the method
	 * doSwitchToSearchlist
	 * 
	 * Sublcasses may overwrite this method to define individual search queries.
	 * 
	 * @return
	 */
	public String getSearchQuery() {
		return searchQuery;
	}

	/**
	 * set the current searchquery
	 * 
	 * @param query
	 */
	public void setSearchQuery(String query) {
		this.searchQuery = query;
	}

	/**
	 * provides a list of writeaccess roles. Roles can be set by a
	 * managed-property tag
	 * 
	 * @return
	 */
	/*
	 * public String getAccessRoles() { return accessRoles; }
	 * 
	 * public void setAccessRoles(String accessRoles) { this.accessRoles =
	 * accessRoles; }
	 */

	/**
	 * updates all attributes by the supported map into the ItemCollection
	 * 
	 * Diese Mehtode wird beim Klick auf einen Datensatz aufgerufen
	 * 
	 * @param ateam
	 */
	public void setWorkitem(ItemCollection aworkitem) {
		if (aworkitem != null)
			workitemItemCollection = aworkitem;
		else
			workitemItemCollection = new ItemCollection();
		workitemAdapter.setItemCollection(workitemItemCollection);
	}

	public ItemCollection getWorkitem() {
		return workitemItemCollection;
	}

	/**
	 * This method is called by a jsf page to create a new instance of a
	 * process.
	 * 
	 * This method creates an empty worktitem
	 * 
	 * The Workflow attributes ProcessID, Modelversion and WorfklowGroup will be
	 * set to the attributes of the corresponding ProcessEntity provided by
	 * param ID in the ActionEvent
	 * 
	 * The Method expects an ID which identifies the ProcessID and optional a
	 * ModelVersion The ID is a String value with the following format:
	 * 
	 * 
	 * modelversion|processID
	 * 
	 * The first Part is the modelversion for the corresponding workflowmodel
	 * the second part will be vast to an integer which corresponds to the start
	 * process id in the model. The modelversion part with the '|' is optional.
	 * 
	 * @see WorkitemServiceBean
	 * 
	 * @param event
	 * @return
	 */
	public void doCreateWorkitem(ActionEvent event) throws Exception {
		// get Process ID out from the ActionEvent Object....
		List children = event.getComponent().getChildren();
		String processEntityIdentifier = "";

		for (int i = 0; i < children.size(); i++) {
			if (children.get(i) instanceof UIParameter) {
				UIParameter currentParam = (UIParameter) children.get(i);
				if (currentParam.getName().equals("id")
						&& currentParam.getValue() != null) {
					processEntityIdentifier = currentParam.getValue()
							.toString();
					break;
				}
			}
		}

		if (processEntityIdentifier != null
				&& !"".equals(processEntityIdentifier)) {
			// find ProcessEntity the Worktiem should be started at
			String sProcessModelVersion = null;
			String sProcessID = processEntityIdentifier;

			// check if modelversion is provided (format: version|id)...
			if (processEntityIdentifier.indexOf('|') > -1) {
				// yes - extract modelversion from process id
				sProcessModelVersion = processEntityIdentifier.substring(0,
						processEntityIdentifier.indexOf('|'));
				sProcessID = processEntityIdentifier
						.substring(processEntityIdentifier.indexOf('|') + 1);

				// update ModelVersion
				this.setModelVersion(sProcessModelVersion);
			}

			// create new Workitem
			doCreateWorkitem(Integer.parseInt(sProcessID));
		} else {
			// no processid specified - create simple workitem
			workitemItemCollection = new ItemCollection();
			FacesContext context = FacesContext.getCurrentInstance();
			ExternalContext externalContext = context.getExternalContext();
			String sUser = externalContext.getRemoteUser();
			workitemItemCollection.replaceItemValue("namCreator", sUser);
			workitemItemCollection.replaceItemValue("type", getType());
			// update the ItemCollectionAdapter...
			this.setWorkitem(workitemItemCollection);
		}
	}

	/**
	 * This method is called by a jsf page to create a new instance of a
	 * process.
	 * 
	 * This method creates an empty worktitem for a defined ProcessID in the
	 * current ModelVersion.
	 * 
	 * The Workflow attributes ProcessID, Modelversion and WorfklowGroup will be
	 * set to the attributes of the corresponding ProcessEntity provided by
	 * param ID in the ActionEvent
	 * 
	 * @param event
	 * @return
	 */
	public void doCreateWorkitem(int ProcessID) throws Exception {
		ItemCollection itemColProcessEntity = null;
		try {
			// find ProcessEntity
			itemColProcessEntity = modelService.getProcessEntityByVersion(
					ProcessID, this.getModelVersion());

			// ProcessEntity found?
			if (itemColProcessEntity == null)
				throw new NullPointerException();
		} catch (Exception eproc) {
			throw new Exception(
					"unable to find ProcessEntity in model version "
							+ this.getModelVersion() + " for ID=" + ProcessID);
		}

		String sEditorID = itemColProcessEntity
				.getItemValueString("txteditorid");
		int processID = itemColProcessEntity
				.getItemValueInteger("numProcessID");
		String sWorkflowGroup = itemColProcessEntity
				.getItemValueString("txtworkflowgroup");

		// Create new empty workitem
		workitemItemCollection = new ItemCollection();
		workitemItemCollection.replaceItemValue("$ProcessID", new Integer(
				processID));
		FacesContext context = FacesContext.getCurrentInstance();
		ExternalContext externalContext = context.getExternalContext();
		String sUser = externalContext.getRemoteUser();

		workitemItemCollection.replaceItemValue("namCreator", sUser);
		workitemItemCollection.replaceItemValue("$WriteAccess", sUser);

		// assigen ModelVersion, group and editor
		workitemItemCollection.replaceItemValue("$modelversion",
				this.getModelVersion());
		workitemItemCollection.replaceItemValue("txtworkflowgroup",
				sWorkflowGroup);
		workitemItemCollection.replaceItemValue("txteditorid", sEditorID);

		workitemItemCollection.replaceItemValue("type", getType());

		// update the ItemCollectionAdapter...
		this.setWorkitem(workitemItemCollection);

	}

	/**
	 * processes the current issue. The method expects an parameter "id" with
	 * the activity ID which should be processed
	 * 
	 * Method should be overwritten to add additional Business logic here.
	 * 
	 * @param event
	 * @return
	 * @throws Exception
	 */
	public void doProcessWorkitem(ActionEvent event) throws Exception {
		// Activity ID raussuchen und in activityID speichern
		List children = event.getComponent().getChildren();
		int activityID = -1;

		for (int i = 0; i < children.size(); i++) {
			if (children.get(i) instanceof UIParameter) {
				UIParameter currentParam = (UIParameter) children.get(i);
				if (currentParam.getName().equals("id")
						&& currentParam.getValue() != null) {
					// Value can be provided as String or Integer Object
					activityID = Integer.parseInt(currentParam.getValue()
							.toString());
					break;
				}
			}
		}

		// update type
		workitemItemCollection.replaceItemValue("type", getType());
		// set activityid
		workitemItemCollection.replaceItemValue("$activityid", activityID);

		// process workitem now...
		workitemItemCollection = workflowService
				.processWorkItem(workitemItemCollection);

		// and finally update the Workitem Reference for the
		// ItemCollectionAdapter....
		this.setWorkitem(workitemItemCollection);
		// reset views
		doReset(event);
	}

	/**
	 * this method is called by datatables to select an workitem
	 * 
	 * @return
	 */
	public void doEdit(ActionEvent event) {
		ItemCollectionAdapter currentSelection = null;
		// suche selektierte Zeile....
		UIComponent component = event.getComponent();
		for (UIComponent parent = component.getParent(); parent != null; parent = parent
				.getParent()) {
			if (!(parent instanceof UIData))
				continue;

			// Zeile gefunden
			currentSelection = (ItemCollectionAdapter) ((UIData) parent)
					.getRowData();
			setWorkitem(currentSelection.getItemCollection());
			break;

		}
	}

	/**
	 * this method removes the current selected workitem from a view. The Method
	 * also deletes also all child workitems recursive
	 * 
	 * @param event
	 * @throws Exception
	 */
	public void doDelete(ActionEvent event) throws Exception {
		ItemCollectionAdapter currentSelection = null;
		// suche selektierte Zeile....
		UIComponent component = event.getComponent();
		for (UIComponent parent = component.getParent(); parent != null; parent = parent
				.getParent()) {
			if (!(parent instanceof UIData))
				continue;

			// Zeile gefunden
			currentSelection = (ItemCollectionAdapter) ((UIData) parent)
					.getRowData();
			break;
		}

		// if nothing found - then try to cath current workitem
		if (currentSelection == null)
			currentSelection = this.workitemAdapter;
		if (currentSelection != null) {
			deleteChilds(currentSelection.getItemCollection());
			entityService.remove(currentSelection.getItemCollection());
			this.setWorkitem(null);
			doReset(event);
		}

	}

	/**
	 * deletes all child workitmes of a workitem - recursive methode call
	 * 
	 * @param parent
	 */
	private void deleteChilds(ItemCollection parent) {
		try {
			String id = parent.getItemValueString("$uniqueid");

			String sQuery = null;
			sQuery = "SELECT";
			sQuery += " wi FROM Entity as wi JOIN wi.textItems as t "
					+ "WHERE ";
			sQuery += " t.itemName = '$uniqueidref' and t.itemValue = '" + id
					+ "'";

			Collection<ItemCollection> col = entityService.findAllEntities(
					sQuery, 0, -1);

			for (ItemCollection aworkitem : col) {
				// recursive method call
				deleteChilds(aworkitem);
				// remove workitem
				entityService.remove(aworkitem);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	/**
	 * resets the current project list and projectMB
	 * 
	 * @return
	 */
	public void doReset(ActionEvent event) {
		workitems = null;
		row = 0;
	}

	/**
	 * refreshes the current workitem list. so the list will be loaded again.
	 * but start pos will not be changed!
	 */
	public void doRefresh(ActionEvent event) {
		workitems = null;
	}

	public void doSwitchToWorklistByAuthor(ActionEvent event) {
		queryType = this.QUERY_WORKLIST_BY_AUTHOR;
		doReset(event);
	}

	public void doSwitchToWorklistByWriteAccess(ActionEvent event) {
		queryType = this.QUERY_WORKLIST_BY_WRITEACCESS;
		doReset(event);
	}

	public void doSwitchToWorklistByOwner(ActionEvent event) {
		queryType = this.QUERY_WORKLIST_BY_OWNER;
		doReset(event);
	}

	public void doSwitchToWorklistByCreator(ActionEvent event) {
		queryType = this.QUERY_WORKLIST_BY_CREATOR;
		doReset(event);
	}

	public void doSwitchToWorklistAll(ActionEvent event) {
		queryType = this.QUERY_WORKLIST_ALL;
		doReset(event);
	}

	public void doSwitchToSearchlist(ActionEvent event) {
		queryType = this.QUERY_SEARCH;
		doReset(event);
	}

	public void doLoadNext(ActionEvent event) {
		row = row + maxSearchResult;
		workitems = null;
	}

	public void doLoadPrev(ActionEvent event) {
		row = row - maxSearchResult;
		if (row < 0)
			row = 0;
		workitems = null;
	}

	public List<ItemCollectionAdapter> getWorkitems() {
		if (workitems == null)
			loadWorkItemList();
		return workitems;
	}
	

	public void setWorkitems(ArrayList<ItemCollectionAdapter> workitems) {
		this.workitems = workitems;
	}

	/***************************************************************************
	 * Navigation
	 */

	public int getRow() {
		return row;
	}

	public boolean isEndOfList() {
		return endOfList;
	}

	/**
	 * this method loads the workitems depending to the current query type The
	 * queryType is set during the getter Methods getWorkList, getStatusList and
	 * getIssueList For each query type a coresponding method is implmented by
	 * the issueService.
	 * 
	 * Caching: ======== The isDirty Flag is diabled because we recognize that
	 * in cases where people working online on share workitems the worklist is
	 * not uptodate. The only way to go back to a caching mechanism would be to
	 * place a refresh-button into the worklist pages.
	 * 
	 * @see org.imixs.shareyouwork.business.WorkitemServiceBean
	 */
	private void loadWorkItemList() {
		workitems = new ArrayList<ItemCollectionAdapter>();
		Collection<ItemCollection> col = null;
		try {
			long lTime = System.currentTimeMillis();

			switch (queryType) {
			case QUERY_WORKLIST_BY_OWNER:
				col = findWorkitemsByOwner(row, maxSearchResult);
				break;
			case QUERY_WORKLIST_BY_CREATOR:
				col = findWorkitemsByCreator(row, maxSearchResult);
				break;

			case QUERY_WORKLIST_BY_AUTHOR:
				col = findWorkitemsByAuthor(row, maxSearchResult);
				break;

			case QUERY_WORKLIST_BY_WRITEACCESS:
				col = findWorkitemsByWriteAccess(row, maxSearchResult);
				break;

			case QUERY_SEARCH:
				col = findWorkitemsByQuery(getSearchQuery(), row,
						maxSearchResult);
				break;

			// default find all
			default:
				col = findAllWorkitems(row, maxSearchResult);
			}

			lTime = System.currentTimeMillis() - lTime;
			System.out.println("  loadWorkItemList (" + lTime + " ms)");

			endOfList = col.size() < maxSearchResult;
			for (ItemCollection aworkitem : col) {
				workitems.add(new ItemCollectionAdapter(aworkitem));
			}
		} catch (Exception ee) {
			ee.printStackTrace();
		}

		// reset current workitem for detail views
		setWorkitem(null);
	}

	private List<ItemCollection> findWorkitemsByQuery(String query, int row,
			int count) {
		ArrayList<ItemCollection> workitemList = new ArrayList<ItemCollection>();

		if (query == null || "".equals(query))
			return workitemList;
		Collection<ItemCollection> col = entityService.findAllEntities(query,
				row, count);
		workitemList.addAll(col);
		return workitemList;
	}

	/**
	 * Returns a collection of workitems where current user is owner (namOwner)
	 * 
	 * @param row
	 * @param count
	 * @return
	 * @throws Exception
	 */
	private List<ItemCollection> findWorkitemsByOwner(int row, int count)
			throws Exception {
		ArrayList<ItemCollection> workList = new ArrayList<ItemCollection>();
		Collection<ItemCollection> col = workflowService.getWorkListByOwner(
				null, row, count, type, getSortOrder());
		workList.addAll(col);

		return workList;

	}

	/**
	 * Returns a collection representing the worklist for the current user
	 * 
	 * @param row
	 * @param count
	 * @return
	 * @throws Exception
	 */
	private List<ItemCollection> findWorkitemsByAuthor(int row, int count)
			throws Exception {
		ArrayList<ItemCollection> workList = new ArrayList<ItemCollection>();
		Collection<ItemCollection> col = workflowService.getWorkList(null, row,
				count, type, getSortOrder());
		workList.addAll(col);

		return workList;

	}

	/**
	 * Returns a collection representing the worklist containing all workitems
	 * where the current user or one of its roles is in the writeAccess table
	 * 
	 * @param row
	 * @param count
	 * @return
	 * @throws Exception
	 */
	private List<ItemCollection> findWorkitemsByWriteAccess(int row, int count)
			throws Exception {
		ArrayList<ItemCollection> workList = new ArrayList<ItemCollection>();
		Collection<ItemCollection> col = workflowService
				.getWorkListByWriteAccess(row, count, type, getSortOrder());
		workList.addAll(col);
		return workList;
	}

	/**
	 * Returns a collection representing the worklist for the current user
	 * 
	 * @param row
	 * @param count
	 * @return
	 * @throws Exception
	 */
	private List<ItemCollection> findWorkitemsByCreator(int row, int count)
			throws Exception {
		ArrayList<ItemCollection> workList = new ArrayList<ItemCollection>();
		Collection<ItemCollection> col = workflowService.getWorkListByCreator(
				null, row, count, type, getSortOrder());
		workList.addAll(col);
		return workList;
	}

	/**
	 * Returns a collection of all Woritems independent of the current user
	 * name!
	 * 
	 * @param row
	 * @param count
	 * @return
	 */
	private List<ItemCollection> findAllWorkitems(int row, int count) {
		ArrayList<ItemCollection> teamList = new ArrayList<ItemCollection>();

		String sQuery = "SELECT wi FROM Entity AS wi " + " WHERE wi.type= '"
				+ getType() + "' ORDER BY wi.modified desc";

		Collection<ItemCollection> col = entityService.findAllEntities(sQuery,
				row, count);

		teamList.addAll(col);

		return teamList;
	}

	/**
	 * returns the last workflow result to control the navigation flow if no
	 * result is found show_worklist will be returned
	 * 
	 * @return
	 */
	public String getWorkflowResult() {
		if (workitemItemCollection == null)
			return "show_worklist";

		String sResult = workitemItemCollection
				.getItemValueString("txtworkflowresultmessage");
		if (sResult == null || "".equals(sResult))
			return "show_worklist";
		else
			return sResult;

	}

	public Map getItem() throws Exception {
		// getWorkitem();
		return workitemAdapter.getItem();
	}

	public Map getItemList() throws Exception {
		// getWorkitem();
		return workitemAdapter.getItemList();
	}

	public Map getItemListArray() throws Exception {
		// getWorkitem();
		return workitemAdapter.getItemListArray();
	}

	/**
	 * returns a arrayList of Activities to the corresponidng processiD of the
	 * current Worktiem. The Method returns the activities corresponding to the
	 * worktiems modelVersionID
	 * 
	 * @return
	 */
	public ArrayList<ItemCollectionAdapter> getActivities() {
		activityList = new ArrayList<ItemCollectionAdapter>();

		if (workitemItemCollection == null)
			return activityList;

		int processId = workitemItemCollection
				.getItemValueInteger("$processid");
		String sversion = workitemItemCollection
				.getItemValueString("$modelversion");

		// get Workflow-Activities by version if provided by the workitem
		List<ItemCollection> col;
		if (sversion != null && !"".equals(sversion))
			col = modelService
					.getPublicActivitiesByVersion(processId, sversion);
		else
			// return activities by defined modelversion
			col = modelService.getPublicActivitiesByVersion(processId,
					getModelVersion());
		for (ItemCollection aworkitem : col) {
			activityList.add(new ItemCollectionAdapter(aworkitem));
		}
		return activityList;
	}

	/**
	 * returns a arrayList of start processes. A Start process is the first
	 * ProcessEntity in a Process group of a model
	 * 
	 * The method only lookups the processEntities once!
	 * 
	 * @return
	 */
	public ArrayList<ItemCollectionAdapter> getStartProcessList() {
		if (startProcessList != null)
			return startProcessList;

		// build startProcessList frist time...
		startProcessList = new ArrayList<ItemCollectionAdapter>();

		// get ProcessEntities by version
		List<ItemCollection> col;
		col = modelService.getAllStartProcessEntitiesByVersion(this
				.getModelVersion());

		for (ItemCollection aworkitem : col) {
			startProcessList.add(new ItemCollectionAdapter(aworkitem));
		}
		return startProcessList;
	}

	/**
	 * indicates if a workitem was processed before by the workflowService
	 * 
	 * @return
	 */
	public boolean isNewWorkitem() {
		try {
			return (!workitemItemCollection.hasItem("numlastactivityid"));
		} catch (Exception e) {
			return true;
		}
	}

	/**
	 * This method can be used to add a Error Messege to the Application Context
	 * during an actionListener Call. Typical this method is used in the
	 * doProcessWrktiem method to display a processing exception to the user.
	 * The method expects the Ressoruce bundle name and the message key inside
	 * the bundle.
	 * 
	 * @param ressourceBundleName
	 * @param messageKey
	 * @param param
	 */
	public void addMessage(String ressourceBundleName, String messageKey,
			Object param) {
		FacesContext context = FacesContext.getCurrentInstance();
		Locale locale = context.getViewRoot().getLocale();

		ResourceBundle rb = ResourceBundle.getBundle(ressourceBundleName,
				locale);
		String msgPattern = rb.getString(messageKey);
		String msg = msgPattern;
		if (param != null) {
			Object[] params = { param };
			msg = MessageFormat.format(msgPattern, params);
		}
		FacesMessage facesMsg = new FacesMessage(FacesMessage.SEVERITY_ERROR,
				msg, msg);
		context.addMessage(null, facesMsg);
	}

}
