package org.imixs.workflow.jee.jsf.util;

import java.util.Collection;
import java.util.Map;
import java.util.Vector;

import javax.ejb.EJB;

import org.imixs.workflow.ItemCollection;
import org.imixs.workflow.util.ItemCollectionAdapter;

/**
 * This BLOBWorkitemController is used to store large objects into a single
 * ItemCollection mapped to a EntityBean. The BLOBWorkitemController supports
 * also the management of file attachment inside an ItemCollection. So the
 * BLOBWorkitemController can be used to save large data objects or files into a Workitem
 * (called BlobWorkitems). The BlobWorkitem is always bounded to a parent
 * workitem by its referrer id ($uniqueidRef). An application can implement a
 * lazy loading for BLOBWorkitems. The read- and write access settings of
 * BLOBWorkitems are always synchronized to the settings of the parent workitem.
 * Before the BlobWorkitem can be accessed the workitem needs to be loaded by
 * the load() method. The Data can be accessed by the embedded Itemcollection
 * through the method getWorkitem(). The BlobWorkitem can be saved by calling
 * the save() method. Both - the load() and the save() method expect the Parent
 * ItemCollection where the BlobWorkitem should be bound to.
 * 
 * @version 0.0.1
 * @author rsoika
 * 
 */
public class BLOBWorkitemController {

	@EJB
	org.imixs.workflow.jee.ejb.EntityService entityService;

	private ItemCollection blobWorkitem = null;
	protected ItemCollectionAdapter workitemAdapter = null;

	public BLOBWorkitemController() {
		super();

		//set the workitemAdapter
		clear();
	}

		
	/**
	 * update the workitemAdapter and the blobworkitem reference
	 * @param aItemcol
	 */
	public void setWorkitem(ItemCollection aItemcol) {
		blobWorkitem =aItemcol;
		workitemAdapter = new ItemCollectionAdapter(blobWorkitem);
	}
	/**
	 * returns the ItemCollection for the curren BlobWorkitem object.
	 * 
	 * @return
	 */
	public ItemCollection getWorkitem() {
		return blobWorkitem;
	}

	/**
	 * Returns a list of file names attached to the current BlobWorkitem. File
	 * Attachments can be added using the method addFile().
	 * 
	 * @return
	 */
	public String[] getFiles() {
		return workitemAdapter.getFiles();
		
	}
	
	/**
	 * Returns a Vector of file names attached to the current BlobWorkitem.
	 * 
	 * @return
	 */
	public Vector<String> getFileList() {
		Vector<String> vFileList = new Vector<String>();
		String[] sFileList = getFiles();
		for (String aName : sFileList)
			vFileList.add(aName);

		return vFileList;
	}
	
	
	/**
	 * Removes the connection to the parend workitem and clear the
	 * itemCollection
	 */
	public void clear() {
		setWorkitem(new ItemCollection());
	}

	/**
	 * This method saves the current BlobWorkitem. Therefore the method copies
	 * the read- and write access list from the given parent workitem into the
	 * BlobWorkitem before save.
	 * 
	 * So this method should be called after a WorkflowProcessing step to update
	 * the read- and write access identically to the parentWorkitem
	 * <p>
	 * The method did not save the blobWorkitem if the parent workitem has no $unqiueID!
	 * <p>
	 * 
	 * @throws Exception
	 */
	public void save(ItemCollection parentWorkitem) throws Exception {

		if (blobWorkitem != null && parentWorkitem != null) {
			
			// verifiy if a uniqueid is still defined. If not return without saving!
			if ("".equals(parentWorkitem
					.getItemValueString("$uniqueID")))
				return;

			// Update Read and write access list from parent workitem
			Vector vAccess = parentWorkitem
					.getItemValue("$ReadAccess");
			blobWorkitem.replaceItemValue("$ReadAccess",
					vAccess);

			vAccess = parentWorkitem.getItemValue("$WriteAccess");
			blobWorkitem.replaceItemValue("$WriteAccess",
					vAccess);

			blobWorkitem.replaceItemValue("$uniqueidRef", parentWorkitem
					.getItemValueString("$uniqueID"));
			blobWorkitem.replaceItemValue("type", "workitemlob");
			// Update BlobWorkitem
			blobWorkitem=entityService.save(blobWorkitem);		
			// update adapter
			setWorkitem(blobWorkitem);

		}
	}

	/**
	 * This method adds a single file to the ItemCollection. files will be
	 * stored into the property $file.
	 * 
	 * @param data
	 *            - byte array with file data
	 * @param fileName
	 *            - name of the file attachment
	 * @param contentType
	 *            - the contenttype (e.g. 'Text/HTML')
	 * @throws Exception
	 */
	public void addFile(byte[] data, String fileName, String contentType)
			throws Exception {
		
		workitemAdapter.addFile(data, fileName, contentType);
		
	}

	/**
	 * This method removes a single file attachment from the BlobWorkitem
	 * 
	 * @throws Exception
	 */
	public void removeFile(String aFilename) throws Exception {
		workitemAdapter.removeFile(aFilename);
	}

	/**
	 * Loads the BlobWorkitem of a given parent Workitem. The BlobWorkitem is
	 * identified by the $unqiueidRef. If no BlobWorkitem still exists the
	 * method creates a new empty BlobWorkitem which can be saved later.
	 * 
	 * @param itemCol
	 *            - parent workitem where the BlobWorkitem will be attached to
	 * @throws Exception
	 */
	public void load(ItemCollection itemCol) throws Exception {

		String sUniqueID = itemCol.getItemValueString("$uniqueid");

		// search entity...
		String sQuery = " SELECT lobitem FROM Entity as lobitem"
				+ " join lobitem.textItems as t2"
				+ " WHERE lobitem.type = 'workitemlob'"
				+ " AND t2.itemName = '$uniqueidref'" + " AND t2.itemValue = '"
				+ sUniqueID + "'";

		Collection<ItemCollection> itemcol = entityService.findAllEntities(
				sQuery, 0, 1);
		if (itemcol != null && itemcol.size() > 0) {

			blobWorkitem = itemcol.iterator().next();
		} else {
			blobWorkitem = new ItemCollection();
			blobWorkitem.replaceItemValue("$uniqueidRef", sUniqueID);
			blobWorkitem.replaceItemValue("type", "workitemlob");
		}

		// update adapter
		setWorkitem(blobWorkitem);
	}

	/**
	 * returns a workitemAdapter Item field
	 * 
	 * @return
	 * @throws Exception
	 */
	public Map getItem() throws Exception {
		return workitemAdapter.getItem();
	}

	/**
	 * returns a workitemAdapter ItemList
	 * 
	 * @return
	 * @throws Exception
	 */
	public Map getItemList() throws Exception {
		return workitemAdapter.getItemList();
	}

	/**
	 * returns a workitemAdapter ItemList Array
	 * 
	 * @return
	 * @throws Exception
	 */
	public Map getItemListArray() throws Exception {
		return workitemAdapter.getItemListArray();
	}

}
