/*******************************************************************************
 *  Imixs IX Workflow Technology
 *  Copyright (C) 2001, 2008 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika
 *******************************************************************************/
package org.imixs.workflow.webservices.soap;

import org.imixs.workflow.xml.EntityCollection;
import org.imixs.workflow.xml.XMLItemCollection;

/**
 * Service Interface for the WorkflowManager. This Service Interface supports methods
 * to create, process and find process instantces. This interface should be used
 * by a workflow service server component. Some methods are mapped directly to
 * the org.imixs.workflow.WorkflowManager Interface. In addition there are
 * getter methods defined which should be used by workflow clients to display a
 * set of workitems.
 * 
 * @author Ralph Soika
 * @version 1.3
 */

public interface WorkflowManagerService {

	

	/**
	 * This method finds and returns an existing workitem (Map) by the unique or
	 * workitem ID. The implementation has to accept both the transfer of a
	 * $uniqueID and a $workItemID. Therefore the $uniqueID and $workitemID must
	 * be unique in the wokflow system the web service belongs to.The web
	 * service has to care that the CallerPrincipal has read access rights for
	 * the workitem to be returned.
	 * If the caller has insufficient rights to read the requested workitem, the
	 * method returns an emtpy worktiem or "null" and dose not return a fault
	 * exception. This is for security reasons so a client can not "test" if a
	 * workitem exists.
	 * 
	 * @param uniqueid
	 * @param attrlist
	 * @return workitem
	 * @throws Exception
	 */
	public XMLItemCollection getWorkItem(String uniqueid)
			throws Exception;

	/**
	 * This method processes the forwarded workitem (Map) through the specified
	 * ActivityInstance. For this purpose, the workitem is processed by the
	 * workflowManager the Web Service belongs to. The Web Service has to
	 * identify the workitem by its $uniqueID attribute and processes the
	 * workitem with the defined activityID. After that the workitem should be
	 * stored into the workflow Systemt by the implemented persistence strategy.
	 * The method returns the processed workitem (Map) which contains the new
	 * attributes assigend by the workflowmanager during processing. The web
	 * service has to care that the CallerPrincipal has write access rights for
	 * the workitem to be procressed.
	 * 
	
	 * If an attribute currently existing in the stored workitem is not
	 * supported in the workitem thranfered by the method the service should not
	 * remove this item. So if an attribute is emtpy the current attribute will
	 * be changed but if the attribute is not transmitted the workitem will not
	 * be changed or removed. To remove a attribute of a workitem permanently
	 * use the plugin technology.
	 * 
	 * If the caller has insufficient rights to write the workitem, the method
	 * returns a fault exception that the user has insufficient write access.
	 * 
	 * @param workitem
	 * @param activityid
	 * @param attrlist
	 * @return
	 * @throws Exception
	 */
	public XMLItemCollection processWorkItem(XMLItemCollection workitem, int activityid) throws Exception;

	/**
	 * This method removes a workitem. The is identified by the unique or
	 * workitem ID. The implementation has to accept both the transfer of a
	 * $uniqueID and a $workItemID. Therefore the $uniqueID and $workitemID must
	 * be unique in the wokflow system the web service belongs to. If a
	 * $workitemID is supported the method should also remove all existing
	 * versions of this workitem. The web service has to care that the
	 * CallerPrincipal has write access rights for the workitem to be
	 * procressed.
	 * 
	 * The method dose not return a value. The method returns a fault exception
	 * if the user has insufficient access to remove the workitem
	 * 
	 * @param uniqueid
	 * @throws Exception
	 */
	public void removeWorkItem(String uniqueid) throws Exception;

	/**
	 * This method returns a collection of workitems (Maps) belonging to a
	 * specified name. The name is a username or role contained in the
	 * nameworkflowWriteAccess attribute of the workitem. The web service should
	 * return a subset of a collection if the start and count parameters differ
	 * form the value -1. The web service has to care that the CallerPrincipal
	 * has read access rights for each workitem to be returned.
	 * 
	
	 * The method returns only workitems the call has sufficient read access
	 * for.
	 * 
	 * @param name
	 * @param startpos
	 * @param count
	
	 * @return
	 * @throws Exception
	 */
	public EntityCollection getWorkList(String name, int startpos, int count) throws Exception;

	/**
	 * This method returns a collection of workitems (Maps) belonging to a
	 * specified workflow group. The behaivor is simmilar to the method
	 * getWorkList.
	 * 
	 * @param name
	 * @param startpos
	 * @param count
	 * @param attrlist
	 * @return
	 * @throws Exception
	 */
	public EntityCollection getWorkListByGroup(String name, int startpos,
			int count) throws Exception;

	/**
	 * This method returns a collection of workitems (Maps) belonging to a
	 * specified process id form the workflow model. The behaivor is simmilar to
	 * the method getWorkList.
	 * 
	 * @param aID
	 * @param startpos
	 * @param count
	 * @param attrlist
	 * @return
	 * @throws Exception
	 */
	public EntityCollection getWorkListByProcessID(int aID, int startpos,
			int count) throws Exception;

	/**
	 * This method returns a collection of workitems (Maps) belonging to a
	 * specified creator of a workitem. The behaivor is simmilar to the method
	 * getWorkList.
	 * 
	 * @param name
	 * @param startpos
	 * @param count
	 * @param attrlist
	 * @return
	 * @throws Exception
	 */
	public EntityCollection getWorkListByCreator(String name, int startpos,
			int count) throws Exception;

	

	/**
	 * This method returns a collection of workitems (Maps) belonging to a
	 * specified workitem identified by the attribute $UniqueIDRef.
	 * 
	 * The behaivor of this Mehtod is simmilar to the method getWorkList.
	 * 
	 * @param aref
	 *            A unique reference to another workitem inside a database *
	 * @param startpos
	 * @param count
	 * @param attrlist
	 * @return List of workitems
	 * @throws Exception
	 */
	public EntityCollection getWorkListByRef(String aref, int startpos, int count) throws Exception;
}
