//
// MIT License
//
// Copyright (c) 2024 Incendo
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
//
package org.incendo.cloud.fabric;

import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.tree.CommandNode;
import com.mojang.brigadier.tree.RootCommandNode;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import net.fabricmc.fabric.api.client.command.v2.ClientCommandManager;
import net.fabricmc.fabric.api.client.command.v2.ClientCommandRegistrationCallback;
import net.fabricmc.fabric.api.client.command.v2.FabricClientCommandSource;
import net.fabricmc.fabric.api.client.networking.v1.ClientPlayConnectionEvents;
import net.fabricmc.fabric.api.command.v2.CommandRegistrationCallback;
import net.minecraft.class_2168;
import net.minecraft.class_2170;
import net.minecraft.class_2172;
import net.minecraft.class_310;
import net.minecraft.class_634;
import net.minecraft.class_7157;
import org.checkerframework.checker.nullness.qual.MonotonicNonNull;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.incendo.cloud.Command;
import org.incendo.cloud.brigadier.CloudBrigadierCommand;
import org.incendo.cloud.component.CommandComponent;
import org.incendo.cloud.internal.CommandRegistrationHandler;
import org.incendo.cloud.minecraft.modded.ModdedCommandMetaKeys;
import org.incendo.cloud.minecraft.modded.internal.ContextualArgumentTypeProvider;

import static org.incendo.cloud.brigadier.util.BrigadierUtil.buildRedirect;

/**
 * A registration handler for Fabric API.
 *
 * <p>Subtypes exist for client and server commands.</p>
 *
 * @param <C> command sender type
 * @param <S> native sender type
 */
abstract class FabricCommandRegistrationHandler<C, S extends class_2172> implements CommandRegistrationHandler<C> {

    private @MonotonicNonNull FabricCommandManager<C, S> commandManager;

    void initialize(final FabricCommandManager<C, S> manager) {
        this.commandManager = manager;
    }

    FabricCommandManager<C, S> commandManager() {
        return this.commandManager;
    }

    static class Client<C> extends FabricCommandRegistrationHandler<C, FabricClientCommandSource> {

        private final Set<Command<C>> registeredCommands = ConcurrentHashMap.newKeySet();
        private volatile boolean registerEventFired = false;

        @Override
        void initialize(final FabricCommandManager<C, FabricClientCommandSource> manager) {
            super.initialize(manager);
            ClientCommandRegistrationCallback.EVENT.register(this::registerCommands);
            ClientPlayConnectionEvents.DISCONNECT.register(($, $$) -> this.registerEventFired = false);
        }

        @Override
        public boolean registerCommand(final @NonNull Command<C> command) {
            this.registeredCommands.add(command);
            if (this.registerEventFired) {
                final class_634 connection = class_310.method_1551().method_1562();
                if (connection == null) {
                    throw new IllegalStateException("Expected connection to be present but it wasn't!");
                }
                final CommandDispatcher<FabricClientCommandSource> dispatcher = ClientCommandManager.getActiveDispatcher();
                if (dispatcher == null) {
                    throw new IllegalStateException("Expected an active dispatcher!");
                }
                ContextualArgumentTypeProvider.withBuildContext(
                        this.commandManager(),
                        class_7157.method_46722(connection.method_29091(), connection.method_45735()),
                        false,
                        () -> this.registerClientCommand(dispatcher, command)
                );
            }
            return true;
        }

        public void registerCommands(
                final CommandDispatcher<FabricClientCommandSource> dispatcher,
                final class_7157 commandBuildContext
        ) {
            this.registerEventFired = true;
            ContextualArgumentTypeProvider.withBuildContext(
                    this.commandManager(),
                    commandBuildContext,
                    true,
                    () -> {
                        for (final Command<C> command : this.registeredCommands) {
                            this.registerClientCommand(dispatcher, command);
                        }
                    }
            );
        }

        private void registerClientCommand(
                final CommandDispatcher<FabricClientCommandSource> dispatcher,
                final Command<C> command
        ) {
            final RootCommandNode<FabricClientCommandSource> rootNode = dispatcher.getRoot();
            final CommandComponent<C> component = command.rootComponent();
            final CommandNode<FabricClientCommandSource> baseNode = this.commandManager()
                    .brigadierManager()
                    .literalBrigadierNodeFactory()
                    .createNode(
                            component.name(),
                            command,
                            new CloudBrigadierCommand<>(this.commandManager(), this.commandManager().brigadierManager())
                    );

            rootNode.addChild(baseNode);

            for (final String alias : component.alternativeAliases()) {
                rootNode.addChild(buildRedirect(alias, baseNode));
            }
        }
    }

    static class Server<C> extends FabricCommandRegistrationHandler<C, class_2168> {

        private final Set<Command<C>> registeredCommands = ConcurrentHashMap.newKeySet();

        @Override
        void initialize(final FabricCommandManager<C, class_2168> manager) {
            super.initialize(manager);
            CommandRegistrationCallback.EVENT.register(this::registerAllCommands);
        }

        @Override
        public boolean registerCommand(final @NonNull Command<C> command) {
            return this.registeredCommands.add(command);
        }

        private void registerAllCommands(
                final CommandDispatcher<class_2168> dispatcher,
                final class_7157 access,
                final class_2170.class_5364 side
        ) {
            this.commandManager().registrationCalled();
            ContextualArgumentTypeProvider.withBuildContext(
                    this.commandManager(),
                    access,
                    true,
                    () -> {
                        for (final Command<C> command : this.registeredCommands) {
                            /* Only register commands in the declared environment */
                            final Commands.CommandSelection env = command.commandMeta().getOrDefault(
                                    ModdedCommandMetaKeys.REGISTRATION_ENVIRONMENT,
                                    Commands.CommandSelection.ALL
                            );

                            if ((env == Commands.CommandSelection.INTEGRATED && !side.includeIntegrated)
                                    || (env == Commands.CommandSelection.DEDICATED && !side.includeDedicated)) {
                                continue;
                            }
                            this.registerCommand(dispatcher.getRoot(), command);
                        }
                    }
            );
        }

        private void registerCommand(final RootCommandNode<class_2168> dispatcher, final Command<C> command) {
            final CommandComponent<C> component = command.rootComponent();
            final CommandNode<class_2168> baseNode = this.commandManager()
                    .brigadierManager()
                    .literalBrigadierNodeFactory()
                    .createNode(
                            component.name(),
                            command,
                            new CloudBrigadierCommand<>(this.commandManager(), this.commandManager().brigadierManager())
                    );

            dispatcher.addChild(baseNode);

            for (final String alias : component.alternativeAliases()) {
                dispatcher.addChild(buildRedirect(alias, baseNode));
            }
        }
    }
}
