/*
 * Copyright 2019-2021 indunet.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.indunet.fastproto;

import lombok.NonNull;
import lombok.val;
import org.indunet.fastproto.graph.Resolver;
import org.indunet.fastproto.pipeline.Pipeline;
import org.indunet.fastproto.pipeline.PipelineContext;

/**
 * FastProto API.
 *
 * @author Deng Ran
 * @since 1.0.0
 */
public class FastProto {
    /**
     * Convert byte array into object.
     *
     * @param bytes      byte array
     * @param clazz class of deserialized object
     * @return deserialize object
     */
    public static <T> T parse(@NonNull byte[] bytes, @NonNull Class<T> clazz) {
        val graph = Resolver.resolve(clazz);
        val context = PipelineContext.builder()
                .bytes(bytes)
                .clazz(clazz)
                .graph(graph)
                .build();

        Pipeline.getDecodeFlow()
                .process(context);

        return context.getObject(clazz);
    }

    /**
     * Helps developers parse binary data through chain api.
     *
     * @param bytes binary data need parsing
     * @return Decoder object which supplies chain api
     */
    public static Decoder parse(byte[] bytes) {
        return new Decoder(bytes);
    }

    /**
     * Convert object into byte array.
     *
     * @param object serialized object
     * @return byte array
     */
    public static byte[] toBytes(Object object) {
        val graph = Resolver.resolve(object.getClass());
        val context = PipelineContext.builder()
                .object(object)
                .clazz(object.getClass())
                .byteBuffer(new ByteBuffer())
                .graph(graph)
                .build();

        Pipeline.getEncodeFlow()
                .process(context);

        return context.getByteBuffer().getBytes();
    }

    /**
     * Convert object into byte array.
     *
     * @param object serialized object
     * @param length the length of byte array
     * @return byte array
     */
    public static byte[] toBytes(Object object, int length) {
        val bytes = new byte[length];

        toBytes(object, bytes);

        return bytes;
    }

    /**
     * Convert object into byte array.
     *
     * @param object serialized object
     * @param buffer write result into buffer
     * @return void
     */
    public static void toBytes(Object object, byte[] buffer) {
        val graph = Resolver.resolve(object.getClass());
        val context = PipelineContext.builder()
                .object(object)
                .clazz(object.getClass())
                .bytes(buffer)
                .byteBuffer(new ByteBuffer(buffer))
                .graph(graph)
                .build();

        Pipeline.getEncodeFlow()
                .process(context);
    }

    /**
     * Helps developers create binary data through chain api.
     *
     * @return Encoder object which supplies chain api.
     */
    public static Encoder create() {
        return new Encoder();
    }

    /**
     * Helps developers create binary data through chain api.
     *
     * @param  length the length of the byte array
     * @return Encoder object which supplies chain api.
     */
    public static Encoder create(int length) {
        return new Encoder(length);
    }
}
