/*
 * Copyright 2014 Google Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.inferred.freebuilder.processor.util;

import static com.google.common.base.Preconditions.checkArgument;

import java.io.Closeable;
import java.io.IOException;
import java.io.Writer;
import java.util.HashSet;
import java.util.Set;

import javax.annotation.processing.Filer;
import javax.annotation.processing.FilerException;
import javax.lang.model.element.Element;
import javax.lang.model.element.PackageElement;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.TypeMirror;

/** Convenience wrapper around the {@link Writer} instances returned by {@link Filer}. */
public class SourceWriter implements Closeable {

  private static final String JAVA_LANG_PACKAGE = "java.lang";
  private final Writer writer;
  private final Set<String> imports = new HashSet<String>();

  SourceWriter(Filer filer, ImpliedClass classToWrite, Element originatingElement)
      throws FilerException {
    String pkg = classToWrite.getEnclosingElement().getQualifiedName().toString();
    try {
      writer = filer
          .createSourceFile(classToWrite.getQualifiedName(), originatingElement)
          .openWriter();
      writer
          .append("// Autogenerated code. Do not modify.\n")
          .append("package ").append(pkg).append(";\n")
          .append("\n");
      imports.add(JAVA_LANG_PACKAGE);
      imports.add(pkg);
    } catch (FilerException e) {
      throw e;
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  SourceWriter(Writer writer) {
    this.writer = writer;
    imports.add(JAVA_LANG_PACKAGE);
  }

  /**
   * Appends formatted text to the source. Formatting is done by {@link String#format},
   * except that:<ul>
   * <li> {@link Package} and {@link PackageElement} instances use their fully-qualified names
   *      (no "package " prefix).
   * <li> {@link Class} and {@link TypeElement} instances use their qualified names where necessary,
   *      or shorter versions if possible.
   * </ul>
   */
  public SourceWriter add(String fmt, Object... args) {
    try {
      Object[] substituteArgs = new Object[args.length];
      for (int i = 0; i < args.length; i++) {
        substituteArgs[i] = substitute(args[i]);
      }
      writer.append(String.format(fmt, substituteArgs));
      return this;
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Appends a formatted line of code to the source. Formatting is done by {@link String#format},
   * except that:<ul>
   * <li> {@link Package} and {@link PackageElement} instances use their fully-qualified names
   *      (no "package " prefix).
   * <li> {@link Class} and {@link TypeElement} instances use their qualified names where necessary,
   *      or shorter versions if possible.
   * </ul>
   */
  public SourceWriter addLine(String fmt, Object... args) {
    return add(fmt + "\n", args);
  }

  @Override
  public void close() {
    try {
      writer.close();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  private Object substitute(Object arg) {
    if (arg instanceof Package) {
      return ((Package) arg).getName();
    } else if (arg instanceof PackageElement) {
      return ((PackageElement) arg).getQualifiedName();
    } else if (arg instanceof Class<?>) {
      Class<?> cls = (Class<?>) arg;
      if (cls.getPackage() != null) {
        return shortTypeName(cls.getCanonicalName(), cls.getPackage().getName());
      } else {
        return cls.getName();
      }
    } else if (arg instanceof TypeElement) {
      TypeElement type = (TypeElement) arg;
      return shortTypeName(type.getQualifiedName(), getPackage(type).getQualifiedName());
    } else if (arg instanceof DeclaredType
        && ((DeclaredType) arg).asElement() instanceof TypeElement) {
      DeclaredType mirror = (DeclaredType) arg;
      checkArgument(isLegalType(mirror), "Cannot write unknown type %s", mirror);
      TypeElement element = (TypeElement) mirror.asElement();
      return shortTypeName(mirror.toString(), getPackage(element).getQualifiedName());
    } else {
      return arg;
    }
  }

  private static boolean isLegalType(TypeMirror mirror) {
    return !(new IsInvalidTypeVisitor().visit(mirror));
  }

  private CharSequence shortTypeName(CharSequence name, CharSequence pkg) {
    if (imports.contains(pkg.toString())) {
      return name.subSequence(pkg.length() + 1, name.length());
    } else {
      return name;
    }
  }

  private static PackageElement getPackage(Element element) {
    Element e = element;
    while (!(e instanceof PackageElement) && e != null) {
      e = e.getEnclosingElement();
    }
    return (PackageElement) e;
  }
}
