/*
 * Copyright 2014 Google Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.inferred.source;

import java.io.Closeable;
import java.io.IOException;
import java.io.Writer;
import java.util.Collection;

import javax.annotation.processing.Filer;
import javax.annotation.processing.FilerException;
import javax.lang.model.element.Element;
import javax.lang.model.element.PackageElement;
import javax.lang.model.element.TypeElement;
import javax.lang.model.util.ElementFilter;
import javax.lang.model.util.Elements;

/** Convenience wrapper around the {@link Writer} instances returned by {@link Filer}. */
public class CompilationUnitWriter implements SourceBuilder, Closeable {

  private final Writer writer;
  private final ImportManager importManager;
  private final SourceStringBuilder source;

  /**
   * Returns a {@link CompilationUnitWriter} for {@code classToWrite}. The file preamble (package
   * and imports) will be generated automatically.
   *
   * @throws FilerException if a Filer guarantee is violated (see the {@link FilerException}
   *     JavaDoc for more information); propagated because this is often seen in GUIDE projects,
   *     so should be downgraded to a warning, whereas runtime exceptions should be flagged as an
   *     internal error to the user
   */
  public CompilationUnitWriter(
      Filer filer,
      Elements elements,
      SourceLevel sourceLevel,
      TypeReference classToWrite,
      Collection<TypeReference> nestedClasses,
      Element originatingElement) throws FilerException {
    try {
      writer = filer
          .createSourceFile(classToWrite.getQualifiedName(), originatingElement)
          .openWriter();
      writer
          .append("// Autogenerated code. Do not modify.\n")
          .append("package ").append(classToWrite.getPackage()).append(";\n")
          .append("\n");
    } catch (FilerException e) {
      throw e;
    } catch (IOException e) {
      throw new RuntimeException(e);
    }

    // Write the source code into an intermediate SourceStringBuilder, as the imports need to be
    // written first, but aren't known yet.
    ImportManager.Builder importManagerBuilder = new ImportManager.Builder();
    importManagerBuilder.addImplicitImport(classToWrite);
    PackageElement pkg = elements.getPackageElement(classToWrite.getPackage());
    for (TypeElement sibling : ElementFilter.typesIn(pkg.getEnclosedElements())) {
      importManagerBuilder.addImplicitImport(TypeReference.to(sibling));
    }
    for (TypeReference nestedClass : nestedClasses) {
      importManagerBuilder.addImplicitImport(nestedClass);
    }
    importManager = importManagerBuilder.build();
    source = new SourceStringBuilder(sourceLevel, importManager, new StringBuilder());
  }

  @Override
  public CompilationUnitWriter add(String fmt, Object... args) {
    source.add(fmt, args);
    return this;
  }

  @Override
  public CompilationUnitWriter addLine(String fmt, Object... args) {
    source.addLine(fmt, args);
    return this;
  }

  @Override
  public SourceLevel getSourceLevel() {
    return source.getSourceLevel();
  }

  @Override
  public void close() {
    try {
      if (!importManager.getClassImports().isEmpty()) {
        for (String classImport : importManager.getClassImports()) {
          writer.append("import ").append(classImport).append(";\n");
        }
        writer.append("\n");
      }
      writer.append(source.toString());
      writer.close();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }
}
