/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.core.paging.impl;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.activemq.artemis.api.core.ActiveMQBuffer;
import org.apache.activemq.artemis.api.core.ActiveMQBuffers;
import org.apache.activemq.artemis.api.core.SimpleString;
import org.apache.activemq.artemis.core.io.SequentialFile;
import org.apache.activemq.artemis.core.io.SequentialFileFactory;
import org.apache.activemq.artemis.core.paging.PagedMessage;
import org.apache.activemq.artemis.core.paging.cursor.LivePageCache;
import org.apache.activemq.artemis.core.paging.cursor.PageSubscriptionCounter;
import org.apache.activemq.artemis.core.paging.impl.PagedMessageImpl;
import org.apache.activemq.artemis.core.persistence.StorageManager;
import org.apache.activemq.artemis.core.server.ActiveMQMessageBundle;
import org.apache.activemq.artemis.core.server.ActiveMQServerLogger;
import org.apache.activemq.artemis.core.server.LargeServerMessage;
import org.apache.activemq.artemis.utils.ConcurrentHashSet;

public final class Page
implements Comparable<Page> {
    private static final boolean isTrace = ActiveMQServerLogger.LOGGER.isTraceEnabled();
    private static final boolean isDebug = ActiveMQServerLogger.LOGGER.isDebugEnabled();
    public static final int SIZE_RECORD = 6;
    private static final byte START_BYTE = 123;
    private static final byte END_BYTE = 125;
    private final int pageId;
    private boolean suspiciousRecords = false;
    private final AtomicInteger numberOfMessages = new AtomicInteger(0);
    private final SequentialFile file;
    private final SequentialFileFactory fileFactory;
    private volatile LivePageCache pageCache;
    private final AtomicInteger size = new AtomicInteger(0);
    private final StorageManager storageManager;
    private final SimpleString storeName;
    private Set<PageSubscriptionCounter> pendingCounters;

    public Page(SimpleString storeName, StorageManager storageManager, SequentialFileFactory factory, SequentialFile file, int pageId) throws Exception {
        this.pageId = pageId;
        this.file = file;
        this.fileFactory = factory;
        this.storageManager = storageManager;
        this.storeName = storeName;
    }

    public int getPageId() {
        return this.pageId;
    }

    public void setLiveCache(LivePageCache pageCache) {
        this.pageCache = pageCache;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized List<PagedMessage> read(StorageManager storage) throws Exception {
        if (isDebug) {
            ActiveMQServerLogger.LOGGER.debug("reading page " + this.pageId + " on address = " + this.storeName);
        }
        if (!this.file.isOpen()) {
            throw ActiveMQMessageBundle.BUNDLE.invalidPageIO();
        }
        ArrayList<PagedMessage> messages = new ArrayList<PagedMessage>();
        this.size.set((int)this.file.size());
        ByteBuffer directBuffer = storage.allocateDirectBuffer((int)this.file.size());
        ActiveMQBuffer fileBuffer = null;
        try {
            this.file.position(0L);
            this.file.read(directBuffer);
            directBuffer.rewind();
            fileBuffer = ActiveMQBuffers.wrappedBuffer(directBuffer);
            fileBuffer.writerIndex(fileBuffer.capacity());
            while (fileBuffer.readable()) {
                int position = fileBuffer.readerIndex();
                byte byteRead = fileBuffer.readByte();
                if (byteRead == 123) {
                    if (fileBuffer.readerIndex() + 4 >= fileBuffer.capacity()) continue;
                    int messageSize = fileBuffer.readInt();
                    int oldPos = fileBuffer.readerIndex();
                    if (fileBuffer.readerIndex() + messageSize < fileBuffer.capacity() && fileBuffer.getByte(oldPos + messageSize) == 125) {
                        PagedMessageImpl msg = new PagedMessageImpl();
                        msg.decode(fileBuffer);
                        byte b = fileBuffer.readByte();
                        if (b != 125) {
                            throw new IllegalStateException("Internal error, it wasn't possible to locate END_BYTE " + b);
                        }
                        msg.initMessage(storage);
                        if (isTrace) {
                            ActiveMQServerLogger.LOGGER.trace("Reading message " + msg + " on pageId=" + this.pageId + " for address=" + this.storeName);
                        }
                        messages.add(msg);
                        continue;
                    }
                    this.markFileAsSuspect(this.file.getFileName(), position, messages.size());
                } else {
                    this.markFileAsSuspect(this.file.getFileName(), position, messages.size());
                }
                break;
            }
        }
        finally {
            if (fileBuffer != null) {
                fileBuffer.byteBuf().unwrap().release();
            }
            storage.freeDirectBuffer(directBuffer);
        }
        this.numberOfMessages.set(messages.size());
        return messages;
    }

    public synchronized void write(PagedMessage message) throws Exception {
        if (!this.file.isOpen()) {
            return;
        }
        ByteBuffer buffer = this.fileFactory.newBuffer(message.getEncodeSize() + 6);
        ActiveMQBuffer wrap = ActiveMQBuffers.wrappedBuffer(buffer);
        wrap.clear();
        wrap.writeByte((byte)123);
        wrap.writeInt(0);
        int startIndex = wrap.writerIndex();
        message.encode(wrap);
        int endIndex = wrap.writerIndex();
        wrap.setInt(1, endIndex - startIndex);
        wrap.writeByte((byte)125);
        buffer.rewind();
        this.file.writeDirect(buffer, false);
        if (this.pageCache != null) {
            this.pageCache.addLiveMessage(message);
        }
        this.numberOfMessages.incrementAndGet();
        this.size.addAndGet(buffer.limit());
        this.storageManager.pageWrite(message, this.pageId);
    }

    public void sync() throws Exception {
        this.file.sync();
    }

    public void open() throws Exception {
        if (!this.file.isOpen()) {
            this.file.open();
        }
        this.size.set((int)this.file.size());
        this.file.position(0L);
    }

    public synchronized void close() throws Exception {
        if (this.storageManager != null) {
            this.storageManager.pageClosed(this.storeName, this.pageId);
        }
        if (this.pageCache != null) {
            this.pageCache.close();
            this.pageCache = null;
        }
        this.file.close();
        Set<PageSubscriptionCounter> counters = this.getPendingCounters();
        if (counters != null) {
            for (PageSubscriptionCounter counter : counters) {
                counter.cleanupNonTXCounters(this.getPageId());
            }
        }
    }

    public boolean isLive() {
        return this.pageCache != null;
    }

    public boolean delete(PagedMessage[] messages) throws Exception {
        if (this.storageManager != null) {
            this.storageManager.pageDeleted(this.storeName, this.pageId);
        }
        if (isDebug) {
            ActiveMQServerLogger.LOGGER.debug("Deleting pageId=" + this.pageId + " on store " + this.storeName);
        }
        if (messages != null) {
            for (PagedMessage msg : messages) {
                if (!msg.getMessage().isLargeMessage()) continue;
                LargeServerMessage lmsg = (LargeServerMessage)msg.getMessage();
                lmsg.decrementDelayDeletionCount();
            }
        }
        try {
            if (this.suspiciousRecords) {
                ActiveMQServerLogger.LOGGER.pageInvalid(this.file.getFileName(), this.file.getFileName());
                this.file.renameTo(this.file.getFileName() + ".invalidPage");
            } else {
                this.file.delete();
            }
            return true;
        }
        catch (Exception e) {
            ActiveMQServerLogger.LOGGER.pageDeleteError(e);
            return false;
        }
    }

    public int getNumberOfMessages() {
        return this.numberOfMessages.intValue();
    }

    public int getSize() {
        return this.size.intValue();
    }

    public String toString() {
        return "Page::pageID=" + this.pageId + ", file=" + this.file;
    }

    @Override
    public int compareTo(Page otherPage) {
        return otherPage.getPageId() - this.pageId;
    }

    protected void finalize() {
        try {
            if (this.file != null && this.file.isOpen()) {
                this.file.close();
            }
        }
        catch (Exception e) {
            ActiveMQServerLogger.LOGGER.pageFinaliseError(e);
        }
    }

    public int hashCode() {
        int prime = 31;
        int result = 1;
        result = 31 * result + this.pageId;
        return result;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        Page other = (Page)obj;
        return this.pageId == other.pageId;
    }

    private void markFileAsSuspect(String fileName, int position, int msgNumber) {
        ActiveMQServerLogger.LOGGER.pageSuspectFile(fileName, position, msgNumber);
        this.suspiciousRecords = true;
    }

    public SequentialFile getFile() {
        return this.file;
    }

    public void addPendingCounter(PageSubscriptionCounter pageSubscriptionCounter) {
        Set<PageSubscriptionCounter> counter = this.getOrCreatePendingCounters();
        this.pendingCounters.add(pageSubscriptionCounter);
    }

    private synchronized Set<PageSubscriptionCounter> getPendingCounters() {
        return this.pendingCounters;
    }

    private synchronized Set<PageSubscriptionCounter> getOrCreatePendingCounters() {
        if (this.pendingCounters == null) {
            this.pendingCounters = new ConcurrentHashSet<PageSubscriptionCounter>();
        }
        return this.pendingCounters;
    }
}

