/*
 * Decompiled with CFR 0.152.
 */
package org.intermine.webservice.server;

import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.util.Properties;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.intermine.web.security.KeySourceException;
import org.intermine.web.security.PublicKeySource;
import org.intermine.webservice.server.JWTVerifier;
import org.json.JSONException;
import org.json.JSONObject;

public class JWTVerifier {
    public static final String VERIFYAUDIENCE = "jwt.verifyaudience";
    public static final String VERIFICATION_STRATEGY = "jwt.verification.strategy";
    public static final String WHITELIST = "jwt.alias.whitelist";
    private static final String EMAIL_CLAIM = "http://wso2.org/claims/emailaddress";
    private static final String NOT_FOR_US = "This token was issued for %s. We are %s";
    private static final String NO_PUBLIC_IDENTITY = "Could not verify audience - no public identity";
    private final Properties options;
    private final PublicKeySource publicKeys;
    private final String strategy;
    private static final Logger LOG = Logger.getLogger(JWTVerifier.class);

    public JWTVerifier(PublicKeySource publicKeys, Properties options) {
        this.publicKeys = publicKeys;
        this.options = options;
        if (publicKeys == null) {
            throw new NullPointerException("publicKeys must not be null");
        }
        if (options == null) {
            throw new NullPointerException("options must not be null");
        }
        this.strategy = this.options.getProperty(VERIFICATION_STRATEGY, "NAMED_ALIAS");
    }

    public Verification verify(String rawString) throws VerificationError {
        String audience;
        String issuer;
        long expiry;
        JSONObject claims;
        JSONObject header;
        Base64 decoder = new Base64();
        if (StringUtils.isBlank((String)rawString)) {
            throw new VerificationError("token is blank");
        }
        String[] pieces = rawString.split("\\.");
        if (pieces.length != 3) {
            throw new VerificationError("Illegal JWT token.");
        }
        try {
            header = new JSONObject(new String(decoder.decode(pieces[0])));
            claims = new JSONObject(new String(decoder.decode(pieces[1])));
            expiry = claims.getLong("exp");
            issuer = claims.getString("iss");
            audience = claims.optString("aud");
        }
        catch (JSONException e) {
            throw new VerificationError("Could not parse token: " + e.getMessage());
        }
        this.verifyAudience(audience);
        long secondsSinceExpiry = System.currentTimeMillis() / 1000L - expiry;
        if (secondsSinceExpiry >= 0L) {
            throw new VerificationError(String.format("This token expired %d seconds ago", secondsSinceExpiry));
        }
        if (!this.canVerify(header, claims, pieces[0] + "." + pieces[1], decoder.decode(pieces[2]))) {
            throw new VerificationError("Could not verify signature.");
        }
        return new Verification(issuer, this.getPrincipal(claims), this.getEmail(claims), null);
    }

    private void verifyAudience(String audience) throws VerificationError {
        boolean verifyAudience = "true".equalsIgnoreCase(this.options.getProperty(VERIFYAUDIENCE, "true"));
        if (verifyAudience && StringUtils.isNotBlank((String)audience)) {
            String self = this.options.getProperty("jwt.publicidentity");
            if (self == null) {
                throw new VerificationError(NO_PUBLIC_IDENTITY);
            }
            if (!self.equals(audience)) {
                throw new VerificationError(String.format(NOT_FOR_US, audience, self));
            }
        }
    }

    private String getKeyAlias(String issuer) {
        return this.options.getProperty("security.keystore.alias." + issuer);
    }

    private String getPrincipal(JSONObject claims) throws VerificationError {
        try {
            String issuer = claims.getString("iss");
            if (this.options.containsKey("jwt.key.sub." + issuer)) {
                return claims.getString(this.options.getProperty("jwt.key.sub." + issuer));
            }
            return claims.getString("sub");
        }
        catch (JSONException e) {
            throw new VerificationError("Could not read principal: " + e.getMessage());
        }
    }

    private String getEmail(JSONObject claims) throws VerificationError {
        try {
            String issuer = claims.getString("iss");
            String emailClaim = this.options.getProperty("jwt.key.email." + issuer, EMAIL_CLAIM);
            if (claims.has(emailClaim)) {
                return claims.getString(emailClaim);
            }
            return null;
        }
        catch (JSONException e) {
            throw new VerificationError("Could not read email: " + e.getMessage());
        }
    }

    private boolean canVerify(JSONObject header, JSONObject claims, String signed, byte[] toVerify) throws VerificationError {
        String algorithm;
        String issuer;
        if (toVerify == null || toVerify.length == 0) {
            throw new VerificationError("Cannot verify an unsigned token");
        }
        try {
            issuer = claims.getString("iss");
            algorithm = header.getString("alg");
        }
        catch (JSONException e) {
            throw new VerificationError("Missing required property: " + e.getMessage());
        }
        if (!algorithm.endsWith("withRSA")) {
            throw new VerificationError("Unsupported signing algorithm: " + algorithm);
        }
        LOG.debug((Object)("Verifying using " + this.strategy + " strategy"));
        try {
            if ("NAMED_ALIAS".equals(this.strategy)) {
                return this.verifyNamedAlias(signed, toVerify, issuer, algorithm);
            }
            if ("ANY".equals(this.strategy)) {
                return this.verifyAnyAlias(signed, toVerify, algorithm);
            }
            if ("WHITELIST".equals(this.strategy)) {
                return this.verifyWhitelistedAliases(signed, toVerify, algorithm);
            }
            throw new VerificationError("Unknown verification strategy: " + this.strategy);
        }
        catch (KeySourceException e) {
            throw new VerificationError("Could not retrieve public key");
        }
    }

    private boolean verifyWhitelistedAliases(String signed, byte[] toVerify, String algorithm) throws VerificationError, KeySourceException {
        Object[] names = this.options.getProperty(WHITELIST, "").split(",");
        LOG.debug((Object)("Using any of " + StringUtils.join((Object[])names, (String)", ") + " to verify JWT"));
        for (PublicKey key : this.publicKeys.getSome((String[])names)) {
            if (!this.verifySignature(key, algorithm, signed, toVerify)) continue;
            return true;
        }
        return false;
    }

    private boolean verifyAnyAlias(String signed, byte[] toVerify, String algorithm) throws VerificationError, KeySourceException {
        for (PublicKey key : this.publicKeys.getAll()) {
            if (!this.verifySignature(key, algorithm, signed, toVerify)) continue;
            return true;
        }
        return false;
    }

    private boolean verifyNamedAlias(String signed, byte[] toVerify, String issuer, String alg) throws VerificationError, KeySourceException {
        String keyAlias = this.getKeyAlias(issuer);
        if (StringUtils.isBlank((String)keyAlias)) {
            throw new VerificationError("Unknown identity issuer: " + issuer);
        }
        LOG.debug((Object)("Using key aliased as " + keyAlias + " to verify JWT"));
        return this.verifySignature(this.publicKeys.get(keyAlias), alg, signed, toVerify);
    }

    private boolean verifySignature(PublicKey key, String algorithm, String signed, byte[] toVerify) throws VerificationError {
        Signature signature;
        try {
            signature = Signature.getInstance(algorithm);
        }
        catch (NoSuchAlgorithmException e) {
            throw new VerificationError(e.getMessage());
        }
        try {
            signature.initVerify(key);
        }
        catch (InvalidKeyException e) {
            throw new VerificationError("Key is invalid. " + e.getMessage());
        }
        try {
            signature.update(signed.getBytes());
        }
        catch (SignatureException e) {
            throw new VerificationError("Error creating signature: " + e.getMessage());
        }
        try {
            return signature.verify(toVerify);
        }
        catch (SignatureException e) {
            throw new VerificationError("Error during verification: " + e.getMessage());
        }
    }
}

