/*
 * CanvasXpress 4.1 - JavaScript Canvas Library
 *
 * Copyright (c) 2009-2010 Isaac Neuhaus
 *
 * imnphd@gmail.com
 *
 * Redistributions of this source code must retain this copyright
 * notice and the following disclaimer.
 *
 * CanvasXpress is licensed under the terms of the Open Source
 * LGPL 3.0 license.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Commercial use is permitted to the extent that this source code
 * do NOT become part of any other Open Source or Commercially licensed
 * development library or toolkit without explicit permission.
 *
 * Network graphs were implemented based on the HeyGraph by Tom Martin
 * <http://www.heychinaski.com>.
 *
 * Thanks to Mingyi Liu for his contributions in extending the Ext panel
 * and adding major capabilities to the network graphs and animations.
 *
 */
window.Canvas = window.Canvas || {};
window.Canvas.Text = {
  equivalentFaces: {
    arial: ["liberation sans", "nimbus sans l", "freesans"],
    "times new roman": ["liberation serif", "linux libertine", "freeserif"],
    "courier new": ["dejavu sans mono", "liberation mono", "nimbus mono l", "freemono"],
    georgia: ["nimbus roman no9 l"],
    helvetica: ["nimbus sans l", "freesans"],
    tahoma: ["dejavu sans", "bitstream vera sans"],
    verdana: ["dejavu sans", "bitstream vera sans"]
  },
  genericFaces: {
    serif: ["times new roman", "georgia", "garamond", "bodoni", "minion web", "itc stone serif", "bitstream cyberbit"],
    "sans-serif": ["arial", "verdana", "trebuchet", "tahoma", "helvetica", "itc avant garde gothic", "univers", "futura", "gill sans", "akzidenz grotesk", "attika", "typiko new era", "itc stone sans", "monotype gill sans 571"],
    monospace: ["courier", "courier new", "prestige", "everson mono"],
    cursive: ["caflisch script", "adobe poetica", "sanvito", "ex ponto", "snell roundhand", "zapf-chancery"],
    fantasy: ["alpha geometrique", "critter", "cottonwood", "fb reactor", "studz"]
  },
  faces: {},
  scaling: 0.962,
  _styleCache: {}
};
(function () {
  var isOpera9 = (window.opera && navigator.userAgent.match(/Opera\/9/)),
    proto = window.CanvasRenderingContext2D ? window.CanvasRenderingContext2D.prototype : document.createElement("canvas").getContext("2d").__proto__,
    ctxt = window.Canvas.Text;
  ctxt.options = {
    fallbackCharacter: " ",
    dontUseMoz: false,
    reimplement: false,
    debug: false,
    autoload: false
  };

  function initialize() {
    var libFileName = "canvas.text.js",
      scripts = document.getElementsByTagName("script"),
      i, j;
    for (i = 0; i < scripts.length; i++) {
      var src = scripts[i].src;
      if (src.indexOf(libFileName) != -1) {
        var parts = src.split("?");
        ctxt.basePath = parts[0].replace(libFileName, "");
        if (parts[1]) {
          var options = parts[1].split("&");
          for (j = options.length - 1; j >= 0; --j) {
            var pair = options[j].split("=");
            ctxt.options[pair[0]] = pair[1]
          }
        }
        break
      }
    }
  }
  initialize();
  var moz = !ctxt.options.dontUseMoz && proto.mozDrawText && !proto.strokeText;
  if (proto.strokeText && !ctxt.options.reimplement) {
    return window._typeface_js = {
      loadFace: function () {}
    }
  }
  function getCSSWeightEquivalent(weight) {
    switch (String(weight)) {
    case "bolder":
    case "bold":
    case "900":
    case "800":
    case "700":
      return "bold";
    case "600":
    case "500":
    case "400":
    default:
    case "normal":
      return "normal"
    }
  }
  function getElementStyle(e) {
    if (document.defaultView && document.defaultView.getComputedStyle) {
      return document.defaultView.getComputedStyle(e, null)
    }
    return e.currentStyle || e.style
  }
  function getXHR() {
    if (!ctxt.xhr) {
      var methods = [function () {
        return new XMLHttpRequest()
      }, function () {
        return new ActiveXObject("Msxml2.XMLHTTP")
      }, function () {
        return new ActiveXObject("Microsoft.XMLHTTP")
      }];
      for (i = 0; i < methods.length; i++) {
        try {
          ctxt.xhr = methods[i]();
          break
        } catch (e) {}
      }
    }
    return ctxt.xhr
  }
  function arrayContains(a, v) {
    var i, l = a.length;
    for (i = l - 1; i >= 0; --i) {
      if (a[i] === v) {
        return true
      }
    }
    return false
  }
  ctxt.lookupFamily = function (family) {
    var faces = this.faces,
      face, i, f, list, equiv = this.equivalentFaces,
      generic = this.genericFaces;
    if (faces[family]) {
      return faces[family]
    }
    if (generic[family]) {
      for (i = 0; i < generic[family].length; i++) {
        if (f = this.lookupFamily(generic[family][i])) {
          return f
        }
      }
    }
    if (!(list = equiv[family])) {
      return false
    }
    for (i = 0; i < list.length; i++) {
      if (face = faces[list[i]]) {
        return face
      }
    }
    return false
  };
  ctxt.getFace = function (family, weight, style) {
    var face = this.lookupFamily(family);
    if (!face) {
      return false
    }
    if (face && face[weight] && face[weight][style]) {
      return face[weight][style]
    }
    if (!this.options.autoload) {
      return false
    }
    var faceName = (family.replace(/[ -]/g, "_") + "-" + weight + "-" + style),
      xhr = this.xhr,
      url = this.basePath + this.options.autoload + "/" + faceName + ".js";
    xhr = getXHR();
    xhr.open("get", url, false);
    xhr.send(null);
    if (xhr.status == 200) {
      eval(xhr.responseText);
      return this.faces[family][weight][style]
    } else {
      throw "Unable to load the font [" + family + " " + weight + " " + style + "]"
    }
    return false
  };
  ctxt.loadFace = function (data) {
    var family = data.familyName.toLowerCase();
    this.faces[family] = this.faces[family] || {};
    this.faces[family][data.cssFontWeight] = this.faces[family][data.cssFontWeight] || {};
    this.faces[family][data.cssFontWeight][data.cssFontStyle] = data;
    return data
  };
  window._typeface_js = {
    faces: ctxt.faces,
    loadFace: ctxt.loadFace
  };
  ctxt.getFaceFromStyle = function (style) {
    var weight = getCSSWeightEquivalent(style.weight),
      families = style.family,
      i, face;
    for (i = 0; i < families.length; i++) {
      if (face = this.getFace(families[i].toLowerCase(), weight, style.style)) {
        return face
      }
    }
    return false
  };
  try {
    proto.font = "10px sans-serif";
    proto.textAlign = "start";
    proto.textBaseline = "alphabetic"
  } catch (e) {}
  proto.parseStyle = function (styleText) {
    if (ctxt._styleCache[styleText]) {
      return this.getComputedStyle(ctxt._styleCache[styleText])
    }
    var style = {},
      computedStyle, families;
    if (!this._elt) {
      this._elt = document.createElement("span");
      this.canvas.appendChild(this._elt)
    }
    this.canvas.font = "10px sans-serif";
    this._elt.style.font = styleText;
    computedStyle = getElementStyle(this._elt);
    style.size = computedStyle.fontSize;
    style.weight = getCSSWeightEquivalent(computedStyle.fontWeight);
    style.style = computedStyle.fontStyle;
    families = computedStyle.fontFamily.split(",");
    for (i = 0; i < families.length; i++) {
      families[i] = families[i].replace(/^["'\s]*/, "").replace(/["'\s]*$/, "")
    }
    style.family = families;
    return this.getComputedStyle(ctxt._styleCache[styleText] = style)
  };
  proto.buildStyle = function (style) {
    return style.style + " " + style.weight + " " + style.size + 'px "' + style.family + '"'
  };
  proto.renderText = function (text, style) {
    var face = ctxt.getFaceFromStyle(style),
      scale = (style.size / face.resolution) * 0.75,
      offset = 0,
      i, chars = text.split(""),
      length = chars.length;
    if (!isOpera9) {
      this.scale(scale, -scale);
      this.lineWidth /= scale
    }
    for (i = 0; i < length; i++) {
      offset += this.renderGlyph(chars[i], face, scale, offset)
    }
  };
  if (isOpera9) {
    proto.renderGlyph = function (c, face, scale, offset) {
      var i, cpx, cpy, outline, action, glyph = face.glyphs[c],
        length;
      if (!glyph) {
        return
      }
      if (glyph.o) {
        outline = glyph._cachedOutline || (glyph._cachedOutline = glyph.o.split(" "));
        length = outline.length;
        for (i = 0; i < length;) {
          action = outline[i++];
          switch (action) {
          case "m":
            this.moveTo(outline[i++] * scale + offset, outline[i++] * -scale);
            break;
          case "l":
            this.lineTo(outline[i++] * scale + offset, outline[i++] * -scale);
            break;
          case "q":
            cpx = outline[i++] * scale + offset;
            cpy = outline[i++] * -scale;
            this.quadraticCurveTo(outline[i++] * scale + offset, outline[i++] * -scale, cpx, cpy);
            break
          }
        }
      }
      return glyph.ha * scale
    }
  } else {
    proto.renderGlyph = function (c, face) {
      var i, cpx, cpy, outline, action, glyph = face.glyphs[c],
        length;
      if (!glyph) {
        return
      }
      if (glyph.o) {
        outline = glyph._cachedOutline || (glyph._cachedOutline = glyph.o.split(" "));
        length = outline.length;
        for (i = 0; i < length;) {
          action = outline[i++];
          switch (action) {
          case "m":
            this.moveTo(outline[i++], outline[i++]);
            break;
          case "l":
            this.lineTo(outline[i++], outline[i++]);
            break;
          case "q":
            cpx = outline[i++];
            cpy = outline[i++];
            this.quadraticCurveTo(outline[i++], outline[i++], cpx, cpy);
            break
          }
        }
      }
      if (glyph.ha) {
        this.translate(glyph.ha, 0)
      }
    }
  }
  proto.getTextExtents = function (text, style) {
    var width = 0,
      height = 0,
      ha = 0,
      face = ctxt.getFaceFromStyle(style),
      i, length = text.length,
      glyph;
    for (i = 0; i < length; i++) {
      glyph = face.glyphs[text.charAt(i)] || face.glyphs[ctxt.options.fallbackCharacter];
      width += Math.max(glyph.ha, glyph.x_max);
      ha += glyph.ha
    }
    return {
      width: width,
      height: face.lineHeight,
      ha: ha
    }
  };
  proto.getComputedStyle = function (style) {
    var p, canvasStyle = getElementStyle(this.canvas),
      computedStyle = {},
      s = style.size,
      canvasFontSize = parseFloat(canvasStyle.fontSize),
      fontSize = parseFloat(s);
    for (p in style) {
      computedStyle[p] = style[p]
    }
    if (typeof s === "number" || s.indexOf("px") != -1) {
      computedStyle.size = fontSize
    } else {
      if (s.indexOf("em") != -1) {
        computedStyle.size = canvasFontSize * fontSize
      } else {
        if (s.indexOf("%") != -1) {
          computedStyle.size = (canvasFontSize / 100) * fontSize
        } else {
          if (s.indexOf("pt") != -1) {
            computedStyle.size = fontSize / 0.75
          } else {
            computedStyle.size = canvasFontSize
          }
        }
      }
    }
    return computedStyle
  };
  proto.getTextOffset = function (text, style, face) {
    var canvasStyle = getElementStyle(this.canvas),
      metrics = this.measureText(text),
      scale = (style.size / face.resolution) * 0.75,
      offset = {
        x: 0,
        y: 0,
        metrics: metrics,
        scale: scale
      };
    switch (this.textAlign) {
    default:
    case null:
    case "left":
      break;
    case "center":
      offset.x = -metrics.width / 2;
      break;
    case "right":
      offset.x = -metrics.width;
      break;
    case "start":
      offset.x = (canvasStyle.direction == "rtl") ? -metrics.width : 0;
      break;
    case "end":
      offset.x = (canvasStyle.direction == "ltr") ? -metrics.width : 0;
      break
    }
    switch (this.textBaseline) {
    case "alphabetic":
      break;
    default:
    case null:
    case "ideographic":
    case "bottom":
      offset.y = face.descender;
      break;
    case "hanging":
    case "top":
      offset.y = face.ascender;
      break;
    case "middle":
      offset.y = (face.ascender + face.descender) / 2;
      break
    }
    offset.y *= scale;
    return offset
  };
  proto.drawText = function (text, x, y, maxWidth, stroke) {
    var style = this.parseStyle(this.font),
      face = ctxt.getFaceFromStyle(style),
      offset = this.getTextOffset(text, style, face);
    this.save();
    this.translate(x + offset.x, y + offset.y);
    if (face.strokeFont && !stroke) {
      this.strokeStyle = this.fillStyle
    }
    this.beginPath();
    if (moz) {
      this.mozTextStyle = this.buildStyle(style);
      this[stroke ? "mozPathText" : "mozDrawText"](text)
    } else {
      this.scale(ctxt.scaling, ctxt.scaling);
      this.renderText(text, style);
      if (face.strokeFont) {
        this.lineWidth = style.size * (style.weight == "bold" ? 0.5 : 0.3)
      }
    }
    this[(stroke || (face.strokeFont && !moz)) ? "stroke" : "fill"]();
    this.closePath();
    this.restore();
    if (ctxt.options.debug) {
      var left = Math.floor(offset.x + x) + 0.5,
        top = Math.floor(y) + 0.5;
      this.save();
      this.strokeStyle = "#F00";
      this.lineWidth = 0.5;
      this.beginPath();
      this.moveTo(left + offset.metrics.width, top);
      this.lineTo(left, top);
      this.moveTo(left - offset.x, top + offset.y);
      this.lineTo(left - offset.x, top + offset.y - style.size);
      this.stroke();
      this.closePath();
      this.restore()
    }
  };
  proto.fillText = function (text, x, y, maxWidth) {
    this.drawText(text, x, y, maxWidth, false)
  };
  proto.strokeText = function (text, x, y, maxWidth) {
    this.drawText(text, x, y, maxWidth, true)
  };
  proto.measureText = function (text) {
    var style = this.parseStyle(this.font),
      dim = {
        width: 0
      };
    if (moz) {
      this.mozTextStyle = this.buildStyle(style);
      dim.width = this.mozMeasureText(text)
    } else {
      var face = ctxt.getFaceFromStyle(style),
        scale = (style.size / face.resolution) * 0.75;
      dim.width = this.getTextExtents(text, style).ha * scale * ctxt.scaling
    }
    return dim
  }
})();
if (_typeface_js && _typeface_js.loadFace) _typeface_js.loadFace({
  "glyphs": {
    "Î¿": {
      "x_min": 41,
      "x_max": 697,
      "ha": 738,
      "o": "m 364 -15 q 130 77 219 -15 q 41 321 41 171 q 131 573 41 474 q 371 672 222 672 q 607 574 518 672 q 697 325 697 477 q 606 79 697 174 q 364 -15 515 -15 m 370 619 q 222 530 268 619 q 177 327 177 441 q 223 125 177 216 q 369 34 269 34 q 515 122 470 34 q 560 325 560 210 q 515 529 560 439 q 370 619 471 619 "
    },
    "S": {
      "x_min": 49.3125,
      "x_max": 626,
      "ha": 685,
      "o": "m 87 208 q 176 89 119 132 q 311 46 232 46 q 448 98 389 46 q 508 227 508 150 q 393 400 508 336 q 181 498 287 448 q 67 688 67 566 q 153 880 67 811 q 363 950 239 950 q 471 936 422 950 q 585 891 520 922 q 559 822 571 858 q 541 757 547 786 l 528 757 q 347 897 493 897 q 226 855 276 897 q 177 742 177 813 q 292 580 177 640 q 509 483 400 532 q 626 286 626 414 q 528 69 626 153 q 295 -15 430 -15 q 163 2 224 -15 q 49 59 102 19 q 67 135 61 104 q 73 205 73 166 l 87 208 "
    },
    "/": {
      "x_min": -35.390625,
      "x_max": 396.078125,
      "ha": 376,
      "o": "m -35 -126 l 333 1025 l 396 1025 l 27 -126 l -35 -126 "
    },
    "Î¤": {
      "x_min": 10.78125,
      "x_max": 698.4375,
      "ha": 711,
      "o": "m 10 838 l 14 884 l 10 926 q 190 926 70 926 q 353 926 310 926 q 533 926 412 926 q 698 926 654 926 q 693 883 693 910 q 693 861 693 870 q 698 834 693 852 q 597 850 653 843 q 493 857 540 857 l 418 857 q 418 767 418 830 q 418 701 418 704 l 418 221 q 432 0 418 122 q 355 8 393 4 q 316 5 337 8 q 283 0 295 3 q 283 132 283 40 q 283 259 283 225 l 283 683 l 283 857 q 10 838 156 857 "
    },
    "Ï•": {
      "x_min": 41,
      "x_max": 942,
      "ha": 986,
      "o": "m 425 -10 q 159 76 277 -10 q 41 316 41 163 q 159 562 41 470 q 425 654 277 654 q 425 838 425 719 q 425 971 425 957 q 456 965 436 968 q 494 963 477 963 q 530 965 511 963 q 568 971 550 968 q 553 654 553 825 q 824 567 706 654 q 942 324 942 481 q 825 79 942 169 q 560 -10 709 -10 q 560 -201 560 -68 q 560 -371 560 -334 q 497 -362 530 -362 q 457 -364 475 -362 q 425 -371 438 -367 q 425 -161 425 -296 q 425 -10 425 -26 m 552 39 q 743 118 677 39 q 809 330 809 198 q 736 523 809 445 q 552 601 664 601 l 552 319 l 552 39 m 432 319 l 432 601 q 248 523 320 601 q 177 330 177 446 q 241 117 177 195 q 432 39 305 39 l 432 319 "
    },
    "y": {
      "x_min": 4.078125,
      "x_max": 651.96875,
      "ha": 651,
      "o": "m 4 653 l 84 647 l 161 653 q 198 536 185 577 q 236 431 210 495 l 356 128 l 464 413 q 541 653 509 537 q 594 647 571 647 q 651 653 620 647 q 408 125 521 388 q 219 -372 295 -137 l 183 -367 q 138 -367 160 -367 q 110 -372 119 -371 l 284 -22 q 4 653 167 294 "
    },
    "â‰ˆ": {
      "x_min": 116,
      "x_max": 999,
      "ha": 1116,
      "o": "m 750 442 q 553 487 686 442 q 369 533 420 533 q 245 506 292 533 q 116 428 199 480 l 116 501 q 241 572 177 545 q 369 600 305 600 q 562 553 430 600 q 750 507 695 507 q 870 534 812 507 q 999 613 928 562 l 999 537 q 874 467 935 493 q 750 442 813 442 m 744 214 q 556 260 688 214 q 369 307 424 307 q 232 273 294 307 q 116 202 170 239 l 116 278 q 242 346 178 320 q 373 372 306 372 q 559 326 437 372 q 750 281 681 281 q 865 305 813 281 q 999 387 917 330 l 999 312 q 876 240 940 266 q 744 214 812 214 "
    },
    "Î ": {
      "x_min": 105.84375,
      "x_max": 909.515625,
      "ha": 1016,
      "o": "m 423 844 l 257 844 q 257 631 257 778 q 257 473 257 484 q 257 235 257 394 q 257 0 257 77 q 224 4 250 0 q 184 8 198 8 q 146 4 173 8 q 114 0 119 0 q 114 239 114 70 q 114 464 114 408 q 114 711 114 620 q 105 926 114 802 q 280 926 163 926 q 455 926 397 926 q 681 926 530 926 q 909 926 833 926 q 900 793 900 870 q 900 659 900 716 l 900 503 q 900 244 900 427 q 900 0 900 62 q 871 4 897 0 q 831 8 845 8 q 795 5 812 8 q 758 0 778 2 q 758 258 758 94 q 758 467 758 421 l 758 729 l 758 844 l 423 844 "
    },
    "Î": {
      "x_min": -41,
      "x_max": 375,
      "ha": 334,
      "o": "m 98 333 l 98 520 q 98 566 98 544 q 93 653 98 588 q 138 647 127 647 q 161 647 148 647 q 228 653 191 647 q 223 555 223 599 q 223 437 223 511 l 223 406 q 223 194 223 337 q 223 0 223 51 q 197 3 212 1 q 161 6 182 6 q 125 3 145 6 q 98 0 104 1 q 98 165 98 51 q 98 333 98 279 m 17 865 q 59 846 41 865 q 77 804 77 828 q 59 761 77 779 q 17 743 41 743 q -23 761 -5 743 q -41 804 -41 779 q -23 846 -41 828 q 17 865 -5 865 m 168 929 q 184 969 175 956 q 220 982 194 982 q 251 971 238 982 q 264 941 264 961 q 251 904 264 919 l 145 743 l 114 743 l 168 929 m 316 865 q 357 846 339 865 q 375 804 375 828 q 357 761 375 779 q 316 743 339 743 q 274 761 290 743 q 258 804 258 779 q 274 846 258 828 q 316 865 290 865 "
    },
    "g": {
      "x_min": 31,
      "x_max": 658,
      "ha": 673,
      "o": "m 79 123 q 110 201 79 169 q 189 252 141 233 l 189 262 q 94 329 127 278 q 62 447 62 380 q 137 610 62 549 q 316 672 213 672 q 412 661 349 672 q 489 651 476 651 l 658 651 l 658 581 q 587 592 623 587 q 526 597 552 597 q 595 458 595 548 q 516 294 595 356 q 334 232 438 232 q 284 235 312 232 q 250 239 257 239 q 203 220 223 239 q 184 173 184 201 q 216 120 184 136 q 289 104 248 104 l 419 104 q 590 56 523 104 q 658 -93 658 9 q 548 -299 658 -226 q 303 -372 439 -372 q 113 -327 195 -372 q 31 -183 31 -283 q 74 -62 31 -110 q 189 8 118 -13 q 109 51 139 25 q 79 123 79 77 m 325 278 q 430 332 393 278 q 468 457 468 386 q 431 575 468 525 q 330 625 395 625 q 227 570 265 625 q 189 447 189 515 q 224 328 189 379 q 325 278 259 278 m 330 -316 q 481 -270 414 -316 q 548 -141 548 -224 q 490 -29 548 -62 q 346 3 432 3 q 195 -36 258 3 q 133 -162 133 -76 q 191 -277 133 -238 q 330 -316 250 -316 "
    },
    "Â²": {
      "x_min": 15,
      "x_max": 403.78125,
      "ha": 485,
      "o": "m 291 743 q 265 830 291 795 q 193 866 239 866 q 117 836 146 866 q 81 760 88 807 l 74 759 q 53 802 66 779 q 31 836 39 824 q 206 901 106 901 q 327 862 273 901 q 382 757 382 824 q 276 567 382 656 q 108 430 171 478 l 286 430 q 343 430 309 430 q 403 436 376 430 l 398 396 l 403 358 l 297 358 l 108 358 l 15 358 l 15 377 q 216 567 141 484 q 291 743 291 650 "
    },
    "Îš": {
      "x_min": 105.84375,
      "x_max": 839.921875,
      "ha": 805,
      "o": "m 256 314 q 256 132 256 250 q 256 0 256 14 q 224 4 250 0 q 185 9 198 9 q 145 4 171 9 q 114 0 119 0 q 114 240 114 71 q 114 465 114 408 q 114 711 114 621 q 105 932 114 802 q 150 928 122 932 q 184 925 179 925 q 227 928 200 925 q 256 932 254 932 l 256 671 l 256 498 q 473 693 372 586 q 673 932 574 800 q 718 930 697 930 q 762 930 739 930 l 820 930 q 594 727 706 830 q 381 522 483 623 q 515 358 456 426 q 653 203 574 290 q 839 -2 732 115 l 732 -2 q 679 -2 702 -2 q 631 -8 656 -2 q 561 92 599 45 q 470 204 523 140 l 256 458 l 256 314 "
    },
    "Ã«": {
      "x_min": 39,
      "x_max": 632,
      "ha": 666,
      "o": "m 398 42 q 590 129 512 42 l 609 126 q 601 90 605 109 q 597 46 597 71 q 486 0 546 14 q 361 -15 426 -15 q 127 73 215 -15 q 39 311 39 162 q 123 562 39 456 q 347 669 207 669 q 551 590 470 669 q 632 387 632 512 l 632 332 q 429 332 551 332 q 307 332 308 332 l 175 332 q 230 127 175 212 q 398 42 285 42 m 214 929 q 266 906 244 929 q 288 854 288 884 q 267 800 288 822 q 217 778 247 778 q 163 800 186 778 q 140 854 140 822 q 161 906 140 884 q 214 929 183 929 m 451 929 q 503 906 483 929 q 524 854 524 884 q 504 799 524 820 q 453 778 484 778 q 399 800 421 778 q 378 854 378 822 q 399 906 378 884 q 451 929 421 929 m 503 390 l 503 436 q 460 563 503 508 q 349 618 418 618 q 228 552 266 618 q 179 390 191 487 l 503 390 "
    },
    "e": {
      "x_min": 41,
      "x_max": 632,
      "ha": 667,
      "o": "m 399 42 q 591 129 513 42 l 607 125 q 600 85 603 104 q 598 44 598 66 q 489 0 551 14 q 363 -15 426 -15 q 128 73 216 -15 q 41 311 41 161 q 125 563 41 455 q 349 672 210 672 q 552 592 473 672 q 632 385 632 513 l 632 332 l 308 332 l 176 332 q 231 126 176 211 q 399 42 286 42 m 503 390 l 503 434 q 461 563 503 508 q 351 618 420 618 q 228 553 265 618 q 178 390 190 489 l 503 390 "
    },
    "ÏŒ": {
      "x_min": 41,
      "x_max": 697,
      "ha": 738,
      "o": "m 364 -15 q 130 77 219 -15 q 41 321 41 171 q 131 573 41 474 q 371 672 222 672 q 607 574 518 672 q 697 325 697 477 q 606 79 697 174 q 364 -15 515 -15 m 515 942 q 548 973 532 963 q 584 984 563 984 q 620 968 607 984 q 634 931 634 953 q 620 896 634 910 q 579 866 606 882 l 382 743 l 335 743 l 515 942 m 370 619 q 222 530 268 619 q 177 327 177 441 q 223 125 177 216 q 369 34 269 34 q 515 122 470 34 q 560 325 560 210 q 515 529 560 439 q 370 619 471 619 "
    },
    "J": {
      "x_min": -69,
      "x_max": 263.109375,
      "ha": 377,
      "o": "m 116 -40 q 127 62 127 9 q 127 184 127 115 q 127 461 127 325 q 127 690 127 598 q 109 932 127 781 q 156 928 127 932 q 189 925 185 925 q 233 927 216 925 q 263 932 250 929 q 263 665 263 843 q 263 457 263 487 l 263 165 q 263 107 263 147 q 263 48 263 68 q 172 -155 263 -88 q -60 -222 81 -222 l -69 -180 q 50 -134 1 -166 q 116 -40 98 -102 "
    },
    "Â»": {
      "x_min": 42.1875,
      "x_max": 532.1875,
      "ha": 595,
      "o": "m 191 322 l 42 606 l 80 644 l 296 322 l 80 0 l 42 37 l 191 322 m 427 322 l 277 606 l 315 644 l 532 322 l 315 0 l 277 37 l 427 322 "
    },
    "Â©": {
      "x_min": 79,
      "x_max": 1037,
      "ha": 1116,
      "o": "m 800 904 q 973 725 909 840 q 1037 481 1037 610 q 894 138 1037 282 q 556 -6 751 -6 q 221 139 363 -6 q 79 481 79 284 q 220 825 79 681 q 558 970 362 970 q 800 904 684 970 m 558 917 q 258 787 385 917 q 132 480 132 658 q 257 175 132 306 q 555 45 382 45 q 855 174 726 45 q 984 478 984 304 q 928 698 984 596 q 771 858 872 800 q 558 917 670 917 m 559 723 q 433 651 475 723 q 392 489 392 580 q 430 319 392 395 q 549 244 469 244 q 651 277 609 244 q 709 368 694 310 l 782 368 q 707 231 773 282 q 551 180 641 180 q 373 267 438 180 q 308 475 308 354 q 373 693 308 603 q 560 784 439 784 q 703 737 643 784 q 777 609 763 690 l 700 609 q 651 691 691 659 q 559 723 612 723 "
    },
    "ÏŽ": {
      "x_min": 39,
      "x_max": 1008,
      "ha": 1046,
      "o": "m 523 526 l 582 526 l 582 305 q 603 117 582 193 q 708 42 625 42 q 833 118 795 42 q 871 293 871 194 q 812 502 871 409 q 651 654 753 594 l 703 648 q 788 654 750 648 q 947 510 886 607 q 1008 304 1008 414 q 929 81 1008 178 q 728 -15 850 -15 q 609 16 664 -15 q 523 104 554 47 q 436 16 490 48 q 316 -15 381 -15 q 115 83 192 -15 q 39 311 39 182 q 98 511 39 419 q 256 654 158 604 q 296 650 272 650 q 342 650 320 650 l 393 654 q 233 501 290 592 q 176 291 176 409 q 214 116 176 191 q 340 42 252 42 q 410 68 381 42 q 451 137 438 94 q 462 216 462 175 q 462 302 462 256 q 462 418 462 352 q 462 526 462 484 l 523 526 m 667 943 q 700 974 685 963 q 735 984 715 984 q 772 969 758 984 q 786 931 786 954 q 771 896 786 912 q 731 866 757 880 l 535 743 l 487 743 l 667 943 "
    },
    "â‰¥": {
      "x_min": 173,
      "x_max": 943,
      "ha": 1116,
      "o": "m 943 462 l 173 195 l 173 266 l 832 491 l 173 717 l 173 787 l 943 522 l 943 462 m 943 26 l 173 26 l 173 93 l 943 93 l 943 26 "
    },
    "^": {
      "x_min": 0,
      "x_max": 382.46875,
      "ha": 395,
      "o": "m 147 978 l 234 978 l 382 743 l 337 743 l 191 875 l 47 743 l 0 743 l 147 978 "
    },
    "Â«": {
      "x_min": 47.640625,
      "x_max": 539,
      "ha": 595,
      "o": "m 149 326 l 303 43 l 264 4 l 47 326 l 264 648 l 303 610 l 149 326 m 386 326 l 539 43 l 499 4 l 283 326 l 499 648 l 539 610 l 386 326 "
    },
    "D": {
      "x_min": 114,
      "x_max": 971,
      "ha": 1021,
      "o": "m 114 466 q 114 704 114 537 q 114 933 114 871 l 206 933 q 343 933 246 933 q 467 933 440 933 q 567 933 542 933 q 695 915 593 933 q 884 774 798 898 q 971 484 971 650 q 835 133 971 263 q 481 4 699 4 l 371 4 l 224 4 l 199 4 q 145 3 169 4 q 114 1 120 2 q 114 240 114 72 q 114 466 114 409 m 391 64 q 709 168 595 64 q 823 476 823 272 q 716 770 823 667 q 420 873 609 873 q 338 873 393 873 q 253 873 284 873 l 253 499 l 253 377 l 253 76 q 322 68 283 72 q 391 64 362 64 "
    },
    "w": {
      "x_min": 4.078125,
      "x_max": 1031.71875,
      "ha": 1026,
      "o": "m 4 653 q 54 647 40 647 q 84 647 68 647 q 161 653 118 647 q 186 540 172 587 q 234 394 200 492 l 322 141 q 488 653 411 386 q 540 647 515 647 q 597 653 566 647 q 624 544 608 601 q 664 420 641 487 l 755 136 l 854 413 q 893 532 875 472 q 925 653 911 592 q 960 647 951 647 q 980 647 969 647 q 1031 653 1001 647 q 941 457 982 555 q 854 235 901 359 q 766 0 807 111 q 739 3 756 0 q 718 7 722 7 q 693 3 709 7 q 672 0 677 0 q 636 127 653 72 q 598 238 619 183 l 507 494 q 423 258 465 383 q 341 0 382 134 q 318 3 329 1 q 291 6 307 6 q 264 3 279 6 q 240 0 249 1 q 159 243 196 140 q 84 446 122 347 q 4 653 46 545 "
    },
    "$": {
      "x_min": 87,
      "x_max": 652,
      "ha": 734,
      "o": "m 136 186 l 142 186 q 209 91 162 119 q 335 49 256 63 l 335 416 q 139 515 192 458 q 87 648 87 573 q 160 819 87 752 q 335 886 234 886 q 322 984 335 935 l 351 979 l 393 984 q 393 924 393 945 q 393 886 393 904 q 499 874 447 886 q 593 834 551 862 q 570 787 580 807 q 544 729 561 767 l 536 729 q 485 810 524 783 q 391 837 446 837 l 391 522 q 597 420 543 480 q 652 279 652 361 q 576 89 652 166 q 389 2 500 12 l 389 -30 q 389 -62 389 -47 q 395 -124 389 -77 l 360 -120 l 322 -124 l 335 0 q 206 13 268 0 q 96 58 144 27 l 136 186 m 335 835 q 232 787 272 825 q 192 685 192 748 q 232 588 192 626 q 335 537 273 551 l 335 835 m 542 231 q 503 337 542 299 q 391 402 464 374 l 391 51 q 501 113 461 66 q 542 231 542 159 "
    },
    "â€§": {
      "x_min": 132,
      "x_max": 304,
      "ha": 437,
      "o": "m 219 636 q 279 611 254 636 q 304 548 304 586 q 280 486 304 512 q 219 461 256 461 q 157 486 183 461 q 132 548 132 511 q 157 610 132 584 q 219 636 182 636 "
    },
    "\\": {
      "x_min": -35.390625,
      "x_max": 394.71875,
      "ha": 376,
      "o": "m -35 1025 l 27 1025 l 394 -126 l 333 -126 l -35 1025 "
    },
    "Î™": {
      "x_min": 107.84375,
      "x_max": 260.875,
      "ha": 377,
      "o": "m 116 465 q 116 711 116 620 q 107 932 116 802 q 153 926 128 926 q 189 926 178 926 q 236 929 208 926 q 260 932 265 932 q 260 788 260 887 q 260 659 260 689 l 260 448 l 260 282 q 260 135 260 238 q 260 0 260 31 q 230 4 257 0 q 189 8 202 8 q 149 5 171 8 q 116 0 128 2 q 116 239 116 70 q 116 465 116 408 "
    },
    "ÎŽ": {
      "x_min": -1.4375,
      "x_max": 1179.875,
      "ha": 1142,
      "o": "m 736 177 l 736 386 q 634 570 697 458 q 531 750 571 681 q 421 932 491 819 q 473 928 439 932 q 511 925 507 925 q 559 927 538 925 q 593 932 580 930 q 652 800 620 866 q 719 676 684 734 l 831 474 q 948 688 891 574 q 1064 932 1005 801 l 1118 926 q 1149 927 1136 926 q 1179 932 1163 929 q 994 627 1076 769 l 875 415 l 875 240 q 875 102 875 198 q 875 0 875 5 q 842 4 866 1 q 802 7 817 7 q 761 4 776 7 q 727 0 746 2 q 736 88 736 36 q 736 177 736 140 m 178 942 q 210 973 195 963 q 246 984 225 984 q 283 968 268 984 q 298 931 298 953 q 283 896 298 910 q 242 866 269 881 l 46 743 l -1 743 l 178 942 "
    },
    "â€™": {
      "x_min": 88.03125,
      "x_max": 299,
      "ha": 366,
      "o": "m 165 858 q 193 922 173 893 q 244 952 213 952 q 282 938 266 952 q 299 902 299 924 q 288 856 299 881 q 263 810 278 832 l 117 567 l 88 575 l 165 858 "
    },
    "Î": {
      "x_min": 96.390625,
      "x_max": 894.046875,
      "ha": 991,
      "o": "m 110 230 q 110 486 110 315 q 110 741 110 656 l 110 950 l 163 950 q 320 766 235 865 q 459 606 405 667 q 596 451 513 544 l 800 229 l 800 605 q 800 765 800 685 q 786 929 800 845 l 839 929 l 894 929 q 881 831 888 884 q 875 741 875 779 q 875 413 875 619 q 875 165 875 206 l 875 -15 l 832 -15 q 716 125 780 49 q 578 281 651 201 l 189 702 l 189 330 q 208 0 189 169 l 146 0 l 96 0 q 110 124 110 79 q 110 230 110 170 "
    },
    "-": {
      "x_min": 57,
      "x_max": 381,
      "ha": 440,
      "o": "m 57 391 l 381 391 l 381 274 l 57 274 l 57 391 "
    },
    "Q": {
      "x_min": 50,
      "x_max": 1052.796875,
      "ha": 1096,
      "o": "m 554 -15 q 190 112 331 -15 q 50 465 50 240 q 188 820 50 690 q 548 950 326 950 q 874 853 738 950 q 1025 654 1010 756 q 1043 525 1040 551 q 1046 462 1046 499 q 1043 405 1046 429 q 1028 305 1040 381 q 940 144 1017 229 q 732 7 863 59 l 911 -112 q 983 -160 943 -135 q 1052 -199 1024 -185 q 987 -228 1014 -213 q 931 -266 960 -242 q 858 -208 893 -234 q 787 -154 823 -181 l 593 -15 l 554 -15 m 198 468 q 285 163 198 283 q 547 43 372 43 q 809 163 723 43 q 896 468 896 284 q 808 770 896 651 q 547 889 720 889 q 341 826 421 889 q 221 638 261 763 q 198 468 198 552 "
    },
    "Ï‚": {
      "x_min": 44,
      "x_max": 601.4375,
      "ha": 631,
      "o": "m 299 -205 q 344 -218 320 -218 q 439 -177 400 -218 q 478 -79 478 -136 q 369 55 478 4 q 157 156 261 106 q 44 365 44 231 q 141 585 44 499 q 373 672 238 672 q 496 658 439 672 q 601 616 553 645 q 575 501 583 559 l 551 501 q 492 586 539 557 q 384 616 445 616 q 229 551 291 616 q 168 394 168 486 q 213 271 168 320 q 333 197 259 223 l 458 149 q 558 76 521 119 q 596 -27 596 34 q 512 -206 596 -136 q 316 -292 428 -276 l 299 -205 "
    },
    "M": {
      "x_min": 54.65625,
      "x_max": 1142.328125,
      "ha": 1213,
      "o": "m 187 950 l 238 950 q 325 772 286 851 q 404 612 363 693 q 494 436 445 531 l 614 213 q 728 435 659 298 q 866 711 798 572 q 981 950 934 849 l 1032 950 q 1062 649 1047 791 q 1097 341 1077 508 q 1142 0 1117 174 q 1100 8 1117 5 q 1067 11 1082 11 q 1029 6 1051 11 q 979 0 1007 2 q 979 226 979 109 q 961 461 979 342 q 925 695 943 579 l 734 312 q 599 0 652 152 l 583 0 l 565 0 q 223 684 395 363 l 185 306 q 169 128 169 179 q 169 0 169 77 q 136 4 155 1 q 109 7 117 7 q 80 4 92 7 q 54 0 68 2 q 119 333 88 167 q 168 652 150 498 q 187 950 187 805 "
    },
    "Î¨": {
      "x_min": 72,
      "x_max": 977,
      "ha": 1049,
      "o": "m 592 0 q 551 5 575 2 q 523 8 527 8 q 486 5 507 8 q 455 0 466 2 q 455 151 455 45 q 455 297 455 258 q 177 382 283 297 q 72 649 72 468 l 72 815 l 72 931 q 138 924 110 924 q 179 927 153 924 q 209 931 204 931 q 199 817 199 860 q 199 722 199 774 l 199 688 l 199 638 q 266 437 199 507 q 457 368 333 368 l 457 481 q 457 735 457 566 q 457 931 457 904 q 491 926 473 928 q 523 924 509 924 q 566 926 547 924 q 590 931 585 929 q 590 634 590 840 q 590 369 590 428 l 634 369 q 798 463 751 369 q 845 690 845 557 q 845 828 845 739 q 845 931 845 918 q 879 926 862 928 q 907 924 896 924 q 953 926 937 924 q 977 931 968 929 l 977 837 l 977 674 q 879 388 977 480 q 592 297 782 297 q 592 142 592 249 q 592 0 592 35 "
    },
    "C": {
      "x_min": 50,
      "x_max": 863.859375,
      "ha": 894,
      "o": "m 812 738 q 541 889 719 889 q 288 768 375 889 q 202 469 202 647 q 292 177 202 305 q 539 50 383 50 q 695 86 624 50 q 838 188 767 122 l 852 183 q 841 122 844 147 q 838 69 838 97 q 511 -15 685 -15 q 176 116 303 -15 q 50 462 50 248 q 185 820 50 690 q 545 950 320 950 q 704 930 625 950 q 863 875 783 911 q 840 809 850 843 q 830 738 830 775 l 812 738 "
    },
    "Å“": {
      "x_min": 38,
      "x_max": 1171,
      "ha": 1202,
      "o": "m 352 -14 q 123 76 208 -14 q 38 317 38 167 q 123 569 38 469 q 355 669 208 669 q 504 637 432 669 q 628 548 576 605 q 878 672 726 672 q 1087 593 1003 672 q 1171 388 1171 514 l 1171 332 q 1001 332 1086 332 q 834 332 917 332 l 694 332 q 749 124 694 207 q 925 42 805 42 q 1123 129 1046 42 l 1139 126 q 1132 82 1135 107 q 1128 46 1130 58 q 1016 0 1078 14 q 886 -15 954 -15 q 739 12 803 -15 q 628 105 674 40 q 504 15 572 44 q 352 -14 435 -14 m 368 622 q 219 531 264 622 q 175 327 175 441 q 219 125 175 215 q 361 35 263 35 q 511 127 461 35 q 562 327 562 220 q 516 531 562 441 q 368 622 470 622 m 1040 390 l 1040 422 q 997 560 1040 503 q 876 618 954 618 q 741 549 789 618 q 694 390 694 480 l 1040 390 "
    },
    "!": {
      "x_min": 133,
      "x_max": 306,
      "ha": 439,
      "o": "m 218 156 q 280 130 254 156 q 306 68 306 105 q 280 8 306 32 q 218 -15 254 -15 q 158 9 183 -15 q 133 68 133 33 q 157 130 133 105 q 218 156 182 156 m 144 752 l 144 841 q 160 919 144 888 q 220 950 177 950 q 277 925 256 950 q 298 863 298 901 q 293 808 298 845 q 289 752 289 770 l 242 250 q 218 253 233 253 q 195 250 202 253 l 144 752 "
    },
    "Ã§": {
      "x_min": 34,
      "x_max": 603.296875,
      "ha": 632,
      "o": "m 596 119 l 581 42 q 481 -2 539 10 q 356 -15 423 -15 q 128 80 223 -15 q 34 313 34 175 q 130 576 34 480 q 390 672 227 672 q 501 659 448 672 q 603 617 555 647 q 586 558 593 587 q 573 493 578 529 l 559 493 q 498 588 537 554 q 399 622 458 622 q 230 534 288 622 q 173 329 173 447 q 234 126 173 211 q 407 42 295 42 q 581 123 512 42 l 596 119 m 198 -212 q 344 -246 268 -246 q 406 -232 381 -246 q 432 -187 432 -219 q 413 -143 432 -159 q 366 -128 395 -128 l 317 -128 l 317 0 l 361 0 l 361 -75 l 397 -75 q 486 -104 449 -75 q 523 -184 523 -134 q 485 -270 523 -238 q 394 -303 448 -303 q 281 -294 329 -303 q 181 -261 234 -286 l 198 -212 "
    },
    "{": {
      "x_min": 114,
      "x_max": 556,
      "ha": 669,
      "o": "m 481 911 q 412 875 436 911 q 389 793 389 840 l 389 744 l 389 583 q 360 434 389 493 q 258 354 332 376 q 360 272 331 332 q 389 125 389 212 l 389 -35 q 405 -149 389 -108 q 462 -197 422 -191 q 519 -204 501 -204 q 556 -204 536 -204 l 556 -276 q 378 -240 449 -276 q 307 -105 307 -204 l 307 -29 l 307 132 q 288 245 307 194 q 234 304 270 295 q 172 314 199 314 q 114 314 145 314 l 114 389 q 263 429 220 389 q 307 576 307 469 l 307 737 q 339 918 307 870 q 440 977 372 966 q 556 983 493 983 l 556 911 l 481 911 "
    },
    "X": {
      "x_min": 0,
      "x_max": 724.109375,
      "ha": 724,
      "o": "m 196 286 l 311 456 q 17 932 156 717 q 62 929 32 932 q 107 926 92 926 q 164 929 144 926 q 194 932 185 932 q 290 742 239 841 q 383 567 341 644 l 479 726 q 585 932 537 824 q 615 927 601 929 q 647 926 628 926 q 679 929 657 926 q 713 932 700 932 q 603 780 660 862 q 514 652 547 699 l 419 512 q 513 347 470 419 q 602 197 555 274 q 724 0 649 119 q 672 3 705 0 q 634 7 639 7 q 583 4 605 7 q 547 0 560 1 q 449 197 502 97 q 345 398 397 298 l 259 249 q 170 96 189 130 q 125 0 152 62 q 87 3 111 0 q 61 7 63 7 q 25 4 42 7 q 0 0 9 1 l 196 286 "
    },
    "Ã´": {
      "x_min": 39,
      "x_max": 698,
      "ha": 737,
      "o": "m 364 -15 q 128 79 218 -15 q 39 322 39 173 q 130 572 39 473 q 372 672 222 672 q 608 575 519 672 q 698 327 698 479 q 607 80 698 175 q 364 -15 516 -15 m 324 977 l 411 977 l 560 743 l 514 743 l 368 874 l 223 743 l 176 743 l 324 977 m 369 622 q 222 532 268 622 q 176 327 176 442 q 222 125 176 216 q 368 34 268 34 q 514 124 469 34 q 559 327 559 214 q 514 531 559 441 q 369 622 470 622 "
    },
    "#": {
      "x_min": 76.21875,
      "x_max": 952.78125,
      "ha": 1029,
      "o": "m 488 647 l 661 647 l 774 969 l 858 969 l 745 647 l 952 647 l 929 576 l 721 576 l 658 391 l 877 391 l 856 319 l 631 319 l 519 0 l 436 0 l 548 319 l 374 319 l 261 0 l 178 0 l 288 319 l 76 319 l 100 391 l 314 391 l 378 576 l 148 576 l 172 647 l 402 647 l 515 969 l 598 969 l 488 647 m 462 576 l 398 390 l 574 390 l 637 576 l 462 576 "
    },
    "Î¹": {
      "x_min": 93.921875,
      "x_max": 228.453125,
      "ha": 335,
      "o": "m 98 333 l 98 520 q 98 566 98 544 q 93 653 98 588 q 138 647 127 647 q 161 647 148 647 q 228 653 191 647 q 223 555 223 599 q 223 437 223 511 l 223 406 q 223 194 223 337 q 223 0 223 51 q 197 3 212 1 q 161 6 182 6 q 125 3 145 6 q 98 0 104 1 q 98 165 98 51 q 98 333 98 279 "
    },
    "Î†": {
      "x_min": 12.5625,
      "x_max": 893.203125,
      "ha": 878,
      "o": "m 322 638 l 450 949 q 480 944 456 946 q 513 949 498 944 q 643 613 591 745 q 756 331 695 480 q 893 0 816 181 q 842 3 874 0 q 807 7 811 7 q 757 4 780 7 q 721 0 735 1 q 659 195 679 135 q 601 355 640 255 l 441 355 l 280 355 l 228 205 q 170 0 190 94 l 114 6 q 75 3 90 6 q 54 0 61 0 q 144 211 99 105 q 225 403 189 317 q 322 638 261 490 m 192 942 q 224 973 209 963 q 260 984 239 984 q 297 968 282 984 q 312 931 312 953 q 297 896 312 910 q 256 866 283 881 l 60 743 l 12 743 l 192 942 m 441 422 l 573 422 l 442 761 l 312 422 l 441 422 "
    },
    ")": {
      "x_min": 65.65625,
      "x_max": 327,
      "ha": 440,
      "o": "m 327 376 q 267 81 327 217 q 95 -183 208 -54 q 65 -151 83 -163 q 191 104 154 -16 q 229 385 229 226 q 188 660 229 533 q 65 917 148 788 q 95 949 87 933 q 269 681 212 816 q 327 376 327 545 "
    },
    "Îµ": {
      "x_min": 52,
      "x_max": 557.359375,
      "ha": 616,
      "o": "m 497 516 q 425 591 459 566 q 346 616 392 616 q 257 580 293 616 q 221 493 221 545 q 263 407 221 438 q 365 376 305 376 l 409 376 l 409 351 l 409 314 l 338 314 q 232 279 275 314 q 189 181 189 245 q 234 76 189 115 q 345 37 279 37 q 452 65 403 37 q 534 144 501 94 q 543 98 538 121 q 557 53 549 75 q 444 1 506 18 q 316 -16 381 -16 q 132 30 212 -16 q 52 174 52 77 q 93 286 52 245 q 209 355 135 326 q 125 408 159 370 q 92 496 92 445 q 162 625 92 578 q 317 672 233 672 q 422 654 372 672 q 531 604 473 637 q 513 564 520 584 q 497 516 505 545 "
    },
    "Î”": {
      "x_min": 0,
      "x_max": 880.640625,
      "ha": 881,
      "o": "m 880 6 q 687 6 816 6 q 491 6 558 6 q 246 6 409 6 q 0 6 83 6 q 220 473 123 251 q 398 932 317 695 q 423 929 406 932 q 447 926 439 926 q 472 927 461 926 q 495 932 483 929 q 677 451 589 663 q 880 6 766 238 m 276 426 q 155 85 208 255 q 284 85 197 85 q 415 85 371 85 l 447 85 q 575 85 539 85 q 677 90 612 85 l 638 189 l 564 383 l 413 777 l 276 426 "
    },
    "Ã¢": {
      "x_min": 42,
      "x_max": 642,
      "ha": 635,
      "o": "m 229 -16 q 96 32 150 -16 q 42 161 42 81 q 104 302 42 272 q 297 363 166 332 q 435 447 428 395 q 395 568 435 521 q 283 616 355 616 q 187 587 228 616 q 122 507 147 559 l 93 520 l 102 590 q 198 651 141 631 q 317 672 255 672 q 490 622 435 672 q 546 454 546 572 l 546 132 q 556 68 546 83 q 605 54 566 54 q 642 57 629 54 l 642 26 q 582 5 613 14 q 526 -6 551 -3 q 430 83 445 -6 q 336 9 381 34 q 229 -16 291 -16 m 273 977 l 360 977 l 511 742 l 461 742 l 317 874 l 173 742 l 123 742 l 273 977 m 173 185 q 200 97 173 132 q 279 63 228 63 q 383 106 336 63 q 430 211 430 150 l 430 345 q 235 292 298 318 q 173 185 173 267 "
    },
    "}": {
      "x_min": 112,
      "x_max": 556,
      "ha": 670,
      "o": "m 363 576 q 398 438 363 487 q 517 389 433 389 l 556 389 l 556 317 q 407 278 452 317 q 363 132 363 239 l 363 -28 q 314 -229 363 -183 q 112 -276 266 -276 l 112 -204 q 247 -172 214 -204 q 281 -83 281 -141 l 281 -35 l 281 125 q 309 271 281 212 q 410 354 337 329 q 306 434 331 382 q 281 583 281 487 l 281 744 q 263 858 281 816 q 188 911 246 901 l 112 911 l 112 982 l 182 982 q 282 960 238 982 q 341 900 326 938 q 359 822 356 862 q 363 737 363 783 l 363 576 "
    },
    "â€°": {
      "x_min": 27,
      "x_max": 1481,
      "ha": 1506,
      "o": "m 783 0 q 633 62 693 0 q 574 218 574 124 q 634 372 574 309 q 783 436 695 436 q 931 373 871 436 q 991 223 991 310 q 976 130 991 176 q 900 35 952 70 q 783 0 847 0 m 1272 0 q 1124 62 1183 0 q 1066 218 1066 124 q 1125 372 1066 308 q 1274 436 1185 436 q 1421 373 1361 436 q 1481 223 1481 310 q 1464 130 1481 169 q 1390 34 1442 69 q 1272 0 1338 0 m 236 448 q 86 510 146 448 q 27 663 27 572 q 87 819 27 754 q 236 884 148 884 q 383 822 322 884 q 444 671 444 760 q 426 580 444 620 q 352 483 404 519 q 236 448 300 448 m 845 1014 l 222 -126 l 154 -126 l 777 1014 l 845 1014 m 878 260 q 853 353 878 310 q 782 397 829 397 q 704 340 722 397 q 686 202 686 283 q 705 86 686 133 q 782 40 724 40 q 848 73 823 40 q 874 149 874 106 q 878 206 878 169 q 878 260 878 242 m 332 684 q 312 796 332 749 q 239 844 293 844 q 157 783 177 844 q 137 645 137 723 q 160 539 137 589 q 232 490 184 490 q 297 524 272 490 q 327 598 322 558 q 332 684 332 638 m 1368 260 q 1345 356 1368 315 q 1270 397 1322 397 q 1194 340 1212 397 q 1176 202 1176 283 q 1195 87 1176 134 q 1271 40 1214 40 q 1338 73 1313 40 q 1364 149 1364 106 q 1368 206 1368 169 q 1368 260 1368 242 "
    },
    "Ã„": {
      "x_min": -16.65625,
      "x_max": 821.9375,
      "ha": 810,
      "o": "m 252 638 l 381 951 q 410 945 386 948 q 443 951 428 945 q 558 651 489 826 q 674 360 627 477 q 821 0 721 242 q 771 3 802 0 q 736 7 740 7 q 687 3 713 7 q 651 0 661 0 q 587 203 602 156 q 530 355 571 250 l 369 355 l 209 355 l 157 205 q 126 108 143 169 q 99 0 109 47 l 43 4 q 18 4 28 4 q -16 0 9 4 q 70 203 28 105 q 165 426 112 300 q 252 638 217 552 m 287 1208 q 338 1186 318 1208 q 358 1133 358 1164 q 337 1080 358 1102 q 287 1058 317 1058 q 233 1080 255 1058 q 212 1133 212 1102 q 233 1185 212 1163 q 287 1208 255 1208 m 523 1208 q 574 1186 553 1208 q 596 1133 596 1164 q 574 1080 596 1103 q 523 1058 553 1058 q 470 1080 492 1058 q 449 1133 449 1102 q 470 1185 449 1163 q 523 1208 492 1208 m 370 422 l 503 422 l 372 762 l 240 422 l 370 422 "
    },
    "a": {
      "x_min": 44,
      "x_max": 642,
      "ha": 635,
      "o": "m 230 -15 q 98 33 152 -15 q 44 162 44 82 q 104 302 44 273 q 297 363 165 332 q 435 448 429 395 q 394 567 435 519 q 282 615 352 615 q 187 587 227 615 q 122 508 147 560 l 94 519 l 102 591 q 201 651 146 631 q 317 672 256 672 q 490 623 436 672 q 545 456 545 574 l 545 132 q 556 70 545 86 q 605 54 568 54 l 642 54 l 642 26 q 583 4 613 15 q 527 -7 554 -7 q 461 15 483 -7 q 429 83 438 38 q 335 9 381 34 q 230 -15 290 -15 m 172 185 q 201 99 172 135 q 278 63 230 63 q 383 106 337 63 q 429 211 429 150 l 429 344 q 235 294 299 320 q 172 185 172 268 "
    },
    "â€”": {
      "x_min": 222,
      "x_max": 1116,
      "ha": 1339,
      "o": "m 222 375 l 1116 375 l 1116 292 l 222 292 l 222 375 "
    },
    "=": {
      "x_min": 166,
      "x_max": 950,
      "ha": 1116,
      "o": "m 950 499 l 166 499 l 166 564 l 950 564 l 950 499 m 950 249 l 166 249 l 166 316 l 950 316 l 950 249 "
    },
    "N": {
      "x_min": 96.390625,
      "x_max": 894.046875,
      "ha": 991,
      "o": "m 110 230 q 110 486 110 315 q 110 741 110 656 l 110 950 l 163 950 q 320 766 235 865 q 459 606 405 667 q 596 451 513 544 l 800 229 l 800 605 q 800 765 800 685 q 786 929 800 845 l 839 929 l 894 929 q 881 831 888 884 q 875 741 875 779 q 875 413 875 619 q 875 165 875 206 l 875 -15 l 832 -15 q 716 125 780 49 q 578 281 651 201 l 189 702 l 189 330 q 208 0 189 169 l 146 0 l 96 0 q 110 124 110 79 q 110 230 110 170 "
    },
    "Ï": {
      "x_min": 69,
      "x_max": 696,
      "ha": 742,
      "o": "m 192 -100 q 192 -236 192 -146 q 192 -372 192 -326 q 159 -367 167 -368 q 131 -367 151 -367 q 94 -369 114 -367 q 69 -372 74 -371 q 69 -190 69 -311 q 69 -8 69 -70 q 69 166 69 67 q 69 328 69 264 q 145 578 69 484 q 372 672 221 672 q 604 573 513 672 q 696 328 696 474 q 613 84 696 184 q 391 -16 530 -16 q 283 6 330 -16 q 192 77 236 29 l 192 -100 m 370 619 q 227 532 266 619 q 187 327 187 446 q 237 102 187 168 q 368 37 287 37 q 514 124 469 37 q 559 325 559 212 q 514 529 559 439 q 370 619 470 619 "
    },
    "2": {
      "x_min": 22,
      "x_max": 610.453125,
      "ha": 734,
      "o": "m 440 648 q 401 789 440 727 q 293 851 363 851 q 170 802 215 851 q 126 676 126 753 l 116 673 q 83 739 98 712 q 46 799 68 767 q 307 911 155 911 q 497 844 417 911 q 577 667 577 777 q 517 479 577 555 q 309 258 457 403 l 166 118 l 434 118 q 520 123 476 118 q 610 136 564 129 q 606 102 607 118 q 605 67 605 87 q 606 37 605 54 q 610 4 607 20 q 430 4 551 4 q 248 4 309 4 q 140 4 192 4 q 22 0 87 4 l 22 40 q 229 238 153 158 q 373 430 306 319 q 440 648 440 541 "
    },
    "Ã¼": {
      "x_min": 91,
      "x_max": 650.53125,
      "ha": 739,
      "o": "m 641 498 q 641 329 641 443 q 641 158 641 215 q 650 0 641 81 q 619 3 634 1 q 586 6 604 6 q 552 3 571 6 q 523 0 533 1 l 523 118 q 430 18 484 52 q 304 -15 376 -15 q 143 50 195 -15 q 91 229 91 115 l 91 354 l 91 516 l 91 655 q 117 650 101 651 q 155 650 133 650 q 188 650 171 650 q 215 655 205 650 q 215 445 215 591 q 215 247 215 299 q 247 115 215 163 q 356 68 279 68 q 463 113 418 68 q 515 217 509 159 q 521 340 521 274 q 521 520 521 394 q 521 655 521 647 q 547 650 532 651 q 585 650 563 650 q 617 650 600 650 q 650 655 634 650 q 641 498 641 573 m 249 929 q 302 906 280 929 q 324 854 324 884 q 303 800 324 822 q 253 778 283 778 q 199 800 221 778 q 177 854 177 823 q 197 906 177 884 q 249 929 217 929 m 488 929 q 539 906 517 929 q 561 854 561 884 q 541 799 561 820 q 490 778 521 778 q 435 800 457 778 q 414 854 414 822 q 435 906 414 884 q 488 929 457 929 "
    },
    "Z": {
      "x_min": 7,
      "x_max": 786,
      "ha": 812,
      "o": "m 7 36 q 218 325 110 176 q 416 605 326 473 l 585 857 l 424 857 q 58 836 242 857 l 64 883 l 58 926 q 257 926 132 926 q 419 926 382 926 q 610 926 481 926 q 786 926 739 926 l 786 903 q 582 630 695 786 q 391 364 469 475 q 197 82 313 253 l 419 82 q 570 82 510 82 q 786 102 630 82 l 781 68 l 780 54 l 780 34 l 786 4 q 494 4 669 4 q 318 4 320 4 q 163 4 266 4 q 7 4 60 4 l 7 36 "
    },
    "u": {
      "x_min": 91,
      "x_max": 649.4375,
      "ha": 739,
      "o": "m 644 497 l 644 157 q 644 83 644 122 q 649 -1 644 45 q 612 3 620 3 q 585 3 604 3 q 561 3 570 3 q 523 -1 552 3 l 523 117 q 430 18 482 52 q 306 -15 378 -15 q 142 49 193 -15 q 91 228 91 113 l 91 353 l 91 516 l 91 653 q 155 647 122 647 q 185 648 174 647 q 215 653 195 649 q 215 450 215 585 q 215 246 215 314 q 247 114 215 163 q 356 66 279 66 q 464 112 419 66 q 516 225 510 158 q 521 340 521 282 q 521 497 521 393 q 521 653 521 601 q 561 647 552 647 q 583 647 570 647 q 614 648 604 647 q 644 653 624 649 l 644 497 "
    },
    "k": {
      "x_min": 98,
      "x_max": 664,
      "ha": 669,
      "o": "m 98 656 q 98 873 98 741 q 98 1024 98 1005 q 161 1018 131 1018 q 223 1024 193 1018 q 223 824 223 962 q 223 622 223 687 l 223 378 l 240 378 q 497 653 383 506 q 552 647 527 647 q 604 647 578 647 q 635 652 625 651 l 342 398 l 537 165 q 595 93 566 127 q 664 19 625 59 l 664 0 q 615 3 646 0 q 579 7 585 7 q 533 3 556 7 q 500 0 510 0 q 429 101 463 54 q 353 197 394 148 l 263 308 l 247 323 l 223 326 q 223 164 223 272 q 223 0 223 55 q 197 4 212 3 q 161 6 182 6 q 125 3 144 6 q 98 0 106 1 q 98 386 98 151 q 98 656 98 620 "
    },
    "Î—": {
      "x_min": 114,
      "x_max": 910.515625,
      "ha": 1018,
      "o": "m 257 317 q 257 141 257 253 q 257 0 257 30 q 222 4 248 0 q 184 8 197 8 q 145 4 171 8 q 114 0 119 0 q 114 243 114 71 q 114 464 114 414 q 114 709 114 539 q 114 932 114 880 q 150 929 119 932 q 184 926 180 926 q 227 929 197 926 q 257 932 258 932 q 257 719 257 853 q 257 544 257 584 q 399 544 296 544 q 507 544 502 544 q 653 544 552 544 q 758 544 755 544 q 758 785 758 641 q 758 932 758 930 q 797 929 766 932 q 832 926 828 926 q 875 929 845 926 q 910 932 906 932 q 901 794 901 866 q 901 659 901 723 l 901 505 q 901 252 901 420 q 901 0 901 84 q 871 4 896 0 q 832 8 845 8 q 796 5 813 8 q 758 0 778 2 q 758 103 758 41 q 758 175 758 166 l 758 317 l 758 466 q 591 466 676 466 q 424 466 507 466 l 257 466 l 257 317 "
    },
    "Î‘": {
      "x_min": -14.96875,
      "x_max": 822.109375,
      "ha": 809,
      "o": "m 253 638 l 379 949 q 394 945 387 946 q 409 944 401 944 q 443 949 428 944 q 565 629 525 733 q 673 359 605 526 q 822 0 740 192 q 773 3 804 0 q 736 7 743 7 q 686 4 709 7 q 650 0 664 1 q 588 199 609 137 q 532 355 567 261 l 370 355 l 210 355 l 159 205 q 127 110 142 161 q 99 0 112 59 l 43 6 q 6 3 20 6 q -14 0 -8 0 q 74 211 29 105 q 155 403 119 317 q 253 638 191 490 m 370 422 l 502 422 l 371 760 l 240 422 l 370 422 "
    },
    "ÃŸ": {
      "x_min": 98,
      "x_max": 712,
      "ha": 750,
      "o": "m 98 432 l 98 611 l 98 761 q 191 948 109 872 q 384 1025 273 1025 q 531 979 467 1025 q 596 852 596 933 q 520 709 596 783 q 444 586 444 636 q 578 457 444 547 q 712 259 712 368 q 635 64 712 143 q 445 -15 559 -15 q 372 -6 411 -15 q 303 15 333 1 l 327 128 l 341 128 q 398 70 361 91 q 476 49 434 49 q 571 92 533 49 q 610 193 610 135 q 475 365 610 280 q 340 525 340 450 q 425 667 340 561 q 510 838 510 773 q 474 939 510 899 q 380 979 438 979 q 267 924 309 979 q 225 795 225 870 l 225 629 l 225 344 q 225 141 225 265 q 225 2 225 17 q 192 5 215 2 q 163 8 170 8 q 121 5 140 8 q 98 2 103 3 q 98 250 98 92 q 98 432 98 409 "
    },
    "Ã©": {
      "x_min": 39,
      "x_max": 632,
      "ha": 666,
      "o": "m 398 42 q 590 129 512 42 l 609 126 q 601 90 605 109 q 597 46 597 71 q 486 0 546 14 q 361 -15 426 -15 q 127 73 215 -15 q 39 311 39 162 q 123 562 39 456 q 347 669 207 669 q 551 590 470 669 q 632 387 632 512 l 632 332 q 429 332 550 332 q 307 332 308 332 l 175 332 q 230 127 175 212 q 398 42 285 42 m 398 945 q 430 975 415 966 q 467 985 445 985 q 518 934 518 985 q 503 895 518 912 q 462 866 488 878 l 266 743 l 217 743 l 398 945 m 503 390 l 503 436 q 460 563 503 508 q 349 618 418 618 q 228 552 266 618 q 179 390 191 487 l 503 390 "
    },
    "s": {
      "x_min": 67.109375,
      "x_max": 520,
      "ha": 581,
      "o": "m 114 161 q 168 69 128 102 q 270 36 209 36 q 370 67 326 36 q 415 152 415 98 q 327 260 415 224 q 164 320 246 290 q 78 460 78 366 q 145 612 78 552 q 303 672 212 672 q 392 660 348 672 q 490 627 436 649 l 451 508 l 439 508 q 393 587 424 561 q 307 614 361 614 q 219 584 257 614 q 181 505 181 555 q 350 374 181 427 q 520 197 520 322 q 441 39 520 93 q 254 -15 362 -15 q 67 23 159 -15 l 101 161 l 114 161 "
    },
    "B": {
      "x_min": 110.5625,
      "x_max": 737,
      "ha": 786,
      "o": "m 116 560 q 110 932 116 759 l 203 932 q 334 932 242 932 q 432 932 426 932 q 618 880 540 932 q 696 724 696 829 q 626 578 696 632 q 456 503 557 524 q 660 438 584 489 q 737 269 737 387 q 628 69 737 135 q 376 3 519 3 l 229 3 q 160 3 208 3 q 116 3 111 3 q 116 289 116 88 q 116 560 116 489 m 255 255 l 255 146 l 255 59 l 330 59 q 516 109 441 59 q 591 266 591 160 q 514 427 591 385 q 310 469 438 469 l 255 469 l 255 255 m 559 705 q 499 831 559 786 q 355 876 439 876 l 257 876 q 248 710 248 799 l 248 522 q 468 558 377 522 q 559 705 559 595 "
    },
    "â€¦": {
      "x_min": 117,
      "x_max": 985,
      "ha": 1136,
      "o": "m 187 131 q 239 109 218 131 q 261 55 261 87 q 239 5 261 25 q 187 -15 218 -15 q 137 4 157 -15 q 117 55 117 23 q 136 109 117 87 q 187 131 156 131 m 550 131 q 600 109 579 131 q 622 55 622 87 q 600 5 622 25 q 550 -15 579 -15 q 498 4 518 -15 q 478 55 478 23 q 498 110 478 89 q 550 131 518 131 m 911 131 q 963 109 942 131 q 985 55 985 87 q 963 5 985 25 q 911 -15 942 -15 q 861 4 881 -15 q 841 55 841 23 q 860 109 841 87 q 911 131 880 131 "
    },
    "?": {
      "x_min": 125.53125,
      "x_max": 510,
      "ha": 590,
      "o": "m 300 155 q 359 129 334 155 q 384 67 384 104 q 360 6 384 29 q 300 -16 336 -16 q 239 7 264 -16 q 215 67 215 31 q 239 129 215 104 q 300 155 264 155 m 322 250 q 210 290 256 250 q 165 399 165 331 q 281 595 165 479 q 398 776 398 712 q 364 858 398 823 q 286 894 331 894 q 203 866 238 894 q 159 794 167 839 l 147 794 q 139 835 143 821 q 125 894 136 849 q 206 936 162 922 q 300 950 250 950 q 448 893 387 950 q 510 748 510 837 q 390 549 510 656 q 271 370 271 443 q 288 316 271 337 q 336 296 305 296 q 389 302 364 296 l 375 256 q 349 251 358 252 q 322 250 341 250 "
    },
    "H": {
      "x_min": 105.84375,
      "x_max": 909.515625,
      "ha": 1016,
      "o": "m 257 316 q 257 142 257 254 q 257 0 257 31 q 224 4 250 0 q 184 9 198 9 q 145 4 171 9 q 114 0 119 0 q 114 239 114 71 q 114 464 114 407 q 114 711 114 620 q 105 932 114 802 q 150 928 122 932 q 184 925 179 925 q 227 928 199 925 q 257 932 254 932 q 257 719 257 854 q 257 544 257 584 l 507 544 l 758 544 q 758 785 758 641 q 758 932 758 930 q 798 928 770 932 q 831 925 826 925 q 876 928 848 925 q 909 932 905 932 q 900 798 900 867 q 900 659 900 728 l 900 448 l 900 283 q 900 135 900 238 q 900 0 900 32 q 871 4 897 0 q 831 9 845 9 q 792 4 818 9 q 758 0 766 0 q 758 101 758 38 q 758 176 758 164 l 758 316 l 758 466 q 591 466 703 466 q 424 466 480 466 l 257 466 l 257 316 "
    },
    "Î½": {
      "x_min": 0.359375,
      "x_max": 620,
      "ha": 642,
      "o": "m 313 8 q 290 5 303 8 q 267 0 278 2 q 196 205 230 120 q 0 653 162 291 q 83 647 45 647 q 111 647 97 647 q 165 653 125 647 q 248 402 200 528 l 352 131 q 460 358 431 267 q 489 520 489 448 q 482 578 489 542 q 468 641 476 615 q 541 649 517 645 q 602 660 565 652 q 616 610 613 623 q 620 576 620 598 q 609 501 620 535 q 549 373 599 466 q 452 189 499 280 q 365 0 404 99 q 339 4 356 0 q 313 8 321 8 "
    },
    "Ã®": {
      "x_min": -24.5,
      "x_max": 358.390625,
      "ha": 334,
      "o": "m 98 144 l 98 522 l 98 655 q 134 650 110 650 q 163 650 159 650 q 229 655 203 650 q 225 507 225 581 q 225 254 225 423 q 225 0 225 85 q 197 3 212 1 q 160 6 182 6 q 123 3 143 6 q 98 0 104 1 l 98 144 m 122 977 l 209 977 l 358 743 l 313 743 l 167 874 l 23 743 l -24 743 l 122 977 "
    },
    "c": {
      "x_min": 35,
      "x_max": 600.75,
      "ha": 631,
      "o": "m 593 118 l 581 41 q 483 -4 536 6 q 357 -16 429 -16 q 128 78 222 -16 q 35 312 35 173 q 131 576 35 480 q 391 672 228 672 q 502 658 449 672 q 600 616 554 645 q 573 493 584 563 l 558 493 q 499 586 538 553 q 397 619 460 619 q 229 532 287 619 q 172 327 172 445 q 232 125 172 209 q 406 41 293 41 q 581 121 509 41 l 593 118 "
    },
    "Â¶": {
      "x_min": 17.40625,
      "x_max": 521,
      "ha": 578,
      "o": "m 292 967 l 521 967 l 521 909 l 464 909 l 464 3 l 407 3 l 407 909 l 299 909 l 299 3 l 242 3 l 242 557 q 97 601 160 557 q 26 689 35 645 q 17 743 17 733 q 88 895 17 824 q 292 967 160 967 "
    },
    "Î²": {
      "x_min": 98,
      "x_max": 723,
      "ha": 764,
      "o": "m 157 -364 q 126 -367 148 -364 q 98 -369 104 -369 q 98 -254 98 -326 q 98 -170 98 -183 l 98 100 l 98 564 q 161 897 98 770 q 410 1025 225 1025 q 593 960 514 1025 q 673 787 673 895 q 629 641 673 705 q 511 548 585 577 q 668 454 614 515 q 723 290 723 394 q 635 73 723 162 q 423 -15 548 -15 q 317 3 364 -15 q 222 61 270 22 q 222 -44 222 20 q 222 -110 222 -108 l 222 -369 q 187 -367 212 -369 q 157 -364 162 -364 m 348 564 q 497 618 449 564 q 546 779 546 673 q 509 917 546 859 q 397 975 472 975 q 270 906 314 975 q 226 752 226 838 q 226 613 226 701 q 226 506 226 525 l 226 355 l 226 230 q 270 94 226 147 q 398 41 315 41 q 541 116 494 41 q 588 297 588 192 q 538 443 588 386 q 404 500 489 500 q 353 497 376 500 q 321 493 330 494 l 321 516 l 321 564 l 348 564 "
    },
    "Îœ": {
      "x_min": 54.65625,
      "x_max": 1142.328125,
      "ha": 1213,
      "o": "m 187 950 l 238 950 q 325 772 286 851 q 404 612 363 693 q 494 436 445 531 l 614 213 q 728 435 659 298 q 866 711 798 572 q 981 950 934 849 l 1032 950 q 1062 649 1047 791 q 1097 341 1077 508 q 1142 0 1117 174 q 1100 8 1117 5 q 1067 11 1082 11 q 1029 6 1051 11 q 979 0 1007 2 q 979 226 979 109 q 961 461 979 342 q 925 695 943 579 l 734 312 q 599 0 652 152 l 583 0 l 565 0 q 223 684 395 363 l 185 306 q 169 128 169 179 q 169 0 169 77 q 136 4 155 1 q 109 7 117 7 q 80 4 92 7 q 54 0 68 2 q 119 333 88 167 q 168 652 150 498 q 187 950 187 805 "
    },
    "ÎŒ": {
      "x_min": -1.109375,
      "x_max": 1278,
      "ha": 1328,
      "o": "m 282 465 q 420 820 282 690 q 780 950 558 950 q 1106 853 970 950 q 1257 654 1242 757 q 1275 525 1272 551 q 1278 462 1278 500 q 1275 402 1278 426 q 1257 277 1272 379 q 1107 80 1242 175 q 781 -15 973 -15 q 670 -10 718 -15 q 530 29 622 -5 q 360 186 438 64 q 282 465 282 308 m 178 943 q 211 974 196 964 q 246 985 226 985 q 283 969 268 985 q 298 932 298 954 q 283 896 298 911 q 242 866 269 882 l 46 743 l -1 743 l 178 943 m 430 468 q 517 162 430 282 q 779 42 604 42 q 1041 162 955 42 q 1128 468 1128 283 q 1040 770 1128 651 q 779 889 952 889 q 573 826 653 889 q 453 638 493 763 q 430 468 430 552 "
    },
    "Î‰": {
      "x_min": -1.4375,
      "x_max": 1209.53125,
      "ha": 1316,
      "o": "m 557 316 q 557 142 557 254 q 557 0 557 31 q 524 4 550 0 q 484 9 498 9 q 445 4 471 9 q 414 0 419 0 q 414 239 414 71 q 414 464 414 407 q 414 711 414 620 q 405 932 414 802 q 450 928 422 932 q 484 925 479 925 q 527 928 499 925 q 557 932 554 932 q 557 719 557 854 q 557 544 557 584 l 807 544 l 1058 544 q 1058 786 1058 641 q 1058 932 1058 930 q 1098 928 1070 932 q 1131 925 1126 925 q 1176 928 1148 925 q 1209 932 1205 932 q 1200 798 1200 868 q 1200 659 1200 729 l 1200 448 l 1200 283 q 1200 135 1200 238 q 1200 0 1200 32 q 1171 4 1197 0 q 1131 9 1145 9 q 1092 4 1118 9 q 1058 0 1066 0 q 1058 101 1058 38 q 1058 176 1058 164 l 1058 316 l 1058 466 q 891 466 1003 466 q 724 466 780 466 l 557 466 l 557 316 m 178 942 q 210 973 195 963 q 246 984 225 984 q 283 968 268 984 q 298 931 298 953 q 283 896 298 910 q 242 866 269 881 l 46 743 l -1 743 l 178 942 "
    },
    "â€¢": {
      "x_min": 200,
      "x_max": 780,
      "ha": 983,
      "o": "m 491 789 q 694 702 609 789 q 780 493 780 615 q 695 286 780 372 q 491 200 611 200 q 285 285 370 200 q 200 493 200 370 q 221 605 200 550 q 306 725 243 661 q 491 789 368 789 "
    },
    "Â¥": {
      "x_min": 31.3125,
      "x_max": 684.359375,
      "ha": 735,
      "o": "m 304 272 q 172 272 263 272 q 49 272 81 272 l 49 332 q 144 332 80 332 q 241 332 208 332 l 306 332 l 306 433 l 173 433 l 49 433 l 49 493 q 179 493 88 493 q 275 493 269 493 l 124 742 l 31 884 l 89 884 l 188 884 q 220 814 204 845 q 282 699 237 783 l 388 512 q 467 653 424 570 q 579 884 510 735 l 630 884 l 684 884 l 612 773 l 543 656 l 445 493 q 581 493 486 493 q 683 493 676 493 l 683 433 l 424 433 l 424 332 l 560 332 q 623 332 582 332 q 683 332 663 332 l 683 272 l 556 272 l 425 272 q 425 141 425 204 q 433 3 425 77 l 357 3 l 293 0 q 304 129 304 54 q 304 272 304 204 "
    },
    "(": {
      "x_min": 112,
      "x_max": 372.96875,
      "ha": 440,
      "o": "m 112 388 q 172 683 112 545 q 344 949 232 821 q 372 917 353 933 q 248 660 285 781 q 211 379 211 538 q 251 103 211 231 q 372 -151 291 -24 q 344 -183 353 -166 q 169 84 227 -49 q 112 388 112 219 "
    },
    "U": {
      "x_min": 101,
      "x_max": 901.609375,
      "ha": 996,
      "o": "m 178 928 q 225 930 192 928 q 259 932 257 932 q 247 803 247 852 q 247 696 247 755 l 247 456 q 312 134 247 212 q 506 57 377 57 q 718 129 641 57 q 795 334 795 201 l 795 458 l 795 655 q 795 797 795 733 q 782 932 795 862 q 811 927 797 929 q 842 926 825 926 q 871 929 852 926 q 901 932 890 932 q 888 778 888 852 q 888 599 888 705 l 888 366 q 781 81 888 177 q 485 -15 674 -15 q 209 66 303 -15 q 115 323 115 147 l 115 424 l 115 698 q 108 826 115 759 q 101 932 101 892 q 136 928 119 928 q 178 928 154 928 "
    },
    "Î³": {
      "x_min": -12,
      "x_max": 686.75,
      "ha": 668,
      "o": "m 629 647 q 686 653 655 647 q 581 439 620 520 q 493 246 542 359 l 405 48 l 405 -90 q 405 -238 405 -139 q 405 -372 405 -337 l 365 -367 q 281 -372 319 -367 q 281 -195 281 -314 q 281 -33 281 -76 q 251 191 281 46 q 164 458 221 336 q 38 580 107 580 l -12 580 l -12 607 l -12 638 q 41 662 13 653 q 98 672 69 672 q 272 554 229 672 q 388 142 316 436 q 485 392 429 243 q 577 653 541 541 q 602 648 594 649 q 629 647 610 647 "
    },
    "Î±": {
      "x_min": 41,
      "x_max": 810.90625,
      "ha": 829,
      "o": "m 691 352 q 787 0 748 155 l 725 0 l 659 0 l 620 172 q 510 36 582 88 q 349 -15 439 -15 q 127 81 213 -15 q 41 316 41 177 q 128 569 41 467 q 361 672 216 672 q 527 622 455 672 q 646 485 598 573 q 697 651 677 569 l 755 651 l 810 651 q 748 505 776 576 q 691 352 719 434 m 369 619 q 222 530 267 619 q 178 326 178 442 q 219 124 178 214 q 360 34 261 34 q 518 137 470 34 q 589 323 565 240 q 516 531 567 444 q 369 619 466 619 "
    },
    "F": {
      "x_min": 105.84375,
      "x_max": 600,
      "ha": 657,
      "o": "m 257 315 q 257 142 257 254 q 257 0 257 31 q 224 4 250 0 q 184 8 198 8 q 145 4 171 8 q 114 0 119 0 q 114 239 114 71 q 114 465 114 408 q 114 712 114 621 q 105 929 114 803 l 350 929 l 600 929 l 600 886 l 600 836 q 494 855 538 851 q 380 860 451 860 l 254 860 l 254 671 l 254 528 l 390 528 q 574 541 470 528 l 574 497 l 574 457 l 386 457 l 257 457 l 257 315 "
    },
    "Â­": {
      "x_min": 0,
      "x_max": 670,
      "ha": 670,
      "o": "m 0 374 l 670 374 l 670 289 l 0 289 l 0 374 "
    },
    ":": {
      "x_min": 132,
      "x_max": 305,
      "ha": 437,
      "o": "m 219 636 q 279 611 254 636 q 304 548 304 586 q 280 486 304 512 q 219 461 256 461 q 157 486 183 461 q 132 548 132 511 q 157 610 132 584 q 219 636 182 636 m 219 156 q 279 131 254 156 q 305 69 305 107 q 280 8 305 32 q 219 -15 255 -15 q 158 9 183 -15 q 133 69 133 33 q 158 131 133 107 q 219 156 183 156 "
    },
    "Î§": {
      "x_min": 0,
      "x_max": 724.109375,
      "ha": 724,
      "o": "m 196 286 l 311 456 q 17 932 156 717 q 62 929 32 932 q 107 926 92 926 q 164 929 144 926 q 194 932 185 932 q 290 742 239 841 q 383 567 341 644 l 479 726 q 585 932 537 824 q 615 927 601 929 q 647 926 628 926 q 679 929 657 926 q 713 932 700 932 q 603 780 660 862 q 514 652 547 699 l 419 512 q 513 347 470 419 q 602 197 555 274 q 724 0 649 119 q 672 3 705 0 q 634 7 639 7 q 583 4 605 7 q 547 0 560 1 q 449 197 502 97 q 345 398 397 298 l 259 249 q 170 96 189 130 q 125 0 152 62 q 87 3 111 0 q 61 7 63 7 q 25 4 42 7 q 0 0 9 1 l 196 286 "
    },
    "*": {
      "x_min": 92.953125,
      "x_max": 569.6875,
      "ha": 662,
      "o": "m 331 940 q 361 944 343 940 q 383 949 379 949 q 362 851 369 902 q 356 747 356 799 q 513 874 434 801 q 538 813 530 830 q 569 779 546 797 q 379 702 477 751 q 467 661 419 680 q 565 629 515 643 q 512 534 530 587 q 433 604 470 573 q 356 661 396 634 q 363 564 356 615 q 384 460 370 514 q 354 463 373 460 q 331 467 335 467 q 301 463 320 467 q 278 460 283 460 q 299 568 291 522 q 308 661 308 615 q 151 534 218 602 q 127 589 137 569 q 97 630 117 608 q 183 661 139 643 q 282 704 227 679 q 186 746 230 729 q 92 777 143 763 q 124 818 112 795 q 148 873 136 841 q 224 806 182 840 q 308 747 265 772 q 304 810 308 784 q 278 950 301 835 q 306 943 293 946 q 331 940 319 940 "
    },
    "Â°": {
      "x_min": 173,
      "x_max": 498,
      "ha": 670,
      "o": "m 173 888 q 223 1009 173 960 q 348 1059 274 1059 q 451 1010 405 1059 q 498 903 498 961 q 446 784 498 838 q 330 730 394 730 q 218 775 264 730 q 173 888 173 820 m 237 879 q 262 804 237 835 q 330 774 287 774 q 406 810 379 774 q 433 898 433 846 q 409 984 433 951 q 337 1017 386 1017 q 276 994 302 1017 q 243 939 250 972 q 237 879 237 905 "
    },
    "V": {
      "x_min": 0,
      "x_max": 835.71875,
      "ha": 836,
      "o": "m 187 477 l 73 759 l 0 929 l 81 929 q 132 929 96 929 q 170 929 167 929 q 202 829 183 884 q 243 704 220 773 l 302 548 l 445 154 l 574 502 q 714 929 653 713 l 774 929 l 835 929 q 630 468 733 720 q 461 0 528 215 q 436 4 449 1 q 411 7 424 7 q 377 3 392 7 q 359 0 362 0 q 274 256 325 123 q 187 477 223 390 "
    },
    "Îž": {
      "x_min": 57.5,
      "x_max": 799,
      "ha": 860,
      "o": "m 429 816 q 253 816 370 816 q 76 816 136 816 q 82 849 82 836 q 82 871 82 863 q 76 927 82 899 q 252 927 135 927 q 428 927 369 927 q 605 927 487 927 q 782 927 724 927 l 776 871 q 779 840 776 860 q 782 816 782 821 q 607 816 724 816 q 429 816 490 816 m 461 425 q 328 425 431 425 q 179 425 226 425 q 184 456 184 433 q 184 481 184 480 q 179 537 184 514 q 299 537 211 537 q 428 537 386 537 q 560 537 472 537 q 680 537 649 537 l 673 481 l 680 425 q 560 425 637 425 q 461 425 484 425 m 428 4 q 245 4 369 4 q 57 4 121 4 l 62 63 l 57 123 q 262 123 132 123 q 428 123 393 123 q 623 123 492 123 q 799 123 754 123 l 799 63 l 799 4 q 613 4 736 4 q 428 4 490 4 "
    },
    "Â ": {
      "x_min": 0,
      "x_max": 0,
      "ha": 368
    },
    "Î«": {
      "x_min": -25.671875,
      "x_max": 730.921875,
      "ha": 694,
      "o": "m 289 177 l 289 387 q 186 577 230 501 q 85 747 143 652 q -25 929 28 841 q 23 929 -13 929 q 64 929 60 929 q 108 930 76 929 q 143 932 140 932 q 204 800 171 866 q 272 676 237 734 l 383 474 q 615 929 509 692 l 669 929 q 707 930 687 929 q 730 932 726 932 q 639 785 688 866 q 546 627 590 705 l 426 415 l 426 241 q 426 97 426 184 q 426 0 426 11 q 387 4 405 2 q 353 7 368 7 q 313 4 329 7 q 280 0 298 2 q 289 88 289 36 q 289 177 289 141 m 228 1208 q 280 1187 260 1208 q 301 1133 301 1166 q 279 1080 301 1103 q 228 1058 258 1058 q 176 1080 198 1058 q 154 1133 154 1103 q 176 1186 154 1164 q 228 1208 198 1208 m 465 1208 q 517 1186 496 1208 q 539 1133 539 1164 q 517 1080 539 1103 q 465 1058 496 1058 q 413 1080 435 1058 q 392 1133 392 1102 q 413 1186 392 1164 q 465 1208 434 1208 "
    },
    "0": {
      "x_min": 48,
      "x_max": 686,
      "ha": 734,
      "o": "m 366 910 q 610 773 535 910 q 686 451 686 637 q 615 117 686 253 q 366 -19 545 -19 q 119 114 190 -19 q 48 444 48 247 q 119 775 48 640 q 366 910 190 910 m 184 366 q 223 137 184 238 q 367 37 262 37 q 449 62 413 37 q 510 142 486 88 q 542 271 535 196 q 549 456 549 346 q 517 737 549 622 q 365 852 485 852 q 241 784 275 852 q 195 647 207 717 q 184 445 184 578 l 184 366 "
    },
    "â€": {
      "x_min": 101.65625,
      "x_max": 559,
      "ha": 612,
      "o": "m 180 857 q 207 924 188 898 q 259 951 225 951 q 314 903 309 951 q 303 856 314 881 q 277 812 293 831 l 132 567 l 101 576 l 180 857 m 424 857 q 458 928 443 905 q 505 951 473 951 q 544 938 529 951 q 559 903 559 926 q 550 859 559 881 q 524 812 541 837 l 379 567 l 348 576 l 424 857 "
    },
    "@": {
      "x_min": 76,
      "x_max": 1263,
      "ha": 1339,
      "o": "m 886 641 l 949 641 l 858 267 l 846 205 q 867 159 846 174 q 921 144 889 144 q 1114 264 1041 144 q 1187 514 1187 385 q 1046 802 1187 694 q 721 910 905 910 q 318 754 485 910 q 151 363 151 599 q 295 2 151 136 q 665 -131 439 -131 q 885 -98 775 -131 q 1083 -3 996 -66 l 1106 -41 q 904 -149 1012 -111 q 680 -187 796 -187 q 252 -42 429 -187 q 76 351 76 101 q 267 792 76 615 q 722 970 459 970 q 1099 844 936 970 q 1263 508 1263 718 q 1163 215 1263 347 q 910 84 1064 84 q 816 104 859 84 q 774 171 774 125 l 774 203 q 694 116 746 150 q 583 82 642 82 q 430 139 482 82 q 378 303 378 197 q 461 554 378 442 q 678 666 545 666 q 782 640 739 666 q 847 557 824 614 l 886 641 m 831 478 q 780 573 817 536 q 686 610 742 610 q 518 512 579 610 q 457 297 457 414 q 493 184 457 230 q 593 139 529 139 q 718 189 665 139 q 790 314 771 239 l 831 478 "
    },
    "ÎŠ": {
      "x_min": -1.4375,
      "x_max": 559.875,
      "ha": 676,
      "o": "m 415 465 q 415 711 415 620 q 406 932 415 802 q 452 926 427 926 q 488 926 477 926 q 535 929 507 926 q 559 932 564 932 q 559 788 559 887 q 559 659 559 690 l 559 448 l 559 283 q 559 135 559 238 q 559 0 559 31 q 529 4 556 0 q 488 8 501 8 q 448 5 470 8 q 415 0 427 2 q 415 239 415 70 q 415 465 415 408 m 178 942 q 210 973 195 963 q 246 984 225 984 q 283 968 268 984 q 298 931 298 953 q 283 896 298 910 q 242 866 269 881 l 46 743 l -1 743 l 178 942 "
    },
    "Ã¶": {
      "x_min": 39,
      "x_max": 698,
      "ha": 737,
      "o": "m 364 -15 q 128 79 218 -15 q 39 322 39 173 q 130 572 39 473 q 372 672 222 672 q 608 575 519 672 q 698 327 698 479 q 607 80 698 175 q 364 -15 516 -15 m 248 929 q 301 906 279 929 q 323 854 323 884 q 302 800 323 822 q 252 778 282 778 q 198 800 220 778 q 176 854 176 823 q 196 906 176 884 q 248 929 216 929 m 487 929 q 538 906 516 929 q 560 854 560 884 q 540 799 560 820 q 489 778 520 778 q 434 800 456 778 q 413 854 413 822 q 434 906 413 884 q 487 929 456 929 m 369 622 q 222 532 268 622 q 176 327 176 442 q 222 125 176 216 q 368 34 268 34 q 514 124 469 34 q 559 327 559 214 q 514 531 559 441 q 369 622 470 622 "
    },
    "i": {
      "x_min": 91,
      "x_max": 243,
      "ha": 334,
      "o": "m 98 144 l 98 520 l 98 653 q 161 647 130 647 q 194 648 182 647 q 228 653 206 649 q 223 579 223 616 q 223 505 223 542 q 223 241 223 418 q 223 0 223 64 q 197 3 212 1 q 161 6 182 6 q 117 3 133 6 q 98 0 102 0 l 98 144 m 167 962 q 221 939 200 962 q 243 880 243 916 q 221 830 243 848 q 167 812 199 812 q 113 832 135 812 q 91 884 91 852 q 112 940 91 918 q 167 962 133 962 "
    },
    "Î’": {
      "x_min": 110.5625,
      "x_max": 737,
      "ha": 786,
      "o": "m 116 560 q 110 932 116 759 l 203 932 q 334 932 242 932 q 432 932 426 932 q 618 880 540 932 q 696 724 696 829 q 626 578 696 632 q 456 503 557 524 q 660 438 584 489 q 737 269 737 387 q 628 69 737 135 q 376 3 519 3 l 229 3 q 160 3 208 3 q 116 3 111 3 q 116 289 116 88 q 116 560 116 489 m 255 255 l 255 146 l 255 59 l 330 59 q 516 109 441 59 q 591 266 591 160 q 514 427 591 385 q 310 469 438 469 l 255 469 l 255 255 m 559 705 q 499 831 559 786 q 355 876 439 876 l 257 876 q 248 710 248 799 l 248 522 q 468 558 377 522 q 559 705 559 595 "
    },
    "â‰¤": {
      "x_min": 173,
      "x_max": 943,
      "ha": 1116,
      "o": "m 283 491 l 943 266 l 943 195 l 173 462 l 173 522 l 943 787 l 943 717 l 283 491 m 943 26 l 173 26 l 173 93 l 943 93 l 943 26 "
    },
    "Ï…": {
      "x_min": 80,
      "x_max": 691,
      "ha": 758,
      "o": "m 691 397 q 586 110 691 236 q 327 -15 482 -15 q 145 54 210 -15 q 80 244 80 123 l 80 430 q 80 542 80 486 q 80 651 80 598 l 145 651 l 207 651 q 207 503 207 608 q 207 366 207 398 l 207 261 q 244 105 207 168 q 367 43 282 43 q 516 133 470 43 q 562 341 562 223 q 540 493 562 418 q 480 646 518 568 q 541 647 521 646 q 617 657 562 649 q 691 397 691 537 "
    },
    "]": {
      "x_min": 81.734375,
      "x_max": 326,
      "ha": 440,
      "o": "m 205 -157 l 81 -157 l 81 -129 l 81 -98 q 180 -104 125 -104 l 226 -104 l 226 386 l 226 879 l 180 879 q 131 876 158 879 q 82 872 105 874 l 82 902 l 82 930 l 200 930 l 326 930 q 323 854 326 912 q 320 769 320 797 l 320 456 l 320 2 l 320 -157 l 205 -157 "
    },
    "m": {
      "x_min": 89.90625,
      "x_max": 1051,
      "ha": 1144,
      "o": "m 94 155 l 94 494 q 94 567 94 529 q 89 654 94 606 q 152 647 121 647 l 215 654 l 215 537 q 313 637 258 604 q 442 671 367 671 q 618 545 570 671 q 720 638 664 605 q 845 672 775 672 q 1001 606 952 672 q 1051 429 1051 540 l 1051 298 l 1051 136 l 1051 0 q 1018 3 1042 0 q 987 7 993 7 q 949 3 973 7 q 926 0 926 0 q 926 202 926 68 q 926 405 926 337 q 895 536 926 486 q 792 586 865 586 q 685 539 729 586 q 636 451 641 492 q 632 375 632 409 q 632 326 632 341 l 632 313 q 632 132 632 258 q 632 0 632 7 q 596 4 610 2 q 570 5 582 5 q 532 2 559 5 q 509 0 506 0 q 509 168 509 55 q 509 320 509 280 l 509 405 q 477 533 509 481 q 374 586 445 586 q 269 540 309 586 q 222 436 229 495 q 215 313 215 376 q 215 133 215 244 q 215 0 215 22 q 180 3 204 0 q 152 7 156 7 q 115 4 132 7 q 94 0 99 1 l 94 155 "
    },
    "Ï‡": {
      "x_min": -1,
      "x_max": 710.859375,
      "ha": 698,
      "o": "m 490 425 q 543 535 517 475 q 595 651 569 595 l 661 651 l 710 651 q 610 506 659 585 q 509 330 562 427 l 435 196 q 563 -86 497 55 q 705 -371 630 -228 q 652 -371 687 -371 q 599 -371 616 -371 l 548 -371 q 435 -116 487 -230 q 351 71 384 -1 l 250 -96 q 174 -237 204 -175 q 122 -371 144 -300 l 61 -371 l 0 -371 q 122 -193 67 -279 q 235 -5 178 -107 l 317 135 l 197 402 q 143 518 170 460 q 42 587 101 587 l -1 582 l -1 637 q 125 672 56 672 q 218 627 177 672 q 286 521 258 582 l 352 374 l 403 259 l 490 425 "
    },
    "8": {
      "x_min": 59,
      "x_max": 677,
      "ha": 734,
      "o": "m 109 695 q 184 852 109 797 q 364 908 260 908 q 548 853 472 908 q 625 694 625 799 q 579 575 625 621 q 459 510 533 529 l 459 498 q 619 413 562 475 q 677 247 677 351 q 587 51 677 120 q 368 -18 497 -18 q 147 48 235 -18 q 59 247 59 115 q 117 413 59 347 q 277 498 175 480 l 277 510 q 155 570 202 521 q 109 695 109 619 m 364 531 q 463 577 429 531 q 498 693 498 623 q 461 809 498 763 q 360 855 425 855 q 267 806 300 855 q 234 693 234 758 q 267 578 234 625 q 364 531 300 531 m 367 31 q 497 97 453 31 q 540 255 540 164 q 497 415 540 348 q 367 482 455 482 q 235 416 279 482 q 192 255 192 351 q 236 96 192 162 q 367 31 280 31 "
    },
    "Î¯": {
      "x_min": 93.921875,
      "x_max": 421,
      "ha": 335,
      "o": "m 98 333 l 98 520 q 98 566 98 544 q 93 653 98 588 q 138 647 127 647 q 161 647 148 647 q 228 653 191 647 q 223 555 223 599 q 223 437 223 511 l 223 406 q 223 194 223 337 q 223 0 223 51 q 197 3 212 1 q 161 6 182 6 q 125 3 145 6 q 98 0 104 1 q 98 165 98 51 q 98 333 98 279 m 302 943 q 334 974 319 964 q 370 985 350 985 q 407 969 393 985 q 421 932 421 954 q 406 896 421 912 q 366 866 392 880 l 170 743 l 122 743 l 302 943 "
    },
    "Î–": {
      "x_min": 7,
      "x_max": 786,
      "ha": 812,
      "o": "m 7 36 q 218 325 110 176 q 416 605 326 473 l 585 857 l 424 857 q 58 836 242 857 l 64 883 l 58 926 q 257 926 132 926 q 419 926 382 926 q 610 926 481 926 q 786 926 739 926 l 786 903 q 582 630 695 786 q 391 364 469 475 q 197 82 313 253 l 419 82 q 570 82 510 82 q 786 102 630 82 l 781 68 l 780 54 l 780 34 l 786 4 q 494 4 669 4 q 318 4 320 4 q 163 4 266 4 q 7 4 60 4 l 7 36 "
    },
    "R": {
      "x_min": 110.5625,
      "x_max": 791.640625,
      "ha": 770,
      "o": "m 251 0 q 219 4 244 0 q 182 8 195 8 q 146 5 166 8 q 116 0 125 3 q 116 306 116 90 q 116 598 116 523 q 116 760 116 671 q 110 932 116 849 l 203 932 l 403 932 q 607 880 520 932 q 695 718 695 828 q 622 559 695 614 q 434 473 550 504 q 609 239 523 355 q 791 3 695 124 l 719 3 l 599 3 q 441 240 520 127 q 278 455 362 353 l 251 455 l 251 307 q 251 138 251 245 q 251 0 251 31 m 560 698 q 495 832 560 789 q 336 876 430 876 l 253 876 q 248 772 248 835 q 248 701 248 710 l 248 504 q 471 542 382 504 q 560 698 560 581 "
    },
    "Ã—": {
      "x_min": 200.078125,
      "x_max": 918.75,
      "ha": 1116,
      "o": "m 873 774 l 918 728 l 604 406 l 918 86 l 873 38 l 559 361 l 242 38 l 200 86 l 514 406 l 200 728 l 242 774 l 559 454 l 873 774 "
    },
    "o": {
      "x_min": 41,
      "x_max": 697,
      "ha": 738,
      "o": "m 364 -15 q 130 77 219 -15 q 41 321 41 171 q 131 573 41 474 q 371 672 222 672 q 607 574 518 672 q 697 325 697 477 q 606 79 697 174 q 364 -15 515 -15 m 370 619 q 222 530 268 619 q 177 327 177 441 q 223 125 177 216 q 369 34 269 34 q 515 122 470 34 q 560 325 560 210 q 515 529 560 439 q 370 619 471 619 "
    },
    "5": {
      "x_min": 73.6875,
      "x_max": 641,
      "ha": 734,
      "o": "m 114 201 q 173 77 129 120 q 298 35 217 35 q 445 98 389 35 q 502 255 502 161 q 449 407 502 346 q 307 469 396 469 q 167 417 223 469 l 147 428 l 154 524 q 154 661 154 568 q 154 800 154 753 l 144 883 l 375 883 l 404 883 q 617 883 510 883 q 613 848 614 870 q 611 817 611 825 l 611 759 l 416 759 l 217 759 q 217 629 217 715 q 217 499 217 543 q 369 534 291 534 q 561 463 481 534 q 641 278 641 392 q 539 60 641 138 q 298 -18 438 -18 q 176 -6 227 -18 q 73 35 125 4 q 99 201 85 118 l 114 201 "
    },
    "7": {
      "x_min": 121.109375,
      "x_max": 716,
      "ha": 734,
      "o": "m 453 554 l 575 771 l 356 771 q 127 755 228 771 q 132 786 132 769 q 132 821 132 804 q 126 885 132 853 l 401 885 l 716 885 l 716 871 q 455 429 571 641 q 247 0 338 217 l 191 8 q 151 5 167 8 q 121 0 136 3 q 210 146 176 92 q 327 339 244 201 q 453 554 409 478 "
    },
    "K": {
      "x_min": 105.84375,
      "x_max": 839.921875,
      "ha": 805,
      "o": "m 256 314 q 256 132 256 250 q 256 0 256 14 q 224 4 250 0 q 185 9 198 9 q 145 4 171 9 q 114 0 119 0 q 114 240 114 71 q 114 465 114 408 q 114 711 114 621 q 105 932 114 802 q 150 928 122 932 q 184 925 179 925 q 227 928 200 925 q 256 932 254 932 l 256 671 l 256 498 q 473 693 372 586 q 673 932 574 800 q 718 930 697 930 q 762 930 739 930 l 820 930 q 594 727 706 830 q 381 522 483 623 q 515 358 456 426 q 653 203 574 290 q 839 -2 732 115 l 732 -2 q 679 -2 702 -2 q 631 -8 656 -2 q 561 92 599 45 q 470 204 523 140 l 256 458 l 256 314 "
    },
    ",": {
      "x_min": 39.6875,
      "x_max": 267,
      "ha": 366,
      "o": "m 129 75 q 157 147 141 120 q 209 175 173 175 q 267 119 267 175 q 254 67 267 91 q 227 18 241 43 l 72 -242 l 39 -231 l 129 75 "
    },
    "d": {
      "x_min": 54,
      "x_max": 658,
      "ha": 742,
      "o": "m 658 762 l 658 137 l 658 -1 q 625 3 640 1 q 598 6 610 6 q 562 2 585 6 q 540 -1 540 -1 l 540 119 q 329 -15 473 -15 q 124 86 195 -15 q 54 330 54 187 q 124 569 54 467 q 326 672 195 672 q 449 643 394 672 q 540 555 503 615 l 540 755 l 540 788 l 540 818 q 530 1024 540 927 q 597 1018 564 1018 q 626 1019 616 1018 q 658 1024 637 1020 l 658 762 m 367 57 q 504 139 464 57 q 545 332 545 222 q 503 522 545 437 q 367 607 461 607 q 231 522 273 607 q 190 332 190 438 q 230 141 190 225 q 367 57 271 57 "
    },
    "Â¨": {
      "x_min": 83,
      "x_max": 539.96875,
      "ha": 612,
      "o": "m 462 659 q 433 592 450 617 q 382 568 416 568 q 343 579 358 568 q 329 613 329 591 q 340 660 329 633 q 364 706 352 687 l 510 949 l 539 940 l 462 659 m 217 659 q 189 592 208 617 q 136 568 171 568 q 83 613 83 568 q 92 658 83 637 q 118 706 102 680 l 263 949 l 295 940 l 217 659 "
    },
    "E": {
      "x_min": 105.84375,
      "x_max": 600.359375,
      "ha": 673,
      "o": "m 114 465 q 114 711 114 620 q 105 928 114 802 l 346 928 l 599 928 l 594 883 l 599 835 q 498 852 551 846 q 399 859 444 859 l 254 859 l 254 670 l 254 527 l 391 527 q 574 540 471 527 l 574 496 l 574 456 l 386 456 l 256 456 l 256 313 l 256 76 q 446 76 371 76 q 600 94 521 76 l 600 46 l 600 1 l 351 1 l 114 1 q 114 240 114 71 q 114 465 114 408 "
    },
    "Y": {
      "x_min": -27.40625,
      "x_max": 730.890625,
      "ha": 693,
      "o": "m 287 177 l 287 386 q 185 570 248 458 q 81 750 121 681 q -27 932 42 819 q 24 928 -9 932 q 62 925 58 925 q 110 927 89 925 q 144 932 131 930 q 203 800 171 866 q 270 676 235 734 l 382 475 q 499 688 442 575 q 615 932 556 801 l 669 926 q 700 927 687 926 q 730 932 714 929 q 545 627 627 769 l 426 415 l 426 240 q 426 102 426 198 q 426 0 426 5 q 393 4 417 1 q 353 7 368 7 q 312 4 327 7 q 278 0 297 2 q 287 88 287 36 q 287 177 287 140 "
    },
    "\"": {
      "x_min": 63,
      "x_max": 310,
      "ha": 373,
      "o": "m 131 587 l 63 587 l 63 956 l 131 956 l 131 587 m 310 587 l 243 587 l 243 956 l 310 956 l 310 587 "
    },
    "Ãª": {
      "x_min": 39,
      "x_max": 632,
      "ha": 666,
      "o": "m 398 42 q 590 129 512 42 l 609 126 q 601 90 605 109 q 597 46 597 71 q 486 0 546 14 q 361 -15 426 -15 q 127 73 215 -15 q 39 311 39 162 q 123 562 39 456 q 347 669 207 669 q 551 590 470 669 q 632 387 632 512 l 632 332 q 429 332 550 332 q 307 332 308 332 l 175 332 q 230 127 175 212 q 398 42 285 42 m 289 977 l 377 977 l 526 743 l 478 743 l 332 874 l 188 743 l 139 743 l 289 977 m 503 390 l 503 436 q 460 563 503 508 q 349 618 418 618 q 228 552 266 618 q 179 390 191 487 l 503 390 "
    },
    "Î´": {
      "x_min": 41,
      "x_max": 657,
      "ha": 695,
      "o": "m 101 840 q 182 981 101 937 q 368 1025 264 1025 q 488 1010 428 1025 q 609 967 548 995 q 583 876 593 923 q 491 947 537 923 q 386 972 445 972 q 275 937 322 972 q 229 840 229 902 q 267 742 229 794 q 426 627 306 691 q 601 504 545 563 q 657 305 657 445 q 569 75 657 165 q 344 -15 482 -15 q 123 73 206 -15 q 41 305 41 162 q 112 526 41 430 q 301 622 183 622 q 143 726 186 674 q 101 840 101 779 m 349 576 q 217 492 257 576 q 178 302 178 409 q 219 118 178 203 q 349 34 261 34 q 481 116 442 34 q 520 307 520 198 q 480 492 520 409 q 349 576 440 576 "
    },
    "Î­": {
      "x_min": 52,
      "x_max": 587,
      "ha": 616,
      "o": "m 497 516 q 425 591 459 566 q 346 616 392 616 q 257 580 293 616 q 221 493 221 545 q 263 407 221 438 q 365 376 305 376 l 409 376 l 409 351 l 409 314 l 338 314 q 232 279 276 314 q 189 183 189 245 q 234 76 189 116 q 345 37 279 37 q 452 65 403 37 q 534 144 501 94 q 543 98 538 121 q 557 53 548 75 q 444 1 506 18 q 316 -16 381 -16 q 132 30 212 -16 q 52 176 52 77 q 94 286 52 245 q 209 355 136 326 q 125 408 159 370 q 92 496 92 445 q 162 625 92 578 q 317 672 233 672 q 422 654 372 672 q 531 604 472 637 q 512 564 520 584 q 497 516 505 545 m 467 943 q 500 974 485 964 q 536 985 516 985 q 572 969 558 985 q 587 932 587 953 q 573 897 587 911 q 532 867 559 883 l 335 744 l 287 744 l 467 943 "
    },
    "Ï‰": {
      "x_min": 39,
      "x_max": 1008,
      "ha": 1046,
      "o": "m 523 526 l 582 526 l 582 305 q 603 117 582 193 q 708 42 625 42 q 833 118 795 42 q 871 293 871 194 q 812 502 871 409 q 651 654 753 594 l 703 648 q 789 654 751 648 q 947 510 887 607 q 1008 304 1008 414 q 929 81 1008 178 q 728 -15 850 -15 q 609 16 664 -15 q 523 104 554 47 q 436 16 490 48 q 316 -15 381 -15 q 115 83 192 -15 q 39 311 39 182 q 98 511 39 419 q 256 654 158 604 q 296 650 272 650 q 342 650 320 650 l 393 654 q 233 501 290 592 q 176 291 176 409 q 214 116 176 191 q 340 42 252 42 q 410 68 381 42 q 451 137 438 94 q 462 216 462 175 q 462 302 462 256 q 462 418 462 352 q 462 526 462 484 l 523 526 "
    },
    "Â´": {
      "x_min": 88.03125,
      "x_max": 299,
      "ha": 366,
      "o": "m 165 858 q 193 922 173 893 q 244 952 213 952 q 282 938 266 952 q 299 902 299 924 q 288 856 299 881 q 263 810 278 832 l 117 567 l 88 575 l 165 858 "
    },
    "Â±": {
      "x_min": 166,
      "x_max": 950,
      "ha": 1116,
      "o": "m 591 548 l 950 548 l 950 481 l 591 481 l 591 247 l 524 247 l 524 481 l 166 481 l 166 548 l 524 548 l 524 779 l 591 779 l 591 548 m 950 33 l 166 33 l 166 100 l 950 100 l 950 33 "
    },
    "|": {
      "x_min": 299,
      "x_max": 368,
      "ha": 669,
      "o": "m 368 449 l 299 449 l 299 956 l 368 956 l 368 449 m 368 -233 l 299 -233 l 299 272 l 368 272 l 368 -233 "
    },
    "Ï‹": {
      "x_min": 80,
      "x_max": 691,
      "ha": 758,
      "o": "m 691 395 q 586 110 691 236 q 327 -15 481 -15 q 145 54 210 -15 q 80 244 80 123 l 80 429 q 80 542 80 486 q 80 651 80 598 l 145 651 l 207 651 q 207 502 207 608 q 207 365 207 397 l 207 261 q 244 105 207 168 q 367 43 282 43 q 516 132 470 43 q 562 340 562 222 q 540 493 562 418 q 480 646 518 568 q 541 647 520 646 q 617 657 562 649 q 691 395 691 537 m 225 928 q 278 907 256 928 q 300 852 300 886 q 279 800 300 822 q 228 778 259 778 q 173 800 195 778 q 152 852 152 822 q 173 906 152 884 q 225 928 194 928 m 462 928 q 514 906 493 928 q 536 852 536 884 q 516 799 536 821 q 465 778 496 778 q 411 800 433 778 q 390 852 390 822 q 410 905 390 883 q 462 928 430 928 "
    },
    "Â§": {
      "x_min": 63,
      "x_max": 608,
      "ha": 662,
      "o": "m 111 36 l 125 36 q 194 -76 145 -33 q 313 -120 242 -120 q 422 -78 377 -120 q 467 27 467 -37 q 362 169 467 116 q 167 252 263 210 q 63 418 63 312 q 90 521 63 475 q 167 608 117 567 q 120 738 120 668 q 191 895 120 836 q 362 955 262 955 q 463 939 413 955 q 565 893 513 924 q 548 863 556 881 q 510 774 540 844 l 498 774 q 436 870 473 838 q 336 902 398 902 q 245 866 282 902 q 208 776 208 831 q 311 643 208 693 q 503 563 407 603 q 608 407 608 506 q 581 299 608 348 q 505 213 554 251 q 549 143 534 178 q 565 65 565 108 q 484 -112 565 -46 q 292 -178 404 -178 q 176 -163 234 -178 q 75 -120 118 -149 q 95 -42 84 -89 q 111 36 106 3 m 135 478 q 252 339 135 395 q 469 237 369 284 q 514 285 498 260 q 530 341 530 309 q 376 489 530 423 q 193 586 223 556 q 153 531 171 563 q 135 478 135 500 "
    },
    "b": {
      "x_min": 75.828125,
      "x_max": 688.109375,
      "ha": 743,
      "o": "m 149 1017 q 215 1023 176 1017 q 207 915 207 965 q 207 788 207 865 l 207 754 l 207 555 q 410 672 277 672 q 616 569 544 672 q 688 322 688 466 q 617 79 688 173 q 405 -15 546 -15 q 289 7 342 -15 q 189 77 236 30 q 156 48 171 63 q 116 -2 141 33 l 75 -2 q 84 105 84 42 q 84 213 84 169 l 84 545 q 84 784 84 624 q 84 1023 84 943 q 149 1017 117 1017 m 375 605 q 238 520 278 605 q 197 323 197 435 q 239 132 197 217 q 375 47 281 47 q 511 131 470 47 q 552 326 552 216 q 511 519 552 434 q 375 605 470 605 "
    },
    "q": {
      "x_min": 53,
      "x_max": 657,
      "ha": 743,
      "o": "m 595 -369 q 567 -369 579 -369 q 528 -373 556 -369 q 537 -213 537 -312 q 537 -101 537 -115 l 537 98 q 331 -15 464 -15 q 127 82 202 -15 q 53 316 53 180 q 123 564 53 456 q 324 672 193 672 q 452 636 396 672 q 542 534 507 601 q 534 655 542 594 q 574 649 564 649 q 598 649 584 649 q 657 655 627 649 l 657 433 q 657 163 657 298 q 657 -106 657 27 l 657 -373 q 629 -369 644 -370 q 595 -369 614 -369 m 365 47 q 504 133 463 47 q 545 330 545 219 q 501 514 545 433 q 364 596 456 596 q 229 512 270 596 q 189 322 189 429 q 231 132 189 217 q 365 47 273 47 "
    },
    "Î©": {
      "x_min": 7.921875,
      "x_max": 1095.375,
      "ha": 1106,
      "o": "m 439 5 q 311 5 400 5 q 218 5 222 5 l 12 5 l 12 47 q 10 74 12 60 q 7 96 9 89 q 116 88 67 88 q 244 88 165 88 q 101 262 149 171 q 54 474 54 353 q 196 822 54 694 q 556 950 338 950 q 912 823 775 950 q 1050 474 1050 697 q 1004 262 1050 350 q 861 91 959 174 q 972 91 884 91 q 1094 91 1061 91 l 1094 34 l 1095 5 l 888 5 q 762 5 850 5 q 668 5 674 5 l 668 75 q 846 218 790 125 q 902 453 902 311 q 816 764 902 640 q 553 889 731 889 q 290 767 376 889 q 205 456 205 646 q 262 220 205 321 q 439 75 320 118 l 439 5 "
    },
    "Ï": {
      "x_min": 80,
      "x_max": 691,
      "ha": 758,
      "o": "m 691 395 q 586 110 691 236 q 327 -15 481 -15 q 145 54 210 -15 q 80 244 80 123 l 80 429 q 80 541 80 485 q 80 650 80 597 l 145 650 l 207 650 q 207 502 207 607 q 207 365 207 397 l 207 261 q 244 105 207 168 q 367 43 282 43 q 516 132 470 43 q 562 340 562 222 q 540 492 562 416 q 480 645 518 567 q 541 646 521 645 q 617 656 562 648 q 691 395 691 536 m 491 941 q 523 972 508 962 q 558 983 538 983 q 595 967 581 983 q 609 930 609 952 q 594 895 609 910 q 555 865 580 879 l 359 742 l 311 742 l 491 941 "
    },
    "Ã–": {
      "x_min": 50,
      "x_max": 1048,
      "ha": 1098,
      "o": "m 50 465 q 188 819 50 688 q 548 950 326 950 q 873 852 736 950 q 1026 659 1011 755 q 1044 535 1041 562 q 1048 462 1048 508 q 1045 395 1048 416 q 1026 275 1042 373 q 876 80 1011 176 q 551 -15 741 -15 q 442 -8 493 -15 q 298 31 391 -2 q 127 187 205 65 q 50 465 50 308 m 431 1209 q 481 1187 460 1209 q 503 1134 503 1165 q 481 1081 503 1104 q 431 1059 460 1059 q 378 1081 401 1059 q 355 1134 355 1103 q 377 1187 355 1165 q 431 1209 399 1209 m 667 1209 q 719 1187 698 1209 q 741 1134 741 1165 q 719 1081 741 1104 q 667 1059 698 1059 q 615 1080 636 1059 q 594 1134 594 1102 q 615 1187 594 1165 q 667 1209 636 1209 m 200 469 q 286 162 200 283 q 548 42 372 42 q 811 162 725 42 q 898 469 898 283 q 809 768 898 649 q 548 888 721 888 q 342 826 421 888 q 221 639 262 764 q 200 469 200 558 "
    },
    "z": {
      "x_min": 15,
      "x_max": 595,
      "ha": 634,
      "o": "m 15 29 q 161 224 87 124 q 296 416 235 323 l 409 586 l 265 586 q 178 581 213 586 q 61 565 142 577 l 66 609 l 61 650 q 178 650 113 650 q 324 650 243 650 l 356 650 l 385 650 q 595 650 490 650 l 595 626 q 444 428 535 549 q 311 246 354 306 q 190 68 268 186 l 336 68 q 447 72 392 68 q 582 86 503 77 l 577 42 l 582 4 l 275 4 q 145 4 232 4 q 15 4 58 4 l 15 29 "
    },
    "â„¢": {
      "x_min": 173,
      "x_max": 900,
      "ha": 1116,
      "o": "m 453 930 l 339 930 l 339 613 l 287 613 l 287 930 l 173 930 l 173 969 l 453 969 l 453 930 m 721 688 l 824 969 l 900 969 l 900 614 l 854 614 l 854 927 l 738 614 l 705 614 l 583 927 l 583 614 l 538 614 l 538 969 l 612 969 l 721 688 "
    },
    "Î®": {
      "x_min": 90,
      "x_max": 647,
      "ha": 740,
      "o": "m 584 -365 q 548 -367 567 -365 q 523 -370 528 -369 q 523 -190 523 -310 q 523 -8 523 -69 l 523 406 q 490 536 523 485 q 383 587 457 587 q 251 508 285 587 q 218 314 218 430 q 218 133 218 259 q 218 1 218 8 l 155 6 l 94 1 l 94 156 l 94 495 q 92 583 94 527 q 90 655 90 639 q 116 650 102 652 q 152 648 130 648 q 181 649 171 648 q 215 655 191 650 l 215 538 q 314 637 258 602 q 444 672 370 672 q 595 603 544 672 q 647 430 647 535 l 647 328 l 647 -185 q 647 -293 647 -227 q 647 -370 647 -359 q 619 -366 634 -367 q 584 -365 604 -365 m 472 943 q 505 974 490 964 q 541 985 521 985 q 578 969 564 985 q 592 932 592 954 q 578 897 592 911 q 537 867 564 883 l 341 744 l 292 744 l 472 943 "
    },
    "Î˜": {
      "x_min": 50,
      "x_max": 1046,
      "ha": 1096,
      "o": "m 50 465 q 189 820 50 690 q 550 950 328 950 q 875 852 739 950 q 1025 653 1010 755 q 1043 525 1040 551 q 1046 462 1046 500 q 1043 402 1046 426 q 1026 280 1040 379 q 873 83 1011 182 q 549 -15 735 -15 q 426 -7 470 -15 q 293 32 382 0 q 127 187 205 65 q 50 465 50 308 m 198 468 q 285 163 198 283 q 547 43 372 43 q 809 163 723 43 q 896 468 896 283 q 809 770 896 652 q 547 889 723 889 q 249 720 300 889 q 198 468 198 552 m 548 510 q 669 510 595 510 q 751 510 742 510 l 751 465 l 751 432 q 637 432 718 432 q 548 432 556 432 q 432 432 513 432 q 345 432 350 432 l 345 471 l 345 510 q 464 510 391 510 q 548 510 537 510 "
    },
    "Â®": {
      "x_min": 79,
      "x_max": 1037,
      "ha": 1116,
      "o": "m 800 904 q 973 725 909 840 q 1037 481 1037 610 q 894 138 1037 282 q 556 -6 751 -6 q 221 139 363 -6 q 79 481 79 284 q 220 825 79 681 q 558 970 362 970 q 800 904 684 970 m 558 917 q 258 787 385 917 q 132 480 132 658 q 257 175 132 306 q 555 45 382 45 q 855 174 726 45 q 984 478 984 304 q 928 698 984 596 q 771 858 872 800 q 558 917 670 917 m 799 618 q 761 520 799 561 q 668 467 724 479 l 793 209 l 697 209 l 581 456 l 453 456 l 453 209 l 369 209 l 369 770 l 570 770 q 730 736 662 770 q 799 618 799 703 m 453 713 l 453 512 l 555 512 q 669 531 623 512 q 715 610 715 550 q 668 691 715 670 q 553 713 622 713 l 453 713 "
    },
    "Ã‰": {
      "x_min": 114,
      "x_max": 601.078125,
      "ha": 671,
      "o": "m 114 465 q 114 709 114 540 q 114 930 114 879 l 350 930 l 598 930 q 597 906 597 920 q 597 885 597 892 l 597 836 q 399 859 496 859 l 254 859 l 254 669 l 254 529 l 390 529 q 575 541 483 529 q 572 517 572 529 q 572 498 572 505 q 575 449 572 464 q 478 458 532 458 q 387 458 424 458 l 254 458 l 254 316 l 254 76 q 443 76 372 76 q 601 94 514 76 l 601 46 l 601 1 l 349 1 l 114 1 q 114 244 114 73 q 114 465 114 415 m 419 1224 q 453 1254 436 1243 q 489 1266 469 1266 q 526 1250 513 1266 q 539 1211 539 1235 q 523 1175 539 1191 q 483 1146 507 1159 l 286 1023 l 237 1023 l 419 1224 "
    },
    "~": {
      "x_min": 279,
      "x_max": 1059,
      "ha": 1341,
      "o": "m 834 650 q 654 750 747 650 q 502 850 562 850 q 389 793 423 850 q 355 650 355 737 l 279 650 q 333 846 279 768 q 499 924 387 924 q 683 824 594 924 q 837 725 773 725 q 950 779 918 725 q 983 924 983 834 l 1059 924 q 1003 727 1059 805 q 834 650 947 650 "
    },
    "Î•": {
      "x_min": 105.84375,
      "x_max": 600.359375,
      "ha": 673,
      "o": "m 114 465 q 114 711 114 620 q 105 928 114 802 l 346 928 l 599 928 l 594 883 l 599 835 q 498 852 551 846 q 399 859 444 859 l 254 859 l 254 670 l 254 527 l 391 527 q 574 540 471 527 l 574 496 l 574 456 l 386 456 l 256 456 l 256 313 l 256 76 q 446 76 371 76 q 600 94 521 76 l 600 46 l 600 1 l 351 1 l 114 1 q 114 240 114 71 q 114 465 114 408 "
    },
    "Â³": {
      "x_min": 47.390625,
      "x_max": 416,
      "ha": 485,
      "o": "m 156 636 l 153 663 q 175 661 164 661 q 197 661 186 661 q 274 690 243 661 q 306 767 306 719 q 208 869 306 869 q 140 846 165 869 q 105 781 115 823 l 100 778 q 83 812 92 795 q 65 845 74 830 q 132 886 96 871 q 207 901 168 901 q 334 873 278 901 q 390 778 390 845 q 348 691 390 724 q 247 643 307 658 q 366 610 317 643 q 416 509 416 578 q 347 385 416 427 q 192 343 278 343 q 47 374 115 343 q 56 422 51 391 q 65 469 62 453 l 74 469 q 118 399 86 424 q 195 374 150 374 q 289 409 250 374 q 328 500 328 444 q 293 589 328 554 q 203 624 259 624 q 176 621 190 624 q 153 617 163 618 l 156 636 "
    },
    "[": {
      "x_min": 114,
      "x_max": 356,
      "ha": 439,
      "o": "m 119 3 l 119 386 l 119 769 l 119 930 l 232 930 l 355 930 l 355 907 l 355 872 q 257 879 308 879 l 213 879 l 213 386 l 213 -106 l 259 -106 q 323 -103 302 -106 q 356 -99 345 -100 l 356 -123 l 356 -157 l 233 -157 l 114 -157 q 116 -80 114 -137 q 119 3 119 -23 "
    },
    "L": {
      "x_min": 105.84375,
      "x_max": 613.609375,
      "ha": 616,
      "o": "m 114 465 q 114 711 114 620 q 105 931 114 801 q 146 928 123 931 q 184 925 169 926 q 229 928 198 925 q 257 931 259 931 q 257 796 257 882 q 257 683 257 710 q 257 331 257 577 q 257 73 257 86 l 393 73 q 501 78 448 73 q 613 93 555 84 l 613 46 l 613 1 l 350 1 l 114 1 q 114 240 114 71 q 114 465 114 408 "
    },
    "Ïƒ": {
      "x_min": 41,
      "x_max": 786.828125,
      "ha": 790,
      "o": "m 614 647 l 786 647 l 782 610 l 786 569 l 614 569 q 678 464 659 527 q 697 327 697 402 q 605 80 697 177 q 364 -15 513 -15 q 130 77 219 -15 q 41 321 41 171 q 129 571 41 471 q 365 672 218 672 q 431 665 396 672 q 499 653 467 659 q 559 647 531 647 q 614 647 587 647 m 370 619 q 223 530 268 619 q 178 325 178 442 q 220 123 178 213 q 363 34 263 34 q 512 122 465 34 q 560 325 560 210 q 515 529 560 439 q 370 619 471 619 "
    },
    "Î¶": {
      "x_min": 73,
      "x_max": 661,
      "ha": 669,
      "o": "m 628 986 l 628 949 q 416 761 513 860 q 254 545 319 662 q 190 303 190 428 q 190 278 190 290 q 198 215 190 267 q 328 141 206 163 q 546 116 437 128 q 661 -7 661 91 q 630 -109 661 -65 q 532 -232 599 -154 l 489 -205 q 553 -113 543 -132 q 564 -66 564 -94 q 540 -20 564 -37 q 285 22 498 2 q 73 240 73 41 q 140 486 73 356 q 291 718 207 616 q 497 944 374 820 l 351 944 q 247 944 294 944 q 127 932 199 944 l 131 977 l 127 1020 q 266 1020 181 1020 q 364 1020 350 1020 q 515 1020 419 1020 q 628 1020 610 1020 l 628 986 "
    },
    "Î¸": {
      "x_min": 48,
      "x_max": 686,
      "ha": 734,
      "o": "m 369 910 q 610 774 535 910 q 686 457 686 639 q 615 118 686 255 q 366 -19 545 -19 q 119 114 190 -19 q 48 444 48 247 q 120 774 48 639 q 369 910 193 910 m 181 384 q 217 136 181 242 q 367 30 254 30 q 524 138 495 30 q 553 425 553 245 l 401 425 l 181 425 l 181 384 m 364 859 q 253 805 295 859 q 193 674 210 751 q 183 580 183 630 q 183 481 183 530 l 317 481 q 453 481 358 481 q 554 481 548 481 q 519 747 554 636 q 364 859 484 859 "
    },
    "ÎŸ": {
      "x_min": 50,
      "x_max": 1046,
      "ha": 1096,
      "o": "m 50 465 q 188 820 50 690 q 548 950 326 950 q 874 853 738 950 q 1025 654 1010 757 q 1043 525 1040 551 q 1046 462 1046 500 q 1043 402 1046 426 q 1025 277 1040 379 q 875 80 1010 175 q 549 -15 741 -15 q 438 -10 486 -15 q 298 29 390 -5 q 128 186 206 64 q 50 465 50 308 m 198 468 q 285 162 198 282 q 547 42 372 42 q 809 162 723 42 q 896 468 896 283 q 808 770 896 651 q 547 889 720 889 q 341 826 421 889 q 221 639 261 764 q 198 468 198 552 "
    },
    "Î“": {
      "x_min": 105.84375,
      "x_max": 614.96875,
      "ha": 616,
      "o": "m 442 930 l 614 930 q 610 900 610 920 q 610 870 610 879 l 614 835 q 393 859 502 859 l 257 859 q 257 583 257 775 q 257 377 257 391 q 257 180 257 311 q 257 2 257 48 l 183 2 l 105 2 q 114 208 114 106 q 114 427 114 310 q 114 686 114 503 q 114 930 114 868 l 442 930 "
    },
    " ": {
      "x_min": 0,
      "x_max": 0,
      "ha": 368
    },
    "%": {
      "x_min": 27,
      "x_max": 991,
      "ha": 1011,
      "o": "m 783 0 q 633 62 693 0 q 574 218 574 124 q 634 372 574 309 q 783 436 695 436 q 931 373 871 436 q 991 223 991 310 q 976 130 991 176 q 900 35 952 70 q 783 0 847 0 m 845 1014 l 222 -126 l 154 -126 l 777 1014 l 845 1014 m 236 451 q 86 513 146 451 q 27 667 27 575 q 86 823 27 759 q 236 887 146 887 q 384 824 324 887 q 444 673 444 762 q 426 581 444 623 q 352 486 404 522 q 236 451 300 451 m 878 260 q 853 353 878 310 q 782 397 829 397 q 704 340 722 397 q 686 202 686 283 q 705 86 686 133 q 782 40 724 40 q 848 73 823 40 q 874 149 874 106 q 878 206 878 169 q 878 260 878 242 m 332 688 q 305 811 332 774 q 235 848 278 848 q 156 787 175 848 q 137 648 137 727 q 160 541 137 590 q 234 493 183 493 q 300 526 274 493 q 326 602 326 560 q 332 688 332 641 "
    },
    "P": {
      "x_min": 110.5625,
      "x_max": 708,
      "ha": 722,
      "o": "m 116 559 q 110 930 116 759 l 230 930 q 333 930 268 930 q 409 930 397 930 q 620 870 533 930 q 708 690 708 810 q 604 490 708 561 q 365 420 501 420 l 252 420 q 252 200 252 347 q 252 0 252 52 q 220 4 245 0 q 182 8 196 8 q 145 5 166 8 q 116 0 125 2 q 116 287 116 86 q 116 559 116 488 m 565 682 q 505 823 565 771 q 358 876 446 876 l 257 876 q 250 782 250 828 q 250 689 250 736 l 250 476 q 498 529 432 476 q 565 682 565 583 "
    },
    "Î": {
      "x_min": -1,
      "x_max": 1327.359375,
      "ha": 1338,
      "o": "m 671 5 q 543 5 632 5 q 450 5 454 5 l 244 5 l 244 47 q 242 74 244 60 q 239 96 241 89 q 348 88 299 88 q 476 88 397 88 q 333 262 381 171 q 286 473 286 353 q 428 822 286 694 q 788 950 570 950 q 1144 823 1007 950 q 1282 473 1282 697 q 1236 262 1282 350 q 1093 91 1191 174 q 1204 91 1116 91 q 1326 91 1293 91 l 1326 34 l 1327 5 l 1120 5 q 994 5 1082 5 q 900 5 906 5 l 900 75 q 1078 218 1022 125 q 1134 453 1134 311 q 1048 764 1134 640 q 785 889 963 889 q 522 767 608 889 q 437 456 437 646 q 494 220 437 321 q 671 75 552 118 l 671 5 m 178 943 q 211 974 196 963 q 246 984 226 984 q 283 969 268 984 q 298 932 298 954 q 283 896 298 911 q 242 866 269 882 l 46 743 l -1 743 l 178 943 "
    },
    "Îˆ": {
      "x_min": -1.4375,
      "x_max": 900.375,
      "ha": 973,
      "o": "m 414 465 q 414 711 414 620 q 405 928 414 801 l 646 928 l 899 928 l 894 883 l 899 835 q 798 852 851 846 q 699 859 744 859 l 554 859 l 554 670 l 554 527 l 691 527 q 874 540 771 527 l 874 496 l 874 456 l 686 456 l 556 456 l 556 313 l 556 76 q 746 76 671 76 q 900 94 821 76 l 900 46 l 900 1 l 651 1 l 414 1 q 414 240 414 71 q 414 465 414 408 m 178 942 q 210 973 195 963 q 246 984 225 984 q 283 968 268 984 q 298 931 298 953 q 283 895 298 910 q 242 865 269 881 l 46 742 l -1 742 l 178 942 "
    },
    "_": {
      "x_min": 0,
      "x_max": 670,
      "ha": 670,
      "o": "m 670 -322 l 0 -322 l 0 -255 l 670 -255 l 670 -322 "
    },
    "Îª": {
      "x_min": -3,
      "x_max": 381,
      "ha": 378,
      "o": "m 116 464 q 116 710 116 539 q 116 932 116 880 q 153 929 121 932 q 189 926 186 926 q 234 929 201 926 q 260 932 266 932 q 260 696 260 863 q 260 464 260 530 q 260 230 260 396 q 260 0 260 63 q 228 4 255 0 q 189 8 201 8 q 151 5 172 8 q 116 0 129 2 q 116 243 116 72 q 116 464 116 414 m 72 1208 q 121 1186 100 1208 q 143 1133 143 1164 q 121 1081 143 1105 q 72 1058 99 1058 q 19 1080 42 1058 q -3 1133 -3 1103 q 18 1185 -3 1163 q 72 1208 40 1208 m 307 1208 q 359 1186 338 1208 q 381 1133 381 1164 q 358 1080 381 1103 q 307 1058 336 1058 q 256 1080 278 1058 q 234 1133 234 1103 q 255 1185 234 1163 q 307 1208 277 1208 "
    },
    "+": {
      "x_min": 166,
      "x_max": 950,
      "ha": 1116,
      "o": "m 590 440 l 950 440 l 950 373 l 590 373 l 590 0 l 525 0 l 525 373 l 166 373 l 166 440 l 525 440 l 525 815 l 590 815 l 590 440 "
    },
    "Â½": {
      "x_min": 81.75,
      "x_max": 1072.453125,
      "ha": 1149,
      "o": "m 241 742 l 241 835 q 177 804 201 817 q 118 767 153 791 q 102 787 115 774 q 81 807 89 800 q 196 850 138 824 q 321 913 254 877 l 324 910 q 324 723 324 849 q 324 550 324 598 l 324 391 q 296 394 314 391 q 276 397 279 397 q 250 394 266 397 q 228 391 234 391 q 241 550 241 472 q 241 742 241 628 m 842 1014 l 911 1014 l 307 -125 l 239 -125 l 842 1014 m 963 361 q 940 442 963 411 q 870 477 917 472 q 805 457 836 477 q 769 406 774 438 l 763 377 l 758 375 q 714 448 739 416 q 882 512 785 512 q 999 475 948 512 q 1051 373 1051 438 q 936 188 1051 284 q 791 66 822 92 l 959 66 q 1072 75 1019 66 q 1069 59 1071 72 q 1067 38 1067 46 q 1072 1 1067 14 q 962 1 1036 1 q 851 1 889 1 q 776 1 826 1 q 701 1 725 1 l 701 21 q 895 195 827 122 q 963 361 963 268 "
    },
    "Î¡": {
      "x_min": 110.5625,
      "x_max": 708,
      "ha": 722,
      "o": "m 116 559 q 110 930 116 759 l 230 930 q 333 930 268 930 q 409 930 397 930 q 620 870 533 930 q 708 690 708 810 q 604 490 708 561 q 365 420 501 420 l 252 420 q 252 200 252 347 q 252 0 252 52 q 220 4 245 0 q 182 8 196 8 q 145 5 166 8 q 116 0 125 2 q 116 287 116 86 q 116 559 116 488 m 565 682 q 505 823 565 771 q 358 876 446 876 l 257 876 q 250 782 250 828 q 250 689 250 736 l 250 476 q 498 529 432 476 q 565 682 565 583 "
    },
    "'": {
      "x_min": 87.296875,
      "x_max": 301,
      "ha": 366,
      "o": "m 166 857 q 192 923 174 896 q 245 951 211 951 q 283 937 266 951 q 301 903 301 923 q 290 857 301 882 q 264 812 279 832 l 119 567 l 87 576 l 166 857 "
    },
    "T": {
      "x_min": 10.78125,
      "x_max": 698.4375,
      "ha": 711,
      "o": "m 10 838 l 14 884 l 10 926 q 190 926 70 926 q 353 926 310 926 q 533 926 412 926 q 698 926 654 926 q 693 883 693 910 q 693 861 693 870 q 698 834 693 852 q 597 850 653 843 q 493 857 540 857 l 418 857 q 418 767 418 830 q 418 701 418 704 l 418 221 q 432 0 418 122 q 355 8 393 4 q 316 5 337 8 q 283 0 295 3 q 283 132 283 40 q 283 259 283 225 l 283 683 l 283 857 q 10 838 156 857 "
    },
    "Î¦": {
      "x_min": 49,
      "x_max": 1047,
      "ha": 1097,
      "o": "m 622 -24 q 548 -15 579 -15 q 516 -17 533 -15 q 475 -24 499 -19 l 475 68 q 178 179 307 68 q 49 465 49 291 q 169 744 49 647 q 475 865 290 842 l 475 958 q 512 952 492 955 q 548 950 532 950 q 584 952 564 950 q 622 958 604 955 l 622 865 q 923 756 800 865 q 1047 460 1047 648 q 925 177 1047 279 q 622 75 804 75 l 622 -24 m 487 502 q 487 677 487 570 q 487 800 487 784 q 273 698 348 786 q 199 466 199 611 q 277 231 199 331 q 487 132 356 132 q 487 363 487 225 q 487 502 487 501 m 897 466 q 822 706 897 612 q 610 800 747 800 q 610 636 610 738 q 610 470 610 533 q 610 301 610 408 q 610 132 610 194 q 822 226 747 132 q 897 466 897 320 "
    },
    "j": {
      "x_min": -54,
      "x_max": 244,
      "ha": 335,
      "o": "m 104 391 q 104 542 104 444 q 104 653 104 641 q 142 647 133 647 q 165 647 151 647 q 199 648 186 647 q 233 653 211 649 q 233 507 233 595 q 233 405 233 419 l 233 -13 l 233 -109 q 153 -303 233 -234 q -54 -372 74 -372 l -54 -333 q 73 -267 42 -323 q 104 -104 104 -212 l 104 -26 l 104 391 m 168 962 q 222 939 201 962 q 244 880 244 916 q 222 830 244 848 q 168 812 200 812 q 114 832 136 812 q 92 884 92 852 q 113 940 92 918 q 168 962 134 962 "
    },
    "Î£": {
      "x_min": 44,
      "x_max": 774.796875,
      "ha": 809,
      "o": "m 714 882 l 718 835 q 266 855 509 855 q 533 500 384 691 q 364 313 438 399 q 215 123 291 226 l 446 123 q 626 123 512 123 q 774 123 740 123 l 768 85 l 768 67 l 768 42 q 769 25 768 33 q 774 4 770 17 q 547 4 680 4 q 408 4 414 4 q 226 4 343 4 q 44 4 109 4 l 44 49 q 166 182 103 109 q 302 343 228 254 l 399 467 q 238 689 309 593 q 75 898 167 784 l 75 929 q 227 929 121 929 q 339 929 333 929 q 556 929 404 929 q 718 929 709 929 l 714 882 "
    },
    "1": {
      "x_min": 71.203125,
      "x_max": 455,
      "ha": 734,
      "o": "m 323 625 q 323 721 323 655 q 323 792 323 788 q 224 736 274 764 q 131 673 174 709 q 101 712 122 688 q 71 742 79 736 q 269 827 174 779 q 449 934 365 874 l 455 928 q 455 551 455 803 q 455 259 455 299 l 455 0 q 413 5 433 2 q 377 8 393 8 q 342 5 361 8 q 306 0 324 2 q 323 289 323 132 q 323 625 323 445 "
    },
    "Ã¤": {
      "x_min": 42,
      "x_max": 642,
      "ha": 635,
      "o": "m 229 -16 q 96 32 150 -16 q 42 161 42 81 q 104 302 42 272 q 297 363 166 332 q 435 447 428 395 q 395 568 435 521 q 283 616 355 616 q 187 587 228 616 q 122 507 147 559 l 93 520 l 102 590 q 198 651 141 631 q 317 672 256 672 q 490 622 435 672 q 546 454 546 572 l 546 132 q 556 68 546 83 q 605 54 566 54 q 642 57 629 54 l 642 26 q 582 5 613 14 q 526 -6 551 -3 q 430 83 444 -6 q 336 9 381 34 q 229 -16 291 -16 m 200 928 q 250 905 228 928 q 272 853 272 883 q 252 798 272 819 q 201 777 232 777 q 147 799 170 777 q 124 853 124 821 q 145 905 124 883 q 200 928 167 928 m 435 928 q 488 905 466 928 q 510 853 510 883 q 489 799 510 821 q 438 777 469 777 q 384 799 406 777 q 362 853 362 822 q 383 905 362 883 q 435 928 405 928 m 173 185 q 200 97 173 132 q 279 63 228 63 q 383 106 336 63 q 430 211 430 150 l 430 345 q 235 292 298 318 q 173 185 173 267 "
    },
    "<": {
      "x_min": 173,
      "x_max": 942,
      "ha": 1116,
      "o": "m 275 406 l 942 130 l 942 56 l 173 379 l 173 433 l 942 756 l 942 683 l 275 406 "
    },
    "Â£": {
      "x_min": 65,
      "x_max": 713.796875,
      "ha": 733,
      "o": "m 65 46 l 65 99 l 113 99 q 199 167 172 110 q 227 292 227 224 q 222 375 227 330 q 217 442 217 421 q 137 442 168 442 q 75 433 105 442 l 75 479 l 74 502 q 103 495 90 498 q 131 493 115 493 l 214 493 q 207 550 210 522 q 205 609 205 579 q 298 825 205 742 q 526 909 391 909 q 619 896 580 909 q 708 864 659 884 q 652 732 679 809 l 643 732 q 593 825 628 791 q 502 859 559 859 q 378 796 420 859 q 336 646 336 734 l 336 493 l 358 493 q 515 502 443 493 l 511 466 l 515 433 q 430 442 483 442 q 341 442 377 442 l 341 379 q 306 238 341 302 q 209 112 271 173 q 516 112 413 112 q 713 134 619 112 l 707 66 q 713 4 707 34 q 531 4 652 4 q 350 4 410 4 q 173 4 280 4 q 65 4 66 4 l 65 46 "
    },
    "Â¹": {
      "x_min": 79.953125,
      "x_max": 334,
      "ha": 485,
      "o": "m 246 731 l 246 833 l 120 759 q 101 780 118 763 q 79 801 85 796 q 204 849 145 821 q 329 916 263 877 l 334 912 q 334 720 334 848 q 334 529 334 592 l 334 355 q 306 358 323 355 q 283 361 290 361 q 259 358 275 361 q 236 355 243 355 q 246 498 246 406 q 246 731 246 590 "
    },
    "t": {
      "x_min": 17.46875,
      "x_max": 407,
      "ha": 417,
      "o": "m 17 586 l 21 630 l 17 654 q 128 643 78 643 q 128 732 128 669 q 128 798 128 795 q 194 826 159 811 q 258 862 230 842 q 248 758 248 811 q 248 643 248 705 q 328 645 306 643 q 392 654 351 647 l 392 617 l 393 586 q 241 594 317 594 l 241 258 l 241 162 q 269 76 241 109 q 347 43 297 43 q 380 45 362 43 q 407 48 397 47 l 407 4 q 343 -10 371 -5 q 286 -15 314 -15 q 175 18 219 -15 q 125 118 130 51 l 125 200 l 125 387 l 125 592 q 79 592 106 592 q 17 586 52 592 "
    },
    "Î»": {
      "x_min": 2.296875,
      "x_max": 639.296875,
      "ha": 644,
      "o": "m 296 670 q 222 871 256 803 q 108 940 188 940 q 76 937 92 940 q 20 923 60 934 l 20 977 q 94 1012 58 999 q 167 1025 130 1025 q 322 947 281 1025 q 418 692 364 869 q 527 340 471 515 q 639 3 582 166 l 567 3 l 493 3 q 328 573 414 305 q 213 301 266 438 q 108 0 160 163 l 59 7 q 26 3 47 7 q 2 0 5 0 q 296 670 161 329 "
    },
    "Ã¹": {
      "x_min": 91,
      "x_max": 650.53125,
      "ha": 739,
      "o": "m 641 498 q 641 329 641 443 q 641 158 641 215 q 650 0 641 81 q 619 3 634 1 q 586 6 604 6 q 552 3 571 6 q 523 0 533 1 l 523 118 q 430 18 484 52 q 304 -15 376 -15 q 143 50 195 -15 q 91 229 91 115 l 91 354 l 91 516 l 91 655 q 117 650 101 651 q 155 650 133 650 q 188 650 171 650 q 215 655 205 650 q 215 446 215 592 q 215 247 215 300 q 247 115 215 163 q 356 68 279 68 q 463 113 418 68 q 515 217 509 159 q 521 340 521 275 q 521 521 521 394 q 521 655 521 647 q 550 650 537 651 q 585 649 563 649 q 650 655 616 649 q 641 498 641 573 m 435 743 l 239 866 q 201 896 219 877 q 184 934 184 915 q 198 970 184 955 q 234 985 213 985 q 272 973 251 985 q 303 945 292 960 l 485 743 l 435 743 "
    },
    "W": {
      "x_min": 0,
      "x_max": 1280.8125,
      "ha": 1281,
      "o": "m 0 932 q 46 927 31 929 q 74 926 61 926 q 119 929 87 926 q 152 932 151 932 q 257 547 196 749 l 372 170 q 460 437 406 272 q 541 693 514 602 q 607 932 568 784 l 658 926 q 686 928 671 926 q 711 932 702 930 q 808 604 771 727 q 865 419 845 481 q 950 170 886 356 l 1065 522 q 1120 719 1098 623 q 1163 932 1143 816 q 1197 929 1173 932 q 1222 926 1220 926 q 1254 928 1237 926 q 1280 932 1272 930 q 1109 467 1193 716 q 970 0 1024 218 q 944 4 960 1 q 918 7 928 7 q 886 3 906 7 q 862 0 865 0 q 745 385 815 184 l 628 733 l 481 287 q 394 0 430 133 q 363 3 383 0 q 337 7 343 7 q 308 4 321 7 q 281 0 296 2 q 202 296 247 143 q 119 568 157 450 q 0 932 81 686 "
    },
    "Ã¯": {
      "x_min": -24,
      "x_max": 359,
      "ha": 335,
      "o": "m 98 144 l 98 522 l 98 655 q 135 650 110 650 q 163 650 159 650 q 229 655 203 650 q 225 507 225 581 q 225 254 225 423 q 225 0 225 85 q 198 3 213 1 q 161 6 183 6 q 125 3 140 6 q 98 0 110 1 l 98 144 m 49 929 q 101 906 79 929 q 123 854 123 884 q 103 799 123 820 q 52 778 83 778 q -2 800 19 778 q -24 854 -24 822 q -3 906 -24 884 q 49 929 16 929 m 285 929 q 337 906 315 929 q 359 854 359 884 q 339 799 359 820 q 289 778 319 778 q 234 800 256 778 q 213 854 213 822 q 233 906 213 884 q 285 929 253 929 "
    },
    ">": {
      "x_min": 173,
      "x_max": 943,
      "ha": 1116,
      "o": "m 943 379 l 173 56 l 173 130 l 840 406 l 173 683 l 173 756 l 943 433 l 943 379 "
    },
    "v": {
      "x_min": 0,
      "x_max": 645.171875,
      "ha": 641,
      "o": "m 0 653 q 53 647 38 647 q 84 647 68 647 q 111 647 98 647 q 164 653 125 647 q 247 402 200 528 l 351 134 l 461 435 q 532 653 498 533 q 558 649 543 650 q 588 648 574 648 q 623 648 605 648 q 645 653 639 652 q 496 340 566 502 q 364 0 427 177 q 340 5 355 2 q 313 8 326 8 q 290 5 303 8 q 266 0 277 2 q 196 205 230 120 q 0 653 161 291 "
    },
    "Ï„": {
      "x_min": 30.375,
      "x_max": 664.4375,
      "ha": 688,
      "o": "m 416 569 l 416 303 q 416 146 416 251 q 416 0 416 42 q 387 3 403 2 q 353 3 372 3 q 327 3 338 3 q 289 0 317 3 l 289 194 l 289 573 q 154 553 204 573 q 57 471 103 534 q 30 560 49 523 q 147 629 82 612 q 302 647 212 647 l 501 647 q 593 647 535 647 q 664 647 652 647 l 659 610 q 664 569 659 587 q 518 569 607 569 q 416 569 429 569 "
    },
    "Ã»": {
      "x_min": 91,
      "x_max": 650.53125,
      "ha": 739,
      "o": "m 641 498 q 641 329 641 443 q 641 158 641 215 q 650 0 641 81 q 619 3 634 1 q 586 6 604 6 q 552 3 571 6 q 523 0 533 1 l 523 118 q 430 18 484 52 q 304 -15 376 -15 q 143 50 195 -15 q 91 229 91 115 l 91 354 l 91 516 l 91 655 q 117 650 101 651 q 155 650 133 650 q 188 650 171 650 q 215 655 205 650 q 215 445 215 591 q 215 247 215 299 q 247 115 215 163 q 356 68 279 68 q 463 113 418 68 q 515 217 509 159 q 521 340 521 274 q 521 520 521 394 q 521 655 521 647 q 547 650 531 651 q 585 650 563 650 q 617 650 600 650 q 650 655 634 650 q 641 498 641 573 m 325 977 l 412 977 l 560 743 l 514 743 l 368 874 l 224 743 l 176 743 l 325 977 "
    },
    "Î¾": {
      "x_min": 63,
      "x_max": 641,
      "ha": 642,
      "o": "m 551 860 q 492 941 528 910 q 406 972 456 972 q 293 917 335 972 q 251 788 251 862 q 306 649 251 700 q 449 598 362 598 l 515 598 l 515 563 l 515 527 q 473 533 495 533 q 421 533 451 533 q 267 492 332 533 q 193 406 202 452 q 184 346 185 360 q 183 325 183 332 q 183 293 183 299 q 187 281 184 287 q 234 199 197 228 q 329 152 271 170 l 485 117 q 598 72 556 99 q 641 -14 641 45 q 615 -104 641 -61 q 501 -238 590 -147 l 457 -211 q 531 -122 520 -140 q 542 -76 542 -104 q 386 15 542 -19 q 146 91 230 49 q 63 276 63 133 q 128 461 63 378 q 293 574 193 544 q 172 651 218 599 q 126 780 126 703 q 209 954 126 883 q 397 1025 293 1025 q 497 1009 449 1025 q 597 965 544 994 q 573 914 588 945 q 551 860 559 883 "
    },
    "&": {
      "x_min": 75,
      "x_max": 899.5,
      "ha": 955,
      "o": "m 353 -18 q 157 41 240 -18 q 75 210 75 101 q 135 381 75 314 q 308 514 196 448 q 244 618 268 568 q 221 721 221 667 q 282 872 221 812 q 432 932 343 932 q 570 891 510 932 q 630 777 630 851 q 576 639 630 701 q 443 531 522 578 q 557 389 499 459 q 677 252 615 320 q 798 530 777 378 l 812 530 l 869 466 q 798 327 839 395 q 713 209 757 259 q 791 115 745 169 q 899 1 836 61 q 838 1 879 1 q 775 1 797 1 l 725 1 l 636 109 q 506 15 575 48 q 353 -18 436 -18 m 335 474 q 230 379 263 419 q 198 273 198 339 q 254 125 198 190 q 393 61 310 61 q 498 85 449 61 q 594 153 548 110 l 335 474 m 536 770 q 514 854 536 823 q 445 886 492 886 q 356 851 394 886 q 319 769 319 817 q 337 685 319 717 q 413 574 356 652 q 504 661 472 614 q 536 770 536 708 "
    },
    "Î›": {
      "x_min": 0,
      "x_max": 835.71875,
      "ha": 836,
      "o": "m 649 452 l 762 171 l 835 2 l 754 2 q 720 2 732 2 q 665 0 707 2 q 634 101 647 54 q 593 225 620 147 l 536 382 l 393 777 l 261 428 q 187 213 221 318 q 122 1 153 107 l 61 1 l 0 1 q 201 458 93 190 q 372 931 308 725 q 396 927 385 928 q 423 926 408 926 q 450 928 432 926 q 477 931 468 931 q 537 743 500 849 q 649 452 574 637 "
    },
    "I": {
      "x_min": 107.84375,
      "x_max": 260.875,
      "ha": 377,
      "o": "m 116 465 q 116 711 116 620 q 107 932 116 802 q 153 926 128 926 q 189 926 178 926 q 236 929 208 926 q 260 932 265 932 q 260 788 260 887 q 260 659 260 689 l 260 448 l 260 282 q 260 135 260 238 q 260 0 260 31 q 230 4 257 0 q 189 8 202 8 q 149 5 171 8 q 116 0 128 2 q 116 239 116 70 q 116 465 116 408 "
    },
    "G": {
      "x_min": 50,
      "x_max": 920,
      "ha": 981,
      "o": "m 568 -15 q 193 107 337 -15 q 50 459 50 229 q 192 815 50 680 q 555 950 335 950 q 740 929 646 950 q 911 869 835 909 q 889 803 898 836 q 877 738 880 769 l 857 738 q 725 855 793 818 q 560 893 657 893 q 299 770 398 893 q 200 479 200 647 q 293 168 200 291 q 568 46 386 46 q 679 56 630 46 q 778 94 727 66 q 778 184 778 123 q 778 251 778 246 q 778 337 778 280 q 778 423 778 394 q 813 417 805 418 q 846 417 821 417 q 883 419 862 417 q 920 425 904 422 l 920 236 q 920 121 920 201 q 920 37 920 41 q 740 -1 823 11 q 568 -15 656 -15 "
    },
    "Î°": {
      "x_min": 80,
      "x_max": 691,
      "ha": 758,
      "o": "m 691 395 q 586 110 691 236 q 327 -15 481 -15 q 145 54 210 -15 q 80 244 80 123 l 80 430 q 80 542 80 486 q 80 651 80 598 l 145 651 l 207 651 q 207 502 207 608 q 207 365 207 397 l 207 261 q 244 105 207 168 q 367 43 282 43 q 516 132 470 43 q 562 340 562 222 q 540 493 562 418 q 480 646 518 568 q 541 647 521 646 q 617 657 562 649 q 691 395 691 537 m 206 865 q 247 846 229 865 q 265 804 265 828 q 248 761 265 780 q 206 743 231 743 q 164 761 182 743 q 147 804 147 779 q 164 846 147 828 q 206 865 182 865 m 356 928 q 372 969 362 956 q 408 981 382 981 q 452 941 452 981 q 439 903 452 919 l 333 743 l 301 743 l 356 928 m 504 865 q 545 846 527 865 q 563 804 563 828 q 545 761 563 779 q 504 743 527 743 q 462 761 478 743 q 446 804 446 779 q 462 846 446 828 q 504 865 478 865 "
    },
    "`": {
      "x_min": 86,
      "x_max": 298.328125,
      "ha": 366,
      "o": "m 219 659 q 192 595 211 622 q 139 568 173 568 q 86 613 86 568 q 96 660 86 636 q 121 706 106 684 l 267 949 l 298 940 l 219 659 "
    },
    "Î¥": {
      "x_min": -27.40625,
      "x_max": 730.890625,
      "ha": 693,
      "o": "m 287 177 l 287 386 q 185 570 248 458 q 81 750 121 681 q -27 932 42 819 q 24 928 -9 932 q 62 925 58 925 q 110 927 89 925 q 144 932 131 930 q 203 800 171 866 q 270 676 235 734 l 382 475 q 499 688 442 575 q 615 932 556 801 l 669 926 q 700 927 687 926 q 730 932 714 929 q 545 627 627 769 l 426 415 l 426 240 q 426 102 426 198 q 426 0 426 5 q 393 4 417 1 q 353 7 368 7 q 312 4 327 7 q 278 0 297 2 q 287 88 287 36 q 287 177 287 140 "
    },
    "r": {
      "x_min": 89.90625,
      "x_max": 459.1875,
      "ha": 478,
      "o": "m 94 120 l 94 399 l 94 432 q 89 653 94 548 q 124 648 113 649 q 160 647 134 647 q 230 653 193 647 q 222 588 222 626 q 222 516 222 551 q 310 627 263 589 q 431 666 358 666 l 459 666 l 453 603 l 459 537 q 422 544 442 541 q 378 551 402 548 q 255 482 290 551 q 221 312 221 413 q 221 131 221 256 q 221 0 221 6 l 156 6 l 94 0 l 94 120 "
    },
    "x": {
      "x_min": 1.359375,
      "x_max": 622.03125,
      "ha": 619,
      "o": "m 258 316 l 155 460 q 76 563 118 508 q 5 653 35 617 q 95 647 50 647 q 138 649 118 647 q 174 653 159 651 q 244 538 209 592 q 328 415 279 484 q 412 533 370 473 q 491 653 454 592 q 513 650 498 652 q 539 647 528 648 q 604 653 570 647 l 363 365 q 467 210 426 268 q 622 0 509 152 q 575 3 604 0 q 540 7 547 7 q 491 4 513 7 q 455 0 469 1 q 372 140 398 98 q 289 263 345 183 q 169 84 190 118 q 121 0 148 51 l 65 6 q 32 3 54 6 q 1 0 10 0 q 129 154 62 72 q 258 316 196 236 "
    },
    "Ã¨": {
      "x_min": 39,
      "x_max": 632,
      "ha": 666,
      "o": "m 398 42 q 590 129 512 42 l 609 126 q 601 90 605 109 q 597 46 597 71 q 486 0 546 14 q 361 -15 426 -15 q 127 73 215 -15 q 39 311 39 162 q 123 562 39 456 q 347 669 207 669 q 551 590 470 669 q 632 387 632 512 l 632 332 q 429 332 550 332 q 307 332 308 332 l 175 332 q 230 127 175 212 q 398 42 285 42 m 398 743 l 203 866 q 163 895 179 880 q 147 934 147 910 q 164 967 147 949 q 198 985 182 985 q 238 969 216 985 q 267 945 260 952 l 447 743 l 398 743 m 503 390 l 503 436 q 460 563 503 508 q 349 618 418 618 q 228 552 266 618 q 179 390 191 487 l 503 390 "
    },
    "Î¼": {
      "x_min": 88,
      "x_max": 657.453125,
      "ha": 739,
      "o": "m 326 -15 q 261 -7 288 -15 q 208 19 235 0 l 208 -127 q 208 -265 208 -177 q 208 -373 208 -354 q 180 -369 195 -370 q 148 -368 165 -368 q 112 -370 131 -368 q 88 -373 93 -372 q 88 -165 88 -304 q 88 43 88 -26 q 88 363 88 148 q 88 653 88 577 q 149 647 118 647 q 210 653 178 647 q 210 495 210 601 q 210 334 210 390 q 233 155 210 227 q 334 75 256 83 q 462 112 412 75 q 520 211 513 149 q 528 331 528 273 q 528 517 528 387 q 528 653 528 647 q 590 647 559 647 q 652 653 619 647 q 652 477 652 566 q 652 256 652 388 q 652 113 652 170 q 657 -1 652 55 l 592 4 l 528 -1 l 528 105 q 442 17 493 50 q 326 -15 390 -15 "
    },
    "Ã·": {
      "x_min": 166,
      "x_max": 950,
      "ha": 1116,
      "o": "m 629 643 q 607 593 629 615 q 555 571 585 571 q 508 592 528 571 q 489 643 489 613 q 509 692 489 670 q 556 715 529 715 q 599 703 569 715 q 629 643 629 691 m 950 373 l 166 373 l 166 440 l 950 440 l 950 373 m 627 170 q 606 120 627 141 q 559 100 586 100 q 509 120 529 100 q 489 170 489 141 q 508 221 489 199 q 556 243 528 243 q 591 235 572 243 q 618 214 610 227 q 627 170 627 201 "
    },
    "h": {
      "x_min": 92.921875,
      "x_max": 651,
      "ha": 744,
      "o": "m 97 136 l 97 859 q 97 934 97 894 q 92 1024 97 974 q 135 1018 125 1018 q 156 1018 144 1018 q 225 1024 186 1018 q 221 957 221 1001 q 221 888 221 913 l 221 868 l 221 542 q 319 637 261 602 q 445 672 376 672 q 601 606 552 672 q 651 429 651 541 l 651 298 l 651 136 l 651 0 q 622 3 637 1 q 587 6 607 6 q 551 3 570 6 q 526 0 531 1 q 526 202 526 68 q 526 405 526 337 q 491 533 526 481 q 386 586 457 586 q 254 508 288 586 q 221 313 221 430 q 221 133 221 244 q 221 0 221 22 q 186 3 210 0 q 161 7 163 7 q 123 5 137 7 q 97 0 109 3 l 97 136 "
    },
    ".": {
      "x_min": 98,
      "x_max": 268,
      "ha": 366,
      "o": "m 183 156 q 243 130 219 156 q 268 68 268 105 q 243 8 268 32 q 183 -15 219 -15 q 122 8 146 -15 q 98 68 98 32 q 122 130 98 105 q 183 156 146 156 "
    },
    "Ï†": {
      "x_min": 38,
      "x_max": 946,
      "ha": 985,
      "o": "m 559 -371 q 496 -362 529 -362 q 456 -364 475 -362 q 424 -371 438 -367 q 424 -163 424 -298 q 424 -11 424 -29 q 153 76 269 -11 q 38 327 38 163 q 142 571 38 486 q 402 656 247 656 q 404 635 404 645 q 404 615 404 625 q 230 516 286 580 q 175 327 175 451 q 242 118 175 197 q 433 38 309 38 l 433 197 l 433 311 q 433 400 433 340 q 433 491 433 461 q 502 619 441 570 q 643 668 564 668 q 861 568 777 668 q 946 331 946 469 q 833 84 946 169 q 559 -15 721 0 q 559 -203 559 -72 q 559 -371 559 -334 m 809 345 q 778 533 809 450 q 662 617 748 617 q 578 571 607 617 q 550 469 550 526 l 550 348 l 550 36 q 744 123 679 36 q 809 345 809 211 "
    },
    ";": {
      "x_min": 72.390625,
      "x_max": 307,
      "ha": 438,
      "o": "m 221 636 q 282 611 257 636 q 307 548 307 586 q 282 486 307 511 q 221 461 257 461 q 160 486 185 461 q 135 548 135 512 q 159 611 135 586 q 221 636 184 636 m 162 75 q 195 157 180 140 q 241 175 211 175 q 300 119 300 175 q 291 75 300 93 q 259 18 283 56 l 102 -242 l 72 -231 l 162 75 "
    },
    "f": {
      "x_min": 12.640625,
      "x_max": 425.203125,
      "ha": 390,
      "o": "m 120 324 l 120 596 q 64 596 88 596 q 14 589 39 596 l 14 626 l 12 654 q 79 649 38 649 q 125 649 119 649 q 189 901 125 777 q 371 1025 253 1025 q 402 1022 393 1025 q 425 1015 411 1019 l 408 895 q 365 911 388 905 q 320 918 341 918 q 248 886 273 918 q 223 805 223 855 q 231 713 223 761 q 242 650 239 665 q 325 650 272 650 q 389 650 377 650 q 385 642 387 647 q 383 633 383 637 l 381 622 l 383 611 q 389 589 383 609 q 319 596 351 596 q 247 596 287 596 l 247 366 q 247 183 247 305 q 247 0 247 61 q 210 3 235 0 q 180 7 184 7 q 142 4 159 7 q 120 0 125 1 q 120 160 120 52 q 120 324 120 268 "
    },
    "â€œ": {
      "x_min": 83,
      "x_max": 539.96875,
      "ha": 612,
      "o": "m 462 659 q 433 592 450 617 q 382 568 416 568 q 343 579 358 568 q 329 613 329 591 q 340 660 329 633 q 364 706 352 687 l 510 949 l 539 940 l 462 659 m 217 659 q 189 592 208 617 q 136 568 171 568 q 83 613 83 568 q 92 658 83 637 q 118 706 102 680 l 263 949 l 295 940 l 217 659 "
    },
    "A": {
      "x_min": -14.96875,
      "x_max": 822.109375,
      "ha": 809,
      "o": "m 253 638 l 379 949 q 394 945 387 946 q 409 944 401 944 q 443 949 428 944 q 565 629 525 733 q 673 359 605 526 q 822 0 740 192 q 773 3 804 0 q 736 7 743 7 q 686 4 709 7 q 650 0 664 1 q 588 199 609 137 q 532 355 567 261 l 370 355 l 210 355 l 159 205 q 127 110 142 161 q 99 0 112 59 l 43 6 q 6 3 20 6 q -14 0 -8 0 q 74 211 29 105 q 155 403 119 317 q 253 638 191 490 m 370 422 l 502 422 l 371 760 l 240 422 l 370 422 "
    },
    "6": {
      "x_min": 63,
      "x_max": 678.109375,
      "ha": 734,
      "o": "m 456 860 q 263 731 318 860 q 207 441 207 602 q 310 514 259 488 q 423 540 361 540 q 606 462 535 540 q 678 270 678 385 q 592 65 678 145 q 382 -15 506 -15 q 139 93 216 -15 q 63 378 63 201 q 163 746 63 582 q 451 909 264 909 q 511 906 489 909 q 565 891 533 903 l 565 828 q 511 852 536 844 q 456 860 486 860 m 542 258 q 499 409 542 347 q 372 471 456 471 q 250 409 294 471 q 206 264 206 348 q 249 105 206 172 q 378 39 292 39 q 477 73 434 39 q 531 148 520 108 q 542 206 542 187 q 542 258 542 225 "
    },
    "â€˜": {
      "x_min": 86,
      "x_max": 298.328125,
      "ha": 366,
      "o": "m 220 659 q 190 594 209 620 q 139 568 171 568 q 86 615 86 568 q 97 660 86 633 q 121 708 109 687 l 267 951 l 298 942 l 220 659 "
    },
    "ÏŠ": {
      "x_min": -29,
      "x_max": 355,
      "ha": 335,
      "o": "m 98 333 l 98 520 q 98 566 98 544 q 93 653 98 588 q 137 647 127 647 q 161 647 147 647 q 228 653 191 647 q 223 555 223 599 q 223 437 223 511 l 223 406 q 223 194 223 337 q 223 0 223 51 q 196 3 211 1 q 161 6 181 6 q 124 3 145 6 q 98 0 104 1 q 98 165 98 51 q 98 333 98 279 m 44 928 q 97 907 75 928 q 119 852 119 886 q 98 800 119 822 q 47 778 78 778 q -7 800 14 778 q -29 852 -29 822 q -7 906 -29 884 q 44 928 13 928 m 281 928 q 333 906 312 928 q 355 852 355 884 q 335 799 355 821 q 284 778 315 778 q 230 800 252 778 q 209 852 209 822 q 229 905 209 883 q 281 928 249 928 "
    },
    "Ï€": {
      "x_min": 19.5,
      "x_max": 938.75,
      "ha": 970,
      "o": "m 689 5 l 627 0 q 627 114 627 44 q 627 195 627 184 l 627 573 l 497 573 l 367 573 l 367 282 q 367 142 367 235 q 367 0 367 49 q 339 3 354 1 q 304 5 324 5 q 267 3 287 5 q 240 0 246 1 l 240 195 l 240 573 q 129 552 169 573 q 44 471 88 531 q 19 560 35 520 q 137 630 71 613 q 300 647 204 647 l 646 647 l 836 647 l 938 647 l 933 610 l 938 568 l 755 568 l 755 308 q 755 153 755 256 q 755 0 755 50 q 726 2 741 0 q 689 5 711 5 "
    },
    "Î¬": {
      "x_min": 41,
      "x_max": 811.390625,
      "ha": 829,
      "o": "m 691 352 q 788 0 748 155 l 725 0 l 660 0 l 620 172 q 511 36 582 88 q 349 -15 439 -15 q 127 81 213 -15 q 41 316 41 177 q 128 569 41 467 q 362 672 216 672 q 527 622 456 672 q 646 485 599 573 q 698 651 678 569 l 755 651 l 811 651 q 748 505 777 576 q 691 352 720 434 m 509 942 q 542 973 526 963 q 578 984 558 984 q 614 968 600 984 q 629 931 629 952 q 615 896 629 910 q 574 866 601 882 l 376 743 l 329 743 l 509 942 m 370 619 q 222 530 267 619 q 178 326 178 442 q 219 124 178 214 q 360 34 261 34 q 518 137 470 34 q 589 323 566 240 q 517 531 567 444 q 370 619 466 619 "
    },
    "O": {
      "x_min": 50,
      "x_max": 1046,
      "ha": 1096,
      "o": "m 50 465 q 188 820 50 690 q 548 950 326 950 q 874 853 738 950 q 1025 654 1010 757 q 1043 525 1040 551 q 1046 462 1046 500 q 1043 402 1046 426 q 1025 277 1040 379 q 875 80 1010 175 q 549 -15 741 -15 q 438 -10 486 -15 q 298 29 390 -5 q 128 186 206 64 q 50 465 50 308 m 198 468 q 285 162 198 282 q 547 42 372 42 q 809 162 723 42 q 896 468 896 283 q 808 770 896 651 q 547 889 720 889 q 341 826 421 889 q 221 639 261 764 q 198 468 198 552 "
    },
    "n": {
      "x_min": 89.90625,
      "x_max": 647,
      "ha": 740,
      "o": "m 94 155 l 94 495 q 94 568 94 530 q 89 655 94 607 q 152 648 121 648 l 215 655 l 215 538 q 314 635 256 599 q 445 672 372 672 q 584 623 522 672 q 647 500 647 575 l 647 298 l 647 136 l 647 0 q 619 3 634 1 q 584 6 604 6 q 548 3 568 6 q 522 0 528 1 q 522 222 522 80 q 522 406 522 364 q 489 536 522 485 q 382 587 456 587 q 251 508 285 587 q 218 313 218 430 q 218 132 218 258 q 218 0 218 7 q 183 3 207 0 q 155 7 159 7 q 117 3 141 7 q 94 0 94 0 l 94 155 "
    },
    "3": {
      "x_min": 73.03125,
      "x_max": 631.109375,
      "ha": 733,
      "o": "m 235 465 l 235 511 l 288 511 q 415 555 367 511 q 464 681 464 600 q 425 806 464 755 q 315 857 386 857 q 215 818 251 857 q 160 711 179 780 l 149 706 q 124 768 132 747 q 96 819 115 788 q 215 887 158 864 q 341 910 272 910 q 516 857 441 910 q 591 707 591 805 q 531 564 591 618 q 375 479 472 511 q 555 422 480 479 q 631 262 631 366 q 531 55 631 129 q 296 -18 432 -18 q 179 -6 235 -18 q 73 32 124 5 q 96 189 89 116 l 109 189 q 178 73 132 115 q 296 31 224 31 q 439 92 384 31 q 495 246 495 154 q 442 389 495 333 q 305 446 390 446 q 233 434 273 446 l 235 465 "
    },
    "9": {
      "x_min": 56,
      "x_max": 674,
      "ha": 734,
      "o": "m 255 38 q 464 166 396 38 q 533 450 533 295 q 432 378 483 403 q 318 354 381 354 q 130 428 204 354 q 56 617 56 502 q 145 828 56 748 q 365 909 235 909 q 599 801 525 909 q 674 520 674 693 q 563 143 674 305 q 247 -18 452 -18 q 182 -14 206 -18 q 125 0 157 -11 l 111 90 q 176 51 140 64 q 255 38 213 38 m 364 419 q 491 482 447 419 q 535 634 535 545 q 494 790 535 725 q 366 855 453 855 q 233 791 278 855 q 189 637 189 727 q 233 482 189 545 q 364 419 278 419 "
    },
    "l": {
      "x_min": 102,
      "x_max": 234.4375,
      "ha": 335,
      "o": "m 102 118 l 102 881 q 102 965 102 915 q 102 1024 102 1016 q 167 1018 136 1018 q 234 1024 200 1018 q 229 872 229 948 q 229 684 229 797 l 229 512 l 229 111 l 229 0 q 203 3 218 1 q 167 6 188 6 q 130 3 151 6 q 102 0 110 1 l 102 118 "
    },
    "Îº": {
      "x_min": 98,
      "x_max": 664,
      "ha": 669,
      "o": "m 98 365 q 98 530 98 427 q 98 653 98 633 q 161 647 131 647 q 223 653 193 647 q 223 516 223 608 q 223 378 223 425 l 252 378 q 378 498 316 432 q 518 653 440 565 q 576 647 552 647 q 627 647 601 647 q 658 652 649 651 l 350 387 l 537 165 q 595 93 566 127 q 664 19 625 59 l 664 0 q 615 3 646 0 q 579 7 585 7 q 533 3 556 7 q 500 0 510 0 q 429 100 468 47 q 353 197 390 154 l 263 308 l 247 323 l 223 326 q 223 164 223 275 q 223 0 223 52 l 161 3 l 98 3 q 98 209 98 80 q 98 365 98 339 "
    },
    "4": {
      "x_min": 38,
      "x_max": 678.15625,
      "ha": 734,
      "o": "m 441 254 l 174 254 l 38 254 l 38 290 q 200 522 108 392 q 338 721 291 653 q 465 904 386 789 l 517 904 l 570 904 q 565 805 570 872 q 560 706 560 739 l 560 345 l 588 345 q 633 345 610 345 q 678 350 656 345 l 670 297 q 672 270 670 287 q 678 244 675 254 q 558 253 616 253 l 558 137 l 558 0 l 496 7 l 428 0 q 441 121 441 51 q 441 254 441 191 m 444 767 l 337 626 q 225 465 292 562 q 141 343 159 368 l 444 343 l 444 767 "
    },
    "p": {
      "x_min": 86,
      "x_max": 689,
      "ha": 743,
      "o": "m 86 -106 q 86 202 86 47 q 86 514 86 358 l 86 655 q 108 651 94 653 q 144 649 122 649 q 174 650 163 649 q 210 655 185 651 q 199 534 199 591 q 293 637 242 602 q 418 672 343 672 q 618 567 548 672 q 689 322 689 463 q 617 84 689 183 q 415 -15 546 -15 q 210 98 282 -15 l 210 -100 q 210 -243 210 -148 q 210 -372 210 -339 q 182 -368 197 -369 q 148 -367 167 -367 q 128 -367 136 -367 q 86 -372 121 -367 q 86 -239 86 -328 q 86 -106 86 -151 m 377 596 q 240 514 283 596 q 197 327 197 433 q 239 133 197 219 q 376 47 280 47 q 511 131 469 47 q 552 322 552 215 q 510 509 552 423 q 377 596 467 596 "
    },
    "Ïˆ": {
      "x_min": 78,
      "x_max": 982,
      "ha": 1017,
      "o": "m 78 294 l 78 477 l 78 651 l 142 651 l 205 651 q 205 517 205 604 q 205 421 205 430 l 205 341 q 262 117 205 197 q 451 38 319 38 l 451 260 q 451 535 451 342 q 451 785 451 728 q 485 778 468 781 q 514 776 502 776 q 554 780 533 776 q 580 785 575 784 q 580 496 580 689 q 580 270 580 302 l 580 39 q 776 133 703 39 q 850 358 850 227 q 807 641 850 506 q 867 644 840 641 q 941 656 894 648 q 982 418 982 544 q 873 116 982 230 q 582 -14 765 3 l 582 -222 l 582 -372 q 549 -366 566 -368 q 514 -364 532 -364 q 480 -366 499 -364 q 450 -372 460 -369 q 450 -162 450 -302 q 450 -13 450 -22 q 190 64 303 -13 q 78 294 78 142 "
    },
    "Ãœ": {
      "x_min": 99.109375,
      "x_max": 900.609375,
      "ha": 995,
      "o": "m 179 929 l 257 929 q 245 805 245 854 q 245 697 245 756 l 245 457 q 310 136 245 212 q 507 60 376 60 q 719 130 642 60 q 797 336 797 201 l 797 458 l 797 656 q 783 929 797 802 l 842 929 l 900 929 q 887 770 887 863 q 887 600 887 677 l 887 366 q 779 82 887 179 q 481 -15 672 -15 q 204 66 299 -15 q 110 325 110 148 l 110 426 l 110 698 q 110 805 110 756 q 99 929 110 854 l 179 929 m 403 1208 q 454 1186 434 1208 q 474 1133 474 1164 q 452 1081 474 1105 q 403 1058 430 1058 q 350 1080 373 1058 q 328 1133 328 1103 q 349 1185 328 1163 q 403 1208 371 1208 m 638 1208 q 691 1187 669 1208 q 713 1133 713 1166 q 690 1080 713 1103 q 638 1058 668 1058 q 586 1078 608 1058 q 565 1133 565 1099 q 586 1185 565 1163 q 638 1208 608 1208 "
    },
    "Ã ": {
      "x_min": 42,
      "x_max": 642,
      "ha": 635,
      "o": "m 229 -16 q 96 32 150 -16 q 42 161 42 81 q 104 302 42 272 q 297 363 166 332 q 435 447 428 395 q 395 568 435 521 q 282 616 355 616 q 187 587 228 616 q 121 507 146 559 l 93 520 l 101 590 q 198 651 141 631 q 316 672 255 672 q 490 622 435 672 q 546 454 546 572 l 546 132 q 556 68 546 83 q 605 54 566 54 q 642 57 629 54 l 642 26 q 582 5 613 14 q 526 -6 551 -3 q 430 83 445 -6 q 336 9 381 34 q 229 -16 291 -16 m 383 742 l 187 865 q 148 895 167 876 q 130 933 130 914 q 145 969 130 954 q 182 984 160 984 q 218 971 195 984 q 249 944 240 957 l 432 742 l 383 742 m 173 185 q 200 97 173 132 q 279 63 228 63 q 383 106 336 63 q 430 211 430 150 l 430 345 q 235 292 298 318 q 173 185 173 267 "
    },
    "Î·": {
      "x_min": 90,
      "x_max": 647,
      "ha": 740,
      "o": "m 584 -365 q 548 -367 567 -365 q 523 -370 528 -369 q 523 -190 523 -310 q 523 -8 523 -69 l 523 406 q 490 536 523 485 q 383 587 457 587 q 251 508 285 587 q 218 314 218 430 q 218 133 218 259 q 218 1 218 8 l 155 6 l 94 1 l 94 156 l 94 495 q 92 583 94 527 q 90 655 90 639 q 116 650 102 652 q 152 648 130 648 q 181 649 171 648 q 215 655 191 650 l 215 538 q 314 637 258 602 q 444 672 370 672 q 595 603 544 672 q 647 430 647 535 l 647 328 l 647 -185 q 647 -293 647 -227 q 647 -370 647 -359 q 619 -366 634 -367 q 584 -365 604 -365 "
    }
  },
  "cssFontWeight": "normal",
  "ascender": 1267,
  "underlinePosition": -133,
  "cssFontStyle": "normal",
  "boundingBox": {
    "yMin": -373.546875,
    "xMin": -69,
    "yMax": 1266,
    "xMax": 1481
  },
  "resolution": 1000,
  "original_font_information": {
    "postscript_name": "Optimer-Regular",
    "version_string": "Version 1.00 2004 initial release",
    "vendor_url": "http://www.magenta.gr/",
    "full_font_name": "Optimer",
    "font_family_name": "Optimer",
    "copyright": "Copyright (c) Magenta Ltd., 2004",
    "description": "",
    "trademark": "",
    "designer": "",
    "designer_url": "",
    "unique_font_identifier": "Magenta Ltd.:Optimer:22-10-104",
    "license_url": "http://www.ellak.gr/fonts/MgOpen/license.html",
    "license_description": "Copyright (c) 2004 by MAGENTA Ltd. All Rights Reserved.\r\n\r\nPermission is hereby granted, free of charge, to any person obtaining a copy of the fonts accompanying this license (\"Fonts\") and associated documentation files (the \"Font Software\"), to reproduce and distribute the Font Software, including without limitation the rights to use, copy, merge, publish, distribute, and/or sell copies of the Font Software, and to permit persons to whom the Font Software is furnished to do so, subject to the following conditions: \r\n\r\nThe above copyright and this permission notice shall be included in all copies of one or more of the Font Software typefaces.\r\n\r\nThe Font Software may be modified, altered, or added to, and in particular the designs of glyphs or characters in the Fonts may be modified and additional glyphs or characters may be added to the Fonts, only if the fonts are renamed to names not containing the word \"MgOpen\", or if the modifications are accepted for inclusion in the Font Software itself by the each appointed Administrator.\r\n\r\nThis License becomes null and void to the extent applicable to Fonts or Font Software that has been modified and is distributed under the \"MgOpen\" name.\r\n\r\nThe Font Software may be sold as part of a larger software package but no copy of one or more of the Font Software typefaces may be sold by itself. \r\n\r\nTHE FONT SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO ANY WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF COPYRIGHT, PATENT, TRADEMARK, OR OTHER RIGHT. IN NO EVENT SHALL MAGENTA OR PERSONS OR BODIES IN CHARGE OF ADMINISTRATION AND MAINTENANCE OF THE FONT SOFTWARE BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, INCLUDING ANY GENERAL, SPECIAL, INDIRECT, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF THE USE OR INABILITY TO USE THE FONT SOFTWARE OR FROM OTHER DEALINGS IN THE FONT SOFTWARE.",
    "manufacturer_name": "Magenta Ltd.",
    "font_sub_family_name": "Regular"
  },
  "descender": -374,
  "familyName": "Optimer",
  "lineHeight": 1640,
  "underlineThickness": 20
});
if (_typeface_js && _typeface_js.loadFace) _typeface_js.loadFace({
  "glyphs": {
    "Î¿": {
      "x_min": 0,
      "x_max": 698,
      "ha": 799,
      "o": "m 349 -25 q 94 88 189 -25 q 0 368 0 201 q 91 642 0 533 q 349 761 189 761 q 605 644 507 761 q 698 368 698 533 q 606 91 698 201 q 349 -25 510 -25 m 349 84 q 517 174 455 84 q 571 369 571 255 q 518 562 571 484 q 349 651 457 651 q 185 561 244 651 q 132 369 132 481 q 183 177 132 256 q 349 84 244 84 "
    },
    "S": {
      "x_min": 0,
      "x_max": 772,
      "ha": 873,
      "o": "m 772 292 q 649 54 772 144 q 389 -26 539 -26 q 114 68 221 -26 q 0 338 0 168 l 128 338 q 195 153 128 221 q 376 85 263 85 q 545 129 469 85 q 637 271 637 183 q 480 429 637 379 q 190 514 334 471 q 33 740 33 585 q 140 965 33 882 q 381 1042 238 1042 q 631 958 531 1042 q 740 717 740 868 l 612 717 q 549 875 612 817 q 387 934 487 934 q 238 892 303 934 q 161 760 161 842 q 318 610 161 657 q 612 526 465 568 q 772 292 772 454 "
    },
    "Â¦": {
      "x_min": 336,
      "x_max": 439,
      "ha": 775,
      "o": "m 439 462 l 336 462 l 336 986 l 439 986 l 439 462 m 439 -242 l 336 -242 l 336 280 l 439 280 l 439 -242 "
    },
    "/": {
      "x_min": 179.671875,
      "x_max": 596.171875,
      "ha": 776,
      "o": "m 596 1042 l 261 -129 l 179 -129 l 510 1042 l 596 1042 "
    },
    "Î¤": {
      "x_min": 0,
      "x_max": 762,
      "ha": 822,
      "o": "m 762 893 l 449 893 l 449 0 l 313 0 l 313 891 l 0 891 l 0 1012 l 762 1012 l 762 893 "
    },
    "y": {
      "x_min": -0.28125,
      "x_max": 670.75,
      "ha": 755,
      "o": "m 670 738 l 380 -83 q 304 -216 349 -167 q 169 -279 247 -279 q 95 -266 130 -279 l 95 -148 q 129 -155 107 -151 q 164 -160 152 -160 q 235 -114 209 -160 q 269 -26 243 -98 l 0 738 l 134 738 l 334 140 l 537 738 l 670 738 "
    },
    "Î ": {
      "x_min": 0,
      "x_max": 787,
      "ha": 899,
      "o": "m 787 0 l 654 0 l 654 886 l 137 886 l 137 0 l 0 0 l 0 1012 l 787 1012 l 787 0 "
    },
    "Î": {
      "x_min": -109,
      "x_max": 332,
      "ha": 354,
      "o": "m 332 800 l 224 800 l 224 925 l 332 925 l 332 800 m -1 800 l -109 800 l -109 925 l -1 925 l -1 800 m 277 3 q 227 -10 252 -5 q 177 -15 202 -15 q 82 26 116 -15 q 40 200 40 79 l 40 738 l 163 738 l 163 215 q 167 142 163 157 q 220 102 178 102 q 242 103 233 102 q 277 113 251 104 l 277 3 m 296 1040 l 111 820 l 29 820 l 161 1040 l 296 1040 "
    },
    "g": {
      "x_min": 0,
      "x_max": 672,
      "ha": 820,
      "o": "m 672 34 q 574 -213 672 -121 q 324 -306 477 -306 q 128 -253 212 -306 q 31 -85 31 -190 l 152 -85 q 224 -175 163 -139 q 338 -207 278 -207 q 504 -109 446 -207 q 553 88 553 -28 q 452 5 511 35 q 329 -24 394 -24 q 87 99 180 -24 q 0 370 0 214 q 85 634 0 521 q 323 758 179 758 q 447 728 390 758 q 552 643 504 699 l 552 737 l 672 737 l 672 34 m 570 367 q 518 560 570 481 q 351 652 458 652 q 185 561 244 652 q 132 368 132 482 q 185 175 132 254 q 353 84 246 84 q 518 175 458 84 q 570 367 570 254 "
    },
    "Â²": {
      "x_min": 0,
      "x_max": 433,
      "ha": 528,
      "o": "m 433 382 l 0 382 q 90 565 0 491 q 255 667 173 616 q 347 797 347 726 q 309 874 347 844 q 223 904 272 904 q 133 868 170 904 q 97 761 97 833 l 14 761 q 80 921 14 863 q 227 973 139 973 q 371 925 310 973 q 433 796 433 877 q 344 634 433 703 q 179 538 314 610 q 90 454 90 490 l 433 454 l 433 382 "
    },
    "â€“": {
      "x_min": 0,
      "x_max": 691,
      "ha": 786,
      "o": "m 691 335 l 0 335 l 0 411 l 691 411 l 691 335 "
    },
    "Îš": {
      "x_min": 0,
      "x_max": 802.953125,
      "ha": 875,
      "o": "m 802 0 l 636 0 l 288 509 l 136 356 l 136 0 l 0 0 l 0 1012 l 136 1012 l 136 526 l 613 1012 l 793 1012 l 387 599 l 802 0 "
    },
    "Æ’": {
      "x_min": -44.53125,
      "x_max": 384,
      "ha": 502,
      "o": "m 384 651 l 255 651 l 77 -279 l -44 -279 l 132 651 l 14 651 l 14 751 l 132 751 q 148 948 132 901 q 298 1040 181 1040 q 327 1040 313 1040 q 384 1034 341 1040 l 384 922 q 331 932 352 932 q 265 884 282 932 q 255 794 255 854 l 255 751 l 384 751 l 384 651 "
    },
    "e": {
      "x_min": 0,
      "x_max": 699.140625,
      "ha": 796,
      "o": "m 699 327 l 137 327 q 196 157 137 228 q 352 86 255 86 q 478 130 422 86 q 549 246 534 175 l 682 246 q 565 48 656 123 q 350 -26 474 -26 q 95 85 191 -26 q 0 363 0 197 q 92 642 0 529 q 350 765 191 765 q 613 627 518 765 q 699 327 699 504 m 564 430 q 497 583 553 522 q 347 650 436 650 q 202 583 260 650 q 137 430 149 522 l 564 430 "
    },
    "ÏŒ": {
      "x_min": 0,
      "x_max": 698,
      "ha": 799,
      "o": "m 349 -25 q 92 91 190 -25 q 0 368 0 202 q 91 642 0 533 q 349 761 189 761 q 605 644 507 761 q 698 368 698 533 q 606 91 698 201 q 349 -25 510 -25 m 349 84 q 517 174 455 84 q 571 369 571 255 q 518 562 571 484 q 349 651 457 651 q 185 561 244 651 q 132 369 132 481 q 183 177 132 256 q 349 84 244 84 m 564 1040 l 379 819 l 297 819 l 429 1040 l 564 1040 "
    },
    "J": {
      "x_min": 0,
      "x_max": 576,
      "ha": 685,
      "o": "m 576 278 q 282 -27 576 -27 q 57 72 124 -27 q 0 327 0 157 l 131 327 q 158 171 131 226 q 283 95 194 95 q 418 170 384 95 q 440 335 440 218 l 440 1012 l 576 1012 l 576 278 "
    },
    "Â»": {
      "x_min": 0,
      "x_max": 494,
      "ha": 589,
      "o": "m 494 302 l 276 137 l 276 256 l 420 372 l 276 486 l 276 608 l 494 440 l 494 302 m 218 302 l 0 137 l 0 256 l 143 372 l 0 486 l 0 608 l 218 440 l 218 302 "
    },
    "Â©": {
      "x_min": -3,
      "x_max": 988,
      "ha": 1083,
      "o": "m 492 -7 q 120 151 258 -7 q -3 501 -3 294 q 120 851 -3 706 q 492 1011 258 1011 q 864 851 725 1011 q 988 501 988 708 q 865 151 988 292 q 492 -7 729 -7 m 492 60 q 813 197 695 60 q 921 501 921 322 q 814 805 921 681 q 492 944 695 944 q 171 805 291 944 q 64 501 64 680 q 170 197 64 320 q 492 60 288 60 m 726 394 q 648 246 716 302 q 486 190 579 190 q 288 285 361 190 q 223 496 223 370 q 289 713 223 624 q 489 812 363 812 q 643 761 576 812 q 718 624 709 710 l 621 624 q 577 703 616 673 q 488 734 538 734 q 368 665 411 734 q 326 503 326 596 q 366 339 326 407 q 479 272 406 272 q 577 304 538 272 q 625 394 616 337 l 726 394 "
    },
    "ÏŽ": {
      "x_min": 0,
      "x_max": 904,
      "ha": 1010,
      "o": "m 673 1040 l 488 820 l 406 820 l 538 1040 l 673 1040 m 904 339 q 839 97 904 203 q 637 -26 765 -26 q 527 9 575 -26 q 451 103 479 44 q 379 12 427 46 q 272 -21 332 -21 q 68 97 144 -21 q 0 339 0 203 q 44 552 0 445 q 157 737 82 643 l 296 737 q 171 553 214 647 q 121 336 121 446 q 152 179 121 249 q 269 88 193 88 q 367 163 334 88 q 392 295 392 220 l 392 573 l 513 573 l 513 295 q 549 135 513 192 q 630 88 579 88 q 746 182 704 88 q 781 342 781 257 q 730 556 781 450 q 606 737 691 638 l 745 737 q 857 551 819 640 q 904 339 904 444 "
    },
    "^": {
      "x_min": 189.1875,
      "x_max": 586.640625,
      "ha": 776,
      "o": "m 586 772 l 504 772 l 387 930 l 272 772 l 189 772 l 319 1012 l 453 1012 l 586 772 "
    },
    "Â«": {
      "x_min": 0,
      "x_max": 497,
      "ha": 592,
      "o": "m 497 137 l 278 302 l 278 440 l 497 608 l 497 486 l 352 372 l 497 256 l 497 137 m 218 137 l 0 302 l 0 440 l 218 608 l 218 486 l 73 372 l 218 256 l 218 137 "
    },
    "D": {
      "x_min": 0,
      "x_max": 811,
      "ha": 916,
      "o": "m 381 1012 q 699 866 581 1012 q 811 520 811 728 q 698 160 811 308 q 374 -1 575 -1 l 0 -1 l 0 1012 l 381 1012 m 368 123 q 594 246 513 123 q 667 509 667 354 q 594 770 667 661 q 368 895 511 895 l 136 895 l 136 123 l 368 123 "
    },
    "âˆ™": {
      "x_min": 0,
      "x_max": 139,
      "ha": 234,
      "o": "m 139 585 l 0 585 l 0 738 l 139 738 l 139 585 "
    },
    "Ã¿": {
      "x_min": 0.28125,
      "x_max": 46,
      "ha": 122,
      "o": "m 46 2 q 36 -7 46 -7 q 27 0 29 -7 q 37 -4 30 -4 q 43 2 43 -1 l 36 0 q 27 9 27 0 q 36 19 27 19 l 43 15 l 46 19 l 46 2 m 36 1 q 43 9 43 1 q 36 17 43 17 q 29 9 29 17 q 36 1 29 1 m 25 0 l 23 21 l 13 0 l 3 21 l 3 0 l 0 24 l 13 3 l 21 24 l 25 0 "
    },
    "w": {
      "x_min": 0,
      "x_max": 989.53125,
      "ha": 1078,
      "o": "m 989 737 l 767 0 l 645 0 l 491 566 l 338 0 l 217 0 l 0 737 l 127 737 l 279 173 l 423 737 l 564 737 l 706 173 l 862 737 l 989 737 "
    },
    "$": {
      "x_min": 0,
      "x_max": 686,
      "ha": 777,
      "o": "m 650 718 l 530 718 q 480 825 519 786 q 373 872 440 865 l 373 551 q 607 446 529 522 q 686 241 686 371 q 605 45 686 116 q 373 -25 525 -25 l 373 -151 l 301 -151 l 301 -24 q 79 62 159 -24 q 0 298 0 149 l 121 298 q 165 147 121 204 q 301 80 210 89 l 301 442 q 79 536 145 469 q 12 725 12 603 q 94 910 12 839 q 301 982 177 982 l 301 1077 l 373 1077 l 373 982 q 562 918 484 982 q 650 718 650 845 m 301 565 l 301 872 q 183 832 228 872 q 139 725 139 791 q 176 618 139 657 q 301 565 213 580 m 373 76 q 551 237 551 97 q 506 362 551 314 q 373 424 462 410 l 373 76 "
    },
    "\\": {
      "x_min": 0,
      "x_max": 416.5,
      "ha": 512,
      "o": "m 416 -129 l 330 -129 l 0 1042 l 81 1042 l 416 -129 "
    },
    "Âµ": {
      "x_min": 0,
      "x_max": 683.21875,
      "ha": 732,
      "o": "m 683 -4 q 616 -14 645 -14 q 488 97 503 -14 q 414 8 461 41 q 307 -24 367 -24 q 203 3 253 -24 q 117 80 153 31 l 117 -278 l 0 -278 l 0 737 l 121 737 l 121 343 q 161 172 121 245 q 302 82 212 82 q 443 177 394 82 q 483 358 483 254 l 483 737 l 604 737 l 604 214 q 610 136 604 159 q 660 92 624 92 q 683 96 670 92 l 683 -4 "
    },
    "Î™": {
      "x_min": 41,
      "x_max": 177,
      "ha": 291,
      "o": "m 177 0 l 41 0 l 41 1012 l 177 1012 l 177 0 "
    },
    "ÎŽ": {
      "x_min": -0.328125,
      "x_max": 1120.75,
      "ha": 1189,
      "o": "m 1120 1012 l 790 416 l 790 0 l 653 0 l 653 416 l 318 1012 l 455 1012 l 720 533 l 983 1012 l 1120 1012 m 271 1040 l 81 800 l 0 800 l 137 1040 l 271 1040 "
    },
    "â€™": {
      "x_min": 0,
      "x_max": 136,
      "ha": 231,
      "o": "m 136 851 q 99 737 136 784 q 0 670 63 691 l 0 734 q 58 787 42 741 q 71 872 71 820 l 0 872 l 0 1012 l 136 1012 l 136 851 "
    },
    "Î": {
      "x_min": 0,
      "x_max": 785,
      "ha": 897,
      "o": "m 785 0 l 637 0 l 128 821 l 128 0 l 0 0 l 0 1012 l 148 1012 l 657 191 l 657 1012 l 785 1012 l 785 0 "
    },
    "-": {
      "x_min": 10,
      "x_max": 345,
      "ha": 469,
      "o": "m 345 317 l 10 317 l 10 428 l 345 428 l 345 317 "
    },
    "Q": {
      "x_min": 0,
      "x_max": 948,
      "ha": 1050,
      "o": "m 934 6 l 869 -78 l 729 36 q 609 -11 673 3 q 473 -26 545 -26 q 125 130 257 -26 q 0 504 0 279 q 125 880 0 729 q 474 1042 257 1042 q 824 885 693 1042 q 948 508 948 736 q 914 294 948 399 q 816 106 881 189 l 934 6 m 708 192 q 786 331 762 249 q 811 500 811 413 q 729 791 811 674 q 473 923 636 923 q 223 792 315 923 q 139 506 139 674 q 223 221 139 338 q 477 91 317 91 q 620 124 555 91 l 510 216 l 575 302 l 708 192 "
    },
    "Ï‚": {
      "x_min": 1,
      "x_max": 662.46875,
      "ha": 725,
      "o": "m 662 461 l 539 461 q 487 595 531 547 q 357 651 438 651 q 195 578 258 651 q 133 401 133 505 q 260 179 133 241 q 497 106 378 142 q 627 -50 627 62 q 612 -158 627 -105 q 570 -278 598 -211 l 455 -278 q 487 -182 479 -211 q 504 -80 504 -126 q 373 12 504 -15 q 131 91 192 51 q 1 388 1 179 q 98 651 1 542 q 346 761 195 761 q 575 680 488 761 q 662 461 662 599 "
    },
    "M": {
      "x_min": 0,
      "x_max": 935,
      "ha": 1045,
      "o": "m 935 0 l 803 0 l 803 867 l 526 0 l 397 0 l 125 864 l 125 0 l 0 0 l 0 1012 l 195 1012 l 462 158 l 743 1012 l 935 1012 l 935 0 "
    },
    "Î¨": {
      "x_min": 0,
      "x_max": 986,
      "ha": 1073,
      "o": "m 986 677 q 896 319 986 429 q 560 200 797 200 l 560 0 l 425 0 l 425 200 q 91 319 190 200 q 0 677 0 429 l 0 1012 l 136 1012 l 136 678 q 187 417 136 492 q 425 326 250 326 l 425 1012 l 560 1012 l 560 326 l 569 326 q 797 423 732 326 q 851 678 851 502 l 851 1012 l 986 1012 l 986 677 "
    },
    "C": {
      "x_min": 0,
      "x_max": 869,
      "ha": 926,
      "o": "m 869 379 q 745 87 869 201 q 446 -26 622 -26 q 110 136 231 -26 q 0 510 0 283 q 115 883 0 738 q 459 1042 240 1042 q 733 955 617 1042 q 861 709 861 860 l 731 709 q 636 863 710 806 q 464 921 562 921 q 214 791 306 921 q 133 509 133 676 q 213 229 133 344 q 461 99 304 99 q 658 179 579 99 q 738 379 738 259 l 869 379 "
    },
    "!": {
      "x_min": 0,
      "x_max": 136,
      "ha": 231,
      "o": "m 136 684 q 114 409 136 628 q 104 244 104 299 l 33 244 q 16 464 33 313 q 0 684 0 616 l 0 1012 l 136 1012 l 136 684 m 136 0 l 0 0 l 0 151 l 136 151 l 136 0 "
    },
    "{": {
      "x_min": 0,
      "x_max": 471,
      "ha": 566,
      "o": "m 471 -286 q 232 -213 297 -286 q 183 -45 183 -159 q 188 48 183 -15 q 194 141 194 112 q 158 263 194 225 q 0 314 113 314 l 0 417 q 159 471 116 417 q 193 605 193 514 q 193 665 193 644 q 193 772 193 769 q 240 941 193 887 q 471 1015 304 1015 l 471 915 q 329 866 367 915 q 303 742 303 829 q 303 662 303 718 q 303 577 303 607 q 280 453 303 500 q 172 365 251 391 q 283 275 253 337 q 306 143 306 226 q 306 84 306 106 q 306 -11 306 -5 q 333 -131 306 -94 q 471 -182 370 -182 l 471 -286 "
    },
    "X": {
      "x_min": 0,
      "x_max": 837.078125,
      "ha": 921,
      "o": "m 837 0 l 669 0 l 415 409 l 163 0 l 0 0 l 340 519 l 17 1012 l 182 1012 l 419 637 l 661 1012 l 819 1012 l 494 520 l 837 0 "
    },
    "#": {
      "x_min": 0,
      "x_max": 944.609375,
      "ha": 1040,
      "o": "m 944 691 l 909 591 l 705 591 l 642 410 l 858 410 l 823 310 l 605 310 l 498 -2 l 386 -2 l 495 310 l 322 310 l 210 -2 l 100 -2 l 208 310 l 0 310 l 35 410 l 243 410 l 306 591 l 84 591 l 118 691 l 341 691 l 450 1006 l 562 1006 l 454 691 l 628 691 l 736 1006 l 847 1006 l 740 691 l 944 691 m 594 591 l 416 591 l 355 410 l 532 410 l 594 591 "
    },
    "Î¹": {
      "x_min": 41,
      "x_max": 278.34375,
      "ha": 355,
      "o": "m 278 3 q 228 -10 253 -5 q 178 -15 203 -15 q 83 26 117 -15 q 41 200 41 79 l 41 738 l 164 738 l 164 215 q 168 142 164 157 q 222 102 179 102 q 243 103 234 102 q 278 113 252 104 l 278 3 "
    },
    "Î†": {
      "x_min": 0,
      "x_max": 888.8125,
      "ha": 962,
      "o": "m 277 1039 l 87 800 l 5 800 l 142 1039 l 277 1039 m 888 0 l 741 0 l 637 302 l 246 302 l 140 0 l 0 0 l 368 1012 l 518 1012 l 888 0 m 597 420 l 443 866 l 287 420 l 597 420 "
    },
    ")": {
      "x_min": -0.1875,
      "x_max": 312,
      "ha": 407,
      "o": "m 312 365 q 252 25 312 191 q 85 -290 193 -141 l 0 -290 q 137 21 91 -128 q 189 360 189 189 q 138 704 189 537 q 0 1024 95 850 l 85 1024 q 252 706 193 871 q 312 365 312 542 "
    },
    "Îµ": {
      "x_min": 0,
      "x_max": 622.53125,
      "ha": 700,
      "o": "m 622 235 q 517 38 622 111 q 294 -25 425 -25 q 96 29 179 -25 q 0 204 0 93 q 36 313 0 265 q 127 390 66 352 q 56 459 82 419 q 26 555 26 505 q 113 712 26 654 q 290 763 188 763 q 490 700 408 763 q 579 515 579 632 l 469 515 q 411 618 455 580 q 302 657 367 657 q 204 630 249 657 q 149 547 149 598 q 234 446 149 469 q 382 435 274 435 l 382 332 l 342 332 q 203 315 250 332 q 123 210 123 287 q 180 108 123 145 q 297 76 229 76 q 427 117 371 76 q 499 235 483 159 l 622 235 "
    },
    "Î”": {
      "x_min": 0,
      "x_max": 933.71875,
      "ha": 1007,
      "o": "m 933 -1 l 0 -1 l 392 1012 l 540 1012 l 933 -1 m 747 123 l 466 866 l 183 123 l 747 123 "
    },
    "}": {
      "x_min": 0,
      "x_max": 471,
      "ha": 566,
      "o": "m 471 314 q 311 262 356 314 q 276 136 276 222 q 278 65 276 97 q 288 -58 288 -48 q 237 -217 288 -166 q 0 -286 171 -286 l 0 -182 q 138 -130 103 -182 q 164 -2 164 -93 q 164 81 164 22 q 164 144 164 140 q 186 275 164 229 q 299 365 215 339 q 187 455 218 391 q 163 589 163 505 q 163 681 163 625 q 163 743 163 737 q 136 865 163 827 q 0 915 100 915 l 0 1015 q 237 942 172 1015 q 286 773 286 888 q 281 675 286 741 q 276 590 276 608 q 311 466 276 505 q 471 417 356 417 l 471 314 "
    },
    "â€°": {
      "x_min": -3,
      "x_max": 1638,
      "ha": 1784,
      "o": "m 828 0 q 650 76 717 0 q 591 244 591 145 q 649 412 591 344 q 829 489 714 489 q 1007 412 940 489 q 1067 244 1067 343 q 1008 76 1067 144 q 828 0 943 0 m 827 103 q 926 143 891 103 q 961 243 961 184 q 928 340 961 301 q 827 385 891 385 q 730 342 766 385 q 694 243 694 300 q 726 147 694 186 q 827 103 764 103 m 871 986 l 278 -26 l 195 -26 l 788 986 l 871 986 m 236 468 q 56 545 123 468 q -3 714 -3 615 q 55 881 -3 813 q 233 958 122 958 q 413 881 346 958 q 473 712 473 813 q 414 544 473 612 q 236 468 349 468 m 236 855 q 134 811 171 855 q 98 710 98 768 q 133 612 98 653 q 235 572 168 572 q 337 614 300 572 q 374 713 374 656 q 340 810 374 771 q 236 855 301 855 m 1399 0 q 1221 76 1288 0 q 1162 244 1162 145 q 1220 412 1162 344 q 1400 489 1285 489 q 1578 412 1511 489 q 1638 244 1638 343 q 1579 76 1638 144 q 1399 0 1514 0 m 1398 103 q 1497 143 1462 103 q 1532 243 1532 184 q 1499 340 1532 301 q 1398 385 1462 385 q 1301 342 1337 385 q 1265 243 1265 300 q 1297 147 1265 186 q 1398 103 1335 103 "
    },
    "a": {
      "x_min": 0,
      "x_max": 685,
      "ha": 779,
      "o": "m 685 0 q 648 -12 665 -7 q 602 -17 630 -17 q 525 12 552 -17 q 489 95 497 41 q 375 6 446 37 q 231 -25 305 -25 q 63 31 127 -25 q 0 194 0 88 q 115 390 0 334 q 321 436 176 420 q 477 483 466 451 q 484 523 484 504 q 432 619 484 584 q 318 654 380 654 q 204 617 251 654 q 149 514 157 580 l 32 514 q 120 705 32 633 q 324 772 202 772 q 516 712 439 772 q 604 532 604 645 l 604 163 q 611 108 604 126 q 650 90 619 90 l 685 94 l 685 0 m 480 262 l 480 372 q 266 329 342 347 q 125 201 125 294 q 163 113 125 144 q 258 83 201 83 q 405 130 338 83 q 480 262 480 183 "
    },
    "â€”": {
      "x_min": 0,
      "x_max": 923,
      "ha": 1018,
      "o": "m 923 335 l 0 335 l 0 411 l 923 411 l 923 335 "
    },
    "=": {
      "x_min": 10,
      "x_max": 767,
      "ha": 777,
      "o": "m 767 510 l 10 510 l 10 606 l 767 606 l 767 510 m 767 235 l 10 235 l 10 332 l 767 332 l 767 235 "
    },
    "N": {
      "x_min": 0,
      "x_max": 785,
      "ha": 895,
      "o": "m 785 0 l 637 0 l 128 821 l 128 0 l 0 0 l 0 1012 l 148 1012 l 657 191 l 657 1012 l 785 1012 l 785 0 "
    },
    "Ï": {
      "x_min": 0,
      "x_max": 698,
      "ha": 781,
      "o": "m 698 368 q 608 93 698 206 q 355 -27 511 -27 q 225 1 287 -27 q 117 82 164 29 l 117 -278 l 0 -278 l 0 361 q 89 642 0 531 q 348 763 186 763 q 605 646 507 763 q 698 368 698 535 m 571 366 q 519 559 571 480 q 352 651 459 651 q 186 562 247 651 q 131 370 131 482 q 185 175 131 256 q 352 84 244 84 q 518 174 457 84 q 571 366 571 253 "
    },
    "2": {
      "x_min": 59,
      "x_max": 718,
      "ha": 777,
      "o": "m 718 0 l 59 0 q 195 313 59 188 q 449 487 196 314 q 587 691 587 580 q 533 819 587 772 q 404 867 479 867 q 268 811 323 867 q 206 630 206 747 l 81 630 q 180 901 81 805 q 402 986 267 986 q 618 909 527 986 q 718 694 718 826 q 602 449 718 541 q 372 316 486 382 q 197 122 231 234 l 718 122 l 718 0 "
    },
    "Â¯": {
      "x_min": 0,
      "x_max": 923,
      "ha": 919,
      "o": "m 923 1033 l 0 1033 l 0 1109 l 923 1109 l 923 1033 "
    },
    "Z": {
      "x_min": 0,
      "x_max": 764,
      "ha": 832,
      "o": "m 764 -1 l 0 -1 l 0 113 l 608 895 l 40 895 l 40 1012 l 764 1012 l 764 886 l 167 123 l 764 123 l 764 -1 "
    },
    "u": {
      "x_min": 0,
      "x_max": 604,
      "ha": 714,
      "o": "m 604 0 l 488 0 l 488 109 q 383 10 450 45 q 240 -25 315 -25 q 59 58 125 -25 q 0 258 0 136 l 0 737 l 123 737 l 123 284 q 154 148 123 202 q 268 82 193 82 q 424 165 362 82 q 483 340 483 242 l 483 737 l 604 737 l 604 0 "
    },
    "k": {
      "x_min": 0,
      "x_max": 599.828125,
      "ha": 683,
      "o": "m 599 739 l 331 465 l 595 0 l 459 0 l 245 382 l 118 251 l 118 0 l 0 0 l 0 1012 l 118 1012 l 118 402 l 447 739 l 599 739 "
    },
    "Î—": {
      "x_min": 0,
      "x_max": 787,
      "ha": 899,
      "o": "m 787 0 l 654 0 l 654 475 l 137 475 l 137 0 l 0 0 l 0 1012 l 137 1012 l 137 599 l 654 599 l 654 1012 l 787 1012 l 787 0 "
    },
    "Î‘": {
      "x_min": 0,
      "x_max": 888.8125,
      "ha": 965,
      "o": "m 888 0 l 741 0 l 637 302 l 246 302 l 140 0 l 0 0 l 368 1012 l 518 1012 l 888 0 m 597 420 l 443 866 l 287 420 l 597 420 "
    },
    "s": {
      "x_min": 0,
      "x_max": 592,
      "ha": 683,
      "o": "m 592 217 q 491 36 592 104 q 287 -23 402 -23 q 84 43 163 -23 q 0 238 0 114 l 118 238 q 171 122 118 164 q 293 85 218 85 q 406 112 356 85 q 469 207 469 147 q 353 310 469 276 q 137 372 138 371 q 20 544 20 426 q 108 708 20 647 q 292 761 185 761 q 482 705 405 761 q 571 532 571 643 l 453 532 q 404 625 453 594 q 292 657 356 657 q 195 636 237 657 q 140 558 140 608 q 257 454 140 486 q 474 394 469 397 q 592 217 592 341 "
    },
    "B": {
      "x_min": 0,
      "x_max": 762,
      "ha": 859,
      "o": "m 569 545 q 709 468 657 534 q 762 310 762 402 q 659 82 762 170 q 423 -1 563 -1 l 0 -1 l 0 1012 l 402 1012 q 616 956 530 1012 q 717 767 717 891 q 676 632 717 692 q 569 545 636 571 m 385 898 l 136 898 l 136 587 l 371 587 q 510 623 453 587 q 580 743 580 666 q 520 858 580 818 q 385 898 461 898 m 411 123 q 555 168 494 123 q 622 302 622 218 q 548 435 622 388 q 394 476 484 476 l 136 476 l 136 123 l 411 123 "
    },
    "â€¦": {
      "x_min": 0,
      "x_max": 601,
      "ha": 694,
      "o": "m 139 0 l 0 0 l 0 151 l 139 151 l 139 0 m 370 0 l 231 0 l 231 151 l 370 151 l 370 0 m 601 0 l 462 0 l 462 151 l 601 151 l 601 0 "
    },
    "?": {
      "x_min": 0,
      "x_max": 595,
      "ha": 690,
      "o": "m 595 777 q 532 599 595 674 q 413 474 472 537 q 350 271 350 391 l 232 271 q 291 487 232 395 q 402 619 293 490 q 464 762 464 691 q 413 885 464 838 q 295 933 363 933 q 175 880 223 933 q 121 707 121 819 l 0 707 q 92 963 0 872 q 296 1044 174 1044 q 501 973 415 1044 q 595 777 595 895 m 362 0 l 226 0 l 226 151 l 362 151 l 362 0 "
    },
    "H": {
      "x_min": 0,
      "x_max": 787,
      "ha": 897,
      "o": "m 787 0 l 654 0 l 654 475 l 137 475 l 137 0 l 0 0 l 0 1012 l 137 1012 l 137 599 l 654 599 l 654 1012 l 787 1012 l 787 0 "
    },
    "Î½": {
      "x_min": 0,
      "x_max": 661.5,
      "ha": 746,
      "o": "m 661 737 l 396 0 l 266 0 l 0 737 l 130 737 l 333 147 l 530 737 l 661 737 "
    },
    "c": {
      "x_min": 1,
      "x_max": 687.328125,
      "ha": 759,
      "o": "m 687 264 q 572 53 668 133 q 346 -26 477 -26 q 89 91 184 -26 q 1 370 1 201 q 90 645 1 537 q 346 761 186 761 q 561 688 469 761 q 676 493 653 615 l 545 493 q 477 606 534 562 q 349 650 420 650 q 183 563 242 650 q 132 372 132 487 q 185 179 132 258 q 352 88 246 88 q 483 136 428 88 q 555 264 537 185 l 687 264 "
    },
    "Â¶": {
      "x_min": -0.421875,
      "x_max": 555,
      "ha": 664,
      "o": "m 20 891 l 51 891 l 96 760 l 142 891 l 172 891 l 172 737 l 153 737 l 153 865 l 105 737 l 86 737 l 39 870 l 39 737 l 20 737 l 20 891 m 302 853 l 302 730 q 248 690 302 690 q 224 690 236 690 q 205 695 210 694 l 202 711 l 247 705 q 281 732 281 705 l 281 763 q 239 740 273 740 q 189 796 189 740 q 257 859 189 859 q 283 859 269 859 q 302 853 296 854 m 283 841 l 258 841 q 208 794 208 841 q 243 755 208 755 q 283 795 283 755 l 283 841 m 555 987 l 492 987 l 492 -1 l 429 -1 l 429 987 l 310 987 l 310 -1 l 247 -1 l 247 601 q 79 652 151 601 q 0 805 0 710 q 98 988 0 917 q 303 1052 189 1052 l 555 1052 l 555 987 "
    },
    "Î²": {
      "x_min": 0,
      "x_max": 647,
      "ha": 730,
      "o": "m 462 549 q 598 449 549 521 q 647 280 647 377 q 567 64 647 151 q 360 -22 487 -22 q 234 4 294 -22 q 124 81 175 31 l 124 -278 l 0 -278 l 0 593 q 53 903 0 801 q 311 1042 125 1042 q 509 964 427 1042 q 591 770 591 887 q 556 643 591 701 q 462 549 521 585 m 330 79 q 466 138 409 79 q 524 278 524 198 q 419 436 524 385 q 222 477 337 477 l 222 583 q 391 620 322 583 q 477 762 477 667 q 427 884 477 833 q 306 935 377 935 q 166 861 215 935 q 124 698 124 797 l 124 362 q 167 169 124 242 q 330 79 220 79 "
    },
    "Îœ": {
      "x_min": 0,
      "x_max": 935,
      "ha": 1047,
      "o": "m 935 0 l 803 0 l 803 867 l 526 0 l 397 0 l 125 864 l 125 0 l 0 0 l 0 1012 l 195 1012 l 462 158 l 743 1012 l 935 1012 l 935 0 "
    },
    "ÎŒ": {
      "x_min": -0.328125,
      "x_max": 1097,
      "ha": 1192,
      "o": "m 1097 505 q 974 132 1097 282 q 628 -29 843 -29 q 284 130 413 -29 q 163 505 163 280 q 288 883 163 730 q 636 1046 421 1046 q 978 882 850 1046 q 1097 505 1097 730 m 957 504 q 877 784 957 669 q 631 915 787 915 q 383 785 474 915 q 300 504 300 669 q 383 224 300 339 q 631 95 475 95 q 876 224 786 95 q 957 504 957 339 m 271 1040 l 81 800 l 0 800 l 137 1040 l 271 1040 "
    },
    "Î‰": {
      "x_min": -0.4375,
      "x_max": 1135,
      "ha": 1249,
      "o": "m 1135 0 l 1002 0 l 1002 475 l 485 475 l 485 0 l 348 0 l 348 1012 l 485 1012 l 485 599 l 1002 599 l 1002 1012 l 1135 1012 l 1135 0 m 271 1040 l 81 800 l 0 800 l 137 1040 l 271 1040 "
    },
    "â€¢": {
      "x_min": 0,
      "x_max": 651,
      "ha": 760,
      "o": "m 651 529 q 555 293 651 391 q 325 196 460 196 q 95 294 190 196 q 0 529 0 393 q 94 763 0 665 q 325 861 189 861 q 555 763 460 861 q 651 529 651 665 "
    },
    "Â¥": {
      "x_min": -0.5,
      "x_max": 802.140625,
      "ha": 867,
      "o": "m 552 561 l 681 561 l 681 486 l 510 486 l 472 416 l 472 380 l 681 380 l 681 306 l 472 306 l 472 0 l 335 0 l 335 306 l 122 306 l 122 380 l 335 380 l 335 416 l 296 486 l 122 486 l 122 561 l 253 561 l 0 1012 l 137 1012 l 402 533 l 664 1012 l 802 1012 l 552 561 "
    },
    "(": {
      "x_min": 0,
      "x_max": 312.1875,
      "ha": 407,
      "o": "m 312 -290 l 226 -290 q 60 23 120 -142 q 0 365 0 190 q 61 712 0 540 q 226 1024 117 869 l 312 1024 q 172 705 215 853 q 123 360 123 542 q 173 22 123 187 q 312 -290 219 -127 "
    },
    "U": {
      "x_min": 0,
      "x_max": 780,
      "ha": 886,
      "o": "m 780 392 q 667 93 780 212 q 378 -25 555 -25 q 99 95 204 -25 q 0 392 0 211 l 0 1012 l 135 1012 l 135 391 q 200 191 135 270 q 386 107 271 107 q 574 191 502 107 q 643 391 643 270 l 643 1012 l 780 1012 l 780 392 "
    },
    "Î³": {
      "x_min": 0,
      "x_max": 730.3125,
      "ha": 807,
      "o": "m 730 737 l 454 53 l 454 -278 l 331 -278 l 331 53 l 151 495 q 102 596 121 569 q 12 651 65 651 l 0 651 l 0 751 l 38 751 q 164 710 118 751 q 237 594 202 675 l 395 208 l 605 737 l 730 737 "
    },
    "Î±": {
      "x_min": 0,
      "x_max": 750,
      "ha": 792,
      "o": "m 750 -3 q 684 -13 711 -13 q 552 97 574 -13 q 456 7 514 40 q 330 -26 399 -26 q 87 98 182 -26 q 0 369 0 212 q 87 637 0 525 q 330 760 181 760 q 456 728 399 760 q 551 637 513 696 l 551 739 l 671 739 l 671 222 q 679 142 671 168 q 733 95 694 95 q 750 96 744 95 l 750 -3 m 572 371 q 520 562 572 484 q 352 653 460 653 q 187 566 249 653 q 132 379 132 489 q 182 180 132 261 q 351 83 242 83 q 517 176 456 83 q 572 371 572 260 "
    },
    "F": {
      "x_min": 0,
      "x_max": 669.203125,
      "ha": 702,
      "o": "m 669 887 l 137 887 l 137 583 l 601 583 l 601 458 l 137 458 l 137 0 l 0 0 l 0 1012 l 669 1012 l 669 887 "
    },
    "Â­": {
      "x_min": 0,
      "x_max": 691,
      "ha": 786,
      "o": "m 691 335 l 0 335 l 0 411 l 691 411 l 691 335 "
    },
    ":": {
      "x_min": 0,
      "x_max": 139,
      "ha": 234,
      "o": "m 139 585 l 0 585 l 0 738 l 139 738 l 139 585 m 139 0 l 0 0 l 0 151 l 139 151 l 139 0 "
    },
    "Î§": {
      "x_min": 0,
      "x_max": 837.078125,
      "ha": 916,
      "o": "m 837 0 l 669 0 l 415 409 l 163 0 l 0 0 l 340 519 l 17 1012 l 182 1012 l 419 637 l 661 1012 l 819 1012 l 494 520 l 837 0 "
    },
    "*": {
      "x_min": 114.140625,
      "x_max": 661.21875,
      "ha": 776,
      "o": "m 661 767 l 466 713 l 598 544 l 508 477 l 387 652 l 269 480 l 175 544 l 308 713 l 114 766 l 150 875 l 336 812 l 336 1012 l 438 1012 l 438 812 l 623 874 l 661 767 "
    },
    "â€ ": {
      "x_min": 0,
      "x_max": 762,
      "ha": 818,
      "o": "m 449 804 l 762 804 l 762 682 l 449 682 l 449 0 l 313 0 l 313 682 l 0 682 l 0 804 l 313 804 l 313 1014 l 449 1014 l 449 804 "
    },
    "Â°": {
      "x_min": 0,
      "x_max": 340,
      "ha": 435,
      "o": "m 170 803 q 42 857 89 803 q 0 978 0 905 q 44 1101 0 1050 q 169 1153 88 1153 q 297 1098 250 1153 q 340 977 340 1050 q 297 857 340 905 q 170 803 251 803 m 170 885 q 233 911 210 885 q 257 973 257 937 q 235 1038 257 1012 q 170 1064 213 1064 q 106 1037 129 1064 q 83 973 83 1011 q 106 911 83 937 q 170 885 129 885 "
    },
    "V": {
      "x_min": 0,
      "x_max": 845.25,
      "ha": 921,
      "o": "m 845 1012 l 495 0 l 353 0 l 0 1012 l 140 1012 l 426 165 l 703 1012 l 845 1012 "
    },
    "Îž": {
      "x_min": 0,
      "x_max": 720,
      "ha": 747,
      "o": "m 709 888 l 9 888 l 9 1012 l 709 1012 l 709 888 m 660 462 l 60 462 l 60 588 l 660 588 l 660 462 m 720 -1 l 0 -1 l 0 123 l 720 123 l 720 -1 "
    },
    "Â ": {
      "x_min": 0,
      "x_max": 0,
      "ha": 836
    },
    "Î«": {
      "x_min": -0.3125,
      "x_max": 802.3125,
      "ha": 870,
      "o": "m 575 1045 l 450 1045 l 450 1188 l 575 1188 l 575 1045 m 352 1045 l 227 1045 l 227 1188 l 352 1188 l 352 1045 m 802 1012 l 472 416 l 472 0 l 335 0 l 335 416 l 0 1012 l 137 1012 l 402 532 l 664 1012 l 802 1012 "
    },
    "0": {
      "x_min": 72,
      "x_max": 701,
      "ha": 776,
      "o": "m 386 -29 q 150 129 238 -29 q 72 479 72 272 q 149 829 72 687 q 386 989 236 989 q 622 829 534 989 q 701 479 701 684 q 623 129 701 270 q 386 -29 536 -29 m 386 89 q 536 211 480 89 q 587 479 587 322 q 537 748 587 640 q 386 871 481 871 q 236 748 292 871 q 186 479 186 637 q 235 211 186 319 q 386 89 291 89 "
    },
    "â€": {
      "x_min": 0,
      "x_max": 340,
      "ha": 445,
      "o": "m 136 851 q 99 737 136 784 q 0 670 63 691 l 0 734 q 58 787 42 741 q 71 872 71 820 l 0 872 l 0 1012 l 136 1012 l 136 851 m 340 851 q 303 737 340 784 q 204 670 267 691 l 204 734 q 262 787 246 741 q 275 872 275 820 l 204 872 l 204 1012 l 340 1012 l 340 851 "
    },
    "@": {
      "x_min": 0,
      "x_max": 1234,
      "ha": 1329,
      "o": "m 1076 -45 q 859 -160 981 -118 q 620 -203 737 -203 q 152 -29 321 -203 q 0 360 0 127 q 172 802 0 616 q 673 1008 364 1008 q 1100 853 949 1008 q 1234 516 1234 717 q 1132 214 1234 339 q 850 80 1023 80 q 756 104 785 80 q 723 201 723 134 q 634 112 686 143 q 517 81 582 81 q 360 146 412 81 q 308 311 308 211 q 393 564 308 452 q 626 690 489 690 q 793 588 744 690 l 832 667 l 919 667 q 860 440 882 531 q 817 224 817 266 q 836 180 817 196 q 885 165 856 165 q 1065 270 992 165 q 1135 510 1135 370 q 1030 793 1135 680 q 673 925 906 925 q 243 747 406 925 q 95 361 95 586 q 221 26 95 159 q 615 -122 363 -122 q 839 -88 722 -122 q 1039 6 957 -54 l 1076 -45 m 770 488 q 724 580 762 545 q 630 615 686 615 q 474 517 538 615 q 417 321 417 430 q 448 202 417 249 q 542 155 480 155 q 708 273 652 155 q 770 488 724 309 "
    },
    "ÎŠ": {
      "x_min": 0.46875,
      "x_max": 489,
      "ha": 601,
      "o": "m 272 1040 l 82 800 l 0 800 l 137 1040 l 272 1040 m 489 0 l 353 0 l 353 1012 l 489 1012 l 489 0 "
    },
    "i": {
      "x_min": 14,
      "x_max": 134,
      "ha": 270,
      "o": "m 134 872 l 14 872 l 14 1012 l 134 1012 l 134 872 m 134 0 l 14 0 l 14 737 l 134 737 l 134 0 "
    },
    "Î’": {
      "x_min": 0,
      "x_max": 762,
      "ha": 859,
      "o": "m 569 545 q 709 468 657 534 q 762 310 762 402 q 659 82 762 170 q 423 -1 563 -1 l 0 -1 l 0 1012 l 402 1012 q 616 956 530 1012 q 717 767 717 891 q 676 632 717 692 q 569 545 636 571 m 385 898 l 136 898 l 136 587 l 371 587 q 510 623 453 587 q 580 743 580 666 q 520 858 580 818 q 385 898 461 898 m 411 123 q 555 168 494 123 q 622 302 622 218 q 548 435 622 388 q 394 476 484 476 l 136 476 l 136 123 l 411 123 "
    },
    "Ï…": {
      "x_min": 0,
      "x_max": 605,
      "ha": 711,
      "o": "m 605 352 q 530 93 605 199 q 302 -24 446 -24 q 74 93 158 -24 q 0 352 0 199 l 0 738 l 124 738 l 124 354 q 166 185 124 257 q 306 98 216 98 q 442 185 394 98 q 482 354 482 257 l 482 738 l 605 738 l 605 352 "
    },
    "]": {
      "x_min": -0.359375,
      "x_max": 269,
      "ha": 364,
      "o": "m 269 -281 l 0 -281 l 0 -187 l 148 -187 l 148 919 l 0 919 l 0 1012 l 269 1012 l 269 -281 "
    },
    "m": {
      "x_min": 0,
      "x_max": 999,
      "ha": 1105,
      "o": "m 999 0 l 879 0 l 879 453 q 843 591 879 535 q 725 659 800 659 q 601 585 646 659 q 562 435 562 521 l 562 0 l 438 0 l 438 453 q 404 589 438 534 q 288 655 364 655 q 162 584 209 655 q 120 435 120 520 l 120 0 l 0 0 l 0 738 l 114 738 l 114 639 q 198 728 147 696 q 310 761 249 761 q 429 728 374 761 q 515 639 484 695 q 608 729 548 697 q 738 762 669 762 q 924 693 850 762 q 999 512 999 624 l 999 0 "
    },
    "Ï‡": {
      "x_min": 8,
      "x_max": 765,
      "ha": 799,
      "o": "m 765 -279 q 701 -295 732 -295 q 604 -258 650 -295 q 537 -176 564 -227 l 371 133 l 140 -278 l 9 -278 l 307 253 l 160 521 q 125 585 129 580 q 35 640 89 640 q 8 637 27 640 l 8 752 l 51 757 q 159 719 111 757 q 231 627 195 690 l 375 372 l 582 737 l 711 737 l 439 249 l 612 -69 q 657 -154 634 -111 q 728 -189 681 -189 q 765 -184 744 -189 l 765 -279 "
    },
    "8": {
      "x_min": 54,
      "x_max": 721,
      "ha": 775,
      "o": "m 560 528 q 680 425 639 492 q 721 281 721 358 q 635 71 721 158 q 387 -26 539 -26 q 139 69 233 -26 q 54 279 54 157 q 94 426 54 360 q 215 528 135 492 q 118 615 150 563 q 87 726 87 668 q 168 904 87 827 q 387 986 256 986 q 606 905 519 986 q 688 728 688 830 q 656 617 688 668 q 560 528 625 565 m 387 566 q 509 611 466 566 q 552 718 552 656 q 511 823 552 782 q 384 872 464 872 q 260 824 307 872 q 220 721 220 783 q 260 614 220 657 q 387 566 307 566 m 387 91 q 534 150 478 91 q 585 281 585 204 q 535 409 585 356 q 387 466 482 466 q 239 409 293 466 q 190 281 190 357 q 239 150 190 203 q 387 91 293 91 "
    },
    "Î¯": {
      "x_min": 41,
      "x_max": 320.53125,
      "ha": 355,
      "o": "m 278 3 q 228 -10 253 -5 q 178 -15 203 -15 q 83 26 117 -15 q 41 200 41 79 l 41 738 l 164 738 l 164 215 q 168 142 164 157 q 222 102 179 102 q 243 103 234 102 q 278 113 252 104 l 278 3 m 320 1040 l 135 820 l 53 820 l 185 1040 l 320 1040 "
    },
    "Î–": {
      "x_min": 0,
      "x_max": 764,
      "ha": 833,
      "o": "m 764 -1 l 0 -1 l 0 113 l 608 895 l 40 895 l 40 1012 l 764 1012 l 764 886 l 167 123 l 764 123 l 764 -1 "
    },
    "R": {
      "x_min": 0,
      "x_max": 766.203125,
      "ha": 889,
      "o": "m 766 -1 l 611 -1 q 576 242 576 52 q 401 433 576 433 l 136 433 l 136 0 l 0 0 l 0 1012 l 388 1012 q 623 945 528 1012 q 734 731 734 867 q 697 596 734 659 q 596 502 660 534 q 704 370 682 474 q 715 207 715 352 q 766 25 715 61 l 766 -1 m 366 551 q 522 594 456 551 q 602 731 602 645 q 521 858 602 814 q 366 895 456 895 l 134 895 l 134 551 l 366 551 "
    },
    "o": {
      "x_min": 0,
      "x_max": 698,
      "ha": 804,
      "o": "m 349 -25 q 92 91 190 -25 q 0 368 0 202 q 91 642 0 533 q 349 761 189 761 q 605 644 507 761 q 698 368 698 533 q 606 91 698 201 q 349 -25 510 -25 m 349 84 q 517 174 455 84 q 571 369 571 255 q 518 562 571 484 q 349 651 457 651 q 185 561 244 651 q 132 369 132 481 q 183 177 132 256 q 349 84 244 84 "
    },
    "5": {
      "x_min": 52.71875,
      "x_max": 722,
      "ha": 775,
      "o": "m 722 313 q 613 59 722 152 q 373 -24 515 -24 q 152 46 243 -24 q 52 255 52 124 l 179 255 q 253 131 199 173 q 373 90 307 90 q 522 148 461 90 q 589 313 589 212 q 527 468 589 410 q 378 527 465 527 q 278 505 324 527 q 192 438 232 484 l 79 438 l 156 958 l 671 958 l 671 840 l 248 840 l 209 578 q 299 626 252 611 q 398 642 346 642 q 622 551 528 642 q 722 313 722 456 "
    },
    "7": {
      "x_min": 59,
      "x_max": 718,
      "ha": 777,
      "o": "m 718 839 q 461 448 550 641 q 329 0 372 255 l 189 0 q 323 441 231 252 q 583 830 414 630 l 59 830 l 59 958 l 718 958 l 718 839 "
    },
    "K": {
      "x_min": 0,
      "x_max": 802.953125,
      "ha": 887,
      "o": "m 802 0 l 636 0 l 288 509 l 136 356 l 136 0 l 0 0 l 0 1012 l 136 1012 l 136 526 l 613 1012 l 793 1012 l 387 599 l 802 0 "
    },
    ",": {
      "x_min": 0,
      "x_max": 139,
      "ha": 234,
      "o": "m 139 -12 q 102 -133 139 -83 q 0 -206 66 -183 l 0 -138 q 55 -83 39 -124 q 68 0 68 -51 l 0 0 l 0 151 l 139 151 l 139 -12 "
    },
    "d": {
      "x_min": 0,
      "x_max": 669,
      "ha": 779,
      "o": "m 669 0 l 552 0 l 552 93 q 446 6 506 38 q 321 -25 387 -25 q 85 99 178 -25 q 0 365 0 215 q 88 639 0 525 q 336 762 183 762 q 552 647 476 762 l 552 1012 l 669 1012 l 669 0 m 570 373 q 518 562 570 484 q 353 653 458 653 q 186 561 247 653 q 132 365 132 479 q 185 175 132 254 q 351 84 246 84 q 518 177 458 84 q 570 373 570 258 "
    },
    "Â¨": {
      "x_min": -107,
      "x_max": 241,
      "ha": 227,
      "o": "m 241 1046 l 116 1046 l 116 1189 l 241 1189 l 241 1046 m 18 1046 l -107 1046 l -107 1189 l 18 1189 l 18 1046 "
    },
    "E": {
      "x_min": 0,
      "x_max": 721.28125,
      "ha": 773,
      "o": "m 721 -1 l 0 -1 l 0 1012 l 710 1012 l 710 888 l 136 888 l 136 584 l 664 584 l 664 466 l 136 466 l 136 124 l 721 124 l 721 -1 "
    },
    "Y": {
      "x_min": 0.171875,
      "x_max": 802.75,
      "ha": 868,
      "o": "m 802 1012 l 472 416 l 472 0 l 335 0 l 335 416 l 0 1012 l 137 1012 l 402 533 l 665 1012 l 802 1012 "
    },
    "\"": {
      "x_min": 0,
      "x_max": 293,
      "ha": 388,
      "o": "m 293 605 l 199 605 l 199 987 l 293 987 l 293 605 m 94 605 l 0 605 l 0 987 l 94 987 l 94 605 "
    },
    "â€¹": {
      "x_min": 18,
      "x_max": 758,
      "ha": 776,
      "o": "m 758 41 l 18 376 l 18 465 l 758 798 l 758 691 l 156 420 l 758 149 l 758 41 "
    },
    "â€ž": {
      "x_min": 0,
      "x_max": 357,
      "ha": 458,
      "o": "m 139 -12 q 102 -133 139 -83 q 0 -206 66 -183 l 0 -138 q 55 -83 39 -124 q 68 0 68 -51 l 0 0 l 0 151 l 139 151 l 139 -12 m 357 -12 q 320 -133 357 -83 q 218 -206 284 -183 l 218 -138 q 273 -83 257 -124 q 286 0 286 -51 l 218 0 l 218 151 l 357 151 l 357 -12 "
    },
    "Î´": {
      "x_min": 1,
      "x_max": 695,
      "ha": 793,
      "o": "m 695 360 q 603 87 695 197 q 349 -28 507 -28 q 96 82 192 -28 q 1 356 1 192 q 98 606 1 509 q 347 703 195 703 q 176 829 282 754 q 69 903 122 866 l 69 1012 l 630 1012 l 630 901 l 252 901 q 452 763 413 794 q 622 592 565 677 q 695 360 695 486 m 571 364 q 540 501 571 446 q 441 602 509 555 q 364 611 402 611 q 192 541 252 611 q 133 355 133 471 q 186 170 133 245 q 350 84 247 84 q 516 173 455 84 q 571 364 571 252 "
    },
    "Î­": {
      "x_min": 0,
      "x_max": 622.53125,
      "ha": 700,
      "o": "m 622 235 q 517 38 622 111 q 294 -25 425 -25 q 96 29 179 -25 q 0 204 0 93 q 36 313 0 265 q 127 390 66 352 q 56 459 82 419 q 26 555 26 505 q 113 712 26 654 q 290 763 188 763 q 490 700 408 763 q 579 515 579 632 l 469 515 q 411 618 455 580 q 302 657 367 657 q 204 630 249 657 q 149 547 149 598 q 234 446 149 469 q 382 435 274 435 l 382 332 l 342 332 q 203 315 250 332 q 123 210 123 287 q 180 108 123 145 q 297 76 229 76 q 427 117 371 76 q 499 235 483 159 l 622 235 m 510 1040 l 325 819 l 243 819 l 375 1040 l 510 1040 "
    },
    "Ï‰": {
      "x_min": 0,
      "x_max": 904,
      "ha": 1010,
      "o": "m 904 339 q 839 97 904 203 q 637 -26 765 -26 q 527 9 575 -26 q 451 103 479 44 q 379 12 427 46 q 272 -21 332 -21 q 68 97 144 -21 q 0 339 0 203 q 44 552 0 445 q 157 737 82 643 l 296 737 q 171 553 214 647 q 121 336 121 446 q 152 179 121 249 q 269 88 193 88 q 367 163 334 88 q 392 295 392 220 l 392 573 l 513 573 l 513 295 q 549 135 513 192 q 630 88 579 88 q 746 182 704 88 q 781 342 781 257 q 730 556 781 450 q 606 737 691 638 l 745 737 q 857 551 819 640 q 904 339 904 444 "
    },
    "Â´": {
      "x_min": 0,
      "x_max": 94,
      "ha": 246,
      "o": "m 94 605 l 0 605 l 0 987 l 94 987 l 94 605 "
    },
    "Â±": {
      "x_min": 11,
      "x_max": 765,
      "ha": 776,
      "o": "m 765 490 l 437 490 l 437 256 l 342 256 l 342 490 l 11 490 l 11 586 l 341 586 l 341 820 l 436 820 l 436 586 l 765 586 l 765 490 m 765 21 l 11 21 l 11 115 l 765 115 l 765 21 "
    },
    "|": {
      "x_min": 336,
      "x_max": 439,
      "ha": 775,
      "o": "m 439 462 l 336 462 l 336 986 l 439 986 l 439 462 m 439 -242 l 336 -242 l 336 280 l 439 280 l 439 -242 "
    },
    "Ï‹": {
      "x_min": 0,
      "x_max": 605,
      "ha": 711,
      "o": "m 473 799 l 365 799 l 365 924 l 473 924 l 473 799 m 234 799 l 126 799 l 126 924 l 234 924 l 234 799 m 605 352 q 530 93 605 199 q 302 -24 446 -24 q 74 93 158 -24 q 0 352 0 199 l 0 738 l 124 738 l 124 354 q 166 185 124 257 q 306 98 216 98 q 442 185 394 98 q 482 354 482 257 l 482 738 l 605 738 l 605 352 "
    },
    "Â§": {
      "x_min": 0,
      "x_max": 581,
      "ha": 676,
      "o": "m 581 425 q 543 311 581 369 q 457 233 505 254 q 527 83 527 172 q 450 -74 527 -12 q 283 -133 379 -133 q 113 -69 180 -133 q 46 96 46 -6 l 163 96 q 195 7 163 40 q 282 -26 228 -26 q 363 -5 325 -26 q 411 60 411 21 q 304 201 411 139 q 106 309 205 255 q 0 490 0 383 q 32 602 0 551 q 121 687 65 654 q 74 743 91 712 q 57 812 57 773 q 130 984 57 920 q 293 1043 197 1043 q 448 987 386 1043 q 518 814 518 925 l 402 814 q 362 908 396 877 q 282 939 329 939 q 207 911 240 939 q 175 841 175 883 q 280 720 175 779 q 474 612 470 615 q 581 425 581 534 m 457 409 q 348 544 457 473 q 191 630 222 612 q 142 587 158 609 q 121 525 121 558 q 233 387 121 462 q 390 298 361 315 q 438 345 420 316 q 457 409 457 375 "
    },
    "b": {
      "x_min": 0,
      "x_max": 671,
      "ha": 768,
      "o": "m 671 372 q 585 99 671 213 q 340 -25 491 -25 q 214 5 272 -25 q 118 92 157 35 l 118 0 l 0 0 l 0 1012 l 118 1012 l 118 634 q 210 723 153 692 q 334 754 266 754 q 579 637 483 754 q 671 372 671 526 m 543 356 q 489 551 543 470 q 322 644 428 644 q 159 556 219 644 q 106 369 106 478 q 157 176 106 256 q 323 83 217 83 q 488 169 427 83 q 543 356 543 245 "
    },
    "q": {
      "x_min": 0,
      "x_max": 669,
      "ha": 858,
      "o": "m 669 -278 l 552 -278 l 552 96 q 464 7 522 38 q 338 -24 406 -24 q 89 92 185 -24 q 0 363 0 202 q 85 634 0 521 q 330 759 180 759 q 456 727 399 759 q 552 637 512 695 l 552 737 l 669 737 l 669 -278 m 570 374 q 516 563 570 487 q 351 651 457 651 q 186 564 247 651 q 132 376 132 487 q 185 178 132 260 q 353 83 246 83 q 519 178 459 83 q 570 374 570 259 "
    },
    "Î©": {
      "x_min": 0,
      "x_max": 950,
      "ha": 1047,
      "o": "m 950 1 l 544 1 l 544 125 q 729 310 661 195 q 798 559 798 424 q 712 813 798 710 q 475 923 620 923 q 239 821 328 923 q 151 569 151 720 q 219 317 151 434 q 404 125 287 201 l 404 1 l 0 1 l 0 125 l 213 125 q 67 328 119 211 q 15 573 15 445 q 142 912 15 782 q 475 1042 269 1042 q 806 910 678 1042 q 934 570 934 778 q 881 327 934 444 q 735 125 829 211 l 950 125 l 950 1 "
    },
    "Ï": {
      "x_min": 0,
      "x_max": 605,
      "ha": 711,
      "o": "m 605 352 q 530 93 605 199 q 302 -24 446 -24 q 74 93 158 -24 q 0 352 0 199 l 0 738 l 124 738 l 124 354 q 166 185 124 257 q 306 98 216 98 q 442 185 394 98 q 482 354 482 257 l 482 738 l 605 738 l 605 352 m 524 1039 l 339 819 l 257 819 l 389 1039 l 524 1039 "
    },
    "z": {
      "x_min": 0,
      "x_max": 602,
      "ha": 684,
      "o": "m 602 0 l 0 0 l 0 99 l 424 629 l 20 629 l 20 737 l 582 737 l 582 635 l 160 109 l 602 109 l 602 0 "
    },
    "â„¢": {
      "x_min": 0,
      "x_max": 875,
      "ha": 980,
      "o": "m 381 951 l 225 951 l 225 502 l 157 502 l 157 948 l 0 948 l 0 1011 l 381 1011 l 381 951 m 875 503 l 810 503 l 810 938 l 671 503 l 607 503 l 471 937 l 471 503 l 408 503 l 408 1011 l 506 1011 l 639 582 l 780 1011 l 875 1011 l 875 503 "
    },
    "Î®": {
      "x_min": 0,
      "x_max": 684,
      "ha": 794,
      "o": "m 684 -278 l 561 -278 l 561 453 q 529 587 561 535 q 416 650 491 650 q 266 579 331 650 q 202 420 202 509 l 202 0 l 79 0 l 79 489 q 70 588 79 562 q 0 644 54 644 l 0 741 q 66 755 36 755 q 155 721 121 755 q 196 630 189 687 q 291 726 230 692 q 426 761 353 761 q 609 692 534 761 q 684 516 684 624 l 684 -278 m 469 1040 l 284 819 l 203 819 l 335 1040 l 469 1040 "
    },
    "Î˜": {
      "x_min": 0,
      "x_max": 940,
      "ha": 1034,
      "o": "m 940 506 q 816 129 940 280 q 466 -32 684 -32 q 121 129 250 -32 q 0 506 0 280 q 121 882 0 731 q 466 1044 250 1044 q 814 882 681 1044 q 940 506 940 731 m 800 499 q 718 788 800 669 q 466 923 625 923 q 219 791 310 923 q 139 506 139 674 q 217 222 139 338 q 466 89 309 89 q 715 218 623 89 q 800 499 800 334 m 702 449 l 238 449 l 238 571 l 702 571 l 702 449 "
    },
    "Â®": {
      "x_min": -3,
      "x_max": 988,
      "ha": 1083,
      "o": "m 493 532 q 602 562 554 532 q 658 658 658 598 q 602 747 658 716 q 493 772 557 772 l 332 772 l 332 532 l 493 532 m 492 -7 q 120 151 258 -7 q -3 501 -3 294 q 120 851 -3 706 q 492 1011 258 1011 q 864 851 725 1011 q 988 501 988 708 q 865 151 988 292 q 492 -7 729 -7 m 492 60 q 813 197 695 60 q 921 501 921 322 q 814 805 921 681 q 492 944 695 944 q 171 805 291 944 q 64 501 64 680 q 170 197 64 320 q 492 60 288 60 m 773 145 l 664 145 q 641 316 641 183 q 517 449 641 449 l 332 449 l 332 145 l 237 145 l 237 854 l 508 854 q 674 808 609 854 q 751 658 751 755 q 725 563 751 606 q 654 497 699 519 q 736 331 736 472 q 773 163 736 190 l 773 145 "
    },
    "~": {
      "x_min": 0,
      "x_max": 817,
      "ha": 912,
      "o": "m 817 958 q 763 753 817 831 q 582 665 702 665 q 394 761 492 665 q 235 857 296 857 q 129 795 163 857 q 102 665 102 745 l 0 665 q 53 867 0 789 q 232 958 114 958 q 421 861 325 958 q 582 765 517 765 q 690 827 657 765 q 715 958 715 874 l 817 958 "
    },
    "Î•": {
      "x_min": 0,
      "x_max": 721.28125,
      "ha": 762,
      "o": "m 721 -1 l 0 -1 l 0 1012 l 710 1012 l 710 888 l 136 888 l 136 584 l 664 584 l 664 466 l 136 466 l 136 124 l 721 124 l 721 -1 "
    },
    "Â³": {
      "x_min": 0,
      "x_max": 441,
      "ha": 536,
      "o": "m 441 552 q 371 414 441 465 q 216 367 307 367 q 68 415 127 367 q 0 567 0 471 l 83 567 q 123 471 83 504 q 220 438 164 438 q 314 468 275 438 q 353 553 353 499 q 312 632 353 608 q 209 657 271 657 q 192 657 200 657 q 173 654 178 654 l 173 722 q 274 733 245 722 q 328 815 328 752 q 295 881 328 856 q 216 907 262 907 q 130 875 165 907 q 95 781 95 844 l 15 781 q 77 926 15 875 q 216 972 133 972 q 358 930 298 972 q 418 817 418 888 q 337 697 418 733 q 413 643 385 681 q 441 552 441 604 "
    },
    "[": {
      "x_min": 0,
      "x_max": 268,
      "ha": 363,
      "o": "m 268 -281 l 0 -281 l 0 1012 l 268 1012 l 268 919 l 121 919 l 121 -187 l 268 -187 l 268 -281 "
    },
    "L": {
      "x_min": 0,
      "x_max": 632.453125,
      "ha": 681,
      "o": "m 632 0 l 0 0 l 0 1012 l 137 1012 l 137 126 l 632 126 l 632 0 "
    },
    "Ïƒ": {
      "x_min": 0,
      "x_max": 788,
      "ha": 876,
      "o": "m 788 628 l 622 628 q 699 368 699 512 q 607 93 699 204 q 351 -25 509 -25 q 92 91 190 -25 q 0 368 0 201 q 92 644 0 533 q 349 761 190 761 q 472 750 390 761 q 597 739 553 739 l 788 739 l 788 628 m 353 84 q 519 180 458 84 q 573 374 573 262 q 517 566 573 490 q 345 651 454 651 q 183 559 242 651 q 132 368 132 479 q 185 175 132 254 q 353 84 246 84 "
    },
    "Î¶": {
      "x_min": 0,
      "x_max": 561,
      "ha": 629,
      "o": "m 561 -40 q 541 -162 561 -97 q 499 -278 532 -193 l 392 -278 q 431 -187 419 -219 q 452 -90 452 -132 q 370 -14 452 -14 q 106 45 193 -14 q 0 289 0 117 q 106 631 0 462 q 345 901 190 767 l 55 901 l 55 1012 l 550 1012 l 550 924 q 255 669 374 831 q 125 301 125 489 q 238 117 125 149 q 447 98 343 108 q 561 -40 561 80 "
    },
    "Î¸": {
      "x_min": 0,
      "x_max": 660,
      "ha": 762,
      "o": "m 660 496 q 589 160 660 304 q 329 -26 498 -26 q 72 153 161 -26 q 0 484 0 296 q 70 840 0 683 q 336 1044 163 1044 q 593 844 506 1044 q 660 496 660 691 m 535 579 q 488 798 535 691 q 329 935 428 935 q 175 798 232 935 q 124 579 134 701 l 535 579 m 535 475 l 124 475 q 167 233 124 348 q 332 80 226 80 q 494 233 438 80 q 535 475 535 346 "
    },
    "ÎŸ": {
      "x_min": 0,
      "x_max": 939,
      "ha": 1033,
      "o": "m 476 1042 q 817 883 689 1042 q 939 511 939 735 q 817 136 939 287 q 472 -26 687 -26 q 123 130 255 -26 q 0 504 0 279 q 125 880 0 729 q 476 1042 258 1042 m 471 98 q 717 225 626 98 q 799 504 799 340 q 718 783 799 670 q 471 913 627 913 q 222 785 314 913 q 139 504 139 671 q 222 224 139 339 q 471 98 313 98 "
    },
    "Î“": {
      "x_min": 0,
      "x_max": 690.984375,
      "ha": 733,
      "o": "m 690 886 l 137 886 l 137 0 l 0 0 l 0 1012 l 690 1012 l 690 886 "
    },
    " ": {
      "x_min": 0,
      "x_max": 0,
      "ha": 368
    },
    "%": {
      "x_min": -3,
      "x_max": 1067,
      "ha": 1162,
      "o": "m 828 0 q 650 76 717 0 q 591 244 591 145 q 649 412 591 344 q 829 489 714 489 q 1007 412 940 489 q 1067 244 1067 343 q 1008 76 1067 144 q 828 0 943 0 m 827 103 q 926 143 891 103 q 961 243 961 184 q 928 340 961 301 q 827 385 891 385 q 730 342 766 385 q 694 243 694 300 q 726 147 694 186 q 827 103 764 103 m 871 986 l 278 -26 l 195 -26 l 788 986 l 871 986 m 236 468 q 56 545 123 468 q -3 714 -3 615 q 55 881 -3 813 q 233 958 122 958 q 413 881 346 958 q 473 712 473 813 q 414 544 473 612 q 236 468 349 468 m 236 855 q 134 811 171 855 q 98 710 98 768 q 133 612 98 653 q 235 572 168 572 q 337 614 300 572 q 374 713 374 656 q 340 810 374 771 q 236 855 301 855 "
    },
    "P": {
      "x_min": 0,
      "x_max": 712,
      "ha": 790,
      "o": "m 416 1012 q 628 930 544 1012 q 712 719 712 849 q 624 506 712 587 q 405 426 537 426 l 137 426 l 137 0 l 0 0 l 0 1012 l 416 1012 m 371 888 l 137 888 l 137 548 l 364 548 q 511 589 450 548 q 581 720 581 636 q 517 844 581 800 q 371 888 454 888 "
    },
    "Îˆ": {
      "x_min": 0.171875,
      "x_max": 1056.28125,
      "ha": 1096,
      "o": "m 1056 -1 l 335 -1 l 335 1012 l 1045 1012 l 1045 888 l 471 888 l 471 584 l 999 584 l 999 466 l 471 466 l 471 124 l 1056 124 l 1056 -1 m 272 1039 l 81 799 l 0 799 l 137 1039 l 272 1039 "
    },
    "Î": {
      "x_min": -0.328125,
      "x_max": 1113,
      "ha": 1210,
      "o": "m 1113 1 l 707 1 l 707 125 q 892 310 824 195 q 961 559 961 424 q 875 813 961 710 q 638 923 783 923 q 402 821 491 923 q 314 569 314 720 q 382 317 314 434 q 567 125 450 201 l 567 1 l 163 1 l 163 125 l 376 125 q 230 328 282 211 q 178 573 178 445 q 305 912 178 782 q 638 1042 432 1042 q 969 910 841 1042 q 1097 570 1097 778 q 1044 327 1097 444 q 898 125 992 211 l 1113 125 l 1113 1 m 271 1041 l 81 801 l 0 801 l 137 1041 l 271 1041 "
    },
    "_": {
      "x_min": 0,
      "x_max": 691,
      "ha": 786,
      "o": "m 691 -333 l 0 -333 l 0 -233 l 691 -233 l 691 -333 "
    },
    "Îª": {
      "x_min": -107,
      "x_max": 241,
      "ha": 270,
      "o": "m 241 1046 l 116 1046 l 116 1189 l 241 1189 l 241 1046 m 18 1046 l -107 1046 l -107 1189 l 18 1189 l 18 1046 m 134 0 l 1 0 l 1 1012 l 134 1012 l 134 0 "
    },
    "+": {
      "x_min": 23,
      "x_max": 753,
      "ha": 776,
      "o": "m 753 372 l 435 372 l 435 0 l 340 0 l 340 372 l 23 372 l 23 468 l 339 468 l 339 840 l 434 840 l 434 468 l 753 468 l 753 372 "
    },
    "Â½": {
      "x_min": 0.171875,
      "x_max": 1031,
      "ha": 1126,
      "o": "m 1031 0 l 613 0 q 698 179 613 108 q 860 277 708 187 q 947 385 947 329 q 913 457 947 430 q 833 484 879 484 q 744 450 782 484 q 707 353 707 416 l 628 353 q 693 502 628 448 q 834 551 751 551 q 967 509 913 551 q 1029 386 1029 462 q 956 251 1029 301 q 812 179 884 215 q 702 68 725 133 l 1031 68 l 1031 0 m 818 984 l 210 -27 l 128 -27 l 735 984 l 818 984 m 219 422 l 139 422 l 139 811 l 0 811 l 0 867 q 102 889 61 867 q 160 973 153 916 l 219 973 l 219 422 "
    },
    "Î¡": {
      "x_min": 0,
      "x_max": 706,
      "ha": 767,
      "o": "m 416 1012 q 625 932 544 1012 q 706 723 706 852 q 621 508 706 591 q 405 426 536 426 l 137 426 l 137 0 l 0 0 l 0 1012 l 416 1012 m 371 888 l 137 888 l 137 548 l 364 548 q 511 589 450 548 q 581 720 581 636 q 517 844 581 800 q 371 888 454 888 "
    },
    "'": {
      "x_min": 0,
      "x_max": 136,
      "ha": 231,
      "o": "m 136 851 q 99 737 136 784 q 0 670 63 691 l 0 734 q 58 787 42 741 q 71 872 71 820 l 0 872 l 0 1012 l 136 1012 l 136 851 "
    },
    "Âª": {
      "x_min": 0,
      "x_max": 343,
      "ha": 389,
      "o": "m 343 624 q 302 616 322 616 q 263 630 277 616 q 245 672 249 645 q 189 627 223 643 q 115 612 155 612 q 32 640 64 612 q 0 721 0 668 q 71 825 0 799 q 239 865 155 844 l 242 886 q 216 933 242 915 q 159 952 190 952 q 103 933 126 952 q 75 881 79 914 l 16 881 q 59 975 16 940 q 163 1010 102 1010 q 261 978 220 1010 q 302 890 302 947 l 302 706 q 305 679 302 688 q 325 670 308 670 l 343 670 l 343 624 m 241 756 l 241 810 q 133 789 171 797 q 63 725 63 772 q 82 681 63 696 q 129 666 101 666 q 202 689 170 666 q 241 756 241 717 "
    },
    "Î…": {
      "x_min": 0,
      "x_max": 441,
      "ha": 542,
      "o": "m 441 800 l 333 800 l 333 925 l 441 925 l 441 800 m 398 1040 l 208 800 l 126 800 l 263 1040 l 398 1040 m 108 800 l 0 800 l 0 925 l 108 925 l 108 800 "
    },
    "T": {
      "x_min": 0,
      "x_max": 762,
      "ha": 818,
      "o": "m 762 893 l 449 893 l 449 0 l 313 0 l 313 891 l 0 891 l 0 1012 l 762 1012 l 762 893 "
    },
    "Î¦": {
      "x_min": 0,
      "x_max": 897,
      "ha": 977,
      "o": "m 516 0 l 381 0 l 381 122 q 108 230 216 122 q 0 509 0 339 q 108 784 0 677 q 381 892 216 892 l 381 1012 l 516 1012 l 516 893 q 788 786 679 893 q 897 509 897 679 q 789 231 897 341 q 516 122 682 122 l 516 0 m 516 226 q 698 310 629 226 q 764 507 764 389 q 698 705 764 627 q 516 787 629 787 l 516 226 m 381 226 l 381 787 q 201 698 269 775 q 133 505 133 620 q 201 308 133 391 q 381 226 270 226 "
    },
    "â‹": {
      "x_min": 0,
      "x_max": 0,
      "ha": 681
    },
    "j": {
      "x_min": -76.046875,
      "x_max": 164,
      "ha": 342,
      "o": "m 164 870 l 41 870 l 41 1012 l 164 1012 l 164 870 m 164 -80 q 119 -231 164 -184 q -25 -278 75 -278 l -76 -278 l -76 -163 l -40 -163 q 26 -142 11 -163 q 41 -64 41 -121 l 41 736 l 164 736 l 164 -80 "
    },
    "Î£": {
      "x_min": 0,
      "x_max": 742,
      "ha": 803,
      "o": "m 742 -1 l 0 -1 l 0 105 l 388 522 l 22 903 l 22 1012 l 731 1012 l 731 888 l 205 888 l 555 522 l 183 124 l 742 124 l 742 -1 "
    },
    "1": {
      "x_min": 212.140625,
      "x_max": 563,
      "ha": 775,
      "o": "m 563 0 l 434 0 l 434 697 l 212 697 l 212 796 q 379 833 325 796 q 466 986 439 875 l 563 986 l 563 0 "
    },
    "â€º": {
      "x_min": 18,
      "x_max": 758,
      "ha": 776,
      "o": "m 758 376 l 18 41 l 18 149 l 617 420 l 18 691 l 18 798 l 758 465 l 758 376 "
    },
    "<": {
      "x_min": 18,
      "x_max": 758,
      "ha": 776,
      "o": "m 758 41 l 18 376 l 18 465 l 758 798 l 758 691 l 156 420 l 758 149 l 758 41 "
    },
    "Â£": {
      "x_min": 0,
      "x_max": 689.859375,
      "ha": 785,
      "o": "m 689 41 q 611 -10 650 5 q 532 -26 571 -26 q 352 15 491 -26 q 237 36 282 36 q 154 23 192 36 q 71 -20 116 11 l 5 76 q 122 196 88 150 q 171 332 171 263 q 143 444 171 384 l 0 444 l 0 513 l 105 513 q 42 735 42 623 q 118 929 42 854 q 350 1010 199 1010 q 567 936 477 1010 q 664 729 664 857 l 664 685 l 540 685 q 493 838 540 780 q 355 896 447 896 q 212 852 258 896 q 173 747 173 815 q 196 628 173 697 q 243 513 213 575 l 459 513 l 459 444 l 273 444 q 291 357 291 399 q 225 194 291 279 q 149 107 206 171 q 222 133 186 125 q 287 142 258 142 q 402 119 331 142 q 506 96 472 96 q 567 105 538 96 q 635 139 595 114 l 689 41 "
    },
    "t": {
      "x_min": 0,
      "x_max": 359.015625,
      "ha": 449,
      "o": "m 359 0 q 305 -5 331 -2 q 257 -8 278 -8 q 142 28 179 -8 q 106 143 106 64 l 106 638 l 0 638 l 0 738 l 106 738 l 106 944 l 227 944 l 227 738 l 359 738 l 359 638 l 227 638 l 227 185 q 242 121 227 140 q 300 101 258 101 q 337 104 323 101 q 359 107 352 107 l 359 0 "
    },
    "Â¬": {
      "x_min": 0.421875,
      "x_max": 692,
      "ha": 787,
      "o": "m 692 411 l 692 158 l 617 158 l 617 335 l 0 335 l 0 411 l 692 411 "
    },
    "Î»": {
      "x_min": -0.46875,
      "x_max": 735,
      "ha": 787,
      "o": "m 735 -6 q 665 -15 702 -15 q 527 59 579 -15 q 457 213 502 97 l 329 551 l 123 0 l 0 0 l 265 705 q 218 837 241 770 q 113 900 186 900 q 88 898 97 900 l 88 1004 q 149 1011 122 1011 q 292 938 239 1011 q 365 783 319 901 l 593 192 q 636 115 616 135 q 702 96 656 96 l 721 96 q 735 97 730 97 l 735 -6 "
    },
    "W": {
      "x_min": 0,
      "x_max": 1238.609375,
      "ha": 1324,
      "o": "m 1238 1012 l 975 0 l 842 0 l 615 835 l 397 0 l 259 0 l 0 1012 l 133 1012 l 336 202 l 545 1012 l 687 1012 l 903 208 l 1110 1012 l 1238 1012 "
    },
    ">": {
      "x_min": 18,
      "x_max": 758,
      "ha": 776,
      "o": "m 758 376 l 18 41 l 18 149 l 617 420 l 18 691 l 18 798 l 758 465 l 758 376 "
    },
    "v": {
      "x_min": 0,
      "x_max": 661.5,
      "ha": 746,
      "o": "m 661 737 l 396 0 l 266 0 l 0 737 l 130 737 l 333 147 l 530 737 l 661 737 "
    },
    "Ï„": {
      "x_min": 0,
      "x_max": 632,
      "ha": 689,
      "o": "m 632 628 l 374 628 l 374 179 q 380 120 374 137 q 427 92 393 92 q 464 94 438 92 q 494 97 491 97 l 494 0 q 444 -9 472 -5 q 393 -14 417 -14 q 271 67 301 -14 q 254 233 254 117 l 254 628 l 0 628 l 0 739 l 632 739 l 632 628 "
    },
    "Î¾": {
      "x_min": 0,
      "x_max": 613,
      "ha": 685,
      "o": "m 613 -37 q 597 -153 613 -96 q 552 -278 581 -211 l 445 -278 q 481 -183 476 -200 q 501 -83 501 -126 q 475 -24 501 -45 q 363 0 443 0 q 317 0 347 0 q 277 0 287 0 q 83 77 166 0 q 0 267 0 155 q 55 432 0 358 q 193 538 106 498 q 92 613 132 562 q 53 730 53 665 q 76 824 53 780 q 140 901 99 867 l 27 901 l 27 1012 l 565 1012 l 565 901 l 372 901 q 239 863 296 901 q 174 745 174 820 q 306 600 174 636 q 521 582 372 582 l 521 479 q 270 455 353 479 q 116 281 116 410 q 162 173 116 217 q 269 120 204 133 q 484 101 377 111 q 613 -37 613 76 "
    },
    "&": {
      "x_min": -3,
      "x_max": 876.25,
      "ha": 972,
      "o": "m 876 0 l 710 0 l 612 123 q 462 0 543 40 q 300 -42 382 -42 q 165 -7 227 -42 q 61 91 103 26 q 14 186 31 138 q -3 276 -3 234 q 54 433 -3 358 q 243 581 112 508 q 166 688 192 640 q 134 816 134 751 q 210 984 134 922 q 376 1040 278 1040 q 537 987 473 1040 q 610 823 610 927 q 552 666 610 738 q 423 556 506 608 l 610 326 q 637 406 627 361 q 650 493 641 426 l 766 493 q 690 229 766 352 l 876 0 m 494 818 q 459 907 494 876 q 376 939 424 939 q 287 907 324 939 q 250 818 250 875 q 284 713 250 766 q 356 627 307 677 q 468 729 438 681 q 494 818 494 770 m 545 209 l 308 499 q 175 395 218 449 q 132 283 132 341 q 143 222 132 252 q 179 158 155 191 q 326 79 236 79 q 545 209 439 79 "
    },
    "Î›": {
      "x_min": 0,
      "x_max": 845.25,
      "ha": 923,
      "o": "m 845 0 l 705 0 l 417 846 l 140 0 l 0 0 l 349 1012 l 491 1012 l 845 0 "
    },
    "I": {
      "x_min": 41,
      "x_max": 177,
      "ha": 287,
      "o": "m 177 0 l 41 0 l 41 1012 l 177 1012 l 177 0 "
    },
    "G": {
      "x_min": 0,
      "x_max": 903,
      "ha": 991,
      "o": "m 903 0 l 815 0 l 785 135 q 641 14 726 57 q 461 -28 556 -28 q 123 133 254 -28 q 0 499 0 284 q 122 881 0 731 q 476 1043 254 1043 q 749 956 634 1043 q 887 708 872 863 l 756 708 q 654 866 732 806 q 476 926 577 926 q 223 795 315 926 q 139 507 139 677 q 223 224 139 342 q 473 94 317 94 q 698 195 612 94 q 780 434 780 291 l 468 434 l 468 548 l 903 548 l 903 0 "
    },
    "Î°": {
      "x_min": 0,
      "x_max": 605,
      "ha": 711,
      "o": "m 513 799 l 405 799 l 405 924 l 513 924 l 513 799 m 180 799 l 72 799 l 72 924 l 180 924 l 180 799 m 605 352 q 530 93 605 199 q 302 -24 446 -24 q 74 93 158 -24 q 0 352 0 199 l 0 738 l 124 738 l 124 354 q 166 185 124 257 q 306 98 216 98 q 442 185 394 98 q 482 354 482 257 l 482 738 l 605 738 l 605 352 m 479 1039 l 293 819 l 212 819 l 344 1039 l 479 1039 "
    },
    "`": {
      "x_min": 0,
      "x_max": 136.046875,
      "ha": 231,
      "o": "m 136 699 l 0 699 l 0 861 q 35 975 0 929 q 136 1041 70 1020 l 136 977 q 80 931 93 969 q 68 839 68 893 l 136 839 l 136 699 "
    },
    "Â·": {
      "x_min": 0,
      "x_max": 139,
      "ha": 234,
      "o": "m 139 585 l 0 585 l 0 738 l 139 738 l 139 585 "
    },
    "Î¥": {
      "x_min": 0.171875,
      "x_max": 802.75,
      "ha": 871,
      "o": "m 802 1012 l 472 416 l 472 0 l 335 0 l 335 416 l 0 1012 l 137 1012 l 402 533 l 665 1012 l 802 1012 "
    },
    "r": {
      "x_min": 0,
      "x_max": 348.65625,
      "ha": 424,
      "o": "m 348 621 l 336 621 q 175 569 231 621 q 120 411 120 518 l 120 0 l 0 0 l 0 737 l 114 737 l 114 604 q 200 719 143 685 q 348 753 257 753 l 348 621 "
    },
    "x": {
      "x_min": 0,
      "x_max": 661.5,
      "ha": 749,
      "o": "m 661 0 l 515 0 l 325 285 l 141 0 l 0 0 l 251 378 l 12 737 l 155 737 l 330 473 l 506 737 l 647 737 l 404 380 l 661 0 "
    },
    "Î¼": {
      "x_min": 0,
      "x_max": 683.21875,
      "ha": 732,
      "o": "m 683 -4 q 616 -14 645 -14 q 488 97 503 -14 q 414 8 461 41 q 307 -24 367 -24 q 203 3 253 -24 q 117 80 153 31 l 117 -278 l 0 -278 l 0 737 l 121 737 l 121 343 q 161 172 121 245 q 302 82 212 82 q 443 177 394 82 q 483 358 483 254 l 483 737 l 604 737 l 604 214 q 610 136 604 159 q 660 92 624 92 q 683 96 670 92 l 683 -4 "
    },
    "h": {
      "x_min": 0,
      "x_max": 603,
      "ha": 709,
      "o": "m 603 472 l 603 0 l 480 0 l 480 454 q 447 590 480 535 q 331 654 407 654 q 183 588 246 654 q 120 436 120 522 l 120 0 l 0 0 l 0 1012 l 120 1012 l 120 633 q 214 727 147 694 q 355 760 281 760 q 541 676 475 760 q 603 472 603 600 "
    },
    ".": {
      "x_min": 0,
      "x_max": 139,
      "ha": 234,
      "o": "m 139 0 l 0 0 l 0 151 l 139 151 l 139 0 "
    },
    "Ï†": {
      "x_min": -1,
      "x_max": 860,
      "ha": 954,
      "o": "m 486 -279 l 370 -279 l 370 -17 q 99 88 200 -17 q -1 367 -1 193 q 68 625 -1 510 q 278 757 148 757 l 278 646 q 165 536 205 625 q 131 372 131 461 q 189 177 131 253 q 370 92 254 92 l 370 757 q 436 763 417 763 q 466 763 454 763 q 749 657 639 763 q 860 376 860 552 q 756 95 860 208 q 486 -17 652 -17 l 486 -279 m 486 92 l 503 92 q 673 182 611 92 q 731 379 731 264 q 677 567 731 490 q 511 656 616 656 l 486 656 l 486 92 "
    },
    ";": {
      "x_min": 0,
      "x_max": 139,
      "ha": 234,
      "o": "m 139 585 l 0 585 l 0 738 l 139 738 l 139 585 m 139 -12 q 102 -133 139 -83 q 0 -206 66 -183 l 0 -138 q 57 -83 42 -123 q 67 0 67 -56 l 0 0 l 0 151 l 139 151 l 139 -12 "
    },
    "f": {
      "x_min": 0,
      "x_max": 370,
      "ha": 463,
      "o": "m 370 637 l 241 637 l 241 0 l 118 0 l 118 637 l 0 637 l 0 737 l 118 737 q 134 934 118 887 q 284 1026 167 1026 q 313 1026 299 1026 q 370 1020 327 1026 l 370 908 q 317 918 338 918 q 251 870 268 918 q 241 780 241 840 l 241 737 l 370 737 l 370 637 "
    },
    "â€œ": {
      "x_min": 1,
      "x_max": 341,
      "ha": 444,
      "o": "m 137 669 l 1 669 l 1 829 q 37 943 1 896 q 137 1010 73 989 l 137 946 q 80 899 94 939 q 66 809 66 860 l 137 809 l 137 669 m 341 669 l 205 669 l 205 829 q 241 943 205 896 q 341 1010 277 989 l 341 946 q 284 899 298 939 q 270 809 270 860 l 341 809 l 341 669 "
    },
    "A": {
      "x_min": 0,
      "x_max": 888.8125,
      "ha": 988,
      "o": "m 888 0 l 741 0 l 637 302 l 246 302 l 140 0 l 0 0 l 368 1012 l 518 1012 l 888 0 m 597 420 l 443 866 l 287 420 l 597 420 "
    },
    "6": {
      "x_min": 52,
      "x_max": 724,
      "ha": 776,
      "o": "m 724 311 q 620 62 724 162 q 392 -31 524 -31 q 159 78 251 -31 q 52 439 52 206 q 174 858 52 711 q 433 985 279 985 q 630 911 548 985 q 718 713 718 832 l 589 713 q 533 829 582 785 q 417 874 484 874 q 262 792 325 874 q 189 516 189 696 q 295 596 235 569 q 419 623 355 623 q 630 539 541 623 q 724 311 724 450 m 591 298 q 529 460 591 399 q 396 515 475 515 q 262 460 317 515 q 202 299 202 401 q 263 137 202 198 q 397 83 318 83 q 531 137 476 83 q 591 298 591 196 "
    },
    "â€˜": {
      "x_min": 1,
      "x_max": 137,
      "ha": 231,
      "o": "m 137 669 l 1 669 l 1 829 q 37 943 1 896 q 137 1010 73 989 l 137 946 q 80 899 94 939 q 66 809 66 860 l 137 809 l 137 669 "
    },
    "ÏŠ": {
      "x_min": -68,
      "x_max": 278.34375,
      "ha": 355,
      "o": "m 278 800 l 170 800 l 170 925 l 278 925 l 278 800 m 40 800 l -68 800 l -68 925 l 40 925 l 40 800 m 278 3 q 228 -10 253 -5 q 178 -15 203 -15 q 83 26 117 -15 q 41 200 41 79 l 41 738 l 164 738 l 164 215 q 168 142 164 157 q 221 102 179 102 q 243 103 234 102 q 278 113 252 104 l 278 3 "
    },
    "Ï€": {
      "x_min": 0,
      "x_max": 758.28125,
      "ha": 840,
      "o": "m 758 -10 l 694 -10 q 565 40 596 -10 q 542 174 542 77 l 542 226 l 542 626 l 218 626 l 218 0 l 95 0 l 95 626 l 0 626 l 0 737 l 758 737 l 758 626 l 663 626 l 663 171 q 682 103 663 117 q 758 90 701 90 l 758 -10 "
    },
    "Î¬": {
      "x_min": 0,
      "x_max": 750,
      "ha": 792,
      "o": "m 750 -3 q 684 -13 711 -13 q 552 97 574 -13 q 456 7 514 40 q 330 -26 399 -26 q 87 98 182 -26 q 0 369 0 212 q 87 637 0 525 q 330 760 181 760 q 456 728 399 760 q 551 637 513 696 l 551 739 l 671 739 l 671 222 q 679 142 671 168 q 733 95 694 95 q 750 96 744 95 l 750 -3 m 572 371 q 520 562 572 484 q 352 653 460 653 q 187 566 249 653 q 132 379 132 489 q 182 180 132 261 q 351 83 242 83 q 517 176 456 83 q 572 371 572 260 m 592 1040 l 407 820 l 325 820 l 457 1040 l 592 1040 "
    },
    "O": {
      "x_min": 0,
      "x_max": 939,
      "ha": 1036,
      "o": "m 476 1042 q 817 883 688 1042 q 939 512 939 735 q 817 136 939 287 q 472 -26 688 -26 q 123 130 255 -26 q 0 504 0 279 q 125 880 0 729 q 476 1042 258 1042 m 471 98 q 717 225 626 98 q 799 504 799 340 q 718 783 799 670 q 471 913 627 913 q 222 785 314 913 q 139 504 139 671 q 222 224 139 339 q 471 98 313 98 "
    },
    "n": {
      "x_min": 0,
      "x_max": 603,
      "ha": 709,
      "o": "m 603 463 l 603 0 l 480 0 l 480 454 q 444 592 480 537 q 324 656 402 656 q 174 585 235 656 q 114 421 114 514 l 114 0 l 0 0 l 0 739 l 114 739 l 114 630 q 213 728 146 693 q 352 764 280 764 q 541 675 474 764 q 603 463 603 593 "
    },
    "3": {
      "x_min": 53,
      "x_max": 723,
      "ha": 776,
      "o": "m 723 284 q 623 55 723 141 q 392 -25 530 -25 q 153 52 243 -25 q 53 308 53 140 l 181 308 q 240 147 181 202 q 388 96 296 96 q 529 140 475 96 q 591 280 591 190 q 500 428 591 390 q 317 454 442 454 l 317 565 q 477 584 432 565 q 554 719 554 617 q 505 835 554 791 q 388 879 457 879 q 250 824 301 879 q 199 661 199 769 l 76 661 q 163 909 76 821 q 379 991 244 991 q 591 920 503 991 q 687 723 687 843 q 655 607 687 656 q 566 524 624 558 q 682 434 642 499 q 723 284 723 369 "
    },
    "9": {
      "x_min": 52,
      "x_max": 724,
      "ha": 776,
      "o": "m 724 524 q 607 94 724 241 q 355 -32 506 -32 q 147 47 237 -32 q 58 243 58 126 l 187 243 q 241 129 187 176 q 366 82 295 82 q 515 161 457 82 q 585 439 585 255 q 356 333 491 333 q 128 431 212 333 q 52 649 52 520 q 132 880 52 785 q 375 986 222 986 q 646 841 555 986 q 724 524 724 719 m 378 448 q 523 513 469 448 q 572 658 572 573 q 523 805 572 744 q 378 874 469 874 q 234 809 285 874 q 184 658 184 745 q 231 513 184 571 q 378 448 284 448 "
    },
    "l": {
      "x_min": 41,
      "x_max": 164,
      "ha": 274,
      "o": "m 164 0 l 41 0 l 41 1012 l 164 1012 l 164 0 "
    },
    "Â¤": {
      "x_min": 39.46875,
      "x_max": 714.15625,
      "ha": 808,
      "o": "m 714 304 l 636 225 l 501 362 q 375 332 448 332 q 251 362 304 332 l 117 225 l 39 304 l 174 441 q 147 552 147 492 q 180 673 147 620 l 39 818 l 117 897 l 262 748 q 314 767 285 760 q 376 774 344 774 q 438 767 409 774 q 490 748 467 760 l 636 897 l 714 818 l 573 675 q 599 618 591 648 q 608 552 608 587 q 579 441 608 491 l 714 304 m 376 682 q 274 643 311 682 q 238 551 238 604 q 274 460 238 498 q 375 422 310 422 q 477 460 440 422 q 514 552 514 499 q 480 641 514 605 q 376 682 441 682 "
    },
    "Îº": {
      "x_min": 0,
      "x_max": 619.171875,
      "ha": 665,
      "o": "m 619 0 l 472 0 l 220 384 l 121 288 l 121 0 l 0 0 l 0 737 l 121 737 l 121 445 l 424 737 l 583 737 l 306 466 l 619 0 "
    },
    "4": {
      "x_min": 48,
      "x_max": 729,
      "ha": 777,
      "o": "m 729 243 l 591 243 l 591 0 l 467 0 l 467 243 l 48 243 l 48 367 l 465 958 l 591 958 l 591 354 l 729 354 l 729 243 m 467 354 l 467 794 l 160 354 l 467 354 "
    },
    "p": {
      "x_min": 0,
      "x_max": 671,
      "ha": 770,
      "o": "m 671 363 q 586 95 671 205 q 342 -24 494 -24 q 118 88 201 -24 l 118 -278 l 0 -278 l 0 739 l 118 739 l 118 633 q 215 726 156 691 q 344 761 274 761 q 586 635 494 761 q 671 363 671 522 m 546 370 q 491 560 546 481 q 323 651 428 651 q 159 559 219 651 q 106 366 106 478 q 159 176 106 253 q 326 85 220 85 q 492 177 432 85 q 546 370 546 258 "
    },
    "â€¡": {
      "x_min": 0,
      "x_max": 762,
      "ha": 818,
      "o": "m 449 238 l 449 0 l 313 0 l 313 236 l 0 236 l 0 360 l 313 360 l 313 682 l 0 682 l 0 804 l 313 804 l 313 1014 l 449 1014 l 449 804 l 762 804 l 762 682 l 449 682 l 449 360 l 762 360 l 762 238 l 449 238 "
    },
    "Ïˆ": {
      "x_min": 0,
      "x_max": 792,
      "ha": 889,
      "o": "m 456 -278 l 335 -278 l 335 -15 q 85 101 177 -15 q 0 377 0 209 l 0 739 l 131 739 l 131 379 q 179 194 131 274 q 335 97 238 97 l 335 922 l 456 922 l 456 97 q 610 194 552 97 q 661 381 661 277 l 661 741 l 792 741 l 792 380 q 706 103 792 212 q 456 -12 614 -12 l 456 -278 "
    },
    "Î·": {
      "x_min": 0,
      "x_max": 684,
      "ha": 794,
      "o": "m 684 -278 l 561 -278 l 561 453 q 529 587 561 535 q 416 650 491 650 q 266 579 331 650 q 202 420 202 509 l 202 0 l 79 0 l 79 489 q 70 588 79 562 q 0 644 54 644 l 0 741 q 66 755 36 755 q 155 721 121 755 q 196 630 189 687 q 291 726 230 692 q 426 761 353 761 q 609 692 534 761 q 684 516 684 624 l 684 -278 "
    }
  },
  "cssFontWeight": "normal",
  "ascender": 1189,
  "underlinePosition": -100,
  "cssFontStyle": "normal",
  "boundingBox": {
    "yMin": -333,
    "xMin": -109,
    "yMax": 1189,
    "xMax": 1638
  },
  "resolution": 1000,
  "original_font_information": {
    "postscript_name": "Helvetiker-Regular",
    "version_string": "Version 1.00 2004 initial release",
    "vendor_url": "http://www.magenta.gr/",
    "full_font_name": "Helvetiker",
    "font_family_name": "Helvetiker",
    "copyright": "Copyright (c) Îœagenta ltd, 2004",
    "description": "",
    "trademark": "",
    "designer": "",
    "designer_url": "",
    "unique_font_identifier": "Îœagenta ltd:Helvetiker:22-10-104",
    "license_url": "http://www.ellak.gr/fonts/MgOpen/license.html",
    "license_description": "Copyright (c) 2004 by MAGENTA Ltd. All Rights Reserved.\r\n\r\nPermission is hereby granted, free of charge, to any person obtaining a copy of the fonts accompanying this license (\"Fonts\") and associated documentation files (the \"Font Software\"), to reproduce and distribute the Font Software, including without limitation the rights to use, copy, merge, publish, distribute, and/or sell copies of the Font Software, and to permit persons to whom the Font Software is furnished to do so, subject to the following conditions: \r\n\r\nThe above copyright and this permission notice shall be included in all copies of one or more of the Font Software typefaces.\r\n\r\nThe Font Software may be modified, altered, or added to, and in particular the designs of glyphs or characters in the Fonts may be modified and additional glyphs or characters may be added to the Fonts, only if the fonts are renamed to names not containing the word \"MgOpen\", or if the modifications are accepted for inclusion in the Font Software itself by the each appointed Administrator.\r\n\r\nThis License becomes null and void to the extent applicable to Fonts or Font Software that has been modified and is distributed under the \"MgOpen\" name.\r\n\r\nThe Font Software may be sold as part of a larger software package but no copy of one or more of the Font Software typefaces may be sold by itself. \r\n\r\nTHE FONT SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO ANY WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF COPYRIGHT, PATENT, TRADEMARK, OR OTHER RIGHT. IN NO EVENT SHALL MAGENTA OR PERSONS OR BODIES IN CHARGE OF ADMINISTRATION AND MAINTENANCE OF THE FONT SOFTWARE BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, INCLUDING ANY GENERAL, SPECIAL, INDIRECT, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF THE USE OR INABILITY TO USE THE FONT SOFTWARE OR FROM OTHER DEALINGS IN THE FONT SOFTWARE.",
    "manufacturer_name": "Îœagenta ltd",
    "font_sub_family_name": "Regular"
  },
  "descender": -334,
  "familyName": "Helvetiker",
  "lineHeight": 1522,
  "underlineThickness": 50
});
if (_typeface_js && _typeface_js.loadFace) _typeface_js.loadFace({
  "glyphs": {
    "á»£": {
      "x_min": 53,
      "x_max": 731,
      "ha": 731,
      "o": "m 528 308 q 510 410 528 362 q 466 496 493 459 q 404 554 438 533 q 336 576 369 576 q 251 556 285 576 q 195 502 216 537 q 165 421 174 468 q 156 320 156 375 q 174 218 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 156 q 528 308 528 250 m 731 706 q 723 669 731 690 q 699 625 716 648 q 652 576 681 602 q 578 526 622 551 q 617 436 604 486 q 631 329 631 386 q 619 240 631 283 q 588 158 608 196 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 471 74 432 q 141 543 114 510 q 202 600 169 576 q 275 637 236 623 q 355 651 314 651 q 465 628 415 651 q 550 564 514 605 q 601 611 585 588 q 617 651 617 634 q 606 688 617 671 q 578 722 596 706 l 696 776 q 721 743 711 761 q 731 706 731 726 m 422 -189 q 416 -225 422 -208 q 399 -254 410 -242 q 375 -274 389 -267 q 344 -282 361 -282 q 301 -265 314 -282 q 289 -217 289 -248 q 294 -181 289 -198 q 311 -151 300 -164 q 335 -132 322 -139 q 365 -125 349 -125 q 422 -189 422 -125 "
    },
    "áº¨": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 351 839 q 390 869 367 856 q 430 893 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 608 957 q 595 938 600 944 q 579 927 590 932 l 397 1068 l 216 927 q 208 932 211 929 q 201 938 204 935 q 195 946 198 942 q 187 957 191 951 l 362 1167 l 434 1167 l 608 957 m 510 1392 q 497 1358 510 1373 q 469 1330 485 1344 q 436 1305 452 1317 q 413 1281 421 1293 q 412 1255 406 1268 q 447 1224 419 1241 q 427 1216 437 1218 q 406 1213 416 1215 q 361 1241 375 1228 q 344 1264 347 1253 q 347 1285 341 1275 q 365 1303 353 1294 q 390 1321 376 1312 q 417 1339 404 1330 q 437 1359 429 1349 q 446 1382 446 1370 q 433 1421 446 1408 q 400 1434 420 1434 q 382 1430 390 1434 q 367 1420 373 1426 q 358 1406 361 1414 q 355 1392 355 1399 q 356 1384 355 1388 q 360 1377 358 1381 q 346 1373 355 1375 q 328 1369 338 1371 q 310 1365 319 1366 q 294 1362 300 1363 l 287 1370 l 287 1379 q 299 1411 287 1395 q 331 1439 312 1426 q 375 1459 351 1451 q 422 1467 399 1467 q 487 1446 464 1467 q 510 1392 510 1426 "
    },
    "Ç»": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 216 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 280 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 165 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 383 843 q 379 874 383 859 q 367 900 375 889 q 349 918 360 911 q 325 925 339 925 q 299 919 311 925 q 276 903 286 913 q 261 877 267 892 q 255 844 255 862 q 259 814 255 828 q 270 788 263 799 q 288 770 277 777 q 312 764 299 764 q 338 769 325 764 q 361 784 351 774 q 377 808 371 794 q 383 843 383 823 m 443 870 q 430 807 443 836 q 396 757 417 778 q 349 723 375 735 q 298 712 323 712 q 256 720 275 712 q 224 742 237 728 q 202 775 210 756 q 195 816 195 794 q 208 880 195 851 q 241 931 221 910 q 287 965 262 953 q 339 977 313 977 q 380 968 361 977 q 413 945 399 960 q 435 911 427 930 q 443 870 443 892 m 290 1008 q 268 1016 278 1010 q 252 1027 259 1021 l 408 1308 q 429 1306 416 1307 q 456 1302 442 1304 q 484 1297 470 1300 q 504 1293 497 1295 l 519 1266 l 290 1008 "
    },
    "Ê‰": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 314 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 281 l 213 281 l 213 226 q 219 146 213 178 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 m 678 281 l 608 281 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 q 662 25 685 39 q 620 0 639 11 q 585 -15 600 -9 q 563 -21 571 -21 q 524 10 539 -21 q 510 112 510 42 q 432 44 466 70 q 369 3 398 18 q 316 -15 340 -10 q 270 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -3 q 125 84 138 43 q 113 188 113 125 l 113 281 l 42 281 l 27 295 q 33 318 30 306 q 42 342 37 331 l 113 342 l 113 481 q 111 532 113 514 q 100 559 109 550 q 75 571 92 568 q 27 578 57 575 l 27 605 q 72 612 51 608 q 112 622 93 616 q 152 634 132 627 q 194 650 172 641 l 213 624 l 213 342 l 508 342 l 508 482 q 505 531 508 512 q 492 559 502 549 q 463 573 482 569 q 414 578 445 578 l 414 605 q 506 624 462 611 q 588 650 550 637 l 608 624 l 608 342 l 678 342 l 693 325 l 678 281 "
    },
    "á»”": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 619 957 q 606 938 611 944 q 590 927 602 932 l 408 1068 l 228 927 q 219 932 223 929 q 212 938 215 935 q 206 946 209 942 q 198 957 203 951 l 373 1167 l 445 1167 l 619 957 m 521 1392 q 508 1358 521 1373 q 480 1330 496 1344 q 447 1305 463 1317 q 424 1281 432 1293 q 423 1255 417 1268 q 458 1224 430 1241 q 438 1216 448 1218 q 417 1213 427 1215 q 372 1241 386 1228 q 355 1264 358 1253 q 358 1285 352 1275 q 376 1303 364 1294 q 401 1321 387 1312 q 428 1339 415 1330 q 448 1359 440 1349 q 457 1382 457 1370 q 444 1421 457 1408 q 411 1434 431 1434 q 393 1430 401 1434 q 378 1420 384 1426 q 369 1406 372 1414 q 366 1392 366 1399 q 367 1384 366 1388 q 371 1377 369 1381 q 357 1373 366 1375 q 339 1369 349 1371 q 321 1365 330 1366 q 305 1362 311 1363 l 298 1370 l 298 1379 q 310 1411 298 1395 q 342 1439 323 1426 q 386 1459 362 1451 q 433 1467 410 1467 q 498 1446 475 1467 q 521 1392 521 1426 "
    },
    "á»ª": {
      "x_min": 33,
      "x_max": 930,
      "ha": 930,
      "o": "m 930 944 q 921 904 930 927 q 892 856 913 882 q 837 803 872 830 q 751 749 803 775 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 804 855 q 816 889 816 873 q 806 926 816 909 q 777 960 796 944 l 895 1014 q 920 981 910 999 q 930 944 930 964 m 567 958 q 553 938 558 945 q 537 927 548 932 l 226 1092 l 231 1122 q 246 1135 236 1128 q 268 1148 256 1141 q 291 1162 280 1156 q 310 1173 302 1168 l 567 958 "
    },
    "Ì‚": {
      "x_min": -552.28125,
      "x_max": -131.59375,
      "ha": 0,
      "o": "m -131 740 q -144 722 -139 728 q -160 710 -148 715 l -342 892 l -522 710 q -531 715 -527 713 q -538 722 -535 718 q -544 730 -540 725 q -552 740 -547 734 l -376 998 l -305 998 l -131 740 "
    },
    "Ã": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 839 q 390 869 367 855 q 430 893 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 310 927 q 293 938 298 931 q 280 957 288 944 l 537 1172 q 555 1162 544 1168 q 579 1148 567 1155 q 600 1134 590 1141 q 616 1122 610 1127 l 622 1091 l 310 927 "
    },
    "á¹‘": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 586 868 q 581 851 585 861 q 574 831 578 841 q 566 812 570 822 q 560 797 563 803 l 152 797 l 137 813 q 142 830 138 821 q 149 850 145 840 q 157 869 153 859 q 164 885 161 878 l 571 885 l 586 868 m 431 975 q 413 962 422 967 q 393 954 405 957 l 160 1209 l 174 1237 q 194 1242 181 1239 q 221 1247 207 1244 q 249 1251 235 1249 q 271 1253 263 1253 l 431 975 "
    },
    "È®": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 476 1045 q 470 1009 476 1026 q 453 980 464 992 q 429 960 443 967 q 398 953 415 953 q 355 969 368 953 q 343 1017 343 986 q 348 1053 343 1036 q 365 1083 354 1070 q 389 1102 376 1095 q 419 1110 403 1110 q 476 1045 476 1110 "
    },
    "Ä¥": {
      "x_min": 37,
      "x_max": 732,
      "ha": 752,
      "o": "m 446 0 l 446 29 q 516 51 493 42 q 539 70 539 61 l 539 429 q 534 496 539 470 q 518 537 529 522 q 491 557 507 552 q 453 563 475 563 q 403 550 430 563 q 347 514 376 538 q 288 455 318 491 q 230 372 258 419 l 230 70 q 254 49 230 60 q 323 29 279 38 l 323 0 l 37 0 l 37 29 q 105 50 80 40 q 130 70 130 59 l 130 880 q 127 924 130 908 q 115 949 125 940 q 87 962 106 958 q 37 969 68 965 l 37 997 q 87 1007 64 1002 q 130 1019 110 1013 q 168 1033 150 1026 q 205 1051 186 1041 l 230 1028 l 230 464 q 293 541 258 507 q 363 600 328 576 q 432 638 398 625 q 492 651 466 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 70 q 659 52 639 61 q 732 29 679 42 l 732 0 l 446 0 m 594 1127 q 581 1108 586 1114 q 565 1096 577 1102 l 383 1237 l 203 1096 q 194 1102 198 1099 q 187 1108 190 1104 q 181 1116 184 1111 q 173 1127 178 1121 l 348 1337 l 420 1337 l 594 1127 "
    },
    "Â»": {
      "x_min": 93.171875,
      "x_max": 607,
      "ha": 661,
      "o": "m 607 291 l 367 11 l 339 31 l 492 316 l 339 598 l 367 619 l 607 341 l 607 291 m 361 291 l 121 11 l 93 31 l 246 316 l 93 598 l 121 619 l 361 341 l 361 291 "
    },
    "á¸º": {
      "x_min": 33,
      "x_max": 627.09375,
      "ha": 648,
      "o": "m 627 165 q 613 63 621 106 q 599 0 605 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 111 q 241 89 236 98 q 261 74 247 80 q 299 64 275 67 q 359 61 322 61 l 454 61 q 505 64 484 61 q 542 79 526 67 q 571 114 558 91 q 598 177 584 137 l 627 165 m 554 -155 q 549 -172 553 -162 q 542 -192 546 -182 q 535 -211 538 -201 q 528 -227 531 -220 l 120 -227 l 105 -210 q 110 -193 107 -202 q 117 -173 113 -183 q 125 -154 121 -164 q 133 -139 129 -145 l 540 -139 l 554 -155 "
    },
    "âˆ†": {
      "x_min": 39.21875,
      "x_max": 727.078125,
      "ha": 780,
      "o": "m 585 112 q 501 335 542 226 q 465 432 484 382 q 429 531 447 482 q 394 627 410 581 q 363 711 377 673 q 330 620 348 668 q 295 520 313 571 q 260 421 277 470 q 225 326 242 371 q 148 112 187 218 q 142 92 143 99 q 148 82 141 85 q 171 79 154 79 q 220 79 188 79 l 511 79 q 561 79 544 79 q 585 82 578 79 q 591 92 592 85 q 585 112 590 99 m 713 0 l 49 0 l 39 34 q 154 348 99 197 q 202 480 177 412 q 249 613 226 548 q 293 736 273 678 q 329 839 314 795 q 367 870 344 856 q 406 893 389 883 q 446 779 424 843 q 495 646 469 716 q 548 504 521 576 q 601 364 575 432 q 727 34 661 205 l 713 0 "
    },
    "á¹Ÿ": {
      "x_min": -44.734375,
      "x_max": 517.765625,
      "ha": 540,
      "o": "m 512 626 q 517 602 517 621 q 512 557 517 582 q 499 504 507 531 q 483 459 491 477 l 455 459 q 443 504 451 485 q 426 534 436 523 q 405 550 417 545 q 381 556 394 556 q 347 543 366 556 q 307 504 327 530 q 266 435 286 477 q 230 336 245 394 l 230 70 q 254 49 230 60 q 342 29 279 38 l 342 0 l 37 0 l 37 29 q 105 49 80 39 q 130 70 130 59 l 130 465 q 128 502 130 487 q 125 527 127 518 q 120 542 123 537 q 114 551 117 547 q 104 560 110 556 q 89 565 98 563 q 68 568 80 568 q 37 568 55 568 l 37 597 q 121 620 80 608 q 196 651 163 632 l 224 627 l 224 473 q 267 543 246 510 q 312 599 288 575 q 362 637 336 623 q 417 651 388 651 q 463 645 438 651 q 512 626 487 640 m 404 -155 q 399 -172 403 -162 q 392 -192 396 -182 q 384 -211 388 -201 q 378 -227 381 -220 l -30 -227 l -44 -210 q -40 -193 -43 -202 q -32 -173 -36 -183 q -24 -154 -28 -164 q -17 -139 -20 -145 l 389 -139 l 404 -155 "
    },
    "á»¹": {
      "x_min": -31,
      "x_max": 658,
      "ha": 672,
      "o": "m 658 601 q 625 592 638 596 q 604 583 612 588 q 592 571 596 578 q 584 555 588 565 l 362 -55 q 300 -184 334 -130 q 229 -272 266 -237 q 155 -322 192 -306 q 84 -339 117 -339 q 38 -335 59 -339 q 1 -327 16 -332 q -22 -315 -13 -322 q -31 -303 -31 -309 q -22 -286 -31 -299 q -2 -260 -14 -274 q 22 -231 9 -245 q 47 -211 36 -218 q 110 -232 79 -232 q 167 -223 142 -232 q 195 -204 179 -219 q 227 -168 211 -189 q 259 -119 243 -146 q 286 -62 274 -91 l 306 -15 l 87 555 q 64 584 81 574 q 14 601 47 593 l 14 630 l 271 630 l 271 601 q 228 594 245 598 q 203 586 212 591 q 193 573 194 581 q 195 555 191 566 l 359 123 l 512 555 q 513 572 515 565 q 502 584 511 579 q 476 593 493 589 q 436 601 460 597 l 436 631 l 658 631 l 658 601 m 579 913 q 551 858 567 885 q 514 809 534 830 q 470 773 493 787 q 420 760 446 760 q 376 771 396 760 q 335 798 355 783 q 295 825 315 813 q 256 837 276 837 q 208 814 230 837 q 161 754 186 792 l 126 768 q 154 823 138 796 q 191 872 171 851 q 235 908 212 894 q 285 922 259 922 q 333 910 310 922 q 375 883 355 898 q 413 856 395 868 q 447 845 431 845 q 496 866 472 845 q 543 928 521 888 l 579 913 "
    },
    "Â«": {
      "x_min": 53,
      "x_max": 567.5,
      "ha": 661,
      "o": "m 53 291 l 53 315 q 53 332 53 325 q 54 341 53 340 l 293 619 l 320 599 l 314 586 q 297 554 307 574 q 272 509 286 534 q 243 457 258 483 q 215 405 229 430 q 191 360 201 380 q 174 328 180 340 q 168 317 167 317 l 321 32 l 293 11 l 53 291 m 299 291 l 299 315 q 299 332 299 325 q 300 341 299 340 l 538 619 l 566 599 l 560 586 q 542 554 553 574 q 518 509 532 534 q 489 457 504 483 q 461 405 475 430 q 437 360 447 380 q 419 328 426 340 q 413 317 413 317 l 567 32 l 538 11 l 299 291 "
    },
    "á»­": {
      "x_min": 27.265625,
      "x_max": 834,
      "ha": 834,
      "o": "m 834 706 q 823 662 834 687 q 787 607 813 636 q 718 548 762 579 q 608 490 674 518 l 608 172 q 611 103 608 130 q 623 70 614 77 q 650 68 631 65 q 704 86 670 71 l 710 55 q 662 25 685 39 q 620 1 639 11 q 586 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 433 44 467 70 q 369 4 398 18 q 316 -14 341 -9 q 271 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 532 113 514 q 100 559 109 550 q 75 572 92 568 q 27 579 57 575 l 27 606 q 72 613 51 608 q 112 622 93 617 q 152 635 132 627 q 194 651 172 642 l 213 624 l 213 226 q 219 147 213 179 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 482 q 505 531 508 513 q 492 560 502 549 q 464 574 482 570 q 415 578 445 578 l 415 606 q 506 625 463 612 q 588 651 550 638 l 608 624 l 608 538 q 659 568 638 552 q 693 598 680 583 q 713 627 707 613 q 720 651 720 640 q 710 689 720 671 q 681 722 700 706 l 799 776 q 824 743 814 761 q 834 706 834 726 m 474 904 q 461 870 474 885 q 432 842 449 855 q 400 817 415 829 q 377 793 385 805 q 376 766 370 780 q 411 736 383 753 q 391 728 401 729 q 370 725 380 726 q 325 753 339 740 q 308 776 311 765 q 311 796 305 787 q 329 815 317 806 q 354 833 340 824 q 381 851 368 842 q 401 871 393 861 q 410 894 410 882 q 397 933 410 920 q 364 946 384 946 q 346 942 354 946 q 331 932 337 938 q 322 918 325 926 q 319 903 319 911 q 320 896 319 900 q 324 889 322 893 q 310 885 319 887 q 292 880 302 882 q 274 877 283 878 q 258 874 264 875 l 251 882 l 251 891 q 263 923 251 907 q 295 951 276 938 q 339 971 315 963 q 386 979 363 979 q 451 958 428 979 q 474 904 474 938 "
    },
    "Ã­": {
      "x_min": 47,
      "x_max": 390.828125,
      "ha": 370,
      "o": "m 47 0 l 47 29 q 115 49 92 38 q 139 70 139 61 l 139 454 q 139 510 139 488 q 130 543 139 531 q 101 560 121 555 q 47 569 82 566 l 47 596 q 90 606 67 600 q 136 619 113 612 q 179 634 158 626 q 217 651 200 642 l 240 651 l 240 70 q 262 50 240 62 q 333 29 284 38 l 333 0 l 47 0 m 161 709 q 140 717 149 712 q 123 728 131 722 l 279 1009 q 300 1007 287 1009 q 327 1003 313 1005 q 354 998 341 1001 q 375 994 368 996 l 390 967 l 161 709 "
    },
    "Ê ": {
      "x_min": 53,
      "x_max": 859,
      "ha": 702,
      "o": "m 495 495 q 433 555 473 533 q 345 576 393 576 q 273 560 308 576 q 212 512 239 544 q 169 433 185 480 q 153 322 153 385 q 169 217 153 264 q 210 137 185 170 q 266 85 236 103 q 323 68 296 68 q 368 77 346 68 q 411 102 390 86 q 453 137 432 117 q 495 177 474 156 l 495 495 m 859 965 q 849 949 859 959 q 826 927 839 938 q 798 906 813 916 q 774 891 784 896 q 746 929 760 915 q 720 953 733 944 q 695 964 707 961 q 675 968 684 968 q 639 957 654 968 q 614 920 624 946 q 599 853 604 895 q 595 750 595 812 l 595 -254 q 615 -275 595 -265 q 687 -296 636 -286 l 687 -325 l 382 -325 l 382 -296 q 469 -276 444 -286 q 495 -254 495 -265 l 495 112 q 445 57 469 81 q 394 15 420 32 q 339 -10 368 -1 q 276 -20 310 -20 q 199 2 239 -20 q 127 65 160 24 q 73 166 94 106 q 53 301 53 226 q 71 411 53 360 q 117 500 90 461 q 174 566 144 539 q 228 606 205 593 q 303 639 266 627 q 366 651 340 651 q 428 643 397 651 q 495 608 459 635 l 495 712 q 500 812 495 770 q 518 885 506 853 q 549 941 530 916 q 593 988 567 965 q 664 1035 627 1019 q 727 1051 701 1051 q 780 1039 755 1051 q 822 1014 804 1028 q 849 985 839 999 q 859 965 859 971 "
    },
    "Çœ": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 467 70 q 369 3 398 18 q 316 -15 340 -10 q 271 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 43 q 113 189 113 125 l 113 481 q 111 532 113 514 q 100 559 109 550 q 75 571 92 568 q 27 578 57 575 l 27 605 q 72 612 51 608 q 112 622 93 616 q 152 634 132 627 q 194 650 172 641 l 213 624 l 213 226 q 219 147 213 178 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 482 q 505 531 508 512 q 492 559 502 549 q 464 573 482 569 q 414 578 446 578 l 414 605 q 507 624 463 611 q 588 650 551 637 l 608 624 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 562 854 q 556 818 562 835 q 539 789 550 801 q 515 769 529 776 q 484 762 501 762 q 441 778 454 762 q 429 826 429 795 q 434 862 429 845 q 451 892 440 879 q 475 911 462 904 q 505 919 489 919 q 562 854 562 919 m 296 854 q 290 818 296 835 q 273 789 284 801 q 249 769 263 776 q 218 762 235 762 q 175 778 188 762 q 163 826 163 795 q 168 862 163 845 q 185 892 174 879 q 209 911 196 904 q 239 919 223 919 q 296 854 296 919 m 438 974 q 420 961 429 966 q 400 953 411 956 l 167 1209 l 181 1237 q 201 1241 188 1239 q 228 1246 214 1243 q 256 1250 242 1249 q 278 1253 270 1252 l 438 974 "
    },
    "á¹¥": {
      "x_min": 63,
      "x_max": 465,
      "ha": 525,
      "o": "m 465 192 q 451 109 465 144 q 417 51 438 75 q 369 13 395 28 q 318 -7 343 0 q 271 -17 293 -15 q 236 -20 248 -20 q 159 -7 204 -20 q 70 29 114 4 q 64 51 65 31 q 63 97 63 71 q 66 150 63 123 q 74 192 69 176 l 104 184 q 117 131 104 156 q 153 87 130 106 q 205 58 175 69 q 269 48 234 48 q 313 55 293 48 q 348 77 333 63 q 370 111 362 91 q 379 154 379 130 q 364 202 379 181 q 326 240 350 223 q 273 273 303 258 q 213 304 244 288 q 159 335 185 319 q 113 371 133 351 q 81 417 93 392 q 69 474 69 442 q 85 549 69 516 q 129 604 102 582 q 192 639 157 627 q 266 651 228 651 q 311 647 287 651 q 361 636 336 643 q 406 620 385 629 q 441 598 427 611 q 442 580 445 594 q 432 549 439 566 q 419 517 426 532 q 408 497 412 502 l 382 502 q 315 570 350 551 q 248 590 281 590 q 209 582 226 590 q 179 563 192 575 q 161 537 167 551 q 155 508 155 522 q 167 469 155 487 q 200 437 180 452 q 247 408 221 421 q 302 379 274 394 q 359 347 330 364 q 412 308 389 330 q 450 258 435 286 q 465 192 465 230 m 234 710 q 213 717 222 712 q 197 729 204 723 l 352 1010 q 373 1008 360 1009 q 400 1004 386 1006 q 428 999 415 1002 q 449 994 441 996 l 464 968 l 234 710 m 330 1099 q 324 1063 330 1080 q 307 1034 318 1046 q 283 1014 297 1021 q 252 1007 269 1007 q 209 1023 222 1007 q 197 1071 197 1040 q 202 1107 197 1090 q 219 1137 208 1124 q 243 1156 230 1149 q 273 1164 257 1164 q 330 1099 330 1164 "
    },
    "Âµ": {
      "x_min": 40.265625,
      "x_max": 720.34375,
      "ha": 734,
      "o": "m 720 55 q 679 29 700 42 q 637 4 657 15 q 600 -13 617 -6 q 572 -21 583 -21 q 548 -9 558 -21 q 531 21 538 2 q 521 67 525 41 q 518 122 518 93 q 413 14 464 50 q 325 -21 363 -21 q 247 6 285 -21 q 180 82 209 33 l 180 78 q 187 -55 180 3 q 208 -160 195 -114 q 238 -237 221 -205 q 272 -290 255 -269 q 245 -300 262 -294 q 210 -314 228 -307 q 175 -328 192 -321 q 151 -339 159 -334 l 126 -319 l 126 495 q 124 537 126 522 q 114 560 122 552 q 88 572 106 568 q 40 579 71 575 l 40 606 q 87 615 66 611 q 128 624 109 619 q 165 635 147 629 q 199 651 182 641 l 202 647 q 209 639 205 644 q 218 630 213 635 q 225 622 222 625 l 225 248 q 233 196 225 224 q 258 145 242 169 q 296 105 273 121 q 347 89 319 89 q 384 92 365 89 q 424 106 403 95 q 468 138 445 118 q 518 194 491 159 l 518 507 q 516 533 518 518 q 509 563 513 548 q 501 591 506 578 q 492 610 497 604 q 552 629 522 618 q 609 651 582 639 l 627 624 q 621 582 625 607 q 618 514 618 557 l 618 181 q 623 92 618 118 q 643 67 629 67 q 674 72 658 67 q 713 86 691 77 l 720 55 "
    },
    "á»³": {
      "x_min": -31,
      "x_max": 658,
      "ha": 672,
      "o": "m 658 601 q 625 592 638 596 q 604 583 612 588 q 592 571 596 578 q 584 555 588 565 l 362 -55 q 300 -184 334 -130 q 229 -272 266 -237 q 155 -322 192 -306 q 84 -339 117 -339 q 38 -335 59 -339 q 1 -327 16 -332 q -22 -315 -13 -322 q -31 -303 -31 -309 q -22 -286 -31 -299 q -2 -260 -14 -274 q 22 -231 9 -245 q 47 -211 36 -218 q 110 -232 79 -232 q 167 -223 142 -232 q 195 -204 179 -219 q 227 -168 211 -189 q 259 -119 243 -146 q 286 -62 274 -91 l 306 -15 l 87 555 q 64 584 81 574 q 14 601 47 593 l 14 630 l 271 630 l 271 601 q 228 594 245 598 q 203 586 212 591 q 193 573 194 581 q 195 555 191 566 l 359 123 l 512 555 q 513 572 515 565 q 502 584 511 579 q 476 593 493 589 q 436 601 460 597 l 436 631 l 658 631 l 658 601 m 428 731 q 411 718 420 723 q 391 710 402 713 l 157 965 l 172 993 q 192 998 179 995 q 219 1003 205 1000 q 247 1007 233 1006 q 269 1010 261 1009 l 428 731 "
    },
    "á¸ž": {
      "x_min": 33,
      "x_max": 595,
      "ha": 650,
      "o": "m 33 0 l 33 28 q 104 49 78 38 q 130 70 130 60 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 575 854 l 595 837 q 592 798 595 819 q 586 756 589 777 q 578 716 582 735 q 569 684 573 697 l 538 684 q 534 736 538 715 q 523 770 530 757 q 507 787 516 782 q 484 793 497 793 l 236 793 l 236 498 l 484 498 l 503 479 q 489 458 497 469 q 473 437 481 447 q 456 417 464 426 q 441 403 447 409 q 420 420 431 413 q 393 432 408 427 q 358 438 378 436 q 310 441 338 441 l 236 441 l 236 70 q 261 51 236 61 q 352 28 287 41 l 352 0 l 33 0 m 381 1044 q 375 1008 381 1025 q 358 979 369 991 q 334 959 348 966 q 303 952 320 952 q 260 968 273 952 q 248 1016 248 985 q 253 1052 248 1035 q 270 1082 259 1069 q 294 1101 281 1094 q 324 1109 308 1109 q 381 1044 381 1109 "
    },
    "M": {
      "x_min": 40,
      "x_max": 1043,
      "ha": 1097,
      "o": "m 1030 825 q 988 818 1010 825 q 944 800 966 812 l 944 70 q 969 50 944 62 q 1043 29 995 38 l 1043 0 l 741 0 l 741 29 q 813 49 786 38 q 840 70 840 61 l 840 705 l 546 0 l 514 0 l 216 700 l 216 70 q 238 50 216 62 q 309 29 261 38 l 309 0 l 40 0 l 40 29 q 112 49 85 38 q 140 70 140 61 l 140 797 q 90 818 115 812 q 46 825 66 825 l 46 854 l 236 854 q 248 851 243 854 q 257 843 252 849 q 266 826 261 837 q 279 797 271 815 l 543 185 l 797 797 q 811 828 806 817 q 821 845 816 840 q 829 852 825 851 q 840 854 834 854 l 1030 854 l 1030 825 "
    },
    "á¸Ž": {
      "x_min": 27.015625,
      "x_max": 746,
      "ha": 806,
      "o": "m 300 817 q 235 817 266 817 l 235 104 q 241 80 235 89 q 277 62 250 68 q 356 57 304 57 q 450 79 402 57 q 540 148 499 102 q 607 265 581 195 q 633 431 633 336 q 611 593 633 521 q 546 714 589 664 q 442 790 504 764 q 300 817 380 817 m 746 456 q 729 305 746 372 q 683 188 712 239 q 617 101 654 137 q 538 43 579 66 q 453 10 496 20 q 371 0 410 0 l 33 0 l 33 29 q 103 49 78 38 q 129 70 129 61 l 129 806 q 78 799 102 802 q 33 792 54 795 l 27 834 q 94 851 56 843 q 175 865 132 859 q 261 872 217 872 q 343 872 304 872 q 510 843 436 872 q 637 762 585 815 q 717 632 689 709 q 746 456 746 554 m 589 -155 q 585 -172 588 -162 q 577 -192 581 -182 q 570 -211 573 -201 q 563 -227 566 -220 l 154 -227 l 140 -210 q 144 -193 141 -202 q 152 -173 148 -183 q 160 -154 156 -164 q 167 -139 164 -145 l 575 -139 l 589 -155 "
    },
    "Å©": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 0 639 11 q 585 -15 600 -9 q 563 -21 571 -21 q 524 10 539 -21 q 510 112 510 42 q 432 44 466 70 q 369 3 398 17 q 316 -15 340 -10 q 270 -21 292 -21 q 211 -12 239 -21 q 160 20 182 -3 q 125 84 138 43 q 113 189 113 125 l 113 482 q 111 532 113 514 q 100 559 109 550 q 75 572 92 568 q 27 578 57 575 l 27 606 q 72 612 51 608 q 112 622 93 616 q 152 634 132 627 q 194 650 172 642 l 213 624 l 213 226 q 219 147 213 178 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 482 q 505 531 508 513 q 492 559 502 549 q 463 574 482 570 q 414 578 445 578 l 414 606 q 506 625 462 612 q 588 650 550 637 l 608 624 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 588 912 q 560 857 576 884 q 523 808 543 830 q 478 773 502 787 q 429 760 455 760 q 384 771 405 760 q 344 798 363 783 q 304 824 324 812 q 265 836 285 836 q 217 814 239 836 q 171 754 195 792 l 135 768 q 163 823 147 795 q 200 872 180 850 q 244 907 221 893 q 294 921 268 921 q 342 909 319 921 q 384 883 364 897 q 422 856 404 868 q 456 845 440 845 q 505 866 481 845 q 552 927 530 888 l 588 912 "
    },
    "Å­": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 466 70 q 369 4 398 18 q 316 -14 340 -9 q 270 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 533 113 515 q 100 560 109 551 q 75 572 92 569 q 27 579 57 576 l 27 606 q 72 613 51 609 q 112 623 93 617 q 152 635 132 628 q 194 651 172 642 l 213 625 l 213 226 q 219 147 213 179 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 483 q 505 532 508 513 q 492 560 502 550 q 463 574 482 570 q 414 579 445 579 l 414 606 q 506 625 462 612 q 588 651 550 638 l 608 625 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 578 923 q 533 830 558 868 q 480 769 508 793 q 423 736 453 746 q 363 726 393 726 q 300 736 330 726 q 242 769 269 746 q 190 830 214 793 q 146 923 166 868 q 162 941 154 935 q 181 953 170 948 q 219 888 198 915 q 265 846 241 862 q 314 823 289 830 q 360 816 338 816 q 409 823 384 816 q 459 846 435 830 q 505 888 483 862 q 542 953 526 915 q 562 941 554 948 q 578 923 570 935 "
    },
    "{": {
      "x_min": 68.078125,
      "x_max": 431.03125,
      "ha": 457,
      "o": "m 431 1052 q 382 1015 405 1036 q 344 968 360 994 q 318 913 327 943 q 309 848 309 883 q 313 784 309 809 q 322 737 317 759 q 331 689 327 715 q 336 625 336 664 q 327 569 336 597 q 304 518 319 542 q 266 474 288 493 q 216 442 244 454 q 272 421 249 436 q 309 382 295 405 q 329 328 323 359 q 336 260 336 298 q 331 191 336 220 q 322 137 327 163 q 313 87 317 112 q 309 28 309 62 q 313 -37 309 -8 q 331 -89 318 -66 q 366 -132 344 -112 q 424 -172 389 -152 l 409 -213 q 313 -165 353 -190 q 248 -108 273 -140 q 210 -39 222 -77 q 199 48 199 0 q 203 114 199 88 q 212 164 207 140 q 221 216 217 188 q 226 285 226 243 q 199 368 226 340 q 122 397 172 397 l 108 397 q 101 396 104 397 q 93 395 97 396 q 79 392 89 394 l 68 428 q 186 491 146 449 q 226 584 226 533 q 224 631 226 611 q 219 667 222 651 q 212 697 216 683 q 205 726 208 710 q 200 762 202 742 q 199 809 199 782 q 213 895 199 855 q 256 970 228 936 q 324 1033 284 1004 q 413 1084 364 1061 l 431 1052 "
    },
    "Â¼": {
      "x_min": 52.703125,
      "x_max": 743.984375,
      "ha": 797,
      "o": "m 631 332 l 497 153 l 631 153 l 631 332 m 743 142 q 729 124 735 132 q 713 111 722 116 l 693 111 l 693 39 q 694 34 693 36 q 700 29 695 32 q 715 24 705 27 q 740 19 724 22 l 740 0 l 558 0 l 558 19 q 596 25 581 22 q 618 31 611 28 q 628 37 626 34 q 631 42 631 39 l 631 111 l 446 111 l 433 121 l 619 381 q 651 393 634 387 q 679 405 668 400 l 693 393 l 693 153 l 733 153 l 743 142 m 64 432 l 64 455 q 111 461 93 457 q 140 468 130 465 q 155 476 151 472 q 159 484 159 480 l 159 732 q 158 752 159 745 q 152 763 157 759 q 144 766 150 765 q 128 768 139 768 q 101 765 118 768 q 60 759 85 763 l 52 781 q 90 792 67 785 q 135 806 112 799 q 179 822 158 814 q 211 837 199 831 l 225 825 l 225 484 q 228 477 225 480 q 240 469 231 473 q 267 461 250 465 q 313 455 285 457 l 313 432 l 64 432 m 209 1 q 179 -10 193 -5 q 147 -20 165 -15 l 131 0 l 637 815 q 666 827 650 820 q 696 837 681 833 l 713 818 l 209 1 "
    },
    "á¸¾": {
      "x_min": 40,
      "x_max": 1043,
      "ha": 1097,
      "o": "m 1030 825 q 988 818 1010 825 q 944 800 966 812 l 944 70 q 969 50 944 62 q 1043 29 995 38 l 1043 0 l 741 0 l 741 29 q 813 49 786 38 q 840 70 840 61 l 840 705 l 546 0 l 514 0 l 216 700 l 216 70 q 238 50 216 62 q 309 29 261 38 l 309 0 l 40 0 l 40 29 q 112 49 85 38 q 140 70 140 61 l 140 797 q 90 818 115 812 q 46 825 66 825 l 46 854 l 236 854 q 248 851 243 854 q 257 843 252 849 q 266 826 261 837 q 279 797 271 815 l 543 185 l 797 797 q 811 828 806 817 q 821 845 816 840 q 829 852 825 851 q 840 854 834 854 l 1030 854 l 1030 825 m 453 926 q 437 937 442 931 q 423 957 432 944 l 681 1172 q 699 1161 687 1167 q 722 1147 710 1155 q 744 1134 734 1140 q 759 1121 754 1127 l 765 1091 l 453 926 "
    },
    "Ä²": {
      "x_min": 47,
      "x_max": 775,
      "ha": 822,
      "o": "m 47 0 l 47 29 q 118 49 92 38 q 144 70 144 61 l 144 782 q 119 803 144 790 q 47 824 95 815 l 47 854 l 346 854 l 346 824 q 275 803 301 815 q 250 782 250 792 l 250 70 q 274 50 250 62 q 346 29 298 38 l 346 0 l 47 0 m 775 824 q 703 803 729 815 q 678 782 678 792 l 678 138 q 667 4 678 58 q 637 -86 656 -49 q 595 -149 619 -124 q 546 -195 571 -173 q 506 -222 528 -210 q 463 -241 484 -233 q 421 -253 441 -249 q 389 -258 402 -258 q 349 -251 369 -258 q 314 -236 329 -245 q 289 -218 299 -228 q 280 -203 280 -209 q 288 -187 280 -198 q 309 -167 297 -177 q 334 -148 321 -157 q 356 -135 347 -138 q 389 -155 375 -147 q 415 -167 403 -163 q 438 -172 427 -171 q 461 -174 449 -174 q 498 -162 479 -174 q 534 -121 518 -150 q 561 -40 550 -92 q 572 90 572 11 l 572 782 q 566 791 572 786 q 547 801 561 796 q 507 812 532 807 q 443 824 482 818 l 443 854 l 775 854 l 775 824 "
    },
    "ÃŠ": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 413 432 428 427 q 377 438 398 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 543 957 q 531 938 535 944 q 515 926 526 932 l 333 1067 l 153 926 q 144 931 147 929 q 137 938 140 934 q 131 946 134 941 q 123 957 128 950 l 298 1166 l 369 1166 l 543 957 "
    },
    ")": {
      "x_min": 26.984375,
      "x_max": 352,
      "ha": 431,
      "o": "m 352 450 q 330 234 352 338 q 268 42 308 130 q 172 -110 228 -44 q 47 -213 116 -176 l 26 -183 q 108 -86 69 -146 q 175 51 146 -27 q 222 225 205 130 q 239 430 239 321 q 225 622 239 528 q 184 798 211 716 q 117 946 157 880 q 26 1054 78 1012 l 47 1085 q 179 987 122 1050 q 274 840 236 923 q 332 657 313 756 q 352 450 352 558 "
    },
    "á¹¼": {
      "x_min": 13,
      "x_max": 857,
      "ha": 884,
      "o": "m 857 825 q 786 807 811 816 q 754 780 760 799 l 499 39 q 481 14 494 24 q 453 -3 468 3 q 423 -14 437 -10 q 399 -20 408 -18 l 107 780 q 77 808 100 797 q 13 825 54 819 l 13 854 l 302 854 l 302 823 q 229 813 248 823 q 216 783 209 803 l 451 137 l 675 780 q 659 809 682 799 q 581 825 635 818 l 581 854 l 857 854 l 857 825 m 668 1102 q 639 1047 656 1074 q 603 998 623 1020 q 558 963 582 977 q 509 950 534 950 q 464 961 485 950 q 423 988 443 973 q 384 1014 403 1002 q 345 1026 364 1026 q 296 1004 318 1026 q 250 944 274 982 l 214 958 q 243 1013 226 985 q 280 1062 259 1040 q 324 1097 300 1083 q 373 1111 348 1111 q 421 1099 399 1111 q 464 1073 444 1087 q 502 1046 484 1058 q 536 1035 520 1035 q 585 1056 561 1035 q 632 1117 610 1078 l 668 1102 "
    },
    "a": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 "
    },
    "Æ": {
      "x_min": -163,
      "x_max": 850,
      "ha": 883,
      "o": "m 754 -21 q 701 -7 720 -16 q 674 13 682 2 l 209 672 l 209 138 q 202 3 209 57 q 181 -87 195 -50 q 148 -149 168 -124 q 102 -195 128 -174 q 63 -221 85 -210 q 19 -241 41 -233 q -21 -253 -1 -249 q -53 -258 -40 -258 q -93 -251 -73 -258 q -128 -236 -113 -245 q -153 -219 -143 -228 q -163 -203 -163 -210 q -154 -188 -163 -198 q -133 -168 -145 -178 q -108 -148 -122 -157 q -86 -136 -95 -139 q -53 -156 -67 -148 q -27 -168 -39 -164 q -4 -173 -15 -172 q 18 -175 6 -175 q 55 -163 35 -175 q 91 -122 75 -151 q 118 -41 107 -92 q 129 89 129 10 l 129 778 q 82 809 106 799 q 32 824 58 820 l 32 854 l 173 854 q 189 852 183 854 q 202 845 196 850 q 216 829 208 839 q 237 801 224 818 l 674 186 l 674 782 q 652 804 674 790 q 577 824 630 817 l 577 854 l 850 854 l 850 824 q 778 805 802 818 q 754 782 754 792 l 754 -21 "
    },
    "Z": {
      "x_min": 39.578125,
      "x_max": 655,
      "ha": 712,
      "o": "m 655 198 q 655 150 655 177 q 655 97 655 124 q 655 45 655 70 q 655 0 655 19 l 58 0 l 39 30 l 512 787 l 207 787 q 183 779 196 787 q 159 755 171 772 q 136 714 147 739 q 118 653 125 688 l 80 661 l 100 865 q 132 858 118 861 q 162 855 147 855 q 191 855 176 855 l 635 855 l 652 825 l 183 68 l 531 68 q 556 74 545 68 q 578 96 567 80 q 599 139 588 112 q 622 208 609 166 l 655 198 "
    },
    "ïˆ‘": {
      "x_min": 53,
      "x_max": 866.8125,
      "ha": 701,
      "o": "m 323 68 q 368 77 346 68 q 411 102 390 86 q 453 137 432 117 q 495 177 474 156 l 495 495 q 433 555 473 533 q 345 576 393 576 q 273 560 308 576 q 212 512 239 544 q 169 433 185 480 q 153 322 153 385 q 169 217 153 264 q 210 137 185 170 q 266 85 236 103 q 323 68 296 68 m 495 112 q 445 57 469 81 q 394 15 420 32 q 339 -10 368 -1 q 276 -20 310 -20 q 199 2 239 -20 q 127 65 160 24 q 73 166 94 106 q 53 301 53 226 q 71 411 53 360 q 117 500 90 461 q 174 566 144 539 q 228 606 205 593 q 303 639 266 628 q 366 651 340 651 q 401 648 384 651 q 438 639 419 646 q 477 620 457 632 q 519 587 497 608 q 540 602 529 594 q 561 619 551 610 q 580 636 571 627 q 595 651 589 644 l 615 630 q 604 589 609 611 q 597 543 600 569 q 595 486 595 516 l 595 -112 q 613 -232 595 -190 q 679 -275 631 -275 q 707 -268 694 -275 q 729 -252 720 -262 q 744 -230 739 -243 q 750 -205 750 -218 q 742 -177 750 -190 q 762 -162 748 -171 q 793 -147 776 -154 q 827 -137 810 -141 q 853 -133 844 -133 l 866 -160 q 847 -219 866 -188 q 796 -277 828 -250 q 720 -321 763 -303 q 630 -339 677 -339 q 563 -323 589 -339 q 522 -283 537 -308 q 500 -224 506 -257 q 495 -154 495 -190 l 495 112 "
    },
    "k": {
      "x_min": 37,
      "x_max": 681.265625,
      "ha": 681,
      "o": "m 37 1 l 37 29 q 105 49 80 41 q 130 70 130 58 l 130 879 q 126 927 130 910 q 112 952 122 944 q 83 963 102 961 q 37 970 65 966 l 37 997 q 127 1019 82 1006 q 205 1051 171 1031 l 230 1028 l 230 359 l 434 542 q 461 573 456 561 q 459 591 465 585 q 438 599 453 597 q 406 602 423 602 l 406 631 l 658 631 l 658 602 q 604 590 628 598 q 551 559 579 583 l 324 374 l 574 75 q 594 56 584 64 q 616 45 604 49 q 643 41 628 41 q 677 41 658 41 l 681 13 q 643 4 661 8 q 607 -1 624 1 q 578 -4 591 -3 q 558 -6 565 -6 q 514 2 531 -6 q 482 29 497 10 l 230 354 l 230 70 q 232 61 230 65 q 241 52 234 57 q 263 43 249 48 q 303 29 278 37 l 303 1 l 37 1 "
    },
    "Ã™": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 847 855 l 847 825 m 567 958 q 553 938 558 945 q 537 927 548 932 l 226 1092 l 231 1122 q 246 1135 236 1128 q 268 1148 256 1141 q 291 1162 280 1156 q 310 1173 302 1168 l 567 958 "
    },
    "Å®": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 615 816 l 568 855 l 847 855 l 847 825 m 513 1060 q 509 1091 513 1076 q 497 1117 505 1106 q 479 1135 490 1128 q 456 1142 469 1142 q 429 1136 442 1142 q 407 1120 417 1130 q 391 1094 397 1109 q 386 1061 386 1079 q 389 1031 386 1045 q 401 1005 393 1016 q 419 987 408 994 q 442 981 429 981 q 468 986 456 981 q 491 1001 481 991 q 507 1025 501 1011 q 513 1060 513 1040 m 573 1088 q 560 1024 573 1053 q 526 974 547 995 q 479 940 505 952 q 428 929 454 929 q 387 937 406 929 q 354 959 368 945 q 333 992 341 973 q 326 1033 326 1011 q 338 1097 326 1068 q 372 1148 351 1127 q 418 1182 393 1170 q 470 1194 444 1194 q 510 1185 491 1194 q 543 1162 529 1177 q 565 1128 557 1147 q 573 1088 573 1109 "
    },
    "Â¢": {
      "x_min": 69,
      "x_max": 573.71875,
      "ha": 638,
      "o": "m 170 417 q 181 331 170 370 q 212 264 192 293 q 258 215 231 234 q 315 186 284 196 l 315 655 q 259 632 285 649 q 213 585 232 615 q 181 515 193 556 q 170 417 170 473 m 375 -5 q 364 -14 369 -11 q 355 -19 359 -17 q 345 -23 351 -21 q 331 -27 339 -25 l 315 -11 l 315 88 q 223 109 267 88 q 145 169 179 130 q 89 265 110 208 q 69 396 69 322 q 87 509 69 456 q 137 604 105 562 q 215 675 170 646 q 315 719 261 705 l 315 823 q 328 832 324 829 q 335 837 332 835 q 343 840 338 838 q 357 844 347 841 l 375 829 l 375 730 q 390 731 382 731 q 407 731 398 731 q 452 727 429 731 q 496 716 475 723 q 535 699 517 709 q 564 677 553 690 q 562 655 565 670 q 553 623 559 640 q 539 592 547 607 q 526 568 532 577 l 503 575 q 486 598 497 585 q 461 623 476 611 q 424 643 445 634 q 375 656 402 653 l 375 178 q 408 179 392 178 q 443 189 424 181 q 487 215 462 198 q 547 265 511 233 l 573 237 q 512 169 539 196 q 461 125 485 142 q 417 100 438 109 q 375 90 396 92 l 375 -5 "
    },
    "É": {
      "x_min": 27,
      "x_max": 592,
      "ha": 632,
      "o": "m 153 0 l 153 29 q 237 49 211 39 q 263 70 263 59 l 263 268 q 279 354 263 317 q 320 421 296 391 q 374 475 345 450 q 427 525 403 500 q 469 576 452 549 q 486 637 486 603 q 435 764 486 721 q 295 807 385 807 q 233 794 261 807 q 185 760 205 781 q 154 714 165 740 q 143 661 143 688 q 145 633 143 646 q 153 608 147 621 q 105 592 132 597 q 49 583 79 586 l 31 603 q 28 619 30 608 q 27 639 27 629 q 52 730 27 687 q 121 805 78 773 q 223 855 165 836 q 347 875 281 875 q 451 860 405 875 q 528 819 497 845 q 575 753 559 792 q 592 668 592 715 q 575 582 592 617 q 534 520 558 547 q 480 472 509 494 q 427 427 451 450 q 385 376 402 404 q 369 308 369 348 l 369 70 q 396 49 369 60 q 479 29 424 38 l 479 0 l 153 0 "
    },
    "Ä“": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 290 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 569 868 q 565 851 568 861 q 557 831 561 841 q 550 812 553 822 q 543 797 546 803 l 135 797 l 121 813 q 125 830 122 821 q 133 850 129 840 q 141 869 137 859 q 148 885 145 878 l 555 885 l 569 868 "
    },
    "áº¸": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 568 855 l 590 838 q 587 799 590 820 q 580 757 584 778 q 571 717 576 736 q 563 685 567 698 l 533 685 q 528 737 533 716 q 517 771 524 758 q 501 788 511 783 q 479 794 491 794 l 236 794 l 236 499 l 505 499 l 522 480 q 509 459 517 470 q 493 438 501 448 q 476 418 485 427 q 461 404 468 410 q 439 421 451 414 q 413 433 427 428 q 377 439 398 437 q 329 442 357 442 l 236 442 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 400 -189 q 394 -225 400 -208 q 377 -254 388 -242 q 353 -274 367 -267 q 322 -282 339 -282 q 279 -265 292 -282 q 267 -217 267 -248 q 272 -181 267 -198 q 289 -151 278 -164 q 313 -132 300 -139 q 343 -125 327 -125 q 400 -189 400 -125 "
    },
    "â‰ ": {
      "x_min": 40.546875,
      "x_max": 556.9375,
      "ha": 597,
      "o": "m 433 519 l 540 519 l 556 502 q 547 472 553 488 q 537 444 541 455 l 387 444 l 307 315 l 540 315 l 556 298 q 547 269 553 284 q 537 241 541 253 l 261 241 l 176 101 q 158 92 170 97 q 133 82 146 87 q 107 74 120 78 q 86 67 94 70 l 70 91 l 162 241 l 57 241 l 40 259 q 49 287 43 272 q 61 315 55 302 l 208 315 l 287 444 l 57 444 l 40 461 q 49 490 43 474 q 61 519 55 506 l 332 519 l 419 659 q 462 679 437 670 q 507 692 487 687 l 527 670 l 433 519 "
    },
    "Â¥": {
      "x_min": -44.375,
      "x_max": 663,
      "ha": 638,
      "o": "m 157 0 l 157 29 q 241 55 217 42 q 266 78 266 68 l 266 296 l 85 296 l 70 313 q 77 335 73 323 q 85 358 81 347 l 266 358 l 266 393 q 217 493 242 443 q 166 588 192 542 q 115 673 140 634 q 67 742 89 713 q 53 756 61 750 q 34 768 46 763 q 4 776 22 773 q -41 780 -13 780 l -44 809 q 34 819 -5 814 q 101 823 73 823 q 153 797 131 823 q 200 730 177 764 q 243 657 222 695 q 286 577 265 619 q 331 486 308 534 l 474 742 q 467 770 484 760 q 400 787 451 780 l 400 817 l 663 817 l 663 787 q 593 769 618 778 q 559 742 569 760 l 372 392 l 372 358 l 552 358 l 569 342 l 552 296 l 372 296 l 372 78 q 377 68 372 74 q 396 55 383 62 q 430 42 409 49 q 483 29 451 35 l 483 0 l 157 0 "
    },
    "È½": {
      "x_min": 29.015625,
      "x_max": 627.09375,
      "ha": 648,
      "o": "m 627 165 q 613 63 621 106 q 599 0 605 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 364 l 43 364 l 29 380 q 36 402 32 391 q 43 425 39 414 l 130 425 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 425 l 395 425 l 411 409 l 395 364 l 236 364 l 236 111 q 241 89 236 98 q 261 74 247 80 q 299 64 275 67 q 359 61 322 61 l 454 61 q 505 64 484 61 q 542 79 526 67 q 571 114 558 91 q 598 177 584 137 l 627 165 "
    },
    "Ä¤": {
      "x_min": 33,
      "x_max": 844,
      "ha": 890,
      "o": "m 33 0 l 33 29 q 104 49 78 38 q 130 70 130 60 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 478 l 641 478 l 641 783 q 617 803 641 791 q 545 825 592 815 l 545 854 l 844 854 l 844 825 q 772 804 798 815 q 747 783 747 792 l 747 70 q 771 50 747 62 q 844 29 795 38 l 844 0 l 545 0 l 545 29 q 615 49 590 38 q 641 70 641 60 l 641 417 l 236 417 l 236 70 q 259 50 236 62 q 332 29 283 38 l 332 0 l 33 0 m 648 957 q 636 938 640 944 q 620 926 631 932 l 437 1067 l 257 926 q 248 932 252 929 q 242 938 245 935 q 236 946 239 941 q 228 957 232 951 l 403 1166 l 474 1166 l 648 957 "
    },
    "U": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 847 855 l 847 825 "
    },
    "Ã‘": {
      "x_min": 33,
      "x_max": 851,
      "ha": 884,
      "o": "m 33 0 l 33 29 q 105 48 82 35 q 129 70 129 61 l 129 778 q 83 810 107 799 q 33 824 58 820 l 33 854 l 173 854 q 190 852 183 854 q 203 845 196 850 q 217 829 209 839 q 237 801 224 818 l 674 187 l 674 782 q 652 804 674 790 q 578 824 630 817 l 578 854 l 851 854 l 851 824 q 778 805 802 818 q 754 782 754 792 l 754 -20 q 701 -6 720 -16 q 674 14 682 2 l 209 673 l 209 70 q 231 49 209 62 q 305 29 253 36 l 305 0 l 33 0 m 668 1103 q 639 1048 656 1075 q 602 999 622 1020 q 558 963 582 977 q 508 950 534 950 q 464 961 485 950 q 423 988 443 973 q 383 1015 403 1003 q 345 1027 364 1027 q 296 1004 318 1027 q 250 944 274 982 l 214 958 q 243 1013 226 986 q 280 1062 259 1041 q 324 1098 300 1084 q 373 1112 347 1112 q 421 1100 398 1112 q 464 1073 444 1088 q 502 1046 484 1058 q 535 1035 520 1035 q 585 1056 561 1035 q 632 1118 609 1078 l 668 1103 "
    },
    "F": {
      "x_min": 33,
      "x_max": 595,
      "ha": 650,
      "o": "m 33 0 l 33 28 q 104 49 78 38 q 130 70 130 60 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 575 854 l 595 837 q 592 798 595 819 q 586 756 589 777 q 578 716 582 735 q 569 684 573 697 l 538 684 q 534 736 538 715 q 523 770 530 757 q 507 787 516 782 q 484 793 497 793 l 236 793 l 236 498 l 484 498 l 503 479 q 489 458 497 469 q 472 437 481 447 q 455 417 464 426 q 441 403 447 409 q 419 420 431 413 q 393 432 408 427 q 358 438 378 436 q 310 441 338 441 l 236 441 l 236 70 q 261 51 236 61 q 352 28 287 41 l 352 0 l 33 0 "
    },
    "áº£": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 148 302 128 288 q 202 330 169 317 q 286 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 194 490 192 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 430 904 q 417 870 430 885 q 389 842 405 855 q 357 817 372 829 q 334 793 341 805 q 333 766 326 780 q 367 736 339 753 q 347 728 357 729 q 326 725 336 726 q 282 753 296 740 q 265 776 268 765 q 268 796 262 787 q 286 815 274 806 q 311 833 297 824 q 337 851 325 842 q 357 871 349 861 q 366 894 366 882 q 353 933 366 920 q 321 946 340 946 q 303 942 311 946 q 288 932 294 938 q 279 918 282 926 q 276 903 276 911 q 277 896 276 900 q 281 889 279 893 q 267 885 276 887 q 249 880 259 882 q 231 877 240 878 q 215 874 221 875 l 208 882 l 208 891 q 220 923 208 907 q 252 951 233 938 q 296 971 272 963 q 343 979 320 979 q 407 958 384 979 q 430 904 430 938 "
    },
    "Ê”": {
      "x_min": 40,
      "x_max": 565,
      "ha": 592,
      "o": "m 133 0 l 133 29 q 246 70 246 49 l 246 336 q 262 432 246 388 q 302 513 278 475 q 355 585 327 551 q 408 652 384 619 q 448 720 432 685 q 465 793 465 755 q 451 870 465 835 q 413 932 437 906 q 357 972 389 957 q 288 987 325 987 q 236 973 261 987 q 192 939 212 960 q 160 891 172 917 q 149 838 149 864 q 151 818 149 829 q 156 799 153 808 q 112 782 135 787 q 59 773 89 776 l 41 793 q 40 804 40 799 l 40 816 q 65 906 40 863 q 132 981 90 949 q 228 1032 174 1013 q 340 1051 282 1051 q 433 1033 391 1051 q 503 984 474 1015 q 549 912 533 954 q 565 824 565 871 q 548 733 565 773 q 508 660 532 693 q 455 595 483 626 q 402 530 427 563 q 362 460 378 498 q 346 376 346 423 l 346 70 q 375 49 346 60 q 459 29 404 38 l 459 0 l 133 0 "
    },
    "á»": {
      "x_min": 53,
      "x_max": 731,
      "ha": 731,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 731 707 q 723 670 731 690 q 699 626 716 649 q 652 577 681 602 q 578 527 622 551 q 617 436 604 486 q 631 329 631 387 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 465 628 415 651 q 550 565 514 606 q 601 612 585 589 q 617 652 617 635 q 606 689 617 671 q 578 722 596 707 l 696 777 q 721 744 711 762 q 731 707 731 726 m 431 731 q 413 718 422 723 q 393 710 405 713 l 160 966 l 174 994 q 194 998 181 996 q 221 1003 207 1000 q 249 1007 235 1006 q 271 1010 263 1009 l 431 731 "
    },
    "Ì¿": {
      "x_min": -683.21875,
      "x_max": 42.53125,
      "ha": 0,
      "o": "m 42 1051 q 37 1034 41 1044 q 30 1014 34 1024 q 22 995 26 1005 q 16 980 19 986 l -668 980 l -683 996 q -678 1013 -681 1004 q -671 1033 -675 1023 q -663 1052 -667 1042 q -655 1068 -659 1061 l 27 1068 l 42 1051 m 42 868 q 37 851 41 861 q 30 831 34 841 q 22 812 26 822 q 16 797 19 803 l -668 797 l -683 813 q -678 830 -681 821 q -671 850 -675 840 q -663 869 -667 859 q -655 885 -659 878 l 27 885 l 42 868 "
    },
    "Ã¥": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 216 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 280 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 165 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 383 843 q 379 874 383 859 q 367 900 375 889 q 349 918 360 911 q 325 925 339 925 q 299 919 311 925 q 276 903 286 913 q 261 877 267 892 q 255 844 255 862 q 259 814 255 828 q 270 788 263 799 q 288 770 277 777 q 312 764 299 764 q 338 769 325 764 q 361 784 351 774 q 377 808 371 794 q 383 843 383 823 m 443 870 q 430 807 443 836 q 396 757 417 778 q 349 723 375 735 q 298 712 323 712 q 256 720 275 712 q 224 742 237 728 q 202 775 210 756 q 195 816 195 794 q 208 880 195 851 q 241 931 221 910 q 287 965 262 953 q 339 977 313 977 q 380 968 361 977 q 413 945 399 960 q 435 911 427 930 q 443 870 443 892 "
    },
    "0": {
      "x_min": 51,
      "x_max": 589,
      "ha": 640,
      "o": "m 480 383 q 465 547 480 476 q 427 664 451 617 q 372 735 403 711 q 307 760 340 760 q 245 741 273 760 q 199 683 218 722 q 170 583 180 644 q 160 437 160 522 q 173 273 160 344 q 209 154 186 202 q 264 82 232 106 q 332 58 295 58 q 394 76 367 58 q 441 134 422 95 q 470 236 460 174 q 480 383 480 297 m 589 408 q 568 243 589 321 q 511 106 548 165 q 422 14 474 48 q 307 -20 370 -20 q 195 14 243 -20 q 115 106 148 48 q 67 243 83 165 q 51 408 51 321 q 71 574 51 496 q 129 710 92 652 q 217 803 166 769 q 332 838 269 838 q 444 804 396 838 q 524 711 491 770 q 572 575 556 653 q 589 408 589 496 "
    },
    "É‹": {
      "x_min": 53,
      "x_max": 866.8125,
      "ha": 701,
      "o": "m 323 68 q 368 77 346 68 q 411 102 390 86 q 453 137 432 117 q 495 177 474 156 l 495 495 q 433 555 473 533 q 345 576 393 576 q 273 560 308 576 q 212 512 239 544 q 169 433 185 480 q 153 322 153 385 q 169 217 153 264 q 210 137 185 170 q 266 85 236 103 q 323 68 296 68 m 495 112 q 445 57 469 81 q 394 15 420 32 q 339 -10 368 -1 q 276 -20 310 -20 q 199 2 239 -20 q 127 65 160 24 q 73 166 94 106 q 53 301 53 226 q 71 411 53 360 q 117 500 90 461 q 174 566 144 539 q 228 606 205 593 q 303 639 266 628 q 366 651 340 651 q 401 648 384 651 q 438 639 419 646 q 477 620 457 632 q 519 587 497 608 q 540 602 529 594 q 561 619 551 610 q 580 636 571 627 q 595 651 589 644 l 615 630 q 604 589 609 611 q 597 543 600 569 q 595 486 595 516 l 595 -112 q 613 -232 595 -190 q 679 -275 631 -275 q 707 -268 694 -275 q 729 -252 720 -262 q 744 -230 739 -243 q 750 -205 750 -218 q 742 -177 750 -190 q 762 -162 748 -171 q 793 -147 776 -154 q 827 -137 810 -141 q 853 -133 844 -133 l 866 -160 q 847 -219 866 -188 q 796 -277 828 -250 q 720 -321 763 -303 q 630 -339 677 -339 q 563 -323 589 -339 q 522 -283 537 -308 q 500 -224 506 -257 q 495 -154 495 -190 l 495 112 "
    },
    "Å": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 586 868 q 581 851 585 861 q 574 831 578 841 q 566 812 570 822 q 560 797 563 803 l 152 797 l 137 813 q 142 830 138 821 q 149 850 145 840 q 157 869 153 859 q 164 885 161 878 l 571 885 l 586 868 "
    },
    "â€": {
      "x_min": 58.59375,
      "x_max": 531,
      "ha": 593,
      "o": "m 251 878 q 242 821 251 852 q 215 760 233 790 q 173 701 198 729 q 117 652 149 673 l 87 676 q 111 709 101 690 q 129 748 122 729 q 139 785 136 767 q 143 816 143 803 q 125 870 143 847 q 69 893 107 893 l 58 924 q 83 943 64 932 q 126 963 103 953 q 171 979 149 973 q 205 986 193 986 q 239 941 227 964 q 251 878 251 917 m 531 878 q 522 821 531 852 q 495 760 513 790 q 454 701 478 729 q 397 652 429 673 l 369 676 q 393 709 383 690 q 410 748 403 729 q 421 785 417 767 q 425 816 425 803 q 406 870 425 847 q 349 893 387 893 l 339 924 q 363 943 344 932 q 406 963 383 953 q 451 979 429 973 q 485 986 474 986 q 519 941 507 964 q 531 878 531 917 "
    },
    "á¸•": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 290 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 569 868 q 565 851 568 861 q 557 831 561 841 q 550 812 553 822 q 543 797 546 803 l 135 797 l 121 813 q 125 830 122 821 q 133 850 129 840 q 141 869 137 859 q 148 885 145 878 l 555 885 l 569 868 m 414 975 q 397 962 405 967 q 377 954 388 957 l 143 1209 l 158 1237 q 177 1242 164 1239 q 205 1247 190 1244 q 233 1251 219 1249 q 255 1253 247 1253 l 414 975 "
    },
    "Ã¶": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 555 855 q 549 819 555 836 q 532 790 543 802 q 508 770 522 777 q 477 763 494 763 q 434 779 447 763 q 422 827 422 796 q 427 863 422 846 q 444 893 433 880 q 468 912 455 905 q 498 920 482 920 q 555 855 555 920 m 289 855 q 283 819 289 836 q 266 790 277 802 q 242 770 256 777 q 211 763 228 763 q 168 779 181 763 q 156 827 156 796 q 161 863 156 846 q 178 893 167 880 q 202 912 189 905 q 232 920 216 920 q 289 855 289 920 "
    },
    "áº‰": {
      "x_min": 13,
      "x_max": 947,
      "ha": 961,
      "o": "m 947 601 q 914 592 926 597 q 895 583 902 588 q 885 573 888 579 q 880 559 882 567 l 733 39 q 718 14 729 24 q 692 -3 706 3 q 664 -14 678 -10 q 642 -20 651 -18 l 483 439 l 347 39 q 331 13 342 24 q 307 -3 320 3 q 281 -14 294 -10 q 259 -20 268 -18 l 85 559 q 13 601 79 586 l 13 631 l 270 631 l 270 601 q 218 594 237 598 q 192 583 200 589 q 184 571 184 578 q 186 559 184 565 l 311 128 l 474 631 l 520 631 l 693 128 l 808 559 q 794 584 812 575 q 731 601 776 594 l 731 631 l 947 631 l 947 601 m 551 -189 q 545 -225 551 -208 q 528 -254 539 -242 q 504 -274 518 -267 q 473 -282 490 -282 q 430 -265 443 -282 q 418 -217 418 -248 q 423 -181 418 -198 q 440 -151 429 -164 q 464 -132 451 -139 q 494 -125 478 -125 q 551 -189 551 -125 "
    },
    "È¦": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 387 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 838 q 390 869 367 855 q 430 893 413 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 465 1044 q 459 1008 465 1025 q 442 979 453 991 q 418 959 432 966 q 387 952 404 952 q 344 968 357 952 q 332 1016 332 985 q 337 1052 332 1035 q 354 1082 343 1069 q 378 1101 365 1094 q 408 1109 392 1109 q 465 1044 465 1109 "
    },
    "Ä‡": {
      "x_min": 53,
      "x_max": 558.40625,
      "ha": 595,
      "o": "m 558 129 q 482 47 515 76 q 421 2 449 17 q 366 -16 393 -12 q 309 -20 340 -20 q 214 2 260 -20 q 132 65 168 24 q 74 166 96 106 q 53 301 53 226 q 79 438 53 374 q 151 549 105 502 q 258 624 196 596 q 392 651 320 651 q 437 647 414 651 q 480 637 459 644 q 519 620 501 629 q 548 598 537 610 q 546 574 549 589 q 537 543 543 560 q 523 511 531 526 q 510 486 516 496 l 485 493 q 468 519 480 505 q 437 547 456 534 q 392 568 418 560 q 332 577 366 577 q 264 561 297 577 q 207 513 232 545 q 167 433 182 481 q 153 322 153 386 q 169 217 153 264 q 215 137 186 170 q 280 85 243 103 q 357 68 317 68 q 391 69 375 68 q 426 80 407 71 q 470 106 445 89 q 532 156 495 124 l 558 129 m 297 710 q 275 718 285 712 q 259 729 266 723 l 414 1010 q 435 1007 422 1009 q 463 1004 448 1006 q 490 999 477 1002 q 511 994 503 996 l 526 968 l 297 710 "
    },
    "Ã¾": {
      "x_min": 37,
      "x_max": 669,
      "ha": 722,
      "o": "m 579 276 q 565 397 579 343 q 529 488 551 450 q 476 546 506 526 q 414 566 446 566 q 383 558 402 566 q 339 533 364 551 q 287 486 315 515 q 230 413 259 456 l 230 144 q 327 82 281 103 q 420 61 374 61 q 487 79 458 61 q 537 128 516 97 q 568 197 557 158 q 579 276 579 235 m 669 333 q 658 253 669 294 q 629 172 648 211 q 585 97 611 133 q 527 36 559 62 q 458 -4 495 10 q 381 -20 421 -20 q 354 -16 371 -20 q 318 -4 338 -12 q 275 17 298 4 q 230 50 253 30 l 230 -253 q 255 -275 230 -264 q 343 -295 280 -286 l 343 -325 l 37 -325 l 37 -295 q 105 -275 80 -285 q 130 -253 130 -265 l 130 878 q 126 926 130 909 q 111 951 122 943 q 83 963 101 960 q 37 969 64 966 l 37 996 q 125 1017 84 1006 q 205 1051 165 1029 l 230 1028 l 230 496 q 297 567 263 538 q 360 615 330 596 q 414 642 389 634 q 459 651 440 651 q 543 629 504 651 q 609 566 581 607 q 653 466 637 525 q 669 333 669 407 "
    },
    "]": {
      "x_min": 27.640625,
      "x_max": 326,
      "ha": 420,
      "o": "m 50 -227 l 28 -207 q 35 -174 30 -191 q 45 -149 40 -157 l 233 -149 l 233 1007 l 50 1007 l 27 1024 q 35 1056 29 1039 q 45 1085 40 1073 l 326 1085 l 326 -227 l 50 -227 "
    },
    "Ç‘": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 445 944 l 374 944 l 198 1186 q 206 1197 203 1192 q 212 1205 209 1202 q 219 1211 215 1209 q 228 1217 223 1214 l 411 1047 l 590 1217 q 606 1205 602 1211 q 619 1186 611 1199 l 445 944 "
    },
    "áº": {
      "x_min": 13,
      "x_max": 947,
      "ha": 961,
      "o": "m 947 601 q 914 592 926 597 q 895 583 902 588 q 885 573 888 579 q 880 559 882 567 l 733 40 q 718 14 729 25 q 692 -2 706 4 q 664 -13 678 -9 q 642 -20 651 -17 l 484 439 l 348 40 q 332 14 343 24 q 307 -3 321 3 q 281 -14 294 -10 q 259 -20 268 -17 l 85 559 q 13 601 79 586 l 13 631 l 270 631 l 270 601 q 218 594 237 598 q 192 583 200 589 q 184 572 184 578 q 186 559 184 565 l 311 129 l 475 631 l 521 631 l 693 129 l 808 559 q 794 584 812 575 q 731 601 776 594 l 731 631 l 947 631 l 947 601 m 574 731 q 556 718 565 723 q 537 710 548 713 l 303 965 l 317 993 q 337 998 324 995 q 364 1003 350 1000 q 392 1007 378 1006 q 414 1010 406 1009 l 574 731 "
    },
    "Èž": {
      "x_min": 33,
      "x_max": 844,
      "ha": 890,
      "o": "m 33 0 l 33 29 q 104 49 78 38 q 130 70 130 60 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 478 l 641 478 l 641 783 q 617 803 641 791 q 545 825 592 815 l 545 854 l 844 854 l 844 825 q 772 804 798 815 q 747 783 747 792 l 747 70 q 771 50 747 62 q 844 29 795 38 l 844 0 l 545 0 l 545 29 q 615 49 590 38 q 641 70 641 60 l 641 417 l 236 417 l 236 70 q 259 50 236 62 q 332 29 283 38 l 332 0 l 33 0 m 474 943 l 403 943 l 228 1151 q 236 1162 232 1157 q 242 1170 239 1167 q 248 1176 245 1174 q 257 1183 252 1179 l 440 1039 l 620 1183 q 636 1170 631 1176 q 648 1151 640 1164 l 474 943 "
    },
    "á»‡": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 527 237 552 q 177 468 192 503 q 156 393 162 434 l 440 393 q 461 398 456 393 q 466 417 466 403 q 461 463 466 435 q 441 517 456 490 q 399 561 426 543 q 329 580 372 580 m 575 377 q 543 352 563 363 q 501 332 523 340 l 153 332 q 166 231 153 279 q 205 147 180 183 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 128 q 561 146 555 155 q 571 132 567 137 q 492 52 526 82 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 233 617 214 607 q 271 634 251 627 q 310 646 291 642 q 348 651 330 651 q 425 638 391 651 q 484 605 458 626 q 526 557 509 585 q 554 499 543 530 q 570 437 565 469 q 575 377 575 405 m 399 -189 q 393 -225 399 -208 q 376 -254 387 -242 q 352 -274 366 -267 q 321 -282 338 -282 q 278 -265 291 -282 q 266 -217 266 -248 q 271 -181 266 -198 q 288 -151 277 -164 q 312 -132 299 -139 q 342 -125 326 -125 q 399 -189 399 -125 m 548 740 q 535 721 540 727 q 519 710 531 715 l 338 891 l 158 710 q 149 715 152 712 q 142 721 145 718 q 136 729 139 724 q 128 740 133 734 l 303 997 l 375 997 l 548 740 "
    },
    "Ä­": {
      "x_min": -26.09375,
      "x_max": 406.109375,
      "ha": 370,
      "o": "m 47 0 l 47 29 q 115 49 92 38 q 139 70 139 61 l 139 454 q 139 510 139 488 q 130 543 139 531 q 101 560 121 555 q 47 569 82 566 l 47 596 q 90 606 67 600 q 136 619 113 612 q 179 634 158 626 q 217 651 200 642 l 240 651 l 240 70 q 262 50 240 62 q 333 29 284 38 l 333 0 l 47 0 m 406 922 q 361 829 386 867 q 308 768 336 792 q 251 735 281 745 q 191 725 221 725 q 128 735 158 725 q 69 768 97 745 q 18 829 42 792 q -26 922 -6 867 q -10 940 -18 934 q 9 952 -2 947 q 47 887 25 914 q 92 845 68 861 q 141 822 116 829 q 188 815 166 815 q 237 822 211 815 q 287 845 263 829 q 333 887 311 861 q 370 952 354 914 q 390 940 382 947 q 406 922 398 934 "
    },
    "8": {
      "x_min": 62,
      "x_max": 576,
      "ha": 638,
      "o": "m 182 644 q 196 592 182 614 q 234 553 210 570 q 289 521 258 536 q 353 492 320 506 q 422 563 402 527 q 443 636 443 599 q 432 697 443 671 q 403 740 422 723 q 358 765 385 757 q 299 774 331 774 q 247 762 269 774 q 210 732 225 750 q 189 690 196 713 q 182 644 182 667 m 470 210 q 454 286 470 254 q 413 341 438 318 q 355 381 387 364 q 287 414 322 398 q 233 367 255 390 q 196 321 210 344 q 174 271 181 297 q 168 214 168 244 q 179 148 168 178 q 212 95 190 117 q 263 60 233 73 q 332 48 294 48 q 395 62 368 48 q 438 99 421 76 q 462 151 454 122 q 470 210 470 180 m 576 239 q 554 139 576 186 q 494 56 532 92 q 406 0 456 21 q 299 -20 355 -20 q 198 -2 242 -20 q 123 45 154 15 q 77 113 93 74 q 62 193 62 151 q 74 264 62 230 q 109 329 86 298 q 164 387 132 360 q 236 437 196 414 q 182 466 207 450 q 137 504 156 483 q 107 554 118 526 q 96 620 96 582 q 114 707 96 667 q 164 776 132 747 q 240 821 196 805 q 333 838 283 838 q 422 824 383 838 q 488 787 461 810 q 529 730 515 763 q 543 657 543 696 q 532 603 543 629 q 504 554 522 578 q 461 509 486 531 q 407 466 436 487 q 470 430 440 450 q 524 383 501 410 q 562 321 548 356 q 576 239 576 285 "
    },
    "á»®": {
      "x_min": 33,
      "x_max": 930,
      "ha": 930,
      "o": "m 930 944 q 921 904 930 927 q 892 856 913 881 q 837 803 872 830 q 751 749 803 775 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 804 855 q 816 889 816 873 q 806 926 816 908 q 777 959 796 943 l 895 1014 q 920 981 910 999 q 930 944 930 963 m 676 1103 q 648 1048 664 1075 q 611 999 631 1020 q 566 963 590 977 q 516 950 542 950 q 472 961 493 950 q 431 988 451 973 q 392 1015 411 1003 q 353 1027 372 1027 q 304 1004 326 1027 q 258 944 283 982 l 223 958 q 251 1013 235 986 q 288 1062 267 1041 q 332 1098 309 1084 q 382 1112 356 1112 q 429 1100 407 1112 q 472 1073 452 1088 q 510 1046 492 1058 q 544 1035 528 1035 q 593 1056 569 1035 q 640 1118 618 1078 l 676 1103 "
    },
    "R": {
      "x_min": 27.015625,
      "x_max": 756.890625,
      "ha": 764,
      "o": "m 33 0 l 33 29 q 104 49 78 38 q 130 70 130 60 l 130 806 q 82 799 106 803 q 33 792 57 795 l 27 834 q 90 849 58 842 q 156 862 122 856 q 229 871 191 868 q 311 875 267 875 q 445 859 388 875 q 540 816 502 843 q 597 750 578 788 q 616 669 616 712 q 602 581 616 620 q 563 510 588 541 q 504 458 538 480 q 428 424 469 437 l 629 94 q 651 69 639 78 q 677 56 663 60 q 709 52 691 52 q 749 54 726 52 l 756 25 q 673 0 713 10 q 608 -9 633 -9 q 570 1 588 -9 q 542 27 552 12 l 341 408 q 324 407 332 407 l 306 407 q 271 408 288 407 q 236 414 254 410 l 236 70 q 259 50 236 62 q 332 29 283 38 l 332 0 l 33 0 m 287 817 q 236 817 261 817 l 236 469 q 267 465 253 465 q 295 465 280 465 q 456 511 400 465 q 512 648 512 558 q 500 716 512 685 q 459 769 487 747 q 389 804 431 792 q 287 817 346 817 "
    },
    "á¸†": {
      "x_min": 27.015625,
      "x_max": 675,
      "ha": 748,
      "o": "m 273 817 q 254 817 264 817 q 235 817 244 817 l 235 492 l 257 492 q 380 507 333 492 q 453 546 427 522 q 488 599 479 569 q 497 658 497 629 q 486 721 497 692 q 450 771 476 750 q 381 804 424 792 q 273 817 338 817 m 344 441 q 285 437 312 441 q 235 430 258 434 l 235 70 q 240 60 235 64 q 266 53 252 55 q 296 49 281 50 q 327 48 312 48 q 356 48 342 48 q 442 60 404 48 q 509 95 481 73 q 551 149 536 117 q 566 218 566 180 q 553 295 566 256 q 513 367 540 334 q 444 420 486 399 q 344 441 402 441 m 675 243 q 652 140 675 186 q 590 62 630 94 q 493 12 549 29 q 370 -5 438 -5 q 335 -5 356 -5 q 292 -5 315 -5 q 244 -5 269 -5 q 196 -5 219 -5 q 81 -5 141 -5 l 33 -5 l 33 28 q 103 49 78 38 q 129 70 129 60 l 129 806 q 79 799 103 803 q 33 791 55 796 l 27 834 q 87 848 52 841 q 159 861 121 855 q 236 871 197 867 q 311 875 275 875 q 431 862 377 875 q 522 826 485 849 q 580 768 560 802 q 601 691 601 734 q 567 566 601 618 q 474 490 533 513 q 553 460 517 482 q 617 405 590 438 q 659 331 644 372 q 675 243 675 290 m 575 -155 q 570 -172 573 -162 q 563 -192 567 -182 q 555 -211 559 -201 q 549 -227 551 -220 l 140 -227 l 125 -210 q 130 -193 127 -202 q 137 -173 133 -183 q 145 -154 141 -164 q 152 -139 149 -145 l 560 -139 l 575 -155 "
    },
    "Å»": {
      "x_min": 38.828125,
      "x_max": 654,
      "ha": 711,
      "o": "m 654 198 q 654 150 654 177 q 654 97 654 124 q 654 45 654 70 q 654 0 654 19 l 58 0 l 38 30 l 511 787 l 206 787 q 183 779 195 787 q 158 755 170 772 q 135 714 146 739 q 117 653 125 688 l 80 661 l 99 865 q 132 858 117 861 q 161 855 147 855 q 191 855 175 855 l 634 855 l 651 825 l 183 68 l 530 68 q 555 74 544 68 q 577 96 567 80 q 598 139 587 112 q 621 208 608 166 l 654 198 m 435 1044 q 429 1008 435 1025 q 412 979 423 991 q 388 959 402 966 q 357 952 374 952 q 314 968 327 952 q 302 1016 302 985 q 307 1052 302 1035 q 324 1082 313 1069 q 348 1101 335 1094 q 378 1109 362 1109 q 435 1044 435 1109 "
    },
    "á¸": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 442 -154 q 430 -203 442 -180 q 394 -245 418 -226 q 330 -278 369 -264 q 237 -300 291 -292 l 222 -267 q 281 -252 257 -261 q 320 -232 305 -243 q 343 -209 336 -221 q 350 -186 350 -197 q 332 -154 350 -163 q 274 -141 315 -145 q 281 -122 275 -139 q 294 -77 285 -108 q 313 -20 302 -55 l 308 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 291 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 493 52 526 83 q 430 6 459 22 q 371 -14 400 -9 q 366 -15 368 -15 l 348 -70 q 383 -80 366 -74 q 413 -97 400 -87 q 434 -121 426 -107 q 442 -154 442 -135 m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 554 923 q 509 830 534 868 q 457 769 484 793 q 399 736 429 746 q 340 726 369 726 q 276 736 307 726 q 218 769 246 746 q 167 830 191 793 q 123 923 142 868 q 138 941 130 935 q 158 953 146 948 q 196 888 174 915 q 241 846 218 862 q 290 823 265 830 q 337 816 315 816 q 386 823 360 816 q 435 846 411 830 q 481 888 460 862 q 519 953 503 915 q 538 941 530 948 q 554 923 546 935 "
    },
    "Ãµ": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 581 913 q 553 858 569 885 q 516 809 536 830 q 472 773 496 787 q 422 760 448 760 q 378 771 399 760 q 337 798 357 783 q 297 825 317 813 q 258 837 278 837 q 210 814 232 837 q 163 754 188 792 l 128 768 q 157 823 140 796 q 193 872 173 851 q 238 908 214 894 q 287 922 261 922 q 335 910 312 922 q 377 883 357 898 q 415 856 397 868 q 449 845 433 845 q 499 866 474 845 q 545 928 523 888 l 581 913 "
    },
    "áº˜": {
      "x_min": 13,
      "x_max": 947,
      "ha": 961,
      "o": "m 947 601 q 914 592 926 597 q 895 583 902 588 q 885 573 888 579 q 880 559 882 567 l 733 40 q 718 14 729 25 q 692 -2 706 4 q 664 -13 678 -9 q 642 -20 651 -17 l 484 439 l 347 40 q 331 14 342 24 q 307 -3 320 3 q 281 -14 294 -10 q 259 -20 268 -17 l 85 559 q 13 601 79 586 l 13 631 l 270 631 l 270 601 q 218 594 237 598 q 192 583 200 589 q 184 572 184 578 q 186 559 184 565 l 311 129 l 474 631 l 520 631 l 693 129 l 808 559 q 794 584 812 575 q 731 601 776 594 l 731 631 l 947 631 l 947 601 m 562 843 q 558 874 562 859 q 546 900 554 889 q 528 918 539 911 q 504 925 518 925 q 478 919 490 925 q 455 903 465 913 q 439 877 445 892 q 434 844 434 862 q 438 814 434 828 q 449 788 442 799 q 467 770 456 777 q 491 764 478 764 q 517 769 504 764 q 540 784 530 774 q 556 808 550 794 q 562 843 562 823 m 622 870 q 609 807 622 836 q 575 757 596 778 q 528 723 554 735 q 477 712 502 712 q 435 720 454 712 q 402 742 416 728 q 381 775 389 756 q 374 816 374 794 q 386 880 374 851 q 420 931 399 910 q 466 965 441 953 q 518 977 492 977 q 558 968 540 977 q 591 945 577 960 q 614 911 606 930 q 622 870 622 892 "
    },
    "áº«": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 529 740 q 516 721 521 727 q 500 710 512 715 l 318 891 l 138 710 q 129 715 132 712 q 122 721 125 718 q 116 729 119 724 q 108 740 112 734 l 283 997 l 354 997 l 529 740 m 545 1218 q 517 1163 533 1190 q 480 1114 500 1135 q 436 1078 460 1092 q 386 1065 412 1065 q 341 1076 362 1065 q 300 1103 320 1088 q 261 1130 281 1118 q 222 1142 241 1142 q 173 1119 195 1142 q 127 1059 152 1097 l 92 1073 q 120 1128 104 1101 q 157 1177 136 1156 q 201 1213 177 1199 q 251 1227 225 1227 q 298 1215 276 1227 q 341 1188 321 1203 q 379 1161 361 1173 q 413 1150 397 1150 q 463 1171 438 1150 q 510 1233 487 1193 l 545 1218 "
    },
    "á¹ ": {
      "x_min": 80,
      "x_max": 589,
      "ha": 649,
      "o": "m 589 255 q 581 193 589 225 q 556 130 573 161 q 516 72 540 99 q 458 24 491 45 q 384 -7 426 4 q 293 -20 342 -20 q 245 -15 271 -20 q 191 -2 219 -10 q 137 18 164 6 q 87 46 110 30 q 81 69 82 48 q 80 116 80 89 q 80 172 80 144 q 87 219 80 201 l 114 216 q 152 147 129 176 q 203 98 175 117 q 263 70 231 79 q 330 61 296 61 q 390 73 360 61 q 444 107 420 86 q 483 158 468 129 q 499 219 499 187 q 481 290 499 261 q 434 343 463 320 q 367 385 405 366 q 291 421 330 403 q 215 460 252 440 q 149 507 178 480 q 102 570 120 535 q 85 655 85 606 q 90 701 85 676 q 110 750 96 725 q 144 797 123 774 q 195 837 165 820 q 263 864 224 854 q 350 875 301 875 q 410 870 380 875 q 466 857 440 865 q 514 837 493 849 q 547 812 535 826 q 547 796 552 808 q 534 770 543 784 q 514 743 525 756 q 495 722 503 729 l 471 726 q 432 763 453 748 q 391 789 412 779 q 349 802 370 798 q 310 807 329 807 q 247 795 273 807 q 203 766 220 783 q 178 727 186 749 q 171 687 171 706 q 188 631 171 655 q 236 585 206 606 q 302 546 265 564 q 379 508 340 528 q 456 465 419 488 q 523 412 494 442 q 571 344 553 382 q 589 255 589 305 m 399 1045 q 393 1009 399 1026 q 376 980 387 992 q 352 960 366 967 q 321 953 338 953 q 278 969 291 953 q 266 1017 266 986 q 271 1053 266 1036 q 288 1083 277 1070 q 312 1102 299 1095 q 342 1110 326 1110 q 399 1045 399 1110 "
    },
    "Ç": {
      "x_min": 47,
      "x_max": 565,
      "ha": 618,
      "o": "m 295 51 q 412 103 370 51 q 463 258 454 156 l 197 258 q 164 242 176 258 q 153 199 153 226 q 165 140 153 167 q 198 93 178 113 q 244 62 218 73 q 295 51 269 51 m 565 329 q 538 180 565 249 q 466 61 512 111 q 382 0 428 19 q 276 -20 336 -20 q 187 -4 228 -20 q 114 40 145 11 q 64 106 82 68 q 47 189 47 144 q 53 239 47 216 q 70 276 59 261 q 113 298 90 287 q 159 319 136 309 l 464 319 q 447 419 464 374 q 404 495 431 464 q 341 543 377 526 q 264 560 304 560 q 227 557 246 560 q 187 546 209 554 q 138 523 165 538 q 78 482 111 507 q 62 496 69 486 q 52 511 56 506 q 127 582 92 555 q 192 625 161 610 q 252 646 222 641 q 311 652 281 652 q 405 629 360 652 q 486 566 451 607 q 543 464 522 525 q 565 329 565 404 "
    },
    "Ë™": {
      "x_min": 66,
      "x_max": 199,
      "ha": 265,
      "o": "m 199 854 q 193 818 199 835 q 176 789 187 801 q 152 769 166 776 q 121 762 138 762 q 78 778 91 762 q 66 826 66 795 q 71 862 66 845 q 88 892 77 879 q 112 911 99 904 q 142 919 126 919 q 199 854 199 919 "
    },
    "Ãª": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 290 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 548 740 q 535 721 540 727 q 519 710 531 715 l 338 891 l 158 710 q 149 715 152 712 q 142 721 145 718 q 136 729 139 724 q 128 740 133 734 l 303 997 l 374 997 l 548 740 "
    },
    "â€ž": {
      "x_min": 58.59375,
      "x_max": 531,
      "ha": 611,
      "o": "m 251 41 q 242 -15 251 15 q 215 -76 233 -46 q 173 -135 198 -107 q 117 -184 149 -163 l 87 -161 q 111 -127 101 -146 q 129 -88 122 -107 q 139 -51 136 -69 q 143 -20 143 -33 q 125 33 143 10 q 69 56 107 56 l 58 87 q 83 106 64 95 q 126 126 103 116 q 171 142 149 136 q 205 149 193 149 q 239 104 227 127 q 251 41 251 80 m 531 41 q 522 -15 531 15 q 495 -76 513 -46 q 454 -135 478 -107 q 397 -184 429 -163 l 369 -161 q 393 -127 383 -146 q 410 -88 403 -107 q 421 -51 417 -69 q 425 -20 425 -33 q 406 33 425 10 q 349 56 387 56 l 339 87 q 363 106 344 95 q 406 126 383 116 q 451 142 429 136 q 485 149 474 149 q 519 104 507 127 q 531 41 531 80 "
    },
    "Ã‚": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 839 q 390 869 367 856 q 430 893 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 608 957 q 595 938 600 944 q 579 927 590 932 l 397 1068 l 216 927 q 208 932 211 929 q 201 938 204 935 q 195 946 198 942 q 187 957 191 951 l 362 1167 l 434 1167 l 608 957 "
    },
    "Â´": {
      "x_min": 144.21875,
      "x_max": 411.390625,
      "ha": 434,
      "o": "m 182 709 q 161 717 170 712 q 144 728 152 722 l 299 1009 q 321 1007 307 1009 q 348 1003 334 1005 q 375 998 362 1000 q 396 994 388 996 l 411 967 l 182 709 "
    },
    "Æ": {
      "x_min": 53,
      "x_max": 639.65625,
      "ha": 692,
      "o": "m 639 145 q 560 67 599 98 q 481 16 522 35 q 397 -11 441 -3 q 303 -20 353 -20 q 195 0 242 -20 q 117 52 148 20 q 69 121 85 83 q 53 192 53 158 q 65 268 53 231 q 100 337 78 305 q 151 392 122 368 q 210 431 179 416 q 110 500 146 454 q 75 614 75 547 q 106 725 75 673 q 198 815 138 778 q 289 858 241 842 q 390 875 336 875 q 523 858 462 875 q 627 804 583 841 q 600 761 615 781 q 567 722 584 742 l 538 727 q 497 759 517 746 q 455 782 477 773 q 409 796 433 791 q 355 800 384 800 q 284 787 318 800 q 225 753 251 775 q 183 701 199 731 q 168 635 168 670 q 178 580 168 607 q 217 531 189 553 q 294 494 245 509 q 419 480 343 480 l 419 425 q 301 405 350 425 q 220 358 252 386 q 173 298 188 330 q 159 234 159 265 q 174 166 159 197 q 217 113 190 135 q 281 77 244 90 q 360 65 317 65 q 420 68 393 65 q 475 84 447 72 q 535 117 503 96 q 611 175 567 139 l 639 145 "
    },
    "á»": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 467 904 q 454 871 467 885 q 426 842 442 856 q 393 817 409 829 q 371 793 378 805 q 370 767 363 780 q 404 736 376 753 q 384 728 394 730 q 363 726 373 727 q 319 753 333 740 q 302 776 305 765 q 305 797 299 787 q 323 815 311 806 q 348 833 334 824 q 374 851 362 842 q 394 871 386 861 q 403 894 403 882 q 390 933 403 920 q 358 946 377 946 q 340 942 348 946 q 325 932 331 938 q 316 918 319 926 q 313 904 313 911 q 314 896 313 900 q 318 889 316 893 q 304 885 313 887 q 286 881 296 883 q 268 877 277 879 q 252 874 258 875 l 245 882 l 245 891 q 257 923 245 907 q 289 951 270 938 q 333 971 309 963 q 380 979 356 979 q 444 958 421 979 q 467 904 467 938 "
    },
    "Êƒ": {
      "x_min": -173,
      "x_max": 474,
      "ha": 335,
      "o": "m 259 44 q 248 -70 259 -21 q 219 -157 237 -119 q 177 -222 201 -194 q 127 -271 153 -249 q 87 -297 109 -285 q 42 -319 66 -310 q -4 -333 19 -328 q -48 -339 -27 -339 q -90 -331 -68 -339 q -130 -315 -112 -324 q -161 -296 -149 -305 q -173 -280 -173 -286 q -163 -264 -173 -274 q -140 -242 -153 -253 q -112 -220 -127 -230 q -88 -205 -98 -210 q -64 -223 -77 -215 q -36 -236 -51 -231 q -8 -244 -22 -241 q 17 -248 5 -248 q 71 -234 47 -248 q 114 -194 96 -221 q 142 -127 132 -167 q 153 -34 153 -87 q 147 83 153 19 q 134 215 142 147 q 117 352 126 283 q 99 485 107 422 q 86 604 91 549 q 81 698 81 659 q 89 805 81 761 q 112 883 97 850 q 148 941 127 917 q 196 988 170 966 q 235 1014 214 1003 q 277 1034 256 1026 q 317 1046 298 1042 q 349 1051 336 1051 q 391 1042 369 1051 q 431 1024 413 1034 q 462 1002 450 1013 q 474 985 474 991 q 464 969 474 980 q 441 947 454 958 q 413 925 428 935 q 389 910 399 915 q 357 935 373 924 q 326 952 341 945 q 298 962 311 959 q 276 966 285 966 q 243 953 259 966 q 214 917 226 940 q 194 857 202 893 q 187 776 187 822 q 192 661 187 724 q 205 529 197 597 q 223 391 213 461 q 240 258 232 322 q 253 139 248 194 q 259 44 259 83 "
    },
    "Äˆ": {
      "x_min": 47,
      "x_max": 677.046875,
      "ha": 730,
      "o": "m 677 143 q 595 65 634 97 q 520 15 556 34 q 449 -11 484 -3 q 379 -20 414 -20 q 258 8 318 -20 q 152 90 199 36 q 75 221 104 144 q 47 397 47 299 q 79 594 47 506 q 166 744 111 682 q 294 841 221 807 q 449 875 368 875 q 575 855 521 875 q 664 807 629 835 q 662 793 668 803 q 646 770 656 783 q 624 744 635 757 q 604 723 612 731 l 581 727 q 500 780 547 759 q 393 800 454 800 q 344 791 371 800 q 290 765 317 783 q 237 717 263 746 q 192 645 212 687 q 161 548 173 603 q 150 422 150 493 q 175 264 150 332 q 241 151 201 196 q 331 83 281 106 q 427 61 380 61 q 521 86 463 61 q 652 173 579 111 q 659 167 655 172 q 666 158 663 163 q 672 149 669 153 q 677 143 675 145 m 618 957 q 605 938 610 944 q 589 927 601 932 l 407 1068 l 227 927 q 218 932 222 929 q 212 938 215 935 q 205 946 209 942 q 197 957 202 951 l 373 1167 l 444 1167 l 618 957 "
    },
    "ÉŠ": {
      "x_min": 44,
      "x_max": 965,
      "ha": 817,
      "o": "m 361 71 q 417 84 388 71 q 477 119 447 98 q 537 169 508 141 q 594 224 567 197 l 594 671 q 523 753 568 724 q 421 783 478 783 q 359 772 391 783 q 296 742 327 762 q 238 693 266 722 q 189 624 210 663 q 156 537 168 585 q 144 432 144 489 q 162 288 144 355 q 212 174 181 222 q 282 98 243 125 q 361 71 321 71 m 594 152 q 532 79 562 111 q 469 24 501 47 q 401 -9 436 2 q 325 -21 365 -21 q 263 -8 295 -21 q 200 28 231 4 q 140 89 168 53 q 90 173 112 126 q 56 278 69 221 q 44 404 44 336 q 55 505 44 457 q 87 595 67 553 q 131 672 106 637 q 181 735 155 707 q 231 783 206 763 q 274 815 255 803 q 321 840 297 830 q 367 859 345 851 q 410 871 390 867 q 445 875 430 875 q 486 872 466 875 q 527 861 507 869 q 570 836 548 852 q 617 795 592 821 q 639 812 628 803 q 660 833 650 822 q 679 854 670 843 q 696 873 688 864 l 724 847 q 712 792 717 822 q 705 730 708 766 q 702 653 702 695 l 702 -72 q 702 -162 702 -124 q 710 -226 702 -201 q 733 -263 718 -251 q 777 -275 749 -275 q 805 -269 792 -275 q 828 -254 818 -264 q 844 -233 838 -245 q 850 -207 850 -221 q 842 -179 850 -194 q 853 -165 842 -173 q 883 -150 865 -158 q 919 -138 900 -143 q 952 -133 938 -133 l 965 -159 q 943 -224 965 -192 q 888 -281 922 -256 q 813 -323 855 -307 q 728 -339 770 -339 q 659 -323 685 -339 q 618 -278 633 -307 q 598 -207 603 -249 q 594 -114 594 -166 l 594 152 "
    },
    "á»œ": {
      "x_min": 47,
      "x_max": 811,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 811 944 q 803 906 811 927 q 778 861 796 885 q 730 812 760 837 q 654 761 700 786 q 731 614 705 698 q 757 438 757 529 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 532 853 482 875 q 622 796 582 832 q 679 846 661 821 q 697 889 697 871 q 687 926 697 908 q 658 959 677 944 l 776 1014 q 801 981 791 999 q 811 944 811 963 m 527 957 q 512 938 517 944 q 497 927 507 931 l 185 1092 l 190 1122 q 205 1134 195 1127 q 227 1148 215 1141 q 251 1162 239 1155 q 269 1172 262 1168 l 527 957 "
    },
    "Î©": {
      "x_min": 54,
      "x_max": 780,
      "ha": 860,
      "o": "m 73 0 l 54 18 q 54 55 54 34 q 57 99 54 76 q 64 145 60 122 q 73 189 68 168 l 103 189 q 114 139 108 161 q 126 103 119 118 q 143 81 133 88 q 167 74 153 74 l 313 74 q 195 185 242 132 q 122 290 149 238 q 83 394 94 342 q 73 501 73 446 q 83 590 73 545 q 116 677 94 635 q 169 754 138 718 q 242 817 201 790 q 335 859 284 844 q 445 875 385 875 q 579 849 518 875 q 684 778 640 824 q 752 665 727 731 q 777 516 777 599 q 761 402 777 456 q 713 295 745 347 q 633 188 681 242 q 523 74 586 134 l 676 74 q 699 81 690 74 q 715 101 708 87 q 731 137 723 115 q 751 192 739 160 l 780 179 q 777 135 780 160 q 772 87 775 111 q 765 40 769 63 q 759 0 762 17 l 465 0 l 465 74 q 534 166 503 126 q 587 242 564 207 q 626 307 610 276 q 652 367 642 337 q 666 430 662 398 q 671 501 671 462 q 655 619 671 563 q 608 719 639 675 q 531 788 577 762 q 425 814 485 814 q 344 798 380 814 q 281 757 308 782 q 235 699 254 731 q 205 632 217 666 q 187 566 193 598 q 182 508 182 533 q 182 434 182 468 q 190 370 182 400 q 212 308 198 339 q 249 242 226 277 q 304 166 272 207 q 382 74 337 126 l 382 0 l 73 0 "
    },
    "È§": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 353 44 383 66 q 295 7 323 21 q 241 -14 266 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 148 302 128 288 q 202 330 169 317 q 286 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 280 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 165 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 385 854 q 379 818 385 835 q 362 789 373 801 q 338 769 352 776 q 307 762 324 762 q 264 778 277 762 q 252 826 252 795 q 257 862 252 845 q 274 892 263 879 q 298 911 285 904 q 328 919 312 919 q 385 854 385 919 "
    },
    "Ã–": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 609 1045 q 603 1009 609 1026 q 586 980 597 992 q 562 960 576 967 q 531 953 548 953 q 488 969 501 953 q 476 1017 476 986 q 481 1053 476 1036 q 498 1083 487 1070 q 522 1102 509 1095 q 552 1110 536 1110 q 609 1045 609 1110 m 343 1045 q 337 1009 343 1026 q 320 980 331 992 q 296 960 310 967 q 265 953 282 953 q 222 969 235 953 q 210 1017 210 986 q 215 1053 210 1036 q 232 1083 221 1070 q 256 1102 243 1095 q 286 1110 270 1110 q 343 1045 343 1110 "
    },
    "á¸": {
      "x_min": 53,
      "x_max": 698.015625,
      "ha": 707,
      "o": "m 698 57 q 643 21 667 36 q 601 -2 619 7 q 570 -15 583 -11 q 548 -20 557 -20 q 512 10 526 -20 q 498 114 498 40 q 446 58 471 83 q 394 16 421 33 q 339 -10 368 -1 q 276 -20 310 -20 q 199 2 239 -20 q 127 65 160 24 q 73 166 94 106 q 53 301 53 226 q 63 381 53 339 q 92 461 73 422 q 140 534 112 499 q 202 595 167 568 q 278 636 237 621 q 366 651 319 651 q 428 643 397 651 q 495 608 459 635 l 495 864 q 493 924 495 902 q 481 958 491 946 q 453 974 472 970 q 398 979 433 979 l 398 1006 q 496 1026 453 1014 q 573 1050 539 1039 l 595 1030 l 595 172 q 595 131 595 148 q 597 103 595 114 q 601 84 599 91 q 608 72 604 76 q 632 67 614 64 q 689 86 649 70 l 698 57 m 495 177 l 495 495 q 433 555 473 533 q 345 576 393 576 q 267 560 303 576 q 207 512 232 544 q 167 433 181 480 q 153 322 153 385 q 169 217 153 264 q 210 137 185 170 q 266 85 236 103 q 323 68 296 68 q 368 77 346 68 q 411 102 390 86 q 453 137 432 117 q 495 177 474 156 m 573 -155 q 568 -172 571 -162 q 561 -192 565 -182 q 553 -211 557 -201 q 547 -227 549 -220 l 139 -227 l 124 -210 q 128 -193 125 -202 q 136 -173 132 -183 q 144 -154 140 -164 q 151 -139 148 -145 l 558 -139 l 573 -155 "
    },
    "z": {
      "x_min": 50.453125,
      "x_max": 549.984375,
      "ha": 602,
      "o": "m 548 168 q 548 121 548 146 q 548 74 548 97 q 548 31 548 51 q 543 0 548 12 l 67 0 l 50 30 l 419 569 l 189 569 q 167 564 178 569 q 146 549 156 560 q 127 518 136 538 q 111 467 118 498 l 85 475 l 85 641 q 117 634 105 637 q 142 631 129 631 q 170 631 155 631 l 535 631 l 549 601 l 179 61 l 460 61 q 490 87 476 61 q 520 174 504 113 l 548 168 "
    },
    "á¸„": {
      "x_min": 27.015625,
      "x_max": 675,
      "ha": 748,
      "o": "m 273 817 q 254 817 264 817 q 235 817 244 817 l 235 492 l 257 492 q 380 507 333 492 q 453 546 427 522 q 488 599 479 569 q 497 658 497 629 q 486 721 497 692 q 450 771 476 750 q 381 804 424 792 q 273 817 338 817 m 344 441 q 285 437 312 441 q 235 430 258 434 l 235 70 q 241 60 235 64 q 267 53 252 55 q 296 49 281 50 q 327 48 312 48 q 356 48 342 48 q 442 60 404 48 q 509 95 481 73 q 551 149 536 117 q 566 218 566 180 q 553 295 566 256 q 513 367 540 334 q 444 420 486 399 q 344 441 402 441 m 675 243 q 652 140 675 186 q 590 62 630 94 q 493 12 549 29 q 370 -5 438 -5 q 335 -5 356 -5 q 292 -5 315 -5 q 244 -5 269 -5 q 196 -5 219 -5 q 81 -5 141 -5 l 33 -5 l 33 28 q 103 49 78 38 q 129 70 129 60 l 129 806 q 79 799 103 803 q 33 791 55 796 l 27 834 q 87 848 52 841 q 159 861 121 855 q 236 871 197 867 q 311 875 275 875 q 431 862 377 875 q 522 826 485 849 q 580 768 560 802 q 601 691 601 734 q 567 566 601 618 q 474 490 533 513 q 553 460 517 482 q 617 405 590 438 q 659 331 644 372 q 675 243 675 290 m 417 -189 q 411 -225 417 -208 q 394 -254 405 -242 q 370 -274 384 -267 q 339 -282 356 -282 q 296 -265 309 -282 q 284 -217 284 -248 q 289 -181 284 -198 q 306 -151 295 -164 q 330 -132 317 -139 q 360 -125 344 -125 q 417 -189 417 -125 "
    },
    "â„¢": {
      "x_min": 40.015625,
      "x_max": 721,
      "ha": 761,
      "o": "m 116 547 l 116 564 q 147 574 139 569 q 155 582 155 578 l 155 827 l 81 827 q 74 825 77 827 q 68 819 71 823 q 61 806 65 814 q 52 783 57 797 l 40 787 q 40 802 40 793 q 41 821 40 811 q 44 841 42 831 q 48 855 46 850 l 308 855 l 316 848 q 315 833 315 842 q 313 815 315 824 q 311 797 312 805 q 308 783 310 789 l 297 783 q 288 815 291 804 q 273 827 284 827 l 203 827 l 203 582 q 210 574 203 579 q 243 564 217 569 l 243 547 l 116 547 m 716 838 q 686 832 702 838 l 686 579 q 694 572 686 576 q 721 564 703 567 l 721 547 l 604 547 l 604 564 q 630 572 621 567 q 640 579 640 576 l 640 787 l 532 548 l 513 548 l 404 786 l 404 579 q 411 572 404 576 q 437 564 419 567 l 437 548 l 333 548 l 333 564 q 360 572 351 567 q 369 579 369 576 l 369 831 q 352 836 360 835 q 337 838 344 838 l 337 855 l 413 855 q 420 851 418 855 q 429 833 423 847 l 528 622 l 623 833 q 632 851 629 848 q 640 855 634 855 l 716 855 l 716 838 "
    },
    "áº·": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 67 q 318 88 273 67 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 233 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 70 200 74 q 233 67 225 67 m 615 55 q 522 -2 559 15 q 466 -20 485 -20 q 429 10 444 -20 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -13 267 -6 q 194 -20 215 -20 q 145 -11 170 -20 q 99 13 120 -3 q 66 59 79 31 q 53 126 53 87 q 70 212 53 177 q 112 272 88 246 q 149 302 128 288 q 202 330 169 317 q 286 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 280 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 165 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 372 -189 q 366 -225 372 -208 q 349 -254 360 -242 q 325 -274 339 -267 q 294 -282 311 -282 q 251 -265 264 -282 q 239 -217 239 -248 q 244 -181 239 -198 q 261 -151 250 -164 q 285 -132 272 -139 q 315 -125 299 -125 q 372 -189 372 -125 m 535 922 q 490 829 515 867 q 437 768 465 792 q 380 735 410 745 q 320 725 350 725 q 257 735 287 725 q 198 768 226 745 q 147 829 171 792 q 102 922 122 867 q 118 940 110 934 q 138 952 126 947 q 176 887 154 914 q 221 845 197 861 q 270 822 245 829 q 317 815 295 815 q 366 822 341 815 q 416 845 392 829 q 462 887 440 861 q 500 952 484 914 q 519 940 511 947 q 535 922 527 934 "
    },
    "Å˜": {
      "x_min": 27.015625,
      "x_max": 756.890625,
      "ha": 764,
      "o": "m 33 0 l 33 29 q 104 49 78 38 q 130 70 130 60 l 130 806 q 82 799 106 803 q 33 792 57 795 l 27 834 q 90 849 58 842 q 156 862 122 856 q 229 871 191 868 q 311 875 267 875 q 445 859 388 875 q 540 816 502 843 q 597 750 578 788 q 616 669 616 712 q 602 581 616 620 q 563 510 588 541 q 504 458 538 480 q 428 424 469 437 l 629 94 q 651 69 639 78 q 677 56 663 60 q 709 52 691 52 q 749 54 726 52 l 756 25 q 673 0 713 10 q 608 -9 633 -9 q 570 1 588 -9 q 542 27 552 12 l 341 408 q 324 407 332 407 l 306 407 q 271 408 288 407 q 236 414 254 410 l 236 70 q 259 50 236 62 q 332 29 283 38 l 332 0 l 33 0 m 287 817 q 236 817 261 817 l 236 469 q 267 465 253 465 q 295 465 280 465 q 456 511 400 465 q 512 648 512 558 q 500 716 512 685 q 459 769 487 747 q 389 804 431 792 q 287 817 346 817 m 381 943 l 310 943 l 135 1151 q 143 1162 139 1157 q 149 1170 146 1167 q 156 1176 152 1174 q 165 1183 159 1179 l 347 1039 l 526 1183 q 542 1170 538 1176 q 555 1151 547 1164 l 381 943 "
    },
    "Å‡": {
      "x_min": 33,
      "x_max": 851,
      "ha": 884,
      "o": "m 33 0 l 33 29 q 105 48 82 35 q 129 70 129 61 l 129 778 q 83 810 107 799 q 33 824 58 820 l 33 854 l 173 854 q 190 852 183 854 q 203 845 196 850 q 217 829 209 839 q 237 801 224 818 l 674 187 l 674 782 q 652 804 674 790 q 578 824 630 817 l 578 854 l 851 854 l 851 824 q 778 805 802 818 q 754 782 754 792 l 754 -20 q 701 -6 720 -16 q 674 14 682 2 l 209 673 l 209 70 q 231 49 209 62 q 305 29 253 36 l 305 0 l 33 0 m 477 943 l 406 943 l 230 1151 q 238 1162 235 1157 q 245 1170 242 1167 q 251 1176 248 1174 q 260 1183 255 1179 l 443 1039 l 623 1183 q 638 1170 634 1176 q 651 1151 643 1164 l 477 943 "
    },
    "á»«": {
      "x_min": 27.265625,
      "x_max": 834,
      "ha": 834,
      "o": "m 834 707 q 823 662 834 688 q 787 608 813 637 q 718 549 762 579 q 608 491 674 519 l 608 172 q 611 103 608 130 q 623 70 614 77 q 650 68 631 65 q 704 86 670 71 l 710 55 q 662 25 685 39 q 620 1 639 11 q 586 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 466 70 q 369 4 398 18 q 316 -14 340 -9 q 270 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 533 113 515 q 100 560 109 551 q 75 572 92 569 q 27 579 57 576 l 27 606 q 72 613 51 609 q 112 623 93 617 q 152 635 132 628 q 194 651 172 642 l 213 625 l 213 226 q 219 147 213 179 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 483 q 505 532 508 513 q 492 560 502 550 q 463 574 482 570 q 415 579 445 579 l 415 606 q 506 625 462 612 q 588 651 550 638 l 608 625 l 608 538 q 659 568 638 553 q 693 599 680 584 q 713 627 707 614 q 720 652 720 641 q 710 689 720 671 q 681 722 700 707 l 799 777 q 824 744 814 762 q 834 707 834 726 m 437 731 q 420 718 428 723 q 400 710 411 713 l 167 966 l 181 994 q 201 998 188 996 q 228 1003 214 1000 q 256 1007 242 1006 q 278 1010 270 1009 l 437 731 "
    },
    "á»¢": {
      "x_min": 47,
      "x_max": 811,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 811 944 q 803 906 811 927 q 778 861 796 885 q 730 812 760 837 q 654 761 700 786 q 731 614 705 698 q 757 438 757 529 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 532 853 482 875 q 622 796 582 832 q 679 846 661 821 q 697 889 697 871 q 687 926 697 908 q 658 959 677 944 l 776 1014 q 801 981 791 999 q 811 944 811 963 m 476 -189 q 470 -225 476 -208 q 453 -254 464 -242 q 429 -274 443 -267 q 398 -282 415 -282 q 355 -265 368 -282 q 343 -217 343 -248 q 348 -181 343 -198 q 365 -151 354 -164 q 389 -132 376 -139 q 419 -125 403 -125 q 476 -189 476 -125 "
    },
    "Æ´": {
      "x_min": -31,
      "x_max": 919,
      "ha": 673,
      "o": "m 271 601 q 228 595 245 598 q 203 586 212 591 q 193 574 194 581 q 195 555 191 566 l 359 123 l 528 599 q 593 733 557 680 q 666 818 628 786 q 741 862 704 849 q 811 875 778 875 q 850 871 831 875 q 885 863 870 868 q 909 852 900 858 q 919 839 919 845 q 911 823 919 835 q 893 797 904 811 q 867 770 881 783 q 841 748 853 756 q 816 761 827 756 q 794 767 804 765 q 774 769 784 769 l 758 769 q 716 757 738 769 q 674 724 695 745 q 634 670 653 702 q 601 598 615 639 l 360 -63 q 295 -196 331 -143 q 222 -281 260 -250 q 146 -326 184 -313 q 76 -339 109 -339 q 37 -335 56 -339 q 2 -327 17 -332 q -21 -316 -12 -322 q -31 -303 -31 -309 q -23 -287 -31 -299 q -4 -261 -16 -275 q 20 -233 6 -247 q 47 -212 33 -220 q 94 -231 73 -229 q 129 -234 115 -234 q 172 -222 150 -234 q 216 -188 194 -210 q 256 -134 237 -166 q 291 -61 276 -102 l 306 -19 l 87 555 q 64 584 81 574 q 14 601 47 594 l 14 631 l 271 631 l 271 601 "
    },
    "Ã‰": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 413 432 428 427 q 377 438 398 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 245 926 q 229 937 234 931 q 216 957 224 944 l 473 1172 q 491 1161 479 1167 q 514 1147 502 1155 q 536 1134 526 1140 q 551 1121 546 1127 l 557 1091 l 245 926 "
    },
    "á¹…": {
      "x_min": 37,
      "x_max": 732,
      "ha": 752,
      "o": "m 446 0 l 446 29 q 516 51 493 42 q 539 70 539 61 l 539 429 q 534 496 539 470 q 519 537 529 522 q 493 557 509 552 q 453 563 477 563 q 408 552 433 563 q 353 520 382 542 q 293 461 325 497 q 230 372 261 425 l 230 70 q 254 49 230 60 q 323 29 279 38 l 323 0 l 37 0 l 37 29 q 105 50 80 40 q 130 70 130 59 l 130 482 q 128 524 130 508 q 117 548 126 539 q 89 561 108 557 q 37 569 70 565 l 37 596 q 120 618 82 604 q 196 651 159 632 l 223 627 l 223 458 q 291 539 256 504 q 362 599 326 575 q 432 637 398 624 q 491 651 465 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 70 q 659 52 639 61 q 732 29 679 42 l 732 0 l 446 0 m 450 855 q 444 819 450 836 q 427 790 438 802 q 403 770 417 777 q 372 763 389 763 q 329 779 342 763 q 317 827 317 796 q 322 863 317 846 q 339 893 328 880 q 363 912 350 905 q 393 920 377 920 q 450 855 450 920 "
    },
    "Â³": {
      "x_min": 24.78125,
      "x_max": 390,
      "ha": 447,
      "o": "m 390 575 q 376 510 390 540 q 338 457 363 480 q 276 421 312 434 q 195 408 240 408 q 109 423 153 408 q 24 475 66 439 l 40 502 q 79 480 61 489 q 114 465 97 470 q 148 458 131 460 q 182 456 165 456 q 270 482 237 456 q 303 561 303 509 q 292 613 303 592 q 265 646 282 633 q 229 665 249 659 q 190 670 209 670 l 180 670 q 174 670 177 670 q 169 669 172 669 q 158 667 165 668 l 152 699 q 218 722 193 709 q 255 748 242 734 q 272 776 268 762 q 276 802 276 790 q 271 830 276 817 q 258 854 267 843 q 233 871 248 865 q 197 878 218 878 q 144 861 162 878 q 131 816 126 844 q 95 804 116 808 q 54 801 73 801 l 41 813 q 56 849 41 830 q 96 885 71 868 q 154 912 121 901 q 223 923 187 923 q 285 913 259 923 q 328 887 311 903 q 353 851 345 871 q 362 810 362 830 q 355 779 362 794 q 336 749 348 763 q 305 723 323 735 q 264 704 287 711 q 314 690 291 701 q 354 662 337 679 q 380 622 370 644 q 390 575 390 599 "
    },
    "á¹¦": {
      "x_min": 80,
      "x_max": 589,
      "ha": 649,
      "o": "m 589 255 q 581 193 589 225 q 556 130 573 161 q 516 72 540 99 q 458 24 491 45 q 384 -7 426 4 q 293 -20 342 -20 q 245 -15 271 -20 q 191 -2 219 -10 q 137 18 164 6 q 87 46 110 30 q 81 69 82 48 q 80 116 80 89 q 80 172 80 144 q 87 219 80 201 l 114 216 q 152 147 129 176 q 203 98 175 117 q 263 70 231 79 q 330 61 296 61 q 390 73 360 61 q 444 107 420 86 q 483 158 468 129 q 499 219 499 187 q 481 290 499 261 q 434 343 463 320 q 367 385 405 366 q 291 421 330 403 q 215 460 252 440 q 149 507 178 480 q 102 570 120 535 q 85 655 85 606 q 90 701 85 676 q 110 750 96 725 q 144 797 123 774 q 195 837 165 820 q 263 864 224 854 q 350 875 301 875 q 410 870 380 875 q 466 857 440 865 q 514 837 493 849 q 547 812 535 826 q 547 796 552 808 q 534 770 543 784 q 514 743 525 756 q 495 722 503 729 l 471 726 q 432 763 453 748 q 391 789 412 779 q 349 802 370 798 q 310 807 329 807 q 247 795 273 807 q 203 766 220 783 q 178 727 186 749 q 171 687 171 706 q 188 631 171 655 q 236 585 206 606 q 302 546 265 564 q 379 508 340 528 q 456 465 419 488 q 523 412 494 442 q 571 344 553 382 q 589 255 589 305 m 368 944 l 297 944 l 122 1152 q 129 1163 126 1159 q 136 1172 133 1168 q 142 1178 139 1175 q 151 1184 146 1180 l 334 1041 l 514 1184 q 530 1172 525 1178 q 543 1152 535 1165 l 368 944 m 399 1262 q 393 1226 399 1243 q 376 1197 387 1209 q 352 1177 366 1184 q 321 1170 338 1170 q 278 1186 291 1170 q 266 1234 266 1203 q 271 1270 266 1253 q 288 1300 277 1287 q 312 1319 299 1312 q 342 1327 326 1327 q 399 1262 399 1327 "
    },
    "á¹—": {
      "x_min": 37,
      "x_max": 669,
      "ha": 722,
      "o": "m 579 288 q 565 398 579 347 q 529 486 551 449 q 476 544 506 523 q 414 566 446 566 q 383 558 402 566 q 339 533 364 551 q 287 486 315 515 q 230 413 259 456 l 230 144 q 285 106 259 121 q 333 83 311 91 q 375 71 355 74 q 413 68 395 68 q 479 82 449 68 q 531 124 509 96 q 566 193 553 152 q 579 288 579 234 m 669 333 q 659 253 669 294 q 631 172 649 211 q 590 97 614 133 q 538 36 566 62 q 478 -4 510 10 q 414 -20 447 -20 q 326 2 375 -20 q 230 66 277 24 l 230 -253 q 255 -275 230 -264 q 342 -295 280 -286 l 342 -325 l 37 -325 l 37 -295 q 105 -275 80 -285 q 130 -253 130 -265 l 130 482 q 127 524 130 507 q 115 550 124 540 q 87 564 105 560 q 37 569 68 569 l 37 596 q 80 607 59 601 q 120 619 100 613 q 158 633 139 625 q 196 651 177 641 l 222 627 l 222 492 q 292 563 259 533 q 357 612 326 593 q 413 641 387 631 q 459 651 440 651 q 543 629 504 651 q 609 566 581 607 q 653 466 637 525 q 669 333 669 407 m 419 855 q 413 819 419 836 q 396 790 407 802 q 372 770 386 777 q 341 763 358 763 q 298 779 311 763 q 286 827 286 796 q 291 863 286 846 q 308 893 297 880 q 332 912 319 905 q 362 920 346 920 q 419 855 419 920 "
    },
    "[": {
      "x_min": 93,
      "x_max": 392.03125,
      "ha": 419,
      "o": "m 392 -168 q 383 -198 388 -182 q 372 -227 378 -215 l 93 -227 l 93 1085 l 368 1085 l 390 1067 q 382 1032 387 1050 q 372 1007 376 1015 l 186 1007 l 186 -149 l 368 -149 l 392 -168 "
    },
    "Ç¸": {
      "x_min": 33,
      "x_max": 851,
      "ha": 884,
      "o": "m 33 0 l 33 29 q 105 48 82 35 q 129 70 129 61 l 129 778 q 83 810 107 799 q 33 824 58 820 l 33 854 l 173 854 q 190 852 183 854 q 203 845 196 850 q 217 829 209 839 q 237 801 224 818 l 674 187 l 674 782 q 652 804 674 790 q 578 824 630 817 l 578 854 l 851 854 l 851 824 q 778 805 802 818 q 754 782 754 792 l 754 -20 q 701 -6 720 -16 q 674 14 682 2 l 209 673 l 209 70 q 231 49 209 62 q 305 29 253 36 l 305 0 l 33 0 m 559 957 q 545 937 550 944 q 529 926 540 931 l 217 1091 l 222 1121 q 237 1134 227 1127 q 259 1147 248 1140 q 283 1161 271 1155 q 301 1172 294 1167 l 559 957 "
    },
    "á¸–": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 413 432 428 427 q 377 438 398 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 564 1058 q 560 1041 563 1051 q 552 1021 556 1031 q 545 1002 548 1012 q 538 987 541 993 l 130 987 l 115 1003 q 120 1020 117 1011 q 127 1040 123 1030 q 135 1059 131 1049 q 143 1075 139 1068 l 550 1075 l 564 1058 m 245 1144 q 229 1155 234 1148 q 216 1174 224 1161 l 473 1389 q 491 1379 479 1385 q 514 1365 502 1372 q 536 1351 526 1358 q 551 1339 546 1344 l 557 1308 l 245 1144 "
    },
    "âˆ": {
      "x_min": 33,
      "x_max": 812,
      "ha": 858,
      "o": "m 33 0 l 33 29 q 104 50 78 38 q 130 70 130 61 l 130 783 q 105 803 130 791 q 33 825 80 816 l 33 855 l 812 855 l 812 825 q 740 804 766 816 q 715 783 715 793 l 715 70 q 739 50 715 62 q 812 29 763 38 l 812 0 l 512 0 l 512 29 q 583 50 557 38 q 609 70 609 61 l 609 750 q 600 766 609 758 q 573 781 591 774 q 530 792 556 788 q 472 797 505 797 l 361 797 q 306 792 330 797 q 267 781 283 788 q 243 767 251 775 q 236 750 236 759 l 236 70 q 260 50 236 62 q 333 29 284 38 l 333 0 l 33 0 "
    },
    "á¸¤": {
      "x_min": 33,
      "x_max": 844,
      "ha": 890,
      "o": "m 33 0 l 33 29 q 104 49 78 38 q 130 70 130 60 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 478 l 641 478 l 641 783 q 617 803 641 791 q 545 825 592 815 l 545 854 l 844 854 l 844 825 q 772 804 798 815 q 747 783 747 792 l 747 70 q 771 50 747 62 q 844 29 795 38 l 844 0 l 545 0 l 545 29 q 615 49 590 38 q 641 70 641 60 l 641 417 l 236 417 l 236 70 q 259 50 236 62 q 332 29 283 38 l 332 0 l 33 0 m 505 -189 q 499 -225 505 -208 q 482 -254 493 -242 q 458 -274 472 -267 q 427 -282 444 -282 q 384 -265 397 -282 q 372 -217 372 -248 q 377 -181 372 -198 q 394 -151 383 -164 q 418 -132 405 -139 q 448 -125 432 -125 q 505 -189 505 -125 "
    },
    "á¸¥": {
      "x_min": 37,
      "x_max": 732,
      "ha": 752,
      "o": "m 446 0 l 446 29 q 516 51 493 42 q 539 70 539 61 l 539 429 q 534 496 539 470 q 518 537 529 522 q 491 557 507 552 q 453 563 475 563 q 403 550 430 563 q 347 514 376 538 q 288 455 318 491 q 230 372 258 419 l 230 70 q 254 49 230 60 q 323 29 279 38 l 323 0 l 37 0 l 37 29 q 105 50 80 40 q 130 70 130 59 l 130 880 q 127 924 130 908 q 115 949 125 940 q 87 961 106 958 q 37 969 68 965 l 37 996 q 87 1007 64 1002 q 130 1019 110 1013 q 168 1033 150 1026 q 205 1051 186 1040 l 230 1028 l 230 464 q 293 541 258 507 q 363 600 328 576 q 432 638 398 625 q 491 651 465 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 70 q 659 52 639 61 q 732 29 679 42 l 732 0 l 446 0 m 450 -189 q 444 -225 450 -208 q 427 -254 438 -242 q 403 -274 417 -267 q 372 -282 389 -282 q 329 -265 342 -282 q 317 -217 317 -248 q 322 -181 317 -198 q 339 -151 328 -164 q 363 -132 350 -139 q 393 -125 377 -125 q 450 -189 450 -125 "
    },
    "Ë‹": {
      "x_min": 0,
      "x_max": 271.15625,
      "ha": 271,
      "o": "m 271 731 q 253 718 262 722 q 233 709 245 713 l 0 965 l 14 993 q 34 997 21 995 q 61 1002 47 1000 q 89 1007 75 1005 q 111 1009 103 1009 l 271 731 "
    },
    "ÄŸ": {
      "x_min": 20,
      "x_max": 657.625,
      "ha": 664,
      "o": "m 459 406 q 447 474 459 442 q 412 531 435 507 q 355 569 389 555 q 276 583 321 583 q 240 574 260 583 q 201 548 219 565 q 171 505 183 531 q 160 446 160 479 q 170 378 160 410 q 204 322 181 346 q 260 284 226 298 q 342 271 294 271 q 382 279 361 271 q 420 305 403 287 q 448 347 437 322 q 459 406 459 372 m 344 -2 q 297 3 319 0 q 257 10 276 6 q 182 -36 210 -15 q 141 -74 155 -57 q 123 -104 127 -90 q 120 -128 120 -118 q 138 -182 120 -157 q 188 -226 157 -208 q 260 -256 219 -245 q 347 -268 301 -268 q 428 -255 391 -268 q 490 -222 464 -243 q 530 -171 516 -200 q 545 -106 545 -141 q 536 -71 545 -87 q 505 -42 527 -55 q 443 -19 482 -29 q 344 -2 405 -9 m 552 434 q 529 339 552 382 q 468 265 506 296 q 384 217 431 234 q 288 200 337 200 l 285 200 q 241 154 253 172 q 230 132 230 136 q 236 116 230 124 q 262 101 243 109 q 315 87 281 94 q 400 74 348 80 q 520 50 472 66 q 595 12 567 33 q 634 -33 623 -8 q 645 -81 645 -57 q 631 -152 645 -118 q 593 -214 617 -185 q 536 -265 569 -242 q 466 -305 504 -288 q 389 -330 429 -321 q 310 -339 349 -339 q 246 -334 279 -339 q 180 -320 212 -330 q 118 -296 147 -311 q 67 -261 89 -282 q 32 -214 45 -240 q 20 -155 20 -188 q 26 -118 20 -137 q 51 -76 32 -99 q 105 -28 71 -54 q 197 28 140 -2 q 137 63 154 44 q 121 103 121 83 q 124 118 121 109 q 138 140 127 127 q 167 170 148 153 q 216 209 186 187 q 155 236 183 218 q 108 280 128 254 q 77 337 88 305 q 67 408 67 370 q 88 502 67 457 q 146 579 110 546 q 228 631 182 612 q 322 651 274 651 q 397 639 362 651 q 460 606 432 627 q 523 615 495 610 q 573 627 551 621 q 613 639 595 633 q 644 651 630 645 l 657 630 q 642 595 649 611 q 619 562 635 579 q 570 554 594 557 q 515 552 545 552 q 542 496 532 525 q 552 434 552 467 m 548 922 q 503 829 528 867 q 451 768 478 792 q 393 735 423 745 q 333 725 363 725 q 270 735 301 725 q 212 768 240 745 q 160 829 185 792 q 116 922 136 867 q 132 940 124 934 q 151 952 140 947 q 190 887 168 914 q 235 845 211 861 q 284 822 259 829 q 331 815 309 815 q 380 822 354 815 q 429 845 405 829 q 475 887 454 861 q 513 952 497 914 q 532 940 524 947 q 548 922 540 934 "
    },
    "á»ž": {
      "x_min": 47,
      "x_max": 811,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 811 944 q 803 906 811 927 q 778 861 796 885 q 730 812 760 837 q 654 761 700 786 q 731 614 705 698 q 757 438 757 529 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 532 853 482 875 q 622 796 582 832 q 679 846 661 821 q 697 889 697 871 q 687 926 697 908 q 658 959 677 944 l 776 1013 q 801 981 791 999 q 811 944 811 963 m 521 1121 q 508 1087 521 1102 q 480 1059 496 1072 q 447 1034 463 1046 q 424 1010 432 1022 q 423 984 417 997 q 458 953 430 970 q 438 945 448 946 q 417 942 427 944 q 372 970 386 957 q 355 993 358 982 q 358 1013 352 1004 q 376 1032 364 1023 q 401 1050 387 1041 q 428 1068 415 1059 q 448 1088 440 1078 q 457 1111 457 1099 q 444 1150 457 1137 q 411 1163 431 1163 q 393 1159 401 1163 q 378 1149 384 1155 q 369 1135 372 1143 q 366 1121 366 1128 q 367 1113 366 1117 q 371 1106 369 1110 q 357 1102 366 1104 q 339 1097 349 1099 q 321 1094 330 1095 q 305 1091 311 1092 l 298 1099 l 298 1108 q 310 1140 298 1124 q 342 1168 323 1155 q 386 1188 362 1180 q 433 1196 410 1196 q 498 1175 475 1196 q 521 1121 521 1155 "
    },
    "á¹˜": {
      "x_min": 27.015625,
      "x_max": 756.890625,
      "ha": 764,
      "o": "m 33 0 l 33 29 q 104 49 78 38 q 130 70 130 60 l 130 806 q 82 799 106 803 q 33 792 57 795 l 27 834 q 90 849 58 842 q 156 862 122 856 q 229 871 191 868 q 311 875 267 875 q 445 859 388 875 q 540 816 502 843 q 597 750 578 788 q 616 669 616 712 q 602 581 616 620 q 563 510 588 541 q 504 458 538 480 q 428 424 469 437 l 629 94 q 651 69 639 78 q 677 56 663 60 q 709 52 691 52 q 749 54 726 52 l 756 25 q 673 0 713 10 q 608 -9 633 -9 q 570 1 588 -9 q 542 27 552 12 l 341 408 q 324 407 332 407 l 306 407 q 271 408 288 407 q 236 414 254 410 l 236 70 q 259 50 236 62 q 332 29 283 38 l 332 0 l 33 0 m 287 817 q 236 817 261 817 l 236 469 q 267 465 253 465 q 295 465 280 465 q 456 511 400 465 q 512 648 512 558 q 500 716 512 685 q 459 769 487 747 q 389 804 431 792 q 287 817 346 817 m 412 1044 q 406 1008 412 1025 q 389 979 400 991 q 365 959 379 966 q 334 952 351 952 q 291 968 304 952 q 279 1016 279 985 q 284 1052 279 1035 q 301 1082 290 1069 q 325 1101 312 1094 q 355 1109 339 1109 q 412 1044 412 1109 "
    },
    "Âª": {
      "x_min": 33,
      "x_max": 342.28125,
      "ha": 349,
      "o": "m 342 520 q 291 490 311 499 q 260 482 270 482 q 241 495 250 482 q 226 538 231 509 q 164 494 195 507 q 110 482 133 482 q 83 486 97 482 q 58 499 70 490 q 40 522 47 508 q 33 559 33 537 q 64 633 33 601 q 163 684 96 665 l 225 700 l 225 712 q 222 749 225 733 q 214 775 220 764 q 197 792 208 786 q 167 798 185 798 q 147 793 157 798 q 128 780 136 788 q 116 759 120 771 q 118 733 113 748 q 110 726 119 730 q 89 719 101 722 q 66 715 77 716 q 50 714 54 713 l 44 728 q 65 769 47 749 q 107 804 83 789 q 156 828 131 819 q 199 837 181 837 q 239 830 223 837 q 266 808 256 823 q 281 766 276 792 q 286 703 286 740 l 286 570 q 288 543 286 552 q 295 530 291 535 q 309 529 298 528 q 338 537 319 530 l 342 520 m 35 397 l 35 447 l 323 447 l 323 397 l 35 397 m 225 569 l 225 672 l 196 664 q 148 646 167 656 q 117 626 129 637 q 102 602 106 614 q 98 575 98 589 q 101 552 98 560 q 110 538 105 543 q 121 531 115 533 q 132 529 127 529 q 176 538 153 529 q 225 569 199 548 "
    },
    "T": {
      "x_min": 6.609375,
      "x_max": 719,
      "ha": 750,
      "o": "m 204 0 l 204 28 q 255 41 234 34 q 289 54 276 48 q 307 67 301 61 q 313 77 313 72 l 313 793 l 114 793 q 96 789 104 793 q 79 773 87 785 q 60 737 71 761 q 35 674 49 713 l 6 687 q 15 772 9 727 q 28 854 21 818 l 699 854 l 719 837 q 713 761 719 802 q 699 677 707 720 l 669 677 q 658 730 663 708 q 646 766 653 752 q 630 786 639 780 q 607 793 621 793 l 419 793 l 419 77 q 442 55 419 68 q 528 28 466 41 l 528 0 l 204 0 "
    },
    "Å¡": {
      "x_min": 53.03125,
      "x_max": 473.640625,
      "ha": 525,
      "o": "m 465 192 q 451 109 465 144 q 417 51 438 75 q 369 13 395 28 q 318 -7 343 0 q 271 -17 293 -15 q 236 -20 248 -20 q 159 -7 204 -20 q 70 29 114 4 q 64 51 65 31 q 63 97 63 71 q 66 150 63 123 q 74 192 69 176 l 104 184 q 117 131 104 156 q 153 87 130 106 q 205 58 175 69 q 269 48 234 48 q 313 55 293 48 q 348 77 333 63 q 370 111 362 91 q 379 154 379 130 q 364 202 379 181 q 327 240 350 223 q 274 273 303 258 q 213 304 244 288 q 159 335 185 319 q 113 371 133 351 q 81 417 93 392 q 69 474 69 442 q 85 549 69 516 q 129 604 102 582 q 192 639 157 627 q 266 651 228 651 q 311 647 287 651 q 361 636 336 643 q 406 620 385 629 q 441 598 427 611 q 442 580 445 594 q 432 549 439 566 q 419 517 426 532 q 408 497 412 502 l 382 502 q 316 570 350 551 q 249 590 281 590 q 209 582 227 590 q 179 563 192 575 q 161 537 167 551 q 155 508 155 522 q 167 469 155 487 q 201 437 180 452 q 248 408 221 421 q 302 379 274 394 q 360 347 331 364 q 412 308 389 330 q 450 258 435 286 q 465 192 465 230 m 299 727 l 228 727 l 53 969 q 60 980 57 975 q 67 988 64 985 q 73 994 70 992 q 82 1000 77 997 l 265 830 l 445 1000 q 461 988 456 994 q 473 969 465 982 l 299 727 "
    },
    "ï‡ª": {
      "x_min": 39.875,
      "x_max": 410.0625,
      "ha": 450,
      "o": "m 410 247 q 400 215 406 232 q 390 186 394 198 l 56 186 l 39 204 q 48 234 43 217 q 60 264 54 250 l 393 264 l 410 247 m 410 422 q 400 391 406 409 l 390 363 l 56 363 l 39 380 q 48 410 43 393 q 60 440 54 427 l 393 440 l 410 422 "
    },
    "Ãž": {
      "x_min": 33,
      "x_max": 653.015625,
      "ha": 712,
      "o": "m 33 0 l 33 29 q 104 50 78 38 q 130 70 130 61 l 130 783 q 105 803 130 791 q 33 825 80 816 l 33 854 l 353 854 l 353 825 q 296 814 318 819 q 260 803 273 808 q 241 793 246 798 q 236 783 236 788 l 236 687 q 339 692 278 692 q 468 676 411 692 q 567 630 526 660 q 630 555 608 599 q 653 451 653 510 q 640 366 653 405 q 605 297 627 327 q 555 245 583 267 q 497 208 528 223 q 437 187 467 194 q 381 180 407 180 q 323 185 350 180 q 274 202 296 190 l 258 250 q 312 234 286 237 q 361 231 338 231 q 425 243 391 231 q 486 279 458 255 q 532 339 514 303 q 550 423 550 375 q 530 519 550 479 q 476 584 510 559 q 397 622 442 610 q 301 634 352 634 q 236 634 267 634 l 236 70 q 241 62 236 66 q 259 51 246 57 q 295 40 272 46 q 353 29 318 35 l 353 0 l 33 0 "
    },
    "j": {
      "x_min": -146,
      "x_max": 256,
      "ha": 352,
      "o": "m 236 67 q 226 -65 236 -10 q 199 -158 216 -119 q 159 -223 182 -196 q 109 -271 136 -249 q 71 -297 91 -285 q 32 -319 51 -310 q -5 -333 12 -328 q -35 -339 -23 -339 q -75 -333 -56 -339 q -111 -321 -95 -328 q -136 -305 -126 -314 q -146 -292 -146 -297 q -136 -275 -146 -286 q -113 -253 -126 -265 q -85 -232 -100 -242 q -61 -217 -71 -222 q -18 -242 -40 -236 q 28 -248 4 -248 q 66 -235 48 -248 q 100 -191 85 -222 q 124 -106 115 -160 q 134 28 134 -53 l 134 454 q 134 509 134 488 q 125 542 134 530 q 96 560 116 554 q 42 569 77 565 l 42 596 q 92 607 70 601 q 133 620 114 613 q 171 634 153 626 q 211 651 190 642 l 236 651 l 236 67 m 256 854 q 250 818 256 835 q 233 789 244 801 q 209 769 223 776 q 179 762 195 762 q 136 778 149 762 q 123 826 123 795 q 128 862 123 845 q 145 892 134 879 q 169 911 155 904 q 199 919 183 919 q 256 854 256 919 "
    },
    "1": {
      "x_min": 92.390625,
      "x_max": 558,
      "ha": 638,
      "o": "m 113 0 l 113 35 q 199 49 164 42 q 254 65 233 57 q 283 81 274 73 q 292 97 292 89 l 292 637 q 290 679 292 664 q 280 704 288 694 q 264 710 276 707 q 232 713 253 713 q 181 708 212 713 q 106 694 150 704 l 92 729 q 158 751 120 737 q 235 782 196 765 q 310 815 274 799 q 367 844 345 832 l 392 820 l 392 97 q 399 82 392 90 q 425 66 406 74 q 475 50 443 58 q 558 35 508 42 l 558 0 l 113 0 "
    },
    "É£": {
      "x_min": 13,
      "x_max": 657,
      "ha": 671,
      "o": "m 327 28 q 290 -50 305 -17 q 267 -106 275 -83 q 255 -144 258 -128 q 252 -170 252 -159 q 259 -217 252 -198 q 277 -247 266 -236 q 302 -263 289 -259 q 329 -268 316 -268 q 354 -261 342 -268 q 377 -245 367 -255 q 393 -220 387 -234 q 400 -191 400 -206 q 397 -156 400 -172 q 389 -119 395 -139 q 371 -72 382 -99 q 343 -6 360 -46 l 327 28 m 657 601 q 624 592 637 597 q 604 583 612 588 q 592 571 597 578 q 583 555 588 565 l 386 149 l 421 72 q 453 0 440 31 q 475 -56 467 -30 q 488 -107 484 -82 q 492 -159 492 -131 q 482 -215 492 -184 q 451 -274 472 -246 q 397 -320 430 -301 q 321 -339 365 -339 q 265 -328 291 -339 q 219 -298 239 -317 q 189 -252 200 -279 q 178 -192 178 -225 q 180 -157 178 -175 q 190 -116 183 -139 q 210 -63 198 -93 q 242 6 222 -33 l 291 108 l 86 555 q 62 583 79 573 q 13 601 46 594 l 13 631 l 270 631 l 270 601 q 228 594 244 598 q 204 585 212 590 q 196 573 196 580 q 200 555 195 565 l 348 230 l 505 555 q 508 572 509 565 q 499 584 507 579 q 475 594 491 590 q 435 601 459 597 l 435 631 l 657 631 l 657 601 "
    },
    "É©": {
      "x_min": 32.609375,
      "x_max": 397.5,
      "ha": 384,
      "o": "m 397 91 q 334 37 363 59 q 281 3 305 15 q 238 -14 257 -9 q 207 -20 219 -20 q 166 -10 181 -20 q 141 18 150 -1 q 128 67 131 37 q 125 137 125 97 l 125 454 q 125 510 125 488 q 116 543 125 531 q 88 560 107 555 q 32 569 68 566 l 32 596 q 76 606 53 600 q 122 619 99 612 q 165 634 144 626 q 202 651 186 642 l 228 651 l 228 205 q 228 133 228 159 q 233 93 228 106 q 248 75 239 79 q 274 72 258 72 q 312 83 281 72 q 386 126 343 95 l 397 91 "
    },
    "Ç›": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 615 816 l 568 855 l 847 855 l 847 825 m 649 1045 q 643 1009 649 1026 q 626 980 637 992 q 602 960 616 967 q 571 953 588 953 q 528 969 541 953 q 516 1017 516 986 q 521 1053 516 1036 q 538 1083 527 1070 q 562 1102 549 1095 q 592 1110 576 1110 q 649 1045 649 1110 m 383 1045 q 377 1009 383 1026 q 360 980 371 992 q 336 960 350 967 q 305 953 322 953 q 262 969 275 953 q 250 1017 250 986 q 255 1053 250 1036 q 272 1083 261 1070 q 296 1102 283 1095 q 326 1110 310 1110 q 383 1045 383 1110 m 567 1175 q 552 1155 557 1162 q 537 1144 547 1149 l 226 1309 l 231 1339 q 246 1352 235 1345 q 268 1365 256 1358 q 291 1379 280 1373 q 310 1390 302 1386 l 567 1175 "
    },
    "Ç’": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 391 727 l 320 727 l 144 969 q 152 980 149 975 q 158 988 156 985 q 165 994 161 992 q 174 1000 169 997 l 357 830 l 536 1000 q 552 988 547 994 q 565 969 557 982 l 391 727 "
    },
    "Ä‰": {
      "x_min": 53,
      "x_max": 558.40625,
      "ha": 595,
      "o": "m 558 129 q 482 47 515 76 q 421 2 449 17 q 366 -16 393 -12 q 309 -20 340 -20 q 214 2 260 -20 q 132 65 168 24 q 74 166 96 106 q 53 301 53 226 q 79 438 53 374 q 151 549 105 502 q 258 624 196 596 q 392 651 320 651 q 437 647 414 651 q 480 637 459 644 q 519 620 501 629 q 548 598 537 610 q 546 574 549 589 q 537 543 543 560 q 523 511 531 526 q 510 486 516 496 l 485 493 q 468 519 480 505 q 437 547 456 534 q 392 568 418 560 q 332 577 366 577 q 264 561 297 577 q 207 513 232 545 q 167 433 182 481 q 153 322 153 386 q 169 217 153 264 q 215 137 186 170 q 280 85 243 103 q 357 68 317 68 q 391 69 375 68 q 426 80 407 71 q 470 106 445 89 q 532 156 495 124 l 558 129 m 535 741 q 523 722 527 728 q 507 710 518 716 l 325 892 l 145 710 q 135 716 139 713 q 129 722 132 718 q 123 730 126 725 q 115 741 119 735 l 290 998 l 361 998 l 535 741 "
    },
    "Æ†": {
      "x_min": 41,
      "x_max": 677.015625,
      "ha": 730,
      "o": "m 677 460 q 648 264 677 352 q 570 113 620 176 q 450 14 519 49 q 300 -20 381 -20 q 186 -1 235 -20 q 105 48 138 17 q 56 114 72 78 q 41 186 41 151 q 49 241 41 214 q 81 293 58 267 q 111 308 95 301 q 144 322 128 315 q 175 333 160 328 q 202 343 190 339 l 217 310 q 186 287 198 298 q 168 263 174 276 q 159 236 161 251 q 157 203 157 222 q 169 152 157 178 q 204 106 181 127 q 258 71 226 85 q 330 58 291 58 q 420 77 376 58 q 498 140 465 96 q 553 256 532 184 q 574 434 574 329 q 548 592 574 525 q 482 706 522 660 q 392 774 441 751 q 295 797 343 797 q 255 792 277 797 q 205 778 233 788 q 144 749 178 767 q 71 702 111 731 q 63 708 67 704 q 57 717 60 712 q 51 726 53 722 q 47 733 48 731 q 130 806 91 778 q 205 849 169 834 q 274 869 241 864 q 342 875 308 875 q 464 847 404 875 q 572 766 524 819 q 648 635 619 712 q 677 460 677 558 "
    },
    "Ä«": {
      "x_min": 25.078125,
      "x_max": 368.234375,
      "ha": 370,
      "o": "m 47 0 l 47 29 q 115 49 92 38 q 139 70 139 61 l 139 454 q 139 510 139 488 q 130 543 139 531 q 101 560 121 555 q 47 569 82 566 l 47 596 q 90 606 67 600 q 136 619 113 612 q 179 634 158 626 q 217 651 200 642 l 240 651 l 240 70 q 262 50 240 62 q 333 29 284 38 l 333 0 l 47 0 m 368 868 q 363 851 366 861 q 356 831 360 841 q 348 812 352 822 q 342 797 344 803 l 39 797 l 25 813 q 29 830 26 821 q 37 850 33 840 q 45 869 41 859 q 52 885 49 878 l 353 885 l 368 868 "
    },
    "ÄŽ": {
      "x_min": 27.015625,
      "x_max": 746,
      "ha": 806,
      "o": "m 300 816 q 235 816 266 816 l 235 104 q 241 80 235 89 q 277 62 250 68 q 356 57 304 57 q 450 79 402 57 q 540 148 499 102 q 607 265 581 195 q 633 431 633 335 q 611 592 633 521 q 546 713 589 663 q 442 789 504 763 q 300 816 380 816 m 746 456 q 729 305 746 372 q 683 188 712 239 q 617 101 654 137 q 538 43 579 66 q 453 10 496 20 q 371 0 410 0 l 33 0 l 33 29 q 103 49 78 38 q 129 70 129 61 l 129 805 q 78 798 102 801 q 33 791 54 794 l 27 833 q 94 850 56 842 q 175 864 132 858 q 261 871 217 871 q 343 871 304 871 q 510 842 436 871 q 637 761 585 814 q 717 631 689 708 q 746 456 746 554 m 400 943 l 329 943 l 154 1151 q 162 1162 158 1157 q 168 1170 165 1167 q 175 1176 171 1174 q 183 1183 178 1179 l 366 1039 l 546 1183 q 562 1170 557 1176 q 575 1151 567 1164 l 400 943 "
    },
    "&": {
      "x_min": 53,
      "x_max": 894.84375,
      "ha": 908,
      "o": "m 295 766 q 306 684 295 727 q 337 596 317 641 q 411 648 383 622 q 456 699 440 674 q 479 748 473 724 q 485 794 485 772 q 479 839 485 818 q 461 876 473 860 q 432 901 449 892 q 391 911 414 911 q 349 898 367 911 q 319 865 331 885 q 301 819 307 845 q 295 766 295 793 m 351 57 q 454 75 407 57 q 539 124 502 93 q 471 200 506 159 q 402 286 436 241 q 337 379 368 331 q 282 476 307 427 q 225 419 248 446 q 187 363 202 391 q 167 308 173 336 q 161 252 161 281 q 176 168 161 205 q 219 107 192 132 q 280 69 246 82 q 351 57 314 57 m 894 507 q 861 472 878 486 q 829 446 845 458 q 801 457 817 452 q 770 466 786 462 q 740 471 755 469 q 714 474 725 474 q 726 451 722 462 q 733 429 731 441 q 736 402 736 417 q 736 368 736 387 q 715 266 736 320 q 657 162 695 212 q 741 86 703 117 q 796 57 767 64 q 864 59 825 50 l 870 30 q 772 -7 812 4 q 718 -20 733 -20 q 667 6 703 -20 q 582 80 630 34 q 467 8 531 36 q 331 -20 403 -20 q 217 -3 269 -20 q 130 45 166 13 q 73 126 93 78 q 53 238 53 175 q 66 316 53 276 q 104 394 79 355 q 167 470 130 432 q 252 542 205 507 q 222 636 233 590 q 211 728 211 683 q 228 829 211 783 q 276 911 246 876 q 345 964 306 945 q 426 984 384 984 q 497 970 468 984 q 544 933 526 956 q 569 879 561 910 q 577 817 577 849 q 533 683 577 745 q 411 571 490 621 q 386 555 398 563 q 365 540 375 547 q 417 451 388 496 q 478 365 446 407 q 544 282 510 322 q 612 206 578 242 q 645 275 634 240 q 656 340 656 309 q 646 393 656 368 q 622 437 637 419 q 588 467 607 456 q 549 478 569 478 l 535 497 q 556 516 542 506 q 584 532 571 526 l 879 532 l 894 507 "
    },
    "á¹»": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 467 70 q 369 4 398 18 q 316 -15 340 -10 q 271 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 532 113 514 q 100 559 109 550 q 75 572 92 569 q 27 579 57 575 l 27 606 q 72 613 51 609 q 112 622 93 617 q 152 635 132 628 q 194 651 172 642 l 213 624 l 213 226 q 219 147 213 179 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 482 q 505 531 508 513 q 492 560 502 550 q 464 574 482 570 q 414 578 446 578 l 414 606 q 507 625 463 612 q 588 651 551 638 l 608 624 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 593 868 q 588 851 592 861 q 581 831 585 841 q 574 812 577 822 q 567 797 570 803 l 159 797 l 144 813 q 148 830 145 821 q 156 850 152 840 q 164 869 160 859 q 171 885 168 878 l 579 885 l 593 868 m 562 1099 q 556 1063 562 1080 q 539 1034 550 1046 q 515 1014 529 1021 q 484 1007 501 1007 q 441 1023 454 1007 q 429 1071 429 1040 q 434 1107 429 1090 q 451 1137 440 1124 q 475 1156 462 1149 q 505 1164 489 1164 q 562 1099 562 1164 m 296 1099 q 290 1063 296 1080 q 273 1034 284 1046 q 249 1014 263 1021 q 218 1007 235 1007 q 175 1023 188 1007 q 163 1071 163 1040 q 168 1107 163 1090 q 185 1137 174 1124 q 209 1156 196 1149 q 239 1164 223 1164 q 296 1099 296 1164 "
    },
    "G": {
      "x_min": 47,
      "x_max": 761,
      "ha": 794,
      "o": "m 693 805 q 691 792 697 802 q 674 770 685 782 q 651 746 664 758 q 629 726 638 734 l 607 730 q 562 765 585 751 q 513 786 538 779 q 459 797 487 794 q 397 800 430 800 q 356 792 381 800 q 303 767 331 784 q 249 722 276 750 q 199 652 221 693 q 163 557 177 611 q 150 432 150 502 q 174 267 150 337 q 239 152 199 198 q 328 83 278 106 q 429 61 378 61 q 518 70 477 61 q 592 99 559 80 l 592 329 q 585 343 592 336 q 563 357 579 350 q 521 371 548 364 q 455 384 495 378 l 455 414 l 761 414 l 761 384 q 707 360 723 375 q 692 329 692 344 l 692 104 q 603 38 642 63 q 532 1 565 14 q 470 -15 499 -11 q 411 -20 441 -20 q 279 5 344 -20 q 162 82 213 30 q 78 212 110 134 q 47 394 47 289 q 81 596 47 507 q 177 747 116 685 q 318 842 237 809 q 490 875 399 875 q 538 870 512 875 q 592 856 564 865 q 645 834 619 847 q 693 805 671 822 "
    },
    "`": {
      "x_min": 19.9375,
      "x_max": 291.09375,
      "ha": 434,
      "o": "m 291 731 q 275 719 285 725 q 253 709 265 713 l 19 965 l 34 993 q 54 997 41 994 q 81 1002 67 999 q 109 1007 95 1004 q 131 1009 123 1009 l 291 731 "
    },
    "á»—": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 565 741 q 552 722 557 728 q 536 710 547 716 l 354 892 l 174 710 q 165 716 169 713 q 159 722 161 718 q 152 730 156 725 q 144 741 149 735 l 320 998 l 391 998 l 565 741 m 581 1218 q 553 1163 569 1190 q 516 1114 536 1136 q 472 1079 496 1093 q 422 1066 448 1066 q 378 1077 399 1066 q 337 1104 357 1089 q 297 1130 317 1118 q 258 1142 278 1142 q 210 1120 232 1142 q 163 1060 188 1098 l 128 1074 q 157 1129 140 1101 q 193 1178 173 1156 q 238 1213 214 1199 q 287 1227 261 1227 q 335 1215 312 1227 q 377 1189 357 1203 q 415 1162 397 1174 q 449 1151 433 1151 q 499 1172 474 1151 q 545 1233 523 1194 l 581 1218 "
    },
    "á»„": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 413 432 428 427 q 377 438 398 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 543 957 q 531 938 535 944 q 515 926 526 932 l 333 1067 l 153 926 q 144 931 147 929 q 137 938 140 934 q 131 946 134 941 q 123 957 128 950 l 298 1166 l 369 1166 l 543 957 m 560 1374 q 531 1319 548 1346 q 494 1270 515 1291 q 450 1234 474 1248 q 400 1221 426 1221 q 356 1232 377 1221 q 315 1259 335 1244 q 276 1286 295 1274 q 237 1298 256 1298 q 188 1275 210 1298 q 142 1215 167 1253 l 107 1229 q 135 1284 119 1257 q 172 1333 151 1312 q 216 1369 192 1355 q 265 1383 240 1383 q 313 1371 291 1383 q 356 1344 336 1359 q 394 1317 376 1329 q 428 1306 412 1306 q 477 1327 453 1306 q 524 1389 501 1349 l 560 1374 "
    },
    "áº±": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 535 922 q 490 829 515 867 q 437 768 465 792 q 380 735 410 745 q 320 725 350 725 q 257 735 287 725 q 198 768 226 745 q 147 829 171 792 q 102 922 122 867 q 118 940 110 934 q 138 952 126 947 q 176 887 154 914 q 221 845 197 861 q 270 822 245 829 q 317 815 295 815 q 366 822 341 815 q 416 845 392 829 q 462 887 440 861 q 500 952 484 914 q 519 940 511 947 q 535 922 527 934 m 394 947 q 377 935 386 939 q 357 926 368 930 l 123 1182 l 138 1210 q 157 1214 144 1212 q 184 1219 170 1217 q 213 1224 199 1222 q 235 1226 227 1225 l 394 947 "
    },
    "Å": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 571 923 q 526 830 551 868 q 473 769 501 793 q 416 736 446 746 q 356 726 386 726 q 293 736 324 726 q 235 769 262 746 q 183 830 207 793 q 139 923 159 868 q 155 941 147 935 q 174 953 163 948 q 212 888 191 915 q 258 846 234 862 q 307 823 282 830 q 353 816 332 816 q 402 823 377 816 q 452 846 428 830 q 498 888 476 862 q 535 953 520 915 q 555 941 547 948 q 571 923 563 935 "
    },
    "áº¢": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 351 838 q 390 869 367 855 q 430 893 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 510 1121 q 497 1087 510 1102 q 469 1059 485 1072 q 436 1034 452 1046 q 413 1010 421 1022 q 412 984 406 997 q 447 953 419 970 q 427 945 437 946 q 406 942 416 944 q 361 970 375 957 q 344 993 347 982 q 347 1013 341 1004 q 365 1032 353 1023 q 390 1050 376 1041 q 417 1068 404 1059 q 437 1088 429 1078 q 446 1111 446 1099 q 433 1150 446 1137 q 400 1163 420 1163 q 382 1159 390 1163 q 367 1149 373 1155 q 358 1135 361 1143 q 355 1120 355 1128 q 356 1113 355 1117 q 360 1106 358 1110 q 346 1102 355 1104 q 328 1097 338 1099 q 310 1094 319 1095 q 294 1091 300 1092 l 287 1099 l 287 1108 q 299 1140 287 1124 q 331 1168 312 1155 q 375 1188 351 1180 q 422 1196 399 1196 q 487 1175 464 1196 q 510 1121 510 1155 "
    },
    "Ã½": {
      "x_min": -31,
      "x_max": 658,
      "ha": 672,
      "o": "m 658 601 q 625 592 638 596 q 604 583 612 588 q 592 571 596 578 q 584 555 588 565 l 362 -55 q 300 -184 334 -130 q 229 -272 266 -237 q 155 -322 192 -306 q 84 -339 117 -339 q 38 -335 59 -339 q 1 -327 16 -332 q -22 -315 -13 -322 q -31 -303 -31 -309 q -22 -286 -31 -299 q -2 -260 -14 -274 q 22 -231 9 -245 q 47 -211 36 -218 q 110 -232 79 -232 q 167 -223 142 -232 q 195 -204 179 -219 q 227 -168 211 -189 q 259 -119 243 -146 q 286 -62 274 -91 l 306 -15 l 87 555 q 64 584 81 574 q 14 601 47 593 l 14 630 l 271 630 l 271 601 q 228 594 245 598 q 203 586 212 591 q 193 573 194 581 q 195 555 191 566 l 359 123 l 512 555 q 513 572 515 565 q 502 584 511 579 q 476 593 493 589 q 436 601 460 597 l 436 631 l 658 631 l 658 601 m 324 710 q 303 717 312 712 q 286 729 294 723 l 441 1010 q 462 1007 449 1009 q 490 1003 475 1006 q 517 999 504 1001 q 538 994 531 996 l 553 968 l 324 710 "
    },
    "Âº": {
      "x_min": 33,
      "x_max": 351,
      "ha": 384,
      "o": "m 289 655 q 280 709 289 684 q 257 754 271 735 q 222 785 242 774 q 181 797 203 797 q 145 786 161 797 q 118 758 129 776 q 101 715 107 740 q 95 662 95 690 q 104 608 95 633 q 128 563 113 582 q 163 532 143 544 q 203 521 182 521 q 236 531 221 521 q 264 559 252 541 q 282 602 275 577 q 289 655 289 627 m 41 397 l 41 447 l 343 447 l 343 397 l 41 397 m 351 666 q 336 598 351 631 q 297 539 321 565 q 242 497 273 513 q 181 482 212 482 q 120 494 147 482 q 73 530 93 507 q 43 584 54 553 q 33 652 33 615 q 46 721 33 687 q 82 780 59 754 q 136 821 106 805 q 202 837 167 837 q 262 824 234 837 q 309 788 289 811 q 340 734 329 765 q 351 666 351 702 "
    },
    "âˆž": {
      "x_min": 53,
      "x_max": 847,
      "ha": 900,
      "o": "m 228 289 q 267 291 250 289 q 303 303 285 294 q 342 329 321 312 q 394 374 364 346 q 364 412 380 392 q 327 447 347 431 q 281 472 306 462 q 225 483 255 483 q 174 473 194 483 q 144 449 155 464 q 130 419 134 435 q 127 391 127 403 q 133 359 127 377 q 152 326 139 342 q 184 299 164 310 q 228 289 203 289 m 671 494 q 631 489 650 494 q 593 475 612 485 q 552 449 574 465 q 503 407 531 432 q 533 369 516 389 q 572 335 551 350 q 619 309 593 319 q 674 300 644 300 q 725 309 705 300 q 755 333 744 318 q 769 363 765 347 q 773 391 773 379 q 766 423 773 405 q 747 456 760 440 q 715 483 735 472 q 671 494 696 494 m 273 573 q 336 562 307 573 q 390 533 365 551 q 436 492 415 515 q 473 448 456 470 q 543 507 511 483 q 602 546 575 531 q 656 566 630 560 q 708 573 681 573 q 758 561 733 573 q 803 528 783 549 q 835 480 823 508 q 847 419 847 452 q 829 343 847 380 q 780 276 811 305 q 710 228 749 246 q 626 210 670 210 q 563 220 592 210 q 508 249 533 231 q 461 289 482 267 q 422 333 440 311 q 348 270 380 295 q 291 233 317 246 q 242 214 265 219 q 192 210 218 210 q 141 221 166 210 q 96 254 116 233 q 64 302 76 274 q 53 363 53 330 q 70 439 53 402 q 119 506 88 477 q 189 554 150 536 q 273 573 229 573 "
    },
    "Åº": {
      "x_min": 50.453125,
      "x_max": 549.984375,
      "ha": 602,
      "o": "m 548 168 q 548 121 548 146 q 548 74 548 97 q 548 31 548 51 q 543 0 548 12 l 67 0 l 50 30 l 419 569 l 189 569 q 167 564 178 569 q 146 549 156 560 q 127 518 136 538 q 111 467 118 498 l 85 475 l 85 641 q 117 634 105 637 q 142 631 129 631 q 170 631 155 631 l 535 631 l 549 601 l 179 61 l 460 61 q 490 87 476 61 q 520 174 504 113 l 548 168 m 290 709 q 269 717 278 711 q 252 728 259 722 l 408 1009 q 429 1007 416 1008 q 456 1003 442 1005 q 484 998 470 1001 q 504 993 497 995 l 520 967 l 290 709 "
    },
    "Æ¯": {
      "x_min": 33,
      "x_max": 930,
      "ha": 930,
      "o": "m 930 944 q 921 904 930 927 q 892 856 913 882 q 837 803 872 830 q 751 749 803 775 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 804 855 q 816 889 816 873 q 806 926 816 909 q 777 960 796 944 l 895 1014 q 920 981 910 999 q 930 944 930 964 "
    },
    "ÍŸ": {
      "x_min": -433.984375,
      "x_max": 433.984375,
      "ha": 0,
      "o": "m 433 -420 q 429 -437 432 -427 q 422 -457 426 -447 q 414 -476 418 -466 q 408 -492 410 -485 l -419 -492 l -433 -475 q -429 -458 -432 -467 q -422 -438 -426 -448 q -414 -419 -418 -429 q -406 -404 -410 -410 l 419 -404 l 433 -420 "
    },
    "Å„": {
      "x_min": 37,
      "x_max": 732,
      "ha": 752,
      "o": "m 446 0 l 446 29 q 516 51 493 42 q 539 70 539 61 l 539 429 q 534 496 539 470 q 519 537 529 522 q 493 557 509 552 q 453 563 477 563 q 408 552 433 563 q 353 520 382 542 q 293 461 325 497 q 230 372 261 425 l 230 70 q 254 49 230 60 q 323 29 279 38 l 323 0 l 37 0 l 37 29 q 105 50 80 40 q 130 70 130 59 l 130 482 q 128 524 130 508 q 117 548 126 539 q 89 561 108 557 q 37 569 70 565 l 37 596 q 120 618 82 604 q 196 651 159 632 l 223 627 l 223 458 q 291 539 256 504 q 362 599 327 575 q 432 637 398 624 q 492 651 466 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 70 q 659 52 639 61 q 732 29 679 42 l 732 0 l 446 0 m 355 709 q 334 717 343 712 q 317 728 325 722 l 473 1009 q 494 1007 481 1009 q 521 1003 507 1005 q 549 998 535 1001 q 569 994 562 996 l 585 967 l 355 709 "
    },
    "á¸´": {
      "x_min": 33,
      "x_max": 779.53125,
      "ha": 786,
      "o": "m 33 0 l 33 29 q 104 50 78 39 q 130 70 130 61 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 438 l 507 765 q 526 794 522 783 q 523 811 530 805 q 500 820 517 817 q 458 824 483 824 l 458 855 l 737 855 l 737 825 q 699 819 715 822 q 669 812 682 817 q 644 801 655 808 q 624 783 634 794 l 333 455 l 653 86 q 679 65 665 73 q 708 55 693 58 q 740 52 724 52 q 774 54 757 52 l 779 25 q 700 1 740 12 q 629 -9 661 -9 q 595 -3 610 -9 q 566 19 580 2 l 236 433 l 236 70 q 259 50 236 62 q 332 29 283 39 l 332 0 l 33 0 m 630 -155 q 626 -172 629 -162 q 618 -192 622 -182 q 611 -211 614 -201 q 604 -227 607 -220 l 196 -227 l 182 -210 q 186 -193 183 -202 q 194 -173 190 -183 q 202 -154 198 -164 q 209 -139 206 -145 l 616 -139 l 630 -155 "
    },
    "ïˆ™": {
      "x_min": 13,
      "x_max": 857,
      "ha": 884,
      "o": "m 13 28 q 83 46 58 37 q 115 73 109 54 l 370 814 q 388 839 375 829 q 416 857 401 850 q 446 868 432 864 q 470 874 461 872 l 762 73 q 792 45 769 56 q 857 28 815 34 l 857 0 l 568 0 l 568 31 q 640 40 621 31 q 653 70 660 50 l 418 716 l 194 73 q 211 44 187 54 q 289 28 234 35 l 289 0 l 13 0 l 13 28 "
    },
    "á¸¹": {
      "x_min": -28.453125,
      "x_max": 420.75,
      "ha": 369,
      "o": "m 40 0 l 40 29 q 88 39 68 34 q 118 49 107 44 q 135 60 130 54 q 140 70 140 65 l 140 879 q 136 926 140 909 q 121 952 132 943 q 92 963 110 960 q 46 970 74 966 l 46 997 q 134 1018 94 1006 q 216 1051 174 1029 l 240 1028 l 240 70 q 262 49 240 60 q 339 29 284 39 l 339 0 l 40 0 m 256 -189 q 250 -225 256 -208 q 233 -254 244 -242 q 209 -274 223 -267 q 178 -282 195 -282 q 135 -265 148 -282 q 123 -217 123 -248 q 128 -181 123 -198 q 145 -151 134 -164 q 169 -132 156 -139 q 199 -125 183 -125 q 256 -189 256 -125 m 420 1275 q 416 1258 419 1268 q 408 1238 412 1248 q 401 1219 404 1229 q 394 1204 397 1210 l -13 1204 l -28 1220 q -23 1237 -27 1228 q -16 1257 -20 1247 q -8 1276 -12 1266 q -1 1292 -4 1285 l 406 1292 l 420 1275 "
    },
    "Â¤": {
      "x_min": 104.65625,
      "x_max": 534,
      "ha": 639,
      "o": "m 241 488 q 216 450 224 471 q 208 407 208 429 q 215 364 208 385 q 240 327 223 344 q 278 302 257 310 q 320 294 299 294 q 362 302 341 294 q 399 327 382 310 q 423 365 415 344 q 432 408 432 386 q 423 450 432 429 q 398 488 415 471 q 361 513 382 505 q 320 522 341 522 q 278 513 299 522 q 241 488 257 505 m 490 190 l 420 260 q 372 237 398 245 q 320 230 346 230 q 267 237 293 230 q 219 261 241 245 l 147 190 l 125 190 q 114 210 120 199 q 104 232 109 221 l 176 304 q 152 353 160 327 q 145 406 145 379 q 152 460 145 433 q 176 509 160 486 l 106 582 l 106 605 q 126 616 115 610 q 148 626 137 621 l 219 553 q 267 578 241 570 q 320 587 293 587 q 372 578 346 587 q 420 553 397 570 l 490 626 l 513 626 l 533 581 l 462 509 q 486 460 478 486 q 495 406 495 433 q 487 353 495 379 q 463 304 479 327 l 534 233 l 534 210 l 490 190 "
    },
    "Ç¢": {
      "x_min": 0,
      "x_max": 1020.84375,
      "ha": 1060,
      "o": "m 515 778 q 508 788 515 785 q 493 791 502 792 q 476 784 484 790 q 462 766 467 778 l 357 498 l 515 498 l 515 778 m 1020 165 q 1006 63 1014 106 q 992 0 997 19 l 419 0 l 419 29 q 489 49 464 38 q 515 70 515 60 l 515 441 l 335 441 l 189 75 q 201 45 180 55 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 70 46 45 37 q 104 75 96 55 l 379 767 q 376 787 383 779 q 353 801 369 795 q 310 813 336 807 q 253 824 285 818 l 253 854 l 955 854 l 975 837 q 972 798 975 819 q 966 756 970 777 q 958 716 962 735 q 948 684 953 697 l 918 684 q 914 736 918 715 q 903 770 910 757 q 887 787 896 782 q 864 793 877 793 l 621 793 l 621 498 l 890 498 l 908 479 q 894 458 902 469 q 879 437 887 447 q 862 417 870 426 q 846 403 854 409 q 825 420 836 413 q 799 432 814 427 q 763 438 783 436 q 714 441 742 441 l 621 441 l 621 104 q 625 86 621 94 q 645 72 630 78 q 687 64 660 67 q 758 61 714 61 l 844 61 q 896 64 874 61 q 934 79 917 67 q 963 114 950 91 q 990 177 976 137 l 1020 165 m 845 1058 q 840 1041 844 1051 q 833 1021 837 1031 q 825 1002 829 1012 q 819 987 822 993 l 410 987 l 395 1003 q 400 1020 397 1011 q 407 1040 403 1030 q 415 1059 411 1049 q 423 1075 419 1068 l 830 1075 l 845 1058 "
    },
    "Æ—": {
      "x_min": 32.390625,
      "x_max": 362.609375,
      "ha": 406,
      "o": "m 346 417 l 250 417 l 250 70 q 274 50 250 62 q 346 29 298 38 l 346 0 l 47 0 l 47 29 q 118 49 92 38 q 144 70 144 60 l 144 417 l 47 417 l 32 433 q 39 455 35 443 q 47 478 43 467 l 144 478 l 144 783 q 119 803 144 791 q 47 825 95 815 l 47 854 l 346 854 l 346 825 q 275 804 300 815 q 250 783 250 792 l 250 478 l 346 478 l 362 462 l 346 417 "
    },
    "Äœ": {
      "x_min": 47,
      "x_max": 761,
      "ha": 794,
      "o": "m 693 805 q 691 792 697 802 q 674 770 685 782 q 651 746 664 758 q 629 726 638 734 l 607 730 q 562 765 585 751 q 513 786 538 779 q 459 797 487 794 q 397 800 430 800 q 356 792 381 800 q 303 767 331 784 q 249 722 276 750 q 199 652 221 693 q 163 557 177 611 q 150 432 150 502 q 174 267 150 337 q 239 152 199 198 q 328 83 278 106 q 429 61 378 61 q 518 70 477 61 q 592 99 559 80 l 592 329 q 585 343 592 336 q 563 357 579 350 q 521 371 548 364 q 455 384 495 378 l 455 414 l 761 414 l 761 384 q 707 360 723 375 q 692 329 692 344 l 692 104 q 603 38 642 63 q 532 1 565 14 q 470 -15 499 -11 q 411 -20 441 -20 q 279 5 344 -20 q 162 82 213 30 q 78 212 110 134 q 47 394 47 289 q 81 596 47 507 q 177 747 116 685 q 318 842 237 809 q 490 875 399 875 q 538 870 512 875 q 592 856 564 865 q 645 834 619 847 q 693 805 671 822 m 636 957 q 623 938 628 944 q 607 927 618 932 l 425 1068 l 245 927 q 236 932 239 929 q 229 938 232 935 q 223 946 226 942 q 215 957 219 951 l 390 1167 l 461 1167 l 636 957 "
    },
    "p": {
      "x_min": 37,
      "x_max": 669,
      "ha": 722,
      "o": "m 579 288 q 565 398 579 347 q 529 486 551 449 q 476 544 506 523 q 414 566 446 566 q 383 558 402 566 q 339 533 364 551 q 287 486 315 515 q 230 413 259 456 l 230 144 q 285 106 259 121 q 333 83 311 91 q 375 71 355 74 q 413 68 395 68 q 479 82 449 68 q 531 124 509 96 q 566 193 553 152 q 579 288 579 234 m 669 333 q 659 253 669 294 q 631 172 649 211 q 590 97 614 133 q 538 36 566 62 q 478 -4 509 10 q 414 -20 447 -20 q 326 2 375 -20 q 230 66 277 24 l 230 -253 q 254 -275 230 -264 q 342 -295 279 -286 l 342 -325 l 37 -325 l 37 -295 q 105 -275 80 -285 q 130 -253 130 -265 l 130 482 q 127 524 130 507 q 115 550 124 540 q 87 564 105 560 q 37 569 68 569 l 37 596 q 80 607 59 601 q 120 619 100 613 q 158 633 139 625 q 196 651 177 641 l 222 627 l 222 492 q 292 563 259 533 q 356 612 326 593 q 413 641 387 631 q 459 651 440 651 q 543 629 504 651 q 609 566 581 607 q 653 466 637 525 q 669 333 669 407 "
    },
    "S": {
      "x_min": 80,
      "x_max": 589,
      "ha": 649,
      "o": "m 589 255 q 581 193 589 225 q 556 130 573 161 q 516 72 540 99 q 458 24 491 45 q 384 -7 426 4 q 293 -20 342 -20 q 245 -15 271 -20 q 191 -2 219 -10 q 137 18 164 6 q 87 46 110 30 q 81 69 82 48 q 80 116 80 89 q 80 172 80 144 q 87 219 80 201 l 114 216 q 152 147 129 176 q 203 98 175 117 q 263 70 231 79 q 330 61 296 61 q 390 73 360 61 q 444 107 420 86 q 483 158 468 129 q 499 219 499 187 q 481 290 499 261 q 434 343 463 320 q 367 385 405 366 q 291 421 330 403 q 215 460 252 440 q 149 507 178 480 q 102 570 120 535 q 85 655 85 606 q 90 701 85 676 q 110 750 96 725 q 144 797 123 774 q 195 837 165 820 q 263 864 224 854 q 350 875 301 875 q 410 870 380 875 q 466 857 440 865 q 514 837 493 849 q 547 812 535 826 q 547 796 552 808 q 534 770 543 784 q 514 743 525 756 q 495 722 503 729 l 471 726 q 432 763 453 748 q 391 789 412 779 q 349 802 370 798 q 310 807 329 807 q 247 795 273 807 q 203 766 220 783 q 178 727 186 749 q 171 687 171 706 q 188 631 171 655 q 236 585 206 606 q 303 546 265 564 q 379 508 340 528 q 456 465 419 488 q 523 412 494 442 q 571 344 553 382 q 589 255 589 305 "
    },
    "á»…": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 290 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 548 740 q 535 721 540 727 q 519 710 531 715 l 338 891 l 158 710 q 149 715 152 712 q 142 721 145 718 q 136 729 139 724 q 128 740 133 734 l 303 997 l 374 997 l 548 740 m 565 1218 q 536 1163 553 1190 q 499 1114 519 1136 q 455 1079 479 1093 q 405 1066 431 1066 q 361 1077 382 1066 q 320 1104 340 1089 q 281 1130 300 1118 q 242 1142 261 1142 q 194 1120 216 1142 q 147 1060 172 1098 l 112 1074 q 140 1129 124 1101 q 177 1178 156 1156 q 221 1213 197 1199 q 270 1227 245 1227 q 318 1215 296 1227 q 361 1189 341 1203 q 399 1162 381 1174 q 432 1151 416 1151 q 482 1172 458 1151 q 529 1233 506 1194 l 565 1218 "
    },
    "/": {
      "x_min": 32.5625,
      "x_max": 604.125,
      "ha": 639,
      "o": "m 140 -192 q 122 -202 133 -197 q 100 -211 112 -207 q 76 -220 88 -216 q 57 -227 65 -224 l 32 -210 l 498 1051 q 538 1071 516 1063 q 579 1085 560 1079 l 604 1070 l 140 -192 "
    },
    "â±¡": {
      "x_min": 46.71875,
      "x_max": 388.265625,
      "ha": 435,
      "o": "m 373 434 l 268 434 l 268 70 q 290 49 268 60 q 367 29 312 38 l 367 0 l 68 0 l 68 29 q 116 38 96 33 q 146 49 135 44 q 163 59 158 54 q 168 70 168 65 l 168 434 l 61 434 l 46 448 q 53 471 50 459 q 61 495 56 484 l 168 495 l 168 570 l 61 570 l 46 584 q 53 607 50 595 q 61 631 56 620 l 168 631 l 168 879 q 164 926 168 909 q 149 952 160 943 q 120 963 138 960 q 74 970 102 966 l 74 997 q 162 1018 122 1006 q 244 1051 202 1029 l 268 1028 l 268 631 l 373 631 l 388 614 l 373 570 l 268 570 l 268 495 l 373 495 l 388 478 l 373 434 "
    },
    "á»Œ": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 476 -189 q 470 -225 476 -208 q 453 -254 464 -242 q 429 -274 443 -267 q 398 -282 415 -282 q 355 -265 368 -282 q 343 -217 343 -248 q 348 -181 343 -198 q 365 -151 354 -164 q 389 -132 376 -139 q 419 -125 403 -125 q 476 -189 476 -125 "
    },
    "Ì¨": {
      "x_min": -468,
      "x_max": -215.84375,
      "ha": 0,
      "o": "m -215 -202 q -248 -238 -229 -221 q -287 -270 -267 -256 q -327 -292 -307 -284 q -364 -301 -347 -301 q -402 -296 -383 -301 q -435 -282 -420 -292 q -459 -252 -450 -271 q -468 -202 -468 -233 q -415 -83 -468 -141 q -263 29 -363 -25 l -233 16 q -310 -42 -280 -13 q -355 -95 -339 -70 q -376 -141 -371 -120 q -382 -179 -382 -162 q -366 -216 -382 -204 q -324 -228 -350 -228 q -283 -215 -308 -228 q -231 -177 -259 -203 l -215 -202 "
    },
    "Ì‹": {
      "x_min": -478.515625,
      "x_max": -74.4375,
      "ha": 0,
      "o": "m -440 709 q -458 716 -451 712 q -478 728 -465 720 l -361 1009 q -342 1006 -353 1008 q -320 1002 -331 1004 q -299 997 -309 1000 q -282 992 -289 995 l -267 964 l -440 709 m -247 709 q -266 716 -259 712 q -285 728 -273 720 l -168 1009 q -149 1006 -160 1008 q -127 1002 -138 1004 q -106 997 -116 1000 q -89 992 -96 995 l -74 964 l -247 709 "
    },
    "y": {
      "x_min": -31,
      "x_max": 658,
      "ha": 672,
      "o": "m 658 601 q 625 592 638 596 q 604 583 612 588 q 592 571 596 578 q 584 555 588 565 l 362 -55 q 300 -184 334 -130 q 229 -272 266 -237 q 155 -322 192 -306 q 84 -339 117 -339 q 38 -335 59 -339 q 1 -327 16 -332 q -22 -315 -13 -322 q -31 -303 -31 -309 q -22 -286 -31 -299 q -2 -260 -14 -274 q 22 -231 9 -245 q 47 -211 36 -218 q 110 -232 79 -232 q 167 -223 142 -232 q 195 -204 179 -219 q 227 -168 211 -189 q 259 -119 243 -146 q 286 -62 274 -91 l 306 -15 l 87 555 q 64 584 81 574 q 14 601 47 593 l 14 630 l 271 630 l 271 601 q 228 594 245 598 q 203 586 212 591 q 193 573 194 581 q 195 555 191 566 l 359 123 l 512 555 q 513 572 515 565 q 502 584 511 579 q 476 593 493 589 q 436 601 460 597 l 436 631 l 658 631 l 658 601 "
    },
    "Î ": {
      "x_min": 33,
      "x_max": 812,
      "ha": 858,
      "o": "m 33 0 l 33 29 q 104 50 78 38 q 130 70 130 61 l 130 783 q 105 803 130 791 q 33 825 80 816 l 33 855 l 812 855 l 812 825 q 740 804 766 816 q 715 783 715 793 l 715 70 q 739 50 715 62 q 812 29 763 38 l 812 0 l 512 0 l 512 29 q 583 50 557 38 q 609 70 609 61 l 609 750 q 600 766 609 758 q 573 781 591 774 q 530 792 556 788 q 472 797 505 797 l 361 797 q 306 792 330 797 q 267 781 283 788 q 243 767 251 775 q 236 750 236 759 l 236 70 q 260 50 236 62 q 333 29 284 38 l 333 0 l 33 0 "
    },
    "ËŠ": {
      "x_min": 0,
      "x_max": 267.171875,
      "ha": 267,
      "o": "m 37 709 q 16 717 25 712 q 0 728 7 722 l 155 1009 q 176 1007 163 1009 q 203 1003 189 1005 q 231 998 217 1001 q 251 994 244 996 l 267 967 l 37 709 "
    },
    "á¸¦": {
      "x_min": 33,
      "x_max": 844,
      "ha": 890,
      "o": "m 33 0 l 33 29 q 104 49 78 38 q 130 70 130 60 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 478 l 641 478 l 641 783 q 617 803 641 791 q 545 825 592 815 l 545 854 l 844 854 l 844 825 q 772 804 798 815 q 747 783 747 792 l 747 70 q 771 50 747 62 q 844 29 795 38 l 844 0 l 545 0 l 545 29 q 615 49 590 38 q 641 70 641 60 l 641 417 l 236 417 l 236 70 q 259 50 236 62 q 332 29 283 38 l 332 0 l 33 0 m 638 1044 q 632 1008 638 1025 q 615 979 626 991 q 591 959 605 966 q 560 952 577 952 q 517 968 530 952 q 505 1016 505 985 q 510 1052 505 1035 q 527 1082 516 1069 q 551 1101 538 1094 q 581 1109 565 1109 q 638 1044 638 1109 m 372 1044 q 366 1008 372 1025 q 349 979 360 991 q 325 959 339 966 q 294 952 311 952 q 251 968 264 952 q 239 1016 239 985 q 244 1052 239 1035 q 261 1082 250 1069 q 285 1101 272 1094 q 315 1109 299 1109 q 372 1044 372 1109 "
    },
    "â€“": {
      "x_min": 40.546875,
      "x_max": 598.140625,
      "ha": 639,
      "o": "m 598 370 q 589 338 594 355 q 578 309 584 320 l 57 309 l 40 325 q 49 356 43 340 q 61 387 55 373 l 582 387 l 598 370 "
    },
    "Ã«": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 502 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 199 597 165 573 q 232 617 214 607 q 270 634 251 627 q 310 646 290 642 q 347 651 330 651 q 424 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 538 855 q 532 819 538 836 q 515 790 526 802 q 491 770 505 777 q 460 763 477 763 q 417 779 430 763 q 405 827 405 796 q 410 863 405 846 q 427 893 416 880 q 451 912 438 905 q 481 920 465 920 q 538 855 538 920 m 272 855 q 266 819 272 836 q 249 790 260 802 q 225 770 239 777 q 194 763 211 763 q 151 779 164 763 q 139 827 139 796 q 144 863 139 846 q 161 893 150 880 q 185 912 172 905 q 215 920 199 920 q 272 855 272 920 "
    },
    "Æ’": {
      "x_min": -166,
      "x_max": 527,
      "ha": 394,
      "o": "m 527 986 q 517 969 527 980 q 494 947 507 959 q 466 925 481 936 q 442 911 452 915 q 409 935 426 925 q 377 953 392 946 q 350 964 363 960 q 329 968 337 968 q 292 955 312 968 q 255 912 272 943 q 227 827 238 881 q 217 689 217 773 l 217 631 l 388 631 l 406 611 q 392 591 400 601 q 374 570 383 580 q 356 553 365 560 q 340 543 347 545 q 296 559 325 551 q 217 567 268 567 l 217 66 q 206 -68 217 -13 q 177 -163 195 -124 q 135 -228 159 -202 q 84 -276 111 -255 q 46 -302 66 -290 q 7 -322 26 -314 q -28 -334 -11 -330 q -58 -339 -45 -339 q -97 -333 -78 -339 q -132 -320 -116 -328 q -156 -304 -147 -312 q -166 -292 -166 -297 q -156 -276 -166 -287 q -134 -253 -147 -265 q -106 -232 -120 -242 q -82 -217 -92 -222 q -57 -233 -68 -227 q -35 -242 -46 -239 q -14 -246 -24 -245 q 9 -248 -3 -248 q 47 -236 28 -248 q 81 -193 66 -224 q 106 -107 96 -161 q 116 30 116 -53 l 116 567 l 27 567 l 13 586 l 65 631 l 116 631 l 116 652 q 126 786 116 732 q 156 878 137 841 q 198 941 174 916 q 249 988 222 967 q 288 1015 267 1003 q 329 1034 309 1026 q 369 1046 350 1042 q 400 1051 387 1051 q 443 1042 420 1051 q 483 1024 465 1034 q 514 1002 502 1013 q 527 986 527 992 "
    },
    "ÈŸ": {
      "x_min": 37,
      "x_max": 732,
      "ha": 752,
      "o": "m 446 0 l 446 29 q 516 51 493 42 q 539 70 539 61 l 539 429 q 534 496 539 470 q 518 537 529 522 q 491 557 507 552 q 453 563 475 563 q 403 550 430 563 q 347 514 376 538 q 288 455 318 491 q 230 372 258 419 l 230 70 q 254 49 230 60 q 323 29 279 38 l 323 0 l 37 0 l 37 29 q 105 50 80 40 q 130 70 130 59 l 130 880 q 127 924 130 908 q 115 949 125 940 q 87 962 106 958 q 37 969 68 965 l 37 996 q 87 1007 64 1002 q 130 1019 110 1013 q 168 1033 150 1026 q 205 1051 186 1041 l 230 1028 l 230 464 q 293 541 258 507 q 363 600 328 576 q 432 638 398 625 q 492 651 466 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 70 q 659 52 639 61 q 732 29 679 42 l 732 0 l 446 0 m 420 1113 l 348 1113 l 173 1321 q 181 1332 178 1327 q 187 1340 184 1337 q 194 1346 190 1344 q 203 1353 198 1349 l 386 1209 l 565 1353 q 581 1340 577 1346 q 594 1321 586 1334 l 420 1113 "
    },
    "áºŽ": {
      "x_min": -0.265625,
      "x_max": 781,
      "ha": 808,
      "o": "m 239 0 l 239 29 q 323 55 299 42 q 348 78 348 68 l 348 364 q 292 478 323 419 q 228 594 260 538 q 165 699 195 651 q 111 780 134 748 q 97 794 105 788 q 78 806 90 801 q 48 814 66 811 q 2 818 30 818 l 0 846 q 78 856 38 852 q 145 861 117 861 q 197 834 175 861 q 250 757 221 802 q 307 663 278 713 q 363 562 335 614 q 413 461 390 510 l 592 780 q 585 808 602 798 q 518 825 569 818 l 518 854 l 781 854 l 781 825 q 711 807 735 816 q 677 780 686 798 l 454 366 l 454 78 q 459 68 454 73 q 478 55 465 62 q 512 42 491 48 q 565 29 533 35 l 565 0 l 239 0 m 468 1044 q 462 1008 468 1025 q 445 979 456 991 q 421 959 435 966 q 390 952 407 952 q 347 968 360 952 q 335 1016 335 985 q 340 1052 335 1035 q 357 1082 346 1069 q 381 1101 368 1094 q 411 1109 395 1109 q 468 1044 468 1109 "
    },
    "J": {
      "x_min": -126,
      "x_max": 369,
      "ha": 416,
      "o": "m 369 824 q 298 803 323 815 q 273 782 273 792 l 273 138 q 262 3 273 57 q 232 -87 251 -50 q 190 -149 214 -124 q 140 -195 166 -174 q 101 -222 122 -210 q 57 -241 79 -233 q 15 -253 35 -249 q -16 -258 -3 -258 q -56 -251 -36 -258 q -91 -236 -76 -245 q -116 -219 -106 -228 q -126 -203 -126 -210 q -117 -188 -126 -198 q -96 -168 -108 -178 q -71 -148 -84 -157 q -48 -136 -58 -139 q -16 -156 -30 -148 q 9 -168 -2 -164 q 32 -173 21 -172 q 56 -175 43 -175 q 93 -163 73 -175 q 129 -122 113 -151 q 156 -41 145 -92 q 167 89 167 10 l 167 782 q 161 791 167 786 q 141 801 156 796 q 102 812 127 807 q 37 824 77 818 l 37 854 l 369 854 l 369 824 "
    },
    "Å·": {
      "x_min": -31,
      "x_max": 658,
      "ha": 672,
      "o": "m 658 601 q 625 592 638 596 q 604 583 612 588 q 592 571 596 578 q 584 555 588 565 l 362 -55 q 300 -184 334 -130 q 229 -272 266 -237 q 155 -322 192 -306 q 84 -339 117 -339 q 38 -335 59 -339 q 1 -327 16 -332 q -22 -315 -13 -322 q -31 -303 -31 -309 q -22 -286 -31 -299 q -2 -260 -14 -274 q 22 -231 9 -245 q 47 -211 36 -218 q 110 -232 79 -232 q 167 -223 142 -232 q 195 -204 179 -219 q 227 -168 211 -189 q 259 -119 243 -146 q 286 -62 274 -91 l 306 -15 l 87 555 q 64 584 81 574 q 14 601 47 593 l 14 630 l 271 630 l 271 601 q 228 594 245 598 q 203 586 212 591 q 193 573 194 581 q 195 555 191 566 l 359 123 l 512 555 q 513 572 515 565 q 502 584 511 579 q 476 593 493 589 q 436 601 460 597 l 436 631 l 658 631 l 658 601 m 562 740 q 550 721 554 728 q 534 710 545 715 l 352 892 l 172 710 q 163 715 167 713 q 156 721 159 718 q 150 730 153 725 q 142 740 147 734 l 318 998 l 388 998 l 562 740 "
    },
    "Å•": {
      "x_min": 37,
      "x_max": 517.765625,
      "ha": 540,
      "o": "m 512 626 q 517 602 517 621 q 512 556 517 582 q 499 503 507 530 q 483 458 491 476 l 455 458 q 443 504 451 485 q 426 534 436 523 q 405 550 417 545 q 381 556 394 556 q 347 543 366 556 q 307 504 327 530 q 266 435 286 477 q 230 336 245 393 l 230 70 q 254 49 230 60 q 342 29 279 38 l 342 0 l 37 0 l 37 29 q 105 49 80 39 q 130 70 130 58 l 130 465 q 128 502 130 487 q 125 527 127 518 q 120 542 123 537 q 114 551 117 547 q 104 560 110 556 q 89 565 98 563 q 68 568 80 568 q 37 568 55 568 l 37 597 q 121 620 80 608 q 196 651 163 632 l 224 627 l 224 472 q 267 543 246 510 q 312 599 288 575 q 362 637 336 623 q 417 651 388 651 q 463 645 438 651 q 512 626 487 640 m 271 709 q 249 717 259 712 q 233 728 240 722 l 388 1009 q 409 1007 396 1009 q 436 1003 422 1005 q 464 998 451 1001 q 485 994 477 996 l 500 967 l 271 709 "
    },
    "á¹": {
      "x_min": 37,
      "x_max": 531.0625,
      "ha": 540,
      "o": "m 512 626 q 517 602 517 621 q 512 556 517 582 q 499 504 507 531 q 483 458 491 476 l 455 458 q 443 504 451 485 q 426 534 436 523 q 405 550 417 545 q 381 556 394 556 q 347 543 366 556 q 307 504 327 530 q 266 435 286 477 q 230 336 245 394 l 230 70 q 254 49 230 60 q 342 29 279 38 l 342 0 l 37 0 l 37 29 q 105 49 80 39 q 130 70 130 59 l 130 465 q 128 502 130 487 q 125 527 127 518 q 120 542 123 537 q 114 551 117 547 q 104 560 109 556 q 89 565 98 563 q 68 568 80 568 q 37 568 55 568 l 37 597 q 121 620 79 608 q 196 651 163 632 l 224 627 l 224 473 q 266 543 246 510 q 311 599 287 575 q 362 637 335 623 q 417 651 388 651 q 463 645 438 651 q 512 626 487 640 m 246 -189 q 240 -225 246 -208 q 223 -254 234 -242 q 199 -274 213 -267 q 168 -282 185 -282 q 125 -265 138 -282 q 113 -217 113 -248 q 118 -181 113 -198 q 135 -151 124 -164 q 159 -132 146 -139 q 189 -125 173 -125 q 246 -189 246 -125 m 531 868 q 526 851 529 861 q 519 831 523 841 q 511 812 515 822 q 505 797 507 803 l 95 797 l 81 813 q 85 830 82 821 q 93 850 89 840 q 101 869 97 859 q 108 885 105 878 l 516 885 l 531 868 "
    },
    "Ë˜": {
      "x_min": 26.578125,
      "x_max": 458.578125,
      "ha": 485,
      "o": "m 458 923 q 413 830 438 868 q 361 769 388 793 q 303 736 333 746 q 243 726 273 726 q 180 736 211 726 q 122 769 150 746 q 70 830 95 793 q 26 923 46 868 q 42 941 34 935 q 61 953 50 948 q 100 888 78 915 q 145 846 121 862 q 194 823 169 830 q 241 816 219 816 q 290 823 264 816 q 339 846 315 830 q 385 888 364 862 q 423 953 407 915 q 442 941 434 948 q 458 923 450 935 "
    },
    "áº‹": {
      "x_min": 13,
      "x_max": 675,
      "ha": 684,
      "o": "m 408 0 l 408 30 q 438 32 424 30 q 461 41 453 35 q 470 58 470 47 q 456 88 470 70 l 325 271 l 199 89 q 189 59 187 70 q 203 42 192 48 q 232 33 215 36 q 265 29 249 30 l 265 0 l 13 0 l 13 29 q 57 39 39 33 q 89 54 76 46 q 111 73 102 63 q 127 93 120 82 l 289 322 l 132 540 q 115 563 124 553 q 94 582 107 574 q 64 594 82 589 q 20 600 46 600 l 20 631 l 299 631 l 299 601 q 263 596 278 600 q 240 586 248 592 q 234 569 233 579 q 247 542 235 558 l 354 391 l 456 542 q 470 569 468 558 q 464 586 471 580 q 442 596 457 593 q 408 601 428 600 l 408 631 l 661 631 l 661 600 q 577 583 607 600 q 530 540 547 566 l 391 340 l 568 93 q 585 73 576 82 q 607 54 595 63 q 636 38 619 45 q 675 29 653 32 l 675 0 l 408 0 m 410 854 q 404 818 410 835 q 387 789 398 801 q 363 769 377 776 q 332 762 349 762 q 289 778 302 762 q 277 826 277 795 q 282 862 277 845 q 299 892 288 879 q 323 911 310 904 q 353 919 337 919 q 410 854 410 919 "
    },
    "D": {
      "x_min": 27.015625,
      "x_max": 746,
      "ha": 806,
      "o": "m 300 816 q 235 816 266 816 l 235 104 q 241 80 235 89 q 277 62 250 68 q 356 57 304 57 q 450 79 402 57 q 540 148 499 102 q 607 265 581 195 q 633 431 633 335 q 611 592 633 521 q 546 713 589 663 q 442 789 504 763 q 300 816 380 816 m 746 456 q 729 305 746 372 q 683 188 712 239 q 617 101 654 137 q 538 43 579 66 q 453 10 496 20 q 371 0 410 0 l 33 0 l 33 29 q 103 49 78 38 q 129 70 129 61 l 129 805 q 78 798 102 801 q 33 791 54 794 l 27 833 q 94 850 56 842 q 175 864 132 858 q 261 871 217 871 q 343 871 304 871 q 510 842 436 871 q 637 761 585 814 q 717 631 689 708 q 746 456 746 554 "
    },
    "Äº": {
      "x_min": 40,
      "x_max": 413.4375,
      "ha": 369,
      "o": "m 40 0 l 40 29 q 88 38 68 33 q 118 49 107 44 q 135 59 130 54 q 140 70 140 65 l 140 878 q 136 926 140 909 q 121 951 132 943 q 92 963 110 960 q 46 969 74 966 l 46 996 q 134 1017 94 1006 q 216 1051 174 1029 l 240 1028 l 240 70 q 262 49 240 60 q 339 29 284 38 l 339 0 l 40 0 m 102 1143 q 85 1155 90 1148 q 72 1174 80 1161 l 329 1389 q 346 1378 335 1385 q 370 1365 358 1372 q 392 1351 382 1358 q 407 1339 402 1344 l 413 1308 l 102 1143 "
    },
    "Å‚": {
      "x_min": 36.375,
      "x_max": 359.296875,
      "ha": 383,
      "o": "m 47 0 l 47 29 q 95 38 75 33 q 125 49 114 44 q 142 59 137 54 q 147 70 147 65 l 147 459 l 52 391 l 36 402 q 39 417 37 411 q 42 428 40 422 q 46 442 43 434 q 55 465 49 450 l 147 530 l 147 878 q 143 926 147 909 q 128 951 139 943 q 99 963 117 960 q 53 969 81 966 l 53 996 q 141 1017 101 1006 q 223 1051 181 1029 l 247 1028 l 247 600 l 344 670 l 359 659 q 355 641 357 648 q 352 628 354 634 q 347 616 350 623 q 340 598 344 609 l 247 530 l 247 70 q 269 49 247 60 q 346 29 291 38 l 346 0 l 47 0 "
    },
    "ï‰ª": {
      "x_min": 112,
      "x_max": 218,
      "ha": 340,
      "o": "m 218 477 q 196 471 209 474 q 169 467 183 469 q 140 464 155 465 q 112 463 125 463 l 112 1031 q 130 1041 117 1035 q 157 1054 143 1048 q 183 1065 171 1060 q 202 1071 196 1069 l 218 1053 l 218 477 "
    },
    "$": {
      "x_min": 60.375,
      "x_max": 573,
      "ha": 639,
      "o": "m 149 649 q 159 604 149 623 q 188 569 170 585 q 232 541 207 553 q 287 517 258 528 l 287 755 q 222 745 248 755 q 179 720 195 735 q 156 686 163 705 q 149 649 149 667 m 483 219 q 472 275 483 251 q 443 318 461 300 q 399 351 424 337 q 347 378 374 365 l 347 76 q 398 91 373 79 q 441 122 422 103 q 471 165 460 140 q 483 219 483 190 m 347 -97 q 336 -105 341 -102 q 328 -110 332 -108 q 318 -114 323 -112 q 305 -118 313 -116 l 287 -103 l 287 -7 q 227 -2 255 -7 q 174 10 200 1 q 122 32 148 19 q 67 64 96 45 q 62 87 64 68 q 60 132 60 106 q 60 184 60 157 q 67 230 60 211 l 96 227 q 177 121 127 157 q 287 77 226 85 l 287 401 q 206 434 245 417 q 135 475 166 451 q 84 533 103 499 q 65 617 65 567 q 75 678 65 645 q 112 740 86 711 q 181 790 138 769 q 287 819 223 812 l 287 918 q 300 927 296 924 q 307 931 304 929 q 315 934 311 933 q 329 937 319 935 l 347 923 l 347 822 q 406 817 377 822 q 460 805 435 813 q 505 785 485 796 q 539 761 525 774 q 537 744 542 757 q 523 717 533 731 q 503 688 514 702 q 483 666 491 674 l 457 671 q 402 721 430 702 q 347 748 374 740 l 347 493 q 428 458 388 477 q 501 411 469 439 q 553 346 533 384 q 573 255 573 308 q 558 172 573 214 q 515 94 544 130 q 444 32 487 58 q 347 -2 402 5 l 347 -97 "
    },
    "w": {
      "x_min": 13,
      "x_max": 947,
      "ha": 961,
      "o": "m 947 601 q 914 592 926 597 q 895 583 902 588 q 885 573 888 579 q 880 559 882 567 l 733 40 q 718 14 729 25 q 692 -2 706 4 q 664 -13 678 -9 q 642 -20 651 -17 l 484 439 l 348 40 q 332 14 343 24 q 307 -3 321 3 q 281 -14 294 -10 q 259 -20 268 -17 l 85 559 q 13 601 79 586 l 13 631 l 270 631 l 270 601 q 218 594 237 598 q 192 583 200 589 q 184 572 184 578 q 186 559 184 565 l 311 129 l 475 631 l 521 631 l 693 129 l 808 559 q 794 584 812 575 q 731 601 776 594 l 731 631 l 947 631 l 947 601 "
    },
    "ïˆŸ": {
      "x_min": 29.015625,
      "x_max": 627.09375,
      "ha": 648,
      "o": "m 627 165 q 613 63 621 106 q 599 0 605 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 364 l 43 364 l 29 380 q 36 402 32 391 q 43 425 39 414 l 130 425 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 425 l 395 425 l 411 409 l 395 364 l 236 364 l 236 111 q 241 89 236 98 q 261 74 247 80 q 299 64 275 67 q 359 61 322 61 l 454 61 q 505 64 484 61 q 542 79 526 67 q 571 114 558 91 q 598 177 584 137 l 627 165 "
    },
    "Ã‡": {
      "x_min": 47,
      "x_max": 676.765625,
      "ha": 730,
      "o": "m 498 -155 q 486 -203 498 -180 q 450 -245 474 -226 q 386 -278 425 -264 q 293 -301 347 -292 l 278 -267 q 337 -252 313 -261 q 376 -232 361 -243 q 399 -209 392 -221 q 406 -187 406 -198 q 388 -154 406 -164 q 330 -141 371 -145 q 337 -122 331 -139 q 350 -78 341 -108 q 369 -19 358 -55 q 258 8 313 -19 q 151 90 199 36 q 75 221 104 143 q 47 397 47 298 q 79 593 47 505 q 166 744 111 682 q 294 840 221 806 q 449 874 367 874 q 575 854 521 874 q 664 806 629 835 q 662 793 668 803 q 645 770 656 783 q 623 744 635 757 q 604 723 612 731 l 581 727 q 500 779 547 759 q 393 800 454 800 q 344 791 371 800 q 290 764 317 783 q 237 716 262 746 q 192 645 212 687 q 161 548 173 603 q 150 422 150 492 q 175 264 150 332 q 241 151 201 196 q 331 83 281 106 q 427 61 380 61 q 521 86 463 61 q 652 173 579 111 q 659 167 655 172 q 666 158 662 163 q 672 149 669 153 q 676 143 674 145 q 595 65 634 96 q 520 14 556 33 q 448 -12 483 -4 q 422 -16 435 -14 l 404 -70 q 439 -81 422 -74 q 469 -97 456 -87 q 490 -122 482 -107 q 498 -155 498 -136 "
    },
    "Åœ": {
      "x_min": 80,
      "x_max": 589,
      "ha": 649,
      "o": "m 589 255 q 581 193 589 225 q 556 130 573 161 q 516 72 540 99 q 458 24 491 45 q 384 -7 426 4 q 293 -20 342 -20 q 245 -15 271 -20 q 191 -2 219 -10 q 137 18 164 6 q 87 46 110 30 q 81 69 82 48 q 80 116 80 89 q 80 172 80 144 q 87 219 80 201 l 114 216 q 152 147 129 176 q 203 98 175 117 q 263 70 231 79 q 330 61 296 61 q 390 73 360 61 q 444 107 420 86 q 483 158 468 129 q 499 219 499 187 q 481 290 499 261 q 434 343 463 320 q 367 385 405 366 q 291 421 330 403 q 215 460 252 440 q 149 507 178 480 q 102 570 120 535 q 85 655 85 606 q 90 701 85 676 q 110 750 96 725 q 144 797 123 774 q 195 837 165 820 q 263 864 224 854 q 350 875 301 875 q 410 870 380 875 q 466 857 440 865 q 514 837 493 849 q 547 812 535 826 q 547 796 552 808 q 534 770 543 784 q 514 743 525 756 q 495 722 503 729 l 471 726 q 432 763 453 748 q 391 789 412 779 q 349 802 370 798 q 310 807 329 807 q 247 795 273 807 q 203 766 220 783 q 178 727 186 749 q 171 687 171 706 q 188 631 171 655 q 236 585 206 606 q 303 546 265 564 q 379 508 340 528 q 456 465 419 488 q 523 412 494 442 q 571 344 553 382 q 589 255 589 305 m 543 957 q 530 938 535 944 q 514 927 525 932 l 332 1068 l 151 927 q 142 932 146 929 q 136 938 139 935 q 129 946 133 942 q 122 957 126 951 l 297 1167 l 368 1167 l 543 957 "
    },
    "C": {
      "x_min": 47,
      "x_max": 677.046875,
      "ha": 730,
      "o": "m 677 143 q 595 65 634 97 q 520 15 556 34 q 449 -11 484 -3 q 379 -20 414 -20 q 258 8 318 -20 q 152 90 199 36 q 75 221 104 144 q 47 397 47 299 q 79 594 47 506 q 166 744 111 682 q 294 841 221 807 q 449 875 368 875 q 575 855 521 875 q 664 807 629 835 q 662 793 668 803 q 646 770 656 783 q 624 744 635 757 q 604 723 612 731 l 581 727 q 500 780 547 759 q 393 800 454 800 q 344 791 371 800 q 290 765 317 783 q 237 717 263 746 q 192 645 212 687 q 161 548 173 603 q 150 422 150 493 q 175 264 150 332 q 241 151 201 196 q 331 83 281 106 q 427 61 380 61 q 521 86 463 61 q 652 173 579 111 q 659 167 655 172 q 666 158 663 163 q 672 149 669 153 q 677 143 675 145 "
    },
    "á¸®": {
      "x_min": -3,
      "x_max": 420.59375,
      "ha": 406,
      "o": "m 47 0 l 47 29 q 117 49 92 38 q 143 70 143 61 l 143 782 q 118 803 143 790 q 47 824 94 815 l 47 854 l 346 854 l 346 824 q 274 803 300 815 q 249 782 249 792 l 249 70 q 273 50 249 62 q 346 29 298 38 l 346 0 l 47 0 m 396 1044 q 390 1008 396 1025 q 373 979 384 991 q 349 959 363 966 q 318 952 335 952 q 275 968 288 952 q 263 1016 263 985 q 268 1052 263 1035 q 285 1082 274 1069 q 309 1101 296 1094 q 339 1109 323 1109 q 396 1044 396 1109 m 130 1044 q 124 1008 130 1025 q 107 979 118 991 q 83 959 97 966 q 52 952 69 952 q 9 968 22 952 q -3 1016 -3 985 q 2 1052 -3 1035 q 19 1082 8 1069 q 43 1101 30 1094 q 73 1109 57 1109 q 130 1044 130 1109 m 108 1143 q 92 1154 97 1148 q 78 1174 87 1161 l 336 1389 q 354 1378 342 1385 q 377 1364 365 1372 q 399 1351 389 1357 q 414 1338 409 1344 l 420 1308 l 108 1143 "
    },
    "Ì‰": {
      "x_min": -455,
      "x_max": -233,
      "ha": 0,
      "o": "m -233 904 q -245 870 -233 885 q -273 842 -257 855 q -306 817 -290 829 q -328 793 -321 805 q -329 766 -336 780 q -295 736 -323 753 q -315 728 -305 729 q -336 725 -326 726 q -380 753 -366 740 q -397 776 -394 765 q -394 796 -400 787 q -376 815 -388 806 q -351 833 -365 824 q -325 851 -337 842 q -305 871 -313 861 q -297 894 -297 882 q -309 933 -297 920 q -341 946 -322 946 q -360 942 -351 946 q -374 932 -368 938 q -383 918 -380 926 q -387 903 -387 911 q -385 896 -387 900 q -381 889 -383 893 q -395 885 -386 887 q -413 880 -403 882 q -431 877 -422 878 q -447 874 -441 875 l -455 882 l -455 891 q -442 923 -455 907 q -410 951 -429 938 q -366 971 -390 963 q -319 979 -343 979 q -255 958 -278 979 q -233 904 -233 938 "
    },
    "É«": {
      "x_min": -0.09375,
      "x_max": 453.09375,
      "ha": 453,
      "o": "m 453 604 q 424 549 441 576 q 387 500 408 522 q 343 465 367 479 q 294 452 320 452 q 277 454 284 452 l 277 70 q 299 49 277 60 q 376 29 321 39 l 376 0 l 77 0 l 77 29 q 125 39 105 34 q 155 49 144 44 q 172 60 167 54 q 177 70 177 65 l 177 511 q 153 523 165 518 q 130 528 141 528 q 81 506 103 528 q 35 446 59 484 l 0 460 q 28 515 11 487 q 65 564 44 542 q 109 599 85 585 q 159 613 133 613 q 177 610 167 613 l 177 878 q 173 926 177 909 q 158 951 169 943 q 129 963 147 960 q 83 969 111 966 l 83 996 q 171 1017 131 1006 q 253 1051 211 1029 l 277 1028 l 277 555 q 299 542 288 547 q 321 537 310 537 q 370 558 346 537 q 417 619 394 580 l 453 604 "
    },
    "áºº": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 412 432 427 427 q 377 438 397 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 445 1121 q 432 1087 445 1102 q 404 1059 420 1072 q 371 1034 387 1046 q 349 1009 356 1022 q 348 983 341 997 q 382 953 354 970 q 362 944 372 946 q 341 942 351 943 q 297 969 311 957 q 280 993 283 982 q 283 1013 277 1004 q 301 1032 289 1023 q 326 1050 312 1041 q 352 1068 340 1059 q 372 1088 364 1078 q 381 1111 381 1099 q 368 1150 381 1137 q 336 1163 355 1163 q 318 1159 326 1163 q 303 1149 309 1155 q 294 1135 297 1143 q 291 1120 291 1128 q 292 1113 291 1117 q 296 1106 294 1110 q 282 1102 291 1104 q 264 1097 274 1099 q 246 1094 255 1095 q 230 1091 236 1092 l 223 1099 l 223 1108 q 235 1140 223 1124 q 267 1168 248 1155 q 311 1188 287 1180 q 358 1196 335 1196 q 422 1175 399 1196 q 445 1121 445 1155 "
    },
    "Ãˆ": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 413 432 428 427 q 377 438 398 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 451 957 q 437 937 441 944 q 421 926 432 931 l 109 1091 l 115 1121 q 130 1134 119 1127 q 152 1147 140 1140 q 175 1161 164 1155 q 194 1172 186 1167 l 451 957 "
    },
    "ï¬": {
      "x_min": 30.03125,
      "x_max": 756,
      "ha": 793,
      "o": "m 543 985 q 533 969 543 980 q 510 946 523 958 q 482 925 497 935 q 458 910 468 914 q 426 935 442 924 q 395 952 410 945 q 367 962 380 959 q 345 966 354 966 q 309 954 328 966 q 272 911 289 942 q 244 826 255 879 q 233 688 233 772 l 233 630 l 404 630 l 423 610 q 408 590 417 601 q 390 569 399 579 q 373 552 381 560 q 358 542 364 544 q 314 558 343 550 q 233 566 286 566 l 233 69 q 239 61 233 65 q 263 52 246 56 q 306 42 280 48 q 372 29 332 36 l 372 0 l 40 0 l 40 29 q 109 49 85 37 q 133 69 133 61 l 133 566 l 43 566 l 30 584 l 81 630 l 133 630 l 133 651 q 143 786 133 731 q 172 878 154 840 q 214 941 190 915 q 265 988 238 966 q 304 1014 283 1003 q 346 1034 326 1026 q 386 1046 367 1042 q 418 1051 405 1051 q 460 1042 438 1051 q 500 1024 482 1034 q 531 1002 519 1013 q 543 985 543 991 m 470 0 l 470 29 q 538 49 515 38 q 562 70 562 61 l 562 454 q 562 510 562 488 q 553 543 562 531 q 524 560 544 555 q 470 569 505 566 l 470 596 q 513 606 490 600 q 559 619 536 611 q 602 634 581 626 q 640 651 623 642 l 663 651 l 663 70 q 685 50 663 62 q 756 29 707 38 l 756 0 l 470 0 m 683 854 q 677 818 683 835 q 660 789 671 801 q 636 769 650 776 q 605 762 622 762 q 562 778 575 762 q 550 826 550 795 q 555 862 550 845 q 572 892 561 879 q 596 911 583 904 q 626 919 610 919 q 683 854 683 919 "
    },
    "ï†•": {
      "x_min": 0,
      "x_max": 271.15625,
      "ha": 271,
      "o": "m 271 731 q 253 718 262 722 q 233 709 245 713 l 0 965 l 14 993 q 34 997 21 995 q 61 1002 47 1000 q 89 1007 75 1005 q 111 1009 103 1009 l 271 731 "
    },
    "X": {
      "x_min": 21,
      "x_max": 817,
      "ha": 838,
      "o": "m 517 0 l 517 28 q 564 36 546 31 q 590 46 582 40 q 597 62 598 53 q 587 85 596 72 l 401 362 l 234 85 q 237 43 215 56 q 320 31 258 31 l 320 0 l 21 0 l 21 30 q 94 42 63 30 q 141 84 124 55 l 352 436 l 130 768 q 111 792 120 783 q 90 808 102 802 q 63 818 78 814 q 27 825 48 821 l 27 854 l 327 854 l 327 825 q 253 808 270 819 q 256 768 236 797 l 420 522 l 570 768 q 578 794 579 784 q 565 810 576 804 q 533 820 554 817 q 484 824 513 824 l 484 855 l 785 855 l 785 825 q 742 818 761 822 q 708 808 723 815 q 682 792 693 802 q 663 768 671 782 l 469 448 l 712 85 q 732 61 722 71 q 755 45 742 52 q 782 35 767 39 q 817 28 798 31 l 817 0 l 517 0 "
    },
    "Ã´": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 565 741 q 552 722 557 728 q 536 710 547 716 l 354 892 l 174 710 q 165 716 169 713 q 159 722 161 718 q 152 730 156 725 q 144 741 149 735 l 320 998 l 391 998 l 565 741 "
    },
    "á¹¹": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 0 639 11 q 585 -15 600 -9 q 563 -21 571 -21 q 524 10 539 -21 q 510 112 510 42 q 432 44 466 70 q 369 3 398 17 q 316 -15 340 -10 q 270 -21 292 -21 q 211 -12 239 -21 q 160 20 182 -3 q 125 84 138 43 q 113 189 113 125 l 113 482 q 111 532 113 514 q 100 559 109 550 q 75 572 92 568 q 27 578 57 575 l 27 606 q 72 612 51 608 q 112 622 93 616 q 152 634 132 627 q 194 650 172 642 l 213 624 l 213 226 q 219 147 213 178 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 482 q 505 531 508 513 q 492 559 502 549 q 463 574 482 570 q 414 578 445 578 l 414 606 q 506 625 462 612 q 588 650 550 637 l 608 624 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 588 912 q 560 857 576 884 q 523 808 543 830 q 478 773 502 787 q 429 760 455 760 q 384 771 405 760 q 344 798 363 783 q 304 824 324 812 q 265 836 285 836 q 217 814 239 836 q 171 754 195 792 l 135 768 q 163 823 147 795 q 200 872 180 850 q 244 907 221 893 q 294 921 268 921 q 342 909 319 921 q 384 883 364 897 q 422 856 404 868 q 456 845 440 845 q 505 866 481 845 q 552 927 530 888 l 588 912 m 333 953 q 312 961 321 955 q 295 972 303 966 l 450 1253 q 471 1250 458 1252 q 499 1247 484 1249 q 526 1242 513 1245 q 547 1237 540 1239 l 562 1211 l 333 953 "
    },
    "ïˆž": {
      "x_min": 47,
      "x_max": 492,
      "ha": 545,
      "o": "m 100 0 l 100 29 q 213 70 213 49 l 213 180 q 226 247 213 219 q 259 296 239 275 q 302 335 279 318 q 345 370 326 352 q 378 407 365 387 q 392 454 392 427 q 380 510 392 485 q 350 552 369 535 q 307 578 331 569 q 255 587 282 587 q 213 578 231 587 q 181 554 194 569 q 162 519 169 539 q 156 479 156 500 q 158 459 156 469 q 163 439 160 449 q 119 422 142 428 q 66 414 96 417 l 48 434 q 47 445 47 439 l 47 456 q 69 534 47 499 q 129 596 92 570 q 212 636 165 622 q 307 651 259 651 q 382 638 348 651 q 441 602 416 625 q 478 550 465 580 q 492 485 492 519 q 478 422 492 448 q 445 377 465 397 q 402 340 426 357 q 359 305 378 323 q 326 263 339 286 q 313 207 313 240 l 313 70 q 341 49 313 60 q 426 29 370 39 l 426 0 l 100 0 "
    },
    "Ä–": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 413 432 427 427 q 377 438 398 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 400 1044 q 394 1008 400 1025 q 377 979 388 991 q 353 959 367 966 q 322 952 339 952 q 279 968 292 952 q 267 1016 267 985 q 272 1052 267 1035 q 289 1082 278 1069 q 313 1101 300 1094 q 343 1109 327 1109 q 400 1044 400 1109 "
    },
    "áº¥": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 529 740 q 516 721 521 727 q 500 710 512 715 l 318 891 l 138 710 q 129 715 132 712 q 122 721 125 718 q 116 729 119 724 q 108 740 112 734 l 283 997 l 354 997 l 529 740 m 290 1015 q 269 1022 278 1017 q 252 1034 259 1028 l 408 1314 q 429 1312 416 1314 q 456 1308 442 1310 q 484 1304 470 1306 q 504 1299 497 1301 l 519 1272 l 290 1015 "
    },
    "Å‹": {
      "x_min": 37,
      "x_max": 639,
      "ha": 745,
      "o": "m 639 59 q 626 -74 639 -19 q 594 -166 614 -128 q 548 -229 573 -203 q 495 -276 522 -254 q 455 -303 477 -291 q 410 -322 432 -314 q 367 -334 387 -330 q 334 -339 347 -339 q 285 -332 311 -339 q 236 -315 259 -325 q 198 -293 213 -305 q 183 -272 183 -282 q 192 -255 183 -265 q 216 -233 202 -244 q 244 -211 229 -221 q 268 -196 259 -201 q 307 -226 287 -215 q 347 -245 328 -238 q 382 -255 366 -252 q 408 -258 398 -258 q 449 -245 426 -258 q 492 -201 472 -232 q 525 -114 512 -169 q 539 23 539 -60 l 539 429 q 534 496 539 470 q 519 536 529 522 q 493 556 509 551 q 453 562 477 562 q 407 552 433 562 q 353 519 382 542 q 292 461 324 497 q 229 372 260 425 l 229 70 q 254 49 229 60 q 323 29 279 38 l 323 0 l 37 0 l 37 29 q 105 49 80 40 q 130 70 130 58 l 130 482 q 128 524 130 508 q 117 548 126 539 q 89 561 108 557 q 37 569 70 565 l 37 597 q 120 618 82 604 q 196 651 159 632 l 223 627 l 223 458 q 290 539 255 503 q 362 599 326 574 q 432 637 398 624 q 491 651 465 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 59 "
    },
    "á»´": {
      "x_min": -0.265625,
      "x_max": 781,
      "ha": 808,
      "o": "m 239 0 l 239 29 q 323 55 299 42 q 348 78 348 68 l 348 364 q 292 478 323 419 q 228 594 260 538 q 165 699 195 651 q 111 780 134 748 q 97 794 105 788 q 78 806 90 801 q 48 814 66 811 q 2 818 30 818 l 0 846 q 78 856 38 852 q 145 861 117 861 q 197 834 175 861 q 250 757 221 802 q 307 663 278 713 q 363 562 335 614 q 413 461 390 510 l 592 780 q 585 808 602 798 q 518 825 569 818 l 518 854 l 781 854 l 781 825 q 711 807 735 816 q 677 780 686 798 l 454 366 l 454 78 q 459 68 454 73 q 478 55 465 62 q 512 42 491 48 q 565 29 533 35 l 565 0 l 239 0 m 468 -189 q 462 -225 468 -208 q 445 -254 456 -242 q 421 -274 435 -267 q 390 -282 407 -282 q 347 -265 360 -282 q 335 -217 335 -248 q 340 -181 335 -198 q 357 -151 346 -164 q 381 -132 368 -139 q 411 -125 395 -125 q 468 -189 468 -125 "
    },
    "ïˆ": {
      "x_min": 29.015625,
      "x_max": 627.09375,
      "ha": 648,
      "o": "m 627 165 q 613 63 621 106 q 599 0 605 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 60 l 130 296 l 43 296 l 29 312 q 36 334 32 323 q 43 357 39 346 l 130 357 l 130 432 l 43 432 l 29 448 q 36 470 32 459 q 43 493 39 482 l 130 493 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 493 l 395 493 l 411 477 l 395 432 l 236 432 l 236 357 l 395 357 l 411 341 l 395 296 l 236 296 l 236 111 q 241 89 236 98 q 261 74 247 80 q 299 64 275 67 q 359 61 322 61 l 454 61 q 505 64 484 61 q 542 79 526 67 q 571 114 558 91 q 598 177 584 137 l 627 165 "
    },
    "á¹‡": {
      "x_min": 37,
      "x_max": 732,
      "ha": 752,
      "o": "m 446 0 l 446 29 q 516 51 493 42 q 539 70 539 61 l 539 429 q 534 496 539 470 q 519 537 529 522 q 493 557 509 552 q 453 563 477 563 q 408 552 433 563 q 353 520 382 542 q 293 461 325 497 q 230 372 261 425 l 230 70 q 254 49 230 60 q 323 29 279 38 l 323 0 l 37 0 l 37 29 q 105 50 80 40 q 130 70 130 59 l 130 482 q 128 524 130 508 q 117 548 126 539 q 89 561 108 557 q 37 569 70 565 l 37 596 q 120 618 82 604 q 196 651 159 632 l 223 627 l 223 458 q 291 539 256 504 q 362 599 326 575 q 432 637 398 624 q 491 651 465 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 70 q 659 52 639 61 q 732 29 679 42 l 732 0 l 446 0 m 450 -189 q 444 -225 450 -208 q 427 -254 438 -242 q 403 -274 417 -267 q 372 -282 389 -282 q 329 -265 342 -282 q 317 -217 317 -248 q 322 -181 317 -198 q 339 -151 328 -164 q 363 -132 350 -139 q 393 -125 377 -125 q 450 -189 450 -125 "
    },
    "Çž": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 387 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 222 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 105 75 98 55 l 352 838 q 390 869 367 855 q 430 893 413 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 568 31 q 600 75 607 52 l 524 302 l 252 302 m 598 1044 q 592 1008 598 1025 q 575 979 586 991 q 551 959 565 966 q 520 952 537 952 q 477 968 490 952 q 465 1016 465 985 q 470 1052 465 1035 q 487 1082 476 1069 q 511 1101 498 1094 q 541 1109 525 1109 q 598 1044 598 1109 m 332 1044 q 326 1008 332 1025 q 309 979 320 991 q 285 959 299 966 q 254 952 271 952 q 211 968 224 952 q 199 1016 199 985 q 204 1052 199 1035 q 221 1082 210 1069 q 245 1101 232 1094 q 275 1109 259 1109 q 332 1044 332 1109 m 629 1275 q 625 1258 628 1268 q 617 1238 621 1248 q 610 1219 613 1229 q 603 1204 606 1210 l 195 1204 l 180 1220 q 185 1237 181 1228 q 192 1257 188 1247 q 200 1276 196 1266 q 207 1292 204 1285 l 615 1292 l 629 1275 "
    },
    "Ã¼": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 467 70 q 369 3 398 18 q 316 -15 340 -10 q 271 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 43 q 113 189 113 125 l 113 481 q 111 532 113 514 q 100 559 109 550 q 75 571 92 568 q 27 578 57 575 l 27 605 q 72 612 51 608 q 112 622 93 616 q 152 634 132 627 q 194 650 172 641 l 213 624 l 213 226 q 219 147 213 178 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 482 q 505 531 508 512 q 492 559 502 549 q 464 573 482 569 q 414 578 446 578 l 414 605 q 507 624 463 611 q 588 650 551 637 l 608 624 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 562 854 q 556 818 562 835 q 539 789 550 801 q 515 769 529 776 q 484 762 501 762 q 441 778 454 762 q 429 826 429 795 q 434 862 429 845 q 451 892 440 879 q 475 911 462 904 q 505 919 489 919 q 562 854 562 919 m 296 854 q 290 818 296 835 q 273 789 284 801 q 249 769 263 776 q 218 762 235 762 q 175 778 188 762 q 163 826 163 795 q 168 862 163 845 q 185 892 174 879 q 209 911 196 904 q 239 919 223 919 q 296 854 296 919 "
    },
    "Å¸": {
      "x_min": -0.046875,
      "x_max": 781,
      "ha": 808,
      "o": "m 239 0 l 239 29 q 323 55 299 42 q 348 78 348 68 l 348 364 q 292 478 323 419 q 228 594 261 538 q 165 699 195 651 q 111 780 134 748 q 97 794 105 788 q 78 806 90 801 q 48 814 66 811 q 2 818 30 818 l 0 846 q 78 856 39 852 q 145 861 117 861 q 197 834 176 861 q 250 757 221 802 q 307 663 278 713 q 363 562 336 614 q 413 461 390 510 l 592 780 q 585 808 602 798 q 518 825 569 818 l 518 854 l 781 854 l 781 825 q 711 807 735 816 q 677 780 687 798 l 454 366 l 454 78 q 459 68 454 73 q 478 55 465 62 q 512 42 491 48 q 565 29 533 35 l 565 0 l 239 0 m 601 1044 q 595 1008 601 1025 q 578 979 589 991 q 554 959 568 966 q 523 952 540 952 q 480 968 493 952 q 468 1016 468 985 q 473 1052 468 1035 q 490 1082 479 1069 q 514 1101 501 1094 q 544 1109 528 1109 q 601 1044 601 1109 m 335 1044 q 329 1008 335 1025 q 312 979 323 991 q 288 959 302 966 q 257 952 274 952 q 214 968 227 952 q 202 1016 202 985 q 207 1052 202 1035 q 224 1082 213 1069 q 248 1101 235 1094 q 278 1109 262 1109 q 335 1044 335 1109 "
    },
    "â‚¬": {
      "x_min": 16.234375,
      "x_max": 615.359375,
      "ha": 639,
      "o": "m 415 317 l 197 317 q 225 197 205 246 q 271 118 244 148 q 331 74 298 88 q 399 61 364 61 q 439 66 419 61 q 481 85 459 72 q 530 120 503 99 q 590 173 556 142 q 597 167 594 172 q 604 158 601 163 q 610 149 608 154 q 615 143 613 145 q 542 65 575 97 q 479 15 510 34 q 417 -11 448 -3 q 349 -20 386 -20 q 254 0 299 -20 q 175 61 210 20 q 117 166 140 102 q 95 317 95 230 l 30 317 l 16 333 q 23 355 19 344 q 30 378 26 367 l 90 378 l 90 380 q 94 466 90 425 l 50 466 l 36 482 q 43 504 39 493 q 50 527 46 516 l 102 527 q 144 661 116 603 q 214 759 173 719 q 307 818 255 798 q 421 838 360 838 q 480 832 454 838 q 527 817 506 827 q 567 796 549 808 q 602 770 585 784 q 600 756 606 766 q 584 733 594 746 q 562 707 574 720 q 542 686 550 694 l 519 690 q 488 719 503 706 q 455 742 473 732 q 416 757 438 752 q 365 763 395 763 q 317 753 342 763 q 269 716 292 743 q 227 644 245 690 q 199 527 208 598 l 435 527 l 452 511 l 435 466 l 194 466 q 194 436 194 451 q 194 405 194 421 l 194 378 l 415 378 l 432 362 l 415 317 "
    },
    "ÃŸ": {
      "x_min": 36.953125,
      "x_max": 709,
      "ha": 769,
      "o": "m 709 191 q 696 113 709 151 q 656 45 683 75 q 587 -2 629 15 q 485 -21 544 -21 q 439 -15 466 -21 q 385 -3 412 -10 q 338 13 359 4 q 312 31 317 23 q 309 46 310 34 q 309 74 309 58 q 309 109 309 91 q 310 144 309 127 q 315 174 312 161 q 321 191 317 186 l 346 182 q 368 120 351 146 q 408 78 384 94 q 458 54 431 61 q 510 47 485 47 q 554 54 533 47 q 590 74 575 61 q 614 106 605 87 q 623 146 623 124 q 610 208 623 181 q 575 254 597 234 q 527 290 554 274 q 472 321 500 306 q 416 352 443 336 q 368 388 389 368 q 333 433 346 408 q 321 493 321 459 q 337 568 321 539 q 378 615 353 597 q 431 647 402 634 q 484 674 460 660 q 525 708 509 688 q 542 760 542 727 q 532 831 542 795 q 503 897 523 867 q 452 946 483 927 q 376 966 420 966 q 317 954 344 966 q 270 911 290 942 q 240 826 251 879 q 230 688 230 772 l 230 0 l 36 0 l 36 28 q 106 48 82 36 q 130 68 130 60 l 130 651 q 140 787 130 732 q 169 879 151 842 q 211 942 187 917 q 261 988 235 966 q 341 1034 297 1018 q 434 1051 386 1051 q 531 1026 491 1051 q 596 966 571 1002 q 632 886 621 929 q 643 806 643 844 q 626 715 643 751 q 583 658 609 680 q 528 621 558 635 q 473 594 498 607 q 431 565 448 581 q 414 521 414 548 q 426 482 414 499 q 460 451 439 466 q 507 424 481 437 q 561 396 534 411 q 615 363 589 381 q 662 321 642 345 q 696 265 683 297 q 709 191 709 233 "
    },
    "Ç©": {
      "x_min": 37,
      "x_max": 681.265625,
      "ha": 681,
      "o": "m 37 1 l 37 29 q 105 49 80 41 q 130 70 130 58 l 130 879 q 126 927 130 910 q 112 952 122 944 q 83 963 102 961 q 37 969 65 966 l 37 997 q 127 1019 82 1006 q 205 1051 171 1031 l 230 1028 l 230 359 l 434 542 q 461 573 456 561 q 459 591 465 585 q 438 599 453 597 q 406 602 423 602 l 406 631 l 658 631 l 658 602 q 604 590 628 598 q 551 559 579 583 l 324 374 l 574 75 q 594 56 584 64 q 616 45 604 49 q 643 41 628 41 q 677 41 658 41 l 681 13 q 643 4 661 8 q 607 -1 624 1 q 578 -4 591 -3 q 558 -6 565 -6 q 514 2 531 -6 q 482 29 497 10 l 230 354 l 230 70 q 232 61 230 65 q 241 52 234 57 q 263 43 249 48 q 303 29 278 37 l 303 1 l 37 1 m 394 1113 l 323 1113 l 148 1321 q 155 1332 152 1327 q 162 1340 159 1337 q 169 1346 165 1344 q 178 1353 172 1349 l 360 1209 l 540 1353 q 556 1340 551 1346 q 568 1321 560 1334 l 394 1113 "
    },
    "á»‚": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 412 432 427 427 q 377 438 397 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 543 957 q 530 938 535 944 q 515 926 526 932 l 333 1067 l 153 926 q 144 931 147 929 q 137 938 140 934 q 131 946 134 941 q 123 957 128 950 l 298 1166 l 369 1166 l 543 957 m 445 1392 q 432 1358 445 1373 q 404 1330 420 1343 q 371 1305 387 1317 q 349 1280 356 1293 q 348 1254 341 1268 q 382 1224 354 1241 q 362 1216 372 1217 q 341 1213 351 1214 q 297 1240 311 1228 q 280 1264 283 1253 q 283 1284 277 1275 q 301 1303 289 1294 q 326 1321 312 1312 q 352 1339 340 1330 q 372 1359 364 1349 q 381 1382 381 1370 q 368 1421 381 1408 q 336 1434 355 1434 q 318 1430 326 1434 q 303 1420 309 1426 q 294 1406 297 1414 q 291 1391 291 1399 q 292 1384 291 1388 q 296 1377 294 1381 q 282 1373 291 1375 q 264 1368 274 1370 q 246 1365 255 1366 q 230 1362 236 1363 l 223 1370 l 223 1379 q 235 1411 223 1395 q 267 1439 248 1426 q 311 1459 287 1451 q 358 1467 335 1467 q 422 1446 399 1467 q 445 1392 445 1426 "
    },
    "Çµ": {
      "x_min": 20,
      "x_max": 657.625,
      "ha": 664,
      "o": "m 459 406 q 447 474 459 442 q 412 531 435 507 q 355 569 389 555 q 276 583 321 583 q 240 574 260 583 q 201 548 219 565 q 171 505 183 531 q 160 446 160 479 q 170 378 160 410 q 204 322 181 346 q 260 284 226 298 q 342 271 294 271 q 382 279 361 271 q 420 305 403 287 q 448 347 437 322 q 459 406 459 372 m 344 -2 q 297 3 319 0 q 257 10 276 6 q 182 -36 210 -15 q 141 -74 155 -57 q 123 -104 127 -90 q 120 -128 120 -118 q 138 -182 120 -157 q 188 -226 157 -208 q 260 -256 219 -245 q 347 -268 301 -268 q 428 -255 391 -268 q 490 -222 464 -243 q 530 -171 516 -200 q 545 -106 545 -141 q 536 -71 545 -87 q 505 -42 527 -55 q 443 -19 482 -29 q 344 -2 405 -9 m 552 434 q 529 339 552 382 q 468 265 506 296 q 384 217 431 234 q 288 200 337 200 l 285 200 q 241 154 253 172 q 230 132 230 136 q 236 116 230 124 q 262 101 243 109 q 315 87 281 94 q 400 74 348 80 q 520 50 472 66 q 595 12 567 33 q 634 -33 623 -8 q 645 -81 645 -57 q 631 -152 645 -118 q 593 -214 617 -185 q 536 -265 569 -242 q 466 -305 504 -288 q 389 -330 429 -321 q 310 -339 349 -339 q 246 -334 279 -339 q 180 -320 212 -330 q 118 -296 147 -311 q 67 -261 89 -282 q 32 -214 45 -240 q 20 -155 20 -188 q 26 -118 20 -137 q 51 -76 32 -99 q 105 -28 71 -54 q 197 28 140 -2 q 137 63 154 44 q 121 103 121 83 q 124 118 121 109 q 138 140 127 127 q 167 170 148 153 q 216 209 186 187 q 155 236 183 218 q 108 280 128 254 q 77 337 88 305 q 67 408 67 370 q 88 502 67 457 q 146 579 110 546 q 228 631 182 612 q 322 651 274 651 q 397 639 362 651 q 460 606 432 627 q 523 615 495 610 q 573 627 551 621 q 613 639 595 633 q 644 651 630 645 l 657 630 q 642 595 649 611 q 619 562 635 579 q 570 554 594 557 q 515 552 545 552 q 542 496 532 525 q 552 434 552 467 m 303 709 q 282 717 291 712 q 265 728 273 722 l 421 1009 q 442 1007 429 1009 q 469 1003 455 1005 q 497 998 484 1001 q 517 994 510 996 l 533 967 l 303 709 "
    },
    "ïˆŽ": {
      "x_min": 46.71875,
      "x_max": 388.265625,
      "ha": 435,
      "o": "m 373 434 l 268 434 l 268 70 q 290 49 268 60 q 367 29 312 38 l 367 0 l 68 0 l 68 29 q 116 38 96 33 q 146 49 135 44 q 163 59 158 54 q 168 70 168 65 l 168 434 l 61 434 l 46 448 q 53 471 50 459 q 61 495 56 484 l 168 495 l 168 570 l 61 570 l 46 584 q 53 607 50 595 q 61 631 56 620 l 168 631 l 168 879 q 164 926 168 909 q 149 952 160 943 q 120 963 138 960 q 74 970 102 966 l 74 997 q 162 1018 122 1006 q 244 1051 202 1029 l 268 1028 l 268 631 l 373 631 l 388 614 l 373 570 l 268 570 l 268 495 l 373 495 l 388 478 l 373 434 "
    },
    "áº³": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 148 302 128 288 q 202 330 169 317 q 286 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 194 490 192 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 535 922 q 490 829 515 867 q 437 768 465 792 q 380 735 410 745 q 320 725 350 725 q 257 735 287 725 q 198 768 226 745 q 147 829 171 792 q 102 922 122 867 q 118 940 110 934 q 138 952 126 947 q 176 887 154 914 q 221 845 197 861 q 270 822 245 829 q 317 815 295 815 q 366 822 340 815 q 416 845 392 829 q 462 887 440 861 q 499 952 483 914 q 519 940 511 947 q 535 922 527 934 m 430 1121 q 417 1087 430 1102 q 389 1059 405 1072 q 357 1034 372 1046 q 334 1010 341 1022 q 333 983 326 997 q 367 953 339 970 q 347 945 357 946 q 326 942 336 943 q 282 969 296 957 q 265 993 268 982 q 268 1013 262 1004 q 286 1032 274 1023 q 311 1050 297 1041 q 337 1068 325 1059 q 357 1088 349 1078 q 366 1111 366 1099 q 353 1150 366 1137 q 321 1163 340 1163 q 303 1159 311 1163 q 288 1149 294 1155 q 279 1135 282 1143 q 276 1120 276 1128 q 277 1113 276 1117 q 281 1106 279 1110 q 267 1102 276 1104 q 249 1097 259 1099 q 231 1094 240 1095 q 215 1091 221 1092 l 208 1099 l 208 1108 q 220 1140 208 1124 q 252 1168 233 1155 q 296 1188 272 1180 q 343 1196 320 1196 q 407 1175 384 1196 q 430 1121 430 1155 "
    },
    "Å°": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 847 855 l 847 825 m 341 927 q 323 933 330 930 q 303 945 316 937 l 420 1227 q 439 1224 428 1225 q 461 1220 450 1222 q 482 1215 472 1217 q 499 1210 492 1213 l 514 1182 l 341 927 m 534 927 q 515 933 522 930 q 496 945 508 937 l 614 1227 q 632 1224 622 1225 q 654 1220 643 1222 q 675 1215 665 1217 q 693 1210 686 1213 l 708 1182 l 534 927 "
    },
    "c": {
      "x_min": 53,
      "x_max": 558.40625,
      "ha": 595,
      "o": "m 558 129 q 482 47 515 76 q 421 2 449 17 q 366 -16 393 -12 q 309 -20 340 -20 q 214 2 260 -20 q 132 65 168 24 q 74 166 96 106 q 53 301 53 226 q 79 438 53 374 q 151 549 105 502 q 258 624 196 596 q 392 651 320 651 q 437 647 414 651 q 480 637 459 644 q 519 620 501 629 q 548 598 537 610 q 546 574 549 589 q 537 543 543 560 q 523 511 531 526 q 510 486 516 496 l 485 493 q 468 519 480 505 q 437 547 456 534 q 392 568 418 560 q 332 577 366 577 q 264 561 297 577 q 207 513 232 545 q 167 433 182 481 q 153 322 153 386 q 169 217 153 264 q 215 137 186 170 q 280 85 243 103 q 357 68 317 68 q 391 69 375 68 q 426 80 407 71 q 470 106 445 89 q 532 156 495 124 l 558 129 "
    },
    "Â¶": {
      "x_min": 33,
      "x_max": 702,
      "ha": 755,
      "o": "m 296 -169 l 296 -140 q 367 -119 341 -131 q 393 -98 393 -108 l 393 360 q 367 357 381 357 q 335 357 353 357 q 219 372 273 357 q 122 419 164 388 q 56 498 80 451 q 33 607 33 544 q 55 710 33 663 q 118 792 77 757 q 214 845 158 826 q 338 864 271 864 q 374 864 352 864 q 420 864 395 864 q 472 864 445 864 q 524 864 498 864 q 653 864 585 864 l 702 864 l 702 824 q 630 803 656 814 q 604 782 604 791 l 604 -98 q 628 -119 604 -107 q 702 -140 653 -131 l 702 -169 l 296 -169 m 393 413 l 393 812 q 325 817 352 817 q 252 802 285 817 q 194 764 218 788 q 156 706 170 739 q 143 635 143 673 q 155 559 143 598 q 195 487 168 520 q 264 434 222 455 q 364 413 305 413 l 393 413 m 526 785 q 499 793 512 789 l 473 800 l 473 -98 q 482 -111 473 -106 q 498 -119 492 -117 q 516 -111 506 -117 q 526 -98 526 -106 l 526 785 "
    },
    "á»¤": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 847 855 l 847 825 m 516 -189 q 510 -225 516 -208 q 493 -254 504 -242 q 469 -274 483 -267 q 438 -282 455 -282 q 395 -265 408 -282 q 383 -217 383 -248 q 388 -181 383 -198 q 405 -151 394 -164 q 429 -132 416 -139 q 459 -125 443 -125 q 516 -189 516 -125 "
    },
    "ïˆ˜": {
      "x_min": 29.015625,
      "x_max": 853.640625,
      "ha": 890,
      "o": "m 468 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 417 l 236 417 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 m 847 825 q 776 804 801 815 q 751 783 751 792 l 751 478 l 837 478 l 853 462 l 837 417 l 751 417 l 751 355 q 729 196 751 266 q 668 78 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 97 q 130 301 130 218 l 130 417 l 43 417 l 29 433 q 36 455 32 444 q 43 478 39 467 l 130 478 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 478 l 665 478 l 665 783 q 640 803 665 791 q 568 825 616 815 l 568 854 l 847 854 l 847 825 "
    },
    "Â­": {
      "x_min": 40.546875,
      "x_max": 418.03125,
      "ha": 459,
      "o": "m 418 370 q 408 338 414 355 q 397 309 402 320 l 57 309 l 40 325 q 49 356 43 340 q 61 387 55 373 l 401 387 l 418 370 "
    },
    "á¹": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 640 1058 q 636 1041 639 1051 q 628 1021 632 1031 q 621 1002 624 1012 q 614 987 617 993 l 205 987 l 191 1003 q 195 1020 192 1011 q 203 1040 199 1030 q 211 1059 207 1049 q 218 1075 215 1068 l 626 1075 l 640 1058 m 527 1174 q 512 1155 517 1161 q 497 1144 507 1148 l 185 1308 l 190 1339 q 205 1351 195 1344 q 227 1365 215 1358 q 251 1379 239 1372 q 269 1389 262 1385 l 527 1174 "
    },
    "È³": {
      "x_min": -31,
      "x_max": 658,
      "ha": 672,
      "o": "m 658 601 q 625 592 638 596 q 604 583 612 588 q 592 571 596 578 q 584 555 588 565 l 362 -55 q 300 -184 334 -130 q 229 -272 266 -237 q 155 -322 192 -306 q 84 -339 117 -339 q 38 -335 59 -339 q 1 -327 16 -332 q -22 -315 -13 -322 q -31 -303 -31 -309 q -22 -286 -31 -299 q -2 -260 -14 -274 q 22 -231 9 -245 q 47 -211 36 -218 q 110 -232 79 -232 q 167 -223 142 -232 q 195 -204 179 -219 q 227 -168 211 -189 q 259 -119 243 -146 q 286 -62 274 -91 l 306 -15 l 87 555 q 64 584 81 574 q 14 601 47 593 l 14 630 l 271 630 l 271 601 q 228 594 245 598 q 203 586 212 591 q 193 573 194 581 q 195 555 191 566 l 359 123 l 512 555 q 513 572 515 565 q 502 584 511 579 q 476 593 493 589 q 436 601 460 597 l 436 631 l 658 631 l 658 601 m 584 868 q 579 851 582 861 q 572 831 576 841 q 564 812 568 822 q 558 797 560 803 l 149 797 l 135 813 q 139 830 136 821 q 147 850 143 840 q 155 869 151 859 q 162 885 159 878 l 569 885 l 584 868 "
    },
    "áº’": {
      "x_min": 39.109375,
      "x_max": 655,
      "ha": 712,
      "o": "m 655 198 q 655 150 655 177 q 655 97 655 124 q 655 45 655 70 q 655 0 655 19 l 58 0 l 39 30 l 511 787 l 206 787 q 183 779 196 787 q 158 755 170 772 q 136 714 146 739 q 118 653 125 688 l 80 661 l 99 865 q 132 858 117 861 q 161 855 147 855 q 191 855 176 855 l 635 855 l 652 825 l 183 68 l 531 68 q 556 74 545 68 q 578 96 567 80 q 599 139 588 112 q 622 208 609 166 l 655 198 m 422 -189 q 416 -225 422 -208 q 399 -254 410 -242 q 375 -274 389 -267 q 344 -282 361 -282 q 301 -265 314 -282 q 289 -217 289 -248 q 294 -181 289 -198 q 311 -151 300 -164 q 335 -132 322 -139 q 365 -125 349 -125 q 422 -189 422 -125 "
    },
    "á¸³": {
      "x_min": 37,
      "x_max": 681.265625,
      "ha": 681,
      "o": "m 37 0 l 37 29 q 105 49 80 41 q 130 70 130 58 l 130 879 q 126 927 130 910 q 112 952 122 944 q 83 963 102 961 q 37 970 65 966 l 37 997 q 127 1019 82 1006 q 205 1051 171 1031 l 230 1028 l 230 359 l 434 542 q 460 573 456 561 q 459 591 465 585 q 437 599 452 597 q 406 602 423 602 l 406 631 l 658 631 l 658 602 q 604 590 628 598 q 551 559 579 583 l 324 374 l 574 75 q 594 56 584 64 q 616 45 604 49 q 643 41 628 41 q 677 41 658 41 l 681 12 q 642 4 661 7 q 607 -1 624 1 q 578 -4 591 -3 q 558 -6 565 -6 q 513 1 530 -6 q 482 29 496 9 l 230 354 l 230 70 q 232 61 230 65 q 241 52 234 57 q 263 43 249 48 q 303 29 278 37 l 303 0 l 37 0 m 425 -189 q 419 -225 425 -208 q 402 -254 413 -242 q 378 -274 392 -267 q 347 -282 364 -282 q 304 -265 317 -282 q 292 -217 292 -248 q 297 -181 292 -198 q 314 -151 303 -164 q 338 -132 325 -139 q 368 -125 352 -125 q 425 -189 425 -125 "
    },
    ":": {
      "x_min": 87,
      "x_max": 245,
      "ha": 311,
      "o": "m 245 83 q 238 39 245 59 q 218 4 231 19 q 189 -18 206 -10 q 153 -27 172 -27 q 102 -7 117 -27 q 87 48 87 12 q 94 91 87 71 q 114 127 101 111 q 143 151 126 142 q 179 160 160 160 q 228 140 212 160 q 245 83 245 120 m 245 575 q 238 531 245 551 q 218 496 231 511 q 189 473 206 481 q 153 465 172 465 q 102 484 117 465 q 87 540 87 504 q 94 583 87 563 q 114 619 101 603 q 143 643 126 634 q 179 652 160 652 q 228 632 212 652 q 245 575 245 612 "
    },
    "Å›": {
      "x_min": 63,
      "x_max": 465,
      "ha": 525,
      "o": "m 465 192 q 451 109 465 144 q 417 51 438 75 q 369 13 395 28 q 318 -7 343 0 q 271 -17 293 -15 q 236 -20 248 -20 q 159 -7 204 -20 q 70 29 114 4 q 64 51 65 31 q 63 97 63 71 q 66 150 63 123 q 74 192 69 176 l 104 184 q 117 131 104 156 q 153 87 130 106 q 205 58 175 69 q 269 48 234 48 q 313 55 293 48 q 348 77 333 63 q 370 111 362 91 q 379 154 379 130 q 364 202 379 181 q 327 240 350 223 q 274 273 303 258 q 213 304 244 288 q 159 335 185 319 q 113 371 133 351 q 81 417 93 392 q 69 474 69 442 q 85 549 69 516 q 129 604 102 582 q 192 639 157 627 q 266 651 228 651 q 311 647 287 651 q 361 636 336 643 q 406 620 385 629 q 441 598 427 611 q 442 580 445 594 q 432 549 439 566 q 419 517 426 532 q 408 497 412 502 l 382 502 q 316 570 350 551 q 249 590 281 590 q 209 582 227 590 q 179 563 192 575 q 161 537 167 551 q 155 508 155 522 q 167 469 155 487 q 201 437 180 452 q 248 408 221 421 q 302 379 274 394 q 360 347 331 364 q 412 308 389 330 q 450 258 435 286 q 465 192 465 230 m 235 709 q 213 717 223 712 q 197 728 204 722 l 353 1009 q 373 1007 360 1009 q 401 1003 386 1005 q 428 998 415 1001 q 449 994 441 996 l 464 967 l 235 709 "
    },
    "Íž": {
      "x_min": -433.984375,
      "x_max": 433.984375,
      "ha": 0,
      "o": "m 433 1187 q 429 1170 432 1180 q 422 1150 426 1160 q 414 1131 418 1141 q 408 1116 410 1122 l -419 1116 l -433 1132 q -429 1149 -432 1140 q -422 1169 -426 1159 q -414 1188 -418 1178 q -406 1204 -410 1197 l 419 1204 l 433 1187 "
    },
    "áº‡": {
      "x_min": 13,
      "x_max": 947,
      "ha": 961,
      "o": "m 947 600 q 914 591 926 596 q 895 582 902 587 q 885 572 888 578 q 880 558 882 566 l 733 39 q 718 13 729 24 q 692 -3 706 3 q 664 -14 678 -10 q 642 -21 651 -18 l 484 438 l 348 39 q 332 13 343 23 q 307 -4 321 2 q 281 -15 294 -11 q 259 -21 268 -18 l 85 558 q 13 600 79 585 l 13 630 l 270 630 l 270 600 q 218 593 237 597 q 192 582 200 588 q 184 571 184 577 q 186 558 184 564 l 311 128 l 475 630 l 521 630 l 693 128 l 808 558 q 794 583 812 574 q 731 600 776 593 l 731 630 l 947 630 l 947 600 m 565 854 q 559 818 565 835 q 542 789 553 801 q 518 769 532 776 q 487 762 504 762 q 444 778 457 762 q 432 826 432 795 q 437 862 432 845 q 454 892 443 879 q 478 911 465 904 q 508 919 492 919 q 565 854 565 919 "
    },
    "Â ": {
      "x_min": 0,
      "x_max": 0,
      "ha": 300
    },
    "Â¾": {
      "x_min": 53.625,
      "x_max": 744.984375,
      "ha": 798,
      "o": "m 210 1 q 180 -10 195 -5 q 148 -20 166 -15 l 132 0 l 638 815 q 667 827 651 820 q 697 837 682 833 l 714 817 l 210 1 m 744 142 q 730 124 736 132 q 714 111 723 116 l 694 111 l 694 39 q 695 34 694 36 q 701 29 696 32 q 716 24 706 27 q 741 19 725 22 l 741 0 l 559 0 l 559 19 q 597 25 582 22 q 619 31 612 28 q 629 37 627 34 q 632 42 632 39 l 632 111 l 447 111 l 434 121 l 620 381 q 652 394 635 387 q 680 405 669 400 l 694 394 l 694 153 l 734 153 l 744 142 m 346 559 q 335 507 346 531 q 304 465 324 483 q 255 436 284 446 q 189 426 226 426 q 122 438 156 426 q 53 479 87 451 l 66 501 q 126 471 100 479 q 179 464 151 464 q 249 485 223 464 q 276 548 276 507 q 267 589 276 572 q 246 615 259 605 q 217 630 233 626 q 186 634 201 634 l 178 634 q 173 634 175 634 q 169 633 171 634 q 160 632 166 633 l 155 657 q 208 676 189 666 q 238 697 228 686 q 252 719 249 708 q 255 740 255 730 q 251 762 255 751 q 240 782 247 773 q 220 795 233 790 q 191 801 208 801 q 149 787 163 801 q 139 751 134 773 q 110 743 126 746 q 77 738 93 739 l 66 748 q 78 777 66 762 q 111 806 90 793 q 157 828 131 820 q 213 837 183 837 q 262 829 241 837 q 297 808 284 821 q 317 779 310 795 q 324 746 324 763 q 303 697 324 722 q 245 662 282 673 q 285 650 266 659 q 317 628 303 641 q 338 596 330 614 q 346 559 346 578 m 632 332 l 498 153 l 632 153 l 632 332 "
    },
    "m": {
      "x_min": 37,
      "x_max": 1075,
      "ha": 1094,
      "o": "m 789 0 l 789 29 q 858 51 836 42 q 881 70 881 61 l 881 429 q 876 498 881 470 q 863 541 872 525 q 840 563 854 557 q 805 570 825 570 q 759 557 783 570 q 708 521 734 544 q 656 464 682 498 q 605 388 630 430 l 605 70 q 625 52 605 61 q 699 29 646 43 l 699 0 l 413 0 l 413 29 q 482 51 460 42 q 505 70 505 61 l 505 429 q 501 498 505 470 q 488 541 497 525 q 464 563 479 557 q 430 570 450 570 q 335 522 385 570 q 230 388 286 475 l 230 70 q 254 49 230 60 q 323 29 279 39 l 323 0 l 37 0 l 37 29 q 105 50 80 41 q 130 70 130 59 l 130 482 q 127 525 130 509 q 115 549 125 540 q 87 561 106 557 q 37 570 68 565 l 37 597 q 83 606 62 601 q 124 619 104 612 q 161 634 143 626 q 196 651 178 642 l 223 627 l 223 472 q 287 550 257 517 q 348 606 318 584 q 406 639 378 628 q 457 651 434 651 q 516 643 489 651 q 563 616 543 635 q 594 567 582 598 q 605 492 605 537 l 605 477 q 663 552 633 520 q 722 606 693 584 q 780 639 752 628 q 833 651 808 651 q 892 642 865 651 q 938 615 919 633 q 969 568 958 596 q 981 502 981 540 l 981 70 q 1001 52 981 61 q 1075 29 1022 43 l 1075 0 l 789 0 "
    },
    "á»Š": {
      "x_min": 47,
      "x_max": 346,
      "ha": 406,
      "o": "m 47 0 l 47 29 q 118 49 92 38 q 144 70 144 61 l 144 782 q 119 803 144 790 q 47 824 94 815 l 47 854 l 346 854 l 346 824 q 275 803 300 815 q 250 782 250 792 l 250 70 q 274 50 250 62 q 346 29 298 38 l 346 0 l 47 0 m 263 -189 q 257 -225 263 -208 q 240 -254 251 -242 q 216 -274 230 -267 q 185 -282 202 -282 q 142 -265 155 -282 q 130 -217 130 -248 q 135 -181 130 -198 q 152 -151 141 -164 q 176 -132 163 -139 q 206 -125 190 -125 q 263 -189 263 -125 "
    },
    "Å¾": {
      "x_min": 50.453125,
      "x_max": 549.984375,
      "ha": 602,
      "o": "m 548 168 q 548 121 548 146 q 548 74 548 97 q 548 31 548 51 q 543 0 548 12 l 67 0 l 50 30 l 419 569 l 189 569 q 167 564 178 569 q 146 549 156 560 q 127 518 136 538 q 111 467 118 498 l 85 475 l 85 641 q 117 634 105 637 q 142 631 129 631 q 170 631 155 631 l 535 631 l 549 601 l 179 61 l 460 61 q 490 87 476 61 q 520 174 504 113 l 548 168 m 355 726 l 283 726 l 108 968 q 116 979 112 974 q 122 987 119 984 q 129 993 125 991 q 138 999 132 996 l 321 829 l 500 999 q 516 987 512 993 q 529 968 521 981 l 355 726 "
    },
    "á»›": {
      "x_min": 53,
      "x_max": 731,
      "ha": 731,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 731 707 q 723 670 731 690 q 699 626 716 649 q 652 577 681 602 q 578 527 622 551 q 617 436 604 486 q 631 329 631 387 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 465 628 415 651 q 550 565 514 606 q 601 612 585 589 q 617 652 617 635 q 606 689 617 671 q 578 722 596 707 l 696 777 q 721 744 711 762 q 731 707 731 726 m 326 710 q 305 718 314 712 q 288 729 296 723 l 444 1010 q 465 1007 452 1009 q 492 1004 478 1006 q 520 999 506 1002 q 540 994 533 996 l 556 968 l 326 710 "
    },
    "Ã¡": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 290 709 q 269 717 278 712 q 252 728 259 722 l 408 1009 q 429 1007 416 1009 q 456 1003 442 1005 q 484 998 470 1001 q 504 994 497 996 l 519 967 l 290 709 "
    },
    "Ã—": {
      "x_min": 62,
      "x_max": 496,
      "ha": 558,
      "o": "m 62 212 l 226 381 l 62 547 l 62 570 q 92 582 75 577 q 123 592 109 588 l 278 434 l 433 592 q 464 582 447 588 q 495 570 481 577 l 495 547 l 330 380 l 496 212 l 496 189 q 464 178 482 184 q 434 168 446 172 l 278 327 l 123 168 q 92 178 110 172 q 62 189 75 184 l 62 212 "
    },
    "á¸": {
      "x_min": 53,
      "x_max": 698.015625,
      "ha": 707,
      "o": "m 698 57 q 643 21 667 36 q 601 -2 619 7 q 570 -15 583 -11 q 548 -20 557 -20 q 512 10 526 -20 q 498 114 498 40 q 446 58 471 83 q 394 16 421 33 q 339 -10 368 -1 q 276 -20 310 -20 q 199 2 239 -20 q 127 65 160 24 q 73 166 94 106 q 53 301 53 226 q 63 381 53 339 q 92 461 73 422 q 140 534 112 499 q 202 595 167 568 q 278 636 237 621 q 365 651 319 651 q 428 643 397 651 q 495 608 459 635 l 495 864 q 493 924 495 902 q 481 958 491 946 q 453 974 472 970 q 398 979 433 979 l 398 1006 q 496 1026 453 1014 q 573 1050 539 1039 l 595 1030 l 595 172 q 595 131 595 148 q 597 103 595 114 q 601 84 599 91 q 608 72 604 76 q 632 67 614 64 q 689 86 649 70 l 698 57 m 495 177 l 495 495 q 433 555 473 533 q 345 576 393 576 q 267 560 303 576 q 207 512 232 544 q 167 433 181 480 q 153 322 153 385 q 169 217 153 264 q 210 137 185 170 q 266 85 236 103 q 323 68 296 68 q 368 77 346 68 q 411 102 390 86 q 453 137 432 117 q 495 177 474 156 m 415 -189 q 409 -225 415 -208 q 392 -254 403 -242 q 368 -274 382 -267 q 337 -282 354 -282 q 294 -265 307 -282 q 282 -217 282 -248 q 287 -181 282 -198 q 304 -151 293 -164 q 328 -132 315 -139 q 358 -125 342 -125 q 415 -189 415 -125 "
    },
    "Çº": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 838 q 390 869 367 855 q 430 893 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 462 1060 q 457 1091 462 1076 q 446 1117 453 1106 q 428 1135 439 1128 q 404 1142 417 1142 q 377 1136 390 1142 q 355 1120 365 1130 q 340 1094 346 1109 q 334 1061 334 1079 q 338 1031 334 1045 q 349 1005 342 1016 q 367 987 356 994 q 391 981 378 981 q 417 986 404 981 q 440 1001 430 991 q 456 1025 450 1011 q 462 1060 462 1040 m 522 1088 q 509 1024 522 1053 q 475 974 496 995 q 428 940 454 952 q 377 929 402 929 q 335 937 354 929 q 302 959 316 945 q 281 992 289 973 q 274 1033 274 1011 q 287 1097 274 1068 q 320 1148 300 1127 q 366 1182 341 1170 q 418 1194 392 1194 q 459 1185 440 1194 q 492 1162 478 1177 q 514 1128 506 1147 q 522 1088 522 1109 m 310 1225 q 293 1237 298 1230 q 280 1256 288 1243 l 537 1471 q 555 1460 544 1467 q 579 1447 567 1454 q 600 1433 590 1440 q 616 1421 610 1426 l 622 1390 l 310 1225 "
    },
    "K": {
      "x_min": 33,
      "x_max": 779.53125,
      "ha": 786,
      "o": "m 33 0 l 33 29 q 104 50 78 39 q 130 70 130 61 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 438 l 507 765 q 526 794 522 783 q 523 811 530 805 q 500 820 517 817 q 458 824 483 824 l 458 855 l 737 855 l 737 825 q 699 819 715 822 q 669 812 682 817 q 644 801 655 808 q 624 783 634 794 l 333 455 l 653 86 q 679 65 665 73 q 708 55 693 58 q 740 52 724 52 q 774 54 757 52 l 779 25 q 700 1 740 12 q 629 -9 661 -9 q 595 -3 610 -9 q 566 19 580 2 l 236 433 l 236 70 q 259 50 236 62 q 332 29 283 39 l 332 0 l 33 0 "
    },
    "7": {
      "x_min": 69.125,
      "x_max": 588.171875,
      "ha": 639,
      "o": "m 588 791 q 526 643 557 717 q 465 496 495 568 q 408 355 435 423 q 358 226 381 286 q 317 115 335 165 q 287 29 298 65 q 238 0 263 12 q 179 -20 212 -11 l 152 1 q 255 199 210 105 q 340 382 301 294 q 415 556 380 471 q 487 728 450 641 l 185 728 q 166 728 175 728 q 146 719 156 728 q 125 690 136 710 q 99 630 113 669 l 69 642 q 74 681 71 658 q 81 730 77 705 q 89 778 85 755 q 97 817 93 802 l 566 817 l 588 791 "
    },
    "Ìˆ": {
      "x_min": -542,
      "x_max": -143,
      "ha": 0,
      "o": "m -143 854 q -148 818 -143 835 q -165 789 -154 801 q -189 769 -175 776 q -220 762 -203 762 q -263 778 -250 762 q -276 826 -276 795 q -270 862 -276 845 q -253 892 -264 879 q -229 911 -242 904 q -199 919 -215 919 q -143 854 -143 919 m -409 854 q -414 818 -409 835 q -431 789 -420 801 q -455 769 -441 776 q -486 762 -469 762 q -529 778 -516 762 q -542 826 -542 795 q -536 862 -542 845 q -519 892 -530 879 q -495 911 -508 904 q -465 919 -481 919 q -409 854 -409 919 "
    },
    "Â¨": {
      "x_min": 66,
      "x_max": 465,
      "ha": 532,
      "o": "m 465 854 q 459 818 465 835 q 442 789 453 801 q 418 769 432 776 q 387 762 404 762 q 344 778 357 762 q 332 826 332 795 q 337 862 332 845 q 354 892 343 879 q 378 911 365 904 q 408 919 392 919 q 465 854 465 919 m 199 854 q 193 818 199 835 q 176 789 187 801 q 152 769 166 776 q 121 762 138 762 q 78 778 91 762 q 66 826 66 795 q 71 862 66 845 q 88 892 77 879 q 112 911 99 904 q 142 919 126 919 q 199 854 199 919 "
    },
    "Y": {
      "x_min": -0.265625,
      "x_max": 781,
      "ha": 808,
      "o": "m 239 0 l 239 29 q 323 55 299 42 q 348 78 348 68 l 348 364 q 292 478 323 419 q 228 594 260 538 q 165 699 195 651 q 111 780 134 748 q 97 794 105 788 q 78 806 90 801 q 48 814 66 811 q 2 818 30 818 l 0 846 q 78 856 38 852 q 145 861 117 861 q 197 834 175 861 q 250 757 221 802 q 307 663 278 713 q 363 562 335 614 q 413 461 390 510 l 592 780 q 585 808 602 798 q 518 825 569 818 l 518 854 l 781 854 l 781 825 q 711 807 735 816 q 677 780 686 798 l 454 366 l 454 78 q 459 68 454 73 q 478 55 465 62 q 512 42 491 48 q 565 29 533 35 l 565 0 l 239 0 "
    },
    "E": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 413 432 428 427 q 377 438 398 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 "
    },
    "Ã”": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 619 957 q 606 938 611 944 q 590 927 602 932 l 408 1068 l 228 927 q 219 932 223 929 q 212 938 215 935 q 206 946 209 942 q 198 957 203 951 l 374 1167 l 445 1167 l 619 957 "
    },
    "á»•": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 565 741 q 552 722 557 728 q 536 710 547 716 l 355 892 l 174 710 q 165 716 169 713 q 159 722 162 718 q 152 730 156 725 q 144 741 149 735 l 320 998 l 391 998 l 565 741 m 467 1209 q 454 1176 467 1191 q 426 1147 442 1161 q 393 1122 409 1134 q 371 1098 378 1111 q 370 1072 363 1086 q 404 1042 376 1058 q 384 1033 394 1035 q 363 1031 373 1032 q 319 1058 333 1046 q 302 1081 305 1071 q 305 1102 299 1092 q 323 1120 311 1111 q 348 1138 334 1129 q 374 1157 362 1147 q 394 1176 386 1166 q 403 1199 403 1187 q 390 1238 403 1225 q 358 1251 377 1251 q 340 1247 348 1251 q 325 1237 331 1243 q 316 1223 319 1231 q 313 1209 313 1216 q 314 1201 313 1205 q 318 1194 316 1198 q 304 1190 313 1192 q 286 1186 296 1188 q 268 1182 277 1184 q 252 1180 258 1180 l 245 1187 l 245 1196 q 257 1228 245 1212 q 289 1256 270 1243 q 333 1276 309 1268 q 380 1284 356 1284 q 444 1263 421 1284 q 467 1209 467 1243 "
    },
    "Ã": {
      "x_min": -3,
      "x_max": 396,
      "ha": 406,
      "o": "m 47 0 l 47 29 q 117 49 92 38 q 143 70 143 61 l 143 782 q 118 803 143 790 q 47 824 94 815 l 47 854 l 346 854 l 346 824 q 274 803 300 815 q 249 782 249 792 l 249 70 q 273 50 249 62 q 346 29 298 38 l 346 0 l 47 0 m 396 1044 q 390 1008 396 1025 q 373 979 384 991 q 349 959 363 966 q 318 952 335 952 q 275 968 288 952 q 263 1016 263 985 q 268 1052 263 1035 q 285 1082 274 1069 q 309 1101 296 1094 q 339 1109 323 1109 q 396 1044 396 1109 m 130 1044 q 124 1008 130 1025 q 107 979 118 991 q 83 959 97 966 q 52 952 69 952 q 9 968 22 952 q -3 1016 -3 985 q 2 1052 -3 1035 q 19 1082 8 1069 q 43 1101 30 1094 q 73 1109 57 1109 q 130 1044 130 1109 "
    },
    "Ä¡": {
      "x_min": 20,
      "x_max": 657.625,
      "ha": 664,
      "o": "m 459 406 q 447 474 459 442 q 412 531 435 507 q 355 569 389 555 q 276 583 321 583 q 240 574 260 583 q 201 548 219 565 q 171 505 183 531 q 160 446 160 479 q 170 378 160 410 q 204 322 181 346 q 260 284 226 298 q 342 271 294 271 q 382 279 361 271 q 420 305 403 287 q 448 347 437 322 q 459 406 459 372 m 344 -2 q 297 3 319 0 q 258 10 276 6 q 182 -36 210 -15 q 141 -74 155 -57 q 123 -104 127 -90 q 120 -128 120 -118 q 138 -182 120 -157 q 188 -226 157 -208 q 260 -256 219 -245 q 347 -268 301 -268 q 428 -255 391 -268 q 490 -222 464 -243 q 530 -171 516 -200 q 545 -106 545 -141 q 536 -71 545 -87 q 505 -42 527 -55 q 443 -19 482 -29 q 344 -2 405 -9 m 552 434 q 529 339 552 382 q 468 265 506 296 q 384 217 431 234 q 288 200 337 200 l 285 200 q 241 154 253 172 q 230 132 230 136 q 236 116 230 124 q 262 101 243 109 q 315 87 281 94 q 400 74 348 80 q 520 50 472 66 q 595 12 567 33 q 634 -33 623 -8 q 645 -81 645 -57 q 631 -152 645 -118 q 593 -214 617 -185 q 536 -265 569 -242 q 466 -305 504 -288 q 389 -330 429 -321 q 310 -339 349 -339 q 246 -334 279 -339 q 180 -320 212 -330 q 118 -296 147 -311 q 67 -261 89 -282 q 32 -214 45 -240 q 20 -155 20 -188 q 26 -118 20 -137 q 51 -76 32 -99 q 105 -28 71 -54 q 197 28 140 -2 q 137 63 154 44 q 121 103 121 83 q 124 118 121 109 q 138 140 127 127 q 167 170 148 153 q 216 209 186 187 q 155 236 183 218 q 108 280 128 254 q 77 337 88 305 q 67 408 67 370 q 88 502 67 457 q 146 579 110 546 q 228 631 182 612 q 322 651 274 651 q 397 639 362 651 q 460 606 432 627 q 523 615 495 610 q 573 627 551 621 q 613 639 595 633 q 644 651 630 645 l 657 630 q 642 595 649 611 q 619 562 635 579 q 570 554 594 557 q 515 552 545 552 q 542 496 532 525 q 552 434 552 467 m 399 854 q 393 818 399 835 q 376 789 387 801 q 352 769 366 776 q 321 762 338 762 q 278 778 291 762 q 266 826 266 795 q 271 862 266 845 q 288 892 277 879 q 312 911 299 904 q 342 919 326 919 q 399 854 399 919 "
    },
    "áº²": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 351 838 q 390 869 367 855 q 430 892 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 614 1139 q 569 1046 594 1084 q 516 985 544 1009 q 459 952 489 962 q 399 942 429 942 q 336 952 366 942 q 277 985 305 962 q 226 1046 250 1009 q 181 1139 201 1084 q 197 1157 189 1151 q 216 1169 205 1164 q 255 1104 233 1131 q 300 1062 276 1078 q 349 1039 324 1046 q 396 1032 374 1032 q 445 1039 419 1032 q 495 1062 471 1046 q 541 1104 519 1078 q 579 1169 563 1131 q 598 1157 590 1164 q 614 1139 606 1151 m 510 1338 q 497 1304 510 1319 q 469 1276 485 1289 q 436 1251 452 1263 q 413 1227 421 1239 q 412 1200 406 1214 q 447 1170 419 1187 q 427 1162 437 1163 q 406 1159 416 1160 q 361 1187 375 1174 q 344 1210 347 1199 q 347 1230 341 1221 q 365 1249 353 1240 q 390 1267 376 1258 q 417 1285 404 1276 q 437 1305 429 1295 q 446 1328 446 1316 q 433 1367 446 1354 q 400 1380 420 1380 q 382 1376 390 1380 q 367 1366 373 1372 q 358 1352 361 1360 q 355 1337 355 1345 q 356 1330 355 1334 q 360 1323 358 1327 q 346 1319 355 1321 q 328 1314 338 1316 q 310 1311 319 1312 q 294 1308 300 1309 l 287 1316 l 287 1325 q 299 1357 287 1341 q 331 1385 312 1372 q 375 1405 351 1397 q 422 1413 399 1413 q 487 1392 464 1413 q 510 1338 510 1372 "
    },
    "á»©": {
      "x_min": 27.265625,
      "x_max": 834,
      "ha": 834,
      "o": "m 834 707 q 823 662 834 688 q 787 608 813 637 q 718 549 762 579 q 608 491 674 519 l 608 172 q 611 103 608 130 q 623 70 614 77 q 650 68 631 65 q 704 86 670 71 l 710 55 q 662 25 685 39 q 620 1 639 11 q 586 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 466 70 q 369 4 398 18 q 316 -14 340 -9 q 270 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 533 113 515 q 100 560 109 551 q 75 572 92 569 q 27 579 57 576 l 27 606 q 72 613 51 609 q 112 623 93 617 q 152 635 132 628 q 194 651 172 642 l 213 625 l 213 226 q 219 147 213 179 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 483 q 505 532 508 513 q 492 560 502 550 q 463 574 482 570 q 415 579 445 579 l 415 606 q 506 625 462 612 q 588 651 550 638 l 608 625 l 608 538 q 659 568 638 553 q 693 599 680 584 q 713 627 707 614 q 720 652 720 641 q 710 689 720 671 q 681 722 700 707 l 799 777 q 824 744 814 762 q 834 707 834 726 m 333 710 q 312 718 321 712 q 295 729 303 723 l 450 1010 q 471 1007 458 1009 q 499 1004 484 1006 q 526 999 513 1002 q 547 994 540 996 l 562 968 l 333 710 "
    },
    "áº‘": {
      "x_min": 50.453125,
      "x_max": 549.984375,
      "ha": 602,
      "o": "m 548 168 q 548 121 548 146 q 548 74 548 97 q 548 31 548 51 q 543 0 548 12 l 67 0 l 50 30 l 419 569 l 189 569 q 167 564 178 569 q 146 549 156 560 q 127 518 136 538 q 111 467 118 498 l 85 475 l 85 641 q 117 634 105 637 q 142 631 129 631 q 170 631 155 631 l 535 631 l 549 601 l 179 61 l 460 61 q 490 87 476 61 q 520 174 504 113 l 548 168 m 529 740 q 516 721 521 727 q 500 709 512 715 l 318 891 l 138 709 q 129 715 132 712 q 122 721 125 717 q 116 729 119 724 q 108 740 112 734 l 283 997 l 355 997 l 529 740 "
    },
    "Æ‰": {
      "x_min": 17.703125,
      "x_max": 746,
      "ha": 806,
      "o": "m 746 456 q 729 305 746 372 q 683 188 712 239 q 617 101 654 137 q 538 43 579 66 q 453 10 496 20 q 371 0 410 0 l 33 0 l 33 29 q 103 49 78 38 q 129 70 129 61 l 129 417 l 32 417 l 17 433 q 24 455 21 444 q 32 478 28 467 l 129 478 l 129 805 q 78 798 102 801 q 33 791 54 795 l 27 833 q 94 850 56 842 q 175 864 132 858 q 261 871 217 871 q 343 871 304 871 q 510 842 436 871 q 637 761 585 814 q 717 631 689 708 q 746 456 746 554 m 300 816 q 235 816 266 816 l 235 478 l 407 478 l 424 462 l 407 417 l 235 417 l 235 104 q 241 80 235 89 q 277 62 250 68 q 356 57 304 57 q 450 79 402 57 q 540 148 499 102 q 607 265 581 195 q 633 431 633 335 q 611 592 633 521 q 546 713 589 663 q 442 789 504 763 q 300 816 380 816 "
    },
    "áº¤": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 839 q 390 869 367 856 q 430 893 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 608 957 q 595 938 600 944 q 579 927 590 932 l 397 1068 l 216 927 q 208 932 211 929 q 201 938 204 935 q 195 946 198 942 q 187 957 191 951 l 362 1167 l 434 1167 l 608 957 m 310 1198 q 293 1209 298 1203 q 280 1228 288 1216 l 537 1443 q 555 1433 544 1439 q 579 1419 567 1426 q 600 1405 590 1412 q 616 1393 610 1399 l 622 1363 l 310 1198 "
    },
    "á»ƒ": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 291 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 548 740 q 535 721 540 727 q 519 710 531 715 l 338 891 l 158 710 q 149 715 152 712 q 142 721 145 718 q 136 729 139 724 q 128 740 133 734 l 303 997 l 374 997 l 548 740 m 450 1209 q 437 1175 450 1190 q 409 1147 425 1160 q 376 1122 392 1134 q 354 1098 361 1110 q 353 1072 346 1085 q 387 1041 359 1058 q 367 1033 377 1034 q 346 1030 356 1032 q 302 1058 316 1045 q 285 1081 288 1070 q 288 1101 282 1092 q 306 1120 294 1111 q 331 1138 317 1129 q 357 1156 345 1147 q 377 1176 369 1166 q 386 1199 386 1187 q 373 1238 386 1225 q 341 1251 360 1251 q 322 1247 331 1251 q 308 1237 314 1243 q 299 1223 302 1231 q 296 1209 296 1216 q 297 1201 296 1205 q 301 1194 299 1198 q 287 1190 296 1192 q 269 1185 279 1187 q 251 1182 260 1183 q 235 1179 241 1180 l 228 1187 l 228 1196 q 240 1228 228 1212 q 272 1256 253 1243 q 316 1276 292 1268 q 363 1284 340 1284 q 427 1263 404 1284 q 450 1209 450 1243 "
    },
    "á¸”": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 413 432 428 427 q 377 438 398 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 564 1058 q 560 1041 563 1051 q 552 1021 556 1031 q 545 1002 548 1012 q 538 987 541 993 l 130 987 l 115 1003 q 120 1020 117 1011 q 127 1040 123 1030 q 135 1059 131 1049 q 143 1075 139 1068 l 550 1075 l 564 1058 m 451 1174 q 437 1155 441 1161 q 421 1144 432 1148 l 109 1308 l 115 1339 q 130 1351 119 1344 q 152 1365 140 1358 q 175 1379 164 1372 q 194 1389 186 1385 l 451 1174 "
    },
    "b": {
      "x_min": 6.59375,
      "x_max": 631,
      "ha": 691,
      "o": "m 631 333 q 621 253 631 294 q 591 172 611 211 q 544 97 572 133 q 481 36 516 62 q 403 -4 445 10 q 311 -20 360 -20 q 276 -14 298 -20 q 225 2 254 -8 q 162 29 196 13 q 93 65 128 44 l 93 878 q 89 926 93 910 q 77 952 86 943 q 50 963 67 960 q 6 969 33 966 l 6 996 q 90 1018 50 1007 q 168 1051 130 1030 l 175 1044 q 183 1036 179 1040 q 193 1028 187 1032 l 193 493 q 258 563 226 534 q 322 612 291 593 q 378 641 352 632 q 423 651 404 651 q 506 629 468 651 q 572 566 544 607 q 615 466 599 525 q 631 333 631 407 m 541 277 q 527 397 541 344 q 490 486 513 450 q 438 540 468 521 q 376 560 408 560 q 345 552 364 560 q 301 529 325 545 q 248 484 276 512 q 193 414 220 456 l 193 137 q 247 103 219 117 q 299 82 274 90 q 344 71 324 74 q 376 68 364 68 q 447 85 417 68 q 498 133 478 103 q 530 200 519 163 q 541 277 541 237 "
    },
    "Ìƒ": {
      "x_min": -568.90625,
      "x_max": -115.640625,
      "ha": 0,
      "o": "m -115 912 q -144 857 -127 884 q -181 808 -160 830 q -225 773 -201 787 q -275 760 -249 760 q -319 771 -298 760 q -360 798 -340 783 q -399 824 -380 812 q -438 836 -419 836 q -487 814 -465 836 q -533 754 -509 792 l -568 768 q -540 823 -556 795 q -503 872 -524 850 q -459 907 -483 893 q -410 921 -435 921 q -362 909 -384 921 q -319 883 -339 897 q -281 856 -299 868 q -247 845 -263 845 q -198 866 -222 845 q -151 927 -174 888 l -115 912 "
    },
    "ï¬‚": {
      "x_min": 30.03125,
      "x_max": 762,
      "ha": 792,
      "o": "m 463 0 l 463 29 q 511 38 491 33 q 541 49 530 44 q 558 59 553 54 q 563 70 563 65 l 563 878 q 559 925 563 909 q 544 951 555 942 q 525 960 536 957 q 510 947 518 953 q 482 925 497 935 q 458 910 468 915 q 426 935 442 924 q 395 952 410 945 q 367 962 380 959 q 345 966 354 966 q 308 954 328 966 q 272 911 289 942 q 244 826 255 879 q 233 688 233 772 l 233 630 l 404 630 l 423 610 q 408 590 417 601 q 390 569 399 579 q 373 552 381 560 q 358 542 364 544 q 314 558 343 550 q 233 566 286 566 l 233 69 q 239 61 233 65 q 263 52 246 56 q 306 42 280 48 q 372 29 332 36 l 372 0 l 40 0 l 40 29 q 109 49 85 37 q 133 69 133 61 l 133 566 l 43 566 l 30 584 l 81 630 l 133 630 l 133 651 q 143 785 133 731 q 172 878 154 840 q 214 941 190 915 q 265 988 238 966 q 304 1014 283 1003 q 346 1034 325 1026 q 386 1046 367 1042 q 418 1051 405 1051 q 460 1042 438 1051 q 500 1023 482 1034 q 524 1008 513 1016 q 557 1017 541 1012 q 639 1051 597 1029 l 663 1027 l 663 70 q 685 49 663 60 q 762 29 707 38 l 762 0 l 463 0 "
    },
    "á¸ ": {
      "x_min": 47,
      "x_max": 761,
      "ha": 794,
      "o": "m 693 805 q 691 792 697 802 q 674 770 685 782 q 651 746 664 758 q 629 726 638 734 l 607 730 q 562 765 585 751 q 513 786 538 779 q 459 797 487 794 q 397 800 430 800 q 356 792 381 800 q 303 767 331 784 q 249 722 276 750 q 199 652 221 693 q 163 557 177 611 q 150 432 150 502 q 174 267 150 337 q 239 152 199 198 q 328 83 278 106 q 429 61 378 61 q 518 70 477 61 q 592 99 559 80 l 592 329 q 585 343 592 336 q 563 357 579 350 q 521 371 548 364 q 455 384 495 378 l 455 414 l 761 414 l 761 384 q 707 360 723 375 q 692 329 692 344 l 692 104 q 603 38 642 63 q 532 1 565 14 q 470 -15 499 -11 q 411 -20 441 -20 q 279 5 344 -20 q 162 82 213 30 q 78 212 110 134 q 47 394 47 289 q 81 596 47 507 q 177 747 116 685 q 318 842 237 809 q 490 875 399 875 q 538 870 512 875 q 592 856 564 865 q 645 834 619 847 q 693 805 671 822 m 657 1058 q 652 1041 656 1051 q 645 1021 649 1031 q 637 1002 641 1012 q 631 987 634 993 l 222 987 l 207 1003 q 212 1020 209 1011 q 219 1040 215 1030 q 227 1059 223 1049 q 235 1075 231 1068 l 642 1075 l 657 1058 "
    },
    "È­": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 581 913 q 553 858 569 885 q 516 809 536 830 q 472 773 496 787 q 422 760 448 760 q 378 771 399 760 q 337 798 357 783 q 297 825 317 813 q 258 837 278 837 q 210 814 232 837 q 163 754 188 792 l 128 768 q 157 823 140 796 q 193 872 173 851 q 238 908 214 894 q 287 922 261 922 q 335 910 312 922 q 377 883 357 898 q 415 856 397 868 q 449 845 433 845 q 499 866 474 845 q 545 928 523 888 l 581 913 m 586 1112 q 581 1095 585 1105 q 574 1075 578 1085 q 566 1056 570 1066 q 560 1041 563 1047 l 152 1041 l 137 1057 q 142 1074 138 1065 q 149 1094 145 1084 q 157 1113 153 1103 q 164 1129 161 1122 l 571 1129 l 586 1112 "
    },
    "ÅŠ": {
      "x_min": 33,
      "x_max": 748,
      "ha": 821,
      "o": "m 748 187 q 733 58 748 117 q 695 -47 719 0 q 640 -132 671 -95 q 575 -195 609 -169 q 534 -221 558 -209 q 485 -241 511 -232 q 434 -253 460 -249 q 384 -258 408 -258 q 326 -250 355 -258 q 273 -231 297 -243 q 234 -208 249 -220 q 220 -184 220 -195 q 230 -167 220 -178 q 255 -145 241 -156 q 285 -123 270 -133 q 310 -109 300 -113 q 349 -140 328 -127 q 390 -162 370 -153 q 431 -174 411 -170 q 468 -179 451 -179 q 531 -158 499 -179 q 586 -97 562 -138 q 626 5 611 -56 q 642 151 642 67 l 642 591 q 634 683 642 647 q 611 740 626 719 q 575 771 596 762 q 530 780 554 780 q 473 768 503 780 q 407 726 442 756 q 329 642 371 695 q 236 507 287 590 l 236 70 q 260 50 236 62 q 333 29 284 38 l 333 0 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 698 q 128 741 130 725 q 117 768 126 758 q 89 783 108 777 q 36 791 70 788 l 36 820 q 129 842 82 829 q 213 875 177 856 l 236 851 l 236 629 q 333 748 287 701 q 421 823 379 795 q 500 863 462 851 q 574 875 538 875 q 633 866 603 875 q 689 833 664 857 q 731 769 715 810 q 748 665 748 729 l 748 187 "
    },
    "Ç“": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 847 855 l 847 825 m 485 944 l 414 944 l 239 1152 q 247 1163 243 1158 q 253 1171 250 1168 q 260 1177 256 1175 q 269 1184 263 1180 l 451 1040 l 631 1184 q 647 1171 642 1177 q 660 1152 652 1165 l 485 944 "
    },
    "Å¨": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 847 855 l 847 825 m 676 1103 q 648 1048 664 1075 q 611 999 631 1020 q 566 963 590 977 q 516 950 542 950 q 472 961 493 950 q 431 988 451 973 q 392 1015 411 1003 q 353 1027 372 1027 q 304 1004 326 1027 q 258 944 283 982 l 223 958 q 251 1013 235 986 q 288 1062 267 1041 q 332 1098 309 1084 q 382 1112 356 1112 q 429 1100 407 1112 q 472 1073 452 1088 q 510 1046 492 1058 q 544 1035 528 1035 q 593 1056 569 1035 q 640 1118 618 1078 l 676 1103 "
    },
    "L": {
      "x_min": 33,
      "x_max": 627.09375,
      "ha": 648,
      "o": "m 627 165 q 613 63 621 106 q 599 0 605 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 111 q 241 89 236 98 q 261 74 247 80 q 299 64 275 67 q 359 61 322 61 l 454 61 q 505 64 484 61 q 542 79 526 67 q 571 114 558 91 q 598 177 584 137 l 627 165 "
    },
    "áºŠ": {
      "x_min": 21,
      "x_max": 817,
      "ha": 838,
      "o": "m 517 0 l 517 28 q 564 36 546 31 q 590 46 582 40 q 597 62 598 53 q 587 85 596 72 l 402 362 l 234 85 q 237 43 215 56 q 320 31 258 31 l 320 0 l 21 0 l 21 30 q 94 42 63 30 q 141 84 124 55 l 353 436 l 130 768 q 111 792 120 783 q 90 808 102 802 q 63 818 78 814 q 27 825 48 821 l 27 854 l 327 854 l 327 825 q 253 808 270 819 q 256 768 236 797 l 420 522 l 570 768 q 578 794 579 784 q 565 810 576 804 q 533 820 554 817 q 484 824 513 824 l 484 855 l 785 855 l 785 825 q 742 818 761 822 q 708 808 723 815 q 682 792 693 802 q 663 768 671 782 l 470 448 l 712 85 q 732 61 722 71 q 755 45 742 52 q 782 35 767 39 q 817 28 798 31 l 817 0 l 517 0 m 486 1044 q 480 1008 486 1025 q 463 979 474 991 q 439 959 453 966 q 408 952 425 952 q 365 968 378 952 q 353 1016 353 985 q 358 1052 353 1035 q 375 1082 364 1069 q 399 1101 386 1094 q 429 1109 413 1109 q 486 1044 486 1109 "
    },
    "â±¢": {
      "x_min": -40.109375,
      "x_max": 627.09375,
      "ha": 648,
      "o": "m 627 165 q 613 63 621 106 q 599 0 605 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 400 q 109 409 119 406 q 90 413 99 413 q 41 390 63 413 q -4 330 19 368 l -40 344 q -11 399 -28 372 q 25 448 4 427 q 69 484 45 470 q 119 498 93 498 q 124 497 121 498 q 130 497 126 497 l 130 783 q 105 803 130 791 q 33 825 81 816 l 33 854 l 332 854 l 332 825 q 261 804 286 816 q 236 783 236 792 l 236 440 q 259 426 247 431 q 281 421 270 421 q 330 442 306 421 q 377 504 354 464 l 413 488 q 384 433 401 461 q 347 384 367 406 q 303 349 327 363 q 253 336 279 336 q 236 338 244 336 l 236 111 q 241 89 236 98 q 261 74 247 80 q 299 64 275 67 q 359 61 322 61 l 454 61 q 505 64 484 61 q 542 79 526 67 q 571 114 558 91 q 598 177 584 137 l 627 165 "
    },
    "á¹¯": {
      "x_min": 6.65625,
      "x_max": 456.25,
      "ha": 469,
      "o": "m 449 79 q 385 36 417 55 q 324 5 353 18 q 270 -13 295 -6 q 228 -19 246 -19 q 184 -10 205 -19 q 147 17 163 -2 q 122 70 131 37 q 113 150 113 102 l 113 567 l 27 567 l 13 585 l 65 631 l 113 631 l 113 797 l 192 868 l 213 851 l 213 631 l 430 631 l 449 611 q 434 591 443 602 q 416 570 425 580 q 398 553 407 561 q 382 543 389 545 q 333 559 366 551 q 247 567 301 567 l 213 567 l 213 208 q 216 140 213 167 q 228 97 220 113 q 251 74 237 81 q 284 67 264 67 q 343 77 308 67 q 430 114 378 86 l 449 79 m 456 -155 q 451 -172 454 -162 q 444 -192 448 -182 q 436 -211 440 -201 q 430 -227 432 -220 l 21 -227 l 6 -210 q 11 -193 7 -202 q 18 -173 14 -183 q 26 -154 22 -164 q 33 -139 30 -145 l 441 -139 l 456 -155 "
    },
    "Ä¬": {
      "x_min": -19.453125,
      "x_max": 412.46875,
      "ha": 406,
      "o": "m 47 0 l 47 29 q 118 49 92 38 q 144 70 144 61 l 144 782 q 119 803 144 790 q 47 824 95 815 l 47 854 l 346 854 l 346 824 q 275 803 300 815 q 250 782 250 792 l 250 70 q 274 50 250 62 q 346 29 298 38 l 346 0 l 47 0 m 412 1139 q 367 1046 392 1084 q 315 985 342 1009 q 257 952 287 962 q 197 942 227 942 q 134 952 165 942 q 76 985 104 962 q 24 1046 48 1009 q -19 1139 0 1084 q -3 1157 -11 1151 q 15 1169 4 1164 q 53 1104 32 1131 q 99 1062 75 1078 q 148 1039 123 1046 q 195 1032 173 1032 q 244 1039 218 1032 q 293 1062 269 1046 q 339 1104 318 1078 q 377 1169 361 1131 q 396 1157 388 1164 q 412 1139 404 1151 "
    },
    "Ã€": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 839 q 390 869 367 855 q 430 893 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 516 957 q 501 938 506 944 q 486 927 496 931 l 173 1091 l 179 1122 q 194 1134 183 1127 q 216 1148 204 1141 q 239 1162 228 1155 q 258 1172 250 1168 l 516 957 "
    },
    "ÌŠ": {
      "x_min": -466,
      "x_max": -219,
      "ha": 0,
      "o": "m -279 843 q -282 874 -279 859 q -294 900 -286 889 q -312 918 -301 911 q -335 925 -322 925 q -362 919 -349 925 q -384 903 -374 913 q -400 877 -394 892 q -406 844 -406 862 q -402 814 -406 828 q -390 788 -398 799 q -372 770 -383 777 q -349 764 -362 764 q -323 769 -335 764 q -300 784 -310 774 q -284 808 -290 794 q -279 843 -279 823 m -219 870 q -231 807 -219 836 q -265 757 -244 778 q -312 723 -286 735 q -363 712 -337 712 q -404 720 -385 712 q -437 742 -423 728 q -458 775 -450 756 q -466 816 -466 794 q -453 880 -466 851 q -419 931 -440 910 q -373 965 -398 953 q -321 977 -347 977 q -281 968 -300 977 q -248 945 -262 960 q -226 911 -234 930 q -219 870 -219 892 "
    },
    "â€‘": {
      "x_min": 40.546875,
      "x_max": 418.03125,
      "ha": 459,
      "o": "m 418 370 q 408 338 414 355 q 397 309 402 320 l 57 309 l 40 325 q 49 356 43 340 q 61 387 55 373 l 401 387 l 418 370 "
    },
    "Â½": {
      "x_min": 52.703125,
      "x_max": 741,
      "ha": 797,
      "o": "m 64 432 l 64 455 q 111 461 93 457 q 140 468 130 465 q 155 476 151 472 q 159 484 159 480 l 159 732 q 158 752 159 745 q 152 763 157 759 q 144 766 150 765 q 128 768 139 768 q 101 765 118 768 q 60 759 85 763 l 52 781 q 90 792 67 785 q 135 806 112 799 q 179 822 158 814 q 211 837 199 831 l 225 825 l 225 484 q 228 477 225 480 q 240 469 231 473 q 267 461 250 465 q 313 455 285 457 l 313 432 l 64 432 m 209 2 q 179 -10 193 -5 q 147 -20 165 -15 l 131 0 l 637 815 q 666 827 651 820 q 697 837 682 833 l 713 818 l 209 2 m 741 0 l 469 0 l 459 23 q 566 135 525 91 q 631 210 607 180 q 662 261 654 241 q 671 300 671 282 q 655 348 671 330 q 601 366 639 366 q 575 360 587 366 q 555 346 563 355 q 542 326 547 337 q 538 303 538 315 q 510 293 524 297 q 478 289 496 289 l 469 297 q 482 333 469 314 q 517 367 495 351 q 568 392 539 382 q 627 402 596 402 q 672 396 651 402 q 707 379 692 391 q 730 350 722 368 q 739 309 739 333 q 729 265 739 288 q 695 214 719 243 q 633 144 671 184 q 537 45 594 103 l 682 45 q 702 53 695 45 q 714 70 710 61 q 720 97 719 82 l 741 93 l 741 0 "
    },
    "á¸Ÿ": {
      "x_min": 30.03125,
      "x_max": 543,
      "ha": 423,
      "o": "m 543 986 q 533 969 543 980 q 510 947 523 958 q 483 925 497 935 q 458 910 468 915 q 426 935 442 925 q 395 953 410 946 q 367 963 380 959 q 345 966 354 966 q 309 954 328 966 q 272 911 289 942 q 244 826 255 880 q 233 688 233 772 l 233 630 l 404 630 l 423 610 q 408 590 417 601 q 390 569 399 579 q 373 552 381 560 q 358 542 364 544 q 314 558 343 550 q 233 566 286 566 l 233 68 q 239 60 233 64 q 263 52 246 56 q 306 42 280 47 q 372 28 332 36 l 372 0 l 40 0 l 40 28 q 109 48 85 37 q 133 68 133 60 l 133 566 l 43 566 l 30 584 l 81 630 l 133 630 l 133 651 q 143 786 133 731 q 172 878 154 840 q 214 941 190 916 q 265 988 238 967 q 304 1015 283 1003 q 346 1034 325 1026 q 386 1046 367 1042 q 418 1051 405 1051 q 460 1042 438 1051 q 500 1024 482 1034 q 531 1002 519 1013 q 543 986 543 992 m 346 1214 q 340 1178 346 1195 q 323 1149 334 1161 q 299 1129 313 1136 q 268 1122 285 1122 q 225 1138 238 1122 q 213 1186 213 1155 q 218 1222 213 1205 q 235 1252 224 1239 q 259 1271 246 1264 q 289 1279 273 1279 q 346 1214 346 1279 "
    },
    "'": {
      "x_min": 106.03125,
      "x_max": 219,
      "ha": 340,
      "o": "m 219 565 q 203 559 212 562 q 183 555 193 557 q 161 552 172 553 q 141 551 150 551 l 106 966 q 127 976 113 970 q 156 987 141 982 q 184 997 171 993 q 205 1003 197 1001 l 219 989 l 219 565 "
    },
    "Ä³": {
      "x_min": 47,
      "x_max": 625,
      "ha": 721,
      "o": "m 47 0 l 47 29 q 115 49 92 38 q 139 70 139 61 l 139 454 q 139 510 139 488 q 130 543 139 531 q 101 560 121 555 q 47 569 82 566 l 47 596 q 90 606 67 600 q 136 619 113 612 q 179 634 158 626 q 217 651 199 642 l 240 651 l 240 70 q 262 50 240 62 q 333 29 284 38 l 333 0 l 47 0 m 260 854 q 254 818 260 835 q 237 789 248 801 q 213 769 227 776 q 182 762 199 762 q 139 778 152 762 q 127 826 127 795 q 132 862 127 845 q 149 892 138 879 q 173 911 160 904 q 203 919 187 919 q 260 854 260 919 m 605 67 q 595 -65 605 -10 q 568 -158 585 -119 q 528 -223 551 -196 q 478 -271 505 -249 q 440 -297 460 -285 q 401 -319 420 -310 q 363 -333 381 -328 q 333 -339 345 -339 q 293 -333 312 -339 q 257 -321 273 -328 q 232 -305 242 -314 q 223 -292 223 -297 q 232 -275 223 -286 q 255 -253 242 -265 q 283 -232 269 -242 q 307 -217 297 -222 q 351 -242 329 -236 q 398 -248 373 -248 q 435 -235 417 -248 q 469 -191 454 -222 q 493 -106 484 -160 q 503 28 503 -53 l 503 454 q 503 509 503 488 q 494 542 503 530 q 465 560 485 554 q 411 569 446 565 l 411 596 q 461 607 439 601 q 502 620 483 613 q 540 634 522 626 q 580 651 559 642 l 605 651 l 605 67 m 625 854 q 619 818 625 835 q 602 789 613 801 q 578 769 592 776 q 548 762 564 762 q 505 778 518 762 q 492 826 492 795 q 497 862 492 845 q 514 892 503 879 q 538 911 524 904 q 568 919 552 919 q 625 854 625 919 "
    },
    "á¸¶": {
      "x_min": 33,
      "x_max": 627.625,
      "ha": 648,
      "o": "m 627 165 q 614 63 622 106 q 600 0 605 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 287 815 q 236 783 236 792 l 236 111 q 241 89 236 98 q 261 74 247 80 q 299 64 275 67 q 359 61 323 61 l 454 61 q 506 64 485 61 q 543 79 527 67 q 572 114 559 91 q 599 177 585 137 l 627 165 m 397 -189 q 391 -225 397 -208 q 374 -254 385 -242 q 350 -274 364 -267 q 319 -282 336 -282 q 276 -265 289 -282 q 264 -217 264 -248 q 269 -181 264 -198 q 286 -151 275 -164 q 310 -132 297 -139 q 340 -125 324 -125 q 397 -189 397 -125 "
    },
    "Ë›": {
      "x_min": 66,
      "x_max": 318.15625,
      "ha": 352,
      "o": "m 318 -202 q 285 -238 304 -221 q 246 -270 266 -256 q 206 -292 226 -284 q 169 -301 186 -301 q 131 -296 150 -301 q 98 -282 113 -292 q 74 -252 83 -271 q 66 -202 66 -233 q 118 -83 66 -141 q 270 29 170 -25 l 300 16 q 223 -42 253 -13 q 178 -95 194 -70 q 157 -141 162 -120 q 152 -179 152 -162 q 167 -216 152 -204 q 209 -228 183 -228 q 250 -215 225 -228 q 302 -177 274 -203 l 318 -202 "
    },
    "Éµ": {
      "x_min": 53,
      "x_max": 631,
      "ha": 684,
      "o": "m 159 282 q 185 193 165 234 q 232 120 205 151 q 290 72 259 89 q 349 54 321 54 q 426 70 393 54 q 481 116 459 86 q 515 187 503 145 q 527 282 527 229 l 159 282 m 527 343 q 502 433 521 391 q 457 507 483 475 q 399 558 431 539 q 335 577 367 577 q 254 559 287 577 q 199 510 220 541 q 167 435 177 478 q 157 343 157 393 l 527 343 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 "
    },
    "É›": {
      "x_min": 53,
      "x_max": 550.8125,
      "ha": 581,
      "o": "m 550 129 q 471 47 507 77 q 402 2 434 17 q 340 -16 370 -12 q 282 -20 311 -20 q 184 -7 227 -20 q 112 28 141 5 q 67 82 82 51 q 53 150 53 113 q 63 205 53 179 q 92 252 74 231 q 133 290 110 274 q 179 316 156 306 q 102 369 129 336 q 75 458 75 402 q 82 500 75 479 q 104 541 90 521 q 137 577 118 560 q 177 607 156 594 q 213 625 193 617 q 255 639 233 634 q 300 647 277 644 q 345 650 323 650 q 393 647 368 650 q 440 636 417 643 q 481 620 462 630 q 514 598 500 610 q 512 574 515 589 q 502 543 508 560 q 490 511 496 526 q 477 486 483 496 l 453 493 q 434 523 447 507 q 402 552 422 539 q 355 575 382 566 q 292 584 327 584 q 240 575 263 584 q 201 551 217 566 q 176 518 185 537 q 168 478 168 499 q 174 437 168 457 q 200 398 180 416 q 256 368 219 381 q 353 356 292 356 l 360 316 q 271 302 308 316 q 209 269 233 288 q 173 226 185 249 q 162 179 162 203 q 172 137 162 158 q 203 101 183 117 q 251 74 223 84 q 316 65 280 65 q 355 67 335 65 q 400 78 375 69 q 455 105 424 87 q 525 157 485 124 l 550 129 "
    },
    "Ç”": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 466 70 q 369 4 398 18 q 316 -14 340 -9 q 270 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 533 113 515 q 100 560 109 551 q 75 572 92 569 q 27 579 57 576 l 27 606 q 72 613 51 609 q 112 623 93 617 q 152 635 132 628 q 194 651 172 642 l 213 625 l 213 226 q 219 147 213 179 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 483 q 505 532 508 513 q 492 560 502 550 q 463 574 482 570 q 414 579 445 579 l 414 606 q 506 625 462 612 q 588 651 550 638 l 608 625 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 397 727 l 326 727 l 151 969 q 159 980 156 975 q 166 988 163 985 q 172 994 169 992 q 181 1000 176 997 l 363 830 l 543 1000 q 559 988 554 994 q 572 969 564 982 l 397 727 "
    },
    "á¹": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 582 913 q 553 858 570 885 q 516 809 536 830 q 472 773 496 787 q 422 760 448 760 q 378 771 399 760 q 337 798 357 783 q 297 825 317 813 q 259 837 278 837 q 210 814 232 837 q 164 754 188 792 l 128 768 q 157 823 140 796 q 193 872 173 851 q 238 908 214 894 q 287 922 261 922 q 335 910 312 922 q 378 883 358 898 q 415 856 398 868 q 449 845 433 845 q 499 866 474 845 q 546 928 523 888 l 582 913 m 555 1099 q 549 1063 555 1080 q 532 1034 543 1046 q 508 1014 522 1021 q 477 1007 494 1007 q 434 1023 447 1007 q 422 1071 422 1040 q 427 1107 422 1090 q 444 1137 433 1124 q 468 1156 455 1149 q 498 1164 482 1164 q 555 1099 555 1164 m 289 1099 q 283 1063 289 1080 q 266 1034 277 1046 q 242 1014 256 1021 q 211 1007 228 1007 q 168 1023 181 1007 q 156 1071 156 1040 q 161 1107 156 1090 q 178 1137 167 1124 q 202 1156 189 1149 q 232 1164 216 1164 q 289 1099 289 1164 "
    },
    "Ä†": {
      "x_min": 47,
      "x_max": 677.046875,
      "ha": 730,
      "o": "m 677 143 q 595 65 634 97 q 520 15 556 34 q 449 -11 484 -3 q 379 -20 414 -20 q 258 8 318 -20 q 152 90 199 36 q 75 221 104 144 q 47 397 47 299 q 79 594 47 506 q 166 744 111 682 q 294 841 221 807 q 449 875 368 875 q 575 855 521 875 q 664 807 629 835 q 662 793 668 803 q 646 770 656 783 q 624 744 635 757 q 604 723 612 731 l 581 727 q 500 780 547 759 q 393 800 454 800 q 344 791 371 800 q 290 765 317 783 q 237 717 263 746 q 192 645 212 687 q 161 548 173 603 q 150 422 150 493 q 175 264 150 332 q 241 151 201 196 q 331 83 281 106 q 427 61 380 61 q 521 86 463 61 q 652 173 579 111 q 659 167 655 172 q 666 158 663 163 q 672 149 669 153 q 677 143 675 145 m 320 927 q 304 938 309 931 q 290 957 299 944 l 548 1172 q 566 1162 554 1168 q 589 1148 577 1155 q 611 1134 601 1141 q 626 1122 621 1127 l 632 1092 l 320 927 "
    },
    "áº“": {
      "x_min": 50.453125,
      "x_max": 549.984375,
      "ha": 602,
      "o": "m 548 168 q 548 122 548 146 q 548 74 548 97 q 548 31 548 51 q 543 0 548 12 l 67 0 l 50 30 l 419 570 l 189 570 q 167 565 178 570 q 146 550 156 561 q 127 519 136 539 q 111 468 118 499 l 85 476 l 85 642 q 117 635 105 638 q 142 632 129 632 q 170 632 155 632 l 535 632 l 549 602 l 179 61 l 460 61 q 490 87 476 61 q 520 175 504 113 l 548 168 m 367 -189 q 361 -225 367 -208 q 344 -254 355 -242 q 320 -274 334 -267 q 289 -282 306 -282 q 246 -265 259 -282 q 234 -217 234 -248 q 239 -181 234 -198 q 256 -151 245 -164 q 280 -132 267 -139 q 310 -125 294 -125 q 367 -189 367 -125 "
    },
    "Â£": {
      "x_min": 42.59375,
      "x_max": 597.25,
      "ha": 639,
      "o": "m 409 363 l 249 363 q 243 274 249 316 q 228 193 238 231 q 203 123 218 155 q 171 67 189 92 q 225 69 201 69 q 267 69 248 69 q 303 69 286 69 q 337 69 320 69 q 374 69 355 69 q 418 69 393 69 q 470 72 449 69 q 509 87 492 75 q 540 121 526 98 q 568 183 554 143 l 597 171 q 590 115 594 142 q 583 66 587 88 q 576 26 579 43 q 570 0 572 9 q 499 -19 540 -19 q 413 -15 459 -19 q 318 -2 367 -11 q 222 7 269 7 q 132 3 175 7 q 57 -20 90 0 l 42 12 q 78 44 63 29 q 104 76 93 59 q 122 116 115 93 q 132 172 129 139 q 136 252 136 205 q 136 363 136 298 l 57 363 l 42 380 q 49 402 45 390 q 57 425 53 414 l 127 425 q 141 597 127 521 q 190 726 156 673 q 278 807 224 779 q 408 836 333 836 q 442 836 425 836 q 476 830 458 836 q 518 816 495 825 q 572 789 540 806 q 569 747 572 772 q 563 695 566 721 q 555 646 559 669 q 546 610 550 622 l 515 610 q 465 732 506 694 q 358 770 425 770 q 323 766 340 770 q 289 750 305 762 q 261 716 274 738 q 242 655 249 693 q 235 560 235 616 q 242 425 235 503 l 409 425 l 425 409 l 409 363 "
    },
    "áº¹": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 527 237 552 q 177 468 192 503 q 156 393 162 434 l 440 393 q 461 398 456 393 q 466 417 466 403 q 461 463 466 435 q 441 517 456 490 q 399 561 426 543 q 329 580 372 580 m 575 377 q 543 352 563 363 q 501 332 523 340 l 153 332 q 166 231 153 279 q 205 147 180 183 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 128 q 561 146 555 155 q 571 132 567 137 q 492 52 526 82 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 233 617 214 607 q 271 634 251 627 q 310 646 291 642 q 348 651 330 651 q 425 638 391 651 q 484 605 458 626 q 526 557 509 585 q 554 499 543 530 q 570 437 565 469 q 575 377 575 405 m 399 -189 q 393 -225 399 -208 q 376 -254 387 -242 q 352 -274 366 -267 q 321 -282 338 -282 q 278 -265 291 -282 q 266 -217 266 -248 q 271 -181 266 -198 q 288 -151 277 -164 q 312 -132 299 -139 q 342 -125 326 -125 q 399 -189 399 -125 "
    },
    "Å¯": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 433 44 467 70 q 369 4 398 18 q 316 -14 341 -9 q 271 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 481 q 111 532 113 514 q 100 559 109 550 q 75 571 92 568 q 27 578 57 575 l 27 605 q 72 612 51 608 q 112 622 93 616 q 152 634 132 627 q 194 650 172 641 l 213 624 l 213 226 q 219 147 213 178 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 482 q 505 531 508 512 q 492 559 502 549 q 464 573 482 569 q 414 578 446 578 l 414 605 q 507 624 463 612 q 588 650 550 637 l 608 624 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 426 843 q 422 874 426 859 q 410 900 418 889 q 392 918 403 911 q 369 925 382 925 q 342 919 355 925 q 320 903 330 913 q 304 877 310 892 q 299 844 299 862 q 302 814 299 828 q 314 788 306 799 q 332 770 321 777 q 355 764 342 764 q 381 769 369 764 q 404 784 394 774 q 420 808 414 794 q 426 843 426 823 m 486 870 q 473 807 486 836 q 439 757 460 778 q 392 723 418 735 q 341 712 367 712 q 300 720 319 712 q 267 742 281 728 q 246 775 254 756 q 239 816 239 794 q 251 880 239 851 q 285 931 264 910 q 331 965 306 953 q 383 977 357 977 q 423 968 404 977 q 456 945 442 960 q 478 911 470 930 q 486 870 486 892 "
    },
    "ÅŒ": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 640 1058 q 636 1041 639 1051 q 628 1021 632 1031 q 621 1002 624 1012 q 614 987 617 993 l 205 987 l 191 1003 q 195 1020 192 1011 q 203 1040 199 1030 q 211 1059 207 1049 q 218 1075 215 1068 l 626 1075 l 640 1058 "
    },
    "á¹º": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 615 816 l 568 855 l 847 855 l 847 825 m 681 1058 q 676 1041 680 1051 q 669 1021 673 1031 q 661 1002 665 1012 q 655 987 657 993 l 246 987 l 232 1003 q 236 1020 233 1011 q 243 1040 239 1030 q 251 1059 247 1049 q 258 1075 255 1068 l 666 1075 l 681 1058 m 649 1262 q 643 1226 649 1243 q 626 1197 637 1209 q 602 1177 616 1184 q 571 1170 588 1170 q 528 1186 541 1170 q 516 1234 516 1203 q 521 1270 516 1253 q 538 1300 527 1287 q 562 1319 549 1312 q 592 1327 576 1327 q 649 1262 649 1327 m 383 1262 q 377 1226 383 1243 q 360 1197 371 1209 q 336 1177 350 1184 q 305 1170 322 1170 q 262 1186 275 1170 q 250 1234 250 1203 q 255 1270 250 1253 q 272 1300 261 1287 q 296 1319 283 1312 q 326 1327 310 1327 q 383 1262 383 1327 "
    },
    "Ç´": {
      "x_min": 47,
      "x_max": 761,
      "ha": 794,
      "o": "m 693 805 q 691 792 697 802 q 674 770 685 782 q 651 746 664 758 q 629 726 638 734 l 607 730 q 562 765 585 751 q 513 786 538 779 q 459 797 487 794 q 397 800 430 800 q 356 792 381 800 q 303 767 331 784 q 249 722 276 750 q 199 652 221 693 q 163 557 177 611 q 150 432 150 502 q 174 267 150 337 q 239 152 199 198 q 328 83 278 106 q 429 61 378 61 q 518 70 477 61 q 592 99 559 80 l 592 329 q 585 343 592 336 q 563 357 579 350 q 521 371 548 364 q 455 384 495 378 l 455 414 l 761 414 l 761 384 q 707 360 723 375 q 692 329 692 344 l 692 104 q 603 38 642 63 q 532 1 565 14 q 470 -15 499 -11 q 411 -20 441 -20 q 279 5 344 -20 q 162 82 213 30 q 78 212 110 134 q 47 394 47 289 q 81 596 47 507 q 177 747 116 685 q 318 842 237 809 q 490 875 399 875 q 538 870 512 875 q 592 856 564 865 q 645 834 619 847 q 693 805 671 822 m 338 927 q 321 938 326 931 q 308 957 316 944 l 565 1172 q 583 1162 572 1168 q 606 1148 594 1155 q 628 1134 618 1141 q 644 1122 638 1127 l 650 1092 l 338 927 "
    },
    "Äž": {
      "x_min": 47,
      "x_max": 761,
      "ha": 794,
      "o": "m 693 805 q 691 792 697 802 q 674 770 685 782 q 651 746 664 758 q 629 726 638 734 l 607 730 q 562 765 585 751 q 513 786 538 779 q 459 797 487 794 q 397 800 430 800 q 356 792 381 800 q 303 767 331 784 q 249 722 276 750 q 199 652 221 693 q 163 557 177 611 q 150 432 150 502 q 174 267 150 337 q 239 152 199 198 q 328 83 278 106 q 429 61 378 61 q 518 70 477 61 q 592 99 559 80 l 592 329 q 585 343 592 336 q 563 357 579 350 q 521 371 548 364 q 455 384 495 378 l 455 414 l 761 414 l 761 384 q 707 360 723 375 q 692 329 692 344 l 692 104 q 603 38 642 63 q 532 1 565 14 q 470 -15 499 -11 q 411 -20 441 -20 q 279 5 344 -20 q 162 82 213 30 q 78 212 110 134 q 47 394 47 289 q 81 596 47 507 q 177 747 116 685 q 318 842 237 809 q 490 875 399 875 q 538 870 512 875 q 592 856 564 865 q 645 834 619 847 q 693 805 671 822 m 642 1140 q 597 1047 622 1085 q 544 986 572 1010 q 487 953 517 963 q 427 943 457 943 q 364 953 394 943 q 305 986 333 963 q 254 1047 278 1010 q 209 1140 229 1085 q 225 1158 217 1152 q 245 1170 233 1165 q 283 1105 261 1132 q 328 1063 304 1079 q 377 1040 352 1047 q 424 1033 402 1033 q 473 1040 447 1033 q 523 1063 498 1047 q 569 1105 547 1079 q 606 1170 590 1132 q 626 1158 618 1165 q 642 1140 634 1152 "
    },
    "v": {
      "x_min": 13,
      "x_max": 657,
      "ha": 671,
      "o": "m 657 601 q 624 593 637 597 q 603 583 611 588 q 591 572 595 578 q 583 555 587 565 l 395 40 q 378 14 389 25 q 354 -2 367 4 q 327 -13 340 -9 q 306 -20 314 -17 l 86 555 q 63 584 80 574 q 13 601 46 594 l 13 631 l 270 631 l 270 601 q 227 595 244 598 q 202 586 211 591 q 192 574 193 581 q 194 555 190 566 l 357 121 l 511 555 q 513 573 514 565 q 504 585 512 580 q 481 594 496 590 q 442 601 466 597 l 442 631 l 657 631 l 657 601 "
    },
    "Ã»": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 466 70 q 369 4 398 18 q 316 -14 340 -9 q 270 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 533 113 515 q 100 560 109 551 q 75 572 92 569 q 27 579 57 576 l 27 606 q 72 613 51 609 q 112 623 93 617 q 152 635 132 628 q 194 651 172 642 l 213 625 l 213 226 q 219 147 213 179 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 483 q 505 532 508 513 q 492 560 502 550 q 463 574 482 570 q 414 579 445 579 l 414 606 q 506 625 462 612 q 588 651 550 638 l 608 625 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 572 741 q 559 722 564 728 q 543 710 554 715 l 361 892 l 181 710 q 172 715 176 713 q 166 722 169 718 q 159 730 163 725 q 151 741 156 734 l 326 998 l 397 998 l 572 741 "
    },
    "áº": {
      "x_min": 39.578125,
      "x_max": 655,
      "ha": 712,
      "o": "m 655 198 q 655 150 655 177 q 655 97 655 124 q 655 45 655 70 q 655 0 655 19 l 58 0 l 39 30 l 512 787 l 207 787 q 183 779 196 787 q 159 755 171 772 q 136 714 147 739 q 118 653 125 688 l 80 661 l 100 865 q 132 858 118 861 q 162 855 147 855 q 191 855 176 855 l 635 855 l 652 825 l 183 68 l 531 68 q 556 74 545 68 q 578 96 567 80 q 599 139 588 112 q 622 208 609 166 l 655 198 m 579 957 q 566 938 571 944 q 550 926 561 932 l 368 1067 l 188 926 q 179 932 183 929 q 172 938 175 935 q 166 946 169 941 q 158 957 163 951 l 333 1166 l 405 1166 l 579 957 "
    },
    "Å¹": {
      "x_min": 39.578125,
      "x_max": 655,
      "ha": 712,
      "o": "m 655 198 q 655 150 655 177 q 655 97 655 124 q 655 45 655 70 q 655 0 655 19 l 58 0 l 39 30 l 512 787 l 207 787 q 183 779 196 787 q 159 755 171 772 q 136 714 147 739 q 118 653 125 688 l 80 661 l 100 865 q 132 858 118 861 q 162 855 147 855 q 191 855 176 855 l 635 855 l 652 825 l 183 68 l 531 68 q 556 74 545 68 q 578 96 567 80 q 599 139 588 112 q 622 208 609 166 l 655 198 m 281 926 q 265 938 270 931 q 251 957 260 944 l 508 1172 q 526 1161 515 1168 q 549 1148 538 1155 q 571 1134 561 1141 q 587 1122 581 1127 l 593 1091 l 281 926 "
    },
    "ï‡©": {
      "x_min": 66,
      "x_max": 224,
      "ha": 284,
      "o": "m 224 493 q 217 449 224 469 q 197 414 210 429 q 168 391 185 399 q 132 383 151 383 q 81 402 96 383 q 66 458 66 422 q 73 501 66 481 q 93 537 80 521 q 122 561 105 552 q 158 570 139 570 q 207 550 191 570 q 224 493 224 530 m 224 164 q 217 120 224 140 q 197 85 210 100 q 168 62 185 70 q 132 54 151 54 q 81 73 96 54 q 66 129 66 93 q 73 172 66 152 q 93 208 80 192 q 122 232 105 223 q 158 241 139 241 q 207 221 191 241 q 224 164 224 201 "
    },
    "á¹€": {
      "x_min": 40,
      "x_max": 1043,
      "ha": 1097,
      "o": "m 1030 825 q 988 818 1010 825 q 944 800 966 812 l 944 70 q 969 50 944 62 q 1043 29 995 38 l 1043 0 l 741 0 l 741 29 q 813 49 786 38 q 840 70 840 60 l 840 705 l 546 0 l 514 0 l 216 700 l 216 70 q 238 50 216 62 q 309 29 261 38 l 309 0 l 40 0 l 40 29 q 112 49 85 38 q 140 70 140 60 l 140 797 q 90 818 115 812 q 46 825 66 825 l 46 854 l 236 854 q 248 851 243 854 q 257 843 252 849 q 266 826 261 837 q 279 797 271 815 l 543 185 l 797 797 q 811 828 806 817 q 821 845 816 840 q 829 852 825 851 q 840 854 834 854 l 1030 854 l 1030 825 m 608 1044 q 602 1008 608 1025 q 585 979 596 991 q 561 959 575 966 q 530 952 547 952 q 487 968 500 952 q 475 1016 475 985 q 480 1052 475 1035 q 497 1082 486 1069 q 521 1101 508 1094 q 551 1109 535 1109 q 608 1044 608 1109 "
    },
    "Ë‰": {
      "x_min": 110.984375,
      "x_max": 560.265625,
      "ha": 671,
      "o": "m 560 868 q 555 851 558 861 q 548 831 552 841 q 540 812 544 822 q 534 797 537 803 l 125 797 l 110 813 q 115 830 112 821 q 122 850 118 840 q 130 869 126 859 q 138 885 134 878 l 545 885 l 560 868 "
    },
    "á¸»": {
      "x_min": -35.09375,
      "x_max": 414.09375,
      "ha": 369,
      "o": "m 40 0 l 40 29 q 88 38 68 33 q 118 49 107 44 q 135 59 130 54 q 140 70 140 65 l 140 878 q 136 926 140 909 q 121 951 132 943 q 92 963 110 960 q 46 969 74 966 l 46 996 q 134 1017 94 1006 q 216 1051 174 1029 l 240 1028 l 240 70 q 262 49 240 60 q 339 29 284 38 l 339 0 l 40 0 m 414 -155 q 409 -172 412 -162 q 402 -192 406 -182 q 394 -211 398 -201 q 388 -227 390 -220 l -20 -227 l -35 -210 q -30 -193 -33 -202 q -23 -173 -27 -183 q -15 -154 -19 -164 q -7 -139 -11 -145 l 399 -139 l 414 -155 "
    },
    "É”": {
      "x_min": 44,
      "x_max": 562,
      "ha": 615,
      "o": "m 562 329 q 538 193 562 256 q 475 81 515 129 q 381 7 435 34 q 265 -20 327 -20 q 177 -5 218 -20 q 107 36 137 9 q 60 100 77 63 q 44 182 44 137 q 50 223 44 203 q 70 257 56 242 q 124 274 95 265 q 178 289 153 283 l 192 257 q 176 240 182 248 q 165 223 169 232 q 158 203 160 213 q 157 177 157 192 q 168 131 157 154 q 198 91 179 108 q 242 62 217 73 q 292 51 266 51 q 361 67 329 51 q 414 116 392 83 q 449 196 436 148 q 462 308 462 245 q 445 413 462 366 q 401 492 429 459 q 337 542 373 524 q 261 560 301 560 q 223 557 242 560 q 183 546 204 554 q 135 523 161 538 q 75 482 109 507 q 59 496 66 486 q 49 511 52 506 q 124 582 89 555 q 189 625 158 610 q 249 646 220 641 q 309 652 278 652 q 403 629 357 652 q 484 566 448 607 q 540 464 519 525 q 562 329 562 404 "
    },
    "Ä¹": {
      "x_min": 33,
      "x_max": 627.09375,
      "ha": 648,
      "o": "m 627 165 q 613 63 621 106 q 599 0 605 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 111 q 241 89 236 98 q 261 74 247 80 q 299 64 275 67 q 359 61 322 61 l 454 61 q 505 64 484 61 q 542 79 526 67 q 571 114 558 91 q 598 177 584 137 l 627 165 m 222 926 q 206 938 211 931 q 192 957 201 944 l 449 1172 q 467 1161 456 1168 q 490 1148 478 1155 q 512 1134 502 1141 q 528 1122 522 1127 l 534 1091 l 222 926 "
    },
    "á»µ": {
      "x_min": -31,
      "x_max": 658,
      "ha": 672,
      "o": "m 658 601 q 625 592 638 596 q 604 583 612 588 q 592 571 596 578 q 584 555 588 565 l 362 -55 q 300 -184 334 -130 q 229 -272 266 -237 q 155 -322 192 -306 q 84 -339 118 -339 q 38 -335 59 -339 q 1 -327 16 -332 q -22 -315 -13 -322 q -31 -303 -31 -309 q -22 -286 -31 -299 q -2 -260 -14 -274 q 22 -232 9 -245 q 47 -211 36 -218 q 110 -232 79 -232 q 167 -223 142 -232 q 195 -204 179 -219 q 227 -168 211 -189 q 259 -119 243 -146 q 286 -62 274 -91 l 306 -15 l 87 555 q 64 584 81 574 q 14 601 47 593 l 14 630 l 271 630 l 271 601 q 228 594 245 598 q 203 586 212 591 q 193 573 194 581 q 195 555 191 566 l 359 123 l 512 555 q 513 572 515 565 q 502 584 511 579 q 476 593 493 589 q 436 601 460 597 l 436 630 l 658 630 l 658 601 m 579 -189 q 573 -225 579 -208 q 556 -254 567 -242 q 532 -274 546 -267 q 501 -282 518 -282 q 458 -265 471 -282 q 446 -217 446 -248 q 451 -181 446 -198 q 468 -151 457 -164 q 492 -132 479 -139 q 522 -125 506 -125 q 579 -189 579 -125 "
    },
    "ïˆ": {
      "x_min": 117,
      "x_max": 219,
      "ha": 333,
      "o": "m 219 443 q 198 437 210 440 q 172 433 186 435 q 144 430 158 431 q 117 429 129 429 l 117 963 q 134 973 122 967 q 159 986 146 980 q 184 997 172 992 q 201 1003 195 1002 l 219 985 l 219 443 "
    },
    "á¸‡": {
      "x_min": 6.59375,
      "x_max": 631,
      "ha": 691,
      "o": "m 631 333 q 621 253 631 294 q 591 172 611 211 q 544 97 572 133 q 481 36 516 62 q 403 -4 445 10 q 311 -20 360 -20 q 276 -14 298 -20 q 225 2 254 -8 q 162 28 196 13 q 93 65 128 44 l 93 878 q 89 927 93 910 q 77 952 86 943 q 50 963 67 960 q 6 969 33 966 l 6 996 q 90 1018 50 1007 q 168 1051 130 1030 l 175 1044 q 183 1036 179 1040 q 193 1028 187 1032 l 193 493 q 258 563 226 534 q 322 612 291 593 q 378 641 352 632 q 423 651 404 651 q 506 629 468 651 q 572 566 544 607 q 615 466 599 525 q 631 333 631 407 m 541 276 q 527 397 541 344 q 490 485 513 450 q 438 540 468 521 q 376 560 408 560 q 345 552 364 560 q 301 529 325 545 q 248 484 276 512 q 193 414 220 456 l 193 137 q 247 103 219 117 q 299 82 274 90 q 344 71 324 74 q 376 67 364 67 q 447 85 417 67 q 498 133 478 103 q 530 200 519 162 q 541 276 541 237 m 542 -155 q 537 -172 541 -162 q 530 -192 534 -182 q 522 -211 526 -201 q 516 -227 519 -220 l 108 -227 l 93 -210 q 98 -193 95 -202 q 105 -173 101 -183 q 113 -154 109 -164 q 121 -139 117 -145 l 527 -139 l 542 -155 "
    },
    "ÄŒ": {
      "x_min": 47,
      "x_max": 677.046875,
      "ha": 730,
      "o": "m 677 143 q 595 65 634 97 q 520 15 556 34 q 449 -11 484 -3 q 379 -20 414 -20 q 258 8 318 -20 q 152 90 199 36 q 75 221 104 144 q 47 397 47 299 q 79 594 47 506 q 166 744 111 682 q 294 841 221 807 q 449 875 368 875 q 575 855 521 875 q 664 807 629 835 q 662 793 668 803 q 646 770 656 783 q 624 744 635 757 q 604 723 612 731 l 581 727 q 500 780 547 759 q 393 800 454 800 q 344 791 371 800 q 290 765 317 783 q 237 717 263 746 q 192 645 212 687 q 161 548 173 603 q 150 422 150 493 q 175 264 150 332 q 241 151 201 196 q 331 83 281 106 q 427 61 380 61 q 521 86 463 61 q 652 173 579 111 q 659 167 655 172 q 666 158 663 163 q 672 149 669 153 q 677 143 675 145 m 444 944 l 373 944 l 197 1152 q 205 1163 202 1158 q 212 1171 209 1168 q 218 1177 215 1175 q 227 1184 222 1180 l 410 1040 l 589 1184 q 605 1171 601 1177 q 618 1152 610 1165 l 444 944 "
    },
    "x": {
      "x_min": 13,
      "x_max": 675,
      "ha": 684,
      "o": "m 408 0 l 408 30 q 438 32 424 30 q 461 41 453 35 q 470 58 470 47 q 456 88 470 70 l 325 271 l 199 89 q 189 59 187 70 q 203 42 192 48 q 232 33 215 36 q 265 29 249 30 l 265 0 l 13 0 l 13 29 q 57 39 39 33 q 89 54 76 46 q 111 73 102 63 q 127 93 120 82 l 289 322 l 133 540 q 115 563 124 553 q 94 582 107 574 q 64 594 82 589 q 20 600 46 600 l 20 631 l 299 631 l 299 601 q 263 596 278 600 q 241 586 248 592 q 234 569 233 579 q 247 542 235 558 l 354 391 l 456 542 q 470 569 468 558 q 464 586 471 580 q 442 596 457 593 q 408 601 428 600 l 408 631 l 661 631 l 661 600 q 577 583 607 600 q 530 540 547 566 l 391 340 l 568 93 q 585 73 576 82 q 607 54 595 63 q 636 38 619 45 q 675 29 653 32 l 675 0 l 408 0 "
    },
    "Ã¨": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 290 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 414 731 q 397 718 405 722 q 377 709 388 713 l 143 965 l 158 993 q 177 997 164 995 q 205 1002 190 1000 q 233 1007 219 1005 q 255 1009 247 1009 l 414 731 "
    },
    "Åƒ": {
      "x_min": 33,
      "x_max": 851,
      "ha": 884,
      "o": "m 33 0 l 33 29 q 105 48 82 35 q 129 70 129 61 l 129 778 q 83 810 107 799 q 33 824 58 820 l 33 854 l 173 854 q 190 852 183 854 q 203 845 196 850 q 217 829 209 839 q 237 801 224 818 l 674 187 l 674 782 q 652 804 674 790 q 578 824 630 817 l 578 854 l 851 854 l 851 824 q 778 805 802 818 q 754 782 754 792 l 754 -20 q 701 -6 720 -16 q 674 14 682 2 l 209 673 l 209 70 q 231 49 209 62 q 305 29 253 36 l 305 0 l 33 0 m 353 926 q 337 937 342 931 q 323 957 332 944 l 581 1172 q 599 1161 587 1167 q 622 1147 610 1155 q 644 1134 634 1140 q 659 1121 654 1127 l 665 1091 l 353 926 "
    },
    "á¸¿": {
      "x_min": 37,
      "x_max": 1075,
      "ha": 1094,
      "o": "m 789 0 l 789 29 q 858 51 836 42 q 881 70 881 61 l 881 429 q 876 498 881 470 q 863 541 872 525 q 840 563 854 557 q 805 570 825 570 q 759 557 783 570 q 708 521 734 544 q 656 464 682 498 q 605 388 630 430 l 605 70 q 625 52 605 61 q 699 29 646 43 l 699 0 l 413 0 l 413 29 q 482 51 460 42 q 505 70 505 61 l 505 429 q 501 498 505 470 q 488 541 497 525 q 464 563 479 557 q 430 570 450 570 q 335 522 385 570 q 230 388 286 475 l 230 70 q 254 49 230 60 q 323 29 279 39 l 323 0 l 37 0 l 37 29 q 105 50 80 41 q 130 70 130 59 l 130 482 q 127 525 130 509 q 115 549 125 540 q 87 561 106 557 q 37 570 68 565 l 37 597 q 83 606 62 601 q 124 619 104 612 q 161 634 143 626 q 196 651 178 642 l 223 627 l 223 472 q 287 550 257 517 q 348 606 318 584 q 406 639 378 628 q 457 651 434 651 q 516 643 489 651 q 563 616 543 635 q 594 567 582 598 q 605 492 605 537 l 605 477 q 663 552 633 520 q 722 606 693 584 q 780 639 752 628 q 833 651 808 651 q 892 642 865 651 q 938 615 919 633 q 969 568 958 596 q 981 502 981 540 l 981 70 q 1001 52 981 61 q 1075 29 1022 43 l 1075 0 l 789 0 m 526 709 q 505 717 514 712 q 488 728 495 722 l 644 1009 q 665 1007 652 1009 q 692 1003 678 1005 q 720 998 707 1001 q 741 994 733 996 l 756 967 l 526 709 "
    },
    "á¹†": {
      "x_min": 33,
      "x_max": 851,
      "ha": 884,
      "o": "m 33 0 l 33 29 q 105 48 82 35 q 129 70 129 61 l 129 778 q 83 810 107 799 q 33 824 58 820 l 33 854 l 173 854 q 190 852 183 854 q 203 845 196 850 q 217 829 209 839 q 237 801 224 818 l 674 187 l 674 782 q 652 804 674 790 q 578 824 630 817 l 578 855 l 851 855 l 851 824 q 778 805 802 818 q 754 782 754 792 l 754 -20 q 701 -6 720 -16 q 674 14 682 2 l 209 673 l 209 70 q 231 49 209 62 q 305 29 253 36 l 305 0 l 33 0 m 508 -189 q 502 -225 508 -208 q 485 -254 496 -242 q 461 -274 475 -267 q 430 -282 447 -282 q 387 -265 400 -282 q 375 -217 375 -248 q 380 -181 375 -198 q 397 -151 386 -164 q 421 -132 408 -139 q 451 -125 435 -125 q 508 -189 508 -125 "
    },
    ".": {
      "x_min": 87,
      "x_max": 245,
      "ha": 311,
      "o": "m 245 83 q 238 39 245 59 q 218 4 231 19 q 189 -18 206 -10 q 153 -27 172 -27 q 102 -7 117 -27 q 87 48 87 12 q 94 91 87 71 q 114 127 101 111 q 143 151 126 142 q 179 160 160 160 q 228 140 212 160 q 245 83 245 120 "
    },
    "áºˆ": {
      "x_min": 13,
      "x_max": 1131,
      "ha": 1157,
      "o": "m 1131 825 q 1082 814 1101 819 q 1051 803 1063 808 q 1035 793 1040 798 q 1029 783 1030 789 l 877 39 q 863 15 874 25 q 838 -2 852 4 q 809 -14 824 -9 q 785 -20 795 -18 l 568 640 l 377 39 q 361 15 372 25 q 336 -1 350 5 q 306 -13 321 -8 q 277 -20 290 -17 l 104 778 q 80 806 101 795 q 13 825 60 817 l 13 855 l 298 855 l 298 825 q 246 817 265 822 q 218 806 228 812 q 207 792 209 800 q 207 778 206 785 l 339 169 l 554 855 l 591 855 l 827 169 l 951 783 q 945 797 952 791 q 924 808 938 804 q 890 817 910 813 q 849 825 871 821 l 849 855 l 1131 855 l 1131 825 m 638 -189 q 632 -225 638 -208 q 615 -254 626 -242 q 591 -274 605 -267 q 560 -282 577 -282 q 517 -265 530 -282 q 505 -217 505 -248 q 510 -181 505 -198 q 527 -151 516 -164 q 551 -132 538 -139 q 581 -125 565 -125 q 638 -189 638 -125 "
    },
    "á¹£": {
      "x_min": 63,
      "x_max": 465,
      "ha": 525,
      "o": "m 465 192 q 451 109 465 144 q 417 51 438 75 q 369 13 395 28 q 318 -7 343 0 q 271 -17 293 -15 q 236 -20 248 -20 q 159 -7 204 -20 q 70 29 114 4 q 64 51 65 31 q 63 97 63 71 q 66 150 63 123 q 74 192 69 176 l 104 184 q 117 131 104 156 q 153 87 130 106 q 205 58 175 69 q 269 48 234 48 q 313 55 293 48 q 348 77 333 63 q 370 111 362 91 q 379 154 379 130 q 364 202 379 181 q 326 240 350 223 q 273 273 303 258 q 213 304 244 288 q 159 335 185 319 q 113 371 133 351 q 81 417 93 392 q 69 474 69 442 q 85 549 69 516 q 129 604 102 582 q 192 639 157 627 q 266 651 228 651 q 311 647 287 651 q 361 636 336 643 q 406 620 385 629 q 441 598 427 611 q 442 580 445 594 q 432 549 439 566 q 419 517 426 532 q 408 497 412 502 l 382 502 q 315 570 350 551 q 248 590 281 590 q 209 582 226 590 q 179 563 192 575 q 161 537 167 551 q 155 508 155 522 q 167 469 155 487 q 200 437 180 452 q 247 408 221 421 q 302 379 274 394 q 359 347 330 364 q 412 308 389 330 q 450 258 435 286 q 465 192 465 230 m 330 -189 q 324 -225 330 -208 q 307 -254 318 -242 q 283 -274 297 -267 q 252 -282 269 -282 q 209 -265 222 -282 q 197 -217 197 -248 q 202 -181 197 -198 q 219 -151 208 -164 q 243 -132 230 -139 q 273 -125 257 -125 q 330 -189 330 -125 "
    },
    "Ç": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 838 q 390 869 367 855 q 430 892 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 434 943 l 362 943 l 187 1151 q 195 1162 191 1157 q 201 1170 198 1167 q 208 1176 204 1174 q 216 1183 211 1179 l 400 1039 l 579 1183 q 595 1170 590 1176 q 608 1151 600 1164 l 434 943 "
    },
    "ÊŠ": {
      "x_min": 40,
      "x_max": 605,
      "ha": 646,
      "o": "m 605 601 q 533 580 563 590 q 508 570 520 575 q 489 559 497 565 q 479 548 481 554 q 483 538 477 542 q 532 483 512 511 q 564 424 552 454 q 581 361 576 393 q 587 294 587 329 q 579 227 587 262 q 555 156 571 191 q 516 89 539 121 q 462 33 493 57 q 393 -5 431 9 q 309 -20 355 -20 q 191 4 240 -20 q 111 68 142 28 q 64 161 79 108 q 50 270 50 213 q 74 414 50 346 q 159 538 98 481 q 163 548 165 543 q 154 560 161 554 q 135 570 147 565 q 111 580 124 576 q 40 601 82 591 l 40 631 l 270 631 l 282 568 q 225 516 250 544 q 184 455 201 488 q 158 381 167 422 q 150 290 150 341 q 165 202 150 245 q 207 127 181 160 q 268 74 233 94 q 341 54 303 54 q 409 73 380 54 q 458 123 438 93 q 487 189 477 152 q 497 260 497 225 q 487 358 497 310 q 461 446 478 405 q 419 517 443 486 q 362 568 394 548 l 375 631 l 605 631 l 605 601 "
    },
    "â€˜": {
      "x_min": 61,
      "x_max": 254.734375,
      "ha": 312,
      "o": "m 254 712 q 229 693 248 704 q 186 672 209 682 q 141 656 163 662 q 107 651 119 650 q 73 694 85 671 q 61 758 61 718 q 70 815 61 784 q 96 876 79 845 q 138 934 113 906 q 196 983 164 962 l 224 959 q 200 926 210 944 q 182 888 189 907 q 172 851 175 869 q 169 818 169 833 q 186 766 169 788 q 244 744 204 744 l 254 712 "
    },
    "Ï€": {
      "x_min": 13.234375,
      "x_max": 702.421875,
      "ha": 716,
      "o": "m 694 73 q 637 26 661 44 q 593 -2 612 7 q 559 -17 574 -13 q 530 -21 544 -21 q 491 -7 507 -21 q 463 33 474 7 q 446 96 451 59 q 441 180 441 133 q 441 233 441 202 q 441 306 441 264 q 441 404 441 349 q 455 534 441 460 q 362 534 408 534 q 256 534 317 534 q 256 372 256 457 q 256 202 256 288 q 256 112 256 153 q 256 33 256 70 q 228 19 247 27 q 187 3 209 11 q 142 -10 164 -3 q 105 -22 120 -18 q 100 -15 103 -20 q 94 -7 97 -11 q 86 2 90 -3 q 118 51 104 27 q 142 101 131 74 q 160 163 152 128 q 174 250 167 199 q 187 370 181 300 q 194 537 194 441 l 161 537 q 124 535 140 537 q 92 529 107 534 q 62 516 77 525 q 28 494 47 508 l 13 519 q 51 563 32 542 q 88 598 69 583 q 126 621 107 613 q 167 630 146 630 l 587 630 q 639 632 615 630 q 683 650 663 635 l 702 630 q 632 555 667 581 q 566 529 598 529 l 522 529 q 517 418 517 474 q 517 309 517 362 q 517 203 517 247 q 524 130 517 159 q 542 89 531 102 q 570 76 554 76 q 590 77 580 76 q 612 81 600 78 q 640 89 624 84 q 679 103 657 95 l 694 73 "
    },
    "âˆ…": {
      "x_min": 53,
      "x_max": 844,
      "ha": 897,
      "o": "m 758 426 q 737 562 758 498 q 678 675 716 625 l 271 121 q 353 71 309 88 q 448 54 398 54 q 568 83 512 54 q 667 163 625 112 q 733 282 709 214 q 758 426 758 350 m 139 426 q 159 291 139 355 q 218 178 180 228 l 624 732 q 542 782 586 765 q 448 800 498 800 q 328 770 384 800 q 229 690 271 741 q 163 571 187 639 q 139 426 139 503 m 53 427 q 67 546 53 489 q 107 653 81 603 q 169 743 132 702 q 249 813 205 784 q 343 859 293 843 q 448 875 394 875 q 567 854 511 875 q 671 796 623 833 l 741 891 q 782 910 759 902 q 825 924 804 919 l 844 902 l 728 743 q 812 601 781 681 q 844 427 844 520 q 812 253 844 334 q 727 111 781 171 q 601 15 673 50 q 448 -20 530 -20 q 329 0 385 -20 q 225 58 273 21 l 156 -34 q 140 -45 151 -40 q 116 -54 129 -50 q 91 -63 103 -59 q 71 -69 79 -66 l 53 -47 l 168 110 q 84 253 115 172 q 53 427 53 334 "
    },
    "á»Ž": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 521 1121 q 508 1087 521 1102 q 480 1059 496 1072 q 447 1034 463 1046 q 424 1010 432 1022 q 423 984 417 997 q 458 953 430 970 q 438 945 448 946 q 417 942 427 944 q 372 970 386 957 q 355 993 358 982 q 358 1013 352 1004 q 376 1032 364 1023 q 401 1050 387 1041 q 428 1068 415 1059 q 448 1088 440 1078 q 457 1111 457 1099 q 444 1150 457 1137 q 411 1163 431 1163 q 393 1159 401 1163 q 378 1149 384 1155 q 369 1135 372 1143 q 366 1121 366 1128 q 367 1113 366 1117 q 371 1106 369 1110 q 357 1102 366 1104 q 339 1097 349 1099 q 321 1094 330 1095 q 305 1091 311 1092 l 298 1099 l 298 1108 q 310 1140 298 1124 q 342 1168 323 1155 q 386 1188 362 1180 q 433 1196 410 1196 q 498 1175 475 1196 q 521 1121 521 1155 "
    },
    "á»š": {
      "x_min": 47,
      "x_max": 811,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 811 944 q 803 906 811 927 q 778 861 796 885 q 730 812 760 837 q 654 761 700 786 q 731 614 705 698 q 757 438 757 529 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 532 853 482 875 q 622 796 582 832 q 679 846 661 821 q 697 889 697 871 q 687 926 697 908 q 658 959 677 944 l 776 1014 q 801 981 791 999 q 811 944 811 963 m 321 927 q 305 938 310 931 q 291 957 300 944 l 548 1172 q 566 1162 555 1168 q 590 1148 578 1155 q 612 1134 602 1141 q 627 1122 622 1127 l 633 1092 l 321 927 "
    },
    "á¹–": {
      "x_min": 27.015625,
      "x_max": 652.015625,
      "ha": 711,
      "o": "m 33 0 l 33 29 q 104 50 78 38 q 130 70 130 61 l 130 806 q 80 799 105 803 q 33 791 56 796 l 27 833 q 96 851 58 843 q 174 865 133 859 q 256 871 214 871 q 338 871 298 871 q 468 855 410 871 q 566 809 525 840 q 629 735 607 779 q 652 632 652 690 q 639 548 652 586 q 604 479 626 509 q 555 427 582 449 q 497 391 527 406 q 436 370 466 377 q 380 363 406 363 q 273 383 318 363 l 258 434 q 311 417 286 421 q 360 414 337 414 q 424 426 391 414 q 485 462 457 438 q 531 522 513 487 q 549 606 549 558 q 529 702 549 662 q 476 767 509 741 q 397 804 442 792 q 300 817 352 817 q 268 817 284 817 q 236 817 251 817 l 236 70 q 241 61 236 66 q 259 51 246 57 q 294 40 272 46 q 352 29 317 35 l 352 0 l 33 0 m 406 1044 q 400 1008 406 1025 q 383 979 394 991 q 359 959 373 966 q 328 952 345 952 q 285 968 298 952 q 273 1016 273 985 q 278 1052 273 1035 q 295 1082 284 1069 q 319 1101 306 1094 q 349 1109 333 1109 q 406 1044 406 1109 "
    },
    "9": {
      "x_min": 66,
      "x_max": 578,
      "ha": 638,
      "o": "m 318 376 q 403 402 364 376 q 470 470 442 428 q 454 612 470 555 q 415 704 438 669 q 363 752 392 738 q 309 767 335 767 q 248 753 275 767 q 203 714 221 739 q 175 655 185 689 q 166 580 166 621 q 180 478 166 518 q 218 416 195 439 q 267 385 240 393 q 318 376 294 376 m 578 496 q 550 304 578 393 q 467 147 523 215 q 326 33 411 79 q 126 -28 241 -11 l 114 11 q 266 70 202 33 q 373 155 330 106 q 438 263 415 203 q 468 391 461 323 q 428 354 450 371 q 384 326 407 337 q 337 308 361 314 q 290 302 313 302 q 195 319 237 302 q 125 369 154 337 q 81 445 96 401 q 66 539 66 488 q 74 609 66 573 q 100 677 83 644 q 142 740 117 711 q 196 791 166 769 q 261 825 226 812 q 335 838 296 838 q 426 818 382 838 q 503 756 469 798 q 557 650 537 715 q 578 496 578 586 "
    },
    "á¹ž": {
      "x_min": 27.015625,
      "x_max": 756.890625,
      "ha": 764,
      "o": "m 33 0 l 33 29 q 104 49 78 38 q 130 70 130 60 l 130 806 q 82 799 106 803 q 33 792 57 795 l 27 834 q 90 849 58 842 q 156 862 122 856 q 229 871 191 868 q 311 875 267 875 q 445 859 388 875 q 540 816 502 843 q 597 750 578 788 q 616 669 616 712 q 602 581 616 620 q 563 510 588 541 q 504 458 538 480 q 428 424 469 437 l 629 94 q 651 69 639 78 q 677 56 663 60 q 709 52 691 52 q 749 54 726 52 l 756 25 q 673 0 713 10 q 608 -9 633 -9 q 570 1 588 -9 q 542 27 552 12 l 341 408 q 324 407 332 407 l 306 407 q 271 408 288 407 q 236 414 254 410 l 236 70 q 259 50 236 62 q 332 29 283 38 l 332 0 l 33 0 m 287 817 q 236 817 261 817 l 236 469 q 267 465 253 465 q 295 465 280 465 q 456 511 400 465 q 512 648 512 558 q 500 716 512 685 q 459 769 487 747 q 389 804 431 792 q 287 817 346 817 m 570 -155 q 565 -172 568 -162 q 558 -192 562 -182 q 550 -211 554 -201 q 544 -227 546 -220 l 135 -227 l 121 -210 q 125 -193 122 -202 q 133 -173 129 -183 q 141 -154 137 -164 q 148 -139 145 -145 l 555 -139 l 570 -155 "
    },
    "l": {
      "x_min": 40,
      "x_max": 339,
      "ha": 369,
      "o": "m 40 0 l 40 29 q 88 38 68 33 q 118 49 107 44 q 135 59 130 54 q 140 70 140 65 l 140 878 q 136 926 140 909 q 121 951 132 943 q 92 963 110 960 q 46 969 74 966 l 46 996 q 134 1017 94 1006 q 216 1051 174 1029 l 240 1028 l 240 70 q 262 49 240 60 q 339 29 284 38 l 339 0 l 40 0 "
    },
    "áºª": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 839 q 390 869 367 856 q 430 893 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 608 957 q 595 938 600 944 q 579 927 590 932 l 397 1068 l 216 927 q 208 932 211 929 q 201 938 204 935 q 195 946 198 942 q 187 957 191 951 l 362 1167 l 434 1167 l 608 957 m 624 1374 q 596 1319 612 1346 q 559 1270 579 1291 q 515 1234 539 1248 q 465 1221 491 1221 q 420 1232 441 1221 q 380 1259 400 1244 q 340 1286 360 1274 q 301 1298 320 1298 q 252 1275 274 1298 q 206 1215 230 1253 l 171 1229 q 199 1284 183 1257 q 236 1333 215 1312 q 280 1369 256 1355 q 330 1383 304 1383 q 378 1371 355 1383 q 420 1344 400 1359 q 458 1317 440 1329 q 492 1306 476 1306 q 542 1327 517 1306 q 588 1389 566 1349 l 624 1374 "
    },
    "È¬": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 636 1103 q 607 1048 624 1075 q 570 999 590 1020 q 526 963 550 977 q 476 950 502 950 q 432 961 453 950 q 391 988 411 973 q 351 1015 371 1003 q 312 1027 332 1027 q 264 1004 286 1027 q 217 944 242 982 l 182 958 q 210 1013 194 986 q 247 1062 227 1041 q 291 1098 268 1084 q 341 1112 315 1112 q 389 1100 366 1112 q 431 1073 411 1088 q 469 1046 451 1058 q 503 1035 487 1035 q 553 1056 528 1035 q 600 1118 577 1078 l 636 1103 m 640 1275 q 636 1258 639 1268 q 628 1238 632 1248 q 621 1219 624 1229 q 614 1204 617 1210 l 205 1204 l 191 1220 q 195 1237 192 1228 q 203 1257 199 1247 q 211 1276 207 1266 q 218 1292 215 1285 l 626 1292 l 640 1275 "
    },
    "Ãœ": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 615 816 l 568 855 l 847 855 l 847 825 m 649 1045 q 643 1009 649 1026 q 626 980 637 992 q 602 960 616 967 q 571 953 588 953 q 528 969 541 953 q 516 1017 516 986 q 521 1053 516 1036 q 538 1083 527 1070 q 562 1102 549 1095 q 592 1110 576 1110 q 649 1045 649 1110 m 383 1045 q 377 1009 383 1026 q 360 980 371 992 q 336 960 350 967 q 305 953 322 953 q 262 969 275 953 q 250 1017 250 986 q 255 1053 250 1036 q 272 1083 261 1070 q 296 1102 283 1095 q 326 1110 310 1110 q 383 1045 383 1110 "
    },
    "Ã ": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 394 731 q 377 718 386 722 q 357 709 368 713 l 123 965 l 138 993 q 157 997 144 995 q 184 1002 170 1000 q 213 1007 199 1005 q 235 1009 227 1009 l 394 731 "
    },
    "Åš": {
      "x_min": 80,
      "x_max": 589,
      "ha": 649,
      "o": "m 589 255 q 581 193 589 225 q 556 130 573 161 q 516 72 540 99 q 458 24 491 45 q 384 -7 426 4 q 293 -20 342 -20 q 245 -15 271 -20 q 191 -2 219 -10 q 137 18 164 6 q 87 46 110 30 q 81 69 82 48 q 80 116 80 89 q 80 172 80 144 q 87 219 80 201 l 114 216 q 152 147 129 176 q 203 98 175 117 q 263 70 231 79 q 330 61 296 61 q 390 73 360 61 q 444 107 420 86 q 483 158 468 129 q 499 219 499 187 q 481 290 499 261 q 434 343 463 320 q 367 385 405 366 q 291 421 330 403 q 215 460 252 440 q 149 507 178 480 q 102 570 120 535 q 85 655 85 606 q 90 701 85 676 q 110 750 96 725 q 144 797 123 774 q 195 837 165 820 q 263 864 224 854 q 350 875 301 875 q 410 870 380 875 q 466 857 440 865 q 514 837 493 849 q 547 812 535 826 q 547 796 552 808 q 534 770 543 784 q 514 743 525 756 q 495 722 503 729 l 471 726 q 432 763 453 748 q 391 789 412 779 q 349 802 370 798 q 310 807 329 807 q 247 795 273 807 q 203 766 220 783 q 178 727 186 749 q 171 687 171 706 q 188 631 171 655 q 236 585 206 606 q 303 546 265 564 q 379 508 340 528 q 456 465 419 488 q 523 412 494 442 q 571 344 553 382 q 589 255 589 305 m 244 927 q 228 938 233 931 q 214 957 223 944 l 472 1172 q 490 1162 479 1168 q 513 1148 501 1155 q 535 1134 525 1141 q 551 1122 545 1127 l 557 1092 l 244 927 "
    },
    "Ã³": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 326 710 q 305 718 314 712 q 288 729 296 723 l 444 1010 q 465 1007 452 1009 q 492 1004 478 1006 q 520 999 506 1002 q 540 994 533 996 l 555 968 l 326 710 "
    },
    "ÇŸ": {
      "x_min": 53,
      "x_max": 615.03125,
      "ha": 624,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 521 -2 558 15 q 466 -20 484 -20 q 429 10 444 -20 q 415 97 415 42 q 353 44 383 66 q 295 7 323 21 q 241 -13 266 -6 q 194 -20 215 -20 q 145 -12 170 -20 q 99 13 120 -4 q 66 58 79 31 q 53 126 53 86 q 70 212 53 177 q 112 271 88 246 q 148 302 128 288 q 202 330 169 317 q 286 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 280 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 165 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 578 74 l 615 55 m 518 854 q 512 818 518 835 q 495 789 506 801 q 471 769 485 776 q 440 762 457 762 q 397 778 410 762 q 385 826 385 795 q 390 862 385 845 q 407 892 396 879 q 431 911 418 904 q 461 919 445 919 q 518 854 518 919 m 252 854 q 246 818 252 835 q 229 789 240 801 q 205 769 219 776 q 174 762 191 762 q 131 778 144 762 q 119 826 119 795 q 124 862 119 845 q 141 892 130 879 q 165 911 152 904 q 195 919 179 919 q 252 854 252 919 m 549 1112 q 545 1095 548 1105 q 537 1075 541 1085 q 530 1056 533 1066 q 523 1041 526 1047 l 115 1041 l 100 1057 q 105 1074 101 1065 q 112 1094 108 1084 q 120 1113 116 1103 q 127 1129 124 1122 l 535 1129 l 549 1112 "
    },
    "áº": {
      "x_min": 13,
      "x_max": 675,
      "ha": 684,
      "o": "m 408 0 l 408 30 q 438 32 424 30 q 461 41 453 35 q 470 58 470 47 q 456 88 470 70 l 325 271 l 199 89 q 189 59 187 70 q 203 42 192 48 q 232 33 215 36 q 265 29 249 30 l 265 0 l 13 0 l 13 29 q 58 39 39 33 q 89 54 76 46 q 111 73 102 63 q 127 93 120 82 l 289 322 l 133 540 q 115 563 124 553 q 94 582 107 574 q 64 594 82 589 q 20 600 46 600 l 20 631 l 299 631 l 299 601 q 264 596 279 600 q 241 586 249 592 q 235 569 234 579 q 248 542 236 558 l 354 391 l 456 542 q 470 569 468 558 q 464 586 471 580 q 442 596 457 593 q 408 601 428 600 l 408 631 l 661 631 l 661 600 q 577 583 607 600 q 530 540 547 566 l 391 340 l 568 93 q 585 73 576 82 q 607 54 594 63 q 636 38 619 45 q 675 29 653 32 l 675 0 l 408 0 m 543 854 q 537 818 543 835 q 520 789 531 801 q 496 769 510 776 q 465 762 482 762 q 422 778 435 762 q 410 826 410 795 q 415 862 410 845 q 432 892 421 879 q 456 911 443 904 q 486 919 470 919 q 543 854 543 919 m 278 854 q 272 818 278 835 q 255 789 266 801 q 231 769 245 776 q 200 762 217 762 q 157 778 170 762 q 145 826 145 795 q 150 862 145 845 q 167 892 156 879 q 191 911 178 904 q 221 919 205 919 q 278 854 278 919 "
    },
    "Â¦": {
      "x_min": 116,
      "x_max": 217,
      "ha": 312,
      "o": "m 217 -240 q 200 -250 210 -245 q 180 -260 191 -255 q 159 -268 169 -265 q 140 -275 148 -272 l 116 -259 l 116 331 q 153 351 135 343 q 192 366 171 360 l 217 349 l 217 -240 m 217 520 q 200 509 210 514 q 180 500 191 504 q 159 491 169 495 q 140 485 148 488 l 116 500 l 116 1099 q 153 1118 135 1110 q 192 1132 171 1127 l 217 1117 l 217 520 "
    },
    "Æ©": {
      "x_min": 39.046875,
      "x_max": 667,
      "ha": 707,
      "o": "m 667 234 q 667 173 667 206 q 661 107 667 140 l 654 46 q 654 0 654 18 l 59 0 l 39 29 l 329 430 l 53 825 l 53 856 l 501 856 q 570 856 538 856 q 644 865 602 856 l 644 661 l 608 653 q 584 730 596 703 q 560 770 572 757 q 538 784 549 782 q 518 787 528 787 l 206 787 l 433 457 l 170 95 l 536 95 q 571 101 557 95 q 594 124 584 107 q 614 168 604 140 q 635 242 623 197 l 667 234 "
    },
    "Ì›": {
      "x_min": -215.53125,
      "x_max": 46,
      "ha": 46,
      "o": "m 46 706 q 34 659 46 686 q -4 602 23 632 q -81 541 -32 572 q -203 481 -129 509 l -215 523 q -148 555 -176 538 q -102 590 -120 573 q -76 623 -84 607 q -68 651 -68 639 q -77 688 -68 671 q -106 722 -87 706 l 11 776 q 36 743 26 761 q 46 706 46 726 "
    },
    "áº”": {
      "x_min": 39.578125,
      "x_max": 655,
      "ha": 712,
      "o": "m 655 198 q 655 150 655 177 q 655 97 655 124 q 655 45 655 70 q 655 0 655 19 l 58 0 l 39 30 l 512 787 l 207 787 q 183 779 196 787 q 159 755 171 772 q 136 714 147 739 q 118 653 125 688 l 80 661 l 100 865 q 132 858 118 861 q 162 855 147 855 q 191 855 176 855 l 635 855 l 652 825 l 183 68 l 531 68 q 556 74 545 68 q 578 96 567 80 q 599 139 588 112 q 622 208 609 166 l 655 198 m 580 -155 q 575 -172 579 -162 q 568 -192 572 -182 q 560 -211 564 -201 q 554 -227 557 -220 l 145 -227 l 131 -210 q 135 -193 132 -202 q 143 -173 139 -183 q 151 -154 147 -164 q 158 -139 155 -145 l 565 -139 l 580 -155 "
    },
    "á»¶": {
      "x_min": -0.265625,
      "x_max": 781,
      "ha": 808,
      "o": "m 239 0 l 239 29 q 323 55 299 42 q 348 78 348 68 l 348 364 q 292 478 323 419 q 228 594 260 538 q 165 699 195 651 q 111 780 134 748 q 97 794 105 788 q 78 806 90 801 q 48 814 66 811 q 2 818 30 818 l 0 846 q 78 856 38 852 q 145 861 117 861 q 197 834 175 861 q 250 757 221 802 q 307 663 278 713 q 363 562 335 614 q 413 461 390 510 l 592 780 q 585 808 602 798 q 518 825 569 818 l 518 854 l 781 854 l 781 825 q 711 807 735 816 q 677 780 686 798 l 454 366 l 454 78 q 459 68 454 73 q 478 55 465 62 q 512 42 491 48 q 565 29 533 35 l 565 0 l 239 0 m 513 1121 q 500 1087 513 1102 q 471 1059 488 1072 q 439 1034 454 1046 q 416 1009 424 1022 q 415 983 409 997 q 450 953 422 970 q 430 945 440 946 q 409 942 419 943 q 364 969 378 957 q 347 993 350 982 q 350 1013 344 1004 q 368 1032 356 1023 q 393 1050 379 1041 q 420 1068 407 1059 q 440 1088 432 1078 q 449 1111 449 1099 q 436 1150 449 1137 q 403 1163 423 1163 q 385 1159 393 1163 q 370 1149 376 1155 q 361 1135 364 1143 q 358 1120 358 1128 q 359 1113 358 1117 q 363 1106 361 1110 q 349 1102 358 1104 q 332 1097 341 1099 q 313 1094 322 1095 q 297 1091 303 1092 l 290 1099 l 290 1108 q 302 1140 290 1124 q 334 1168 315 1155 q 378 1188 354 1180 q 425 1196 402 1196 q 490 1175 467 1196 q 513 1121 513 1155 "
    },
    "Å": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 300 927 q 282 933 289 929 q 263 945 276 937 l 380 1226 q 399 1223 388 1225 q 420 1219 409 1222 q 442 1215 431 1217 q 459 1209 452 1212 l 473 1182 l 300 927 m 494 927 q 475 933 481 929 q 456 945 468 937 l 573 1226 q 592 1223 581 1225 q 613 1219 602 1222 q 635 1215 624 1217 q 652 1209 645 1212 l 667 1182 l 494 927 "
    },
    "á¹­": {
      "x_min": 13.3125,
      "x_max": 449.53125,
      "ha": 469,
      "o": "m 449 79 q 385 36 416 55 q 324 5 353 18 q 270 -13 295 -6 q 228 -19 246 -19 q 184 -10 205 -19 q 147 17 163 -2 q 122 70 131 37 q 113 150 113 102 l 113 567 l 27 567 l 13 585 l 65 631 l 113 631 l 113 797 l 192 868 l 213 851 l 213 631 l 430 631 l 449 611 q 434 591 443 602 q 416 570 425 580 q 398 553 406 561 q 382 543 389 545 q 333 559 366 551 q 247 567 301 567 l 213 567 l 213 208 q 216 140 213 167 q 228 97 220 113 q 251 74 237 81 q 284 68 264 68 q 343 77 308 68 q 430 114 378 87 l 449 79 m 298 -189 q 292 -225 298 -208 q 275 -254 286 -242 q 251 -274 265 -267 q 220 -282 237 -282 q 177 -265 190 -282 q 165 -217 165 -248 q 170 -181 165 -198 q 187 -151 176 -164 q 211 -132 198 -139 q 241 -125 225 -125 q 298 -189 298 -125 "
    },
    "á¹”": {
      "x_min": 27.015625,
      "x_max": 652.015625,
      "ha": 711,
      "o": "m 33 0 l 33 29 q 104 50 78 38 q 130 70 130 61 l 130 806 q 80 799 105 803 q 33 791 56 796 l 27 833 q 96 851 58 843 q 174 865 133 859 q 256 871 214 871 q 338 871 298 871 q 468 855 410 871 q 566 809 525 840 q 629 735 607 779 q 652 632 652 690 q 639 548 652 586 q 604 479 626 509 q 555 427 583 449 q 497 391 527 406 q 436 370 466 377 q 380 363 406 363 q 273 383 318 363 l 258 434 q 311 417 286 421 q 360 414 337 414 q 424 426 391 414 q 485 462 457 438 q 531 522 513 487 q 549 606 549 558 q 529 702 549 662 q 476 767 509 741 q 397 804 442 792 q 300 817 352 817 q 268 817 284 817 q 236 817 251 817 l 236 70 q 241 61 236 66 q 259 51 246 57 q 294 40 272 46 q 352 29 317 35 l 352 0 l 33 0 m 251 926 q 235 937 240 931 q 222 957 230 944 l 478 1172 q 496 1161 485 1168 q 519 1148 507 1155 q 541 1134 531 1140 q 556 1121 551 1127 l 562 1091 l 251 926 "
    },
    "Å½": {
      "x_min": 39.578125,
      "x_max": 655,
      "ha": 712,
      "o": "m 655 198 q 655 150 655 177 q 655 97 655 124 q 655 45 655 70 q 655 0 655 19 l 58 0 l 39 30 l 512 787 l 207 787 q 183 779 196 787 q 159 755 171 772 q 136 714 147 739 q 118 653 125 688 l 80 661 l 100 865 q 132 858 118 861 q 162 855 147 855 q 191 855 176 855 l 635 855 l 652 825 l 183 68 l 531 68 q 556 74 545 68 q 578 96 567 80 q 599 139 588 112 q 622 208 609 166 l 655 198 m 405 943 l 333 943 l 158 1151 q 166 1162 163 1157 q 172 1170 169 1167 q 179 1176 175 1174 q 188 1183 183 1179 l 371 1039 l 550 1183 q 566 1170 561 1176 q 579 1151 571 1164 l 405 943 "
    },
    "Æª": {
      "x_min": 20,
      "x_max": 651,
      "ha": 470,
      "o": "m 178 823 q 206 828 193 823 q 230 840 219 833 q 249 857 241 848 q 264 875 258 867 q 174 987 244 987 q 147 980 160 987 q 124 961 134 973 q 108 935 114 950 q 103 905 103 920 q 109 871 103 886 q 126 845 115 856 q 150 829 136 835 q 178 823 164 823 m 20 869 q 34 931 20 899 q 75 990 49 963 q 137 1033 101 1016 q 215 1051 173 1051 q 281 1031 253 1051 q 327 979 309 1011 q 354 904 345 946 q 363 818 363 862 q 363 608 363 715 q 363 398 363 502 q 363 196 363 294 q 363 14 363 98 q 370 -122 363 -67 q 391 -211 377 -177 q 424 -260 405 -245 q 468 -275 443 -275 q 500 -265 486 -275 q 523 -241 514 -256 q 533 -210 532 -227 q 527 -179 535 -193 q 538 -168 525 -176 q 570 -151 551 -159 q 608 -137 589 -143 q 637 -133 627 -131 l 651 -159 q 633 -216 651 -185 q 585 -275 615 -247 q 513 -320 554 -302 q 428 -339 472 -339 q 352 -317 384 -339 q 299 -255 320 -295 q 267 -157 278 -214 q 257 -27 257 -99 q 257 178 257 73 q 257 386 257 282 q 257 588 257 489 q 257 777 257 688 q 257 790 257 784 q 257 804 257 797 q 208 755 241 770 q 141 741 175 741 q 90 750 112 741 q 52 777 67 760 q 28 818 36 795 q 20 869 20 842 "
    },
    "ÃŽ": {
      "x_min": -14.140625,
      "x_max": 406.484375,
      "ha": 406,
      "o": "m 47 0 l 47 29 q 118 49 92 38 q 144 70 144 61 l 144 782 q 119 803 144 790 q 47 824 95 815 l 47 854 l 346 854 l 346 824 q 275 803 300 815 q 250 782 250 792 l 250 70 q 274 50 250 62 q 346 29 298 38 l 346 0 l 47 0 m 406 957 q 393 938 398 944 q 377 926 389 932 l 196 1067 l 15 926 q 6 931 10 929 q 0 938 3 934 q -6 946 -2 941 q -14 957 -9 950 l 161 1166 l 232 1166 l 406 957 "
    },
    "e": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 290 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 "
    },
    "á»€": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 413 432 428 427 q 377 438 398 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 543 957 q 531 938 535 944 q 515 926 526 932 l 333 1067 l 153 926 q 144 931 147 929 q 137 938 140 934 q 131 946 134 941 q 123 957 128 950 l 298 1166 l 369 1166 l 543 957 m 451 1228 q 437 1208 441 1215 q 421 1197 432 1202 l 109 1362 l 115 1393 q 130 1405 119 1398 q 152 1419 140 1412 q 175 1432 164 1426 q 194 1443 186 1439 l 451 1228 "
    },
    "Ä”": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 413 432 428 427 q 377 438 398 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 549 1139 q 504 1046 529 1084 q 452 985 479 1009 q 394 952 424 962 q 335 942 364 942 q 272 952 302 942 q 213 985 241 962 q 162 1046 186 1009 q 117 1139 137 1084 q 133 1157 125 1151 q 153 1169 141 1164 q 191 1104 169 1131 q 236 1062 212 1078 q 285 1039 260 1046 q 332 1032 310 1032 q 381 1039 355 1032 q 431 1062 406 1046 q 476 1104 455 1078 q 514 1169 498 1131 q 533 1157 525 1164 q 549 1139 541 1151 "
    },
    "á»‹": {
      "x_min": 47,
      "x_max": 333,
      "ha": 370,
      "o": "m 47 0 l 47 29 q 115 49 92 38 q 139 70 139 61 l 139 454 q 139 510 139 488 q 130 543 139 531 q 101 560 121 555 q 47 569 82 566 l 47 596 q 90 606 67 600 q 136 619 113 612 q 179 634 158 626 q 217 651 200 642 l 240 651 l 240 70 q 262 50 240 62 q 333 29 284 38 l 333 0 l 47 0 m 260 854 q 254 818 260 835 q 237 789 248 801 q 213 769 227 776 q 182 762 199 762 q 139 778 152 762 q 127 826 127 795 q 132 862 127 845 q 149 892 138 879 q 173 911 160 904 q 203 919 187 919 q 260 854 260 919 m 256 -189 q 250 -225 256 -208 q 233 -254 244 -242 q 209 -274 223 -267 q 178 -282 195 -282 q 135 -265 148 -282 q 123 -217 123 -248 q 128 -181 123 -198 q 145 -151 134 -164 q 169 -132 156 -139 q 199 -125 183 -125 q 256 -189 256 -125 "
    },
    "á¹‚": {
      "x_min": 40,
      "x_max": 1043,
      "ha": 1097,
      "o": "m 1030 825 q 988 818 1010 825 q 944 800 966 812 l 944 70 q 969 50 944 62 q 1043 29 995 38 l 1043 0 l 741 0 l 741 29 q 813 49 786 38 q 840 70 840 61 l 840 705 l 546 0 l 514 0 l 216 700 l 216 70 q 238 50 216 62 q 309 29 261 38 l 309 0 l 40 0 l 40 29 q 112 49 85 38 q 140 70 140 61 l 140 797 q 90 818 115 812 q 46 825 66 825 l 46 854 l 236 854 q 248 851 243 854 q 257 843 252 849 q 266 826 261 837 q 279 797 271 815 l 543 185 l 797 797 q 811 828 806 817 q 821 845 816 840 q 829 852 825 851 q 840 854 834 854 l 1030 854 l 1030 825 m 608 -189 q 602 -225 608 -208 q 585 -254 596 -242 q 561 -274 575 -267 q 530 -282 547 -282 q 487 -265 500 -282 q 475 -217 475 -248 q 480 -181 475 -198 q 497 -151 486 -164 q 521 -132 508 -139 q 551 -125 535 -125 q 608 -189 608 -125 "
    },
    "â—Œ": {
      "x_min": 50,
      "x_max": 659,
      "ha": 709,
      "o": "m 323 588 q 331 611 323 602 q 354 621 340 621 q 376 611 367 621 q 386 588 386 602 q 376 566 386 575 q 354 557 367 557 q 331 566 340 557 q 323 588 323 575 m 323 31 q 331 54 323 45 q 354 64 340 64 q 376 54 367 64 q 386 31 386 45 q 376 9 386 18 q 354 0 367 0 q 331 9 340 0 q 323 31 323 18 m 429 579 q 441 594 433 589 q 459 600 449 600 q 481 589 472 600 q 491 566 491 579 q 481 544 491 553 q 459 535 471 535 q 436 545 445 535 q 428 568 428 556 q 429 579 428 573 m 220 64 q 233 79 224 74 q 250 85 242 85 q 272 74 263 85 q 282 52 282 64 q 272 30 282 39 q 250 21 262 21 q 227 31 236 21 q 218 52 218 41 q 218 58 218 55 q 220 64 219 61 m 524 530 q 547 540 535 540 q 569 530 560 540 q 578 507 578 521 q 569 484 578 493 q 547 475 560 475 q 524 485 533 475 q 516 507 516 495 q 524 530 516 520 m 139 135 q 160 146 149 146 q 183 136 174 146 q 193 114 193 127 q 184 91 193 101 q 161 82 175 82 q 139 91 149 82 q 130 114 130 101 q 139 135 130 126 m 594 447 q 600 449 597 448 q 606 450 602 450 q 628 440 618 450 q 638 417 638 431 q 628 395 638 405 q 605 385 619 385 q 584 394 593 385 q 575 416 575 403 q 580 434 575 426 q 594 447 585 443 m 89 233 q 102 236 95 236 q 124 227 115 236 q 134 205 134 218 q 124 182 134 192 q 101 172 115 172 q 79 181 89 172 q 70 205 70 191 q 75 222 70 214 q 89 233 80 229 m 626 343 q 649 333 640 343 q 659 310 659 324 q 650 288 659 297 q 627 279 641 279 l 625 279 q 604 288 613 279 q 595 310 595 297 q 604 332 595 323 q 626 343 614 341 m 81 342 q 104 332 94 342 q 114 311 114 323 q 104 287 114 297 q 82 278 94 278 l 80 278 q 58 287 67 278 q 50 310 50 297 q 59 332 50 323 q 81 342 68 342 m 618 233 q 632 221 627 229 q 638 203 638 213 q 628 181 638 190 q 606 172 618 172 q 582 181 590 172 q 574 203 574 191 q 584 227 574 219 q 606 236 594 236 q 618 233 612 236 m 114 447 q 128 434 123 443 q 134 416 134 425 q 123 393 134 402 q 101 385 113 385 q 78 395 87 385 q 70 417 70 405 q 79 440 70 431 q 101 450 89 450 q 107 449 105 450 q 114 447 110 448 m 569 137 q 579 114 579 127 q 570 91 579 101 q 547 82 561 82 q 524 91 533 82 q 515 114 515 101 q 523 136 515 127 q 545 146 532 146 q 569 137 558 146 m 182 530 q 193 508 193 521 q 184 484 193 493 q 161 476 175 476 q 139 485 148 476 q 130 507 130 494 q 139 530 130 520 q 161 540 148 540 q 182 530 173 540 m 488 65 q 490 59 489 62 q 491 53 491 56 q 481 31 491 41 q 459 21 472 21 q 437 30 447 21 q 427 54 427 39 q 435 75 427 65 q 457 86 444 86 q 475 80 466 86 q 488 65 483 75 m 278 580 q 281 567 281 575 q 272 544 281 554 q 251 535 264 535 q 227 544 237 535 q 218 567 218 553 q 227 589 218 579 q 250 600 237 600 q 267 594 259 600 q 278 580 274 589 "
    },
    "Ã²": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 431 731 q 413 718 422 723 q 393 710 405 713 l 160 966 l 174 994 q 194 998 181 996 q 221 1003 207 1000 q 249 1007 235 1006 q 271 1010 263 1009 l 431 731 "
    },
    "^": {
      "x_min": 66.453125,
      "x_max": 602.796875,
      "ha": 670,
      "o": "m 602 430 q 570 404 586 415 q 532 383 554 393 l 507 401 l 319 891 l 153 430 q 136 416 146 423 q 117 403 127 409 q 98 391 107 396 q 81 383 88 386 l 66 401 l 280 991 q 300 1008 288 999 q 323 1024 311 1016 q 347 1039 335 1032 q 368 1051 359 1046 l 602 430 "
    },
    "âˆ™": {
      "x_min": 33,
      "x_max": 166,
      "ha": 199,
      "o": "m 166 488 q 160 452 166 469 q 143 423 154 435 q 119 403 133 410 q 88 396 105 396 q 45 412 58 396 q 33 460 33 429 q 38 496 33 479 q 55 526 44 513 q 79 545 66 538 q 109 553 93 553 q 166 488 166 553 "
    },
    "Ç˜": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 467 70 q 369 3 398 18 q 316 -15 340 -10 q 271 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 43 q 113 189 113 125 l 113 481 q 111 532 113 514 q 100 559 109 550 q 75 571 92 568 q 27 578 57 575 l 27 605 q 72 612 51 608 q 112 622 93 616 q 152 634 132 627 q 194 650 172 641 l 213 624 l 213 226 q 219 147 213 178 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 482 q 505 531 508 512 q 492 559 502 549 q 464 573 482 569 q 414 578 446 578 l 414 605 q 507 624 463 611 q 588 650 551 637 l 608 624 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 562 854 q 556 818 562 835 q 539 789 550 801 q 515 769 529 776 q 484 762 501 762 q 441 778 454 762 q 429 826 429 795 q 434 862 429 845 q 451 892 440 879 q 475 911 462 904 q 505 919 489 919 q 562 854 562 919 m 296 854 q 290 818 296 835 q 273 789 284 801 q 249 769 263 776 q 218 762 235 762 q 175 778 188 762 q 163 826 163 795 q 168 862 163 845 q 185 892 174 879 q 209 911 196 904 q 239 919 223 919 q 296 854 296 919 m 333 953 q 312 961 321 955 q 296 972 303 966 l 451 1253 q 472 1250 459 1252 q 499 1247 485 1249 q 527 1242 513 1245 q 547 1237 540 1239 l 563 1211 l 333 953 "
    },
    "á¹‰": {
      "x_min": 37,
      "x_max": 732,
      "ha": 752,
      "o": "m 446 0 l 446 29 q 516 51 493 42 q 539 70 539 61 l 539 429 q 534 496 539 470 q 519 537 529 522 q 493 557 509 552 q 453 563 477 563 q 408 552 433 563 q 353 520 382 542 q 293 461 325 497 q 230 372 261 425 l 230 70 q 254 49 230 60 q 323 29 279 38 l 323 0 l 37 0 l 37 29 q 105 50 80 40 q 130 70 130 59 l 130 482 q 128 524 130 508 q 117 548 126 539 q 89 561 108 557 q 37 569 70 565 l 37 596 q 120 618 82 604 q 196 651 159 632 l 223 627 l 223 458 q 291 539 256 504 q 362 599 327 575 q 432 637 398 624 q 492 651 466 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 70 q 659 52 639 61 q 732 29 679 42 l 732 0 l 446 0 m 609 -155 q 604 -172 607 -162 q 597 -192 601 -182 q 589 -211 593 -201 q 583 -227 585 -220 l 174 -227 l 159 -210 q 164 -193 160 -202 q 171 -173 167 -183 q 179 -154 175 -164 q 186 -139 183 -145 l 594 -139 l 609 -155 "
    },
    "Å«": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 466 70 q 369 4 398 18 q 316 -15 340 -10 q 270 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 532 113 514 q 100 559 109 550 q 75 572 92 569 q 27 579 57 575 l 27 606 q 72 613 51 609 q 112 622 93 617 q 152 635 132 628 q 194 651 172 642 l 213 624 l 213 226 q 219 147 213 179 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 482 q 505 531 508 513 q 492 560 502 550 q 463 574 482 570 q 414 578 445 578 l 414 606 q 506 625 462 612 q 588 651 550 638 l 608 624 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 593 868 q 588 851 591 861 q 581 831 585 841 q 573 812 577 822 q 567 797 570 803 l 159 797 l 144 813 q 148 830 145 821 q 156 850 152 840 q 164 869 160 859 q 171 885 168 878 l 578 885 l 593 868 "
    },
    "Ë†": {
      "x_min": 32.5625,
      "x_max": 453.265625,
      "ha": 487,
      "o": "m 453 740 q 440 722 445 728 q 424 710 435 715 l 242 892 l 62 710 q 53 715 57 713 q 46 722 49 718 q 40 730 43 725 q 32 740 37 734 l 208 998 l 279 998 l 453 740 "
    },
    "áº„": {
      "x_min": 13,
      "x_max": 1131,
      "ha": 1157,
      "o": "m 1131 824 q 1082 813 1101 818 q 1051 802 1063 807 q 1035 792 1040 797 q 1029 782 1030 788 l 877 39 q 863 14 874 25 q 838 -2 852 4 q 809 -14 824 -9 q 785 -20 795 -18 l 568 639 l 377 39 q 361 15 372 25 q 335 -2 350 4 q 306 -13 321 -9 q 277 -20 290 -18 l 104 777 q 80 805 101 794 q 13 824 60 816 l 13 854 l 298 854 l 298 824 q 246 816 265 821 q 218 805 228 811 q 207 791 209 799 q 207 777 206 784 l 339 169 l 554 854 l 591 854 l 827 169 l 951 782 q 945 797 952 790 q 924 807 938 803 q 890 816 910 812 q 849 824 871 820 l 849 854 l 1131 854 l 1131 824 m 771 1044 q 765 1008 771 1025 q 748 979 759 991 q 724 959 738 966 q 693 952 710 952 q 650 968 663 952 q 638 1016 638 985 q 643 1052 638 1035 q 660 1082 649 1069 q 684 1101 671 1094 q 714 1109 698 1109 q 771 1044 771 1109 m 505 1044 q 499 1008 505 1025 q 482 979 493 991 q 458 959 472 966 q 427 952 444 952 q 384 968 397 952 q 372 1016 372 985 q 377 1052 372 1035 q 394 1082 383 1069 q 418 1101 405 1094 q 448 1109 432 1109 q 505 1044 505 1109 "
    },
    "È·": {
      "x_min": -146,
      "x_max": 236,
      "ha": 352,
      "o": "m 236 67 q 226 -65 236 -10 q 199 -158 216 -119 q 159 -223 182 -196 q 109 -271 136 -249 q 71 -297 91 -285 q 32 -319 51 -310 q -5 -333 12 -328 q -35 -339 -23 -339 q -75 -333 -56 -339 q -111 -321 -95 -328 q -136 -305 -126 -313 q -146 -292 -146 -297 q -136 -275 -146 -285 q -113 -253 -126 -264 q -85 -231 -100 -241 q -61 -216 -71 -221 q -18 -241 -40 -235 q 28 -247 4 -247 q 66 -234 48 -247 q 100 -190 85 -221 q 124 -106 115 -159 q 134 29 134 -52 l 134 454 q 134 510 134 489 q 125 542 134 531 q 96 560 116 554 q 42 569 77 565 l 42 596 q 92 607 70 601 q 133 620 114 613 q 171 634 153 626 q 211 651 190 642 l 236 651 l 236 67 "
    },
    "Ä": {
      "x_min": 53,
      "x_max": 558.40625,
      "ha": 595,
      "o": "m 558 129 q 482 47 515 76 q 421 2 449 17 q 366 -16 393 -12 q 309 -20 340 -20 q 214 2 260 -20 q 132 65 168 24 q 74 166 96 106 q 53 301 53 226 q 79 438 53 374 q 151 549 105 502 q 258 624 196 596 q 392 651 320 651 q 437 647 414 651 q 480 637 459 644 q 519 620 501 629 q 548 598 537 610 q 546 574 549 589 q 537 543 543 560 q 523 511 531 526 q 510 486 516 496 l 485 493 q 468 519 480 505 q 437 547 456 534 q 392 568 418 560 q 332 577 366 577 q 264 561 297 577 q 207 513 232 545 q 167 433 182 481 q 153 322 153 386 q 169 217 153 264 q 215 137 186 170 q 280 85 243 103 q 357 68 317 68 q 391 69 375 68 q 426 80 407 71 q 470 106 445 89 q 532 156 495 124 l 558 129 m 361 727 l 290 727 l 115 969 q 122 980 119 975 q 129 988 126 985 q 135 994 132 992 q 145 1000 139 997 l 327 830 l 507 1000 q 523 988 518 994 q 535 969 527 982 l 361 727 "
    },
    "â€™": {
      "x_min": 58.59375,
      "x_max": 251,
      "ha": 312,
      "o": "m 251 878 q 242 821 251 852 q 215 760 233 790 q 173 701 198 729 q 117 652 149 673 l 87 676 q 111 709 101 690 q 129 748 122 729 q 139 785 136 767 q 143 816 143 803 q 125 870 143 847 q 69 893 107 893 l 58 924 q 83 943 64 932 q 126 963 103 953 q 171 979 149 973 q 205 986 193 986 q 239 941 227 964 q 251 878 251 917 "
    },
    "-": {
      "x_min": 40.546875,
      "x_max": 418.03125,
      "ha": 459,
      "o": "m 418 370 q 408 338 414 355 q 397 309 402 320 l 57 309 l 40 325 q 49 356 43 340 q 61 387 55 373 l 401 387 l 418 370 "
    },
    "Q": {
      "x_min": 47,
      "x_max": 852.703125,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 852 -79 q 820 -132 836 -109 q 787 -172 803 -156 q 757 -198 771 -189 q 733 -207 743 -207 q 651 -186 691 -207 q 571 -136 611 -165 q 492 -74 531 -106 q 412 -17 452 -41 q 382 -20 397 -20 q 247 14 309 -20 q 141 108 185 49 q 71 246 96 167 q 47 415 47 325 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 678 q 757 438 757 521 q 738 299 757 367 q 686 176 719 232 q 607 76 653 119 q 509 8 562 32 q 569 -28 539 -7 q 629 -66 600 -48 q 685 -96 657 -84 q 737 -109 712 -109 q 754 -106 745 -109 q 774 -97 762 -103 q 800 -80 785 -91 q 834 -55 814 -70 l 852 -79 "
    },
    "Ä›": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 290 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 374 727 l 303 727 l 128 969 q 136 980 133 975 q 142 988 139 985 q 149 994 145 992 q 158 1000 152 997 l 340 830 l 519 1000 q 535 988 531 994 q 548 969 540 982 l 374 727 "
    },
    "Å“": {
      "x_min": 53,
      "x_max": 1026,
      "ha": 1079,
      "o": "m 786 581 q 727 567 755 581 q 678 528 699 553 q 642 469 656 503 q 621 394 627 435 l 890 394 q 910 399 904 394 q 916 418 916 404 q 911 464 916 436 q 892 517 906 492 q 852 562 878 543 q 786 581 827 581 m 516 309 q 499 411 516 363 q 453 496 482 460 q 389 555 425 533 q 315 577 353 577 q 250 557 279 577 q 199 503 220 537 q 166 422 178 469 q 155 321 155 375 q 172 219 155 267 q 219 133 190 170 q 283 75 248 97 q 355 54 319 54 q 420 73 391 54 q 471 127 450 93 q 504 208 492 161 q 516 309 516 254 m 1026 378 q 992 352 1012 364 q 951 332 972 340 l 618 332 q 631 231 618 279 q 669 148 645 184 q 728 91 693 112 q 806 70 763 70 q 843 72 825 70 q 882 83 861 74 q 931 111 904 93 q 996 160 959 129 q 1011 147 1004 156 q 1020 132 1017 138 q 943 53 976 83 q 880 6 909 22 q 824 -14 851 -9 q 766 -20 797 -20 q 702 -10 733 -20 q 643 18 671 0 q 592 64 615 37 q 552 125 568 90 q 504 66 531 93 q 447 20 477 39 q 383 -9 416 1 q 315 -20 350 -20 q 210 4 259 -20 q 127 71 162 28 q 72 173 92 114 q 53 301 53 232 q 64 389 53 346 q 95 472 75 433 q 144 544 116 511 q 205 600 172 576 q 277 638 239 624 q 355 651 315 651 q 424 641 391 651 q 485 612 457 631 q 537 568 514 594 q 577 509 560 542 q 615 557 594 535 q 661 597 636 579 q 694 617 675 607 q 733 635 713 627 q 771 647 752 643 q 805 652 790 652 q 878 639 846 652 q 935 606 911 627 q 977 558 960 586 q 1005 500 994 531 q 1021 438 1016 470 q 1026 378 1026 406 "
    },
    "á»˜": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 476 -189 q 470 -225 476 -208 q 453 -254 464 -242 q 429 -274 443 -267 q 398 -282 415 -282 q 355 -265 368 -282 q 343 -217 343 -248 q 348 -181 343 -198 q 365 -151 354 -164 q 389 -132 376 -139 q 419 -125 403 -125 q 476 -189 476 -125 m 619 957 q 606 938 611 944 q 590 927 602 932 l 408 1068 l 228 927 q 219 932 223 929 q 212 938 215 935 q 206 946 209 942 q 198 957 203 951 l 374 1167 l 445 1167 l 619 957 "
    },
    "á¹©": {
      "x_min": 63,
      "x_max": 465,
      "ha": 525,
      "o": "m 465 192 q 451 109 465 144 q 417 51 438 75 q 369 13 395 28 q 318 -7 343 0 q 271 -17 293 -15 q 236 -20 248 -20 q 159 -7 204 -20 q 70 29 114 4 q 64 51 65 31 q 63 97 63 71 q 66 150 63 123 q 74 192 69 176 l 104 184 q 117 131 104 156 q 153 87 130 106 q 205 58 175 69 q 269 48 234 48 q 313 55 293 48 q 348 77 333 63 q 370 111 362 91 q 379 154 379 130 q 364 202 379 181 q 326 240 350 223 q 273 273 303 258 q 213 304 244 288 q 159 335 185 319 q 113 371 133 351 q 81 417 93 392 q 69 474 69 442 q 85 549 69 516 q 129 604 102 582 q 192 639 157 627 q 266 651 228 651 q 311 647 287 651 q 361 636 336 643 q 406 620 385 629 q 441 598 427 611 q 442 580 445 594 q 432 549 439 566 q 419 517 426 532 q 408 497 412 502 l 382 502 q 315 570 350 551 q 248 590 281 590 q 209 582 226 590 q 179 563 192 575 q 161 537 167 551 q 155 508 155 522 q 167 469 155 487 q 200 437 180 452 q 247 408 221 421 q 302 379 274 394 q 359 347 330 364 q 412 308 389 330 q 450 258 435 286 q 465 192 465 230 m 330 -189 q 324 -225 330 -208 q 307 -254 318 -242 q 283 -274 297 -267 q 252 -282 269 -282 q 209 -265 222 -282 q 197 -217 197 -248 q 202 -181 197 -198 q 219 -151 208 -164 q 243 -132 230 -139 q 273 -125 257 -125 q 330 -189 330 -125 m 330 854 q 324 818 330 835 q 307 789 318 801 q 283 769 297 776 q 252 762 269 762 q 209 778 222 762 q 197 826 197 795 q 202 862 197 845 q 219 892 208 879 q 243 911 230 904 q 273 919 257 919 q 330 854 330 919 "
    },
    "áº¬": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 387 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 839 q 390 869 367 856 q 430 893 413 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 465 -189 q 459 -225 465 -208 q 442 -254 453 -242 q 418 -274 432 -267 q 387 -282 404 -282 q 344 -265 357 -282 q 332 -217 332 -248 q 337 -181 332 -198 q 354 -151 343 -164 q 378 -132 365 -139 q 408 -125 392 -125 q 465 -189 465 -125 m 608 957 q 595 938 600 944 q 579 927 590 932 l 397 1068 l 216 927 q 208 932 211 929 q 201 938 204 935 q 195 946 198 942 q 187 957 191 951 l 363 1167 l 434 1167 l 608 957 "
    },
    "ï„°": {
      "x_min": 29.859375,
      "x_max": 226,
      "ha": 226,
      "o": "m 226 0 l 226 -199 l 200 -199 l 200 -26 l 29 -26 l 29 0 l 226 0 "
    },
    "#": {
      "x_min": 57.828125,
      "x_max": 652.640625,
      "ha": 639,
      "o": "m 265 400 l 390 400 l 439 574 l 314 574 l 265 400 m 535 652 l 636 652 l 652 635 q 643 602 649 620 q 632 574 637 585 l 513 574 l 465 400 l 565 400 l 580 381 q 571 350 576 368 q 561 322 566 333 l 443 322 l 385 115 q 357 98 373 105 q 324 86 341 92 l 307 99 l 368 322 l 243 322 l 185 115 q 158 98 173 105 q 125 86 142 92 l 108 99 l 170 322 l 72 322 l 57 338 q 66 369 61 352 q 77 400 71 386 l 192 400 l 240 574 l 144 574 l 128 590 q 137 621 131 605 q 148 652 143 638 l 262 652 l 317 848 q 346 864 332 859 q 376 876 361 870 l 395 861 l 336 652 l 461 652 l 516 848 q 545 864 531 859 q 576 876 560 870 l 594 861 l 535 652 "
    },
    "Ç¦": {
      "x_min": 47,
      "x_max": 761,
      "ha": 794,
      "o": "m 693 805 q 691 792 697 802 q 674 770 685 782 q 651 746 664 758 q 629 726 638 734 l 607 730 q 562 765 585 751 q 513 786 538 779 q 459 797 487 794 q 397 800 430 800 q 356 792 381 800 q 303 767 331 784 q 249 722 276 750 q 199 652 221 693 q 163 557 177 611 q 150 432 150 502 q 174 267 150 337 q 239 152 199 198 q 328 83 278 106 q 429 61 378 61 q 518 70 477 61 q 592 99 559 80 l 592 329 q 585 343 592 336 q 563 357 579 350 q 521 371 548 364 q 455 384 495 378 l 455 414 l 761 414 l 761 384 q 707 360 723 375 q 692 329 692 344 l 692 104 q 603 38 642 63 q 532 1 565 14 q 470 -15 499 -11 q 411 -20 441 -20 q 279 5 344 -20 q 162 82 213 30 q 78 212 110 134 q 47 394 47 289 q 81 596 47 507 q 177 747 116 685 q 318 842 237 809 q 490 875 399 875 q 538 870 512 875 q 592 856 564 865 q 645 834 619 847 q 693 805 671 822 m 461 944 l 390 944 l 215 1152 q 223 1163 219 1158 q 229 1171 226 1168 q 236 1177 232 1175 q 245 1184 239 1180 l 427 1040 l 607 1184 q 623 1171 618 1177 q 636 1152 628 1165 l 461 944 "
    },
    "É‚": {
      "x_min": 47,
      "x_max": 492,
      "ha": 545,
      "o": "m 100 0 l 100 29 q 213 70 213 49 l 213 180 q 226 247 213 219 q 259 296 239 275 q 302 335 279 318 q 345 370 326 352 q 378 407 365 387 q 392 454 392 427 q 380 510 392 485 q 350 552 369 535 q 307 578 331 569 q 255 587 282 587 q 213 578 231 587 q 181 554 194 569 q 162 519 169 539 q 156 479 156 500 q 158 459 156 469 q 163 439 160 449 q 119 422 142 428 q 66 414 96 417 l 48 434 q 47 445 47 439 l 47 456 q 69 534 47 499 q 129 596 92 570 q 212 636 165 622 q 307 651 259 651 q 382 638 348 651 q 441 602 416 625 q 478 550 465 580 q 492 485 492 519 q 478 422 492 448 q 445 377 465 397 q 402 340 426 357 q 359 305 378 323 q 326 263 339 286 q 313 207 313 240 l 313 70 q 341 49 313 60 q 426 29 370 39 l 426 0 l 100 0 "
    },
    "êžŒ": {
      "x_min": 117,
      "x_max": 219,
      "ha": 333,
      "o": "m 219 443 q 198 437 210 440 q 172 433 186 435 q 144 430 158 431 q 117 429 129 429 l 117 963 q 134 973 122 967 q 159 986 146 980 q 184 997 172 992 q 201 1003 195 1002 l 219 985 l 219 443 "
    },
    "â± ": {
      "x_min": 29.015625,
      "x_max": 627.09375,
      "ha": 648,
      "o": "m 627 165 q 613 63 621 106 q 599 0 605 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 60 l 130 296 l 43 296 l 29 312 q 36 334 32 323 q 43 357 39 346 l 130 357 l 130 432 l 43 432 l 29 448 q 36 470 32 459 q 43 493 39 482 l 130 493 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 493 l 395 493 l 411 477 l 395 432 l 236 432 l 236 357 l 395 357 l 411 341 l 395 296 l 236 296 l 236 111 q 241 89 236 98 q 261 74 247 80 q 299 64 275 67 q 359 61 322 61 l 454 61 q 505 64 484 61 q 542 79 526 67 q 571 114 558 91 q 598 177 584 137 l 627 165 "
    },
    "ÆŸ": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 651 478 q 624 598 645 541 q 571 700 604 656 q 495 770 539 744 q 401 797 452 797 q 300 774 345 797 q 223 710 255 752 q 172 610 191 669 q 153 478 153 551 l 651 478 m 151 417 q 172 279 151 343 q 227 165 193 214 q 306 89 261 117 q 401 61 351 61 q 502 83 456 61 q 581 151 548 106 q 633 263 614 196 q 653 417 653 329 l 151 417 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 "
    },
    "Ã…": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 838 q 390 869 367 855 q 430 893 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 462 1060 q 457 1091 462 1076 q 446 1117 453 1106 q 428 1135 439 1128 q 404 1142 417 1142 q 377 1136 390 1142 q 355 1120 365 1130 q 340 1094 346 1109 q 334 1061 334 1079 q 338 1031 334 1045 q 349 1005 342 1016 q 367 987 356 994 q 391 981 378 981 q 417 986 404 981 q 440 1001 430 991 q 456 1025 450 1011 q 462 1060 462 1040 m 522 1088 q 509 1024 522 1053 q 475 974 496 995 q 428 940 454 952 q 377 929 402 929 q 335 937 354 929 q 302 959 316 945 q 281 992 289 973 q 274 1033 274 1011 q 287 1097 274 1068 q 320 1148 300 1127 q 366 1182 341 1170 q 418 1194 392 1194 q 459 1185 440 1194 q 492 1162 478 1177 q 514 1128 506 1147 q 522 1088 522 1109 "
    },
    "Èª": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 609 1045 q 603 1009 609 1026 q 586 980 597 992 q 562 960 576 967 q 531 953 548 953 q 488 969 501 953 q 476 1017 476 986 q 481 1053 476 1036 q 498 1083 487 1070 q 522 1102 509 1095 q 552 1110 536 1110 q 609 1045 609 1110 m 343 1045 q 337 1009 343 1026 q 320 980 331 992 q 296 960 310 967 q 265 953 282 953 q 222 969 235 953 q 210 1017 210 986 q 215 1053 210 1036 q 232 1083 221 1070 q 256 1102 243 1095 q 286 1110 270 1110 q 343 1045 343 1110 m 640 1275 q 636 1258 639 1268 q 628 1238 632 1248 q 621 1219 624 1229 q 614 1204 617 1210 l 206 1204 l 191 1220 q 196 1237 192 1228 q 203 1257 199 1247 q 211 1276 207 1266 q 218 1292 215 1285 l 626 1292 l 640 1275 "
    },
    "ÇŽ": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 354 726 l 283 726 l 108 968 q 116 979 112 974 q 122 987 119 984 q 129 993 125 991 q 138 999 132 996 l 320 829 l 500 999 q 516 987 512 993 q 529 968 521 981 l 354 726 "
    },
    "Â¸": {
      "x_min": 51.40625,
      "x_max": 271,
      "ha": 312,
      "o": "m 271 -155 q 259 -203 271 -180 q 223 -245 247 -226 q 159 -278 198 -264 q 66 -300 120 -292 l 51 -267 q 110 -252 85 -261 q 149 -232 134 -243 q 172 -209 165 -221 q 179 -186 179 -197 q 161 -154 179 -163 q 103 -141 144 -145 q 109 -122 104 -139 q 124 -77 114 -108 q 151 9 133 -47 l 203 9 l 177 -70 q 212 -80 195 -74 q 242 -97 229 -87 q 263 -121 255 -107 q 271 -155 271 -136 "
    },
    "=": {
      "x_min": 40.546875,
      "x_max": 556.9375,
      "ha": 597,
      "o": "m 556 298 q 547 269 553 284 q 537 241 541 253 l 57 241 l 40 259 q 49 287 43 272 q 61 315 55 302 l 540 315 l 556 298 m 556 502 q 547 472 553 488 q 537 444 541 455 l 57 444 l 40 461 q 49 490 43 474 q 61 519 55 506 l 540 519 l 556 502 "
    },
    "áº¡": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 67 q 318 88 273 67 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 233 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 70 200 74 q 233 67 225 67 m 615 55 q 522 -2 559 15 q 466 -20 485 -20 q 429 10 444 -20 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -13 267 -6 q 194 -20 215 -20 q 145 -11 170 -20 q 99 13 120 -3 q 66 59 79 31 q 53 126 53 87 q 70 212 53 177 q 112 272 88 246 q 149 302 128 288 q 202 330 169 317 q 286 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 280 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 165 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 372 -189 q 366 -225 372 -208 q 349 -254 360 -242 q 325 -274 339 -267 q 294 -282 311 -282 q 251 -265 264 -282 q 239 -217 239 -248 q 244 -181 239 -198 q 261 -151 250 -164 q 285 -132 272 -139 q 315 -125 299 -125 q 372 -189 372 -125 "
    },
    "Ç•": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 615 816 l 568 855 l 847 855 l 847 825 m 649 1045 q 643 1009 649 1026 q 626 980 637 992 q 602 960 616 967 q 571 953 588 953 q 528 969 541 953 q 516 1017 516 986 q 521 1053 516 1036 q 538 1083 527 1070 q 562 1102 549 1095 q 592 1110 576 1110 q 649 1045 649 1110 m 383 1045 q 377 1009 383 1026 q 360 980 371 992 q 336 960 350 967 q 305 953 322 953 q 262 969 275 953 q 250 1017 250 986 q 255 1053 250 1036 q 272 1083 261 1070 q 296 1102 283 1095 q 326 1110 310 1110 q 383 1045 383 1110 m 681 1275 q 676 1258 680 1268 q 669 1238 673 1248 q 661 1219 665 1229 q 655 1204 657 1210 l 246 1204 l 232 1220 q 236 1237 233 1228 q 243 1257 239 1247 q 251 1276 247 1266 q 258 1292 255 1285 l 666 1292 l 681 1275 "
    },
    "Ãº": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 466 70 q 369 4 398 18 q 316 -14 340 -9 q 270 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 533 113 515 q 100 560 109 551 q 75 572 92 569 q 27 579 57 576 l 27 606 q 72 613 51 609 q 112 623 93 617 q 152 635 132 628 q 194 651 172 642 l 213 625 l 213 226 q 219 147 213 179 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 483 q 505 532 508 513 q 492 560 502 550 q 463 574 482 570 q 414 579 445 579 l 414 606 q 506 625 462 612 q 588 651 550 638 l 608 625 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 333 710 q 312 718 321 712 q 295 729 303 723 l 450 1010 q 471 1007 458 1009 q 499 1004 484 1006 q 526 999 513 1002 q 547 994 540 996 l 562 968 l 333 710 "
    },
    "Ëš": {
      "x_min": 47,
      "x_max": 294,
      "ha": 341,
      "o": "m 234 843 q 230 874 234 859 q 218 900 226 889 q 200 918 211 911 q 177 925 190 925 q 150 919 163 925 q 128 903 138 913 q 112 877 118 892 q 107 844 107 862 q 110 814 107 828 q 122 788 114 799 q 140 770 129 777 q 163 764 150 764 q 189 769 177 764 q 212 784 202 774 q 228 808 222 794 q 234 843 234 823 m 294 870 q 281 807 294 836 q 247 757 268 778 q 200 723 226 735 q 149 712 175 712 q 108 720 127 712 q 75 742 89 728 q 54 775 62 756 q 47 816 47 794 q 59 880 47 851 q 93 931 72 910 q 139 965 114 953 q 191 977 165 977 q 231 968 212 977 q 264 945 250 960 q 286 911 278 930 q 294 870 294 892 "
    },
    "Â¯": {
      "x_min": 110.984375,
      "x_max": 560.265625,
      "ha": 671,
      "o": "m 560 868 q 555 851 558 861 q 548 831 552 841 q 540 812 544 822 q 534 797 537 803 l 125 797 l 110 813 q 115 830 112 821 q 122 850 118 840 q 130 869 126 859 q 138 885 134 878 l 545 885 l 560 868 "
    },
    "u": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 466 70 q 369 4 398 18 q 316 -14 340 -9 q 270 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 533 113 515 q 100 560 109 551 q 75 572 92 569 q 27 579 57 576 l 27 606 q 72 613 51 609 q 112 623 93 617 q 152 635 132 628 q 194 651 172 642 l 213 625 l 213 226 q 219 147 213 179 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 483 q 505 532 508 513 q 492 560 502 550 q 463 574 482 570 q 414 579 445 579 l 414 606 q 506 625 462 612 q 588 651 550 638 l 608 625 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 "
    },
    "á¹›": {
      "x_min": 37,
      "x_max": 517.765625,
      "ha": 540,
      "o": "m 512 626 q 517 602 517 621 q 512 557 517 582 q 499 504 507 531 q 483 459 491 477 l 455 459 q 443 504 451 485 q 426 534 436 523 q 405 550 417 545 q 381 556 394 556 q 347 543 366 556 q 307 504 327 530 q 266 435 286 477 q 230 336 245 394 l 230 70 q 254 49 230 60 q 342 29 279 38 l 342 0 l 37 0 l 37 29 q 105 49 80 39 q 130 70 130 59 l 130 465 q 128 502 130 487 q 125 527 127 518 q 120 542 123 537 q 114 551 117 547 q 104 560 109 556 q 89 565 98 563 q 68 568 80 568 q 37 568 55 568 l 37 597 q 121 620 79 608 q 196 651 163 632 l 224 627 l 224 473 q 266 543 246 510 q 311 599 287 575 q 362 637 335 623 q 417 651 388 651 q 463 645 438 651 q 512 626 487 640 m 246 -189 q 240 -225 246 -208 q 223 -254 234 -242 q 199 -274 213 -267 q 168 -282 185 -282 q 125 -265 138 -282 q 113 -217 113 -248 q 118 -181 113 -198 q 135 -151 124 -164 q 159 -132 146 -139 q 189 -125 173 -125 q 246 -189 246 -125 "
    },
    "ï†—": {
      "x_min": 0,
      "x_max": 267.171875,
      "ha": 267,
      "o": "m 37 709 q 16 717 25 712 q 0 728 7 722 l 155 1009 q 176 1007 163 1009 q 203 1003 189 1005 q 231 998 217 1001 q 251 994 244 996 l 267 967 l 37 709 "
    },
    "áº»": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 291 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 450 904 q 437 870 450 885 q 409 842 425 855 q 376 817 392 829 q 354 793 361 805 q 353 766 346 780 q 387 736 359 753 q 367 728 377 729 q 346 725 356 726 q 302 753 316 740 q 285 776 288 765 q 288 796 282 787 q 306 815 294 806 q 331 833 317 824 q 357 851 345 842 q 377 871 369 861 q 386 894 386 882 q 373 933 386 920 q 341 946 360 946 q 322 942 331 946 q 308 932 314 938 q 299 918 302 926 q 296 903 296 911 q 297 896 296 900 q 301 889 299 893 q 287 885 296 887 q 269 880 279 882 q 251 877 260 878 q 235 874 241 875 l 228 882 l 228 891 q 240 923 228 907 q 272 951 253 938 q 316 971 292 963 q 363 979 340 979 q 427 958 404 979 q 450 904 450 938 "
    },
    "á¹Ž": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 636 1103 q 607 1048 624 1075 q 570 999 590 1020 q 526 963 550 977 q 476 950 502 950 q 432 961 453 950 q 391 988 411 973 q 351 1015 371 1003 q 313 1027 332 1027 q 264 1004 286 1027 q 217 944 242 982 l 182 958 q 210 1013 194 986 q 247 1062 227 1041 q 292 1098 268 1084 q 341 1112 315 1112 q 389 1100 366 1112 q 432 1073 412 1088 q 470 1046 452 1058 q 503 1035 487 1035 q 553 1056 529 1035 q 600 1118 577 1078 l 636 1103 m 609 1262 q 603 1226 609 1243 q 586 1197 597 1209 q 562 1177 576 1184 q 531 1170 548 1170 q 488 1186 501 1170 q 476 1234 476 1203 q 481 1270 476 1253 q 498 1300 487 1287 q 522 1319 509 1312 q 552 1327 536 1327 q 609 1262 609 1327 m 343 1262 q 337 1226 343 1243 q 320 1197 331 1209 q 296 1177 310 1184 q 265 1170 282 1170 q 222 1186 235 1170 q 210 1234 210 1203 q 215 1270 210 1253 q 232 1300 221 1287 q 256 1319 243 1312 q 286 1327 270 1327 q 343 1262 343 1327 "
    },
    "áº—": {
      "x_min": 13.390625,
      "x_max": 449.609375,
      "ha": 469,
      "o": "m 449 79 q 385 36 417 55 q 324 4 353 17 q 270 -14 295 -7 q 228 -20 246 -20 q 184 -11 205 -20 q 147 17 163 -2 q 122 69 131 37 q 113 149 113 102 l 113 566 l 27 566 l 13 584 l 65 630 l 113 630 l 113 797 l 192 867 l 213 850 l 213 630 l 430 630 l 449 610 q 434 590 443 601 q 416 569 425 579 q 398 552 407 560 q 382 542 389 544 q 333 558 366 550 q 247 566 301 566 l 213 566 l 213 208 q 216 140 213 167 q 228 97 220 113 q 251 74 237 81 q 284 68 264 68 q 343 77 308 68 q 430 114 378 86 l 449 79 m 431 1037 q 425 1001 431 1018 q 408 972 419 984 q 384 952 398 959 q 353 945 370 945 q 310 961 323 945 q 298 1009 298 978 q 303 1045 298 1028 q 320 1075 309 1062 q 344 1094 331 1087 q 374 1102 358 1102 q 431 1037 431 1102 m 165 1037 q 159 1001 165 1018 q 142 972 153 984 q 118 952 132 959 q 87 945 104 945 q 44 961 57 945 q 32 1009 32 978 q 37 1045 32 1028 q 54 1075 43 1062 q 78 1094 65 1087 q 108 1102 92 1102 q 165 1037 165 1102 "
    },
    "áºµ": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 535 922 q 490 829 515 867 q 437 768 465 792 q 380 735 410 745 q 320 725 350 725 q 257 735 287 725 q 198 768 226 745 q 147 829 171 792 q 102 922 122 867 q 118 940 110 934 q 138 952 126 947 q 176 887 154 913 q 221 845 197 861 q 270 822 245 829 q 317 815 295 815 q 366 822 341 815 q 416 845 392 829 q 462 887 440 861 q 500 952 484 913 q 519 940 511 947 q 535 922 527 934 m 545 1129 q 517 1074 533 1101 q 480 1025 500 1047 q 436 990 460 1004 q 386 977 412 977 q 341 988 362 977 q 300 1015 320 1000 q 261 1041 281 1029 q 222 1053 241 1053 q 173 1031 195 1053 q 127 971 152 1009 l 92 985 q 120 1040 104 1012 q 157 1089 136 1067 q 201 1124 177 1110 q 251 1138 225 1138 q 298 1126 276 1138 q 341 1100 321 1114 q 379 1073 361 1085 q 413 1062 397 1062 q 463 1083 438 1062 q 510 1144 487 1105 l 545 1129 "
    },
    "á»¯": {
      "x_min": 27.265625,
      "x_max": 834,
      "ha": 834,
      "o": "m 834 706 q 823 661 834 687 q 787 607 813 636 q 718 548 762 578 q 608 490 674 518 l 608 172 q 611 103 608 130 q 623 70 614 77 q 650 68 631 65 q 704 86 670 71 l 710 55 q 662 25 685 39 q 620 0 639 11 q 586 -15 600 -9 q 563 -21 571 -21 q 524 10 539 -21 q 510 112 510 42 q 432 44 466 70 q 369 3 398 17 q 316 -15 340 -10 q 270 -21 292 -21 q 211 -12 239 -21 q 160 20 182 -3 q 125 84 138 43 q 113 189 113 125 l 113 482 q 111 532 113 514 q 100 559 109 550 q 75 572 92 568 q 27 578 57 575 l 27 606 q 72 612 51 608 q 112 622 93 616 q 152 634 132 627 q 194 650 172 642 l 213 624 l 213 226 q 219 147 213 178 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 482 q 505 531 508 513 q 492 559 502 549 q 463 574 482 570 q 415 578 445 578 l 415 606 q 506 625 462 612 q 588 650 550 637 l 608 624 l 608 538 q 659 568 638 552 q 693 598 680 583 q 713 627 707 613 q 720 651 720 640 q 710 688 720 671 q 681 722 700 706 l 799 776 q 824 743 814 761 q 834 706 834 725 m 588 912 q 560 857 576 884 q 523 808 543 830 q 478 773 502 787 q 429 760 455 760 q 384 771 405 760 q 344 798 364 783 q 304 824 324 812 q 265 836 285 836 q 217 814 239 836 q 171 754 195 792 l 135 768 q 163 823 147 795 q 200 872 180 850 q 244 907 221 893 q 294 921 268 921 q 342 909 319 921 q 384 883 364 897 q 422 856 404 868 q 456 845 440 845 q 505 866 481 845 q 552 927 530 888 l 588 912 "
    },
    "É—": {
      "x_min": 53,
      "x_max": 859,
      "ha": 708,
      "o": "m 859 964 q 849 948 859 959 q 826 927 839 938 q 798 906 813 915 q 774 891 784 896 q 746 929 760 914 q 720 951 733 943 q 695 962 707 959 q 675 966 684 966 q 639 955 654 966 q 614 919 624 944 q 599 852 604 894 q 595 749 595 811 l 595 172 q 595 131 595 148 q 597 103 595 114 q 601 84 599 91 q 608 72 604 76 q 632 67 614 64 q 689 86 649 70 l 698 57 q 643 21 667 36 q 601 -2 619 6 q 570 -16 583 -12 q 548 -21 558 -21 q 512 9 526 -21 q 498 114 498 40 q 446 58 471 83 q 395 16 421 33 q 339 -10 368 -1 q 276 -20 310 -20 q 199 2 239 -20 q 127 65 160 24 q 73 166 94 106 q 53 301 53 226 q 63 381 53 339 q 92 461 73 422 q 140 534 112 499 q 202 595 168 568 q 278 636 237 621 q 366 651 319 651 q 428 643 397 651 q 495 608 459 635 l 495 712 q 500 811 495 770 q 518 885 506 853 q 549 941 530 916 q 593 988 567 965 q 664 1035 627 1019 q 727 1051 701 1051 q 780 1039 755 1051 q 822 1014 804 1028 q 849 985 839 999 q 859 964 859 970 m 495 177 l 495 495 q 433 555 473 533 q 345 576 393 576 q 267 560 303 576 q 207 512 232 544 q 167 433 181 480 q 153 322 153 385 q 169 217 153 264 q 210 137 185 170 q 266 85 236 103 q 323 68 296 68 q 368 77 346 68 q 411 102 390 86 q 453 137 432 117 q 495 177 474 156 "
    },
    "Ã©": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 290 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 310 709 q 288 717 298 712 q 272 728 279 722 l 427 1009 q 448 1007 435 1009 q 475 1003 461 1005 q 503 998 489 1001 q 523 994 516 996 l 539 967 l 310 709 "
    },
    "á¸ƒ": {
      "x_min": 6.59375,
      "x_max": 631,
      "ha": 691,
      "o": "m 631 333 q 621 252 631 293 q 591 171 611 211 q 544 97 572 132 q 481 35 516 61 q 403 -5 445 9 q 311 -21 360 -21 q 276 -15 298 -21 q 225 1 254 -9 q 162 28 196 12 q 93 65 128 44 l 93 878 q 89 926 93 909 q 77 951 86 943 q 50 963 67 960 q 6 969 33 965 l 6 996 q 90 1018 50 1007 q 168 1050 130 1029 l 175 1043 q 183 1036 179 1039 q 193 1027 187 1032 l 193 492 q 259 563 226 533 q 322 612 291 593 q 378 641 352 631 q 423 651 404 651 q 506 629 468 651 q 572 566 544 607 q 615 466 599 525 q 631 333 631 407 m 541 276 q 527 396 541 344 q 490 485 513 449 q 438 540 468 521 q 376 559 408 559 q 345 552 364 559 q 301 528 325 545 q 248 483 276 511 q 193 413 220 455 l 193 137 q 247 103 219 117 q 299 82 274 90 q 344 71 324 74 q 376 68 364 68 q 447 85 417 68 q 498 133 478 103 q 530 200 519 162 q 541 276 541 237 m 459 854 q 453 818 459 835 q 436 789 447 801 q 412 769 426 776 q 381 762 398 762 q 338 778 351 762 q 326 826 326 795 q 331 862 326 845 q 348 892 337 879 q 372 911 359 904 q 402 919 386 919 q 459 854 459 919 "
    },
    "B": {
      "x_min": 27.015625,
      "x_max": 675,
      "ha": 748,
      "o": "m 273 817 q 254 817 264 817 q 235 817 244 817 l 235 491 l 257 491 q 380 506 333 491 q 453 545 427 521 q 488 598 479 568 q 497 657 497 628 q 486 720 497 691 q 450 771 476 749 q 381 804 424 792 q 273 817 338 817 m 344 440 q 285 436 312 440 q 235 429 258 433 l 235 70 q 240 60 235 64 q 266 53 252 55 q 296 49 281 50 q 327 48 312 48 q 356 48 342 48 q 442 60 404 48 q 509 95 481 73 q 551 149 536 117 q 566 218 566 180 q 553 294 566 255 q 513 366 540 333 q 444 419 486 398 q 344 440 402 440 m 675 243 q 652 140 675 186 q 590 62 630 94 q 493 12 549 29 q 370 -5 438 -5 q 335 -5 356 -5 q 292 -5 315 -5 q 244 -5 269 -5 q 196 -5 219 -5 q 81 -5 141 -5 l 33 -5 l 33 28 q 103 49 78 38 q 129 70 129 60 l 129 806 q 79 799 103 803 q 33 791 55 795 l 27 834 q 87 848 52 841 q 159 861 121 855 q 236 871 197 867 q 311 875 275 875 q 431 862 377 875 q 522 826 485 849 q 580 768 560 802 q 601 691 601 734 q 567 565 601 618 q 474 489 533 512 q 553 459 517 481 q 617 404 590 437 q 659 330 644 372 q 675 243 675 289 "
    },
    "â€¦": {
      "x_min": 87,
      "x_max": 843,
      "ha": 910,
      "o": "m 245 83 q 238 39 245 59 q 218 4 231 19 q 189 -18 206 -10 q 153 -27 172 -27 q 102 -7 117 -27 q 87 48 87 12 q 94 91 87 71 q 114 127 101 111 q 143 151 126 142 q 179 160 160 160 q 228 140 212 160 q 245 83 245 120 m 544 83 q 537 39 544 59 q 517 4 530 19 q 488 -18 505 -10 q 452 -27 471 -27 q 401 -7 416 -27 q 386 48 386 12 q 393 91 386 71 q 413 127 400 111 q 442 151 425 142 q 478 160 459 160 q 527 140 511 160 q 544 83 544 120 m 843 83 q 836 39 843 59 q 816 4 829 19 q 787 -18 804 -10 q 751 -27 770 -27 q 700 -7 715 -27 q 685 48 685 12 q 692 91 685 71 q 712 127 699 111 q 741 151 724 142 q 777 160 758 160 q 826 140 810 160 q 843 83 843 120 "
    },
    "á»¦": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 847 855 l 847 825 m 561 1121 q 548 1087 561 1102 q 520 1059 536 1073 q 487 1034 503 1046 q 465 1010 472 1022 q 464 984 457 997 q 498 953 470 970 q 478 945 489 947 q 457 943 467 944 q 413 970 427 957 q 396 993 399 982 q 399 1014 393 1004 q 417 1032 405 1023 q 442 1050 428 1041 q 468 1068 456 1059 q 488 1088 480 1078 q 497 1111 497 1099 q 484 1150 497 1137 q 452 1163 471 1163 q 434 1159 442 1163 q 419 1149 425 1155 q 410 1135 413 1143 q 407 1121 407 1128 q 408 1113 407 1117 q 412 1106 410 1110 q 398 1102 407 1104 q 380 1098 390 1100 q 362 1094 371 1095 q 346 1091 352 1092 l 339 1099 l 339 1108 q 351 1140 339 1124 q 383 1168 364 1155 q 427 1188 403 1180 q 474 1196 450 1196 q 538 1175 515 1196 q 561 1121 561 1155 "
    },
    "H": {
      "x_min": 33,
      "x_max": 844,
      "ha": 890,
      "o": "m 33 0 l 33 29 q 104 49 78 38 q 130 70 130 60 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 478 l 641 478 l 641 783 q 617 803 641 791 q 545 825 592 815 l 545 854 l 844 854 l 844 825 q 772 804 798 815 q 747 783 747 792 l 747 70 q 771 50 747 62 q 844 29 795 38 l 844 0 l 545 0 l 545 29 q 615 49 590 38 q 641 70 641 60 l 641 417 l 236 417 l 236 70 q 259 50 236 62 q 332 29 283 38 l 332 0 l 33 0 "
    },
    "Ã®": {
      "x_min": -20.78125,
      "x_max": 400.125,
      "ha": 370,
      "o": "m 47 0 l 47 29 q 115 49 92 38 q 139 70 139 61 l 139 454 q 139 510 139 488 q 130 543 139 531 q 101 560 121 555 q 47 569 82 566 l 47 596 q 90 606 67 600 q 136 619 113 612 q 179 634 158 626 q 217 651 200 642 l 240 651 l 240 70 q 262 50 240 62 q 333 29 284 38 l 333 0 l 47 0 m 400 740 q 387 721 392 727 q 371 710 382 715 l 189 891 l 9 710 q 0 715 3 712 q -6 721 -3 718 q -12 729 -9 724 q -20 740 -16 734 l 154 997 l 226 997 l 400 740 "
    },
    "Æ°": {
      "x_min": 27.265625,
      "x_max": 834,
      "ha": 834,
      "o": "m 834 707 q 823 662 834 688 q 787 608 813 637 q 718 549 762 579 q 608 491 674 519 l 608 172 q 611 103 608 130 q 623 70 614 77 q 650 68 631 65 q 704 86 670 71 l 710 55 q 662 25 685 39 q 620 1 639 11 q 586 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 466 70 q 369 4 398 18 q 316 -14 340 -9 q 270 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 533 113 515 q 100 560 109 551 q 75 572 92 569 q 27 579 57 576 l 27 606 q 72 613 51 609 q 112 623 93 617 q 152 635 132 628 q 194 651 172 642 l 213 625 l 213 226 q 219 147 213 179 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 483 q 505 532 508 513 q 492 560 502 550 q 463 574 482 570 q 415 579 445 579 l 415 606 q 506 625 462 612 q 588 651 550 638 l 608 625 l 608 538 q 659 568 638 553 q 693 599 680 584 q 713 627 707 614 q 720 652 720 641 q 710 689 720 671 q 681 722 700 707 l 799 777 q 824 744 814 762 q 834 707 834 726 "
    },
    "âˆ’": {
      "x_min": 40.546875,
      "x_max": 517.0625,
      "ha": 558,
      "o": "m 517 402 q 508 372 513 388 q 497 344 502 355 l 57 344 l 40 360 q 49 389 43 373 q 61 418 55 405 l 501 418 l 517 402 "
    },
    "É“": {
      "x_min": 93,
      "x_max": 631,
      "ha": 691,
      "o": "m 541 276 q 527 396 541 343 q 490 485 513 449 q 438 539 468 520 q 376 559 408 559 q 345 551 364 559 q 301 528 325 544 q 248 483 276 511 q 193 413 220 455 l 193 136 q 247 102 219 116 q 299 81 274 89 q 344 70 324 73 q 376 67 364 67 q 447 84 417 67 q 498 132 478 102 q 530 199 519 162 q 541 276 541 236 m 631 333 q 621 252 631 293 q 591 171 611 211 q 544 97 572 132 q 481 35 516 61 q 403 -5 445 9 q 311 -21 360 -21 q 276 -15 298 -21 q 225 1 254 -9 q 162 28 196 12 q 93 64 128 43 l 93 596 q 104 741 93 681 q 136 847 115 802 q 186 925 157 892 q 252 988 216 959 q 292 1014 269 1003 q 336 1034 314 1026 q 378 1046 359 1042 q 410 1051 398 1051 q 469 1039 440 1051 q 519 1013 497 1028 q 554 984 541 998 q 568 963 568 969 q 558 947 568 958 q 535 925 548 936 q 507 903 521 913 q 482 888 492 893 q 417 945 454 924 q 340 966 379 966 q 293 950 318 966 q 245 896 267 934 q 207 795 222 858 q 193 634 193 731 l 193 492 q 258 563 226 533 q 322 612 291 593 q 378 641 352 631 q 423 651 404 651 q 506 629 468 651 q 572 566 544 607 q 615 466 599 525 q 631 333 631 407 "
    },
    "á¸§": {
      "x_min": 37,
      "x_max": 732,
      "ha": 752,
      "o": "m 446 0 l 446 29 q 516 51 493 42 q 539 70 539 61 l 539 429 q 534 496 539 470 q 518 537 529 522 q 491 557 507 552 q 453 563 475 563 q 403 550 430 563 q 347 514 376 538 q 288 455 318 491 q 230 372 258 419 l 230 70 q 254 49 230 60 q 323 29 279 39 l 323 0 l 37 0 l 37 29 q 105 50 80 41 q 130 70 130 59 l 130 880 q 127 924 130 908 q 115 949 125 940 q 87 962 106 958 q 37 969 68 965 l 37 997 q 87 1007 64 1002 q 130 1019 110 1013 q 167 1033 149 1026 q 204 1051 185 1041 l 230 1028 l 230 464 q 293 541 258 507 q 363 600 328 576 q 432 638 398 625 q 491 651 465 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 70 q 659 52 639 61 q 732 29 679 43 l 732 0 l 446 0 m 583 1214 q 577 1178 583 1195 q 560 1149 571 1161 q 536 1129 550 1136 q 505 1122 522 1122 q 462 1138 475 1122 q 450 1186 450 1155 q 455 1222 450 1205 q 472 1252 461 1239 q 496 1271 483 1264 q 526 1279 510 1279 q 583 1214 583 1279 m 317 1214 q 311 1178 317 1195 q 294 1149 305 1161 q 270 1129 284 1136 q 239 1122 256 1122 q 196 1138 209 1122 q 184 1186 184 1155 q 189 1222 184 1205 q 206 1252 195 1239 q 230 1271 217 1264 q 260 1279 244 1279 q 317 1214 317 1279 "
    },
    "Ä": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -20 485 -20 q 429 11 444 -20 q 415 97 415 42 q 354 44 384 66 q 295 7 324 21 q 241 -13 267 -6 q 194 -20 215 -20 q 145 -12 170 -20 q 99 13 120 -3 q 66 59 79 31 q 53 126 53 86 q 70 212 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 550 868 q 545 851 549 861 q 538 831 542 841 q 530 812 534 822 q 524 797 527 803 l 115 797 l 100 813 q 105 830 102 821 q 112 850 108 840 q 120 869 116 859 q 128 885 124 878 l 535 885 l 550 868 "
    },
    "á¹¤": {
      "x_min": 80,
      "x_max": 589,
      "ha": 649,
      "o": "m 589 255 q 581 193 589 225 q 556 130 573 161 q 516 72 540 99 q 458 24 491 45 q 384 -7 426 4 q 293 -20 342 -20 q 245 -15 271 -20 q 191 -2 219 -10 q 137 18 164 6 q 87 46 110 30 q 81 69 82 48 q 80 116 80 89 q 80 172 80 144 q 87 219 80 201 l 114 216 q 152 147 129 176 q 203 98 175 117 q 263 70 231 79 q 330 61 296 61 q 390 73 360 61 q 444 107 420 86 q 483 158 468 129 q 499 219 499 187 q 481 290 499 261 q 434 343 463 320 q 367 385 405 366 q 291 421 330 403 q 215 460 252 440 q 149 507 178 480 q 102 570 120 535 q 85 655 85 606 q 90 701 85 676 q 110 750 96 725 q 144 797 123 774 q 195 837 165 820 q 263 864 224 854 q 350 875 301 875 q 410 870 380 875 q 466 857 440 865 q 514 837 493 849 q 547 812 535 826 q 547 796 552 808 q 534 770 543 784 q 514 743 525 756 q 495 722 503 729 l 471 726 q 432 763 453 748 q 391 789 412 779 q 349 802 370 798 q 310 807 329 807 q 247 795 273 807 q 203 766 220 783 q 178 727 186 749 q 171 687 171 706 q 188 631 171 655 q 236 585 206 606 q 302 546 265 564 q 379 508 340 528 q 456 465 419 488 q 523 412 494 442 q 571 344 553 382 q 589 255 589 305 m 244 927 q 228 938 233 932 q 214 957 223 945 l 472 1173 q 490 1162 479 1169 q 513 1149 501 1156 q 535 1135 525 1142 q 551 1123 545 1128 l 557 1092 l 244 927 m 399 1262 q 393 1226 399 1243 q 376 1197 387 1209 q 352 1177 366 1184 q 321 1170 338 1170 q 278 1186 291 1170 q 266 1234 266 1203 q 271 1270 266 1253 q 288 1300 277 1287 q 312 1319 299 1312 q 342 1327 326 1327 q 399 1262 399 1327 "
    },
    "Ä¨": {
      "x_min": -30.09375,
      "x_max": 423.09375,
      "ha": 406,
      "o": "m 47 0 l 47 29 q 118 49 92 38 q 144 70 144 61 l 144 782 q 119 803 144 790 q 47 824 95 815 l 47 854 l 346 854 l 346 824 q 275 803 300 815 q 250 782 250 792 l 250 70 q 274 50 250 62 q 346 29 298 38 l 346 0 l 47 0 m 423 1103 q 394 1048 411 1075 q 357 999 378 1020 q 313 963 337 977 q 263 950 289 950 q 219 961 240 950 q 179 988 198 973 q 139 1015 159 1003 q 100 1027 119 1027 q 51 1004 73 1027 q 5 944 29 982 l -30 958 q -1 1013 -18 986 q 35 1062 14 1041 q 79 1098 55 1084 q 129 1112 103 1112 q 177 1100 154 1112 q 219 1073 199 1088 q 257 1046 239 1058 q 291 1035 275 1035 q 340 1056 316 1035 q 387 1118 364 1078 l 423 1103 "
    },
    "*": {
      "x_min": 47,
      "x_max": 561,
      "ha": 608,
      "o": "m 324 805 l 498 938 q 530 916 512 929 q 561 893 549 904 l 561 865 l 345 769 l 544 682 q 544 642 544 665 q 537 604 544 619 l 512 588 l 325 731 l 350 512 q 334 503 343 508 q 316 494 325 498 q 297 485 306 489 q 281 480 288 482 l 255 493 l 283 731 l 109 598 q 93 609 102 603 q 75 621 84 615 q 59 633 66 627 q 47 644 51 638 l 47 673 l 262 768 l 61 855 q 63 874 63 863 q 63 895 63 884 q 64 916 63 906 q 69 933 66 926 l 92 948 l 282 804 l 256 1026 q 273 1034 263 1030 q 292 1044 282 1039 q 311 1052 301 1049 q 327 1058 320 1056 l 351 1044 l 324 805 "
    },
    "Äƒ": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 535 922 q 490 829 515 867 q 437 768 465 792 q 380 735 410 745 q 320 725 350 725 q 257 735 287 725 q 198 768 226 745 q 147 829 171 792 q 102 922 122 867 q 118 940 110 934 q 138 952 126 947 q 176 887 154 914 q 221 845 197 861 q 270 822 245 829 q 317 815 295 815 q 366 822 341 815 q 416 845 392 829 q 462 887 440 861 q 500 952 484 914 q 519 940 511 947 q 535 922 527 934 "
    },
    "â€ ": {
      "x_min": 46.734375,
      "x_max": 602.578125,
      "ha": 649,
      "o": "m 602 728 q 578 676 593 705 q 548 622 564 647 q 456 646 502 638 q 355 654 410 654 q 360 557 355 607 q 379 461 366 508 q 360 310 360 386 q 360 169 360 234 q 360 85 360 130 q 360 0 360 40 q 332 -15 348 -9 q 301 -26 316 -21 l 282 -12 q 286 80 286 32 q 286 169 286 128 q 286 311 286 236 q 266 461 286 386 q 284 558 279 508 q 290 655 290 608 q 231 652 259 655 q 175 645 203 649 q 120 635 148 641 q 63 622 93 629 l 46 651 q 70 703 56 674 q 101 757 84 731 q 190 733 145 741 q 290 725 235 725 q 282 842 290 786 q 256 948 274 897 q 310 979 281 965 q 360 1003 338 994 l 389 986 q 365 856 373 922 q 357 727 357 790 q 415 727 387 727 q 472 731 444 727 q 528 743 500 736 q 585 757 556 750 l 602 728 "
    },
    "Â°": {
      "x_min": 93,
      "x_max": 394,
      "ha": 487,
      "o": "m 322 674 q 317 712 322 694 q 302 744 312 730 q 280 766 293 758 q 251 775 267 775 q 219 768 234 775 q 192 748 204 761 q 173 717 180 735 q 166 677 166 699 q 170 639 166 657 q 184 607 175 621 q 206 585 193 594 q 236 577 219 577 q 267 583 252 577 q 294 602 282 589 q 314 633 307 615 q 322 674 322 651 m 394 709 q 378 631 394 667 q 336 570 362 596 q 280 529 311 544 q 218 515 249 515 q 168 525 191 515 q 128 553 144 535 q 102 593 111 570 q 93 643 93 616 q 108 720 93 685 q 149 782 124 756 q 205 823 174 808 q 268 838 237 838 q 317 827 294 838 q 357 799 340 816 q 384 758 374 781 q 394 709 394 734 "
    },
    "É…": {
      "x_min": 13,
      "x_max": 857,
      "ha": 884,
      "o": "m 13 28 q 83 46 58 37 q 115 73 109 54 l 370 814 q 388 839 375 829 q 416 857 401 850 q 446 868 432 864 q 470 874 461 872 l 762 73 q 792 45 769 56 q 857 28 815 34 l 857 0 l 568 0 l 568 31 q 640 40 621 31 q 653 70 660 50 l 418 716 l 194 73 q 211 44 187 54 q 289 28 234 35 l 289 0 l 13 0 l 13 28 "
    },
    "á»“": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 565 741 q 552 722 557 728 q 536 710 547 716 l 354 892 l 174 710 q 165 716 169 713 q 159 722 161 718 q 152 730 156 725 q 144 741 149 735 l 320 998 l 391 998 l 565 741 m 431 1036 q 413 1023 422 1028 q 393 1015 405 1019 l 160 1271 l 174 1299 q 194 1303 181 1301 q 221 1308 207 1305 q 249 1313 235 1311 q 271 1315 263 1314 l 431 1036 "
    },
    "Åµ": {
      "x_min": 13,
      "x_max": 947,
      "ha": 961,
      "o": "m 947 601 q 914 592 926 597 q 895 583 902 588 q 885 573 888 579 q 880 559 882 567 l 733 40 q 718 14 729 25 q 692 -2 706 4 q 664 -13 678 -9 q 642 -20 651 -17 l 484 439 l 348 40 q 332 14 343 24 q 307 -3 321 3 q 281 -14 294 -10 q 259 -20 268 -17 l 85 559 q 13 601 79 586 l 13 631 l 270 631 l 270 601 q 218 594 237 598 q 192 583 200 589 q 184 572 184 578 q 186 559 184 565 l 311 129 l 475 631 l 521 631 l 693 129 l 808 559 q 794 584 812 575 q 731 601 776 594 l 731 631 l 947 631 l 947 601 m 708 740 q 695 721 700 728 q 679 710 691 715 l 497 892 l 317 710 q 308 715 312 713 q 302 721 305 718 q 295 730 299 725 q 287 740 292 734 l 463 998 l 534 998 l 708 740 "
    },
    "Ç½": {
      "x_min": 53,
      "x_max": 889,
      "ha": 942,
      "o": "m 889 378 q 855 352 875 364 q 814 332 835 340 l 515 332 q 514 308 514 320 l 514 283 q 524 206 514 244 q 554 137 534 168 q 603 88 574 107 q 669 70 632 70 q 709 72 690 70 q 749 83 728 74 q 797 111 771 93 q 858 160 823 129 q 873 147 867 156 q 883 133 880 138 q 810 54 843 84 q 746 7 777 24 q 687 -14 716 -8 q 627 -20 658 -20 q 569 -10 598 -20 q 516 17 541 0 q 471 62 491 35 q 437 122 451 88 q 375 60 408 86 q 308 15 341 33 q 245 -11 275 -2 q 194 -20 215 -20 q 145 -11 170 -20 q 99 13 120 -3 q 66 59 79 31 q 53 126 53 87 q 68 203 53 166 q 115 272 84 240 q 194 330 146 304 q 307 375 242 356 q 357 386 329 382 q 414 390 384 390 l 414 448 q 410 499 414 474 q 394 542 406 523 q 363 571 383 560 q 310 583 342 583 q 264 573 289 583 q 221 546 240 563 q 193 506 202 529 q 193 458 183 483 q 181 445 195 452 q 147 431 167 437 q 108 422 128 425 q 80 421 88 419 l 71 445 q 90 498 74 472 q 130 546 106 523 q 184 588 154 569 q 244 621 213 607 q 303 643 274 635 q 355 651 332 651 q 448 626 413 651 q 497 553 482 602 q 542 599 518 580 q 590 629 566 618 q 638 645 615 640 q 680 651 661 651 q 752 639 721 651 q 806 607 783 627 q 845 560 829 587 q 871 503 861 533 q 885 440 881 473 q 889 378 889 408 m 423 165 q 416 205 418 185 q 414 248 414 225 l 414 329 q 374 326 392 329 q 349 321 357 324 q 205 253 249 296 q 162 155 162 211 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 q 274 74 251 68 q 321 92 296 80 q 372 122 346 104 q 423 165 398 141 m 660 583 q 625 575 645 583 q 585 547 605 568 q 547 489 565 526 q 521 394 529 452 l 752 394 q 772 399 766 394 q 778 418 778 404 q 771 480 778 450 q 750 532 764 510 q 714 569 736 555 q 660 583 692 583 m 461 709 q 440 717 449 712 q 423 728 431 722 l 579 1009 q 600 1007 587 1009 q 627 1003 613 1005 q 654 998 641 1001 q 675 994 668 996 l 690 967 l 461 709 "
    },
    "á¸°": {
      "x_min": 33,
      "x_max": 779.53125,
      "ha": 786,
      "o": "m 33 0 l 33 29 q 104 50 78 39 q 130 70 130 61 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 438 l 507 765 q 526 794 522 783 q 523 811 530 805 q 500 820 517 817 q 458 824 483 824 l 458 855 l 737 855 l 737 825 q 699 819 715 822 q 669 812 682 817 q 644 801 655 808 q 624 783 634 794 l 333 455 l 653 86 q 679 65 665 73 q 708 55 693 58 q 740 52 724 52 q 774 54 757 52 l 779 25 q 700 1 740 12 q 629 -9 661 -9 q 595 -3 610 -9 q 566 19 580 2 l 236 433 l 236 70 q 259 50 236 62 q 332 29 283 39 l 332 0 l 33 0 m 304 927 q 287 938 292 932 q 274 958 283 945 l 531 1173 q 549 1162 537 1168 q 572 1148 560 1156 q 594 1135 584 1141 q 609 1122 604 1128 l 615 1092 l 304 927 "
    },
    "Ã•": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 636 1103 q 607 1048 624 1075 q 570 999 590 1020 q 526 963 550 977 q 476 950 502 950 q 432 961 453 950 q 391 988 411 973 q 351 1015 371 1003 q 312 1027 332 1027 q 264 1004 286 1027 q 217 944 242 982 l 182 958 q 210 1013 194 986 q 247 1062 227 1041 q 291 1098 268 1084 q 341 1112 315 1112 q 389 1100 366 1112 q 431 1073 411 1088 q 469 1046 451 1058 q 503 1035 487 1035 q 553 1056 528 1035 q 600 1118 577 1078 l 636 1103 "
    },
    "áº": {
      "x_min": -31,
      "x_max": 658,
      "ha": 672,
      "o": "m 658 601 q 625 592 638 596 q 604 583 612 588 q 592 571 596 578 q 584 555 588 565 l 362 -55 q 300 -184 334 -130 q 229 -272 266 -237 q 155 -322 192 -306 q 84 -339 117 -339 q 38 -335 59 -339 q 1 -327 16 -332 q -22 -315 -13 -322 q -31 -303 -31 -309 q -22 -286 -31 -299 q -2 -260 -14 -274 q 22 -231 9 -245 q 47 -211 36 -218 q 110 -232 79 -232 q 167 -223 142 -232 q 194 -204 179 -219 q 227 -168 210 -189 q 258 -119 243 -146 q 286 -62 274 -91 l 305 -15 l 87 555 q 64 584 81 574 q 14 601 47 593 l 14 630 l 271 630 l 271 601 q 228 594 245 598 q 203 586 212 591 q 193 573 194 581 q 195 555 191 566 l 358 123 l 512 555 q 513 572 515 565 q 502 584 511 579 q 476 593 493 589 q 436 601 460 597 l 436 631 l 658 631 l 658 601 m 419 854 q 413 818 419 835 q 396 789 407 801 q 372 769 386 776 q 341 762 358 762 q 298 778 311 762 q 286 826 286 795 q 291 862 286 845 q 308 892 297 879 q 332 911 319 904 q 362 919 346 919 q 419 854 419 919 "
    },
    "êžŠ": {
      "x_min": 39.875,
      "x_max": 410.0625,
      "ha": 450,
      "o": "m 410 247 q 400 215 406 232 q 390 186 394 198 l 56 186 l 39 204 q 48 234 43 217 q 60 264 54 250 l 393 264 l 410 247 m 410 422 q 400 391 406 409 l 390 363 l 56 363 l 39 380 q 48 410 43 393 q 60 440 54 427 l 393 440 l 410 422 "
    },
    "á¸µ": {
      "x_min": 37,
      "x_max": 681.265625,
      "ha": 681,
      "o": "m 37 0 l 37 29 q 105 49 80 41 q 130 70 130 58 l 130 879 q 126 927 130 910 q 112 952 122 944 q 83 963 102 961 q 37 970 65 966 l 37 997 q 127 1019 82 1006 q 205 1051 171 1031 l 230 1028 l 230 359 l 434 542 q 461 573 456 561 q 459 591 465 585 q 438 599 453 597 q 406 602 423 602 l 406 631 l 658 631 l 658 602 q 604 590 628 598 q 551 559 579 583 l 324 374 l 574 75 q 594 56 584 64 q 616 45 604 49 q 643 41 628 41 q 677 41 658 41 l 681 12 q 643 4 661 7 q 607 -1 624 1 q 578 -4 591 -3 q 558 -6 565 -6 q 514 1 531 -6 q 482 29 497 9 l 230 354 l 230 70 q 232 61 230 65 q 241 52 234 57 q 263 43 249 48 q 303 29 278 37 l 303 0 l 37 0 m 583 -155 q 578 -172 582 -162 q 571 -192 575 -182 q 563 -211 567 -201 q 557 -227 560 -220 l 148 -227 l 134 -210 q 138 -193 135 -202 q 146 -173 142 -183 q 154 -154 150 -164 q 161 -139 158 -145 l 568 -139 l 583 -155 "
    },
    "5": {
      "x_min": 51.953125,
      "x_max": 555,
      "ha": 638,
      "o": "m 555 278 q 537 165 555 219 q 486 70 519 111 q 403 4 452 28 q 292 -20 354 -20 q 170 3 230 -20 q 51 81 109 26 l 75 126 q 139 86 110 101 q 192 62 168 71 q 237 51 217 54 q 277 48 258 48 q 353 64 320 48 q 406 109 385 81 q 438 174 427 138 q 449 249 449 210 q 438 333 449 295 q 404 397 427 371 q 348 439 382 424 q 267 454 314 454 q 235 451 253 454 q 198 440 218 447 q 159 423 178 434 q 122 400 139 413 l 91 422 q 100 479 95 446 q 111 547 105 511 q 122 620 117 583 q 133 694 128 658 q 141 761 138 729 q 145 817 145 793 l 435 817 q 468 818 454 817 q 494 823 483 820 q 515 829 505 825 l 534 809 q 517 786 527 798 q 498 764 507 775 q 478 744 488 753 q 460 729 468 735 l 203 729 q 199 678 203 709 q 191 615 196 647 q 182 554 187 583 q 175 511 177 526 q 237 524 201 519 q 307 529 273 529 q 413 508 367 529 q 491 453 460 487 q 538 373 522 418 q 555 278 555 327 "
    },
    "o": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 "
    },
    "Ì†": {
      "x_min": -558.265625,
      "x_max": -126.28125,
      "ha": 0,
      "o": "m -126 923 q -171 830 -146 868 q -223 769 -196 793 q -281 736 -251 746 q -340 726 -311 726 q -404 736 -373 726 q -462 769 -434 746 q -514 830 -489 793 q -558 923 -538 868 q -542 941 -550 935 q -523 953 -534 948 q -484 888 -506 915 q -439 846 -463 862 q -390 823 -415 830 q -343 816 -365 816 q -294 823 -320 816 q -244 846 -269 830 q -199 888 -220 862 q -161 953 -177 915 q -142 941 -150 948 q -126 923 -134 935 "
    },
    "áºŒ": {
      "x_min": 21,
      "x_max": 817,
      "ha": 838,
      "o": "m 517 0 l 517 28 q 564 36 546 31 q 590 46 582 40 q 597 62 598 53 q 586 85 596 72 l 402 362 l 234 85 q 237 43 216 56 q 320 31 259 31 l 320 0 l 21 0 l 21 30 q 94 42 63 30 q 141 84 125 55 l 353 436 l 131 768 q 111 792 121 783 q 90 808 102 802 q 64 818 79 814 q 27 825 49 821 l 27 854 l 327 854 l 327 825 q 253 808 271 819 q 256 768 236 797 l 420 522 l 570 768 q 578 794 579 784 q 565 810 576 804 q 533 820 554 817 q 484 824 513 824 l 484 855 l 785 855 l 785 825 q 742 818 760 822 q 708 808 723 815 q 682 792 693 802 q 663 768 671 782 l 470 448 l 712 85 q 732 61 722 71 q 755 45 742 52 q 782 35 767 39 q 817 28 798 31 l 817 0 l 517 0 m 618 1044 q 612 1008 618 1025 q 595 979 606 991 q 571 959 585 966 q 540 952 557 952 q 497 968 510 952 q 485 1016 485 985 q 490 1052 485 1035 q 507 1082 496 1069 q 531 1101 518 1094 q 561 1109 545 1109 q 618 1044 618 1109 m 353 1044 q 347 1008 353 1025 q 330 979 341 991 q 306 959 320 966 q 275 952 292 952 q 232 968 245 952 q 220 1016 220 985 q 225 1052 220 1035 q 242 1082 231 1069 q 266 1101 253 1094 q 296 1109 280 1109 q 353 1044 353 1109 "
    },
    "Ç": {
      "x_min": -14.140625,
      "x_max": 406.484375,
      "ha": 406,
      "o": "m 47 0 l 47 29 q 118 49 92 38 q 144 70 144 61 l 144 782 q 119 803 144 790 q 47 824 95 815 l 47 854 l 346 854 l 346 824 q 275 803 300 815 q 250 782 250 792 l 250 70 q 274 50 250 62 q 346 29 298 38 l 346 0 l 47 0 m 232 943 l 161 943 l -14 1151 q -6 1162 -9 1157 q 0 1170 -2 1167 q 6 1176 3 1174 q 15 1183 10 1179 l 198 1039 l 377 1183 q 393 1170 389 1176 q 406 1151 398 1164 l 232 943 "
    },
    "Ì§": {
      "x_min": -450.609375,
      "x_max": -231,
      "ha": 0,
      "o": "m -231 -155 q -242 -203 -231 -180 q -278 -245 -254 -226 q -342 -278 -303 -264 q -435 -300 -381 -292 l -450 -267 q -391 -252 -416 -261 q -352 -232 -367 -243 q -329 -209 -336 -221 q -323 -186 -323 -197 q -340 -154 -323 -163 q -398 -141 -357 -145 q -392 -122 -397 -139 q -377 -77 -387 -108 q -350 9 -368 -47 l -298 9 l -324 -70 q -290 -80 -307 -74 q -260 -97 -273 -87 q -239 -121 -247 -107 q -231 -155 -231 -136 "
    },
    "d": {
      "x_min": 53,
      "x_max": 698.015625,
      "ha": 707,
      "o": "m 698 57 q 643 21 667 36 q 601 -2 619 7 q 570 -15 583 -11 q 548 -20 557 -20 q 512 10 526 -20 q 498 114 498 40 q 446 58 471 83 q 394 16 421 33 q 339 -10 368 -1 q 276 -20 310 -20 q 199 2 239 -20 q 127 65 160 24 q 73 166 94 106 q 53 301 53 226 q 63 381 53 340 q 92 461 73 422 q 140 534 112 500 q 202 595 167 569 q 278 636 237 621 q 366 651 319 651 q 428 643 397 651 q 495 608 459 636 l 495 865 q 493 925 495 903 q 481 959 491 947 q 453 975 472 971 q 398 980 433 980 l 398 1007 q 496 1027 453 1015 q 573 1051 539 1040 l 595 1031 l 595 172 q 595 131 595 148 q 597 103 595 114 q 601 84 599 91 q 608 72 604 76 q 632 67 614 64 q 689 86 649 70 l 698 57 m 495 177 l 495 495 q 433 555 473 534 q 345 577 393 577 q 267 561 303 577 q 207 513 232 545 q 167 433 181 481 q 153 322 153 386 q 169 217 153 264 q 210 137 185 170 q 266 85 236 103 q 323 68 296 68 q 368 77 346 68 q 411 102 390 87 q 453 137 432 117 q 495 177 474 156 "
    },
    ",": {
      "x_min": 58.28125,
      "x_max": 259,
      "ha": 312,
      "o": "m 259 48 q 248 -11 259 21 q 219 -79 238 -44 q 172 -147 199 -114 q 110 -206 144 -180 l 80 -182 q 111 -140 98 -161 q 133 -99 124 -120 q 146 -53 142 -77 q 150 0 150 -29 q 130 46 150 28 q 68 64 111 64 l 58 94 q 83 113 64 102 q 125 133 102 123 q 170 149 148 143 q 205 156 193 156 q 248 113 237 140 q 259 48 259 86 "
    },
    "êž‹": {
      "x_min": 112,
      "x_max": 218,
      "ha": 340,
      "o": "m 218 477 q 196 471 209 474 q 169 467 183 469 q 140 464 155 465 q 112 463 125 463 l 112 1031 q 130 1041 117 1035 q 157 1054 143 1048 q 183 1065 171 1060 q 202 1071 196 1069 l 218 1053 l 218 477 "
    },
    "\"": {
      "x_min": 106.03125,
      "x_max": 458,
      "ha": 578,
      "o": "m 219 565 q 203 559 212 562 q 183 555 193 557 q 161 552 172 553 q 141 551 150 551 l 106 966 q 127 976 113 970 q 156 987 141 982 q 184 997 171 993 q 205 1003 197 1001 l 219 989 l 219 565 m 458 565 q 442 559 451 562 q 421 555 432 557 q 400 552 411 553 q 379 551 389 551 l 344 966 q 365 976 351 970 q 394 987 378 982 q 423 997 409 993 q 444 1003 436 1001 l 458 989 l 458 565 "
    },
    "Ä—": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 199 597 165 573 q 232 617 214 607 q 270 634 251 627 q 310 646 290 642 q 347 651 330 651 q 424 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 405 855 q 399 819 405 836 q 382 790 393 802 q 358 770 372 777 q 327 763 344 763 q 284 779 297 763 q 272 827 272 796 q 277 863 272 846 q 294 893 283 880 q 318 912 305 905 q 348 920 332 920 q 405 855 405 920 "
    },
    "á»": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 290 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 548 740 q 535 721 540 727 q 519 710 531 715 l 338 891 l 158 710 q 149 715 152 712 q 142 721 145 718 q 136 729 139 724 q 128 740 133 734 l 303 997 l 374 997 l 548 740 m 414 1036 q 397 1023 405 1028 q 377 1015 388 1018 l 143 1270 l 158 1298 q 177 1303 164 1300 q 205 1308 190 1305 q 233 1312 219 1310 q 255 1314 247 1314 l 414 1036 "
    },
    "Ã": {
      "x_min": 47,
      "x_max": 420.4375,
      "ha": 406,
      "o": "m 47 0 l 47 29 q 118 49 92 38 q 144 70 144 61 l 144 782 q 119 803 144 790 q 47 824 95 815 l 47 854 l 346 854 l 346 824 q 275 803 300 815 q 250 782 250 792 l 250 70 q 274 50 250 62 q 346 29 298 38 l 346 0 l 47 0 m 109 926 q 92 937 97 931 q 79 957 87 944 l 336 1172 q 353 1161 342 1167 q 377 1147 365 1155 q 399 1134 389 1140 q 414 1121 409 1127 l 420 1091 l 109 926 "
    },
    "Ãš": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 847 855 l 847 825 m 361 927 q 345 938 350 932 q 332 958 340 945 l 589 1173 q 607 1162 596 1168 q 630 1148 618 1156 q 652 1135 642 1141 q 668 1122 662 1128 l 674 1092 l 361 927 "
    },
    "Æ ": {
      "x_min": 47,
      "x_max": 811,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 811 944 q 803 906 811 927 q 778 861 796 885 q 730 812 760 837 q 654 761 700 786 q 731 614 705 698 q 757 438 757 529 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 532 853 482 875 q 622 796 582 832 q 679 846 661 821 q 697 889 697 871 q 687 926 697 908 q 658 959 677 944 l 776 1014 q 801 981 791 999 q 811 944 811 963 "
    },
    "Å¶": {
      "x_min": -0.265625,
      "x_max": 781,
      "ha": 808,
      "o": "m 239 0 l 239 29 q 323 55 299 42 q 348 78 348 68 l 348 364 q 292 478 323 419 q 228 594 260 538 q 165 699 195 651 q 111 780 134 748 q 97 794 105 788 q 78 806 90 801 q 48 814 66 811 q 2 818 30 818 l 0 846 q 78 856 38 852 q 145 861 117 861 q 197 834 175 861 q 250 757 221 802 q 307 663 278 713 q 363 562 335 614 q 413 461 390 510 l 592 780 q 585 808 602 798 q 518 825 569 818 l 518 854 l 781 854 l 781 825 q 711 807 735 816 q 677 780 686 798 l 454 366 l 454 78 q 459 68 454 73 q 478 55 465 62 q 512 42 491 48 q 565 29 533 35 l 565 0 l 239 0 m 611 957 q 598 938 603 944 q 582 926 594 932 l 400 1067 l 220 926 q 211 932 215 929 q 204 938 207 934 q 198 946 201 941 q 190 957 195 951 l 365 1166 l 436 1166 l 611 957 "
    },
    "áº†": {
      "x_min": 13,
      "x_max": 1131,
      "ha": 1157,
      "o": "m 1131 824 q 1082 813 1101 818 q 1051 802 1063 807 q 1035 792 1040 797 q 1029 782 1030 788 l 877 39 q 863 14 874 25 q 838 -2 852 4 q 809 -14 824 -9 q 785 -20 795 -18 l 568 639 l 377 39 q 361 15 372 25 q 336 -2 350 4 q 306 -13 321 -9 q 277 -20 290 -18 l 104 777 q 80 805 101 794 q 13 824 60 816 l 13 854 l 298 854 l 298 824 q 246 816 265 821 q 218 805 228 811 q 207 791 209 799 q 207 777 206 784 l 339 169 l 554 854 l 591 854 l 827 169 l 951 782 q 945 797 952 790 q 924 807 938 803 q 890 816 910 812 q 849 824 871 820 l 849 854 l 1131 854 l 1131 824 m 638 1044 q 632 1008 638 1025 q 615 979 626 991 q 591 959 605 966 q 560 952 577 952 q 517 968 530 952 q 505 1016 505 985 q 510 1052 505 1035 q 527 1082 516 1069 q 551 1101 538 1094 q 581 1109 565 1109 q 638 1044 638 1109 "
    },
    "á»°": {
      "x_min": 33,
      "x_max": 930,
      "ha": 930,
      "o": "m 930 944 q 921 904 930 927 q 892 856 913 882 q 837 803 872 830 q 751 749 803 775 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 804 855 q 816 889 816 873 q 806 926 816 909 q 777 960 796 944 l 895 1014 q 920 981 910 999 q 930 944 930 964 m 516 -189 q 510 -225 516 -208 q 493 -254 504 -242 q 469 -274 483 -267 q 438 -282 455 -282 q 395 -265 408 -282 q 383 -217 383 -248 q 388 -181 383 -198 q 405 -151 394 -164 q 429 -132 416 -139 q 459 -125 443 -125 q 516 -189 516 -125 "
    },
    "Ã": {
      "x_min": -0.265625,
      "x_max": 781,
      "ha": 808,
      "o": "m 239 0 l 239 29 q 323 55 299 42 q 348 78 348 68 l 348 364 q 292 478 323 419 q 228 594 260 538 q 165 699 195 651 q 111 780 134 748 q 97 794 105 788 q 78 806 90 801 q 48 814 66 811 q 2 818 30 818 l 0 846 q 78 856 38 852 q 145 861 117 861 q 197 834 175 861 q 250 757 221 802 q 307 663 278 713 q 363 562 335 614 q 413 461 390 510 l 592 780 q 585 808 602 798 q 518 825 569 818 l 518 854 l 781 854 l 781 825 q 711 807 735 816 q 677 780 686 798 l 454 366 l 454 78 q 459 68 454 73 q 478 55 465 62 q 512 42 491 48 q 565 29 533 35 l 565 0 l 239 0 m 313 926 q 297 937 302 931 q 283 957 292 944 l 540 1172 q 558 1161 547 1168 q 582 1148 570 1155 q 604 1134 594 1141 q 619 1122 614 1127 l 625 1091 l 313 926 "
    },
    "Å": {
      "x_min": 53.03125,
      "x_max": 473.640625,
      "ha": 525,
      "o": "m 465 192 q 451 109 465 144 q 417 51 438 75 q 369 13 395 28 q 318 -7 343 0 q 271 -17 293 -15 q 236 -20 248 -20 q 159 -7 204 -20 q 70 29 114 4 q 64 51 65 31 q 63 97 63 71 q 66 150 63 123 q 74 192 69 176 l 104 184 q 117 131 104 156 q 153 87 130 106 q 205 58 175 69 q 269 48 234 48 q 313 55 293 48 q 348 77 333 63 q 370 111 362 91 q 379 154 379 130 q 364 202 379 181 q 327 240 350 223 q 274 273 303 258 q 213 304 244 288 q 159 335 185 319 q 113 371 133 351 q 81 417 93 392 q 69 474 69 442 q 85 549 69 516 q 129 604 102 582 q 192 639 157 627 q 266 651 228 651 q 311 647 287 651 q 361 636 336 643 q 406 620 385 629 q 441 598 427 611 q 442 580 445 594 q 432 549 439 566 q 419 517 426 532 q 408 497 412 502 l 382 502 q 316 570 350 551 q 249 590 281 590 q 209 582 227 590 q 179 563 192 575 q 161 537 167 551 q 155 508 155 522 q 167 469 155 487 q 201 437 180 452 q 248 408 221 421 q 302 379 274 394 q 360 347 331 364 q 412 308 389 330 q 450 258 435 286 q 465 192 465 230 m 473 740 q 461 721 465 727 q 445 710 456 715 l 263 891 l 82 710 q 73 715 77 712 q 67 721 70 718 q 61 729 64 724 q 53 740 57 734 l 228 997 l 299 997 l 473 740 "
    },
    "Ç§": {
      "x_min": 20,
      "x_max": 657.625,
      "ha": 664,
      "o": "m 459 406 q 447 474 459 442 q 412 531 435 507 q 355 569 389 555 q 276 583 321 583 q 240 574 260 583 q 201 548 219 565 q 171 505 183 531 q 160 446 160 479 q 170 378 160 410 q 204 322 181 346 q 260 284 226 298 q 342 271 294 271 q 382 279 361 271 q 420 305 403 287 q 448 347 437 322 q 459 406 459 372 m 344 -2 q 297 3 319 0 q 257 10 276 6 q 182 -36 210 -15 q 141 -74 155 -57 q 123 -104 127 -90 q 120 -128 120 -118 q 138 -182 120 -157 q 188 -226 157 -208 q 260 -256 219 -245 q 347 -268 301 -268 q 428 -255 391 -268 q 490 -222 464 -243 q 530 -171 516 -200 q 545 -106 545 -141 q 536 -71 545 -87 q 505 -42 527 -55 q 443 -19 482 -29 q 344 -2 405 -9 m 552 434 q 529 339 552 382 q 468 265 506 296 q 384 217 431 234 q 288 200 337 200 l 285 200 q 241 154 253 172 q 230 132 230 136 q 236 116 230 124 q 262 101 243 109 q 315 87 281 94 q 400 74 348 80 q 520 50 472 66 q 595 12 567 33 q 634 -33 623 -8 q 645 -81 645 -57 q 631 -152 645 -118 q 593 -214 617 -185 q 536 -265 569 -242 q 466 -305 504 -288 q 389 -330 429 -321 q 310 -339 349 -339 q 246 -334 279 -339 q 180 -320 212 -330 q 118 -296 147 -311 q 67 -261 89 -282 q 32 -214 45 -240 q 20 -155 20 -188 q 26 -118 20 -137 q 51 -76 32 -99 q 105 -28 71 -54 q 197 28 140 -2 q 137 63 154 44 q 121 103 121 83 q 124 118 121 109 q 138 140 127 127 q 167 170 148 153 q 216 209 186 187 q 155 236 183 218 q 108 280 128 254 q 77 337 88 305 q 67 408 67 370 q 88 502 67 457 q 146 579 110 546 q 228 631 182 612 q 322 651 274 651 q 397 639 362 651 q 460 606 432 627 q 523 615 495 610 q 573 627 551 621 q 613 639 595 633 q 644 651 630 645 l 657 630 q 642 595 649 611 q 619 562 635 579 q 570 554 594 557 q 515 552 545 552 q 542 496 532 525 q 552 434 552 467 m 368 726 l 297 726 l 121 968 q 129 979 126 974 q 136 987 133 984 q 142 993 139 991 q 151 999 146 996 l 334 829 l 513 999 q 529 987 525 993 q 542 968 534 981 l 368 726 "
    },
    "È«": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 555 855 q 549 819 555 836 q 532 790 543 802 q 508 770 522 777 q 477 763 494 763 q 434 779 447 763 q 422 827 422 796 q 427 863 422 846 q 444 893 433 880 q 468 912 455 905 q 498 920 482 920 q 555 855 555 920 m 289 855 q 283 819 289 836 q 266 790 277 802 q 242 770 256 777 q 211 763 228 763 q 168 779 181 763 q 156 827 156 796 q 161 863 156 846 q 178 893 167 880 q 202 912 189 905 q 232 920 216 920 q 289 855 289 920 m 586 1112 q 582 1095 585 1105 q 574 1075 578 1085 q 567 1056 570 1066 q 560 1041 563 1047 l 152 1041 l 137 1057 q 142 1074 138 1065 q 149 1094 145 1084 q 157 1113 153 1103 q 164 1129 161 1122 l 572 1129 l 586 1112 "
    },
    "á¹•": {
      "x_min": 37,
      "x_max": 669,
      "ha": 722,
      "o": "m 579 288 q 565 398 579 347 q 529 486 551 449 q 476 544 506 523 q 414 566 446 566 q 383 558 402 566 q 339 533 364 551 q 287 486 315 515 q 230 413 259 456 l 230 144 q 285 106 259 121 q 333 83 311 91 q 375 71 355 74 q 413 68 395 68 q 479 82 449 68 q 531 124 509 96 q 566 193 553 152 q 579 288 579 234 m 669 333 q 659 253 669 294 q 631 172 649 211 q 590 97 614 133 q 538 36 566 62 q 478 -4 509 10 q 414 -20 447 -20 q 326 2 375 -20 q 230 66 277 24 l 230 -253 q 254 -275 230 -264 q 342 -295 279 -286 l 342 -325 l 37 -325 l 37 -295 q 105 -275 80 -285 q 130 -253 130 -265 l 130 482 q 127 524 130 507 q 115 550 124 540 q 87 564 105 560 q 37 569 68 569 l 37 596 q 80 607 59 601 q 120 619 100 613 q 158 633 139 625 q 196 651 177 641 l 222 627 l 222 492 q 292 563 259 533 q 356 612 326 593 q 413 641 387 631 q 459 651 440 651 q 543 629 504 651 q 609 566 581 607 q 653 466 637 525 q 669 333 669 407 m 323 709 q 302 717 311 712 q 285 728 292 722 l 441 1009 q 462 1007 449 1009 q 489 1003 475 1005 q 517 998 503 1001 q 537 994 530 996 l 553 967 l 323 709 "
    },
    "áº®": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 838 q 390 869 367 855 q 430 892 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 614 1139 q 569 1046 594 1084 q 516 985 544 1009 q 459 952 489 962 q 399 942 429 942 q 336 952 366 942 q 277 985 305 962 q 226 1046 250 1009 q 181 1139 201 1084 q 197 1157 189 1151 q 216 1169 205 1164 q 255 1104 233 1131 q 300 1062 276 1078 q 349 1039 324 1046 q 396 1032 374 1032 q 445 1039 419 1032 q 495 1062 471 1046 q 541 1104 519 1078 q 579 1169 563 1131 q 598 1157 590 1164 q 614 1139 606 1151 m 310 1143 q 293 1155 298 1148 q 280 1174 288 1161 l 537 1389 q 555 1378 544 1385 q 579 1365 567 1372 q 600 1351 590 1358 q 616 1339 610 1344 l 622 1308 l 310 1143 "
    },
    "Ã£": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 545 912 q 517 857 533 884 q 480 808 500 830 q 436 773 460 787 q 386 760 412 760 q 341 771 362 760 q 300 798 320 783 q 261 824 281 812 q 222 836 241 836 q 173 814 195 836 q 127 754 152 792 l 92 768 q 120 823 104 795 q 157 872 136 850 q 201 907 177 893 q 251 921 225 921 q 298 909 276 921 q 341 883 321 897 q 379 856 361 868 q 413 845 397 845 q 463 866 438 845 q 510 927 487 888 l 545 912 "
    },
    "ÆŠ": {
      "x_min": 20,
      "x_max": 899,
      "ha": 959,
      "o": "m 666 748 q 617 781 642 767 q 559 802 591 794 q 485 814 527 810 q 389 818 444 818 l 389 104 q 395 80 389 89 q 431 62 404 68 q 509 57 458 57 q 604 79 555 57 q 693 148 653 102 q 760 265 734 195 q 786 432 786 336 q 779 523 786 478 q 758 609 772 568 q 720 686 743 650 q 666 748 698 721 m 20 679 q 50 766 20 730 q 136 826 81 803 q 268 860 191 849 q 437 872 344 872 q 533 872 491 872 q 611 864 576 872 q 674 846 646 857 q 728 818 703 835 q 807 748 775 788 q 860 663 840 709 q 890 566 881 617 q 899 460 899 515 q 882 308 899 375 q 836 189 865 241 q 770 102 807 138 q 691 43 732 66 q 606 10 649 21 q 524 0 563 0 l 186 0 l 186 29 q 257 49 231 38 q 283 70 283 61 l 283 812 q 166 774 203 802 q 129 702 129 746 q 141 659 129 678 q 174 630 154 640 q 162 619 172 626 q 137 604 152 611 q 106 590 122 596 q 76 580 90 583 q 35 616 51 591 q 20 679 20 642 "
    },
    "Ã¦": {
      "x_min": 53,
      "x_max": 889,
      "ha": 942,
      "o": "m 889 378 q 855 352 875 364 q 814 332 835 340 l 515 332 q 514 308 514 320 l 514 283 q 524 206 514 244 q 554 137 534 168 q 603 88 574 107 q 669 70 632 70 q 709 72 690 70 q 749 83 728 74 q 797 111 771 93 q 858 160 823 129 q 873 147 867 156 q 883 133 880 138 q 810 54 843 84 q 746 7 777 24 q 687 -14 716 -8 q 627 -20 658 -20 q 569 -10 598 -20 q 516 17 541 0 q 471 62 491 35 q 437 122 451 88 q 375 60 408 86 q 308 15 341 33 q 245 -11 275 -2 q 194 -20 215 -20 q 145 -11 170 -20 q 99 13 120 -3 q 66 59 79 31 q 53 126 53 87 q 68 203 53 166 q 115 272 84 240 q 194 330 146 304 q 307 375 242 356 q 357 386 329 382 q 414 390 384 390 l 414 448 q 410 499 414 474 q 394 542 406 523 q 363 571 383 560 q 310 583 342 583 q 264 573 289 583 q 221 546 240 563 q 193 506 202 529 q 193 458 183 483 q 181 445 195 452 q 147 431 167 437 q 108 422 128 425 q 80 421 88 419 l 71 445 q 90 498 74 472 q 130 546 106 523 q 184 588 154 569 q 244 621 213 607 q 303 643 274 635 q 355 651 332 651 q 448 626 413 651 q 497 553 482 602 q 542 599 518 580 q 590 629 566 618 q 638 645 615 640 q 680 651 661 651 q 752 639 721 651 q 806 607 783 627 q 845 560 829 587 q 871 503 861 533 q 885 440 881 473 q 889 378 889 408 m 423 165 q 416 205 418 185 q 414 248 414 225 l 414 329 q 374 326 392 329 q 349 321 357 324 q 205 253 249 296 q 162 155 162 211 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 q 274 74 251 68 q 321 92 296 80 q 372 122 346 104 q 423 165 398 141 m 660 583 q 625 575 645 583 q 585 547 605 568 q 547 489 565 526 q 521 394 529 452 l 752 394 q 772 399 766 394 q 778 418 778 404 q 771 480 778 450 q 750 532 764 510 q 714 569 736 555 q 660 583 692 583 "
    },
    "Ä©": {
      "x_min": -36.734375,
      "x_max": 416.75,
      "ha": 370,
      "o": "m 47 0 l 47 29 q 115 49 92 38 q 139 70 139 61 l 139 454 q 139 510 139 488 q 130 543 139 531 q 101 560 121 555 q 47 569 82 566 l 47 596 q 90 606 67 600 q 136 619 113 612 q 179 634 158 626 q 217 651 200 642 l 240 651 l 240 70 q 262 50 240 62 q 333 29 284 38 l 333 0 l 47 0 m 416 913 q 388 858 404 885 q 351 809 371 830 q 307 773 331 787 q 257 760 283 760 q 213 771 234 760 q 172 798 192 783 q 132 825 152 813 q 93 837 113 837 q 45 814 67 837 q -1 754 23 792 l -36 768 q -8 823 -24 796 q 28 872 7 851 q 72 908 49 894 q 122 922 96 922 q 170 910 147 922 q 212 883 192 898 q 250 856 232 868 q 284 845 268 845 q 334 866 309 845 q 380 928 358 888 l 416 913 "
    },
    "~": {
      "x_min": 32.5625,
      "x_max": 631.375,
      "ha": 664,
      "o": "m 631 524 q 596 455 617 490 q 548 390 574 419 q 492 341 522 360 q 430 323 461 323 q 370 339 402 323 q 307 377 339 356 q 243 414 275 397 q 183 431 212 431 q 123 404 150 431 q 67 321 95 378 l 32 339 q 68 408 46 373 q 115 473 89 444 q 172 522 142 503 q 233 541 202 541 q 296 524 264 541 q 360 487 328 507 q 422 449 392 466 q 479 433 452 433 q 539 459 511 433 q 596 541 568 485 l 631 524 "
    },
    "ÄŠ": {
      "x_min": 47,
      "x_max": 677.03125,
      "ha": 730,
      "o": "m 677 143 q 595 65 634 97 q 520 15 556 34 q 449 -11 484 -3 q 379 -20 414 -20 q 258 8 318 -20 q 152 90 199 36 q 75 221 104 144 q 47 397 47 299 q 79 594 47 506 q 166 744 111 682 q 294 841 221 807 q 449 875 368 875 q 575 855 521 875 q 664 807 629 835 q 662 793 668 803 q 646 770 656 783 q 624 744 635 757 q 604 723 612 731 l 581 727 q 500 780 547 759 q 393 800 454 800 q 344 791 371 800 q 290 765 317 783 q 237 717 263 746 q 192 645 212 687 q 161 548 173 603 q 150 422 150 493 q 175 264 150 332 q 241 151 201 196 q 331 83 281 106 q 427 61 380 61 q 521 86 463 61 q 652 173 579 111 q 659 167 655 172 q 666 158 663 163 q 672 149 669 153 q 677 143 675 145 m 475 1045 q 469 1009 475 1026 q 452 980 463 992 q 428 960 442 967 q 397 953 414 953 q 354 969 367 953 q 342 1017 342 986 q 347 1053 342 1036 q 364 1083 353 1070 q 388 1102 375 1095 q 418 1110 402 1110 q 475 1045 475 1110 "
    },
    "Â¡": {
      "x_min": 102,
      "x_max": 260,
      "ha": 370,
      "o": "m 229 -327 q 184 -359 207 -344 q 142 -382 161 -374 l 129 -368 l 129 359 q 159 375 146 370 q 195 386 172 380 l 229 373 l 229 -327 m 260 575 q 252 532 260 552 q 232 496 245 512 q 203 472 220 481 q 168 464 187 464 q 118 484 134 464 q 102 542 102 504 q 109 585 102 565 q 128 620 116 605 q 157 643 140 635 q 194 652 174 652 q 244 631 229 652 q 260 575 260 611 "
    },
    "áº…": {
      "x_min": 13,
      "x_max": 947,
      "ha": 961,
      "o": "m 947 600 q 914 591 926 596 q 895 582 902 587 q 885 572 888 578 q 880 558 882 566 l 733 39 q 718 13 729 24 q 692 -3 706 3 q 665 -14 678 -10 q 642 -21 651 -18 l 484 438 l 348 39 q 331 13 342 23 q 307 -4 320 2 q 281 -15 294 -11 q 259 -21 268 -18 l 85 558 q 13 600 79 585 l 13 630 l 270 630 l 270 600 q 218 593 237 597 q 192 582 200 588 q 184 571 184 577 q 186 558 184 564 l 311 128 l 475 630 l 521 630 l 694 128 l 808 558 q 794 583 812 574 q 731 600 776 593 l 731 630 l 947 630 l 947 600 m 698 854 q 692 818 698 835 q 675 789 686 801 q 651 769 665 776 q 620 762 637 762 q 577 778 590 762 q 565 826 565 795 q 570 862 565 845 q 587 892 576 879 q 611 911 598 904 q 641 919 625 919 q 698 854 698 919 m 432 854 q 426 818 432 835 q 409 789 420 801 q 385 769 399 776 q 354 762 371 762 q 311 778 324 762 q 299 826 299 795 q 304 862 299 845 q 321 892 310 879 q 345 911 332 904 q 375 919 359 919 q 432 854 432 919 "
    },
    "áº­": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 67 q 318 88 273 67 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 233 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 70 200 74 q 233 67 225 67 m 615 55 q 522 -2 559 15 q 466 -20 485 -20 q 429 10 444 -20 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -13 267 -6 q 194 -20 215 -20 q 145 -11 170 -20 q 99 13 120 -3 q 66 59 79 31 q 53 126 53 87 q 70 212 53 177 q 112 272 88 246 q 149 302 128 288 q 202 330 169 317 q 286 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 280 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 165 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 372 -189 q 366 -225 372 -208 q 349 -254 360 -242 q 325 -274 339 -267 q 294 -282 311 -282 q 251 -265 264 -282 q 239 -217 239 -248 q 244 -181 239 -198 q 261 -151 250 -164 q 285 -132 272 -139 q 315 -125 299 -125 q 372 -189 372 -125 m 529 740 q 516 721 521 727 q 500 710 512 715 l 318 891 l 138 710 q 129 715 132 712 q 122 721 125 718 q 116 729 119 724 q 108 740 112 734 l 283 997 l 354 997 l 529 740 "
    },
    "Ç¡": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -20 485 -20 q 429 10 444 -20 q 415 97 415 42 q 353 44 383 66 q 295 7 323 21 q 241 -13 266 -6 q 194 -20 215 -20 q 145 -12 170 -20 q 99 13 120 -3 q 66 59 79 31 q 53 126 53 86 q 70 212 53 177 q 112 271 88 246 q 148 302 128 288 q 202 330 169 317 q 286 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 280 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 165 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 385 854 q 379 818 385 835 q 362 789 373 801 q 338 769 352 776 q 307 762 324 762 q 264 778 277 762 q 252 826 252 795 q 257 862 252 845 q 274 892 263 879 q 298 911 285 904 q 328 919 312 919 q 385 854 385 919 m 550 1112 q 545 1095 549 1105 q 538 1075 542 1085 q 530 1056 534 1066 q 524 1041 527 1047 l 115 1041 l 100 1057 q 105 1074 102 1065 q 112 1094 108 1084 q 120 1113 116 1103 q 128 1129 124 1122 l 535 1129 l 550 1112 "
    },
    "á¹": {
      "x_min": 37,
      "x_max": 1075,
      "ha": 1094,
      "o": "m 789 0 l 789 29 q 858 51 836 42 q 881 70 881 61 l 881 429 q 876 498 881 470 q 863 541 872 525 q 840 563 854 557 q 805 570 825 570 q 759 557 783 570 q 708 521 734 544 q 656 464 682 498 q 605 388 630 430 l 605 70 q 625 52 605 61 q 699 29 646 43 l 699 0 l 413 0 l 413 29 q 483 51 461 42 q 505 70 505 61 l 505 429 q 501 498 505 470 q 488 541 497 525 q 464 563 479 557 q 430 570 450 570 q 335 522 385 570 q 230 388 286 475 l 230 70 q 254 49 230 60 q 323 29 279 39 l 323 0 l 37 0 l 37 29 q 105 50 80 41 q 130 70 130 59 l 130 482 q 127 525 130 509 q 115 549 125 540 q 87 561 106 557 q 37 570 68 565 l 37 597 q 83 606 62 601 q 124 619 104 612 q 161 634 143 626 q 196 651 178 642 l 223 627 l 223 472 q 287 550 257 517 q 348 606 318 584 q 406 639 378 628 q 457 651 434 651 q 516 643 489 651 q 563 616 543 635 q 594 567 582 598 q 605 492 605 537 l 605 477 q 663 552 633 520 q 722 606 693 584 q 780 639 752 628 q 833 651 808 651 q 892 642 865 651 q 938 615 919 633 q 969 568 958 596 q 981 502 981 540 l 981 70 q 1001 52 981 61 q 1075 29 1022 43 l 1075 0 l 789 0 m 622 855 q 616 819 622 836 q 599 790 610 802 q 575 770 589 777 q 544 763 561 763 q 501 779 514 763 q 489 827 489 796 q 494 863 489 846 q 511 893 500 880 q 535 912 522 905 q 565 920 549 920 q 622 855 622 920 "
    },
    "á»¬": {
      "x_min": 33,
      "x_max": 930,
      "ha": 930,
      "o": "m 930 944 q 921 904 930 927 q 892 856 913 881 q 837 803 872 830 q 751 749 803 775 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 804 855 q 816 889 816 873 q 806 926 816 909 q 777 959 796 944 l 895 1014 q 920 981 910 999 q 930 944 930 963 m 561 1121 q 548 1087 561 1102 q 520 1059 536 1073 q 487 1034 503 1046 q 465 1010 472 1022 q 464 984 457 997 q 498 953 470 970 q 478 945 489 947 q 457 943 467 944 q 413 970 427 957 q 396 993 399 982 q 399 1014 393 1004 q 417 1032 405 1023 q 442 1050 428 1041 q 468 1068 456 1059 q 488 1088 480 1078 q 497 1111 497 1099 q 484 1150 497 1137 q 452 1163 471 1163 q 434 1159 442 1163 q 419 1149 425 1155 q 410 1135 413 1143 q 407 1121 407 1128 q 408 1113 407 1117 q 412 1106 410 1110 q 398 1102 407 1104 q 380 1098 390 1100 q 362 1094 371 1095 q 346 1091 352 1092 l 339 1099 l 339 1108 q 351 1140 339 1124 q 383 1168 364 1155 q 427 1188 403 1180 q 474 1196 450 1196 q 538 1175 515 1196 q 561 1121 561 1155 "
    },
    "P": {
      "x_min": 27.015625,
      "x_max": 652.015625,
      "ha": 711,
      "o": "m 33 0 l 33 29 q 104 50 78 38 q 130 70 130 61 l 130 806 q 80 799 105 803 q 33 791 56 796 l 27 833 q 96 851 58 843 q 174 865 133 859 q 256 871 214 871 q 338 871 298 871 q 468 855 410 871 q 566 809 525 840 q 629 735 607 779 q 652 632 652 690 q 639 548 652 586 q 604 479 626 509 q 555 427 583 449 q 497 391 527 406 q 436 370 466 377 q 380 363 406 363 q 273 383 318 363 l 258 434 q 311 417 286 421 q 360 414 337 414 q 424 426 391 414 q 485 462 457 438 q 531 522 513 487 q 549 606 549 558 q 529 702 549 662 q 476 767 509 741 q 397 804 442 792 q 300 817 352 817 q 268 817 284 817 q 236 817 251 817 l 236 70 q 241 61 236 66 q 259 51 246 57 q 294 40 272 46 q 352 29 317 35 l 352 0 l 33 0 "
    },
    "%": {
      "x_min": 51,
      "x_max": 889,
      "ha": 940,
      "o": "m 794 196 q 785 278 794 243 q 763 337 777 313 q 731 371 749 360 q 694 383 713 383 q 665 373 679 383 q 640 343 651 363 q 621 294 628 324 q 615 225 615 265 q 622 140 615 176 q 642 81 629 104 q 673 46 655 58 q 712 35 691 35 q 743 45 729 35 q 769 75 758 55 q 787 126 780 95 q 794 196 794 156 m 889 209 q 873 121 889 162 q 830 48 857 79 q 768 -1 804 16 q 694 -20 733 -20 q 623 -1 656 -20 q 567 48 591 16 q 531 121 544 79 q 518 209 518 162 q 534 298 518 256 q 577 370 550 339 q 639 420 604 402 q 712 438 674 438 q 784 420 752 438 q 840 371 817 402 q 876 298 863 340 q 889 209 889 257 m 238 11 q 220 2 232 7 q 195 -7 209 -3 q 169 -15 182 -11 q 148 -22 156 -20 l 132 1 l 701 805 q 744 825 720 817 q 790 839 769 833 l 809 816 l 238 11 m 328 595 q 319 678 328 643 q 297 736 311 713 q 265 770 283 759 q 227 782 247 782 q 198 772 213 782 q 172 742 184 762 q 154 693 161 723 q 147 624 147 664 q 154 539 147 575 q 174 480 161 503 q 206 445 187 457 q 245 434 224 434 q 276 444 261 434 q 303 474 291 454 q 321 525 314 494 q 328 595 328 555 m 424 608 q 408 520 424 561 q 365 447 392 478 q 302 397 338 415 q 227 379 267 379 q 156 397 188 379 q 100 447 124 415 q 63 520 76 478 q 51 608 51 561 q 66 697 51 655 q 109 769 82 738 q 172 818 136 800 q 245 837 207 837 q 318 819 285 837 q 374 769 350 801 q 411 697 398 738 q 424 608 424 656 "
    },
    "Æ·": {
      "x_min": 62,
      "x_max": 648,
      "ha": 708,
      "o": "m 648 294 q 621 164 648 222 q 549 65 594 106 q 445 2 504 24 q 323 -20 386 -20 q 218 -4 266 -20 q 135 30 170 10 q 81 71 100 51 q 62 103 62 92 q 71 119 62 108 q 94 143 80 131 q 124 166 108 156 q 153 179 140 177 q 189 132 167 154 q 237 95 211 111 q 292 70 264 79 q 349 61 321 61 q 427 72 391 61 q 490 108 464 84 q 532 170 517 132 q 548 259 548 208 q 534 338 548 300 q 496 405 521 376 q 434 451 470 434 q 353 469 399 469 q 297 460 323 469 q 238 435 270 451 l 206 418 l 204 420 q 193 432 200 424 q 183 447 186 441 l 183 448 l 183 449 l 183 449 l 457 787 l 203 787 q 180 777 193 787 q 156 751 168 768 q 134 709 144 734 q 117 653 124 684 l 79 661 l 99 865 q 125 858 113 861 q 150 855 138 855 q 179 855 163 855 l 594 855 l 611 825 l 379 533 q 401 536 391 535 q 423 538 412 538 q 506 522 466 538 q 578 475 547 506 q 629 399 610 445 q 648 294 648 354 "
    },
    "_": {
      "x_min": 40.546875,
      "x_max": 596.15625,
      "ha": 639,
      "o": "m 596 -105 q 587 -137 592 -120 q 576 -167 582 -154 l 57 -167 l 40 -148 q 49 -118 43 -135 q 61 -89 55 -102 l 580 -89 l 596 -105 "
    },
    "Ã±": {
      "x_min": 37,
      "x_max": 732,
      "ha": 752,
      "o": "m 446 0 l 446 29 q 516 51 493 42 q 539 70 539 61 l 539 429 q 534 496 539 470 q 519 537 529 522 q 493 557 509 552 q 453 563 477 563 q 408 552 433 563 q 353 520 382 542 q 293 461 325 497 q 230 372 261 425 l 230 70 q 254 49 230 60 q 323 29 279 38 l 323 0 l 37 0 l 37 29 q 105 50 80 40 q 130 70 130 59 l 130 482 q 128 524 130 508 q 117 548 126 539 q 89 561 108 557 q 37 569 70 565 l 37 596 q 120 618 82 604 q 196 651 159 632 l 223 627 l 223 458 q 291 539 256 504 q 362 599 327 575 q 432 637 398 624 q 492 651 466 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 70 q 659 52 639 61 q 732 29 679 42 l 732 0 l 446 0 m 611 913 q 582 858 599 885 q 545 809 565 830 q 501 773 525 787 q 451 760 477 760 q 407 771 428 760 q 366 798 386 783 q 326 825 346 813 q 287 837 307 837 q 239 814 261 837 q 192 754 217 792 l 157 768 q 185 823 169 796 q 222 872 201 851 q 266 908 243 894 q 316 922 290 922 q 364 910 341 922 q 406 883 386 898 q 444 856 426 868 q 478 845 462 845 q 528 866 503 845 q 575 928 552 888 l 611 913 "
    },
    "Å”": {
      "x_min": 27.015625,
      "x_max": 756.890625,
      "ha": 764,
      "o": "m 33 0 l 33 29 q 104 49 78 38 q 130 70 130 60 l 130 806 q 82 799 106 803 q 33 792 57 795 l 27 834 q 90 849 58 842 q 156 862 122 856 q 229 871 191 868 q 311 875 267 875 q 445 859 388 875 q 540 816 502 843 q 597 750 578 788 q 616 669 616 712 q 602 581 616 620 q 563 510 588 541 q 504 458 538 480 q 428 424 469 437 l 629 94 q 651 69 639 78 q 677 56 663 60 q 709 52 691 52 q 749 54 726 52 l 756 25 q 673 0 713 10 q 608 -9 633 -9 q 570 1 588 -9 q 542 27 552 12 l 341 408 q 324 407 332 407 l 306 407 q 271 408 288 407 q 236 414 254 410 l 236 70 q 259 50 236 62 q 332 29 283 38 l 332 0 l 33 0 m 287 817 q 236 817 261 817 l 236 469 q 267 465 253 465 q 295 465 280 465 q 456 511 400 465 q 512 648 512 558 q 500 716 512 685 q 459 769 487 747 q 389 804 431 792 q 287 817 346 817 m 257 927 q 241 938 246 931 q 228 957 236 944 l 485 1172 q 503 1162 491 1168 q 526 1148 514 1155 q 548 1134 538 1141 q 563 1122 558 1127 l 569 1092 l 257 927 "
    },
    "â€š": {
      "x_min": 58.59375,
      "x_max": 251,
      "ha": 312,
      "o": "m 251 41 q 242 -15 251 15 q 215 -76 233 -46 q 173 -135 198 -107 q 117 -184 149 -163 l 87 -161 q 111 -127 101 -146 q 129 -88 122 -107 q 139 -51 136 -69 q 143 -20 143 -33 q 125 33 143 10 q 69 56 107 56 l 58 87 q 83 106 64 95 q 126 126 103 116 q 171 142 149 136 q 205 149 193 149 q 239 104 227 127 q 251 41 251 80 "
    },
    "Ã†": {
      "x_min": 0,
      "x_max": 1020.84375,
      "ha": 1060,
      "o": "m 515 778 q 508 788 515 785 q 493 791 502 792 q 476 784 484 790 q 462 766 467 778 l 357 498 l 515 498 l 515 778 m 1020 165 q 1006 63 1014 106 q 992 0 997 19 l 419 0 l 419 29 q 489 49 464 38 q 515 70 515 60 l 515 441 l 335 441 l 189 75 q 201 45 180 55 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 70 46 45 37 q 104 75 96 55 l 379 767 q 376 787 383 779 q 353 801 369 795 q 310 813 336 807 q 253 824 285 818 l 253 854 l 955 854 l 975 837 q 972 798 975 819 q 966 756 970 777 q 958 716 962 735 q 948 684 953 697 l 918 684 q 914 736 918 715 q 903 770 910 757 q 887 787 896 782 q 864 793 877 793 l 621 793 l 621 498 l 890 498 l 908 479 q 894 458 902 469 q 879 437 887 447 q 862 417 870 426 q 846 403 854 409 q 825 420 836 413 q 799 432 814 427 q 763 438 783 436 q 714 441 742 441 l 621 441 l 621 104 q 625 86 621 94 q 645 72 630 78 q 687 64 660 67 q 758 61 714 61 l 844 61 q 896 64 874 61 q 934 79 917 67 q 963 114 950 91 q 990 177 976 137 l 1020 165 "
    },
    "á¹": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 581 913 q 553 858 569 885 q 516 809 536 830 q 472 773 496 787 q 422 760 448 760 q 378 771 399 760 q 337 798 357 783 q 297 825 317 813 q 258 837 278 837 q 210 814 232 837 q 163 754 188 792 l 128 768 q 157 823 140 796 q 193 872 173 851 q 238 908 214 894 q 287 922 261 922 q 335 910 312 922 q 377 883 357 898 q 415 856 397 868 q 449 845 433 845 q 499 866 474 845 q 545 928 523 888 l 581 913 m 326 954 q 305 962 314 956 q 288 973 296 967 l 444 1254 q 465 1251 452 1253 q 492 1248 478 1250 q 520 1243 506 1246 q 540 1238 533 1240 l 555 1212 l 326 954 "
    },
    "á¹®": {
      "x_min": 6.609375,
      "x_max": 719,
      "ha": 750,
      "o": "m 204 0 l 204 29 q 255 42 234 35 q 289 55 276 49 q 307 68 301 62 q 313 78 313 73 l 313 794 l 114 794 q 96 790 104 794 q 79 774 87 786 q 60 738 71 762 q 35 675 49 714 l 6 688 q 15 773 9 728 q 28 855 21 819 l 699 855 l 719 838 q 713 762 719 803 q 699 678 707 721 l 669 678 q 658 731 663 709 q 646 767 653 753 q 630 787 639 781 q 607 794 621 794 l 419 794 l 419 78 q 442 56 419 69 q 528 29 466 42 l 528 0 l 204 0 m 587 -155 q 582 -172 585 -162 q 575 -192 579 -182 q 567 -211 571 -201 q 561 -227 563 -220 l 152 -227 l 138 -210 q 142 -193 139 -202 q 150 -173 146 -183 q 158 -154 154 -164 q 165 -139 162 -145 l 572 -139 l 587 -155 "
    },
    "Åª": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 847 855 l 847 825 m 681 1058 q 676 1041 680 1051 q 669 1021 673 1031 q 661 1002 665 1012 q 655 987 658 993 l 246 987 l 232 1003 q 236 1020 233 1011 q 243 1040 239 1030 q 251 1059 247 1049 q 259 1075 255 1068 l 666 1075 l 681 1058 "
    },
    "Å’": {
      "x_min": 47,
      "x_max": 1046.859375,
      "ha": 1086,
      "o": "m 401 62 q 451 62 428 62 q 495 65 474 62 q 533 75 515 69 q 568 94 551 82 l 568 745 q 490 785 532 773 q 401 798 449 798 q 295 772 342 798 q 216 699 249 746 q 167 585 184 652 q 150 435 150 518 q 170 291 150 359 q 224 172 190 223 q 304 91 259 121 q 401 62 350 62 m 1046 165 q 1033 63 1041 106 q 1019 0 1024 19 l 585 0 q 524 -3 550 0 q 477 -10 499 -6 q 433 -16 455 -13 q 382 -20 411 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 155 737 105 674 q 274 837 206 800 q 421 875 342 875 q 467 872 447 875 q 504 865 486 869 q 542 858 522 861 q 588 855 562 855 l 981 855 l 1001 838 q 998 799 1001 820 q 992 757 996 778 q 984 717 988 736 q 975 685 979 698 l 944 685 q 940 737 944 716 q 929 770 936 757 q 913 787 922 782 q 890 793 903 793 l 676 793 l 676 499 l 916 499 l 935 480 q 921 460 929 470 q 905 439 913 449 q 888 421 896 430 q 874 407 880 413 q 852 424 864 417 q 826 436 841 431 q 790 442 811 440 q 742 445 770 445 l 676 445 l 676 105 q 676 87 676 95 q 686 73 677 79 q 718 65 695 68 q 784 62 741 62 l 870 62 q 922 65 900 62 q 960 80 943 68 q 990 115 976 92 q 1018 178 1003 138 l 1046 165 "
    },
    "áº ": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 387 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 839 q 390 869 367 855 q 430 893 413 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 465 -189 q 459 -225 465 -208 q 442 -254 453 -242 q 418 -274 432 -267 q 387 -282 404 -282 q 344 -265 357 -282 q 332 -217 332 -248 q 337 -181 332 -198 q 354 -151 343 -164 q 378 -132 365 -139 q 408 -125 392 -125 q 465 -189 465 -125 "
    },
    "Æ³": {
      "x_min": -0.265625,
      "x_max": 921,
      "ha": 921,
      "o": "m 239 0 l 239 29 q 323 55 299 42 q 348 77 348 68 l 348 363 q 292 478 323 418 q 228 594 260 538 q 165 699 195 650 q 111 780 134 747 q 97 794 105 787 q 78 805 90 800 q 48 813 66 810 q 2 817 30 817 l 0 846 q 78 856 38 852 q 145 861 117 861 q 197 834 175 861 q 250 757 221 801 q 307 663 278 712 q 363 561 335 614 q 413 460 390 509 l 572 745 q 609 803 590 776 q 651 851 627 831 q 704 883 675 871 q 774 895 734 895 q 832 882 805 895 q 878 850 859 870 q 909 802 898 829 q 921 746 921 776 q 918 717 921 732 q 913 693 916 701 q 894 675 909 684 q 860 658 879 666 q 822 643 841 649 q 794 634 804 637 l 782 654 q 798 688 791 672 q 805 729 805 705 q 801 762 805 746 q 788 788 797 777 q 768 807 780 800 q 739 814 755 814 q 692 792 713 814 q 656 742 672 770 l 454 366 l 454 77 q 459 68 454 73 q 478 55 465 62 q 512 42 491 48 q 565 29 533 35 l 565 0 l 239 0 "
    },
    "á¹¡": {
      "x_min": 63,
      "x_max": 465,
      "ha": 525,
      "o": "m 465 192 q 451 109 465 144 q 417 51 438 75 q 369 13 395 28 q 318 -7 343 0 q 271 -17 293 -15 q 236 -20 248 -20 q 159 -7 204 -20 q 70 29 114 4 q 64 51 65 31 q 63 97 63 71 q 66 150 63 123 q 74 192 69 176 l 104 184 q 117 131 104 156 q 153 87 130 106 q 205 58 175 69 q 269 48 234 48 q 313 55 293 48 q 348 77 333 63 q 370 111 362 91 q 379 154 379 130 q 364 202 379 181 q 326 240 350 223 q 273 273 303 258 q 213 304 244 288 q 159 335 185 319 q 113 371 133 351 q 81 417 93 392 q 69 474 69 442 q 85 549 69 516 q 129 604 102 582 q 192 639 157 627 q 266 651 228 651 q 311 647 287 651 q 361 636 336 643 q 406 620 385 629 q 441 598 427 611 q 442 580 445 594 q 432 549 439 566 q 419 517 426 532 q 408 497 412 502 l 382 502 q 315 570 350 551 q 248 590 281 590 q 209 582 226 590 q 179 563 192 575 q 161 537 167 551 q 155 508 155 522 q 167 469 155 487 q 200 437 180 452 q 247 408 221 421 q 302 379 274 394 q 359 347 330 364 q 412 308 389 330 q 450 258 435 286 q 465 192 465 230 m 330 855 q 324 819 330 836 q 307 790 318 802 q 283 770 297 777 q 252 763 269 763 q 209 779 222 763 q 197 827 197 796 q 202 863 197 846 q 219 893 208 880 q 243 912 230 905 q 273 920 257 920 q 330 855 330 920 "
    },
    "á»·": {
      "x_min": -31,
      "x_max": 658,
      "ha": 672,
      "o": "m 658 601 q 625 592 638 596 q 604 583 612 588 q 592 571 596 578 q 584 555 588 565 l 362 -55 q 300 -184 334 -130 q 229 -272 266 -237 q 155 -322 192 -306 q 84 -339 117 -339 q 38 -335 59 -339 q 1 -327 16 -332 q -22 -315 -13 -322 q -31 -303 -31 -309 q -22 -286 -31 -299 q -2 -260 -14 -274 q 22 -231 9 -245 q 47 -211 36 -218 q 110 -232 79 -232 q 166 -223 142 -232 q 194 -204 178 -219 q 226 -168 210 -189 q 258 -119 243 -146 q 286 -62 274 -91 l 305 -15 l 87 555 q 64 584 81 574 q 14 601 47 593 l 14 630 l 271 630 l 271 601 q 228 594 244 598 q 203 586 212 591 q 192 573 194 581 q 195 555 191 566 l 358 123 l 512 555 q 512 572 515 565 q 501 584 510 579 q 476 593 492 589 q 436 601 460 597 l 436 631 l 658 631 l 658 601 m 464 904 q 451 870 464 885 q 423 842 439 855 q 391 817 406 829 q 368 793 375 805 q 367 767 360 780 q 401 736 373 753 q 381 728 392 729 q 360 725 370 727 q 316 753 330 740 q 299 776 302 765 q 302 796 296 787 q 320 815 308 806 q 345 833 331 824 q 371 851 359 842 q 391 871 383 861 q 400 894 400 882 q 387 933 400 920 q 355 946 374 946 q 337 942 345 946 q 322 932 328 938 q 313 918 316 926 q 310 904 310 911 q 311 896 310 900 q 315 889 313 893 q 301 885 310 887 q 284 880 293 883 q 265 877 274 878 q 249 874 256 875 l 242 882 l 242 891 q 254 923 242 907 q 286 951 267 938 q 330 971 306 963 q 377 979 354 979 q 441 958 418 979 q 464 904 464 938 "
    },
    "â€º": {
      "x_min": 93.171875,
      "x_max": 361,
      "ha": 415,
      "o": "m 361 291 l 121 11 l 93 31 l 246 316 l 93 598 l 121 619 l 361 341 l 361 291 "
    },
    "<": {
      "x_min": 41,
      "x_max": 557.734375,
      "ha": 598,
      "o": "m 557 218 q 544 206 550 211 q 532 195 539 200 q 519 185 526 190 q 501 173 511 179 l 57 343 l 41 359 q 41 361 41 360 q 42 363 42 362 l 44 369 q 45 375 44 370 l 47 383 l 49 389 q 50 393 49 391 q 51 398 50 396 l 53 403 l 57 415 l 60 423 l 61 426 l 541 609 l 557 592 q 553 576 555 585 q 548 558 551 567 q 543 539 545 548 q 538 524 540 530 l 169 385 l 546 242 l 557 218 "
    },
    "Â¬": {
      "x_min": 40.15625,
      "x_max": 592,
      "ha": 639,
      "o": "m 592 157 q 565 138 578 147 q 534 123 551 128 l 518 140 l 518 343 l 56 343 l 40 359 q 49 389 43 372 q 60 419 54 405 l 572 419 l 592 404 l 592 157 "
    },
    "t": {
      "x_min": 13.3125,
      "x_max": 449.609375,
      "ha": 469,
      "o": "m 449 79 q 385 36 417 55 q 324 4 353 17 q 270 -14 295 -7 q 228 -20 246 -20 q 184 -11 205 -20 q 147 17 163 -2 q 122 69 131 37 q 113 149 113 102 l 113 566 l 27 566 l 13 584 l 65 630 l 113 630 l 113 796 l 192 867 l 213 850 l 213 630 l 430 630 l 449 610 q 434 590 443 601 q 416 569 425 579 q 398 552 407 560 q 382 542 389 544 q 333 558 366 550 q 247 566 301 566 l 213 566 l 213 208 q 216 140 213 167 q 228 97 220 113 q 251 74 237 81 q 284 68 264 68 q 343 77 308 68 q 430 114 378 86 l 449 79 "
    },
    "Ã¹": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 466 70 q 369 4 398 18 q 316 -14 340 -9 q 270 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 533 113 515 q 100 560 109 551 q 75 572 92 569 q 27 579 57 576 l 27 606 q 72 613 51 609 q 112 623 93 617 q 152 635 132 628 q 194 651 172 642 l 213 625 l 213 226 q 219 147 213 179 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 483 q 505 532 508 513 q 492 560 502 550 q 463 574 482 570 q 414 579 445 579 l 414 606 q 506 625 462 612 q 588 651 550 638 l 608 625 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 437 731 q 420 718 428 723 q 400 710 411 713 l 167 966 l 181 994 q 201 998 188 996 q 228 1003 214 1000 q 256 1007 242 1006 q 278 1010 270 1009 l 437 731 "
    },
    "È²": {
      "x_min": -0.265625,
      "x_max": 781,
      "ha": 808,
      "o": "m 239 0 l 239 29 q 323 55 299 42 q 348 78 348 68 l 348 364 q 292 478 323 419 q 228 594 260 538 q 165 699 195 651 q 111 780 134 748 q 97 794 105 788 q 78 806 90 801 q 48 814 66 811 q 2 818 30 818 l 0 846 q 78 856 38 852 q 145 861 117 861 q 197 834 175 861 q 250 757 221 802 q 307 663 278 713 q 363 562 335 614 q 413 461 390 510 l 592 780 q 585 808 602 798 q 518 825 569 818 l 518 854 l 781 854 l 781 825 q 711 807 735 816 q 677 780 686 798 l 454 366 l 454 78 q 459 68 454 73 q 478 55 465 62 q 512 42 491 48 q 565 29 533 35 l 565 0 l 239 0 m 632 1058 q 627 1041 631 1051 q 620 1021 624 1031 q 613 1002 616 1012 q 606 987 609 993 l 197 987 l 183 1003 q 187 1020 184 1011 q 195 1040 191 1030 q 203 1059 199 1049 q 210 1075 207 1068 l 618 1075 l 632 1058 "
    },
    "Ã¯": {
      "x_min": -10,
      "x_max": 389,
      "ha": 369,
      "o": "m 46 0 l 46 29 q 114 49 91 38 q 138 70 138 61 l 138 454 q 138 510 138 488 q 129 543 138 531 q 100 560 120 555 q 46 569 81 566 l 46 596 q 89 606 66 600 q 135 619 112 612 q 178 634 157 626 q 216 651 199 642 l 239 651 l 239 70 q 261 50 239 62 q 332 29 283 38 l 332 0 l 46 0 m 389 854 q 383 818 389 835 q 366 789 377 801 q 342 769 356 776 q 311 762 328 762 q 268 778 281 762 q 256 826 256 795 q 261 862 256 845 q 278 892 267 879 q 302 911 289 904 q 332 919 316 919 q 389 854 389 919 m 123 854 q 117 818 123 835 q 100 789 111 801 q 76 769 90 776 q 45 762 62 762 q 2 778 15 762 q -10 826 -10 795 q -4 862 -10 845 q 12 892 1 879 q 36 911 23 904 q 66 919 50 919 q 123 854 123 919 "
    },
    "Ã’": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 527 957 q 512 938 517 944 q 497 927 507 931 l 185 1092 l 190 1122 q 205 1134 195 1127 q 227 1148 215 1141 q 251 1162 239 1155 q 269 1172 262 1168 l 527 957 "
    },
    "ï‰‚": {
      "x_min": -40.109375,
      "x_max": 627.09375,
      "ha": 648,
      "o": "m 627 165 q 613 63 621 106 q 599 0 605 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 400 q 109 409 119 406 q 90 413 99 413 q 41 390 63 413 q -4 330 19 368 l -40 344 q -11 399 -28 372 q 25 448 4 427 q 69 484 45 470 q 119 498 93 498 q 124 497 121 498 q 130 497 126 497 l 130 783 q 105 803 130 791 q 33 825 81 816 l 33 854 l 332 854 l 332 825 q 261 804 286 816 q 236 783 236 792 l 236 440 q 259 426 247 431 q 281 421 270 421 q 330 442 306 421 q 377 504 354 464 l 413 488 q 384 433 401 461 q 347 384 367 406 q 303 349 327 363 q 253 336 279 336 q 236 338 244 336 l 236 111 q 241 89 236 98 q 261 74 247 80 q 299 64 275 67 q 359 61 322 61 l 454 61 q 505 64 484 61 q 542 79 526 67 q 571 114 558 91 q 598 177 584 137 l 627 165 "
    },
    "áº§": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 529 740 q 516 721 521 727 q 500 710 512 715 l 318 891 l 138 710 q 129 715 132 712 q 122 721 125 718 q 116 729 119 724 q 108 740 112 734 l 283 997 l 354 997 l 529 740 m 394 1036 q 377 1023 386 1028 q 357 1015 368 1018 l 123 1270 l 138 1298 q 157 1303 144 1300 q 184 1308 170 1305 q 213 1312 199 1310 q 235 1314 227 1314 l 394 1036 "
    },
    "á¹ª": {
      "x_min": 6.609375,
      "x_max": 719,
      "ha": 750,
      "o": "m 204 0 l 204 28 q 255 41 234 34 q 289 54 276 48 q 307 67 301 61 q 313 77 313 72 l 313 793 l 114 793 q 96 789 104 793 q 79 773 87 785 q 60 737 71 761 q 35 674 49 713 l 6 687 q 15 772 9 727 q 28 854 21 818 l 699 854 l 719 837 q 713 761 719 802 q 699 677 707 720 l 669 677 q 658 730 663 708 q 646 766 653 752 q 630 786 639 780 q 607 793 621 793 l 419 793 l 419 77 q 442 55 419 68 q 528 28 466 41 l 528 0 l 204 0 m 429 1044 q 423 1008 429 1025 q 406 979 417 991 q 382 959 396 966 q 351 952 368 952 q 308 968 321 952 q 296 1016 296 985 q 301 1052 296 1035 q 318 1082 307 1069 q 342 1101 329 1094 q 372 1109 356 1109 q 429 1044 429 1109 "
    },
    "á»’": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 619 957 q 606 938 611 944 q 590 927 602 932 l 408 1068 l 228 927 q 219 932 223 929 q 212 938 215 935 q 206 946 209 942 q 198 957 203 951 l 374 1167 l 445 1167 l 619 957 m 527 1228 q 512 1209 517 1216 q 497 1198 507 1203 l 185 1363 l 190 1393 q 205 1405 195 1399 q 227 1419 215 1412 q 251 1433 239 1426 q 269 1443 262 1439 l 527 1228 "
    },
    "I": {
      "x_min": 47,
      "x_max": 346,
      "ha": 406,
      "o": "m 47 0 l 47 29 q 118 49 92 38 q 144 70 144 61 l 144 782 q 119 803 144 790 q 47 824 95 815 l 47 854 l 346 854 l 346 824 q 275 803 300 815 q 250 782 250 792 l 250 70 q 274 50 250 62 q 346 29 298 38 l 346 0 l 47 0 "
    },
    "Ë": {
      "x_min": 53.171875,
      "x_max": 457.25,
      "ha": 510,
      "o": "m 91 709 q 73 716 79 712 q 53 728 66 720 l 170 1009 q 189 1006 178 1008 q 211 1002 200 1004 q 232 997 221 1000 q 249 992 242 995 l 263 964 l 91 709 m 284 709 q 265 716 271 712 q 246 728 258 720 l 363 1009 q 382 1006 371 1008 q 403 1002 392 1004 q 425 997 414 1000 q 442 992 435 995 l 457 964 l 284 709 "
    },
    "É™": {
      "x_min": 47,
      "x_max": 565,
      "ha": 618,
      "o": "m 295 51 q 412 103 370 51 q 463 258 454 156 l 197 258 q 164 242 176 258 q 153 199 153 226 q 165 140 153 167 q 198 93 178 113 q 244 62 218 73 q 295 51 269 51 m 565 329 q 538 180 565 249 q 466 61 512 111 q 382 0 428 19 q 276 -20 336 -20 q 187 -4 228 -20 q 114 40 145 11 q 64 106 82 68 q 47 189 47 144 q 53 239 47 216 q 70 276 59 261 q 113 298 90 287 q 159 319 136 309 l 464 319 q 447 419 464 374 q 404 495 431 464 q 341 543 377 526 q 264 560 304 560 q 227 557 246 560 q 187 546 209 554 q 138 523 165 538 q 78 482 111 507 q 62 496 69 486 q 52 511 56 506 q 127 582 92 555 q 192 625 161 610 q 252 646 222 641 q 311 652 281 652 q 405 629 360 652 q 486 566 451 607 q 543 464 522 525 q 565 329 565 404 "
    },
    "Â·": {
      "x_min": 33,
      "x_max": 166,
      "ha": 199,
      "o": "m 166 488 q 160 452 166 469 q 143 423 154 435 q 119 403 133 410 q 88 396 105 396 q 45 412 58 396 q 33 460 33 429 q 38 496 33 479 q 55 526 44 513 q 79 545 66 538 q 109 553 93 553 q 166 488 166 553 "
    },
    "á¹œ": {
      "x_min": 27.015625,
      "x_max": 756.890625,
      "ha": 764,
      "o": "m 33 0 l 33 29 q 104 49 78 38 q 130 70 130 60 l 130 806 q 82 799 106 803 q 33 792 57 795 l 27 834 q 90 849 58 842 q 156 862 122 856 q 229 871 191 868 q 311 875 267 875 q 445 859 388 875 q 540 816 502 843 q 597 750 578 788 q 616 669 616 712 q 602 581 616 620 q 563 510 588 541 q 504 458 538 480 q 428 424 469 437 l 629 94 q 651 69 639 78 q 677 56 663 60 q 709 52 691 52 q 749 54 726 52 l 756 25 q 673 0 713 10 q 608 -9 633 -9 q 570 1 588 -9 q 542 27 552 12 l 341 408 q 324 407 332 407 l 306 407 q 271 408 288 407 q 236 414 254 410 l 236 70 q 259 50 236 62 q 332 29 283 38 l 332 0 l 33 0 m 287 817 q 236 817 261 817 l 236 469 q 267 465 253 465 q 295 465 280 465 q 456 511 400 465 q 512 648 512 558 q 500 716 512 685 q 459 769 487 747 q 389 804 431 792 q 287 817 346 817 m 412 -189 q 406 -225 412 -208 q 389 -254 400 -242 q 365 -274 379 -267 q 334 -282 351 -282 q 291 -265 304 -282 q 279 -217 279 -248 q 284 -181 279 -198 q 301 -151 290 -164 q 325 -132 312 -139 q 355 -125 339 -125 q 412 -189 412 -125 m 576 1058 q 572 1041 575 1051 q 564 1021 568 1031 q 557 1002 560 1012 q 550 987 553 993 l 142 987 l 128 1003 q 132 1020 129 1011 q 139 1040 135 1030 q 147 1059 143 1049 q 155 1075 151 1068 l 562 1075 l 576 1058 "
    },
    "áº•": {
      "x_min": 50.453125,
      "x_max": 549.984375,
      "ha": 602,
      "o": "m 548 168 q 548 122 548 146 q 548 74 548 97 q 548 31 548 51 q 543 0 548 12 l 67 0 l 50 30 l 419 570 l 189 570 q 167 565 178 570 q 146 550 156 561 q 127 519 136 539 q 111 468 118 499 l 85 476 l 85 642 q 117 635 105 638 q 142 632 129 632 q 170 632 155 632 l 535 632 l 549 602 l 179 61 l 460 61 q 490 87 476 61 q 520 175 504 113 l 548 168 m 525 -155 q 520 -172 524 -162 q 513 -192 517 -182 q 505 -211 509 -201 q 499 -227 502 -220 l 90 -227 l 75 -210 q 80 -193 77 -202 q 87 -173 83 -183 q 95 -154 91 -164 q 102 -139 99 -145 l 510 -139 l 525 -155 "
    },
    "Â¿": {
      "x_min": 53,
      "x_max": 536,
      "ha": 589,
      "o": "m 278 359 q 301 376 290 370 q 331 387 312 381 l 352 373 l 352 294 q 339 217 352 255 q 306 142 327 179 q 261 68 286 104 q 214 -1 236 33 q 177 -70 192 -36 q 163 -136 163 -103 q 200 -267 163 -220 q 307 -315 237 -315 q 352 -303 331 -315 q 391 -274 374 -292 q 417 -231 407 -255 q 427 -180 427 -206 q 424 -160 427 -170 q 417 -142 421 -151 q 463 -122 440 -128 q 516 -117 486 -117 l 534 -136 q 536 -147 536 -142 l 536 -158 q 513 -246 536 -206 q 451 -316 490 -286 q 361 -362 412 -345 q 255 -379 311 -379 q 168 -363 206 -379 q 105 -319 130 -347 q 66 -252 79 -291 q 53 -166 53 -213 q 68 -75 53 -116 q 108 1 84 -34 q 159 68 131 36 q 214 135 187 101 q 259 208 240 170 q 278 294 278 246 l 278 359 m 389 575 q 381 532 389 552 q 361 496 374 512 q 331 472 348 481 q 296 464 314 464 q 246 483 262 464 q 231 541 231 503 q 237 584 231 564 q 257 619 244 604 q 286 642 269 634 q 321 651 302 651 q 373 631 357 651 q 389 575 389 611 "
    },
    "á»¨": {
      "x_min": 33,
      "x_max": 930,
      "ha": 930,
      "o": "m 930 944 q 921 904 930 927 q 892 856 913 882 q 837 803 872 830 q 751 749 803 775 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 804 855 q 816 889 816 873 q 806 926 816 909 q 777 960 796 944 l 895 1014 q 920 981 910 999 q 930 944 930 964 m 361 927 q 345 938 350 932 q 332 958 340 945 l 589 1173 q 607 1162 596 1168 q 630 1148 618 1156 q 652 1135 642 1141 q 668 1122 662 1128 l 674 1092 l 361 927 "
    },
    "Å±": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 466 70 q 369 4 398 18 q 316 -14 340 -9 q 270 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 533 113 515 q 100 560 109 551 q 75 572 92 569 q 27 579 57 576 l 27 606 q 72 613 51 609 q 112 623 93 617 q 152 635 132 628 q 194 651 172 642 l 213 625 l 213 226 q 219 147 213 179 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 483 q 505 532 508 513 q 492 560 502 550 q 463 574 482 570 q 414 579 445 579 l 414 606 q 506 625 462 612 q 588 651 550 638 l 608 625 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 254 710 q 236 716 242 713 q 216 728 229 720 l 332 1010 q 351 1007 340 1008 q 373 1003 362 1005 q 394 998 384 1000 q 411 993 405 996 l 426 965 l 254 710 m 446 710 q 427 716 434 713 q 409 728 420 720 l 526 1010 q 544 1007 534 1008 q 566 1003 555 1005 q 587 998 577 1000 q 605 993 598 996 l 619 965 l 446 710 "
    },
    "É–": {
      "x_min": 53,
      "x_max": 866.8125,
      "ha": 685,
      "o": "m 323 68 q 368 77 346 68 q 411 102 390 86 q 453 137 432 117 q 495 177 474 156 l 495 495 q 433 555 473 533 q 345 576 393 576 q 267 560 303 576 q 207 512 232 544 q 167 433 181 480 q 153 322 153 385 q 169 217 153 264 q 210 137 185 170 q 266 85 236 103 q 323 68 296 68 m 595 -112 q 613 -232 595 -190 q 679 -275 631 -275 q 714 -265 698 -275 q 739 -241 729 -255 q 750 -210 748 -227 q 743 -179 751 -193 q 754 -167 741 -176 q 786 -151 767 -159 q 824 -137 805 -143 q 853 -132 843 -131 l 866 -159 q 848 -216 866 -185 q 797 -274 830 -247 q 721 -320 765 -302 q 629 -339 677 -339 q 563 -323 589 -339 q 522 -283 537 -308 q 500 -224 506 -257 q 495 -153 495 -190 l 495 112 q 445 57 469 81 q 394 15 420 32 q 339 -10 368 -1 q 276 -20 310 -20 q 199 2 239 -20 q 127 65 160 24 q 73 166 94 106 q 53 301 53 226 q 63 381 53 339 q 92 461 73 422 q 140 534 112 499 q 202 595 167 568 q 278 636 237 621 q 366 651 319 651 q 428 643 397 651 q 493 608 459 635 l 493 864 q 493 924 493 902 q 483 958 493 946 q 454 974 474 970 q 399 979 435 979 l 399 1006 q 497 1026 454 1014 q 574 1050 539 1039 l 595 1030 l 595 -112 "
    },
    "á¹¸": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 847 855 l 847 825 m 676 1103 q 648 1048 664 1075 q 611 999 631 1020 q 566 963 590 977 q 516 950 542 950 q 472 961 493 950 q 431 988 451 973 q 392 1015 411 1003 q 353 1027 372 1027 q 304 1004 326 1027 q 258 944 283 982 l 223 958 q 251 1013 235 986 q 288 1062 267 1041 q 332 1098 309 1084 q 382 1112 356 1112 q 429 1100 407 1112 q 472 1073 452 1088 q 510 1046 492 1058 q 544 1035 528 1035 q 593 1056 569 1035 q 640 1118 618 1078 l 676 1103 m 361 1144 q 345 1155 350 1149 q 332 1175 340 1162 l 589 1390 q 607 1379 596 1385 q 630 1365 618 1373 q 652 1352 642 1358 q 668 1339 662 1345 l 674 1309 l 361 1144 "
    },
    "á¸Œ": {
      "x_min": 27.015625,
      "x_max": 746,
      "ha": 806,
      "o": "m 300 817 q 235 817 266 817 l 235 104 q 241 80 235 89 q 277 62 250 68 q 356 57 304 57 q 450 79 402 57 q 540 148 499 102 q 607 265 581 195 q 633 431 633 336 q 611 593 633 521 q 546 714 589 664 q 442 790 504 764 q 300 817 380 817 m 746 456 q 729 305 746 372 q 683 188 712 239 q 617 101 654 137 q 538 43 579 66 q 453 10 496 20 q 371 0 410 0 l 33 0 l 33 29 q 103 49 78 38 q 129 70 129 61 l 129 806 q 78 799 102 802 q 33 792 54 795 l 27 834 q 94 851 56 843 q 175 865 132 859 q 261 872 217 872 q 343 872 304 872 q 510 843 436 872 q 637 762 585 815 q 717 632 689 709 q 746 456 746 554 m 431 -189 q 425 -225 431 -208 q 408 -254 419 -242 q 384 -274 398 -267 q 353 -282 370 -282 q 310 -265 323 -282 q 298 -217 298 -248 q 303 -181 298 -198 q 320 -151 309 -164 q 344 -132 331 -139 q 374 -125 358 -125 q 431 -189 431 -125 "
    },
    "Ç¼": {
      "x_min": 0,
      "x_max": 1020.84375,
      "ha": 1060,
      "o": "m 515 778 q 508 788 515 785 q 493 791 502 792 q 476 784 484 790 q 462 766 467 778 l 357 498 l 515 498 l 515 778 m 1020 165 q 1006 63 1014 106 q 992 0 997 19 l 419 0 l 419 29 q 489 49 464 38 q 515 70 515 60 l 515 441 l 335 441 l 189 75 q 201 45 180 55 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 70 46 45 37 q 104 75 96 55 l 379 767 q 376 787 383 779 q 353 801 369 795 q 310 813 336 807 q 253 824 285 818 l 253 854 l 955 854 l 975 837 q 972 798 975 819 q 966 756 970 777 q 958 716 962 735 q 948 684 953 697 l 918 684 q 914 736 918 715 q 903 770 910 757 q 887 787 896 782 q 864 793 877 793 l 621 793 l 621 498 l 890 498 l 908 479 q 894 458 902 469 q 879 437 887 447 q 862 417 870 426 q 846 403 854 409 q 825 420 836 413 q 799 432 814 427 q 763 438 783 436 q 714 441 742 441 l 621 441 l 621 104 q 625 86 621 94 q 645 72 630 78 q 687 64 660 67 q 758 61 714 61 l 844 61 q 896 64 874 61 q 934 79 917 67 q 963 114 950 91 q 990 177 976 137 l 1020 165 m 526 926 q 510 937 515 931 q 496 957 505 944 l 753 1172 q 771 1161 760 1167 q 795 1147 783 1155 q 816 1134 807 1140 q 832 1121 826 1127 l 838 1091 l 526 926 "
    },
    ";": {
      "x_min": 58.421875,
      "x_max": 259,
      "ha": 312,
      "o": "m 259 48 q 248 -11 259 21 q 219 -79 238 -44 q 172 -147 199 -114 q 110 -206 144 -180 l 80 -182 q 111 -140 98 -161 q 133 -99 124 -120 q 146 -53 142 -77 q 150 0 150 -29 q 130 46 150 28 q 69 64 111 64 l 58 94 q 83 113 64 102 q 125 133 102 123 q 170 149 148 143 q 205 156 193 156 q 248 113 237 140 q 259 48 259 86 m 245 575 q 238 531 245 551 q 218 496 231 511 q 189 473 206 481 q 153 465 172 465 q 102 484 117 465 q 87 540 87 504 q 94 583 87 563 q 114 619 101 603 q 143 643 126 634 q 179 652 160 652 q 228 632 212 652 q 245 575 245 612 "
    },
    "Ä ": {
      "x_min": 47,
      "x_max": 761,
      "ha": 794,
      "o": "m 693 805 q 691 792 697 802 q 674 770 685 782 q 651 746 664 758 q 629 726 638 734 l 607 730 q 562 765 585 751 q 513 786 538 779 q 459 797 487 794 q 397 800 430 800 q 356 792 381 800 q 303 767 331 784 q 249 722 276 750 q 199 652 221 693 q 163 557 177 611 q 150 432 150 502 q 174 267 150 337 q 239 152 199 198 q 328 83 278 106 q 429 61 378 61 q 518 70 477 61 q 592 99 559 80 l 592 329 q 585 343 592 336 q 563 357 579 350 q 521 371 548 364 q 455 384 495 378 l 455 414 l 761 414 l 761 384 q 707 360 723 375 q 692 329 692 344 l 692 104 q 603 38 642 63 q 532 1 565 14 q 470 -15 499 -11 q 411 -20 441 -20 q 279 5 344 -20 q 162 82 213 30 q 78 212 110 134 q 47 394 47 289 q 81 596 47 507 q 177 747 116 685 q 318 842 237 809 q 490 875 399 875 q 538 870 512 875 q 592 856 564 865 q 645 834 619 847 q 693 805 671 822 m 492 1045 q 486 1009 492 1026 q 469 980 480 992 q 445 960 459 967 q 414 953 431 953 q 371 969 384 953 q 359 1017 359 986 q 364 1053 359 1036 q 381 1083 370 1070 q 405 1102 392 1095 q 435 1110 419 1110 q 492 1045 492 1110 "
    },
    "6": {
      "x_min": 73,
      "x_max": 586,
      "ha": 639,
      "o": "m 332 448 q 257 428 298 448 q 179 363 216 409 q 191 224 179 283 q 228 127 204 165 q 286 69 252 88 q 362 51 320 51 q 421 68 398 51 q 459 112 445 85 q 480 172 474 139 q 486 236 486 205 q 471 341 486 300 q 433 405 456 382 q 383 438 410 429 q 332 448 356 448 m 586 279 q 579 213 586 247 q 557 145 572 178 q 522 82 543 112 q 473 29 501 52 q 411 -6 446 7 q 336 -20 377 -20 q 234 4 281 -20 q 150 74 186 29 q 93 181 114 119 q 73 321 73 244 q 99 504 73 416 q 182 662 126 592 q 323 781 238 733 q 524 847 408 830 l 536 807 q 397 752 458 789 q 292 666 336 715 q 221 559 248 617 q 185 441 194 501 q 232 479 208 464 q 279 504 256 495 q 323 518 302 514 q 359 522 344 522 q 455 505 413 522 q 526 457 497 488 q 570 380 555 425 q 586 279 586 335 "
    },
    "n": {
      "x_min": 37,
      "x_max": 732,
      "ha": 752,
      "o": "m 446 0 l 446 29 q 516 51 493 42 q 539 70 539 61 l 539 429 q 534 496 539 470 q 519 537 529 522 q 493 557 509 552 q 453 563 477 563 q 408 552 433 563 q 353 520 382 542 q 293 461 325 497 q 230 372 261 425 l 230 70 q 254 49 230 60 q 323 29 279 38 l 323 0 l 37 0 l 37 29 q 105 50 80 40 q 130 70 130 59 l 130 482 q 128 524 130 508 q 117 548 126 539 q 89 561 108 557 q 37 569 70 565 l 37 596 q 120 618 82 604 q 196 651 159 632 l 223 627 l 223 458 q 291 539 256 504 q 362 599 327 575 q 432 637 398 624 q 492 651 466 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 70 q 659 52 639 61 q 732 29 679 42 l 732 0 l 446 0 "
    },
    "ÊŒ": {
      "x_min": 13,
      "x_max": 657,
      "ha": 671,
      "o": "m 13 29 q 45 37 32 33 q 66 47 58 42 q 78 58 74 52 q 86 75 82 65 l 274 590 q 291 616 280 605 q 315 633 302 626 q 342 644 329 640 q 363 651 355 648 l 583 75 q 606 46 589 56 q 657 29 623 36 l 657 0 l 400 0 l 400 29 q 442 35 425 32 q 467 44 458 39 q 477 56 476 49 q 475 75 479 64 l 312 509 l 158 75 q 156 57 155 65 q 165 45 157 50 q 188 36 173 40 q 228 29 203 33 l 228 0 l 13 0 l 13 29 "
    },
    "á¹ˆ": {
      "x_min": 33,
      "x_max": 851,
      "ha": 884,
      "o": "m 33 0 l 33 29 q 105 48 82 35 q 129 70 129 61 l 129 778 q 83 810 107 799 q 33 824 58 820 l 33 854 l 173 854 q 190 852 183 854 q 203 845 196 850 q 217 829 209 839 q 237 801 224 818 l 674 187 l 674 782 q 652 804 674 790 q 578 824 630 817 l 578 854 l 851 854 l 851 824 q 778 805 802 818 q 754 782 754 792 l 754 -20 q 701 -6 720 -16 q 674 14 682 2 l 209 673 l 209 70 q 231 49 209 62 q 305 29 253 36 l 305 0 l 33 0 m 666 -155 q 661 -172 664 -162 q 654 -192 658 -182 q 646 -211 650 -201 q 640 -227 642 -220 l 231 -227 l 216 -210 q 221 -193 218 -202 q 228 -173 224 -183 q 236 -154 232 -164 q 244 -139 240 -145 l 651 -139 l 666 -155 "
    },
    "á¸¯": {
      "x_min": -10,
      "x_max": 390.328125,
      "ha": 369,
      "o": "m 46 0 l 46 29 q 114 49 91 38 q 138 70 138 61 l 138 454 q 138 510 138 488 q 129 543 138 531 q 100 560 120 555 q 46 569 81 566 l 46 596 q 89 606 66 600 q 135 619 112 612 q 178 634 157 626 q 216 651 199 642 l 239 651 l 239 70 q 261 50 239 62 q 332 29 283 38 l 332 0 l 46 0 m 389 854 q 383 818 389 835 q 366 789 377 801 q 342 769 356 776 q 311 762 328 762 q 268 778 281 762 q 256 826 256 795 q 261 862 256 845 q 278 892 267 879 q 302 911 289 904 q 332 919 316 919 q 389 854 389 919 m 123 854 q 117 818 123 835 q 100 789 111 801 q 76 769 90 776 q 45 762 62 762 q 2 778 15 762 q -10 826 -10 795 q -4 862 -10 845 q 12 892 1 879 q 36 911 23 904 q 66 919 50 919 q 123 854 123 919 m 160 953 q 139 961 148 955 q 123 972 130 966 l 278 1253 q 299 1250 286 1252 q 326 1247 312 1249 q 354 1242 340 1245 q 374 1237 367 1239 l 390 1211 l 160 953 "
    },
    "á»¥": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 467 70 q 369 4 398 18 q 316 -14 340 -9 q 271 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 533 113 515 q 100 560 109 551 q 75 572 92 569 q 27 579 57 576 l 27 606 q 72 613 51 609 q 112 623 93 617 q 152 635 132 628 q 194 651 172 642 l 213 625 l 213 226 q 219 147 213 178 q 239 96 226 115 q 271 68 252 77 q 314 60 290 60 q 357 67 335 60 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 483 q 505 532 508 513 q 492 560 502 550 q 464 574 482 570 q 414 579 446 579 l 414 606 q 506 625 463 612 q 588 651 550 638 l 608 625 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 67 631 64 q 703 85 669 71 l 710 55 m 429 -189 q 423 -225 429 -208 q 406 -254 417 -242 q 382 -274 396 -267 q 351 -282 368 -282 q 308 -265 321 -282 q 296 -217 296 -248 q 301 -181 296 -198 q 318 -151 307 -164 q 342 -132 329 -139 q 372 -125 356 -125 q 429 -189 429 -125 "
    },
    "áº´": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 838 q 390 869 367 855 q 430 892 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 624 1367 q 596 1312 612 1339 q 559 1263 579 1284 q 515 1227 539 1241 q 465 1214 491 1214 q 420 1225 441 1214 q 380 1252 400 1237 q 340 1279 360 1267 q 301 1291 320 1291 q 252 1268 274 1291 q 206 1208 230 1246 l 171 1222 q 199 1277 183 1250 q 236 1326 215 1305 q 280 1362 256 1348 q 330 1376 304 1376 q 378 1364 355 1376 q 420 1337 400 1352 q 458 1310 440 1322 q 492 1299 476 1299 q 542 1320 517 1299 q 588 1382 566 1342 l 624 1367 m 614 1138 q 569 1046 594 1083 q 516 985 544 1009 q 459 952 489 962 q 399 942 429 942 q 336 952 366 942 q 277 985 305 962 q 226 1046 250 1009 q 181 1138 201 1083 q 197 1157 189 1151 q 216 1169 205 1163 q 255 1104 233 1130 q 300 1062 276 1078 q 349 1039 324 1046 q 396 1032 374 1032 q 445 1039 419 1032 q 495 1062 471 1046 q 541 1104 519 1078 q 579 1169 563 1130 q 598 1157 590 1163 q 614 1138 606 1151 "
    },
    "Ç¨": {
      "x_min": 33,
      "x_max": 779.53125,
      "ha": 786,
      "o": "m 33 0 l 33 29 q 104 50 78 39 q 130 70 130 61 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 438 l 507 765 q 526 794 522 783 q 523 811 530 805 q 500 820 517 817 q 458 824 483 824 l 458 855 l 737 855 l 737 825 q 699 819 715 822 q 669 812 682 817 q 644 801 655 808 q 624 783 634 794 l 333 455 l 653 86 q 679 65 665 73 q 708 55 693 58 q 740 52 724 52 q 774 54 757 52 l 779 25 q 700 1 740 12 q 629 -9 661 -9 q 595 -3 610 -9 q 566 19 580 2 l 236 433 l 236 70 q 259 50 236 62 q 332 29 283 39 l 332 0 l 33 0 m 427 943 l 356 943 l 181 1151 q 189 1162 186 1157 q 195 1170 192 1167 q 202 1176 198 1174 q 211 1183 206 1179 l 393 1039 l 572 1183 q 588 1170 584 1176 q 601 1151 593 1164 l 427 943 "
    },
    "á¸¡": {
      "x_min": 20,
      "x_max": 657.625,
      "ha": 664,
      "o": "m 459 406 q 447 474 459 442 q 412 531 435 507 q 355 569 389 555 q 276 583 321 583 q 240 574 260 583 q 201 548 219 565 q 171 505 183 531 q 160 446 160 479 q 170 378 160 410 q 204 322 181 346 q 260 284 226 298 q 342 271 294 271 q 382 279 361 271 q 420 305 403 287 q 448 347 437 322 q 459 406 459 372 m 344 -2 q 297 3 319 0 q 257 10 276 6 q 182 -36 210 -15 q 141 -74 155 -57 q 123 -104 127 -90 q 120 -128 120 -118 q 138 -182 120 -157 q 188 -226 157 -208 q 260 -256 219 -245 q 347 -268 301 -268 q 428 -255 391 -268 q 490 -222 464 -243 q 530 -171 516 -200 q 545 -106 545 -141 q 536 -71 545 -87 q 505 -42 527 -55 q 443 -19 482 -29 q 344 -2 405 -9 m 552 434 q 529 339 552 382 q 468 265 506 296 q 384 217 431 234 q 288 200 337 200 l 285 200 q 241 154 253 172 q 230 132 230 136 q 236 116 230 124 q 262 101 243 109 q 315 87 281 94 q 400 74 348 80 q 520 50 472 66 q 595 12 567 33 q 634 -33 623 -8 q 645 -81 645 -57 q 631 -152 645 -118 q 593 -214 617 -185 q 536 -265 569 -242 q 466 -305 504 -288 q 389 -330 429 -321 q 310 -339 349 -339 q 246 -334 279 -339 q 180 -320 212 -330 q 118 -296 147 -311 q 67 -261 89 -282 q 32 -214 45 -240 q 20 -155 20 -188 q 26 -118 20 -137 q 51 -76 32 -99 q 105 -28 71 -54 q 197 28 140 -2 q 137 63 154 44 q 121 103 121 83 q 124 118 121 109 q 138 140 127 127 q 167 170 148 153 q 216 209 186 187 q 155 236 183 218 q 108 280 128 254 q 77 337 88 305 q 67 408 67 370 q 88 502 67 457 q 146 579 110 546 q 228 631 182 612 q 322 651 274 651 q 397 639 362 651 q 460 606 432 627 q 523 615 495 610 q 573 627 551 621 q 613 639 595 633 q 644 651 630 645 l 657 630 q 642 595 649 611 q 619 562 635 579 q 570 555 594 557 q 515 552 545 552 q 542 496 532 525 q 552 434 552 467 m 563 868 q 559 851 562 861 q 551 831 555 841 q 544 812 547 822 q 537 797 540 803 l 129 797 l 114 813 q 119 830 116 821 q 126 850 122 840 q 134 869 130 859 q 141 885 138 878 l 549 885 l 563 868 "
    },
    "âˆ‚": {
      "x_min": 53,
      "x_max": 627,
      "ha": 680,
      "o": "m 521 397 q 487 468 510 435 q 436 525 465 501 q 375 563 407 549 q 315 577 343 577 q 246 557 276 577 q 195 503 216 537 q 165 422 175 469 q 155 321 155 375 q 172 219 155 267 q 216 133 190 170 q 275 75 243 97 q 336 54 307 54 q 408 78 374 54 q 467 145 442 102 q 506 251 492 189 q 521 390 521 313 l 521 397 m 627 490 q 612 309 627 387 q 572 175 597 231 q 515 80 547 118 q 450 20 484 43 q 386 -10 417 -1 q 329 -20 354 -20 q 213 4 264 -20 q 126 71 161 28 q 71 173 90 114 q 53 301 53 232 q 76 431 53 368 q 140 544 100 495 q 230 622 179 593 q 335 652 282 652 q 381 641 355 652 q 433 615 406 631 q 482 578 459 599 q 519 535 505 557 q 497 705 519 636 q 442 817 475 775 q 369 879 409 860 q 291 898 329 898 q 253 895 271 898 q 215 886 234 893 q 172 865 195 879 q 119 828 149 851 l 89 851 l 168 948 q 240 973 206 964 q 309 982 275 982 q 383 974 345 982 q 457 946 421 966 q 523 890 492 925 q 577 801 553 855 q 613 669 600 746 q 627 490 627 593 "
    },
    "â€¡": {
      "x_min": 46.421875,
      "x_max": 602.578125,
      "ha": 649,
      "o": "m 602 727 q 578 675 593 704 q 548 621 564 646 q 457 644 502 636 q 356 653 411 653 q 361 572 356 611 q 380 487 367 533 q 362 401 368 443 q 357 321 357 360 q 415 323 387 321 q 471 330 443 326 q 527 341 499 335 q 585 354 556 347 l 602 325 q 578 273 593 302 q 548 218 564 244 q 457 240 503 231 q 356 250 411 250 q 364 132 356 185 q 390 28 372 79 q 337 -3 365 11 q 286 -27 308 -17 l 257 -10 q 282 118 273 54 q 291 251 291 182 q 232 247 260 251 q 176 240 204 244 q 120 231 148 236 q 63 218 92 225 l 46 247 q 70 299 55 271 q 100 354 84 328 q 190 330 145 338 q 291 322 236 322 q 285 402 291 360 q 267 487 279 443 q 285 573 280 534 q 291 654 291 612 q 176 644 230 654 q 63 621 122 634 l 46 650 q 70 702 55 673 q 100 756 84 731 q 190 732 145 740 q 291 725 236 725 q 282 842 291 788 q 257 947 273 896 q 311 978 283 964 q 361 1003 339 993 l 390 986 q 367 857 377 922 q 358 725 358 792 q 472 733 418 725 q 585 756 526 741 l 602 727 "
    },
    "Åˆ": {
      "x_min": 37,
      "x_max": 732,
      "ha": 752,
      "o": "m 446 0 l 446 29 q 516 51 493 42 q 539 70 539 61 l 539 429 q 534 496 539 470 q 519 537 529 522 q 493 557 509 552 q 453 563 477 563 q 408 552 433 563 q 353 520 382 542 q 293 461 325 497 q 230 372 261 425 l 230 70 q 254 49 230 60 q 323 29 279 38 l 323 0 l 37 0 l 37 29 q 105 50 80 40 q 130 70 130 59 l 130 482 q 128 524 130 508 q 117 548 126 539 q 89 561 108 557 q 37 569 70 565 l 37 596 q 120 618 82 604 q 196 651 159 632 l 223 627 l 223 458 q 291 539 256 504 q 362 599 327 575 q 432 637 398 624 q 492 651 466 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 70 q 659 52 639 61 q 732 29 679 42 l 732 0 l 446 0 m 420 726 l 348 726 l 173 968 q 181 979 178 974 q 187 987 184 984 q 194 993 190 991 q 203 999 198 996 l 386 829 l 565 999 q 581 987 577 993 q 594 968 586 981 l 420 726 "
    },
    "âˆš": {
      "x_min": 13.953125,
      "x_max": 822.78125,
      "ha": 804,
      "o": "m 822 968 q 813 936 819 953 q 803 907 807 919 l 704 907 l 475 40 q 458 14 469 25 q 433 -3 447 4 q 407 -14 420 -10 q 386 -20 394 -18 l 121 552 l 30 552 l 13 570 q 22 600 17 583 q 34 630 28 616 l 196 630 l 431 127 l 662 985 l 806 985 l 822 968 "
    },
    "á»‘": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 565 741 q 552 722 557 728 q 536 710 547 716 l 354 892 l 174 710 q 165 716 169 713 q 159 722 161 718 q 152 730 156 725 q 144 741 149 735 l 320 998 l 391 998 l 565 741 m 326 1015 q 305 1023 314 1017 q 288 1034 296 1028 l 444 1315 q 465 1313 452 1314 q 492 1309 478 1311 q 520 1304 506 1307 q 540 1299 533 1301 l 555 1273 l 326 1015 "
    },
    "áº¶": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 387 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 838 q 390 869 367 855 q 430 892 413 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 465 -189 q 459 -225 465 -208 q 442 -254 453 -242 q 418 -274 432 -267 q 387 -282 404 -282 q 344 -265 357 -282 q 332 -217 332 -248 q 337 -181 332 -198 q 354 -151 343 -164 q 378 -132 365 -139 q 408 -125 392 -125 q 465 -189 465 -125 m 614 1139 q 569 1046 594 1084 q 516 985 544 1009 q 459 952 489 962 q 399 942 429 942 q 336 952 366 942 q 277 985 305 962 q 226 1046 250 1009 q 181 1139 201 1084 q 197 1157 189 1151 q 216 1169 205 1164 q 255 1104 233 1131 q 300 1062 276 1078 q 349 1039 324 1046 q 396 1032 374 1032 q 445 1039 419 1032 q 495 1062 471 1046 q 541 1104 519 1078 q 579 1169 563 1131 q 598 1157 590 1164 q 614 1139 606 1151 "
    },
    "áº¾": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 413 432 428 427 q 377 438 398 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 543 957 q 531 938 535 944 q 515 926 526 932 l 333 1067 l 153 926 q 144 931 147 929 q 137 938 140 934 q 131 946 134 941 q 123 957 128 950 l 298 1166 l 369 1166 l 543 957 m 245 1197 q 229 1208 234 1202 q 216 1228 224 1215 l 473 1443 q 491 1432 479 1439 q 514 1419 502 1426 q 536 1405 526 1412 q 551 1393 546 1398 l 557 1362 l 245 1197 "
    },
    "á¹«": {
      "x_min": 13.3125,
      "x_max": 449.53125,
      "ha": 469,
      "o": "m 449 79 q 385 36 416 55 q 324 4 353 17 q 270 -14 295 -7 q 228 -20 246 -20 q 184 -11 205 -20 q 147 17 163 -2 q 122 69 131 37 q 113 149 113 102 l 113 566 l 27 566 l 13 584 l 65 630 l 113 630 l 113 797 l 192 867 l 213 850 l 213 630 l 430 630 l 449 610 q 434 590 443 601 q 416 569 425 579 q 398 552 406 560 q 382 542 389 544 q 333 558 366 550 q 247 566 301 566 l 213 566 l 213 208 q 216 140 213 167 q 228 97 220 113 q 251 74 237 81 q 284 68 264 68 q 343 77 308 68 q 430 114 378 86 l 449 79 m 298 1037 q 292 1001 298 1018 q 275 972 286 984 q 251 952 265 959 q 220 945 237 945 q 177 961 190 945 q 165 1009 165 978 q 170 1045 165 1028 q 187 1075 176 1062 q 211 1094 198 1087 q 241 1102 225 1102 q 298 1037 298 1102 "
    },
    "áº¯": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 535 922 q 490 829 515 867 q 437 768 465 792 q 380 735 410 745 q 320 725 350 725 q 257 735 287 725 q 198 768 226 745 q 147 829 171 792 q 102 922 122 867 q 118 940 110 934 q 138 952 126 947 q 176 887 154 914 q 221 845 197 861 q 270 822 245 829 q 317 815 295 815 q 366 822 341 815 q 416 845 392 829 q 462 887 440 861 q 500 952 484 914 q 519 940 511 947 q 535 922 527 934 m 290 926 q 269 934 278 928 q 252 945 259 939 l 408 1226 q 429 1224 416 1225 q 456 1220 442 1222 q 484 1215 470 1218 q 504 1211 497 1213 l 519 1184 l 290 926 "
    },
    "á¹„": {
      "x_min": 33,
      "x_max": 851,
      "ha": 884,
      "o": "m 33 0 l 33 29 q 105 48 82 35 q 129 70 129 61 l 129 778 q 83 810 107 799 q 33 824 58 820 l 33 854 l 173 854 q 190 852 183 854 q 203 845 196 850 q 217 829 209 839 q 237 801 224 818 l 674 187 l 674 782 q 652 804 674 790 q 578 824 630 817 l 578 854 l 851 854 l 851 824 q 778 805 802 818 q 754 782 754 792 l 754 -20 q 701 -6 720 -16 q 674 14 682 2 l 209 673 l 209 70 q 231 49 209 62 q 305 29 253 36 l 305 0 l 33 0 m 508 1044 q 502 1008 508 1025 q 485 979 496 991 q 461 959 475 966 q 430 952 447 952 q 387 968 400 952 q 375 1016 375 985 q 380 1052 375 1035 q 397 1082 386 1069 q 421 1101 408 1094 q 451 1109 435 1109 q 508 1044 508 1109 "
    },
    "â‰ˆ": {
      "x_min": 37.21875,
      "x_max": 560.265625,
      "ha": 597,
      "o": "m 560 328 q 399 219 493 219 q 344 229 371 219 q 290 252 317 240 q 236 275 263 264 q 182 286 209 286 q 126 267 151 286 q 72 216 101 249 l 37 254 q 198 365 103 365 q 257 354 227 365 q 314 331 287 343 q 367 308 342 319 q 413 298 392 298 q 444 303 428 298 q 474 319 459 309 q 501 341 488 328 q 524 368 514 354 l 560 328 m 560 504 q 399 395 493 395 q 344 405 371 395 q 290 428 317 416 q 236 451 263 440 q 182 462 209 462 q 126 443 151 462 q 72 393 101 425 l 37 430 q 198 541 103 541 q 257 530 227 541 q 314 507 287 519 q 367 484 342 495 q 413 474 392 474 q 444 479 428 474 q 474 495 459 485 q 501 517 488 504 q 524 544 514 530 l 560 504 "
    },
    "g": {
      "x_min": 20,
      "x_max": 657.625,
      "ha": 664,
      "o": "m 459 406 q 447 474 459 442 q 412 531 435 507 q 355 569 389 555 q 276 583 321 583 q 240 574 260 583 q 201 548 219 565 q 171 505 183 531 q 160 446 160 479 q 170 378 160 410 q 204 322 181 346 q 260 284 226 298 q 342 271 294 271 q 382 279 361 271 q 420 305 403 287 q 448 347 437 322 q 459 406 459 372 m 344 -2 q 297 3 319 0 q 257 10 276 6 q 182 -36 210 -15 q 141 -74 155 -57 q 123 -104 127 -90 q 120 -128 120 -118 q 138 -182 120 -157 q 188 -226 157 -208 q 260 -256 219 -245 q 347 -268 301 -268 q 428 -255 391 -268 q 490 -222 464 -243 q 530 -171 516 -200 q 545 -106 545 -141 q 536 -71 545 -87 q 505 -42 527 -55 q 443 -19 482 -29 q 344 -2 405 -9 m 552 434 q 529 339 552 382 q 468 265 506 296 q 384 217 431 234 q 288 200 337 200 l 285 200 q 241 154 253 172 q 230 132 230 136 q 236 116 230 124 q 262 101 243 109 q 315 87 281 94 q 400 74 348 80 q 520 50 472 66 q 595 12 567 33 q 634 -33 623 -8 q 645 -81 645 -57 q 631 -152 645 -118 q 593 -214 617 -185 q 536 -265 569 -242 q 466 -305 504 -288 q 389 -330 429 -321 q 310 -339 349 -339 q 246 -334 279 -339 q 180 -320 212 -330 q 118 -296 147 -311 q 67 -261 89 -282 q 32 -214 45 -240 q 20 -155 20 -188 q 26 -118 20 -137 q 51 -76 32 -99 q 105 -28 71 -54 q 197 28 140 -2 q 137 63 154 44 q 121 103 121 83 q 124 118 121 109 q 138 140 127 127 q 167 170 148 153 q 216 209 186 187 q 155 236 183 218 q 108 280 128 254 q 77 337 88 305 q 67 408 67 370 q 88 502 67 457 q 146 579 110 546 q 228 631 182 612 q 322 651 274 651 q 397 639 362 651 q 460 606 432 627 q 523 615 495 610 q 573 627 551 621 q 613 639 595 633 q 644 651 630 645 l 657 630 q 642 595 649 611 q 619 562 635 579 q 570 554 594 557 q 515 552 545 552 q 542 496 532 525 q 552 434 552 467 "
    },
    "Ç¿": {
      "x_min": 53,
      "x_max": 631,
      "ha": 684,
      "o": "m 529 309 q 518 388 529 349 q 490 458 507 426 l 236 114 q 292 70 263 86 q 347 54 321 54 q 429 72 395 54 q 485 123 463 90 q 518 203 507 157 q 529 309 529 250 m 155 321 q 165 242 155 279 q 195 171 176 204 l 450 516 q 394 561 424 545 q 336 577 365 577 q 251 557 285 577 q 195 503 216 537 q 164 422 173 469 q 155 321 155 375 m 628 644 l 560 553 q 612 453 594 510 q 631 329 631 396 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 370 -20 q 244 -7 283 -20 q 173 29 205 5 l 160 12 q 143 0 154 7 q 120 -14 132 -7 q 97 -27 109 -21 q 77 -35 85 -32 l 58 -13 l 125 76 q 71 176 90 119 q 53 301 53 233 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 313 651 q 440 638 400 651 q 512 600 479 625 l 526 619 q 545 632 535 625 q 566 646 555 640 q 587 659 577 653 q 607 667 598 664 l 628 644 m 320 710 q 298 718 308 712 q 282 729 289 723 l 437 1010 q 458 1007 445 1009 q 485 1004 471 1006 q 513 999 499 1002 q 533 994 526 996 l 549 968 l 320 710 "
    },
    "Â²": {
      "x_min": 39.375,
      "x_max": 392,
      "ha": 447,
      "o": "m 392 420 l 52 420 l 39 449 q 135 548 94 505 q 206 625 177 592 q 255 683 235 658 q 285 728 274 708 q 300 764 296 748 q 305 795 305 780 q 285 855 305 833 q 217 878 265 878 q 185 871 199 878 q 160 853 170 864 q 143 828 149 842 q 138 799 138 814 q 103 787 122 792 q 64 780 85 782 l 52 792 q 68 836 52 813 q 112 878 84 859 q 175 910 139 897 q 249 923 210 923 q 305 916 279 923 q 349 895 331 909 q 378 859 368 880 q 389 807 389 837 q 376 752 389 780 q 334 687 363 724 q 257 599 305 650 q 137 476 209 549 l 318 476 q 343 486 333 476 q 357 508 352 496 q 365 541 363 522 l 392 536 l 392 420 "
    },
    "Ì": {
      "x_min": -427.34375,
      "x_max": -160.171875,
      "ha": 0,
      "o": "m -389 709 q -410 717 -401 712 q -427 728 -420 722 l -271 1009 q -250 1007 -263 1009 q -223 1003 -237 1005 q -196 998 -209 1001 q -175 994 -182 996 l -160 967 l -389 709 "
    },
    "á¸£": {
      "x_min": 37,
      "x_max": 732,
      "ha": 752,
      "o": "m 446 0 l 446 29 q 516 51 493 42 q 539 70 539 61 l 539 429 q 534 496 539 470 q 518 537 529 522 q 491 557 507 552 q 453 563 475 563 q 403 550 430 563 q 347 514 376 538 q 288 455 318 491 q 230 372 258 419 l 230 70 q 254 49 230 60 q 323 29 279 38 l 323 0 l 37 0 l 37 29 q 105 50 80 40 q 130 70 130 59 l 130 880 q 127 924 130 908 q 115 949 125 940 q 87 962 106 958 q 37 969 68 965 l 37 997 q 87 1007 64 1002 q 130 1019 110 1013 q 168 1033 150 1026 q 205 1051 186 1041 l 230 1028 l 230 464 q 293 541 258 507 q 363 600 328 576 q 432 638 398 625 q 491 651 465 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 70 q 659 52 639 61 q 732 29 679 42 l 732 0 l 446 0 m 450 1214 q 444 1178 450 1195 q 427 1149 438 1161 q 403 1129 417 1136 q 372 1122 389 1122 q 329 1138 342 1122 q 317 1186 317 1155 q 322 1222 317 1205 q 339 1252 328 1239 q 363 1271 350 1264 q 393 1279 377 1279 q 450 1214 450 1279 "
    },
    "á¸‰": {
      "x_min": 53,
      "x_max": 557.953125,
      "ha": 595,
      "o": "m 435 -155 q 423 -204 435 -181 q 387 -246 411 -227 q 323 -279 362 -265 q 230 -301 284 -293 l 215 -268 q 274 -253 250 -262 q 313 -233 298 -244 q 336 -210 329 -222 q 343 -187 343 -198 q 325 -155 343 -164 q 267 -142 308 -146 q 274 -123 268 -140 q 288 -78 278 -109 q 306 -21 295 -56 q 213 0 258 -21 q 132 64 168 23 q 74 165 96 105 q 53 300 53 225 q 79 437 53 373 q 150 548 105 501 q 258 623 196 595 q 391 650 320 650 q 436 646 413 650 q 480 636 459 643 q 519 619 501 628 q 547 597 536 609 q 545 573 549 588 q 536 542 542 559 q 523 510 530 525 q 510 485 516 495 l 485 492 q 468 518 480 504 q 437 546 456 533 q 392 567 418 559 q 332 576 366 576 q 264 560 297 576 q 207 512 232 544 q 167 432 182 480 q 153 321 153 385 q 169 216 153 263 q 215 136 186 169 q 280 84 243 102 q 356 67 316 67 q 391 68 375 67 q 426 79 407 70 q 469 105 445 88 q 532 155 494 123 l 557 128 q 481 46 514 75 q 420 1 448 16 q 366 -16 393 -13 q 359 -18 363 -17 l 341 -71 q 376 -81 359 -75 q 406 -98 393 -88 q 427 -122 419 -108 q 435 -155 435 -136 m 296 709 q 275 717 284 711 q 258 728 266 722 l 414 1009 q 435 1006 422 1008 q 462 1003 448 1005 q 490 998 476 1001 q 510 993 503 995 l 526 967 l 296 709 "
    },
    "Ãƒ": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 838 q 390 869 367 855 q 430 893 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 624 1103 q 596 1048 612 1075 q 559 999 579 1020 q 515 963 539 977 q 465 950 491 950 q 420 961 441 950 q 380 988 400 973 q 340 1015 360 1003 q 301 1027 320 1027 q 252 1004 274 1027 q 206 944 230 982 l 171 958 q 199 1013 183 986 q 236 1062 215 1041 q 280 1098 256 1084 q 330 1112 304 1112 q 378 1100 355 1112 q 420 1073 400 1088 q 458 1046 440 1058 q 492 1035 476 1035 q 542 1056 517 1035 q 588 1118 566 1078 l 624 1103 "
    },
    "Ë€": {
      "x_min": 21,
      "x_max": 402,
      "ha": 414,
      "o": "m 165 550 l 165 622 q 176 679 165 653 q 204 728 187 705 q 241 771 222 751 q 278 812 261 792 q 307 852 296 832 q 319 896 319 872 q 309 941 319 921 q 283 975 299 961 q 246 997 267 990 q 202 1005 225 1005 q 169 998 186 1005 q 140 980 153 991 q 119 954 127 969 q 111 923 111 939 q 112 911 111 917 q 116 899 114 905 q 78 889 100 893 q 34 884 55 886 l 21 896 q 21 903 21 899 l 21 910 q 38 964 21 938 q 86 1009 56 990 q 155 1039 116 1028 q 237 1051 194 1051 q 306 1040 276 1051 q 358 1011 337 1029 q 390 967 379 992 q 402 914 402 943 q 390 860 402 884 q 362 816 379 836 q 325 777 344 796 q 288 738 305 758 q 259 696 270 718 q 248 646 248 674 l 248 550 l 165 550 "
    },
    "Ì„": {
      "x_min": -566.90625,
      "x_max": -117.640625,
      "ha": 0,
      "o": "m -117 868 q -122 851 -118 861 q -129 831 -125 841 q -137 812 -133 822 q -143 797 -140 803 l -552 797 l -566 813 q -562 830 -565 821 q -554 850 -558 840 q -546 869 -550 859 q -539 885 -542 878 l -132 885 l -117 868 "
    },
    "á¹Œ": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 636 1103 q 607 1048 624 1075 q 570 999 590 1020 q 526 963 550 977 q 476 950 502 950 q 432 961 453 950 q 391 988 411 973 q 351 1015 371 1003 q 312 1027 332 1027 q 264 1004 286 1027 q 217 944 242 982 l 182 958 q 210 1013 194 986 q 247 1062 227 1041 q 291 1098 268 1084 q 341 1112 315 1112 q 389 1100 366 1112 q 431 1073 411 1088 q 469 1046 451 1058 q 503 1035 487 1035 q 553 1056 528 1035 q 600 1118 577 1078 l 636 1103 m 321 1144 q 305 1155 310 1149 q 291 1175 300 1162 l 548 1390 q 566 1379 555 1385 q 590 1365 578 1373 q 612 1352 602 1358 q 627 1339 622 1345 l 633 1309 l 321 1144 "
    },
    "Â©": {
      "x_min": 66,
      "x_max": 903,
      "ha": 969,
      "o": "m 676 250 q 619 201 646 221 q 567 169 592 181 q 517 152 542 157 q 469 147 493 147 q 385 164 426 147 q 311 215 343 182 q 258 298 278 249 q 238 408 238 346 q 260 531 238 476 q 320 626 282 587 q 410 686 359 665 q 518 708 461 708 q 605 695 568 708 q 668 665 642 682 q 666 654 670 663 q 655 634 662 645 q 639 613 648 623 q 625 598 631 603 l 609 601 q 555 636 585 620 q 485 653 526 653 q 436 642 464 653 q 383 605 408 631 q 341 535 358 579 q 325 425 325 490 q 344 330 325 370 q 393 262 364 289 q 456 220 423 234 q 518 206 490 206 q 546 209 532 206 q 577 219 560 212 q 613 238 593 225 q 658 269 632 250 q 668 259 663 267 q 676 250 672 252 m 846 428 q 819 579 846 508 q 743 703 792 650 q 629 786 695 755 q 483 817 562 817 q 339 786 405 817 q 225 703 273 755 q 149 579 176 650 q 123 428 123 508 q 149 276 123 347 q 225 152 176 204 q 339 68 273 99 q 483 38 405 38 q 629 68 562 38 q 743 152 695 99 q 819 276 792 204 q 846 428 846 347 m 903 428 q 888 309 903 366 q 848 202 874 251 q 784 111 821 152 q 700 41 746 70 q 599 -4 653 11 q 483 -20 544 -20 q 316 15 392 -20 q 184 111 240 50 q 97 253 128 171 q 66 428 66 335 q 97 602 66 520 q 184 744 128 683 q 316 839 240 804 q 483 875 392 875 q 599 859 544 875 q 700 813 653 843 q 784 744 746 784 q 848 653 821 703 q 888 546 874 603 q 903 428 903 489 "
    },
    "â‰¥": {
      "x_min": 40.546875,
      "x_max": 556.9375,
      "ha": 597,
      "o": "m 556 193 q 547 164 553 179 q 537 136 541 148 l 57 136 l 40 154 q 49 182 43 167 q 61 210 55 197 l 540 210 l 556 193 m 40 647 q 70 670 55 659 q 96 691 84 681 l 541 521 l 556 505 q 548 471 553 489 q 536 437 543 452 l 57 255 l 40 272 q 45 287 42 279 q 50 306 47 296 q 55 324 53 315 q 61 338 58 332 l 427 478 l 51 620 l 40 647 "
    },
    "áº™": {
      "x_min": -31,
      "x_max": 658,
      "ha": 672,
      "o": "m 658 601 q 625 592 638 596 q 604 583 612 588 q 592 571 597 578 q 585 555 588 565 l 362 -55 q 300 -184 334 -130 q 229 -272 266 -237 q 155 -322 192 -306 q 84 -339 118 -339 q 38 -335 59 -339 q 1 -327 16 -332 q -22 -315 -13 -322 q -31 -303 -31 -309 q -22 -286 -31 -299 q -2 -260 -14 -274 q 22 -231 9 -245 q 47 -211 36 -218 q 110 -232 79 -232 q 166 -223 142 -232 q 194 -204 178 -219 q 226 -168 210 -189 q 258 -119 243 -146 q 286 -62 274 -91 l 305 -15 l 87 555 q 64 584 81 574 q 14 601 47 593 l 14 630 l 271 630 l 271 601 q 228 594 244 598 q 203 586 212 591 q 192 573 194 581 q 195 555 191 566 l 359 123 l 512 555 q 513 572 516 565 q 502 584 511 579 q 477 593 493 589 q 436 601 460 597 l 436 631 l 658 631 l 658 601 m 417 843 q 412 874 417 859 q 401 900 408 889 q 383 918 394 911 q 359 925 372 925 q 333 919 345 925 q 310 903 320 913 q 295 877 301 892 q 289 844 289 862 q 293 814 289 828 q 304 788 297 799 q 322 770 311 777 q 346 764 333 764 q 372 769 359 764 q 395 784 385 774 q 411 808 405 794 q 417 843 417 823 m 477 870 q 464 807 477 836 q 430 757 451 778 q 383 723 409 735 q 332 712 357 712 q 290 720 309 712 q 257 742 271 728 q 236 775 244 756 q 229 816 229 794 q 242 880 229 851 q 275 931 255 910 q 321 965 296 953 q 373 977 347 977 q 414 968 395 977 q 447 945 433 960 q 469 911 461 930 q 477 870 477 892 "
    },
    "Ä‚": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 838 q 390 869 367 855 q 430 892 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 614 1139 q 569 1046 594 1084 q 516 985 544 1009 q 459 952 489 962 q 399 942 429 942 q 336 952 366 942 q 277 985 305 962 q 226 1046 250 1009 q 181 1139 201 1084 q 197 1157 189 1151 q 216 1169 205 1164 q 255 1104 233 1131 q 300 1062 276 1078 q 349 1039 324 1046 q 396 1032 374 1032 q 445 1039 419 1032 q 495 1062 471 1046 q 541 1104 519 1078 q 579 1169 563 1131 q 598 1157 590 1164 q 614 1139 606 1151 "
    },
    "Ç–": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 467 70 q 369 3 398 18 q 316 -15 340 -10 q 271 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 43 q 113 189 113 125 l 113 482 q 111 532 113 514 q 100 559 109 550 q 75 572 92 569 q 27 579 57 575 l 27 606 q 72 613 51 608 q 112 622 93 617 q 152 635 132 627 q 194 650 172 642 l 213 624 l 213 226 q 219 147 213 179 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 482 q 505 531 508 513 q 492 560 502 549 q 464 574 482 570 q 414 578 446 578 l 414 606 q 507 625 463 612 q 588 650 551 638 l 608 624 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 562 854 q 556 818 562 835 q 539 789 550 801 q 515 769 529 776 q 484 762 501 762 q 441 778 454 762 q 429 826 429 795 q 434 862 429 845 q 451 892 440 879 q 475 911 462 904 q 505 919 489 919 q 562 854 562 919 m 296 854 q 290 818 296 835 q 273 789 284 801 q 249 769 263 776 q 218 762 235 762 q 175 778 188 762 q 163 826 163 795 q 168 862 163 845 q 185 892 174 879 q 209 911 196 904 q 239 919 223 919 q 296 854 296 919 m 593 1112 q 588 1095 592 1105 q 581 1075 585 1085 q 574 1056 577 1066 q 567 1041 570 1047 l 159 1041 l 144 1057 q 148 1074 145 1065 q 156 1094 152 1084 q 164 1113 160 1103 q 171 1129 168 1122 l 579 1129 l 593 1112 "
    },
    "Ç¹": {
      "x_min": 37,
      "x_max": 732,
      "ha": 752,
      "o": "m 446 0 l 446 29 q 516 51 493 42 q 539 70 539 61 l 539 429 q 534 496 539 470 q 519 537 529 522 q 493 557 509 552 q 453 563 477 563 q 408 552 433 563 q 353 520 382 542 q 293 461 325 497 q 230 372 261 425 l 230 70 q 254 49 230 60 q 323 29 279 38 l 323 0 l 37 0 l 37 29 q 105 50 80 40 q 130 70 130 59 l 130 482 q 128 524 130 508 q 117 548 126 539 q 89 561 108 557 q 37 569 70 565 l 37 596 q 120 618 82 604 q 196 651 159 632 l 223 627 l 223 458 q 291 539 256 504 q 362 599 327 575 q 432 637 398 624 q 492 651 466 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 70 q 659 52 639 61 q 732 29 679 42 l 732 0 l 446 0 m 459 731 q 442 718 451 722 q 422 709 434 713 l 188 965 l 203 993 q 223 997 210 995 q 250 1002 235 1000 q 278 1007 264 1005 q 300 1009 292 1009 l 459 731 "
    },
    "Ã¿": {
      "x_min": -31,
      "x_max": 658,
      "ha": 672,
      "o": "m 658 601 q 625 592 638 596 q 604 583 612 588 q 592 571 596 578 q 584 555 588 565 l 362 -55 q 300 -184 334 -130 q 229 -272 266 -237 q 155 -322 192 -306 q 84 -339 117 -339 q 38 -335 59 -339 q 1 -327 16 -332 q -22 -315 -13 -322 q -31 -303 -31 -309 q -22 -286 -31 -299 q -2 -260 -14 -274 q 22 -231 9 -245 q 47 -211 36 -218 q 111 -232 79 -232 q 167 -223 142 -232 q 195 -204 179 -219 q 227 -168 211 -189 q 259 -119 243 -146 q 286 -62 274 -91 l 306 -15 l 88 555 q 64 584 82 574 q 14 601 47 593 l 14 630 l 271 630 l 271 601 q 228 594 245 598 q 203 586 212 591 q 193 573 194 581 q 195 555 191 566 l 358 123 l 512 555 q 513 572 515 565 q 501 584 510 579 q 476 593 493 589 q 436 601 460 597 l 436 631 l 658 631 l 658 601 m 552 854 q 546 818 552 835 q 529 789 540 801 q 505 769 519 776 q 474 762 491 762 q 431 778 444 762 q 419 826 419 795 q 424 862 419 845 q 441 892 430 879 q 465 911 452 904 q 495 919 479 919 q 552 854 552 919 m 287 854 q 281 818 287 835 q 264 789 275 801 q 240 769 254 776 q 209 762 226 762 q 166 778 179 762 q 154 826 154 795 q 159 862 154 845 q 176 892 165 879 q 200 911 187 904 q 230 919 214 919 q 287 854 287 919 "
    },
    "á¸¸": {
      "x_min": 33,
      "x_max": 627.625,
      "ha": 648,
      "o": "m 627 165 q 614 63 622 106 q 600 0 605 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 287 815 q 236 783 236 792 l 236 111 q 241 89 236 98 q 261 74 247 80 q 299 64 275 67 q 359 61 323 61 l 454 61 q 506 64 485 61 q 543 79 527 67 q 572 114 559 91 q 599 177 585 137 l 627 165 m 397 -189 q 391 -225 397 -208 q 374 -254 385 -242 q 350 -274 364 -267 q 319 -282 336 -282 q 276 -265 289 -282 q 264 -217 264 -248 q 269 -181 264 -198 q 286 -151 275 -164 q 310 -132 297 -139 q 340 -125 324 -125 q 397 -189 397 -125 m 541 1058 q 537 1041 540 1051 q 529 1021 533 1031 q 522 1002 525 1012 q 515 987 518 993 l 107 987 l 92 1003 q 97 1020 93 1011 q 104 1040 100 1030 q 112 1059 108 1049 q 119 1075 116 1068 l 527 1075 l 541 1058 "
    },
    "Å": {
      "x_min": 27.015625,
      "x_max": 627.09375,
      "ha": 648,
      "o": "m 627 165 q 613 63 621 106 q 599 0 605 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 361 l 45 318 l 27 332 q 33 347 29 339 q 42 365 37 356 q 52 381 47 374 q 61 393 57 389 l 130 428 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 482 l 420 578 l 437 561 q 422 531 432 549 q 402 502 411 513 l 236 416 l 236 111 q 241 89 236 98 q 261 74 247 80 q 299 64 275 67 q 359 61 322 61 l 454 61 q 505 64 484 61 q 542 79 526 67 q 571 114 558 91 q 598 177 584 137 l 627 165 "
    },
    "âˆ«": {
      "x_min": -137,
      "x_max": 483,
      "ha": 369,
      "o": "m 483 965 q 473 949 483 959 q 450 927 463 938 q 422 906 437 916 q 398 892 408 896 q 372 930 384 915 q 348 953 359 944 q 326 964 336 961 q 307 968 315 968 q 274 957 288 968 q 252 920 261 946 q 239 853 243 895 q 235 750 235 811 q 235 650 235 708 q 235 526 235 592 q 235 390 235 460 q 235 252 235 319 q 235 125 235 185 q 235 20 235 65 q 223 -84 235 -38 q 194 -165 212 -130 q 153 -226 175 -200 q 108 -271 130 -252 q 71 -298 90 -285 q 33 -319 52 -310 q -2 -333 14 -328 q -31 -339 -19 -339 q -70 -333 -51 -339 q -103 -321 -88 -328 q -127 -305 -118 -314 q -137 -292 -137 -297 q -127 -275 -137 -286 q -104 -253 -117 -265 q -76 -232 -91 -242 q -52 -217 -62 -222 q -11 -242 -31 -236 q 32 -248 9 -248 q 65 -236 48 -248 q 97 -197 83 -225 q 119 -121 110 -169 q 129 0 129 -73 q 129 93 129 37 q 129 215 129 149 q 129 352 129 281 q 129 489 129 423 q 129 614 129 556 q 129 712 129 672 q 135 812 129 770 q 154 885 141 853 q 185 941 166 916 q 229 988 204 965 q 297 1035 261 1019 q 357 1051 332 1051 q 407 1039 384 1051 q 447 1014 430 1028 q 473 985 464 999 q 483 965 483 971 "
    },
    "\\": {
      "x_min": 36.546875,
      "x_max": 601.46875,
      "ha": 639,
      "o": "m 578 -227 q 559 -220 570 -224 q 538 -211 548 -216 q 518 -202 527 -207 q 502 -192 508 -197 l 36 1070 l 61 1085 q 136 1051 99 1074 l 601 -210 l 578 -227 "
    },
    "ÃŒ": {
      "x_min": -27.421875,
      "x_max": 346,
      "ha": 406,
      "o": "m 47 0 l 47 29 q 118 49 92 38 q 144 70 144 61 l 144 782 q 119 803 144 790 q 47 824 95 815 l 47 854 l 346 854 l 346 824 q 275 803 300 815 q 250 782 250 792 l 250 70 q 274 50 250 62 q 346 29 298 38 l 346 0 l 47 0 m 314 957 q 299 937 304 944 q 284 926 295 931 l -27 1091 l -22 1121 q -7 1134 -17 1127 q 15 1147 3 1140 q 38 1161 27 1155 q 57 1172 49 1167 l 314 957 "
    },
    "È°": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 476 1045 q 470 1009 476 1026 q 453 980 464 992 q 429 960 443 967 q 398 953 415 953 q 355 969 368 953 q 343 1017 343 986 q 348 1053 343 1036 q 365 1083 354 1070 q 389 1102 376 1095 q 419 1110 403 1110 q 476 1045 476 1110 m 640 1275 q 636 1258 639 1268 q 628 1238 632 1248 q 621 1219 624 1229 q 614 1204 617 1210 l 205 1204 l 191 1220 q 195 1237 192 1228 q 203 1257 199 1247 q 211 1276 207 1266 q 218 1292 215 1285 l 626 1292 l 640 1275 "
    },
    "á¸²": {
      "x_min": 33,
      "x_max": 779.53125,
      "ha": 786,
      "o": "m 33 0 l 33 29 q 104 50 78 39 q 130 70 130 61 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 438 l 507 765 q 526 794 522 783 q 524 811 530 805 q 500 820 517 817 q 458 824 484 824 l 458 855 l 737 855 l 737 825 q 699 819 715 822 q 669 812 682 817 q 645 801 656 808 q 624 783 634 794 l 333 455 l 654 86 q 679 65 666 73 q 708 55 693 58 q 741 52 724 52 q 774 54 757 52 l 779 25 q 700 1 740 12 q 629 -9 661 -9 q 595 -3 610 -9 q 567 19 580 2 l 236 433 l 236 70 q 259 50 236 62 q 332 29 283 39 l 332 0 l 33 0 m 473 -189 q 467 -225 473 -208 q 450 -254 461 -242 q 426 -274 440 -267 q 395 -282 412 -282 q 352 -265 365 -282 q 340 -217 340 -248 q 345 -181 340 -198 q 362 -151 351 -164 q 386 -132 373 -139 q 416 -125 400 -125 q 473 -189 473 -125 "
    },
    "á¸—": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 290 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 569 868 q 565 851 568 861 q 557 831 561 841 q 550 812 553 822 q 543 797 546 803 l 135 797 l 121 813 q 125 830 122 821 q 133 850 129 840 q 141 869 137 859 q 148 885 145 878 l 555 885 l 569 868 m 310 954 q 288 961 298 956 q 272 973 279 967 l 427 1253 q 448 1251 435 1253 q 475 1247 461 1249 q 503 1243 489 1245 q 523 1238 516 1240 l 539 1211 l 310 954 "
    },
    "á¹™": {
      "x_min": 37,
      "x_max": 517.765625,
      "ha": 540,
      "o": "m 512 626 q 517 602 517 621 q 512 556 517 582 q 499 503 507 530 q 483 458 491 476 l 455 458 q 443 504 451 485 q 426 534 436 523 q 405 550 417 545 q 381 556 394 556 q 347 543 366 556 q 307 504 327 530 q 266 435 286 477 q 230 336 245 393 l 230 70 q 254 49 230 60 q 342 28 279 38 l 342 0 l 37 0 l 37 28 q 105 48 80 39 q 130 70 130 58 l 130 465 q 128 502 130 487 q 125 527 127 517 q 120 542 123 537 q 114 551 117 547 q 104 560 110 556 q 89 565 98 563 q 68 568 80 568 q 37 568 55 568 l 37 597 q 121 620 80 608 q 196 651 163 632 l 224 627 l 224 472 q 267 543 246 510 q 312 599 288 575 q 362 637 336 623 q 417 651 388 651 q 463 645 438 651 q 512 626 487 640 m 366 854 q 360 818 366 835 q 343 789 354 801 q 319 769 333 776 q 288 762 305 762 q 245 778 258 762 q 233 826 233 795 q 238 862 233 845 q 255 892 244 879 q 279 911 266 904 q 309 919 293 919 q 366 854 366 919 "
    },
    "á¸Š": {
      "x_min": 27.015625,
      "x_max": 746,
      "ha": 806,
      "o": "m 300 816 q 235 816 266 816 l 235 104 q 241 80 235 89 q 277 62 250 68 q 356 57 304 57 q 450 79 402 57 q 540 148 499 102 q 607 265 581 195 q 633 431 633 335 q 611 592 633 521 q 546 713 589 663 q 442 789 504 763 q 300 816 380 816 m 746 456 q 729 305 746 372 q 683 188 712 239 q 617 101 654 137 q 538 43 579 66 q 453 10 496 20 q 371 0 410 0 l 33 0 l 33 29 q 103 49 78 38 q 129 70 129 61 l 129 805 q 78 798 102 801 q 33 791 54 794 l 27 833 q 94 850 56 842 q 175 864 132 858 q 261 871 217 871 q 343 871 304 871 q 510 842 436 871 q 637 761 585 814 q 717 631 689 708 q 746 456 746 554 m 431 1044 q 425 1008 431 1025 q 408 979 419 991 q 384 959 398 966 q 353 952 370 952 q 310 968 323 952 q 298 1016 298 985 q 303 1052 298 1035 q 320 1082 309 1069 q 344 1101 331 1094 q 374 1109 358 1109 q 431 1044 431 1109 "
    },
    "Ä’": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 413 432 428 427 q 377 438 398 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 564 1058 q 560 1041 563 1051 q 552 1021 556 1031 q 545 1002 548 1012 q 538 987 541 993 l 130 987 l 115 1003 q 120 1020 117 1011 q 127 1040 123 1030 q 135 1059 131 1049 q 143 1075 139 1068 l 550 1075 l 564 1058 "
    },
    "!": {
      "x_min": 101,
      "x_max": 259,
      "ha": 370,
      "o": "m 259 83 q 252 39 259 59 q 232 4 245 19 q 203 -18 220 -10 q 167 -27 186 -27 q 116 -7 131 -27 q 101 48 101 12 q 108 91 101 71 q 128 127 115 111 q 157 151 140 142 q 193 160 174 160 q 242 140 226 160 q 259 83 259 120 m 232 263 q 202 248 215 254 q 167 237 189 242 l 132 251 l 132 947 q 177 979 153 964 q 220 1003 200 994 l 232 986 l 232 263 "
    },
    "Ã§": {
      "x_min": 53,
      "x_max": 557.953125,
      "ha": 595,
      "o": "m 435 -155 q 423 -204 435 -181 q 387 -246 411 -227 q 323 -279 362 -265 q 230 -301 284 -293 l 215 -268 q 274 -253 250 -262 q 313 -233 298 -244 q 336 -210 329 -222 q 343 -187 343 -198 q 325 -155 343 -164 q 267 -142 308 -146 q 274 -123 268 -140 q 288 -78 278 -109 q 306 -21 295 -56 q 213 0 258 -21 q 132 64 168 23 q 74 165 96 105 q 53 300 53 225 q 79 437 53 373 q 150 548 105 501 q 258 623 196 595 q 391 650 320 650 q 436 646 413 650 q 480 636 459 643 q 519 619 501 628 q 547 597 536 609 q 545 573 549 588 q 536 542 542 559 q 523 510 530 525 q 510 485 516 495 l 485 492 q 468 518 480 504 q 437 546 456 533 q 392 567 418 559 q 332 576 366 576 q 264 560 297 576 q 207 512 232 544 q 167 432 182 480 q 153 321 153 385 q 169 216 153 263 q 215 136 186 169 q 280 84 243 102 q 356 67 316 67 q 391 68 375 67 q 426 79 407 70 q 469 105 445 88 q 532 155 494 123 l 557 128 q 481 46 514 75 q 420 1 448 16 q 366 -16 393 -13 q 359 -18 363 -17 l 341 -71 q 376 -81 359 -75 q 406 -98 393 -88 q 427 -122 419 -108 q 435 -155 435 -136 "
    },
    "Ç¯": {
      "x_min": 25,
      "x_max": 565,
      "ha": 612,
      "o": "m 565 -17 q 537 -148 565 -89 q 465 -249 509 -207 q 365 -315 420 -292 q 253 -339 309 -339 q 164 -326 206 -339 q 91 -295 122 -313 q 42 -258 60 -277 q 25 -228 25 -240 q 34 -214 25 -224 q 57 -193 43 -204 q 87 -173 71 -183 q 117 -162 103 -164 q 195 -240 156 -213 q 279 -268 235 -268 q 353 -253 319 -268 q 412 -212 387 -239 q 451 -144 437 -184 q 465 -52 465 -103 q 450 36 465 -6 q 407 113 435 79 q 338 167 379 147 q 245 187 297 187 q 191 181 218 187 q 139 167 165 176 q 126 181 134 170 q 115 197 119 191 l 399 570 l 174 570 q 152 565 163 570 q 131 550 141 561 q 112 519 121 539 q 96 468 103 499 l 70 476 l 70 642 q 102 635 90 638 q 128 632 115 632 q 156 632 141 632 l 518 632 l 535 602 l 271 251 q 293 254 282 254 q 314 254 303 254 q 413 235 367 254 q 492 182 459 217 q 545 97 526 148 q 565 -17 565 47 m 330 726 l 259 726 l 84 934 q 92 945 89 940 q 98 953 95 950 q 105 959 101 957 q 114 966 109 962 l 297 822 l 476 966 q 492 953 487 959 q 505 934 497 947 l 330 726 "
    },
    "Ç ": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 387 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 838 q 390 869 367 855 q 430 893 413 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 465 1044 q 459 1008 465 1025 q 442 979 453 991 q 418 959 432 966 q 387 952 404 952 q 344 968 357 952 q 332 1016 332 985 q 337 1052 332 1035 q 354 1082 343 1069 q 378 1101 365 1094 q 408 1109 392 1109 q 465 1044 465 1109 m 629 1275 q 624 1258 628 1268 q 617 1238 621 1248 q 609 1219 613 1229 q 603 1204 606 1210 l 194 1204 l 179 1220 q 184 1237 181 1228 q 191 1257 187 1247 q 199 1276 195 1266 q 207 1292 203 1285 l 614 1292 l 629 1275 "
    },
    "È¨": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 443 -155 q 431 -203 443 -180 q 395 -245 419 -226 q 331 -279 370 -264 q 238 -301 293 -293 l 224 -268 q 282 -253 258 -262 q 321 -233 306 -243 q 344 -210 337 -222 q 351 -187 351 -198 q 333 -155 351 -164 q 275 -142 316 -146 q 282 -123 276 -140 q 296 -78 286 -108 q 320 0 305 -50 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 412 432 427 427 q 377 438 397 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 q 620 63 628 107 q 606 0 611 19 l 373 0 l 349 -70 q 384 -81 367 -75 q 414 -98 401 -87 q 435 -122 427 -108 q 443 -155 443 -136 "
    },
    "Ì£": {
      "x_min": -409,
      "x_max": -276,
      "ha": 0,
      "o": "m -276 -189 q -281 -225 -276 -208 q -298 -254 -287 -242 q -322 -274 -308 -267 q -353 -282 -336 -282 q -396 -265 -383 -282 q -409 -217 -409 -248 q -403 -181 -409 -198 q -386 -151 -397 -164 q -362 -132 -375 -139 q -332 -125 -348 -125 q -276 -189 -276 -125 "
    },
    "Ä‘": {
      "x_min": 53,
      "x_max": 705.984375,
      "ha": 707,
      "o": "m 698 57 q 643 21 667 36 q 601 -2 619 6 q 570 -16 583 -12 q 548 -21 557 -21 q 512 9 526 -21 q 498 114 498 40 q 446 58 471 83 q 395 16 421 33 q 339 -10 368 -1 q 276 -20 310 -20 q 199 2 239 -20 q 127 65 160 24 q 73 166 94 106 q 53 301 53 226 q 63 381 53 340 q 92 461 73 422 q 140 534 112 500 q 202 595 168 569 q 278 636 237 621 q 366 651 319 651 q 428 643 397 651 q 495 608 459 636 l 495 773 l 309 773 l 293 787 q 300 810 297 798 q 309 834 303 823 l 495 834 l 495 864 q 493 924 495 902 q 481 958 491 946 q 453 975 472 971 q 398 980 433 980 l 398 1007 q 496 1027 453 1015 q 574 1051 539 1040 l 595 1031 l 595 834 l 690 834 l 705 817 l 690 773 l 595 773 l 595 172 q 595 131 595 148 q 597 103 595 114 q 601 84 599 91 q 608 72 604 76 q 632 67 614 64 q 689 86 649 70 l 698 57 m 495 177 l 495 495 q 433 555 473 534 q 345 577 393 577 q 267 561 303 577 q 207 513 232 545 q 167 433 181 481 q 153 322 153 386 q 169 217 153 264 q 210 137 185 170 q 266 85 236 103 q 323 68 296 68 q 368 77 346 68 q 411 102 390 87 q 453 137 432 117 q 495 177 474 156 "
    },
    "Ä‹": {
      "x_min": 53,
      "x_max": 558.140625,
      "ha": 595,
      "o": "m 558 129 q 482 47 514 76 q 421 2 449 17 q 366 -16 393 -12 q 308 -20 340 -20 q 214 2 260 -20 q 132 65 167 24 q 74 166 96 106 q 53 301 53 226 q 79 438 53 374 q 151 549 105 502 q 258 624 196 596 q 391 651 320 651 q 436 647 413 651 q 480 637 459 644 q 519 620 501 629 q 548 598 536 610 q 546 574 549 589 q 536 543 542 560 q 523 511 530 526 q 510 486 516 496 l 485 493 q 468 519 480 505 q 437 547 456 534 q 392 568 418 560 q 332 577 366 577 q 264 561 297 577 q 207 513 232 545 q 167 433 182 481 q 153 322 153 386 q 169 217 153 264 q 215 137 186 170 q 280 85 243 103 q 356 68 317 68 q 391 69 375 68 q 426 80 407 71 q 470 106 445 89 q 532 156 495 124 l 558 129 m 392 855 q 386 819 392 836 q 369 790 380 802 q 345 770 359 777 q 314 763 331 763 q 271 779 284 763 q 259 827 259 796 q 264 863 259 846 q 281 893 270 880 q 305 912 292 905 q 335 920 319 920 q 392 855 392 920 "
    },
    "Ä€": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 839 q 390 870 367 856 q 430 893 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 629 1058 q 624 1041 628 1051 q 617 1021 621 1031 q 609 1002 613 1012 q 603 987 606 993 l 194 987 l 179 1003 q 184 1020 181 1011 q 191 1040 187 1030 q 199 1059 195 1049 q 207 1075 203 1068 l 614 1075 l 629 1058 "
    },
    "áº‚": {
      "x_min": 13,
      "x_max": 1131,
      "ha": 1157,
      "o": "m 1131 824 q 1082 813 1101 818 q 1051 802 1063 807 q 1035 792 1040 797 q 1029 782 1030 788 l 877 39 q 863 14 874 25 q 838 -2 852 4 q 809 -14 824 -9 q 785 -20 795 -18 l 568 639 l 377 39 q 361 15 373 25 q 336 -2 350 4 q 306 -13 321 -9 q 277 -20 290 -18 l 104 777 q 80 805 101 794 q 13 824 60 816 l 13 854 l 298 854 l 298 824 q 246 816 265 821 q 218 805 228 811 q 207 791 209 799 q 207 777 206 784 l 339 169 l 555 854 l 591 854 l 827 169 l 951 782 q 945 797 952 790 q 924 807 938 803 q 890 816 910 812 q 849 824 871 820 l 849 854 l 1131 854 l 1131 824 m 484 926 q 467 937 472 931 q 454 957 462 944 l 711 1172 q 729 1161 718 1167 q 752 1147 740 1155 q 774 1134 764 1140 q 789 1121 784 1127 l 795 1091 l 484 926 "
    },
    "Ã¸": {
      "x_min": 53,
      "x_max": 631,
      "ha": 684,
      "o": "m 529 309 q 518 388 529 349 q 490 458 507 426 l 236 114 q 292 70 263 86 q 347 54 321 54 q 429 72 395 54 q 485 123 463 90 q 518 203 507 157 q 529 309 529 250 m 155 321 q 165 242 155 279 q 195 171 176 204 l 450 516 q 394 561 424 545 q 336 577 365 577 q 251 557 285 577 q 195 503 216 537 q 164 422 173 469 q 155 321 155 375 m 628 644 l 560 553 q 612 453 594 510 q 631 329 631 396 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 370 -20 q 244 -7 283 -20 q 173 29 205 5 l 160 12 q 143 0 154 7 q 120 -14 132 -7 q 97 -27 109 -21 q 77 -35 85 -32 l 58 -13 l 125 76 q 71 176 90 119 q 53 301 53 233 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 313 651 q 440 638 400 651 q 512 600 479 625 l 526 619 q 545 632 535 625 q 566 646 555 640 q 587 659 577 653 q 607 667 598 664 l 628 644 "
    },
    "Ã¢": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 149 302 128 288 q 202 330 169 317 q 287 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 529 740 q 516 721 521 727 q 500 710 512 715 l 318 891 l 138 710 q 129 715 132 712 q 122 721 125 718 q 116 729 119 724 q 108 740 112 734 l 283 997 l 354 997 l 529 740 "
    },
    "}": {
      "x_min": 25.640625,
      "x_max": 390.25,
      "ha": 457,
      "o": "m 390 441 q 271 379 311 420 q 232 286 232 338 q 233 239 232 258 q 238 204 235 220 q 245 175 241 188 q 251 145 248 161 q 256 110 254 129 q 258 62 258 90 q 243 -23 258 16 q 200 -98 228 -63 q 132 -161 172 -132 q 43 -213 92 -190 l 25 -180 q 73 -143 51 -163 q 112 -96 96 -122 q 138 -41 129 -71 q 148 24 148 -11 q 144 88 148 63 q 135 134 140 112 q 126 182 130 156 q 122 247 122 207 q 129 302 122 275 q 153 353 137 329 q 191 396 169 377 q 242 428 213 416 q 185 450 208 434 q 148 488 162 465 q 127 543 133 512 q 122 611 122 574 q 126 680 122 651 q 135 734 130 708 q 144 784 140 759 q 148 842 148 809 q 143 909 148 880 q 125 962 138 938 q 89 1004 112 985 q 32 1043 66 1024 l 47 1085 q 143 1037 103 1062 q 208 980 183 1012 q 246 911 234 949 q 258 824 258 873 q 254 757 258 784 q 245 707 250 731 q 236 655 240 683 q 232 586 232 628 q 258 503 232 531 q 335 475 285 475 l 348 475 q 355 475 352 475 q 363 476 359 475 l 378 479 l 390 441 "
    },
    "â€°": {
      "x_min": 51,
      "x_max": 1329,
      "ha": 1380,
      "o": "m 1232 196 q 1223 278 1232 243 q 1202 337 1215 313 q 1170 371 1188 360 q 1133 383 1152 383 q 1103 373 1118 383 q 1077 343 1089 363 q 1059 294 1066 324 q 1052 225 1052 265 q 1059 140 1052 176 q 1079 81 1066 104 q 1110 46 1092 58 q 1150 35 1129 35 q 1181 45 1166 35 q 1207 75 1196 55 q 1225 126 1218 95 q 1232 196 1232 156 m 1329 209 q 1313 120 1329 162 q 1270 47 1297 78 q 1208 -2 1243 15 q 1134 -21 1172 -21 q 1062 -2 1094 -21 q 1005 47 1029 15 q 968 120 981 78 q 956 209 956 162 q 972 298 956 256 q 1015 371 988 339 q 1077 420 1042 402 q 1150 439 1112 439 q 1223 421 1190 439 q 1279 371 1255 403 q 1316 298 1303 340 q 1329 209 1329 257 m 794 196 q 785 278 794 243 q 763 337 777 313 q 731 371 749 360 q 694 383 713 383 q 665 373 679 383 q 640 343 651 363 q 621 294 628 324 q 615 225 615 265 q 622 140 615 176 q 642 81 629 104 q 673 46 655 58 q 712 35 691 35 q 743 45 729 35 q 769 75 758 55 q 787 126 780 95 q 794 196 794 156 m 889 209 q 873 120 889 162 q 830 47 857 78 q 768 -2 804 15 q 694 -21 733 -21 q 623 -2 656 -21 q 567 47 591 15 q 531 120 544 78 q 518 209 518 162 q 534 298 518 256 q 577 371 550 339 q 639 420 604 402 q 712 439 674 439 q 784 421 752 439 q 840 371 817 403 q 876 298 863 340 q 889 209 889 257 m 238 11 q 220 1 232 6 q 195 -8 209 -3 q 169 -16 182 -12 q 148 -23 156 -21 l 132 0 l 701 805 q 744 825 720 817 q 790 839 769 833 l 809 816 l 238 11 m 328 595 q 319 678 328 643 q 297 736 311 713 q 265 770 283 759 q 227 782 247 782 q 198 772 213 782 q 172 742 184 762 q 154 693 161 723 q 147 624 147 664 q 154 539 147 575 q 174 480 161 503 q 206 445 187 457 q 245 434 224 434 q 276 444 261 434 q 303 474 291 454 q 321 525 314 494 q 328 595 328 555 m 424 608 q 408 519 424 561 q 365 446 392 477 q 302 396 338 414 q 227 378 267 378 q 156 396 188 378 q 100 446 124 414 q 63 519 76 477 q 51 608 51 561 q 66 697 51 655 q 109 770 82 738 q 172 819 136 801 q 245 838 207 838 q 318 820 285 838 q 374 770 350 802 q 411 697 398 739 q 424 608 424 656 "
    },
    "Ã„": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 387 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 222 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 105 75 98 55 l 352 838 q 390 869 367 855 q 430 893 413 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 568 31 q 600 75 607 52 l 524 302 l 252 302 m 598 1044 q 592 1008 598 1025 q 575 979 586 991 q 551 959 565 966 q 520 952 537 952 q 477 968 490 952 q 465 1016 465 985 q 470 1052 465 1035 q 487 1082 476 1069 q 511 1101 498 1094 q 541 1109 525 1109 q 598 1044 598 1109 m 332 1044 q 326 1008 332 1025 q 309 979 320 991 q 285 959 299 966 q 254 952 271 952 q 211 968 224 952 q 199 1016 199 985 q 204 1052 199 1035 q 221 1082 210 1069 q 245 1101 232 1094 q 275 1109 259 1109 q 332 1044 332 1109 "
    },
    "Å™": {
      "x_min": 37,
      "x_max": 517.765625,
      "ha": 540,
      "o": "m 512 626 q 517 602 517 621 q 512 556 517 582 q 499 503 507 530 q 483 458 491 476 l 455 458 q 443 504 451 485 q 426 534 436 523 q 405 550 417 545 q 381 556 394 556 q 347 543 366 556 q 307 504 327 530 q 266 435 286 477 q 230 336 245 393 l 230 70 q 254 49 230 60 q 342 29 279 38 l 342 0 l 37 0 l 37 29 q 105 49 80 39 q 130 70 130 58 l 130 465 q 128 502 130 487 q 125 527 127 518 q 120 542 123 537 q 114 551 117 547 q 104 560 110 556 q 89 565 98 563 q 68 568 80 568 q 37 568 55 568 l 37 597 q 121 620 80 608 q 196 651 163 632 l 224 627 l 224 472 q 267 543 246 510 q 312 599 288 575 q 362 637 336 623 q 417 651 388 651 q 463 645 438 651 q 512 626 487 640 m 335 726 l 264 726 l 88 968 q 96 979 93 974 q 103 987 100 984 q 109 993 106 991 q 118 999 113 996 l 301 829 l 481 999 q 497 987 492 993 q 509 968 501 981 l 335 726 "
    },
    "á¹¢": {
      "x_min": 80,
      "x_max": 589,
      "ha": 649,
      "o": "m 589 255 q 581 193 589 225 q 556 130 573 161 q 516 72 540 99 q 458 24 491 45 q 384 -7 426 4 q 293 -20 342 -20 q 245 -15 271 -20 q 191 -2 219 -10 q 137 18 164 6 q 87 46 110 30 q 81 69 82 48 q 80 116 80 89 q 80 172 80 144 q 87 219 80 201 l 114 216 q 152 147 129 176 q 203 98 175 117 q 263 70 231 79 q 330 61 296 61 q 390 73 360 61 q 444 107 420 86 q 483 158 468 129 q 499 219 499 187 q 481 290 499 261 q 434 343 463 320 q 367 385 405 366 q 291 421 330 403 q 215 460 252 440 q 149 507 178 480 q 102 570 120 535 q 85 655 85 606 q 90 701 85 676 q 110 750 96 725 q 144 797 123 774 q 195 837 165 820 q 263 864 224 854 q 350 875 301 875 q 410 870 380 875 q 466 857 440 865 q 514 837 493 849 q 547 812 535 826 q 547 796 552 808 q 534 770 543 784 q 514 743 525 756 q 495 722 503 729 l 471 726 q 432 763 453 748 q 391 789 412 779 q 349 802 370 798 q 310 807 329 807 q 247 795 273 807 q 203 766 220 783 q 178 727 186 749 q 171 687 171 706 q 188 631 171 655 q 236 585 206 606 q 302 546 265 564 q 379 508 340 528 q 456 465 419 488 q 523 412 494 442 q 571 344 553 382 q 589 255 589 305 m 399 -189 q 393 -225 399 -208 q 376 -254 387 -242 q 352 -274 366 -267 q 321 -282 338 -282 q 278 -265 291 -282 q 266 -217 266 -248 q 271 -181 266 -198 q 288 -151 277 -164 q 312 -132 299 -139 q 342 -125 326 -125 q 399 -189 399 -125 "
    },
    "â€”": {
      "x_min": 40.546875,
      "x_max": 1002.21875,
      "ha": 1043,
      "o": "m 1002 370 q 993 338 998 355 q 982 309 987 320 l 57 309 l 40 325 q 49 356 43 340 q 61 387 55 373 l 986 387 l 1002 370 "
    },
    "N": {
      "x_min": 33,
      "x_max": 851,
      "ha": 884,
      "o": "m 33 0 l 33 29 q 105 48 82 35 q 129 70 129 61 l 129 778 q 83 810 107 799 q 33 824 58 820 l 33 854 l 173 854 q 190 852 183 854 q 203 845 196 850 q 217 829 209 839 q 237 801 224 818 l 674 187 l 674 782 q 652 804 674 790 q 578 824 630 817 l 578 854 l 851 854 l 851 824 q 778 805 802 818 q 754 782 754 792 l 754 -20 q 701 -6 720 -16 q 674 14 682 2 l 209 673 l 209 70 q 231 49 209 62 q 305 29 253 36 l 305 0 l 33 0 "
    },
    "á¹¾": {
      "x_min": 13,
      "x_max": 857,
      "ha": 884,
      "o": "m 857 825 q 786 807 811 816 q 754 780 760 799 l 500 39 q 481 14 494 24 q 453 -3 468 3 q 423 -14 438 -10 q 400 -20 408 -18 l 107 780 q 77 808 100 797 q 13 825 54 819 l 13 854 l 302 854 l 302 823 q 229 813 248 823 q 216 783 209 803 l 452 137 l 675 780 q 659 809 682 799 q 581 825 635 818 l 581 855 l 857 855 l 857 825 m 502 -189 q 496 -225 502 -208 q 479 -254 490 -242 q 455 -274 469 -267 q 424 -282 441 -282 q 381 -265 394 -282 q 369 -217 369 -248 q 374 -181 369 -198 q 391 -151 380 -164 q 415 -132 402 -139 q 445 -125 429 -125 q 502 -189 502 -125 "
    },
    "â„": {
      "x_min": 131.59375,
      "x_max": 713.78125,
      "ha": 798,
      "o": "m 210 2 q 180 -10 194 -5 q 148 -20 166 -15 l 131 0 l 638 815 q 666 827 651 821 q 697 837 682 834 l 713 818 l 210 2 "
    },
    "2": {
      "x_min": 65.046875,
      "x_max": 550,
      "ha": 638,
      "o": "m 550 0 l 84 0 l 65 50 q 202 215 144 143 q 303 342 261 287 q 372 439 345 397 q 416 514 400 481 q 438 573 432 546 q 443 626 443 600 q 436 684 443 657 q 413 732 429 712 q 371 764 397 752 q 310 776 346 776 q 260 762 283 776 q 220 728 237 748 q 195 682 204 707 q 186 633 186 656 q 165 621 174 626 q 147 611 156 616 q 127 604 138 607 q 100 599 116 601 l 83 620 q 106 693 83 655 q 168 763 129 731 q 255 816 206 795 q 355 837 304 837 q 430 825 395 837 q 491 790 465 813 q 532 730 517 767 q 547 645 547 694 q 538 584 547 614 q 511 518 529 553 q 464 440 493 482 q 394 345 435 398 q 299 228 352 293 q 176 82 245 163 l 454 82 q 474 87 465 82 q 490 100 483 92 q 502 119 497 109 q 511 142 507 130 q 522 202 519 168 l 550 194 l 550 0 "
    },
    "Ã“": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 321 927 q 305 938 310 931 q 291 957 300 944 l 548 1172 q 566 1162 555 1168 q 590 1148 578 1155 q 612 1134 602 1141 q 627 1122 622 1127 l 633 1092 l 321 927 "
    },
    "Ëœ": {
      "x_min": 46.515625,
      "x_max": 499.78125,
      "ha": 546,
      "o": "m 499 912 q 471 857 487 884 q 434 808 454 830 q 390 773 414 787 q 340 760 366 760 q 296 771 317 760 q 255 798 275 783 q 215 824 235 812 q 176 836 196 836 q 128 814 150 836 q 81 754 106 792 l 46 768 q 74 823 58 795 q 111 872 91 850 q 155 907 132 893 q 205 921 179 921 q 253 909 230 921 q 295 883 275 897 q 333 856 315 868 q 367 845 351 845 q 417 866 392 845 q 463 927 441 888 l 499 912 "
    },
    "Ë‡": {
      "x_min": 33.234375,
      "x_max": 453.921875,
      "ha": 487,
      "o": "m 279 726 l 208 726 l 33 968 q 41 979 37 974 q 47 987 44 984 q 54 993 50 991 q 63 999 57 996 l 245 829 l 425 999 q 441 987 436 993 q 453 968 445 981 l 279 726 "
    },
    "ï„±": {
      "x_min": 0,
      "x_max": 196.140625,
      "ha": 226,
      "o": "m 196 0 l 196 -26 l 26 -26 l 26 -199 l 0 -199 l 0 0 l 196 0 "
    },
    "Å¬": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 847 855 l 847 825 m 666 1140 q 621 1047 646 1085 q 568 986 596 1010 q 511 953 541 963 q 451 943 481 943 q 388 953 418 943 q 330 986 357 963 q 278 1047 302 1010 q 234 1140 253 1085 q 249 1158 241 1152 q 269 1170 257 1165 q 307 1105 285 1132 q 352 1063 328 1079 q 401 1040 376 1047 q 448 1033 426 1033 q 497 1040 471 1033 q 547 1063 523 1047 q 593 1105 571 1079 q 630 1170 614 1132 q 650 1158 642 1165 q 666 1140 658 1152 "
    },
    "ÌŒ": {
      "x_min": -552.28125,
      "x_max": -131.59375,
      "ha": 0,
      "o": "m -305 726 l -376 726 l -552 968 q -544 979 -547 974 q -538 987 -540 984 q -531 993 -535 991 q -522 999 -527 996 l -339 829 l -160 999 q -144 987 -148 993 q -131 968 -139 981 l -305 726 "
    },
    "Ä": {
      "x_min": 20,
      "x_max": 657.625,
      "ha": 664,
      "o": "m 459 406 q 447 474 459 442 q 412 531 435 507 q 355 569 389 555 q 276 583 321 583 q 240 574 260 583 q 201 548 219 565 q 171 505 183 531 q 160 446 160 479 q 170 378 160 410 q 204 322 181 346 q 260 284 226 298 q 342 271 294 271 q 382 279 361 271 q 420 305 403 287 q 448 347 437 322 q 459 406 459 372 m 344 -2 q 297 3 319 0 q 257 10 276 6 q 182 -36 210 -15 q 141 -74 155 -57 q 123 -104 127 -90 q 120 -128 120 -118 q 138 -182 120 -157 q 188 -226 157 -208 q 260 -256 219 -245 q 347 -268 301 -268 q 428 -255 391 -268 q 490 -222 464 -243 q 530 -171 516 -200 q 545 -106 545 -141 q 536 -71 545 -87 q 505 -42 527 -55 q 443 -19 482 -29 q 344 -2 405 -9 m 552 434 q 529 339 552 382 q 468 265 506 296 q 384 217 431 234 q 288 200 337 200 l 285 200 q 241 154 253 172 q 230 132 230 136 q 236 116 230 124 q 262 101 243 109 q 315 87 281 94 q 400 74 348 80 q 520 50 472 66 q 595 12 567 33 q 634 -33 623 -8 q 645 -81 645 -57 q 631 -152 645 -118 q 593 -214 617 -185 q 536 -265 569 -242 q 466 -305 504 -288 q 389 -330 429 -321 q 310 -339 349 -339 q 246 -334 279 -339 q 180 -320 212 -330 q 118 -296 147 -311 q 67 -261 89 -282 q 32 -214 45 -240 q 20 -155 20 -188 q 26 -118 20 -137 q 51 -76 32 -99 q 105 -28 71 -54 q 197 28 140 -2 q 137 63 154 44 q 121 103 121 83 q 124 118 121 109 q 138 140 127 127 q 167 170 148 153 q 216 209 186 187 q 155 236 183 218 q 108 280 128 254 q 77 337 88 305 q 67 408 67 370 q 88 502 67 457 q 146 579 110 546 q 228 631 182 612 q 322 651 274 651 q 397 639 362 651 q 460 606 432 627 q 523 615 495 610 q 573 627 551 621 q 613 639 595 633 q 644 651 630 645 l 657 630 q 642 595 649 611 q 619 562 635 579 q 570 554 594 557 q 515 552 545 552 q 542 496 532 525 q 552 434 552 467 m 542 740 q 529 721 534 727 q 513 710 525 715 l 331 891 l 151 710 q 142 715 146 712 q 136 721 139 718 q 129 729 133 724 q 121 740 126 734 l 297 997 l 368 997 l 542 740 "
    },
    "â„¦": {
      "x_min": 54,
      "x_max": 780,
      "ha": 860,
      "o": "m 73 0 l 54 18 q 54 55 54 34 q 57 99 54 76 q 64 145 60 122 q 73 189 68 168 l 103 189 q 114 139 108 161 q 126 103 119 118 q 143 81 133 88 q 167 74 153 74 l 313 74 q 195 185 242 132 q 122 290 149 238 q 83 394 94 342 q 73 501 73 446 q 83 590 73 545 q 116 677 94 635 q 169 754 138 718 q 242 817 201 790 q 335 859 284 844 q 445 875 385 875 q 579 849 518 875 q 684 778 640 824 q 752 665 727 731 q 777 516 777 599 q 761 402 777 456 q 713 295 745 347 q 633 188 681 242 q 523 74 586 134 l 676 74 q 699 81 690 74 q 715 101 708 87 q 731 137 723 115 q 751 192 739 160 l 780 179 q 777 135 780 160 q 772 87 775 111 q 765 40 769 63 q 759 0 762 17 l 465 0 l 465 74 q 534 166 503 126 q 587 242 564 207 q 626 307 610 276 q 652 367 642 337 q 666 430 662 398 q 671 501 671 462 q 655 619 671 563 q 608 719 639 675 q 531 788 577 762 q 425 814 485 814 q 344 798 380 814 q 281 757 308 782 q 235 699 254 731 q 205 632 217 666 q 187 566 193 598 q 182 508 182 533 q 182 434 182 468 q 190 370 182 400 q 212 308 198 339 q 249 242 226 277 q 304 166 272 207 q 382 74 337 126 l 382 0 l 73 0 "
    },
    "s": {
      "x_min": 63,
      "x_max": 465,
      "ha": 525,
      "o": "m 465 192 q 451 109 465 144 q 417 51 438 75 q 369 13 395 28 q 318 -7 343 0 q 271 -17 293 -15 q 236 -20 248 -20 q 159 -7 204 -20 q 70 29 114 4 q 64 51 65 31 q 63 97 63 71 q 66 150 63 123 q 74 192 69 176 l 104 184 q 117 131 104 156 q 153 87 130 106 q 205 58 175 69 q 269 48 234 48 q 313 55 293 48 q 348 77 333 63 q 370 111 362 91 q 379 154 379 130 q 364 202 379 181 q 327 240 350 223 q 274 273 303 258 q 213 304 244 288 q 159 335 185 319 q 113 371 133 351 q 81 417 93 392 q 69 474 69 442 q 85 549 69 516 q 129 604 102 582 q 192 639 157 627 q 266 651 228 651 q 311 647 287 651 q 361 636 336 643 q 406 620 385 629 q 441 598 427 611 q 442 580 445 594 q 432 549 439 566 q 419 517 426 532 q 408 497 412 502 l 382 502 q 316 570 350 551 q 249 590 281 590 q 209 582 227 590 q 179 563 192 575 q 161 537 167 551 q 155 508 155 522 q 167 469 155 487 q 201 437 180 452 q 248 408 221 421 q 302 379 274 394 q 360 347 331 364 q 412 308 389 330 q 450 258 435 286 q 465 192 465 230 "
    },
    "Çš": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 467 70 q 369 3 398 18 q 316 -15 340 -10 q 271 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 43 q 113 189 113 125 l 113 481 q 111 532 113 514 q 100 559 109 550 q 75 571 92 568 q 27 578 57 575 l 27 605 q 72 612 51 608 q 112 622 93 616 q 152 634 132 627 q 194 650 172 641 l 213 624 l 213 226 q 219 147 213 178 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 482 q 505 531 508 512 q 492 559 502 549 q 464 573 482 569 q 414 578 446 578 l 414 605 q 507 624 463 611 q 588 650 551 637 l 608 624 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 562 854 q 556 818 562 835 q 539 789 550 801 q 515 769 529 776 q 484 762 501 762 q 441 778 454 762 q 429 826 429 795 q 434 862 429 845 q 451 892 440 879 q 475 911 462 904 q 505 919 489 919 q 562 854 562 919 m 296 854 q 290 818 296 835 q 273 789 284 801 q 249 769 263 776 q 218 762 235 762 q 175 778 188 762 q 163 826 163 795 q 168 862 163 845 q 185 892 174 879 q 209 911 196 904 q 239 919 223 919 q 296 854 296 919 m 397 970 l 327 970 l 151 1212 q 159 1223 156 1218 q 166 1231 163 1228 q 172 1237 169 1235 q 181 1243 176 1240 l 364 1073 l 543 1243 q 559 1231 555 1237 q 572 1212 564 1225 l 397 970 "
    },
    "Ì€": {
      "x_min": -483.171875,
      "x_max": -212.015625,
      "ha": 0,
      "o": "m -212 731 q -229 718 -220 722 q -249 709 -237 713 l -483 965 l -468 993 q -448 997 -461 995 q -421 1002 -435 1000 q -393 1007 -407 1005 q -371 1009 -379 1009 l -212 731 "
    },
    "?": {
      "x_min": 53,
      "x_max": 536,
      "ha": 590,
      "o": "m 536 790 q 520 698 536 739 q 480 622 504 658 q 428 555 456 587 q 374 488 400 523 q 329 415 348 454 q 311 328 311 377 l 311 263 q 287 247 297 254 q 257 236 277 241 l 237 250 l 237 328 q 249 405 237 367 q 282 481 262 444 q 327 554 302 518 q 375 625 353 590 q 412 693 397 660 q 427 759 427 727 q 389 891 427 844 q 281 939 351 939 q 236 927 257 939 q 197 898 214 916 q 171 855 181 879 q 162 804 162 831 q 164 784 162 795 q 170 765 166 774 q 125 748 148 753 q 72 739 102 742 l 54 759 q 53 770 53 765 l 53 782 q 75 870 53 829 q 137 940 98 911 q 227 987 176 970 q 333 1004 277 1004 q 420 988 382 1004 q 483 944 458 972 q 522 877 509 916 q 536 790 536 837 m 358 83 q 351 39 358 59 q 331 4 344 19 q 302 -18 319 -10 q 266 -27 285 -27 q 215 -7 230 -27 q 200 48 200 12 q 207 91 200 71 q 227 127 214 111 q 256 151 239 142 q 292 160 273 160 q 341 140 325 160 q 358 83 358 120 "
    },
    "á»¡": {
      "x_min": 53,
      "x_max": 731,
      "ha": 731,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 731 706 q 723 669 731 690 q 699 625 716 649 q 652 577 681 602 q 578 527 622 551 q 617 436 604 486 q 631 329 631 387 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 465 628 415 651 q 550 565 514 606 q 601 612 585 589 q 617 652 617 635 q 606 689 617 671 q 578 722 596 706 l 696 776 q 721 743 711 761 q 731 706 731 726 m 582 913 q 553 858 570 885 q 516 809 536 830 q 472 773 496 787 q 422 760 448 760 q 378 771 399 760 q 337 798 357 783 q 297 825 317 813 q 258 837 278 837 q 210 814 232 837 q 163 754 188 792 l 128 768 q 157 823 140 796 q 193 872 173 851 q 238 908 214 894 q 287 922 261 922 q 335 910 312 922 q 377 883 357 898 q 415 856 397 868 q 449 845 433 845 q 499 866 474 845 q 546 928 523 888 l 582 913 "
    },
    "Äª": {
      "x_min": 31.71875,
      "x_max": 374.578125,
      "ha": 406,
      "o": "m 47 0 l 47 29 q 118 49 92 38 q 144 70 144 61 l 144 782 q 119 803 144 790 q 47 824 95 815 l 47 854 l 346 854 l 346 824 q 275 803 300 815 q 250 782 250 792 l 250 70 q 274 50 250 62 q 346 29 298 38 l 346 0 l 47 0 m 374 1085 q 369 1068 373 1078 q 362 1048 366 1058 q 354 1029 358 1039 q 348 1014 351 1020 l 46 1014 l 31 1030 q 36 1047 33 1038 q 43 1067 39 1057 q 51 1086 47 1076 q 59 1102 55 1095 l 359 1102 l 374 1085 "
    },
    "â€¢": {
      "x_min": 47,
      "x_max": 331,
      "ha": 378,
      "o": "m 331 448 q 319 373 331 407 q 286 315 307 339 q 237 277 265 291 q 173 264 208 264 q 117 275 141 264 q 78 306 94 286 q 54 354 62 327 q 47 415 47 382 q 59 489 47 455 q 92 548 71 523 q 142 586 114 572 q 205 600 171 600 q 259 588 235 600 q 298 557 282 577 q 322 509 314 537 q 331 448 331 482 "
    },
    "(": {
      "x_min": 80,
      "x_max": 405,
      "ha": 431,
      "o": "m 383 -213 q 251 -114 308 -178 q 156 31 194 -51 q 99 214 118 115 q 80 422 80 314 q 101 637 80 533 q 163 828 123 741 q 259 982 202 916 q 383 1085 315 1048 l 405 1054 q 323 958 361 1017 q 255 819 284 898 q 209 645 226 741 q 193 441 193 549 q 206 248 193 342 q 248 72 220 154 q 314 -74 275 -8 q 405 -183 353 -140 l 383 -213 "
    },
    "â—Š": {
      "x_min": 0.671875,
      "x_max": 451.265625,
      "ha": 452,
      "o": "m 84 448 l 85 448 l 239 142 l 366 424 l 366 425 l 212 731 l 84 448 m 0 425 l 182 816 q 198 829 189 823 q 216 842 206 836 q 234 854 225 848 q 251 863 243 859 l 451 448 l 269 57 q 253 44 262 51 q 234 31 244 37 q 216 20 225 25 q 200 11 207 14 l 0 425 "
    },
    "á»–": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 619 957 q 606 938 611 944 q 590 927 602 932 l 408 1068 l 228 927 q 219 932 223 929 q 212 938 215 935 q 206 946 209 942 q 198 957 203 951 l 374 1167 l 445 1167 l 619 957 m 636 1374 q 607 1319 624 1346 q 570 1270 590 1292 q 526 1235 550 1249 q 476 1222 502 1222 q 432 1233 453 1222 q 391 1260 411 1245 q 351 1286 371 1274 q 312 1298 332 1298 q 264 1276 286 1298 q 217 1216 242 1254 l 182 1230 q 210 1285 194 1257 q 247 1334 227 1312 q 291 1369 268 1355 q 341 1383 315 1383 q 389 1371 366 1383 q 431 1345 411 1359 q 469 1318 451 1330 q 503 1307 487 1307 q 553 1328 528 1307 q 600 1389 577 1350 l 636 1374 "
    },
    "á¸…": {
      "x_min": 6.59375,
      "x_max": 631,
      "ha": 691,
      "o": "m 631 333 q 621 253 631 294 q 591 172 611 211 q 544 97 572 133 q 481 36 516 62 q 403 -4 445 10 q 311 -20 360 -20 q 276 -14 298 -20 q 225 2 254 -8 q 162 29 196 13 q 93 65 128 44 l 93 878 q 89 926 93 910 q 77 952 86 943 q 50 963 67 960 q 6 969 33 966 l 6 996 q 90 1018 50 1007 q 168 1051 130 1030 l 175 1044 q 183 1036 179 1040 q 193 1028 187 1032 l 193 493 q 258 563 226 534 q 322 612 291 593 q 378 641 352 632 q 423 651 404 651 q 506 629 468 651 q 572 566 544 607 q 615 466 599 525 q 631 333 631 407 m 541 277 q 527 397 541 344 q 490 486 513 450 q 438 540 468 521 q 376 560 408 560 q 345 552 364 560 q 301 529 325 545 q 248 484 276 512 q 193 414 220 456 l 193 137 q 247 103 219 117 q 299 82 274 90 q 344 71 324 74 q 376 68 364 68 q 447 85 417 68 q 498 133 478 103 q 530 200 519 163 q 541 277 541 237 m 385 -189 q 379 -225 385 -208 q 362 -254 373 -242 q 338 -274 352 -267 q 307 -282 324 -282 q 264 -265 277 -282 q 252 -217 252 -248 q 257 -181 252 -198 q 274 -151 263 -164 q 298 -132 285 -139 q 328 -125 312 -125 q 385 -189 385 -125 "
    },
    "Ã›": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 616 816 l 568 855 l 847 855 l 847 825 m 660 958 q 647 939 652 945 q 631 927 642 932 l 449 1068 l 269 927 q 260 932 263 930 q 253 939 256 935 q 247 947 250 942 q 239 958 243 951 l 414 1167 l 485 1167 l 660 958 "
    },
    "áº¦": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 839 q 390 869 367 856 q 430 893 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 608 957 q 595 938 600 944 q 579 927 590 932 l 397 1068 l 216 927 q 208 932 211 929 q 201 938 204 935 q 195 946 198 942 q 187 957 191 951 l 362 1167 l 434 1167 l 608 957 m 516 1228 q 501 1209 506 1216 q 486 1198 496 1203 l 173 1363 l 179 1393 q 194 1405 183 1399 q 216 1419 204 1412 q 239 1433 228 1426 q 258 1443 250 1439 l 516 1228 "
    },
    "V": {
      "x_min": 13,
      "x_max": 857,
      "ha": 884,
      "o": "m 857 825 q 786 807 811 816 q 754 780 760 799 l 499 39 q 481 14 494 24 q 453 -3 468 3 q 423 -14 437 -10 q 399 -20 408 -18 l 107 780 q 77 808 100 797 q 13 825 54 819 l 13 854 l 302 854 l 302 823 q 229 813 248 823 q 216 783 209 803 l 451 137 l 675 780 q 659 809 682 799 q 581 825 635 818 l 581 854 l 857 854 l 857 825 "
    },
    "á»¸": {
      "x_min": -0.265625,
      "x_max": 781,
      "ha": 808,
      "o": "m 239 0 l 239 29 q 323 55 299 42 q 348 78 348 68 l 348 364 q 292 478 323 419 q 228 594 260 538 q 165 699 195 651 q 111 780 134 748 q 97 794 105 788 q 78 806 90 801 q 48 814 66 811 q 2 818 30 818 l 0 846 q 78 856 38 852 q 145 861 117 861 q 197 834 175 861 q 250 757 221 802 q 307 663 278 713 q 363 562 335 614 q 413 461 390 510 l 592 780 q 585 808 602 798 q 518 825 569 818 l 518 854 l 781 854 l 781 825 q 711 807 735 816 q 677 780 686 798 l 454 366 l 454 78 q 459 68 454 73 q 478 55 465 62 q 512 42 491 48 q 565 29 533 35 l 565 0 l 239 0 m 627 1103 q 599 1048 615 1075 q 562 999 582 1020 q 518 963 542 977 q 468 950 494 950 q 423 961 444 950 q 383 988 402 973 q 343 1015 363 1003 q 304 1027 324 1027 q 256 1004 278 1027 q 209 944 234 982 l 174 958 q 202 1013 186 986 q 239 1062 219 1041 q 283 1098 260 1084 q 333 1112 307 1112 q 381 1100 358 1112 q 423 1073 403 1088 q 461 1046 443 1058 q 495 1035 479 1035 q 545 1056 520 1035 q 592 1118 569 1078 l 627 1103 "
    },
    "á¹¿": {
      "x_min": 13,
      "x_max": 657,
      "ha": 671,
      "o": "m 657 601 q 624 592 637 597 q 603 583 611 588 q 591 572 595 578 q 583 555 587 565 l 395 39 q 378 14 389 24 q 353 -3 367 3 q 327 -14 340 -10 q 305 -20 314 -18 l 86 555 q 63 584 80 574 q 13 601 46 594 l 13 631 l 270 631 l 270 601 q 227 595 244 598 q 202 586 211 591 q 192 573 193 581 q 194 555 190 566 l 357 120 l 511 555 q 513 572 514 565 q 504 585 512 580 q 481 594 496 590 q 442 601 466 597 l 442 631 l 657 631 l 657 601 m 409 -189 q 403 -225 409 -208 q 386 -254 397 -242 q 362 -274 376 -267 q 331 -282 348 -282 q 288 -265 301 -282 q 276 -217 276 -248 q 281 -181 276 -198 q 298 -151 287 -164 q 322 -132 309 -139 q 352 -125 336 -125 q 409 -189 409 -125 "
    },
    "Ì±": {
      "x_min": -566.90625,
      "x_max": -117.640625,
      "ha": 0,
      "o": "m -117 -155 q -122 -172 -118 -162 q -129 -192 -125 -182 q -137 -211 -133 -201 q -143 -227 -140 -220 l -552 -227 l -566 -210 q -562 -193 -565 -202 q -554 -173 -558 -183 q -546 -154 -550 -164 q -539 -139 -542 -145 l -132 -139 l -117 -155 "
    },
    "@": {
      "x_min": 47,
      "x_max": 1053,
      "ha": 1100,
      "o": "m 674 495 q 632 555 655 533 q 570 576 610 576 q 503 560 534 576 q 451 512 473 544 q 417 433 429 480 q 406 322 406 385 q 419 217 406 264 q 454 136 433 170 q 500 85 475 103 q 548 67 525 67 q 573 72 561 67 q 599 88 585 76 q 630 119 612 99 q 674 169 648 139 l 674 495 m 1053 372 q 1040 261 1053 313 q 1004 166 1027 209 q 953 88 982 123 q 893 29 924 53 q 829 -7 861 5 q 769 -20 797 -20 q 738 -12 753 -20 q 711 9 723 -5 q 691 49 699 25 q 678 109 682 74 q 629 46 650 71 q 588 6 608 21 q 548 -14 569 -8 q 501 -20 527 -20 q 432 2 467 -20 q 369 65 397 24 q 323 166 341 106 q 306 301 306 226 q 314 381 306 339 q 341 461 323 422 q 383 534 358 499 q 440 595 408 568 q 509 636 472 621 q 590 651 547 651 q 618 648 606 651 q 643 639 631 646 q 670 620 656 632 q 701 588 684 608 q 736 614 718 599 q 773 651 754 629 l 793 630 q 783 588 787 610 q 776 543 779 568 q 774 490 774 518 l 774 193 q 793 103 774 132 q 845 74 813 74 q 884 95 862 74 q 926 153 906 116 q 959 240 945 190 q 973 347 973 290 q 944 550 973 461 q 865 699 915 639 q 743 791 814 759 q 588 823 673 823 q 461 805 521 823 q 353 756 402 787 q 266 681 304 724 q 201 585 228 637 q 160 475 174 533 q 147 355 147 416 q 163 202 147 273 q 211 72 180 131 q 283 -31 241 14 q 376 -108 325 -77 q 483 -156 426 -140 q 600 -173 541 -173 q 710 -161 656 -173 q 809 -131 763 -149 q 890 -93 855 -113 q 946 -55 926 -73 l 964 -92 q 902 -146 940 -118 q 813 -199 863 -175 q 701 -238 763 -222 q 568 -254 639 -254 q 358 -214 453 -254 q 193 -99 262 -174 q 85 79 124 -25 q 47 315 47 185 q 67 465 47 392 q 124 604 87 539 q 213 725 162 670 q 328 819 265 779 q 464 881 392 859 q 614 904 537 904 q 794 869 713 904 q 932 768 875 835 q 1021 601 990 701 q 1053 372 1053 502 "
    },
    "Ê¼": {
      "x_min": 53.3125,
      "x_max": 246,
      "ha": 304,
      "o": "m 246 900 q 215 807 246 854 q 120 719 185 760 l 94 741 q 137 794 122 767 q 153 852 153 821 q 148 887 153 871 q 132 915 143 903 q 104 935 122 928 q 60 943 86 943 l 53 971 q 76 986 58 978 q 115 1001 93 994 q 157 1012 137 1008 q 188 1015 177 1016 q 216 995 205 1007 q 234 967 228 983 q 243 935 241 952 q 246 900 246 917 "
    },
    "i": {
      "x_min": 47,
      "x_max": 333,
      "ha": 370,
      "o": "m 47 0 l 47 29 q 115 49 92 38 q 139 70 139 61 l 139 454 q 139 510 139 488 q 130 543 139 531 q 101 560 121 555 q 47 569 82 566 l 47 596 q 90 606 67 600 q 136 619 113 612 q 179 634 158 626 q 217 651 200 642 l 240 651 l 240 70 q 262 50 240 62 q 333 29 284 38 l 333 0 l 47 0 m 260 854 q 254 818 260 835 q 237 789 248 801 q 213 769 227 776 q 182 762 199 762 q 139 778 152 762 q 127 826 127 795 q 132 862 127 845 q 149 892 138 879 q 173 911 160 904 q 203 919 187 919 q 260 854 260 919 "
    },
    "È¯": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 422 855 q 416 819 422 836 q 399 790 410 802 q 375 770 389 777 q 344 763 361 763 q 301 779 314 763 q 289 827 289 796 q 294 863 289 846 q 311 893 300 880 q 335 912 322 905 q 365 920 349 920 q 422 855 422 920 "
    },
    "â‰¤": {
      "x_min": 40.34375,
      "x_max": 556.734375,
      "ha": 597,
      "o": "m 556 300 q 543 287 549 293 q 531 277 538 282 q 518 266 525 272 q 500 255 510 261 l 56 424 q 47 435 50 431 q 42 441 44 439 q 41 443 41 443 l 41 445 q 48 476 44 460 q 60 508 53 492 l 540 691 l 556 674 q 552 658 554 666 q 547 639 550 649 q 542 621 544 630 q 537 606 539 612 l 168 466 l 545 324 l 556 300 m 556 193 q 547 164 553 179 q 537 136 541 148 l 56 136 l 40 154 q 49 182 43 167 q 60 210 54 197 l 540 210 l 556 193 "
    },
    "áº½": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 290 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 565 913 q 536 858 553 885 q 499 809 519 830 q 455 773 479 787 q 405 760 431 760 q 361 771 382 760 q 320 798 340 783 q 281 825 300 813 q 242 837 261 837 q 194 814 216 837 q 147 754 172 792 l 112 768 q 140 823 124 796 q 177 872 156 851 q 221 908 197 894 q 270 922 245 922 q 318 910 296 922 q 361 883 341 898 q 399 856 381 868 q 432 845 416 845 q 482 866 458 845 q 529 928 506 888 l 565 913 "
    },
    "Ä•": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 290 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 554 923 q 509 830 534 868 q 457 769 484 793 q 399 736 429 746 q 340 726 369 726 q 276 736 307 726 q 218 769 246 746 q 167 830 191 793 q 123 923 142 868 q 138 941 130 935 q 158 953 146 948 q 196 888 174 915 q 241 846 218 862 q 290 823 265 830 q 337 816 315 816 q 386 823 360 816 q 435 846 411 830 q 481 888 460 862 q 519 953 503 915 q 538 941 530 948 q 554 923 546 935 "
    },
    "á¹§": {
      "x_min": 53.03125,
      "x_max": 473.640625,
      "ha": 525,
      "o": "m 465 192 q 451 109 465 144 q 417 51 438 75 q 369 13 395 28 q 318 -7 343 0 q 271 -17 293 -15 q 236 -20 248 -20 q 159 -7 204 -20 q 70 29 114 4 q 64 51 65 31 q 63 97 63 71 q 66 150 63 123 q 74 192 69 176 l 104 184 q 117 131 104 156 q 153 87 130 106 q 205 58 175 69 q 269 48 234 48 q 313 55 293 48 q 348 77 333 63 q 370 111 362 91 q 379 154 379 130 q 364 202 379 181 q 326 240 350 223 q 273 273 303 258 q 213 304 244 288 q 159 335 185 319 q 113 371 133 351 q 81 417 93 392 q 69 474 69 442 q 85 549 69 516 q 129 604 102 582 q 192 639 157 627 q 266 651 228 651 q 311 647 287 651 q 361 636 336 643 q 406 620 385 629 q 441 598 427 611 q 442 580 445 594 q 432 549 439 566 q 419 517 426 532 q 408 497 412 502 l 382 502 q 315 570 350 551 q 248 590 281 590 q 209 582 226 590 q 179 563 192 575 q 161 537 167 551 q 155 508 155 522 q 167 469 155 487 q 200 437 180 452 q 247 408 221 421 q 302 379 274 394 q 359 347 330 364 q 412 308 389 330 q 450 258 435 286 q 465 192 465 230 m 299 727 l 228 727 l 53 969 q 60 980 57 975 q 67 988 64 985 q 73 994 70 992 q 82 1000 77 997 l 265 830 l 445 1000 q 461 988 456 994 q 473 969 465 982 l 299 727 m 330 1072 q 324 1036 330 1053 q 307 1007 318 1019 q 283 987 297 994 q 252 980 269 980 q 209 996 222 980 q 197 1044 197 1013 q 202 1080 197 1063 q 219 1110 208 1097 q 243 1129 230 1122 q 273 1137 257 1137 q 330 1072 330 1137 "
    },
    "á»ˆ": {
      "x_min": 47,
      "x_max": 346,
      "ha": 406,
      "o": "m 47 0 l 47 29 q 118 49 92 38 q 144 70 144 61 l 144 782 q 119 803 144 790 q 47 824 95 815 l 47 854 l 346 854 l 346 824 q 275 803 300 815 q 250 782 250 792 l 250 70 q 274 50 250 62 q 346 29 298 38 l 346 0 l 47 0 m 308 1121 q 295 1087 308 1102 q 267 1059 283 1072 q 234 1034 250 1046 q 212 1009 219 1022 q 211 983 204 997 q 245 953 217 970 q 225 944 235 946 q 204 942 214 943 q 160 969 174 957 q 143 993 146 982 q 146 1013 140 1004 q 164 1032 152 1023 q 189 1050 175 1041 q 215 1068 203 1059 q 235 1088 227 1078 q 244 1111 244 1099 q 231 1150 244 1137 q 199 1163 218 1163 q 181 1159 189 1163 q 166 1149 172 1155 q 157 1135 160 1143 q 154 1120 154 1128 q 155 1113 154 1117 q 159 1106 157 1110 q 145 1102 154 1104 q 127 1097 137 1099 q 109 1094 118 1095 q 93 1091 99 1092 l 86 1099 l 86 1108 q 98 1140 86 1124 q 130 1168 111 1155 q 174 1188 150 1180 q 221 1196 198 1196 q 285 1175 262 1196 q 308 1121 308 1155 "
    },
    "Å¼": {
      "x_min": 50.453125,
      "x_max": 548.984375,
      "ha": 601,
      "o": "m 547 168 q 547 121 547 146 q 547 74 547 97 q 547 31 547 51 q 542 0 547 12 l 67 0 l 50 30 l 418 569 l 189 569 q 167 564 178 569 q 146 549 156 560 q 127 518 136 538 q 111 467 118 498 l 85 475 l 85 641 q 117 634 105 637 q 142 631 129 631 q 170 631 155 631 l 534 631 l 548 601 l 179 61 l 459 61 q 489 87 475 61 q 519 174 503 113 l 547 168 m 385 854 q 379 818 385 835 q 362 789 373 801 q 338 769 352 776 q 307 762 324 762 q 264 778 277 762 q 252 826 252 795 q 257 862 252 845 q 274 892 263 879 q 298 911 285 904 q 328 919 312 919 q 385 854 385 919 "
    },
    "Æ˜": {
      "x_min": 33,
      "x_max": 819,
      "ha": 839,
      "o": "m 33 0 l 33 29 q 104 50 78 39 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 332 854 l 332 824 q 261 803 286 815 q 236 782 236 792 l 236 438 l 476 747 q 566 832 519 803 q 664 861 612 861 q 723 852 695 861 q 772 825 751 844 q 806 778 793 807 q 819 709 819 749 q 811 639 819 677 q 796 625 809 633 q 765 608 783 616 q 727 592 747 599 q 692 580 707 584 l 680 599 q 697 652 692 628 q 702 696 702 676 q 683 758 702 737 q 637 780 665 780 q 609 773 623 780 q 583 756 596 767 q 557 732 570 746 q 534 706 545 719 l 332 456 l 654 86 q 679 65 666 73 q 708 55 693 58 q 741 52 724 52 q 775 54 757 52 l 779 25 q 701 1 740 12 q 629 -10 661 -10 q 595 -3 610 -10 q 567 19 580 2 l 236 433 l 236 70 q 259 50 236 62 q 332 29 283 39 l 332 0 l 33 0 "
    },
    "Å‘": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 247 710 q 229 717 235 713 q 209 728 222 720 l 326 1010 q 344 1007 334 1008 q 366 1003 355 1005 q 388 998 377 1000 q 405 993 398 996 l 419 965 l 247 710 m 440 710 q 421 717 427 713 q 402 728 414 720 l 519 1010 q 537 1007 527 1008 q 559 1003 548 1005 q 580 998 570 1000 q 598 993 591 996 l 613 965 l 440 710 "
    },
    "ï‡ˆ": {
      "x_min": 21,
      "x_max": 402,
      "ha": 414,
      "o": "m 165 550 l 165 622 q 176 679 165 653 q 204 728 187 705 q 241 771 222 751 q 278 812 261 792 q 307 852 296 832 q 319 896 319 872 q 309 941 319 921 q 283 975 299 961 q 246 997 267 990 q 202 1005 225 1005 q 169 998 186 1005 q 140 980 153 991 q 119 954 127 969 q 111 923 111 939 q 112 911 111 917 q 116 899 114 905 q 78 889 100 893 q 34 884 55 886 l 21 896 q 21 903 21 899 l 21 910 q 38 964 21 938 q 86 1009 56 990 q 155 1039 116 1028 q 237 1051 194 1051 q 306 1040 276 1051 q 358 1011 337 1029 q 390 967 379 992 q 402 914 402 943 q 390 860 402 884 q 362 816 379 836 q 325 777 344 796 q 288 738 305 758 q 259 696 270 718 q 248 646 248 674 l 248 550 l 165 550 "
    },
    "á»±": {
      "x_min": 27.265625,
      "x_max": 834,
      "ha": 834,
      "o": "m 834 707 q 823 662 834 688 q 787 608 813 637 q 718 549 762 579 q 608 491 674 519 l 608 172 q 611 103 608 130 q 623 70 614 77 q 650 67 631 64 q 704 85 670 71 l 710 55 q 662 25 685 39 q 620 1 639 11 q 586 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 432 44 467 70 q 369 4 398 18 q 316 -14 340 -9 q 271 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 533 113 515 q 100 560 109 551 q 75 572 92 569 q 27 579 57 576 l 27 606 q 72 613 51 609 q 112 623 93 617 q 152 635 132 628 q 194 651 172 642 l 213 625 l 213 226 q 219 147 213 178 q 239 96 226 115 q 271 68 252 77 q 314 60 290 60 q 357 67 335 60 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 483 q 505 532 508 513 q 492 560 502 550 q 464 574 482 570 q 415 579 446 579 l 415 606 q 506 625 463 612 q 588 651 550 638 l 608 625 l 608 538 q 659 568 638 553 q 693 599 680 584 q 713 627 707 614 q 720 652 720 641 q 710 689 720 671 q 681 722 700 707 l 799 777 q 824 744 814 762 q 834 707 834 726 m 429 -189 q 423 -225 429 -208 q 406 -254 417 -242 q 382 -274 396 -267 q 351 -282 368 -282 q 308 -265 321 -282 q 296 -217 296 -248 q 301 -181 296 -198 q 318 -151 307 -164 q 342 -132 329 -139 q 372 -125 356 -125 q 429 -189 429 -125 "
    },
    "ÅŽ": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 625 1140 q 580 1047 605 1085 q 527 986 555 1010 q 470 953 500 963 q 410 943 440 943 q 347 953 378 943 q 289 986 316 963 q 237 1047 261 1010 q 193 1140 213 1085 q 209 1158 201 1152 q 228 1170 217 1165 q 266 1105 245 1132 q 312 1063 288 1079 q 361 1040 336 1047 q 407 1033 386 1033 q 456 1040 431 1033 q 506 1063 482 1047 q 552 1105 530 1079 q 590 1170 574 1132 q 609 1158 601 1165 q 625 1140 617 1152 "
    },
    "È±": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 422 855 q 416 819 422 836 q 399 790 410 802 q 375 770 389 777 q 344 763 361 763 q 301 779 314 763 q 289 827 289 796 q 294 863 289 846 q 311 893 300 880 q 335 912 322 905 q 365 920 349 920 q 422 855 422 920 m 586 1112 q 581 1095 585 1105 q 574 1075 578 1085 q 566 1056 570 1066 q 560 1041 563 1047 l 152 1041 l 137 1057 q 142 1074 138 1065 q 149 1094 145 1084 q 157 1113 153 1103 q 164 1129 161 1122 l 571 1129 l 586 1112 "
    },
    "áº©": {
      "x_min": 53,
      "x_max": 615.6875,
      "ha": 625,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 522 -2 559 15 q 466 -21 485 -21 q 429 10 444 -21 q 415 97 415 41 q 354 44 384 66 q 295 7 324 21 q 241 -14 267 -7 q 194 -21 215 -21 q 145 -12 170 -21 q 99 12 120 -4 q 66 58 79 30 q 53 125 53 86 q 70 211 53 177 q 112 271 88 246 q 148 302 128 288 q 202 330 169 317 q 286 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 281 583 q 224 557 239 569 q 200 527 209 544 q 194 490 192 511 q 180 476 194 484 q 146 463 166 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 579 74 l 615 55 m 529 740 q 516 721 521 727 q 500 710 511 715 l 318 891 l 138 710 q 129 715 132 712 q 122 721 125 718 q 116 729 119 724 q 108 740 112 734 l 283 997 l 354 997 l 529 740 m 430 1209 q 417 1175 430 1190 q 389 1147 405 1160 q 357 1122 372 1134 q 334 1098 341 1110 q 333 1072 326 1085 q 367 1041 339 1058 q 347 1033 357 1034 q 326 1030 336 1032 q 282 1058 296 1045 q 265 1081 268 1070 q 268 1101 262 1092 q 286 1120 274 1111 q 311 1138 297 1129 q 337 1156 325 1147 q 357 1176 349 1166 q 366 1199 366 1187 q 353 1238 366 1225 q 321 1251 340 1251 q 303 1247 311 1251 q 288 1237 294 1243 q 279 1223 282 1231 q 276 1209 276 1216 q 277 1201 276 1205 q 281 1194 279 1198 q 267 1190 276 1192 q 249 1185 259 1187 q 231 1182 240 1183 q 215 1179 221 1180 l 208 1187 l 208 1196 q 220 1228 208 1212 q 252 1256 233 1243 q 296 1276 272 1268 q 343 1284 320 1284 q 407 1263 384 1284 q 430 1209 430 1243 "
    },
    "Ä°": {
      "x_min": 47,
      "x_max": 346,
      "ha": 406,
      "o": "m 47 0 l 47 29 q 118 49 92 38 q 144 70 144 61 l 144 782 q 119 803 144 790 q 47 824 94 815 l 47 854 l 346 854 l 346 824 q 275 803 300 815 q 250 782 250 792 l 250 70 q 274 50 250 62 q 346 29 298 38 l 346 0 l 47 0 m 263 1044 q 257 1008 263 1025 q 240 979 251 991 q 216 959 230 966 q 185 952 202 952 q 142 968 155 952 q 130 1016 130 985 q 135 1052 130 1035 q 152 1082 141 1069 q 176 1101 163 1094 q 206 1109 190 1109 q 263 1044 263 1109 "
    },
    "Äš": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 413 432 428 427 q 377 438 398 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 369 943 l 298 943 l 123 1151 q 131 1162 127 1157 q 137 1170 134 1167 q 144 1176 140 1174 q 153 1183 147 1179 l 335 1039 l 515 1183 q 531 1170 526 1176 q 543 1151 535 1164 l 369 943 "
    },
    "á»": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 619 957 q 606 938 611 944 q 590 927 602 932 l 408 1068 l 228 927 q 219 932 223 929 q 212 938 215 935 q 206 946 209 942 q 198 957 203 951 l 374 1167 l 445 1167 l 619 957 m 321 1198 q 305 1209 310 1203 q 291 1228 300 1216 l 548 1443 q 566 1433 555 1439 q 590 1419 578 1426 q 612 1405 602 1412 q 627 1393 622 1399 l 633 1363 l 321 1198 "
    },
    "Ç£": {
      "x_min": 53,
      "x_max": 889,
      "ha": 942,
      "o": "m 889 378 q 855 352 875 364 q 814 332 835 340 l 515 332 q 514 308 514 320 l 514 283 q 524 206 514 244 q 554 137 534 168 q 603 88 574 107 q 669 70 632 70 q 709 72 690 70 q 749 83 728 74 q 797 111 771 93 q 858 160 823 129 q 873 147 867 156 q 883 133 880 138 q 810 54 843 84 q 746 7 777 24 q 687 -14 716 -8 q 627 -20 658 -20 q 569 -10 598 -20 q 516 17 541 0 q 471 62 491 35 q 437 122 451 88 q 375 60 408 86 q 308 15 341 33 q 245 -11 275 -2 q 194 -20 215 -20 q 145 -11 170 -20 q 99 13 120 -3 q 66 59 79 31 q 53 126 53 87 q 68 203 53 166 q 115 272 84 240 q 194 330 146 304 q 307 375 242 356 q 357 386 329 382 q 414 390 384 390 l 414 448 q 410 499 414 474 q 394 542 406 523 q 363 571 383 560 q 310 583 342 583 q 264 573 289 583 q 221 546 240 563 q 193 506 202 529 q 193 458 183 483 q 181 445 195 452 q 147 431 167 437 q 108 422 128 425 q 80 421 88 419 l 71 445 q 90 498 74 472 q 130 546 106 523 q 184 588 154 569 q 244 621 213 607 q 303 643 274 635 q 355 651 332 651 q 448 626 413 651 q 497 553 482 602 q 542 599 518 580 q 590 629 566 618 q 638 645 615 640 q 680 651 661 651 q 752 639 721 651 q 806 607 783 627 q 845 560 829 587 q 871 503 861 533 q 885 440 881 473 q 889 378 889 408 m 423 165 q 416 205 418 185 q 414 248 414 225 l 414 329 q 374 326 392 329 q 349 321 357 324 q 205 253 249 296 q 162 155 162 211 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 q 274 74 251 68 q 321 92 296 80 q 372 122 346 104 q 423 165 398 141 m 660 583 q 625 575 645 583 q 585 547 605 568 q 547 489 565 526 q 521 394 529 452 l 752 394 q 772 399 766 394 q 778 418 778 404 q 771 480 778 450 q 750 532 764 510 q 714 569 736 555 q 660 583 692 583 m 721 868 q 716 851 720 861 q 709 831 713 841 q 701 812 705 822 q 695 797 698 803 l 287 797 l 272 813 q 277 830 273 821 q 284 850 280 840 q 292 869 288 859 q 299 885 296 878 l 706 885 l 721 868 "
    },
    "É„": {
      "x_min": 29.015625,
      "x_max": 853.640625,
      "ha": 890,
      "o": "m 468 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 417 l 236 417 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 m 847 825 q 776 804 801 815 q 751 783 751 792 l 751 478 l 837 478 l 853 462 l 837 417 l 751 417 l 751 355 q 729 196 751 266 q 668 78 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 97 q 130 301 130 218 l 130 417 l 43 417 l 29 433 q 36 455 32 444 q 43 478 39 467 l 130 478 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 478 l 665 478 l 665 783 q 640 803 665 791 q 568 825 616 815 l 568 854 l 847 854 l 847 825 "
    },
    "á¹š": {
      "x_min": 27.015625,
      "x_max": 756.890625,
      "ha": 764,
      "o": "m 33 0 l 33 29 q 104 49 78 38 q 130 70 130 60 l 130 806 q 82 799 106 803 q 33 792 57 795 l 27 834 q 90 849 58 842 q 156 862 122 856 q 229 871 191 868 q 311 875 267 875 q 445 859 388 875 q 540 816 502 843 q 597 750 578 788 q 616 669 616 712 q 602 581 616 620 q 563 510 588 541 q 504 458 538 480 q 428 424 469 437 l 629 94 q 651 69 639 78 q 677 56 663 60 q 709 52 691 52 q 749 54 726 52 l 756 25 q 673 0 713 10 q 608 -9 633 -9 q 570 1 588 -9 q 542 27 552 12 l 341 408 q 324 407 332 407 l 306 407 q 271 408 288 407 q 236 414 254 410 l 236 70 q 259 50 236 62 q 332 29 283 38 l 332 0 l 33 0 m 287 817 q 236 817 261 817 l 236 469 q 267 465 253 465 q 295 465 280 465 q 456 511 400 465 q 512 648 512 558 q 500 716 512 685 q 459 769 487 747 q 389 804 431 792 q 287 817 346 817 m 412 -189 q 406 -225 412 -208 q 389 -254 400 -242 q 365 -274 379 -267 q 334 -282 351 -282 q 291 -265 304 -282 q 279 -217 279 -248 q 284 -181 279 -198 q 301 -151 290 -164 q 325 -132 312 -139 q 355 -125 339 -125 q 412 -189 412 -125 "
    },
    "á¸·": {
      "x_min": 40,
      "x_max": 339,
      "ha": 369,
      "o": "m 40 0 l 40 29 q 88 38 68 33 q 118 49 107 44 q 135 59 130 54 q 140 70 140 65 l 140 878 q 136 926 140 909 q 121 951 132 943 q 92 963 110 960 q 46 969 74 966 l 46 996 q 134 1017 94 1006 q 216 1051 174 1029 l 240 1028 l 240 70 q 262 49 240 60 q 339 29 284 38 l 339 0 l 40 0 m 256 -189 q 250 -225 256 -208 q 233 -254 244 -242 q 209 -274 223 -267 q 178 -282 195 -282 q 135 -265 148 -282 q 123 -217 123 -248 q 128 -181 123 -198 q 145 -151 134 -164 q 169 -132 156 -139 q 199 -125 183 -125 q 256 -189 256 -125 "
    },
    "Ç™": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 615 816 l 568 855 l 847 855 l 847 825 m 649 1045 q 643 1009 649 1026 q 626 980 637 992 q 602 960 616 967 q 571 953 588 953 q 528 969 541 953 q 516 1017 516 986 q 521 1053 516 1036 q 538 1083 527 1070 q 562 1102 549 1095 q 592 1110 576 1110 q 649 1045 649 1110 m 383 1045 q 377 1009 383 1026 q 360 980 371 992 q 336 960 350 967 q 305 953 322 953 q 262 969 275 953 q 250 1017 250 986 q 255 1053 250 1036 q 272 1083 261 1070 q 296 1102 283 1095 q 326 1110 310 1110 q 383 1045 383 1110 m 485 1161 l 414 1161 l 239 1369 q 247 1380 243 1375 q 253 1388 250 1385 q 259 1394 256 1392 q 268 1401 263 1397 l 451 1257 l 631 1401 q 647 1388 642 1394 q 660 1369 652 1382 l 485 1161 "
    },
    "â€¹": {
      "x_min": 53,
      "x_max": 321.5,
      "ha": 415,
      "o": "m 292 11 l 53 291 l 53 315 q 53 332 53 325 q 54 341 53 340 l 292 619 l 320 599 l 314 586 q 296 554 307 574 q 272 509 286 534 q 243 457 258 483 q 215 405 229 430 q 191 360 201 380 q 173 328 180 340 q 167 317 167 317 l 321 32 l 292 11 "
    },
    "á»§": {
      "x_min": 27.265625,
      "x_max": 710.359375,
      "ha": 721,
      "o": "m 710 55 q 662 25 685 39 q 620 1 639 11 q 585 -14 600 -8 q 563 -20 571 -20 q 524 11 539 -20 q 510 112 510 42 q 433 44 467 70 q 369 4 398 18 q 316 -14 341 -9 q 271 -20 292 -20 q 211 -11 239 -20 q 160 20 182 -2 q 125 84 138 44 q 113 189 113 125 l 113 482 q 111 532 113 514 q 100 559 109 550 q 75 572 92 568 q 27 579 57 575 l 27 606 q 72 613 51 608 q 112 622 93 617 q 152 635 132 627 q 194 651 172 642 l 213 624 l 213 226 q 219 147 213 179 q 239 96 226 115 q 271 69 252 77 q 314 61 290 61 q 357 67 335 61 q 403 87 380 73 q 452 123 427 101 q 508 177 478 145 l 508 482 q 505 531 508 513 q 492 560 502 549 q 464 574 482 570 q 414 578 445 578 l 414 606 q 506 625 463 612 q 588 651 550 638 l 608 624 l 608 172 q 610 103 608 130 q 623 70 613 77 q 650 68 631 65 q 703 86 669 71 l 710 55 m 474 904 q 461 870 474 885 q 432 842 449 855 q 400 817 415 829 q 377 793 385 805 q 376 766 370 780 q 411 736 383 753 q 390 728 401 729 q 370 725 380 726 q 325 753 339 740 q 308 776 311 765 q 311 796 305 787 q 329 815 317 806 q 354 833 340 824 q 381 851 368 842 q 401 871 393 861 q 410 894 410 882 q 397 933 410 920 q 364 946 384 946 q 346 942 354 946 q 331 932 337 938 q 322 918 325 926 q 319 903 319 911 q 320 896 319 900 q 324 889 322 893 q 310 885 319 887 q 292 880 302 882 q 274 877 283 878 q 258 874 264 875 l 251 882 l 251 891 q 263 923 251 907 q 295 951 276 938 q 339 971 315 963 q 386 979 363 979 q 451 958 428 979 q 474 904 474 938 "
    },
    "áº°": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 838 q 390 869 367 855 q 430 892 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 m 614 1139 q 569 1046 594 1084 q 516 985 544 1009 q 459 952 489 962 q 399 942 429 942 q 336 952 366 942 q 277 985 305 962 q 226 1046 250 1009 q 181 1139 201 1084 q 197 1157 189 1151 q 216 1169 205 1164 q 255 1104 233 1131 q 300 1062 276 1078 q 349 1039 324 1046 q 396 1032 374 1032 q 445 1039 419 1032 q 495 1062 471 1046 q 541 1104 519 1078 q 579 1169 563 1131 q 598 1157 590 1164 q 614 1139 606 1151 m 516 1174 q 501 1155 506 1161 q 486 1143 496 1148 l 173 1308 l 179 1339 q 194 1351 183 1344 q 216 1365 204 1358 q 239 1378 228 1372 q 258 1389 250 1385 l 516 1174 "
    },
    "Ê’": {
      "x_min": 25,
      "x_max": 565,
      "ha": 612,
      "o": "m 565 -17 q 537 -148 565 -89 q 465 -249 509 -207 q 365 -315 420 -292 q 253 -339 309 -339 q 164 -326 206 -339 q 91 -295 122 -313 q 42 -258 60 -277 q 25 -228 25 -240 q 34 -214 25 -224 q 57 -193 43 -204 q 87 -173 71 -183 q 117 -162 103 -164 q 195 -240 156 -213 q 279 -268 235 -268 q 353 -253 319 -268 q 412 -212 387 -239 q 451 -144 437 -184 q 465 -52 465 -103 q 450 36 465 -6 q 407 113 435 79 q 338 167 379 147 q 245 187 297 187 q 191 181 218 187 q 139 167 165 176 q 126 181 134 170 q 115 197 119 191 l 399 570 l 174 570 q 152 565 163 570 q 131 550 141 561 q 112 519 121 539 q 96 468 103 499 l 70 476 l 70 642 q 102 635 90 638 q 128 632 115 632 q 156 632 141 632 l 518 632 l 535 602 l 271 251 q 293 254 282 254 q 314 254 303 254 q 413 235 367 254 q 492 182 459 217 q 545 97 526 148 q 565 -17 565 47 "
    },
    "á¸¢": {
      "x_min": 33,
      "x_max": 844,
      "ha": 890,
      "o": "m 33 0 l 33 29 q 104 49 78 38 q 130 70 130 60 l 130 783 q 105 803 130 791 q 33 825 81 815 l 33 854 l 332 854 l 332 825 q 261 804 286 815 q 236 783 236 792 l 236 478 l 641 478 l 641 783 q 617 803 641 791 q 545 825 592 815 l 545 854 l 844 854 l 844 825 q 772 804 798 815 q 747 783 747 792 l 747 70 q 771 50 747 62 q 844 29 795 38 l 844 0 l 545 0 l 545 29 q 615 49 590 38 q 641 70 641 60 l 641 417 l 236 417 l 236 70 q 259 50 236 62 q 332 29 283 38 l 332 0 l 33 0 m 505 1044 q 499 1008 505 1025 q 482 979 493 991 q 458 959 472 966 q 427 952 444 952 q 384 968 397 952 q 372 1016 372 985 q 377 1052 372 1035 q 394 1082 383 1069 q 418 1101 405 1094 q 448 1109 432 1109 q 505 1044 505 1109 "
    },
    "Ã¬": {
      "x_min": -5.5,
      "x_max": 333,
      "ha": 370,
      "o": "m 47 0 l 47 29 q 115 49 92 38 q 139 70 139 61 l 139 454 q 139 510 139 488 q 130 543 139 531 q 101 560 121 555 q 47 569 82 566 l 47 596 q 90 606 67 600 q 136 619 113 612 q 179 634 158 626 q 217 651 200 642 l 240 651 l 240 70 q 262 50 240 62 q 333 29 284 38 l 333 0 l 47 0 m 265 731 q 248 718 257 722 q 228 709 240 713 l -5 965 l 9 993 q 28 997 15 995 q 56 1002 41 1000 q 84 1007 70 1005 q 106 1009 98 1009 l 265 731 "
    },
    "Â±": {
      "x_min": 40.953125,
      "x_max": 517.046875,
      "ha": 558,
      "o": "m 517 146 q 508 116 513 132 q 497 88 502 99 l 57 88 l 40 104 q 49 133 44 117 q 61 162 55 149 l 501 162 l 517 146 m 317 248 q 287 235 303 241 q 257 227 272 230 l 241 241 l 241 432 l 57 432 l 40 448 q 49 477 44 461 q 61 506 55 493 l 241 506 l 241 692 q 270 702 253 696 q 301 712 286 708 l 317 695 l 317 506 l 501 506 l 517 490 q 508 460 513 476 q 497 432 502 443 l 317 432 l 317 248 "
    },
    "|": {
      "x_min": 120,
      "x_max": 213,
      "ha": 312,
      "o": "m 213 -239 q 180 -259 201 -250 q 143 -274 159 -269 l 120 -259 l 120 1099 q 154 1118 138 1110 q 189 1132 169 1127 l 213 1117 l 213 -239 "
    },
    "Â§": {
      "x_min": 79,
      "x_max": 559,
      "ha": 638,
      "o": "m 357 358 q 392 339 374 349 q 426 320 410 330 q 459 355 447 334 q 472 402 472 376 q 459 459 472 433 q 422 508 447 484 q 358 556 397 532 q 268 609 320 580 q 239 626 253 617 q 212 645 225 635 q 191 630 200 641 q 177 607 183 620 q 168 581 171 595 q 166 557 166 567 q 177 501 166 525 q 211 455 188 477 q 271 410 235 433 q 357 358 307 387 m 559 444 q 552 388 559 414 q 535 341 546 363 q 510 304 524 320 q 482 276 496 287 q 509 233 499 257 q 520 177 520 209 q 506 96 520 129 q 472 40 493 62 q 424 5 450 18 q 373 -12 399 -6 q 326 -19 348 -19 q 291 -19 304 -19 q 255 -15 275 -19 q 214 -3 236 -11 q 170 13 192 3 q 125 35 147 23 q 119 57 120 37 q 118 103 118 77 q 121 156 118 130 q 129 198 124 182 l 159 191 q 172 136 159 162 q 208 90 185 110 q 260 57 230 69 q 324 46 289 46 q 403 69 372 46 q 434 140 434 92 q 419 187 434 167 q 381 222 405 206 q 328 252 358 238 q 268 283 298 266 q 189 333 225 308 q 130 385 154 357 q 92 447 105 414 q 79 525 79 481 q 86 573 79 550 q 107 617 94 597 q 136 655 120 638 q 172 683 153 671 q 136 742 149 710 q 124 813 124 774 q 140 886 124 855 q 184 937 157 917 q 247 966 212 957 q 321 976 283 976 q 366 971 341 976 q 415 958 391 966 q 461 939 440 950 q 496 917 482 929 q 497 898 500 912 q 487 867 494 884 q 474 835 481 850 q 462 815 466 820 l 437 820 q 370 892 405 869 q 304 915 336 915 q 235 895 260 915 q 210 840 210 876 q 219 797 210 816 q 247 759 228 778 q 293 723 265 741 q 357 683 320 704 q 438 634 402 658 q 502 584 475 611 q 544 524 529 557 q 559 444 559 491 "
    },
    "È©": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 442 -154 q 430 -203 442 -180 q 394 -245 418 -226 q 330 -278 369 -264 q 237 -300 291 -292 l 222 -267 q 281 -252 257 -261 q 320 -232 305 -243 q 343 -209 336 -221 q 350 -186 350 -197 q 332 -154 350 -163 q 274 -141 315 -145 q 281 -122 275 -139 q 294 -77 285 -108 q 313 -20 302 -55 l 308 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 291 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 493 52 526 83 q 430 6 459 22 q 371 -14 400 -9 q 366 -15 368 -15 l 348 -70 q 383 -80 366 -74 q 413 -97 400 -87 q 434 -121 426 -107 q 442 -154 442 -135 m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 "
    },
    "É¨": {
      "x_min": 31.71875,
      "x_max": 347.625,
      "ha": 370,
      "o": "m 260 854 q 254 818 260 835 q 237 789 248 801 q 213 769 227 776 q 182 762 199 762 q 139 778 152 762 q 127 826 127 795 q 132 862 127 845 q 149 892 138 879 q 173 911 160 904 q 203 919 187 919 q 260 854 260 919 m 47 0 l 47 28 q 116 49 92 38 q 140 70 140 60 l 140 281 l 47 281 l 31 295 q 38 318 35 306 q 47 342 41 331 l 139 342 l 139 453 q 139 509 139 488 q 130 543 139 531 q 101 560 121 554 q 47 569 82 565 l 47 596 q 90 606 67 600 q 136 619 113 611 q 179 634 158 626 q 217 650 200 642 l 240 650 l 240 342 l 333 342 l 347 325 l 333 281 l 240 281 l 240 70 q 262 50 240 61 q 333 28 284 38 l 333 0 l 47 0 "
    },
    "Ë": {
      "x_min": 110.984375,
      "x_max": 560.265625,
      "ha": 671,
      "o": "m 560 -66 q 555 -83 558 -73 q 548 -103 552 -93 q 540 -122 544 -112 q 534 -138 537 -131 l 125 -138 l 110 -121 q 115 -104 112 -113 q 122 -84 118 -94 q 130 -65 126 -75 q 138 -50 134 -56 l 545 -50 l 560 -66 "
    },
    "ïˆ’": {
      "x_min": 44,
      "x_max": 965,
      "ha": 817,
      "o": "m 361 71 q 417 84 388 71 q 477 119 447 98 q 537 169 508 141 q 594 224 567 197 l 594 671 q 523 753 568 724 q 421 783 478 783 q 359 772 391 783 q 296 742 327 762 q 238 693 266 722 q 189 624 210 663 q 156 537 168 585 q 144 432 144 489 q 162 288 144 355 q 212 174 181 222 q 282 98 243 125 q 361 71 321 71 m 594 152 q 532 79 562 111 q 469 24 501 47 q 401 -9 436 2 q 325 -21 365 -21 q 263 -8 295 -21 q 200 28 231 4 q 140 89 168 53 q 90 173 112 126 q 56 278 69 221 q 44 404 44 336 q 55 505 44 457 q 87 595 67 553 q 131 672 106 637 q 181 735 155 707 q 231 783 206 763 q 274 815 255 803 q 321 840 297 830 q 367 859 345 851 q 410 871 390 867 q 445 875 430 875 q 486 872 466 875 q 527 861 507 869 q 570 836 548 852 q 617 795 592 821 q 639 812 628 803 q 660 833 650 822 q 679 854 670 843 q 696 873 688 864 l 724 847 q 712 792 717 822 q 705 730 708 766 q 702 653 702 695 l 702 -72 q 702 -162 702 -124 q 710 -226 702 -201 q 733 -263 718 -251 q 777 -275 749 -275 q 805 -269 792 -275 q 828 -254 818 -264 q 844 -233 838 -245 q 850 -207 850 -221 q 842 -179 850 -194 q 853 -165 842 -173 q 883 -150 865 -158 q 919 -138 900 -143 q 952 -133 938 -133 l 965 -159 q 943 -224 965 -192 q 888 -281 922 -256 q 813 -323 855 -307 q 728 -339 770 -339 q 659 -323 685 -339 q 618 -278 633 -307 q 598 -207 603 -249 q 594 -114 594 -166 l 594 152 "
    },
    "q": {
      "x_min": 53,
      "x_max": 687,
      "ha": 701,
      "o": "m 323 68 q 368 77 346 68 q 411 102 390 87 q 453 137 432 117 q 495 177 474 156 l 495 495 q 433 555 473 534 q 345 577 393 577 q 273 561 308 577 q 212 513 239 545 q 169 433 185 481 q 153 322 153 386 q 169 217 153 264 q 210 137 185 170 q 266 85 236 103 q 323 68 296 68 m 382 -325 l 382 -295 q 469 -275 444 -286 q 495 -253 495 -265 l 495 112 q 445 57 469 81 q 394 15 420 32 q 339 -10 368 -1 q 276 -20 310 -20 q 199 2 239 -20 q 127 65 160 24 q 73 166 94 106 q 53 301 53 226 q 71 411 53 361 q 117 500 90 461 q 174 566 144 539 q 228 607 205 593 q 303 640 266 628 q 366 651 340 651 q 402 649 384 651 q 438 640 420 646 q 477 621 457 633 q 519 588 497 608 q 540 603 529 594 q 561 620 551 611 q 580 637 571 628 q 595 651 589 645 l 615 631 q 604 589 609 612 q 597 543 600 570 q 595 486 595 517 l 595 -253 q 615 -275 595 -264 q 687 -295 636 -286 l 687 -325 l 382 -325 "
    },
    "É‘": {
      "x_min": 53,
      "x_max": 669.984375,
      "ha": 679,
      "o": "m 669 57 q 571 -1 611 17 q 522 -20 531 -20 q 482 9 494 -20 q 470 110 470 38 q 417 49 440 74 q 370 8 393 24 q 321 -13 346 -6 q 268 -20 297 -20 q 194 2 232 -20 q 124 65 155 24 q 72 166 92 106 q 53 301 53 226 q 62 381 53 340 q 91 461 72 422 q 137 534 110 500 q 197 595 163 569 q 272 636 231 621 q 358 651 312 651 q 391 649 376 651 q 422 640 406 646 q 454 620 437 633 q 492 587 471 608 q 530 613 511 598 q 568 651 548 629 l 587 631 q 577 589 581 611 q 570 544 573 569 q 568 490 568 519 l 568 172 q 570 106 568 131 q 579 74 573 80 q 606 68 585 65 q 662 87 628 72 q 663 83 662 88 q 665 72 664 78 q 669 57 667 65 m 468 173 l 468 495 q 416 555 446 534 q 338 577 385 577 q 262 561 296 577 q 203 513 227 545 q 166 433 179 481 q 153 322 153 386 q 168 217 153 264 q 207 137 183 170 q 260 85 231 103 q 316 68 288 68 q 383 93 348 68 q 468 173 417 119 "
    },
    "á»™": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 308 q 510 410 528 362 q 466 496 493 459 q 404 554 438 533 q 336 576 369 576 q 251 556 285 576 q 195 502 216 537 q 165 421 174 468 q 156 320 156 375 q 174 218 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 156 q 528 308 528 250 m 631 329 q 619 240 631 283 q 588 158 608 196 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 471 74 432 q 141 543 114 510 q 202 600 169 576 q 275 637 236 623 q 355 651 314 651 q 468 626 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 422 -189 q 416 -225 422 -208 q 399 -254 410 -242 q 375 -274 389 -267 q 344 -282 361 -282 q 301 -265 314 -282 q 289 -217 289 -248 q 294 -181 289 -198 q 311 -151 300 -164 q 335 -132 322 -139 q 365 -125 349 -125 q 422 -189 422 -125 m 565 740 q 552 721 557 727 q 536 710 547 715 l 354 891 l 174 710 q 165 715 169 712 q 159 721 161 718 q 152 729 156 724 q 144 740 149 734 l 320 997 l 391 997 l 565 740 "
    },
    "Â®": {
      "x_min": 23,
      "x_max": 462,
      "ha": 485,
      "o": "m 462 734 q 445 643 462 685 q 399 568 428 600 q 329 518 370 536 q 241 500 289 500 q 153 518 193 500 q 84 568 113 536 q 39 643 55 600 q 23 734 23 685 q 39 825 23 783 q 84 900 55 868 q 153 950 113 932 q 241 969 193 969 q 329 950 289 969 q 399 900 370 932 q 445 825 428 868 q 462 734 462 783 m 423 734 q 409 810 423 774 q 371 872 395 845 q 314 913 348 898 q 241 929 281 929 q 169 913 202 929 q 113 872 136 898 q 75 810 89 845 q 62 734 62 774 q 75 658 62 693 q 113 596 89 623 q 169 555 136 570 q 241 540 202 540 q 314 555 281 540 q 371 596 348 570 q 409 658 395 623 q 423 734 423 693 m 142 600 l 142 613 q 171 626 171 619 l 171 846 l 141 842 l 141 862 q 181 871 160 867 q 230 875 202 875 q 306 856 280 875 q 332 810 332 838 q 313 757 332 778 q 264 729 294 737 l 333 632 q 346 623 338 625 q 368 622 353 621 l 369 608 q 341 600 356 604 q 320 597 327 597 q 308 601 314 597 q 299 609 302 605 l 229 725 l 228 725 q 208 727 218 725 l 208 626 q 236 613 208 619 l 236 600 l 142 600 m 223 851 l 215 851 q 208 850 212 851 l 208 751 q 216 750 213 750 l 225 750 q 275 764 259 750 q 291 804 291 778 q 276 837 291 823 q 223 851 261 851 "
    },
    "á¹¬": {
      "x_min": 6.609375,
      "x_max": 719,
      "ha": 750,
      "o": "m 204 0 l 204 29 q 255 42 234 35 q 289 55 276 49 q 307 68 301 62 q 313 78 313 73 l 313 794 l 114 794 q 96 790 104 794 q 79 774 87 786 q 60 738 71 762 q 35 675 49 714 l 6 688 q 15 773 9 728 q 28 855 21 819 l 699 855 l 719 838 q 713 762 719 803 q 699 678 707 721 l 669 678 q 658 731 663 709 q 646 767 653 753 q 630 787 639 781 q 607 794 621 794 l 419 794 l 419 78 q 442 56 419 69 q 528 29 466 42 l 528 0 l 204 0 m 429 -189 q 423 -225 429 -208 q 406 -254 417 -242 q 382 -274 396 -267 q 351 -282 368 -282 q 308 -265 321 -282 q 296 -217 296 -248 q 301 -181 296 -198 q 318 -151 307 -164 q 342 -132 329 -139 q 372 -125 356 -125 q 429 -189 429 -125 "
    },
    "á¹“": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 631 329 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 468 627 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 586 868 q 581 851 585 861 q 574 831 578 841 q 566 812 570 822 q 560 797 563 803 l 152 797 l 137 813 q 142 830 138 821 q 149 850 145 840 q 157 869 153 859 q 164 885 161 878 l 571 885 l 586 868 m 326 954 q 305 961 314 956 q 288 973 296 967 l 444 1253 q 465 1251 452 1253 q 492 1247 478 1249 q 520 1243 506 1245 q 540 1238 533 1240 l 555 1211 l 326 954 "
    },
    "á¸±": {
      "x_min": 37,
      "x_max": 681.265625,
      "ha": 681,
      "o": "m 37 1 l 37 29 q 105 49 80 41 q 130 70 130 58 l 130 879 q 126 927 130 910 q 112 952 122 944 q 83 963 102 961 q 37 970 65 966 l 37 997 q 127 1019 82 1006 q 205 1051 171 1031 l 230 1028 l 230 359 l 434 542 q 461 573 456 561 q 459 591 465 585 q 438 599 453 597 q 406 602 423 602 l 406 631 l 658 631 l 658 602 q 604 590 628 598 q 551 559 579 583 l 324 374 l 574 75 q 594 56 584 64 q 616 45 604 49 q 643 41 628 41 q 677 41 658 41 l 681 13 q 643 4 661 8 q 607 -1 624 1 q 578 -4 591 -3 q 558 -6 565 -6 q 514 2 531 -6 q 482 29 497 10 l 230 354 l 230 70 q 232 61 230 65 q 241 52 234 57 q 263 43 249 48 q 303 29 278 37 l 303 1 l 37 1 m 271 1096 q 254 1108 259 1101 q 241 1127 249 1114 l 498 1342 q 516 1331 505 1338 q 539 1318 527 1325 q 561 1304 551 1311 q 576 1292 571 1297 l 582 1261 l 271 1096 "
    },
    "á»": {
      "x_min": 53,
      "x_max": 631.015625,
      "ha": 684,
      "o": "m 528 308 q 510 410 528 362 q 466 496 493 459 q 404 554 438 533 q 336 576 369 576 q 251 556 285 576 q 195 502 216 537 q 165 421 174 468 q 156 320 156 375 q 174 218 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 156 q 528 308 528 250 m 631 329 q 619 240 631 283 q 588 158 608 196 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 471 74 432 q 141 543 114 510 q 202 600 169 576 q 275 637 236 623 q 355 651 314 651 q 468 626 417 651 q 555 559 519 602 q 611 457 591 516 q 631 329 631 398 m 422 -189 q 416 -225 422 -208 q 399 -254 410 -242 q 375 -274 389 -267 q 344 -282 361 -282 q 301 -265 314 -282 q 289 -217 289 -248 q 294 -181 289 -198 q 311 -151 300 -164 q 335 -132 322 -139 q 365 -125 349 -125 q 422 -189 422 -125 "
    },
    "áº–": {
      "x_min": 37,
      "x_max": 732,
      "ha": 752,
      "o": "m 446 0 l 446 29 q 516 51 493 42 q 539 70 539 61 l 539 429 q 534 496 539 470 q 518 537 529 522 q 491 557 507 552 q 453 563 475 563 q 403 550 430 563 q 347 514 376 538 q 288 455 318 491 q 230 372 258 419 l 230 70 q 254 49 230 60 q 323 29 279 38 l 323 0 l 37 0 l 37 29 q 105 50 80 40 q 130 70 130 59 l 130 880 q 127 924 130 908 q 115 949 125 940 q 87 961 106 958 q 37 969 68 965 l 37 996 q 87 1007 64 1002 q 130 1019 110 1013 q 168 1033 150 1026 q 205 1051 186 1040 l 230 1028 l 230 464 q 293 541 258 507 q 363 600 328 576 q 432 638 398 625 q 492 651 466 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 70 q 659 52 639 61 q 732 29 679 42 l 732 0 l 446 0 m 609 -155 q 604 -172 607 -162 q 597 -192 601 -182 q 589 -211 593 -201 q 583 -227 585 -220 l 174 -227 l 159 -210 q 163 -193 160 -202 q 171 -173 167 -183 q 179 -154 175 -164 q 186 -139 183 -145 l 594 -139 l 609 -155 "
    },
    "áº¿": {
      "x_min": 53,
      "x_max": 575,
      "ha": 628,
      "o": "m 329 580 q 266 566 294 580 q 215 528 237 552 q 177 469 192 503 q 156 394 162 435 l 440 394 q 461 399 456 394 q 466 418 466 404 q 461 464 466 436 q 441 517 456 491 q 399 561 426 543 q 329 580 372 580 m 575 378 q 543 353 563 364 q 501 333 523 341 l 153 333 q 166 232 153 280 q 205 147 180 184 q 267 89 231 111 q 349 68 303 68 q 389 70 369 68 q 431 82 408 73 q 481 110 454 92 q 546 160 509 129 q 561 146 555 155 q 571 132 567 137 q 492 52 526 83 q 429 6 459 22 q 371 -14 400 -9 q 308 -20 342 -20 q 211 2 257 -20 q 130 65 165 24 q 73 164 94 106 q 53 294 53 222 q 62 384 53 339 q 91 467 72 428 q 138 539 110 506 q 200 597 165 573 q 232 617 214 607 q 271 634 251 627 q 310 646 290 642 q 348 651 330 651 q 425 638 391 651 q 484 606 458 626 q 526 557 509 585 q 554 500 543 530 q 570 438 565 469 q 575 378 575 406 m 548 740 q 535 721 540 727 q 519 710 531 715 l 338 891 l 158 710 q 149 715 152 712 q 142 721 145 718 q 136 729 139 724 q 128 740 133 734 l 303 997 l 374 997 l 548 740 m 310 1015 q 288 1022 298 1017 q 272 1034 279 1028 l 427 1314 q 448 1312 435 1314 q 475 1308 461 1310 q 503 1304 489 1306 q 523 1299 516 1301 l 539 1272 l 310 1015 "
    },
    " ": {
      "x_min": 0,
      "x_max": 0,
      "ha": 300
    },
    "á¸ˆ": {
      "x_min": 47,
      "x_max": 676.765625,
      "ha": 730,
      "o": "m 498 -155 q 486 -203 498 -180 q 450 -245 474 -226 q 386 -278 425 -264 q 293 -301 347 -292 l 278 -267 q 337 -252 313 -261 q 376 -232 361 -243 q 399 -209 392 -221 q 406 -187 406 -198 q 388 -154 406 -164 q 330 -141 371 -145 q 337 -122 331 -139 q 350 -78 341 -108 q 369 -19 358 -55 q 258 8 313 -19 q 151 90 199 36 q 75 221 104 143 q 47 397 47 298 q 79 593 47 505 q 166 744 111 682 q 294 840 221 806 q 449 874 367 874 q 575 854 521 874 q 664 806 629 835 q 662 793 668 803 q 645 770 656 783 q 623 744 635 757 q 604 723 612 731 l 581 727 q 500 779 547 759 q 393 800 454 800 q 344 791 371 800 q 290 764 317 783 q 237 716 262 746 q 192 645 212 687 q 161 548 173 603 q 150 422 150 492 q 175 264 150 332 q 241 151 201 196 q 331 83 281 106 q 427 61 380 61 q 521 86 463 61 q 652 173 579 111 q 659 167 655 172 q 666 158 662 163 q 672 149 669 153 q 676 143 674 145 q 595 65 634 96 q 520 14 556 33 q 448 -12 483 -4 q 422 -16 435 -14 l 404 -70 q 439 -81 422 -74 q 469 -97 456 -87 q 490 -122 482 -107 q 498 -155 498 -136 m 320 926 q 304 937 309 931 q 290 957 299 944 l 547 1172 q 565 1161 554 1168 q 589 1148 577 1155 q 610 1134 601 1141 q 626 1122 620 1127 l 632 1091 l 320 926 "
    },
    "âˆ‘": {
      "x_min": 39.046875,
      "x_max": 667,
      "ha": 707,
      "o": "m 667 234 q 667 173 667 206 q 661 107 667 140 l 654 46 q 654 0 654 18 l 59 0 l 39 29 l 329 430 l 53 825 l 53 856 l 501 856 q 570 856 538 856 q 644 865 602 856 l 644 661 l 608 653 q 584 730 596 703 q 560 770 572 757 q 538 784 549 782 q 518 787 528 787 l 206 787 l 433 457 l 170 95 l 536 95 q 571 101 557 95 q 594 124 584 107 q 614 168 604 140 q 635 242 623 197 l 667 234 "
    },
    "áºƒ": {
      "x_min": 13,
      "x_max": 947,
      "ha": 961,
      "o": "m 947 601 q 914 592 926 597 q 895 583 902 588 q 885 573 888 579 q 880 559 882 567 l 733 40 q 718 14 729 25 q 692 -2 706 4 q 664 -13 678 -9 q 642 -20 651 -17 l 484 439 l 348 40 q 332 14 343 24 q 307 -3 321 3 q 281 -14 294 -10 q 259 -20 268 -17 l 85 559 q 13 601 79 586 l 13 631 l 270 631 l 270 601 q 218 594 237 598 q 192 583 200 589 q 184 572 184 578 q 186 559 184 565 l 311 129 l 475 631 l 521 631 l 693 129 l 808 559 q 794 584 812 575 q 731 601 776 594 l 731 631 l 947 631 l 947 601 m 469 710 q 448 717 457 712 q 432 729 439 723 l 587 1010 q 608 1007 595 1009 q 635 1003 621 1006 q 663 999 649 1001 q 683 994 676 996 l 699 968 l 469 710 "
    },
    "+": {
      "x_min": 40.953125,
      "x_max": 517.046875,
      "ha": 558,
      "o": "m 317 160 q 287 148 303 153 q 257 139 272 142 l 241 154 l 241 344 l 57 344 l 40 360 q 49 389 44 373 q 61 418 55 405 l 241 418 l 241 604 q 270 614 253 608 q 301 624 286 620 l 317 608 l 317 418 l 501 418 l 517 402 q 508 372 513 388 q 497 344 502 355 l 317 344 l 317 160 "
    },
    "á¸‹": {
      "x_min": 53,
      "x_max": 698.015625,
      "ha": 707,
      "o": "m 698 57 q 643 21 667 36 q 601 -2 619 7 q 570 -15 583 -11 q 548 -20 557 -20 q 512 10 526 -20 q 498 114 498 40 q 446 58 471 83 q 394 16 421 33 q 339 -10 368 -1 q 276 -20 310 -20 q 199 2 239 -20 q 127 65 160 24 q 73 166 94 106 q 53 301 53 226 q 63 381 53 340 q 92 461 73 422 q 140 534 112 500 q 202 595 167 569 q 278 636 237 621 q 366 651 319 651 q 428 643 397 651 q 495 608 459 636 l 495 865 q 493 925 495 903 q 481 959 491 947 q 453 975 472 971 q 398 980 433 980 l 398 1007 q 496 1027 453 1015 q 573 1051 539 1040 l 595 1031 l 595 172 q 595 131 595 148 q 597 103 595 114 q 601 84 599 91 q 608 72 604 76 q 632 67 614 64 q 689 86 649 70 l 698 57 m 495 177 l 495 495 q 433 555 473 534 q 345 577 393 577 q 267 561 303 577 q 207 513 232 545 q 167 433 181 481 q 153 322 153 386 q 169 217 153 264 q 210 137 185 170 q 266 85 236 103 q 323 68 296 68 q 368 77 346 68 q 411 102 390 87 q 453 137 432 117 q 495 177 474 156 m 289 855 q 283 819 289 836 q 266 790 277 802 q 242 770 256 777 q 211 763 228 763 q 168 779 181 763 q 156 827 156 796 q 161 863 156 846 q 178 893 167 880 q 202 912 189 905 q 232 920 216 920 q 289 855 289 920 "
    },
    "á¹’": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 m 640 1058 q 636 1041 639 1051 q 628 1021 632 1031 q 621 1002 624 1012 q 614 987 617 993 l 205 987 l 191 1003 q 195 1020 192 1011 q 203 1040 199 1030 q 211 1059 207 1049 q 218 1075 215 1068 l 626 1075 l 640 1058 m 321 1144 q 305 1155 310 1148 q 291 1174 300 1161 l 548 1389 q 566 1379 555 1385 q 590 1365 578 1372 q 612 1351 602 1358 q 627 1339 622 1344 l 633 1308 l 321 1144 "
    },
    "Ë—": {
      "x_min": 40.546875,
      "x_max": 418.03125,
      "ha": 459,
      "o": "m 418 370 q 408 338 414 355 q 397 309 402 320 l 57 309 l 40 325 q 49 356 43 340 q 61 387 55 373 l 401 387 l 418 370 "
    },
    "Ã‹": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 413 432 427 427 q 377 438 398 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 533 1044 q 527 1008 533 1025 q 510 979 521 991 q 486 959 500 966 q 455 952 472 952 q 412 968 425 952 q 400 1016 400 985 q 405 1052 400 1035 q 422 1082 411 1069 q 446 1101 433 1094 q 476 1109 460 1109 q 533 1044 533 1109 m 267 1044 q 261 1008 267 1025 q 244 979 255 991 q 220 959 234 966 q 189 952 206 952 q 146 968 159 952 q 134 1016 134 985 q 139 1052 134 1035 q 156 1082 145 1069 q 180 1101 167 1094 q 210 1109 194 1109 q 267 1044 267 1109 "
    },
    "Å ": {
      "x_min": 80,
      "x_max": 589,
      "ha": 649,
      "o": "m 589 255 q 581 193 589 225 q 556 130 573 161 q 516 72 540 99 q 458 24 491 45 q 384 -7 426 4 q 293 -20 342 -20 q 245 -15 271 -20 q 191 -2 219 -10 q 137 18 164 6 q 87 46 110 30 q 81 69 82 48 q 80 116 80 89 q 80 172 80 144 q 87 219 80 201 l 114 216 q 152 147 129 176 q 203 98 175 117 q 263 70 231 79 q 330 61 296 61 q 390 73 360 61 q 444 107 420 86 q 483 158 468 129 q 499 219 499 187 q 481 290 499 261 q 434 343 463 320 q 367 385 405 366 q 291 421 330 403 q 215 460 252 440 q 149 507 178 480 q 102 570 120 535 q 85 655 85 606 q 90 701 85 676 q 110 750 96 725 q 144 797 123 774 q 195 837 165 820 q 263 864 224 854 q 350 875 301 875 q 410 870 380 875 q 466 857 440 865 q 514 837 493 849 q 547 812 535 826 q 547 796 552 808 q 534 770 543 784 q 514 743 525 756 q 495 722 503 729 l 471 726 q 432 763 453 748 q 391 789 412 779 q 349 802 370 798 q 310 807 329 807 q 247 795 273 807 q 203 766 220 783 q 178 727 186 749 q 171 687 171 706 q 188 631 171 655 q 236 585 206 606 q 303 546 265 564 q 379 508 340 528 q 456 465 419 488 q 523 412 494 442 q 571 344 553 382 q 589 255 589 305 m 368 944 l 297 944 l 122 1152 q 129 1163 126 1158 q 136 1171 133 1168 q 142 1177 139 1175 q 151 1184 146 1180 l 334 1040 l 514 1184 q 530 1171 525 1177 q 543 1152 535 1165 l 368 944 "
    },
    "Æ™": {
      "x_min": 37,
      "x_max": 681.265625,
      "ha": 681,
      "o": "m 681 12 q 643 3 661 7 q 607 -2 624 0 q 578 -5 591 -4 q 558 -7 565 -7 q 514 1 531 -7 q 482 29 497 9 l 230 353 l 230 70 q 232 61 230 65 q 241 52 234 57 q 263 42 249 48 q 303 29 278 37 l 303 0 l 37 0 l 37 29 q 105 49 80 41 q 130 70 130 58 l 130 607 q 141 747 130 688 q 173 850 153 805 q 223 926 194 894 q 289 989 253 959 q 331 1016 309 1004 q 376 1035 354 1027 q 417 1047 397 1043 q 450 1051 437 1051 q 505 1039 477 1051 q 555 1013 533 1028 q 591 984 577 998 q 605 963 605 969 q 595 947 605 957 q 572 924 585 936 q 544 902 559 913 q 520 888 530 892 q 454 944 489 923 q 377 966 418 966 q 330 950 356 966 q 282 896 305 934 q 244 795 259 858 q 230 634 230 731 l 230 359 l 434 541 q 461 572 456 561 q 459 590 465 584 q 438 598 453 596 q 406 601 423 601 l 406 630 l 658 630 l 658 601 q 604 589 628 597 q 551 559 579 582 l 324 374 l 575 75 q 594 56 584 64 q 616 45 604 49 q 643 41 628 41 q 677 41 658 41 l 681 12 "
    },
    "á¹½": {
      "x_min": 13,
      "x_max": 657,
      "ha": 671,
      "o": "m 657 600 q 624 592 637 596 q 603 582 611 587 q 591 571 595 577 q 583 554 587 564 l 395 39 q 378 13 389 24 q 354 -3 367 3 q 327 -14 340 -10 q 306 -21 314 -18 l 86 554 q 63 583 80 573 q 13 600 46 593 l 13 630 l 270 630 l 270 600 q 227 594 244 597 q 202 585 211 590 q 192 573 193 580 q 194 554 190 565 l 357 120 l 511 554 q 513 572 514 564 q 504 584 512 579 q 481 593 496 589 q 442 600 466 596 l 442 630 l 657 630 l 657 600 m 582 912 q 553 857 570 884 q 517 808 537 830 q 472 773 496 787 q 423 760 448 760 q 378 771 399 760 q 338 798 357 783 q 298 824 318 812 q 259 836 279 836 q 211 814 233 836 q 164 754 189 792 l 129 768 q 157 823 141 795 q 194 872 174 850 q 238 907 215 893 q 288 921 262 921 q 336 909 313 921 q 378 883 358 897 q 416 856 398 868 q 450 845 434 845 q 499 866 475 845 q 546 927 523 888 l 582 912 "
    },
    "á»Ÿ": {
      "x_min": 53,
      "x_max": 731,
      "ha": 731,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 731 707 q 723 670 731 690 q 699 626 716 649 q 652 577 681 602 q 578 527 622 551 q 617 436 604 486 q 631 329 631 387 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 465 628 415 651 q 550 565 514 606 q 601 612 585 589 q 617 652 617 635 q 606 689 617 671 q 578 722 596 707 l 696 776 q 721 744 711 761 q 731 707 731 726 m 467 904 q 454 871 467 885 q 426 842 442 856 q 393 817 409 829 q 371 793 378 805 q 370 767 363 780 q 404 736 376 753 q 384 728 394 730 q 363 726 373 727 q 319 753 333 740 q 302 776 305 765 q 305 797 299 787 q 323 815 311 806 q 348 833 334 824 q 374 851 362 842 q 394 871 386 861 q 403 894 403 882 q 390 933 403 920 q 358 946 377 946 q 340 942 348 946 q 325 932 331 938 q 316 918 319 926 q 313 904 313 911 q 314 896 313 900 q 318 889 316 893 q 304 885 313 887 q 286 881 296 883 q 268 877 277 879 q 252 874 258 875 l 245 882 l 245 891 q 257 923 245 907 q 289 951 270 938 q 333 971 309 963 q 380 979 356 979 q 444 958 421 979 q 467 904 467 938 "
    },
    "Ã°": {
      "x_min": 53,
      "x_max": 621,
      "ha": 677,
      "o": "m 521 397 q 485 469 508 436 q 434 526 462 502 q 375 563 406 550 q 315 577 343 577 q 246 557 276 577 q 196 503 216 537 q 166 422 176 469 q 156 321 156 375 q 173 219 156 267 q 217 133 190 170 q 275 75 243 97 q 335 54 307 54 q 405 73 372 54 q 464 130 439 91 q 505 230 490 169 q 521 374 521 290 q 521 386 521 380 q 521 397 521 392 m 621 401 q 606 261 621 323 q 567 152 591 199 q 513 73 543 106 q 449 19 482 40 q 385 -10 417 -1 q 329 -20 354 -20 q 213 4 264 -20 q 126 71 161 28 q 71 173 90 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 140 544 113 511 q 198 600 167 576 q 264 638 229 624 q 335 651 299 651 q 376 644 355 651 q 419 625 398 638 q 460 597 440 613 q 497 562 480 581 q 445 686 478 631 q 365 790 412 741 l 215 724 q 195 726 202 725 q 183 727 189 726 q 172 730 178 728 q 154 736 165 732 l 148 761 l 317 836 q 279 866 299 852 q 238 888 260 879 q 192 901 216 897 q 140 900 167 904 l 136 928 l 269 977 q 400 873 344 926 l 538 935 q 562 930 554 932 q 576 928 571 929 q 587 924 582 926 q 601 920 592 922 l 605 896 l 446 826 q 525 723 492 775 q 580 616 559 670 q 611 509 601 563 q 621 401 621 454 "
    },
    "á» ": {
      "x_min": 47,
      "x_max": 811,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 811 944 q 803 906 811 927 q 778 861 796 885 q 730 812 760 837 q 654 761 700 786 q 731 614 705 698 q 757 438 757 529 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 532 853 482 875 q 622 796 582 832 q 679 846 661 821 q 697 889 697 871 q 687 926 697 908 q 658 959 677 943 l 776 1014 q 801 981 791 999 q 811 944 811 963 m 636 1103 q 607 1048 624 1075 q 570 999 590 1020 q 526 963 550 977 q 476 950 502 950 q 432 961 453 950 q 391 988 411 973 q 351 1015 371 1003 q 312 1027 332 1027 q 264 1004 286 1027 q 217 944 242 982 l 182 958 q 210 1013 194 986 q 247 1062 227 1041 q 291 1098 268 1084 q 341 1112 315 1112 q 389 1100 366 1112 q 431 1073 411 1088 q 469 1046 451 1058 q 503 1035 487 1035 q 553 1056 528 1035 q 600 1118 577 1078 l 636 1103 "
    },
    "á¸œ": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 443 -155 q 431 -203 443 -180 q 395 -245 419 -226 q 331 -279 370 -264 q 238 -301 293 -293 l 224 -268 q 282 -253 258 -262 q 321 -233 306 -243 q 344 -210 337 -222 q 351 -187 351 -198 q 333 -155 351 -164 q 275 -142 316 -146 q 282 -123 276 -140 q 296 -78 286 -108 q 320 0 305 -50 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 412 432 427 427 q 377 438 397 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 q 620 63 628 107 q 606 0 611 19 l 373 0 l 349 -70 q 384 -81 367 -75 q 414 -98 401 -87 q 435 -122 427 -108 q 443 -155 443 -136 m 549 1139 q 504 1046 529 1084 q 452 985 479 1009 q 394 952 424 962 q 335 942 364 942 q 272 952 302 942 q 213 985 241 962 q 162 1046 186 1009 q 117 1139 137 1084 q 133 1157 125 1151 q 153 1169 141 1164 q 191 1104 169 1131 q 236 1062 212 1078 q 285 1039 260 1046 q 332 1032 310 1032 q 381 1039 355 1032 q 430 1062 406 1046 q 476 1104 455 1078 q 514 1169 498 1131 q 533 1157 525 1164 q 549 1139 541 1151 "
    },
    "Ä±": {
      "x_min": 47,
      "x_max": 333,
      "ha": 370,
      "o": "m 47 0 l 47 29 q 115 49 92 38 q 139 70 139 61 l 139 454 q 139 510 139 488 q 130 543 139 531 q 101 560 121 555 q 47 569 82 566 l 47 596 q 90 606 67 600 q 136 619 113 612 q 179 634 158 626 q 217 651 200 642 l 240 651 l 240 70 q 262 50 240 62 q 333 29 284 38 l 333 0 l 47 0 "
    },
    "Æš": {
      "x_min": 46.71875,
      "x_max": 388.265625,
      "ha": 435,
      "o": "m 373 502 l 268 502 l 268 70 q 290 49 268 60 q 367 29 312 38 l 367 0 l 68 0 l 68 29 q 116 39 96 34 q 146 49 135 44 q 163 59 158 54 q 168 70 168 65 l 168 502 l 61 502 l 46 516 q 53 539 50 527 q 61 563 56 552 l 168 563 l 168 879 q 164 926 168 909 q 149 951 160 943 q 120 963 138 960 q 74 969 102 966 l 74 997 q 162 1018 122 1006 q 244 1051 202 1029 l 268 1028 l 268 563 l 373 563 l 388 546 l 373 502 "
    },
    "Ã¤": {
      "x_min": 53,
      "x_max": 615.03125,
      "ha": 624,
      "o": "m 233 68 q 318 88 273 68 q 414 154 363 109 l 414 329 q 321 311 358 320 q 262 293 285 302 q 225 273 239 284 q 200 252 211 263 q 172 210 183 234 q 162 155 162 187 q 169 111 162 128 q 189 84 177 94 q 212 71 200 74 q 233 68 225 68 m 615 55 q 521 -2 558 15 q 466 -20 484 -20 q 429 10 444 -20 q 415 97 415 42 q 353 44 383 66 q 295 7 323 21 q 241 -13 266 -6 q 194 -20 215 -20 q 145 -12 170 -20 q 99 13 120 -4 q 66 58 79 31 q 53 126 53 86 q 70 212 53 177 q 112 271 88 246 q 148 302 128 288 q 202 330 169 317 q 286 356 236 344 q 414 380 337 368 l 414 475 q 408 518 414 498 q 390 552 403 538 q 356 575 378 567 q 302 583 334 583 q 260 576 280 583 q 224 557 239 569 q 200 527 209 544 q 193 490 191 511 q 180 476 194 484 q 146 463 165 469 q 107 453 126 456 q 80 452 88 450 l 71 478 q 118 543 86 512 q 189 598 149 575 q 274 636 229 622 q 361 651 319 651 q 474 611 434 651 q 514 503 514 572 l 514 120 q 521 80 514 92 q 540 68 528 68 q 564 71 549 68 q 605 86 578 74 l 615 55 m 518 854 q 512 818 518 835 q 495 789 506 801 q 471 769 485 776 q 440 762 457 762 q 397 778 410 762 q 385 826 385 795 q 390 862 385 845 q 407 892 396 879 q 431 911 418 904 q 461 919 445 919 q 518 854 518 919 m 252 854 q 246 818 252 835 q 229 789 240 801 q 205 769 219 776 q 174 762 191 762 q 131 778 144 762 q 119 826 119 795 q 124 862 119 845 q 141 892 130 879 q 165 911 152 904 q 195 919 179 919 q 252 854 252 919 "
    },
    "Ç®": {
      "x_min": 62,
      "x_max": 648,
      "ha": 708,
      "o": "m 648 294 q 621 164 648 222 q 549 65 594 106 q 445 2 504 24 q 323 -20 386 -20 q 218 -4 266 -20 q 135 30 170 10 q 81 71 100 51 q 62 103 62 92 q 71 119 62 108 q 94 143 80 131 q 124 166 108 156 q 153 179 140 177 q 189 132 167 154 q 237 95 211 111 q 292 70 264 79 q 349 61 321 61 q 427 72 391 61 q 490 108 464 84 q 532 170 517 132 q 548 259 548 208 q 534 338 548 300 q 496 405 521 376 q 434 451 470 434 q 353 469 399 469 q 297 460 323 469 q 238 435 270 451 l 206 418 l 204 420 q 193 432 200 424 q 183 447 186 441 l 183 448 l 183 449 l 183 449 l 457 787 l 203 787 q 180 777 193 787 q 156 751 168 768 q 134 709 144 734 q 117 653 124 684 l 79 661 l 99 865 q 125 858 113 861 q 150 855 138 855 q 179 855 163 855 l 594 855 l 611 825 l 379 533 q 401 536 391 535 q 423 538 412 538 q 506 522 466 538 q 578 475 547 506 q 629 399 610 445 q 648 294 648 354 m 390 944 l 319 944 l 143 1186 q 151 1197 148 1192 q 158 1205 155 1202 q 164 1211 161 1209 q 173 1217 168 1214 l 356 1047 l 536 1217 q 552 1205 547 1211 q 564 1186 556 1199 l 390 944 "
    },
    "Â¹": {
      "x_min": 64.375,
      "x_max": 391,
      "ha": 448,
      "o": "m 79 420 l 79 448 q 138 456 115 452 q 174 466 161 461 q 193 475 188 470 q 198 485 198 480 l 198 795 q 196 820 198 811 q 189 835 195 829 q 179 838 186 837 q 159 840 172 840 q 125 837 146 840 q 74 829 104 835 l 64 856 q 111 870 83 861 q 168 889 139 879 q 222 908 197 898 q 263 926 248 918 l 281 911 l 281 485 q 285 476 281 480 q 300 466 289 471 q 334 456 312 461 q 391 448 355 452 l 391 420 l 79 420 "
    },
    "W": {
      "x_min": 13,
      "x_max": 1131,
      "ha": 1157,
      "o": "m 1131 824 q 1082 813 1101 818 q 1051 802 1063 807 q 1035 792 1040 797 q 1029 782 1030 788 l 877 39 q 863 14 874 25 q 838 -2 852 4 q 809 -14 824 -9 q 785 -20 795 -18 l 568 639 l 377 39 q 361 15 373 25 q 336 -2 350 4 q 306 -13 321 -9 q 277 -20 290 -18 l 104 777 q 80 805 101 794 q 13 824 60 816 l 13 854 l 298 854 l 298 824 q 246 816 265 821 q 218 805 228 811 q 207 791 209 799 q 207 777 206 784 l 339 169 l 555 854 l 591 854 l 827 169 l 951 782 q 945 797 952 790 q 924 807 938 803 q 890 816 910 812 q 849 824 871 820 l 849 854 l 1131 854 l 1131 824 "
    },
    "á»‰": {
      "x_min": 47,
      "x_max": 333,
      "ha": 370,
      "o": "m 47 0 l 47 29 q 115 49 92 38 q 139 70 139 61 l 139 454 q 139 510 139 488 q 130 543 139 531 q 101 560 121 555 q 47 569 82 566 l 47 596 q 90 606 67 600 q 136 619 113 612 q 179 634 158 626 q 217 651 199 642 l 240 651 l 240 70 q 261 50 240 62 q 333 29 283 38 l 333 0 l 47 0 m 301 904 q 288 870 301 885 q 260 842 276 855 q 228 817 243 829 q 205 793 212 805 q 204 766 197 780 q 238 736 210 753 q 218 728 229 729 q 197 725 207 726 q 153 753 167 740 q 136 776 139 765 q 139 796 133 787 q 157 815 145 806 q 182 833 168 824 q 208 851 196 842 q 228 871 220 861 q 237 894 237 882 q 224 933 237 920 q 192 946 211 946 q 174 942 182 946 q 159 932 165 938 q 150 918 153 926 q 147 903 147 911 q 148 896 147 900 q 152 889 150 893 q 138 885 147 887 q 120 880 130 882 q 102 877 111 878 q 86 874 92 875 l 79 882 l 79 891 q 91 923 79 907 q 123 951 104 938 q 167 971 143 963 q 214 979 190 979 q 278 958 255 979 q 301 904 301 938 "
    },
    "É²": {
      "x_min": -159,
      "x_max": 725,
      "ha": 745,
      "o": "m 439 0 l 439 29 q 509 51 487 42 q 532 70 532 61 l 532 429 q 527 496 532 470 q 513 537 523 522 q 487 557 503 552 q 446 563 470 563 q 401 552 427 563 q 346 520 375 542 q 286 461 317 497 q 223 372 254 425 l 223 59 q 212 -73 223 -19 q 183 -165 201 -127 q 140 -228 165 -203 q 89 -275 116 -253 q 51 -302 72 -291 q 9 -322 30 -314 q -29 -334 -11 -330 q -60 -339 -48 -339 q -91 -334 -74 -339 q -123 -322 -108 -329 q -148 -307 -138 -315 q -159 -292 -159 -298 q -149 -275 -159 -286 q -126 -253 -139 -264 q -98 -231 -113 -241 q -74 -216 -84 -221 q -49 -234 -62 -228 q -26 -243 -36 -240 q -5 -247 -15 -247 q 12 -247 4 -247 q 51 -236 32 -247 q 87 -195 71 -225 q 113 -112 103 -165 q 123 24 123 -59 l 123 482 q 121 524 123 508 q 110 548 119 539 q 82 561 101 557 q 29 569 63 565 l 29 596 q 114 618 75 604 q 189 651 153 632 l 215 627 l 215 458 q 283 539 247 504 q 354 599 318 575 q 425 637 391 624 q 485 651 459 651 q 540 642 513 651 q 587 615 566 633 q 619 568 607 596 q 632 502 632 540 l 632 70 q 652 52 632 61 q 725 29 672 42 l 725 0 l 439 0 "
    },
    ">": {
      "x_min": 40.546875,
      "x_max": 556.9375,
      "ha": 597,
      "o": "m 556 423 q 547 390 553 410 q 536 355 542 370 l 57 173 l 40 190 q 45 206 42 197 q 50 224 47 214 q 55 242 53 233 q 61 257 58 250 l 427 396 l 51 539 l 40 565 q 70 588 55 577 q 96 609 84 599 l 541 440 l 556 423 "
    },
    "á»†": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 568 855 l 590 838 q 587 799 590 820 q 580 757 584 778 q 571 717 576 736 q 563 685 567 698 l 533 685 q 528 737 533 716 q 517 771 524 758 q 501 788 511 783 q 479 794 491 794 l 236 794 l 236 499 l 505 499 l 522 480 q 509 459 517 470 q 493 438 501 448 q 476 418 485 427 q 461 404 468 410 q 439 421 451 414 q 413 433 427 428 q 377 439 398 437 q 329 442 357 442 l 236 442 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 400 -189 q 394 -225 400 -208 q 377 -254 388 -242 q 353 -274 367 -267 q 322 -282 339 -282 q 279 -265 292 -282 q 267 -217 267 -248 q 272 -181 267 -198 q 289 -151 278 -164 q 313 -132 300 -139 q 343 -125 327 -125 q 400 -189 400 -125 m 543 958 q 531 939 535 945 q 515 927 526 933 l 332 1068 l 153 927 q 144 932 147 930 q 137 939 140 935 q 131 947 134 942 q 123 958 128 951 l 298 1167 l 369 1167 l 543 958 "
    },
    "á¸‚": {
      "x_min": 27.015625,
      "x_max": 675,
      "ha": 748,
      "o": "m 273 817 q 254 817 264 817 q 235 817 244 817 l 235 491 l 257 491 q 380 506 333 491 q 453 545 427 521 q 488 598 479 568 q 497 657 497 628 q 486 720 497 691 q 450 771 476 749 q 381 804 424 792 q 273 817 338 817 m 344 440 q 285 436 312 440 q 235 429 258 433 l 235 70 q 241 60 235 64 q 267 53 252 55 q 296 49 281 50 q 327 48 312 48 q 356 48 342 48 q 442 60 404 48 q 509 95 481 73 q 551 149 536 117 q 566 218 566 180 q 553 294 566 255 q 513 366 540 333 q 444 419 486 398 q 344 440 402 440 m 675 243 q 652 140 675 186 q 590 62 630 94 q 493 12 549 29 q 370 -5 438 -5 q 335 -5 356 -5 q 292 -5 315 -5 q 244 -5 269 -5 q 196 -5 219 -5 q 81 -5 141 -5 l 33 -5 l 33 28 q 103 49 78 38 q 129 70 129 60 l 129 806 q 79 799 103 803 q 33 791 55 795 l 27 834 q 87 848 52 841 q 159 861 121 855 q 236 871 197 867 q 311 875 275 875 q 431 862 377 875 q 522 826 485 849 q 580 768 560 802 q 601 691 601 734 q 567 565 601 618 q 474 489 533 512 q 553 459 517 481 q 617 404 590 437 q 659 330 644 372 q 675 243 675 289 m 417 1044 q 411 1008 417 1025 q 394 979 405 991 q 370 959 384 966 q 339 952 356 952 q 296 968 309 952 q 284 1016 284 985 q 289 1052 284 1035 q 306 1082 295 1069 q 330 1101 317 1094 q 360 1109 344 1109 q 417 1044 417 1109 "
    },
    "Å´": {
      "x_min": 13,
      "x_max": 1131,
      "ha": 1157,
      "o": "m 1131 824 q 1082 813 1101 818 q 1051 802 1063 807 q 1035 792 1040 797 q 1029 782 1030 788 l 877 39 q 863 14 874 25 q 838 -2 852 4 q 809 -14 824 -9 q 785 -20 795 -18 l 568 639 l 377 39 q 361 15 373 25 q 336 -2 350 4 q 306 -13 321 -9 q 277 -20 290 -18 l 104 777 q 80 805 101 794 q 13 824 60 816 l 13 854 l 298 854 l 298 824 q 246 816 265 821 q 218 805 228 811 q 207 791 209 799 q 207 777 206 784 l 339 169 l 555 854 l 591 854 l 827 169 l 951 782 q 945 797 952 790 q 924 807 938 803 q 890 816 910 812 q 849 824 871 820 l 849 854 l 1131 854 l 1131 824 m 781 957 q 769 938 773 944 q 753 926 764 932 l 571 1067 l 391 926 q 382 932 385 929 q 375 938 378 934 q 369 946 372 941 q 361 957 365 951 l 536 1167 l 607 1167 l 781 957 "
    },
    "Ã": {
      "x_min": 17.703125,
      "x_max": 746,
      "ha": 806,
      "o": "m 746 456 q 729 305 746 372 q 683 188 712 239 q 617 101 654 137 q 538 43 579 66 q 453 10 496 20 q 371 0 410 0 l 33 0 l 33 29 q 103 49 78 38 q 129 70 129 61 l 129 417 l 32 417 l 17 433 q 24 455 21 444 q 32 478 28 467 l 129 478 l 129 805 q 78 798 102 801 q 33 791 54 795 l 27 833 q 94 850 56 842 q 175 864 132 858 q 261 871 217 871 q 343 871 304 871 q 510 842 436 871 q 637 761 585 814 q 717 631 689 708 q 746 456 746 554 m 300 816 q 235 816 266 816 l 235 478 l 407 478 l 424 462 l 407 417 l 235 417 l 235 104 q 241 80 235 89 q 277 62 250 68 q 356 57 304 57 q 450 79 402 57 q 540 148 499 102 q 607 265 581 195 q 633 431 633 335 q 611 592 633 521 q 546 713 589 663 q 442 789 504 763 q 300 816 380 816 "
    },
    "r": {
      "x_min": 37,
      "x_max": 517.765625,
      "ha": 540,
      "o": "m 512 626 q 517 602 517 621 q 512 556 517 582 q 499 503 507 530 q 483 458 491 476 l 455 458 q 443 504 451 485 q 426 534 436 523 q 405 550 417 545 q 381 556 394 556 q 347 543 366 556 q 307 504 327 530 q 266 435 286 477 q 230 336 245 393 l 230 70 q 254 49 230 60 q 342 29 279 38 l 342 0 l 37 0 l 37 29 q 105 49 80 39 q 130 70 130 58 l 130 465 q 128 502 130 487 q 125 527 127 518 q 120 542 123 537 q 114 551 117 547 q 104 560 110 556 q 89 565 98 563 q 68 568 80 568 q 37 568 55 568 l 37 597 q 121 620 80 608 q 196 651 163 632 l 224 627 l 224 472 q 267 543 246 510 q 312 599 288 575 q 362 637 336 623 q 417 651 388 651 q 463 645 438 651 q 512 626 487 640 "
    },
    "Ã˜": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 640 544 653 487 q 602 650 627 601 l 251 137 q 321 81 282 101 q 402 61 360 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 150 434 q 163 313 150 371 q 203 208 177 255 l 554 721 q 485 776 524 756 q 402 797 447 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 m 667 745 q 734 603 711 683 q 757 438 757 523 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 277 -1 324 -20 q 190 50 229 17 l 155 -1 q 138 -11 149 -6 q 114 -21 127 -16 q 89 -29 102 -25 q 69 -35 77 -32 l 50 -13 l 136 109 q 69 249 92 170 q 47 415 47 328 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 526 856 479 875 q 612 805 574 837 l 648 857 q 689 877 667 868 q 733 890 712 885 l 751 868 l 667 745 "
    },
    "Ç": {
      "x_min": -20.78125,
      "x_max": 400.125,
      "ha": 370,
      "o": "m 47 0 l 47 29 q 115 49 92 38 q 139 70 139 61 l 139 454 q 139 510 139 488 q 130 543 139 531 q 101 560 121 555 q 47 569 82 566 l 47 596 q 90 606 67 600 q 136 619 113 612 q 179 634 158 626 q 217 651 200 642 l 240 651 l 240 70 q 262 50 240 62 q 333 29 284 38 l 333 0 l 47 0 m 226 726 l 154 726 l -20 968 q -12 979 -16 974 q -6 987 -9 984 q 0 993 -3 991 q 9 999 3 996 l 192 829 l 371 999 q 387 987 382 993 q 400 968 392 981 l 226 726 "
    },
    "á»²": {
      "x_min": -0.265625,
      "x_max": 781,
      "ha": 808,
      "o": "m 239 0 l 239 29 q 323 55 299 42 q 348 78 348 68 l 348 364 q 292 478 323 419 q 228 594 260 538 q 165 699 195 651 q 111 780 134 748 q 97 794 105 788 q 78 806 90 801 q 48 814 66 811 q 2 818 30 818 l 0 846 q 78 856 38 852 q 145 861 117 861 q 197 834 175 861 q 250 757 221 802 q 307 663 278 713 q 363 562 335 614 q 413 461 390 510 l 592 780 q 585 808 602 798 q 518 825 569 818 l 518 854 l 781 854 l 781 825 q 711 807 735 816 q 677 780 686 798 l 454 366 l 454 78 q 459 68 454 73 q 478 55 465 62 q 512 42 491 48 q 565 29 533 35 l 565 0 l 239 0 m 518 957 q 504 937 509 944 q 488 926 499 931 l 177 1091 l 182 1122 q 197 1134 187 1127 q 219 1148 207 1141 q 242 1161 231 1155 q 261 1172 254 1168 l 518 957 "
    },
    "áº¼": {
      "x_min": 33,
      "x_max": 633.859375,
      "ha": 675,
      "o": "m 633 165 q 620 63 628 107 q 606 0 611 19 l 33 0 l 33 29 q 104 49 78 38 q 130 70 130 61 l 130 782 q 105 803 130 790 q 33 824 81 815 l 33 854 l 568 854 l 590 837 q 587 798 590 819 q 580 756 584 777 q 571 716 576 735 q 563 684 567 697 l 533 684 q 528 736 533 715 q 517 770 524 757 q 501 787 511 782 q 479 793 491 793 l 236 793 l 236 498 l 505 498 l 522 479 q 509 458 517 469 q 493 437 501 447 q 476 417 485 426 q 461 403 468 409 q 439 420 451 413 q 413 432 428 427 q 377 438 398 436 q 329 441 357 441 l 236 441 l 236 104 q 240 86 236 94 q 259 72 245 78 q 301 64 274 67 q 371 61 327 61 l 457 61 q 509 64 487 61 q 547 79 530 67 q 577 114 564 91 q 605 177 591 137 l 633 165 m 560 1103 q 531 1048 548 1075 q 494 999 515 1020 q 450 963 474 977 q 400 950 426 950 q 356 961 377 950 q 315 988 335 973 q 276 1015 295 1003 q 237 1027 256 1027 q 188 1004 210 1027 q 142 944 167 982 l 107 958 q 135 1013 119 986 q 172 1062 151 1041 q 216 1098 192 1084 q 265 1112 240 1112 q 313 1100 291 1112 q 356 1073 336 1088 q 394 1046 376 1058 q 428 1035 412 1035 q 477 1056 453 1035 q 524 1118 501 1078 l 560 1103 "
    },
    "Ã·": {
      "x_min": 40.578125,
      "x_max": 517.421875,
      "ha": 559,
      "o": "m 332 230 q 327 202 332 215 q 314 178 322 188 q 294 162 306 168 q 270 157 283 157 q 236 170 247 157 q 226 209 226 184 q 230 237 226 223 q 243 260 235 250 q 263 276 252 270 q 287 282 274 282 q 332 230 332 282 m 332 556 q 327 528 332 541 q 314 504 322 514 q 294 488 306 494 q 270 483 283 483 q 236 496 247 483 q 226 535 226 510 q 230 563 226 549 q 243 586 235 576 q 263 602 252 596 q 287 608 274 608 q 332 556 332 608 m 517 402 q 508 372 514 388 q 497 344 502 355 l 57 344 l 40 360 q 49 389 43 373 q 61 418 55 405 l 501 418 l 517 402 "
    },
    "h": {
      "x_min": 37,
      "x_max": 732,
      "ha": 752,
      "o": "m 446 0 l 446 29 q 516 51 493 42 q 539 70 539 61 l 539 429 q 534 496 539 470 q 518 537 529 522 q 491 557 507 552 q 453 563 475 563 q 403 550 430 563 q 347 514 376 538 q 288 455 318 491 q 230 372 258 419 l 230 70 q 254 49 230 60 q 323 29 279 38 l 323 0 l 37 0 l 37 29 q 105 50 80 40 q 130 70 130 59 l 130 880 q 127 924 130 908 q 115 949 125 940 q 87 961 106 958 q 37 969 68 965 l 37 996 q 87 1007 64 1002 q 130 1019 110 1013 q 168 1033 150 1026 q 205 1051 186 1040 l 230 1028 l 230 464 q 293 541 258 507 q 363 600 328 576 q 432 638 398 625 q 492 651 466 651 q 547 642 520 651 q 594 615 573 633 q 626 568 614 596 q 639 502 639 540 l 639 70 q 659 52 639 61 q 732 29 679 42 l 732 0 l 446 0 "
    },
    "á¹ƒ": {
      "x_min": 37,
      "x_max": 1075,
      "ha": 1094,
      "o": "m 789 0 l 789 29 q 858 51 836 42 q 881 70 881 61 l 881 429 q 876 498 881 470 q 863 541 872 525 q 840 563 854 557 q 805 570 825 570 q 759 557 783 570 q 708 521 734 544 q 656 464 682 498 q 605 388 630 430 l 605 70 q 625 52 605 61 q 699 29 646 43 l 699 0 l 413 0 l 413 29 q 483 51 461 42 q 505 70 505 61 l 505 429 q 501 498 505 470 q 488 541 497 525 q 464 563 479 557 q 430 570 450 570 q 335 522 385 570 q 230 388 286 475 l 230 70 q 254 49 230 60 q 323 29 279 39 l 323 0 l 37 0 l 37 29 q 105 50 80 41 q 130 70 130 59 l 130 482 q 127 525 130 509 q 115 549 125 540 q 87 561 106 557 q 37 570 68 565 l 37 597 q 83 606 62 601 q 124 619 104 612 q 161 634 143 626 q 196 651 178 642 l 223 627 l 223 472 q 287 550 257 517 q 348 606 318 584 q 406 639 378 628 q 457 651 434 651 q 516 643 489 651 q 563 616 543 635 q 594 567 582 598 q 605 492 605 537 l 605 477 q 663 552 633 520 q 722 606 693 584 q 780 639 752 628 q 833 651 808 651 q 892 642 865 651 q 938 615 919 633 q 969 568 958 596 q 981 502 981 540 l 981 70 q 1001 52 981 61 q 1075 29 1022 43 l 1075 0 l 789 0 m 622 -189 q 616 -225 622 -208 q 599 -254 610 -242 q 575 -274 589 -267 q 544 -282 561 -282 q 501 -265 514 -282 q 489 -217 489 -248 q 494 -181 489 -198 q 511 -151 500 -164 q 535 -132 522 -139 q 565 -125 549 -125 q 622 -189 622 -125 "
    },
    "f": {
      "x_min": 30.03125,
      "x_max": 543,
      "ha": 423,
      "o": "m 543 985 q 533 969 543 980 q 510 947 523 958 q 483 925 497 935 q 458 910 468 915 q 426 935 442 924 q 395 952 410 945 q 367 962 380 959 q 345 966 354 966 q 309 954 328 966 q 272 911 289 942 q 244 826 255 879 q 233 688 233 772 l 233 630 l 404 630 l 423 610 q 408 590 417 601 q 390 569 399 579 q 373 552 381 560 q 358 542 364 544 q 314 558 343 550 q 233 566 286 566 l 233 68 q 239 60 233 64 q 263 52 246 56 q 306 42 280 47 q 372 28 332 36 l 372 0 l 40 0 l 40 28 q 109 48 85 37 q 133 68 133 60 l 133 566 l 43 566 l 30 584 l 81 630 l 133 630 l 133 651 q 143 785 133 731 q 172 878 154 840 q 214 941 190 915 q 265 988 238 966 q 304 1014 283 1003 q 346 1034 326 1026 q 386 1046 367 1042 q 418 1051 405 1051 q 460 1042 438 1051 q 500 1024 482 1034 q 531 1002 519 1013 q 543 985 543 991 "
    },
    "â€œ": {
      "x_min": 61,
      "x_max": 534.734375,
      "ha": 593,
      "o": "m 254 712 q 228 693 248 704 q 186 672 209 682 q 141 656 163 662 q 107 651 119 650 q 73 694 85 671 q 61 758 61 718 q 69 815 61 784 q 96 876 78 845 q 138 934 113 906 q 196 983 164 962 l 224 959 q 200 926 210 944 q 182 888 189 907 q 172 851 175 869 q 169 818 169 833 q 186 766 169 788 q 243 744 204 744 l 254 712 m 534 712 q 509 693 528 704 q 466 672 489 682 q 421 656 443 662 q 387 651 399 650 q 368 671 377 661 q 353 694 359 682 q 344 722 347 707 q 341 758 341 738 q 349 815 341 784 q 376 876 358 845 q 418 934 393 906 q 476 983 444 962 l 504 959 q 479 926 490 944 q 462 888 469 907 q 452 851 455 869 q 449 818 449 833 q 466 766 449 788 q 523 744 484 744 l 534 712 "
    },
    "Ç—": {
      "x_min": 33,
      "x_max": 847,
      "ha": 890,
      "o": "m 847 825 q 776 804 801 816 q 751 783 751 793 l 751 355 q 729 197 751 266 q 668 79 708 127 q 571 5 628 30 q 442 -20 514 -20 q 316 0 374 -20 q 218 58 259 18 q 153 158 176 98 q 130 301 130 218 l 130 783 q 105 804 130 791 q 33 825 81 816 l 33 855 l 332 855 l 332 825 q 261 804 286 816 q 236 783 236 793 l 236 335 q 250 218 236 270 q 295 131 265 167 q 368 76 325 95 q 468 58 411 58 q 552 81 516 58 q 614 142 589 104 q 652 229 639 180 q 665 327 665 277 l 665 783 q 640 804 665 791 q 568 825 615 816 l 568 855 l 847 855 l 847 825 m 649 1045 q 643 1009 649 1026 q 626 980 637 992 q 602 960 616 967 q 571 953 588 953 q 528 969 541 953 q 516 1017 516 986 q 521 1053 516 1036 q 538 1083 527 1070 q 562 1102 549 1095 q 592 1110 576 1110 q 649 1045 649 1110 m 383 1045 q 377 1009 383 1026 q 360 980 371 992 q 336 960 350 967 q 305 953 322 953 q 262 969 275 953 q 250 1017 250 986 q 255 1053 250 1036 q 272 1083 261 1070 q 296 1102 283 1095 q 326 1110 310 1110 q 383 1045 383 1110 m 361 1144 q 345 1155 350 1149 q 332 1175 340 1162 l 589 1390 q 607 1379 595 1386 q 630 1365 618 1373 q 652 1352 642 1358 q 668 1339 662 1345 l 674 1309 l 361 1144 "
    },
    "Ì‡": {
      "x_min": -409,
      "x_max": -276,
      "ha": 0,
      "o": "m -276 854 q -281 818 -276 835 q -298 789 -287 801 q -322 769 -308 776 q -353 762 -336 762 q -396 778 -383 762 q -409 826 -409 795 q -403 862 -409 845 q -386 892 -397 879 q -362 911 -375 904 q -332 919 -348 919 q -276 854 -276 919 "
    },
    "A": {
      "x_min": 0,
      "x_max": 796,
      "ha": 810,
      "o": "m 504 363 l 386 711 l 272 363 l 504 363 m 252 302 l 179 75 q 197 45 172 54 q 276 29 221 35 l 276 0 l 0 0 l 0 29 q 71 46 45 37 q 104 75 98 55 l 352 839 q 390 869 367 855 q 430 893 412 883 l 709 75 q 718 58 712 66 q 734 45 724 51 q 759 36 744 40 q 796 29 774 32 l 796 0 l 516 0 l 516 31 q 587 41 567 31 q 600 75 607 52 l 524 302 l 252 302 "
    },
    "Æ": {
      "x_min": 20,
      "x_max": 828,
      "ha": 901,
      "o": "m 601 772 q 566 794 585 785 q 523 809 547 803 q 466 815 498 815 q 389 815 433 815 l 389 492 l 411 492 q 534 507 487 492 q 607 546 581 522 q 642 600 633 570 q 651 659 651 630 q 638 723 651 693 q 601 772 625 752 m 498 440 q 439 437 466 440 q 389 430 412 434 l 389 70 q 394 60 389 64 q 420 53 406 55 q 450 49 435 50 q 481 48 466 48 q 509 48 496 48 q 596 60 557 48 q 662 95 634 73 q 704 149 689 117 q 719 218 719 180 q 706 295 719 255 q 666 366 693 334 q 597 419 639 398 q 498 440 556 440 m 20 679 q 45 757 20 722 q 119 818 71 793 q 236 857 167 843 q 391 871 305 871 q 493 871 448 871 q 574 863 538 871 q 637 845 609 856 q 689 817 665 834 q 737 764 719 797 q 755 693 755 731 q 720 566 755 619 q 628 490 686 513 q 707 460 670 482 q 770 405 743 438 q 812 331 797 372 q 828 243 828 290 q 805 140 828 186 q 743 62 783 94 q 647 12 703 29 q 524 -5 591 -5 q 489 -5 509 -5 q 446 -5 469 -5 q 398 -5 422 -5 q 350 -5 373 -5 q 234 -5 295 -5 l 186 -5 l 186 28 q 257 49 231 38 q 283 70 283 60 l 283 812 q 220 799 248 808 q 171 777 191 790 q 139 744 150 763 q 129 702 129 726 q 141 659 129 678 q 174 630 154 640 q 162 619 172 626 q 137 604 152 611 q 106 590 122 596 q 76 580 90 583 q 35 616 51 591 q 20 679 20 642 "
    },
    "á¹¨": {
      "x_min": 80,
      "x_max": 589,
      "ha": 649,
      "o": "m 589 255 q 581 193 589 225 q 556 130 573 161 q 516 72 540 99 q 458 24 491 45 q 384 -7 426 4 q 293 -20 342 -20 q 245 -15 271 -20 q 191 -2 219 -10 q 137 18 164 6 q 87 46 110 30 q 81 69 82 48 q 80 116 80 89 q 80 172 80 144 q 87 219 80 201 l 114 216 q 152 147 129 176 q 203 98 175 117 q 263 70 231 79 q 330 61 296 61 q 390 73 360 61 q 444 107 420 86 q 483 158 468 129 q 499 219 499 187 q 481 290 499 261 q 434 343 463 320 q 367 385 405 366 q 291 421 330 403 q 215 460 252 440 q 149 507 178 480 q 102 570 120 535 q 85 655 85 606 q 90 701 85 676 q 110 750 96 725 q 144 797 123 774 q 195 837 165 820 q 263 864 224 854 q 350 875 301 875 q 410 870 380 875 q 466 857 440 865 q 514 837 493 849 q 547 812 535 826 q 547 796 552 808 q 534 770 543 784 q 514 743 525 756 q 495 722 503 729 l 471 726 q 432 763 453 748 q 391 789 412 779 q 349 802 370 798 q 310 807 329 807 q 247 795 273 807 q 203 766 220 783 q 178 727 186 749 q 171 687 171 706 q 188 631 171 655 q 236 585 206 606 q 302 546 265 564 q 379 508 340 528 q 456 465 419 488 q 523 412 494 442 q 571 344 553 382 q 589 255 589 305 m 399 -189 q 393 -225 399 -208 q 376 -254 387 -242 q 352 -274 366 -267 q 321 -282 338 -282 q 278 -265 291 -282 q 266 -217 266 -248 q 271 -181 266 -198 q 288 -151 277 -164 q 312 -132 299 -139 q 342 -125 326 -125 q 399 -189 399 -125 m 399 1044 q 393 1008 399 1025 q 376 979 387 991 q 352 959 366 966 q 321 952 338 952 q 278 968 291 952 q 266 1016 266 985 q 271 1052 266 1035 q 288 1082 277 1069 q 312 1101 299 1094 q 342 1109 326 1109 q 399 1044 399 1109 "
    },
    "O": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 645 519 653 473 q 622 606 637 565 q 584 682 606 647 q 534 742 562 716 q 473 782 506 767 q 402 797 439 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 q 170 290 150 358 q 224 171 190 223 q 304 90 259 120 q 402 61 350 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 757 439 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 240 15 302 -20 q 135 112 177 51 q 69 251 92 172 q 47 415 47 329 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 566 838 503 875 q 671 740 628 801 q 735 600 713 679 q 757 439 757 521 "
    },
    "Ä": {
      "x_min": 17.703125,
      "x_max": 746,
      "ha": 806,
      "o": "m 746 456 q 729 305 746 372 q 683 188 712 239 q 617 101 654 137 q 538 43 579 66 q 453 10 496 20 q 371 0 410 0 l 33 0 l 33 29 q 103 49 78 38 q 129 70 129 61 l 129 417 l 32 417 l 17 433 q 24 455 21 444 q 32 478 28 467 l 129 478 l 129 805 q 78 798 102 801 q 33 791 54 795 l 27 833 q 94 850 56 842 q 175 864 132 858 q 261 871 217 871 q 343 871 304 871 q 510 842 436 871 q 637 761 585 814 q 717 631 689 708 q 746 456 746 554 m 300 816 q 235 816 266 816 l 235 478 l 407 478 l 424 462 l 407 417 l 235 417 l 235 104 q 241 80 235 89 q 277 62 250 68 q 356 57 304 57 q 450 79 402 57 q 540 148 499 102 q 607 265 581 195 q 633 431 633 335 q 611 592 633 521 q 546 713 589 663 q 442 789 504 763 q 300 816 380 816 "
    },
    "3": {
      "x_min": 45.109375,
      "x_max": 548,
      "ha": 638,
      "o": "m 548 258 q 530 150 548 201 q 476 62 512 100 q 389 2 441 24 q 269 -20 338 -20 q 216 -15 243 -20 q 161 1 189 -10 q 103 32 132 13 q 45 81 75 52 l 67 126 q 122 88 97 103 q 169 64 147 73 q 213 51 192 55 q 261 48 235 48 q 334 61 301 48 q 392 98 368 74 q 429 158 416 123 q 442 236 442 193 q 426 322 442 287 q 386 380 410 358 q 331 413 362 403 q 272 423 301 423 l 258 423 q 250 422 254 423 q 242 420 247 421 q 227 418 237 420 l 217 458 q 315 498 279 475 q 371 544 352 520 q 397 592 391 568 q 404 637 404 616 q 397 685 404 660 q 375 730 390 710 q 337 764 360 751 q 283 777 314 777 q 235 767 256 777 q 200 741 214 757 q 181 704 186 725 q 179 659 175 683 q 136 640 156 647 q 86 634 115 634 l 69 654 q 89 715 69 682 q 144 774 110 747 q 224 819 179 801 q 320 838 270 838 q 406 821 370 838 q 464 778 441 805 q 499 718 488 751 q 510 650 510 684 q 500 599 510 624 q 471 551 490 574 q 427 508 453 527 q 368 473 401 488 q 439 450 406 469 q 496 402 472 431 q 534 337 520 374 q 548 258 548 300 "
    },
    "Ç¾": {
      "x_min": 47,
      "x_max": 757,
      "ha": 817,
      "o": "m 653 426 q 640 544 653 487 q 602 650 627 601 l 251 137 q 321 81 282 101 q 402 61 360 61 q 503 84 457 61 q 583 153 549 107 q 635 268 616 200 q 653 426 653 337 m 150 434 q 163 313 150 371 q 203 208 177 255 l 554 721 q 485 776 524 756 q 402 797 447 797 q 295 771 342 797 q 216 698 249 746 q 167 584 184 651 q 150 434 150 517 m 667 745 q 734 603 711 683 q 757 438 757 523 q 727 263 757 346 q 645 117 697 180 q 526 17 594 54 q 382 -20 458 -20 q 277 -1 324 -20 q 190 50 229 17 l 155 -1 q 138 -11 149 -6 q 114 -21 127 -16 q 89 -29 102 -25 q 69 -35 77 -32 l 50 -13 l 136 109 q 69 249 92 170 q 47 415 47 328 q 76 590 47 507 q 156 737 105 674 q 274 837 206 800 q 421 875 342 875 q 526 856 479 875 q 612 805 574 837 l 648 857 q 689 877 667 868 q 733 890 712 885 l 751 868 l 667 745 m 314 927 q 297 938 302 931 q 284 957 292 944 l 541 1172 q 559 1162 548 1168 q 582 1148 570 1155 q 604 1134 594 1141 q 620 1122 614 1127 l 626 1092 l 314 927 "
    },
    "4": {
      "x_min": 36.3125,
      "x_max": 580.640625,
      "ha": 638,
      "o": "m 389 704 l 141 311 l 389 311 l 389 704 m 580 291 q 555 259 566 269 q 528 237 543 248 l 482 237 l 482 70 q 485 60 482 64 q 500 50 489 55 q 528 40 511 45 q 574 28 546 34 l 574 0 l 249 0 l 249 28 q 319 42 292 36 q 362 54 347 49 q 383 66 377 60 q 389 77 389 71 l 389 237 l 58 237 l 36 259 l 366 795 q 416 820 394 807 q 457 843 438 833 l 482 820 l 482 311 l 563 311 l 580 291 "
    },
    "ÆŽ": {
      "x_min": 43,
      "x_max": 634,
      "ha": 675,
      "o": "m 64 0 l 43 16 q 47 62 43 33 q 56 120 51 90 q 67 175 61 149 q 76 214 72 202 l 107 214 q 110 155 107 183 q 119 106 113 127 q 133 73 124 85 q 153 61 141 61 l 297 61 q 366 63 339 61 q 407 72 392 66 q 426 87 421 77 q 431 110 431 96 l 431 443 l 291 443 q 257 443 275 443 q 221 440 239 443 q 188 436 204 438 q 161 430 172 433 l 144 449 q 157 470 149 459 q 173 491 164 480 q 189 510 181 501 q 205 524 198 519 q 227 509 215 514 q 253 503 238 505 q 288 501 268 501 l 431 501 l 431 793 l 269 793 q 217 789 238 793 q 178 774 195 786 q 148 739 162 762 q 120 676 134 715 l 94 688 q 104 790 94 746 q 119 854 113 834 l 634 854 l 634 824 q 562 804 588 815 q 537 783 537 793 l 537 71 q 561 50 537 63 q 634 29 585 38 l 634 0 l 64 0 "
    },
    "áº€": {
      "x_min": 13,
      "x_max": 1131,
      "ha": 1157,
      "o": "m 1131 824 q 1082 813 1101 818 q 1051 802 1063 807 q 1035 792 1040 797 q 1029 782 1030 788 l 877 39 q 863 14 874 25 q 838 -2 852 4 q 809 -14 824 -9 q 785 -20 795 -18 l 568 639 l 377 39 q 361 15 373 25 q 336 -2 350 4 q 306 -13 321 -9 q 277 -20 290 -18 l 104 777 q 80 805 101 794 q 13 824 60 816 l 13 854 l 298 854 l 298 824 q 246 816 265 821 q 218 805 228 811 q 207 791 209 799 q 207 777 206 784 l 339 169 l 555 854 l 591 854 l 827 169 l 951 782 q 945 797 952 790 q 924 807 938 803 q 890 816 910 812 q 849 824 871 820 l 849 854 l 1131 854 l 1131 824 m 689 957 q 675 937 680 944 q 659 926 670 931 l 347 1091 l 353 1121 q 368 1134 357 1127 q 390 1147 378 1140 q 413 1161 402 1155 q 432 1172 424 1167 l 689 957 "
    },
    "Å¤": {
      "x_min": 6.609375,
      "x_max": 719,
      "ha": 750,
      "o": "m 204 0 l 204 28 q 255 41 234 34 q 289 54 276 48 q 307 67 301 61 q 313 77 313 72 l 313 793 l 114 793 q 96 789 104 793 q 79 773 87 785 q 60 737 71 761 q 35 674 49 713 l 6 687 q 15 772 9 727 q 28 854 21 818 l 699 854 l 719 837 q 713 761 719 802 q 699 677 707 720 l 669 677 q 658 730 663 708 q 646 766 653 752 q 630 786 639 780 q 607 793 621 793 l 419 793 l 419 77 q 442 55 419 68 q 528 28 466 41 l 528 0 l 204 0 m 398 943 l 327 943 l 152 1151 q 160 1162 156 1157 q 166 1170 163 1167 q 173 1176 169 1174 q 182 1183 176 1179 l 364 1039 l 543 1183 q 559 1170 555 1176 q 572 1151 564 1164 l 398 943 "
    },
    "Æ¡": {
      "x_min": 53,
      "x_max": 731,
      "ha": 731,
      "o": "m 528 309 q 510 411 528 362 q 466 496 493 459 q 404 555 438 533 q 336 577 369 577 q 251 557 285 577 q 195 503 216 537 q 165 422 174 469 q 156 321 156 375 q 174 219 156 267 q 221 133 193 170 q 283 75 249 97 q 347 54 317 54 q 429 72 395 54 q 484 123 462 90 q 517 203 506 157 q 528 309 528 250 m 731 707 q 723 670 731 690 q 699 626 716 649 q 652 577 681 602 q 578 527 622 551 q 617 436 604 486 q 631 329 631 387 q 619 240 631 283 q 588 158 608 197 q 540 86 568 119 q 479 30 512 53 q 407 -6 445 6 q 329 -20 369 -20 q 215 4 266 -20 q 128 71 164 28 q 72 173 92 114 q 53 301 53 232 q 63 389 53 346 q 94 472 74 433 q 141 544 114 511 q 202 600 169 576 q 275 638 236 624 q 355 651 314 651 q 465 628 415 651 q 550 565 514 606 q 601 612 585 589 q 617 652 617 635 q 606 689 617 671 q 578 722 596 707 l 696 777 q 721 744 711 762 q 731 707 731 726 "
    },
    "êž‰": {
      "x_min": 66,
      "x_max": 224,
      "ha": 284,
      "o": "m 224 493 q 217 449 224 469 q 197 414 210 429 q 168 391 185 399 q 132 383 151 383 q 81 402 96 383 q 66 458 66 422 q 73 501 66 481 q 93 537 80 521 q 122 561 105 552 q 158 570 139 570 q 207 550 191 570 q 224 493 224 530 m 224 164 q 217 120 224 140 q 197 85 210 100 q 168 62 185 70 q 132 54 151 54 q 81 73 96 54 q 66 129 66 93 q 73 172 66 152 q 93 208 80 192 q 122 232 105 223 q 158 241 139 241 q 207 221 191 241 q 224 164 224 201 "
    }
  },
  "cssFontWeight": "normal",
  "ascender": 1214,
  "underlinePosition": -250,
  "cssFontStyle": "normal",
  "boundingBox": {
    "yMin": -492,
    "xMin": -683.21875,
    "yMax": 1471.359375,
    "xMax": 1329
  },
  "resolution": 1000,
  "original_font_information": {
    "postscript_name": "Gentilis-Regular",
    "version_string": "Version 1.100",
    "vendor_url": "http://scripts.sil.org/",
    "full_font_name": "Gentilis",
    "font_family_name": "Gentilis",
    "copyright": "Copyright (c) SIL International, 2003-2008.",
    "description": "",
    "trademark": "Gentium is a trademark of SIL International.",
    "designer": "J. Victor Gaultney and Annie Olsen",
    "designer_url": "http://www.sil.org/~gaultney",
    "unique_font_identifier": "SIL International:Gentilis:2-3-108",
    "license_url": "http://scripts.sil.org/OFL",
    "license_description": "Copyright (c) 2003-2008, SIL International (http://www.sil.org/) with Reserved Font Names \"Gentium\" and \"SIL\".\r\n\r\nThis Font Software is licensed under the SIL Open Font License, Version 1.1. This license is copied below, and is also available with a FAQ at: http://scripts.sil.org/OFL\r\n\r\n\r\n-----------------------------------------------------------\r\nSIL OPEN FONT LICENSE Version 1.1 - 26 February 2007\r\n-----------------------------------------------------------\r\n\r\nPREAMBLE\r\nThe goals of the Open Font License (OFL) are to stimulate worldwide development of collaborative font projects, to support the font creation efforts of academic and linguistic communities, and to provide a free and open framework in which fonts may be shared and improved in partnership with others.\r\n\r\nThe OFL allows the licensed fonts to be used, studied, modified and redistributed freely as long as they are not sold by themselves. The fonts, including any derivative works, can be bundled, embedded, redistributed and/or sold with any software provided that any reserved names are not used by derivative works. The fonts and derivatives, however, cannot be released under any other type of license. The requirement for fonts to remain under this license does not apply to any document created using the fonts or their derivatives.\r\n\r\nDEFINITIONS\r\n\"Font Software\" refers to the set of files released by the Copyright Holder(s) under this license and clearly marked as such. This may include source files, build scripts and documentation.\r\n\r\n\"Reserved Font Name\" refers to any names specified as such after the copyright statement(s).\r\n\r\n\"Original Version\" refers to the collection of Font Software components as distributed by the Copyright Holder(s).\r\n\r\n\"Modified Version\" refers to any derivative made by adding to, deleting, or substituting -- in part or in whole -- any of the components of the Original Version, by changing formats or by porting the Font Software to a new environment.\r\n\r\n\"Author\" refers to any designer, engineer, programmer, technical writer or other person who contributed to the Font Software.\r\n\r\nPERMISSION & CONDITIONS\r\nPermission is hereby granted, free of charge, to any person obtaining a copy of the Font Software, to use, study, copy, merge, embed, modify, redistribute, and sell modified and unmodified copies of the Font Software, subject to the following conditions:\r\n\r\n1) Neither the Font Software nor any of its individual components, in Original or Modified Versions, may be sold by itself.\r\n\r\n2) Original or Modified Versions of the Font Software may be bundled, redistributed and/or sold with any software, provided that each copy contains the above copyright notice and this license. These can be included either as stand-alone text files, human-readable headers or in the appropriate machine-readable metadata fields within text or binary files as long as those fields can be easily viewed by the user.\r\n\r\n3) No Modified Version of the Font Software may use the Reserved Font Name(s) unless explicit written permission is granted by the corresponding Copyright Holder. This restriction only applies to the primary font name as presented to the users.\r\n\r\n4) The name(s) of the Copyright Holder(s) or the Author(s) of the Font Software shall not be used to promote, endorse or advertise any Modified Version, except to acknowledge the contribution(s) of the Copyright Holder(s) and the Author(s) or with their explicit written permission.\r\n\r\n5) The Font Software, modified or unmodified, in part or in whole, must be distributed entirely under this license, and must not be distributed under any other license. The requirement for fonts to remain under this license does not apply to any document created using the Font Software.\r\n\r\nTERMINATION\r\nThis license becomes null and void if any of the above conditions are not met.\r\n\r\nDISCLAIMER\r\nTHE FONT SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO ANY WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF COPYRIGHT, PATENT, TRADEMARK, OR OTHER RIGHT. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, INCLUDING ANY GENERAL, SPECIAL, INDIRECT, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF THE USE OR INABILITY TO USE THE FONT SOFTWARE OR FROM OTHER DEALINGS IN THE FONT SOFTWARE.",
    "manufacturer_name": "SIL International",
    "font_sub_family_name": "Regular"
  },
  "descender": -394,
  "familyName": "Gentilis",
  "lineHeight": 1607,
  "underlineThickness": 100
});

function str_repeat(b, a) {
  for (var c = []; a > 0; c[--a] = b) {}
  return (c.join(""))
}
function sprintf() {
  var g = 0,
    e, h = arguments[g++],
    k = [],
    d, j, l, b;
  while (h) {
    if (d = /^[^\x25]+/.exec(h)) {
      k.push(d[0])
    } else {
      if (d = /^\x25{2}/.exec(h)) {
        k.push("%")
      } else {
        if (d = /^\x25(?:(\d+)\$)?(\+)?(0|'[^$])?(-)?(\d+)?(?:\.(\d+))?([b-fosuxX])/.exec(h)) {
          if (((e = arguments[d[1] || g++]) == null) || (e == undefined)) {
            throw ("Too few arguments.")
          }
          if (/[^s]/.test(d[7]) && (typeof (e) != "number")) {
            throw ("Expecting number but found " + typeof (e))
          }
          switch (d[7]) {
          case "b":
            e = e.toString(2);
            break;
          case "c":
            e = String.fromCharCode(e);
            break;
          case "d":
            e = parseInt(e);
            break;
          case "e":
            e = d[6] ? e.toExponential(d[6]) : e.toExponential();
            break;
          case "f":
            e = d[6] ? parseFloat(e).toFixed(d[6]) : parseFloat(e);
            break;
          case "o":
            e = e.toString(8);
            break;
          case "s":
            e = ((e = String(e)) && d[6] ? e.substring(0, d[6]) : e);
            break;
          case "u":
            e = Math.abs(e);
            break;
          case "x":
            e = e.toString(16);
            break;
          case "X":
            e = e.toString(16).toUpperCase();
            break
          }
          e = (/[def]/.test(d[7]) && d[2] && e > 0 ? "+" + e : e);
          l = d[3] ? d[3] == "0" ? "0" : d[3].charAt(1) : " ";
          b = d[5] - String(e).length;
          j = d[5] ? str_repeat(l, b) : "";
          k.push(d[4] ? e + j : j + e)
        } else {
          throw ("Huh ?!")
        }
      }
    }
    h = h.substring(d[0].length)
  }
  return k.join("")
};
var dateFormat = function () {
    var a = /d{1,4}|m{1,4}|yy(?:yy)?|([HhMsTt])\1?|[LloSZ]|"[^"]*"|'[^']*'/g,
      b = /\b(?:[PMCEA][SDP]T|(?:Pacific|Mountain|Central|Eastern|Atlantic) (?:Standard|Daylight|Prevailing) Time|(?:GMT|UTC)(?:[-+]\d{4})?)\b/g,
      d = /[^-+\dA-Z]/g,
      c = function (f, e) {
        f = String(f);
        e = e || 2;
        while (f.length < e) {
          f = "0" + f
        }
        return f
      };
    return function (i, v, q) {
      var g = dateFormat;
      if (arguments.length == 1 && Object.prototype.toString.call(i) == "[object String]" && !/\d/.test(i)) {
        v = i;
        i = undefined
      }
      i = i ? new Date(i) : new Date;
      if (isNaN(i)) {
        throw SyntaxError("invalid date")
      }
      v = String(g.masks[v] || v || g.masks["default"]);
      if (v.slice(0, 4) == "UTC:") {
        v = v.slice(4);
        q = true
      }
      var t = q ? "getUTC" : "get",
        l = i[t + "Date"](),
        e = i[t + "Day"](),
        j = i[t + "Month"](),
        p = i[t + "FullYear"](),
        r = i[t + "Hours"](),
        k = i[t + "Minutes"](),
        u = i[t + "Seconds"](),
        n = i[t + "Milliseconds"](),
        f = q ? 0 : i.getTimezoneOffset(),
        h = {
          d: l,
          dd: c(l),
          ddd: g.i18n.dayNames[e],
          dddd: g.i18n.dayNames[e + 7],
          m: j + 1,
          mm: c(j + 1),
          mmm: g.i18n.monthNames[j],
          mmmm: g.i18n.monthNames[j + 12],
          yy: String(p).slice(2),
          yyyy: p,
          h: r % 12 || 12,
          hh: c(r % 12 || 12),
          H: r,
          HH: c(r),
          M: k,
          MM: c(k),
          s: u,
          ss: c(u),
          l: c(n, 3),
          L: c(n > 99 ? Math.round(n / 10) : n),
          t: r < 12 ? "a" : "p",
          tt: r < 12 ? "am" : "pm",
          T: r < 12 ? "A" : "P",
          TT: r < 12 ? "AM" : "PM",
          Z: q ? "UTC" : (String(i).match(b) || [""]).pop().replace(d, ""),
          o: (f > 0 ? "-" : "+") + c(Math.floor(Math.abs(f) / 60) * 100 + Math.abs(f) % 60, 4),
          S: ["th", "st", "nd", "rd"][l % 10 > 3 ? 0 : (l % 100 - l % 10 != 10) * l % 10]
        };
      return v.replace(a, function (m) {
        return m in h ? h[m] : m.slice(1, m.length - 1)
      })
    }
  }();
dateFormat.masks = {
  "default": "ddd mmm dd yyyy HH:MM:ss",
  shortDate: "m/d/yy",
  mediumDate: "mmm d, yyyy",
  longDate: "mmmm d, yyyy",
  fullDate: "dddd, mmmm d, yyyy",
  shortTime: "h:MM TT",
  mediumTime: "h:MM:ss TT",
  longTime: "h:MM:ss TT Z",
  isoDate: "yyyy-mm-dd",
  isoTime: "HH:MM:ss",
  isoDateTime: "yyyy-mm-dd'T'HH:MM:ss",
  isoUtcDateTime: "UTC:yyyy-mm-dd'T'HH:MM:ss'Z'"
};
dateFormat.i18n = {
  dayNames: ["Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"],
  monthNames: ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec", "January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"]
};
Date.prototype.format = function (a, b) {
  return dateFormat(this, a, b)
};
if (typeof (CanvasXpress) == "undefined") {
  CanvasXpress = {}
}
var CanvasXpress = function (f, d, a, b, c, e) {
    if (!f) {
      f = this.createNewTarget()
    } else {
      if (typeof (f) == "object") {
        d = f.data || false;
        a = f.config || false;
        b = f.events || false;
        c = f.hidden || false;
        e = f.info || false;
        f = f.renderTo || this.createNewTarget()
      }
    }
    this.validateParameters = function () {
      this.validateData();
      this.validateConfig();
      this.validateEvents();
      this.validateInfo()
    };
    this.validateData = function () {
      if (d && !this.subBrowser) {
        try {
          JSON.stringify(d)
        } catch (g) {
          alert("Data object malformed:\n" + g)
        }
      }
    };
    this.validateConfig = function () {
      if (!a) {
        a = {}
      } else {
        if (!this.subBrowser) {
          try {
            JSON.stringify(a)
          } catch (g) {
            alert("Config object malformed:\n" + g)
          }
        }
      }
      this.userConfig = a
    };
    this.validateEvents = function () {};
    this.validateInfo = function () {
      if (!e) {
        e = ""
      }
    };
    this.createNewTarget = function () {
      var k = document.createElement("canvas");
      var j = 0;
      var g = "canvasXpress" + j;
      var h = document.getElementById(g);
      while (h) {
        j++;
        g = "canvasXpress" + j;
        h = document.getElementById(g)
      }
      k.id = g;
      document.body.appendChild(k);
      return g
    };
    this.insertTarget = function (j, l, i, k, g) {
      if (j && l) {
        var m = document.getElementById(j);
        if (m) {
          return
        } else {
          m = document.createElement("canvas")
        }
        m.id = j;
        m.width = i;
        m.height = k;
        if (g) {
          l.parentNode.insertBefore(m, l.nextSibling)
        } else {
          l.parentNode.insertBefore(m, l)
        }
      }
    };
    this.removeTarget = function (g) {
      var h = document.getElementById(g);
      if (h) {
        h.parentNode.removeChild(h)
      }
    };
    this.setConfig = function (g) {
      if (a.graphType == "Network") {
        if (!a.backgroundType) {
          this.backgroundType = "gradient";
          this.nodeFontColor = this.background
        }
      }
      if (a) {
        for (var h = 0; h < g.length; h++) {
          var j = g[h];
          if (a.hasOwnProperty(j)) {
            this[j] = a[j]
          }
        }
      }
      if (this.isVML) {
        this.showAnimation = false
      }
    };
    this.setInitialConfig = function () {
      if (a) {
        for (var g in a) {
          this[g] = a[g]
        }
      }
    };
    this.resetConfig = function (g) {
      var j = {};
      for (var h = 0; h < g.length; h++) {
        j[g[h]] = this[g[h]]
      }
      this.initConfig(true);
      for (var h in j) {
        this[h] = j[h]
      }
    };
    this.updateConfig = function (h) {
      if (h) {
        this.userConfig = h;
        for (var g in h) {
          this[g] = h[g]
        }
      }
    };
    this.getConfig = function () {
      var k = {};
      for (var h = 0; h < this.config.length; h++) {
        var g = this.config[h];
        if (this[g] != undefined) {
          k[g] = this[g]
        }
      }
      return k
    };
    this.getUserConfig = function () {
      return this.userConfig
    };
    this.updateData = function (g) {
      this.initData(g);
      this.initEvents();
      this.initGraph()
    };
    this.save = function () {
      return {
        renderTo: this.target,
        data: this.data,
        config: this.getConfig(),
        events: this.events
      }
    };
    this.print = function (g) {
      return function (h) {
        alert("A new window will open so you can right click on the graph and save it");
        if (g.isIE) {
          var j = g.canvas.parentNode.childNodes[0];
          return window.open().document.write("<html><body>" + j.innerHTML + "</body></html>")
        } else {
          var i = g.canvas.toDataURL("image/png");
          return window.open().document.write('<html><body><img src="' + i + '" /></body></html>')
        }
      }
    }(this);
    this.prettyJSON = function (g, p) {
      var q = function (i) {
          if (typeof (i) == "object") {
            if (i === null) {
              return "null"
            }
            if (i.constructor == (new Array).constructor) {
              return "array"
            }
            if (i.constructor == (new Date).constructor) {
              return "date"
            }
            if (i.constructor == (new RegExp).constructor) {
              return "regex"
            }
            return "object"
          }
          return typeof (i)
        };
      if (!p) {
        p = ""
      }
      var r;
      var m = "  ";
      var h = q(g);
      var j = 0;
      if (h == "array") {
        if (g.length == 0) {
          return "[]"
        }
        r = "["
      } else {
        for (var n in g) {
          j++;
          break
        }
        if (j == 0) {
          return "{}"
        }
        r = "{"
      }
      j = 0;
      for (var l in g) {
        v = g[l];
        if (j > 0) {
          r += ","
        }
        if (h == "array") {
          r += ("\n" + p + m)
        } else {
          r += ("\n" + p + m + '"' + l + '": ')
        }
        switch (q(v)) {
        case "array":
        case "object":
          r += this.prettyJSON(v, (p + m));
          break;
        case "boolean":
        case "number":
          r += v.toString();
          break;
        case "null":
          r += "null";
          break;
        case "string":
          r += ('"' + v + '"');
          break;
        default:
          r += ("TYPEOF: " + typeof (v))
        }
        j++
      }
      if (h == "array") {
        r += ("\n" + p + "]")
      } else {
        r += ("\n" + p + "}")
      }
      return r
    };
    this.dumpToConsole = function (g) {
      console.log(this.target);
      console.log(this.prettyJSON(g))
    };
    this.stack = function () {
      if (this.debug) {
        for (var g in this) {
          if (arguments.callee.caller === this[g]) {
            CanvasXpress.stack[this.target].push(g);
            return
          }
        }
      }
    };
    this.setCanvas = function () {
      this.version = 4.2;
      this.target = f;
      this.events = b;
      this.info = e;
      var g = document.getElementById("wrapper-" + this.target);
      if (!g) {
        var i = document.createElement("div");
        i.id = "wrapper-" + this.target;
        i.style.position = "relative";
        i.style.cssFloat = "left";
        if (c) {
          i.style.display = "none"
        }
        var j = document.getElementById(this.target);
        var h = j.parentNode;
        h.insertBefore(i, j);
        i.appendChild(j.parentNode.appendChild(j))
      }
      this.canvas = this.browser == "Explorer" && (this.browserVersion < 9 && navigator.userAgent.toLowerCase().indexOf("chromeframe") < 0) ? window.G_vmlCanvasManager.initElement(j) : j;
      this.ctx = this.canvas.getContext("2d")
    };
    this.setPath = function () {
      var j = document.getElementsByTagName("script");
      if (j) {
        for (var h = 0; h < j.length; h++) {
          if (!j[h].src) {
            continue
          }
          var k = j[h].src;
          var g = k.lastIndexOf("/");
          var m = k.substring(0, g + 1);
          var l = k.substring(g + 1);
          if (l.match(/canvasXpress/)) {
            this.path = m
          }
        }
      }
    };
    this.initialize = function () {
      this.initBrowser();
      this.validateParameters();
      this.addCSS();
      this.setPath();
      this.setCanvas();
      this.initConfig();
      this.isGoogleChromeFrame();
      this.initData(d);
      this.initLayout();
      this.initEvents();
      this.initAnimation();
      this.initRemote();
      this.initGraph()
    };
    this.initialize();
    CanvasXpress.references.push(this);
    CanvasXpress.stack[this.target] = []
  };
CanvasXpress.references = [];
CanvasXpress.cacheImages = {};
CanvasXpress.stack = {};
CanvasXpress.prototype.addCSS = function () {
  var a = "table.shorts {border: 1px solid; border-collapse: collapse; font-size: x-small}";
  a += "td.k {text-align: center; padding: 2px 5px; margin: 2px 5px; border: 1px solid;}";
  a += "td.d {padding: 2px 5px; margin: 2px 5px; border: 1px solid;}";
  a += "th.d {padding: 2px 5px; margin: 2px 5px; border: 1px solid;}";
  a += "th.cxx, th.cxz, th.cxb, th.cxs, th.cxv, td.cxx, td.cxz, td.cxb, td.cxs, td.cxv {padding: 2px 5px; margin: 2px 5px; border: 1px solid; valign: center;}";
  a += "td.cxx, td.cxz, td.cxb, td.cxs, td.cxv {background: #fefefe}";
  a += "th.cxx, th.cxz, th.cxb, th.cxs, th.cxv {background: #d4d0c8}";
  a += "td.cxy {border: 1px solid; valign: center; text-align: right;}";
  a += "div.cxc {width: 100%; height: 100%; overflow: hidden;}";
  var b = document.createElement("style");
  b.type = "text/css";
  if (b.styleSheet) {
    b.styleSheet.cssText = a
  } else {
    b.appendChild(document.createTextNode(a))
  }
  document.getElementsByTagName("head")[0].appendChild(b)
};
CanvasXpress.prototype.initBrowser = function () {
  this.isGoogleChromeFrame = function () {
    if (this.browser == "Explorer" && this.checkGoogleChromeFrame) {
      if (navigator.userAgent.toLowerCase().indexOf("chromeframe") < 0) {
        try {
          var b = document.createElement("script");
          b.type = "text/javascript";
          b.src = "http://ajax.googleapis.com/ajax/libs/chrome-frame/1/CFInstall.min.js";
          document.getElementsByTagName("head")[0].appendChild(b);
          CFInstall.check({
            mode: "overlay"
          })
        } catch (a) {
          alert("CanvasXpress works much faster in fully HTML 5 compliant browsers.\nPlease consider installing Google chrome frame to make this browser compliant.")
        }
      }
    }
  };
  this.dataBrowser = [{
    string: navigator.userAgent,
    subString: "Chrome",
    identity: "Chrome"
  }, {
    string: navigator.userAgent,
    subString: "OmniWeb",
    versionSearch: "OmniWeb/",
    identity: "OmniWeb"
  }, {
    string: navigator.vendor,
    subString: "Apple",
    identity: "Safari",
    versionSearch: "Version"
  }, {
    prop: window.opera,
    identity: "Opera"
  }, {
    string: navigator.vendor,
    subString: "iCab",
    identity: "iCab"
  }, {
    string: navigator.vendor,
    subString: "KDE",
    identity: "Konqueror"
  }, {
    string: navigator.userAgent,
    subString: "Firefox",
    identity: "Firefox"
  }, {
    string: navigator.vendor,
    subString: "Camino",
    identity: "Camino"
  }, {
    string: navigator.userAgent,
    subString: "Netscape",
    identity: "Netscape"
  }, {
    string: navigator.userAgent,
    subString: "MSIE",
    identity: "Explorer",
    versionSearch: "MSIE"
  }, {
    string: navigator.userAgent,
    subString: "Gecko",
    identity: "Mozilla",
    versionSearch: "rv"
  }, {
    string: navigator.userAgent,
    subString: "Mozilla",
    identity: "Netscape",
    versionSearch: "Mozilla"
  }, {
    string: navigator.platform,
    subString: "iPhone",
    identity: "iPhone"
  }, {
    string: navigator.platform,
    subString: "iPod",
    identity: "iPod"
  }, {
    string: navigator.userAgent,
    subString: "iPad",
    identity: "iPad"
  }, {
    string: navigator.userAgent,
    subString: "Android",
    identity: "Android"
  }, {
    string: navigator.userAgent,
    subString: "BlackBerry",
    identity: "BlackBerry"
  }];
  this.dataOS = [{
    string: navigator.platform,
    subString: "Win",
    identity: "Windows"
  }, {
    string: navigator.platform,
    subString: "Mac",
    identity: "Mac"
  }, {
    string: navigator.platform,
    subString: "iPhone",
    identity: "iPhone"
  }, {
    string: navigator.platform,
    subString: "iPod",
    identity: "iPod"
  }, {
    string: navigator.userAgent,
    subString: "iPad",
    identity: "iPad"
  }, {
    string: navigator.userAgent,
    subString: "Android",
    identity: "Android"
  }, {
    string: navigator.userAgent,
    subString: "BlackBerry",
    identity: "BlackBerry"
  }, {
    string: navigator.platform,
    subString: "Linux",
    identity: "Linux"
  }];
  this.isSubstandardBrowser = function () {
    if ((this.browser == "Firefox" && this.browserVersion > 4) || (this.browser == "Opera" && this.browserVersion > 9) || (this.browser == "Safari" && this.browserVersion > 4) || (this.browser == "Chrome" && this.browserVersion > 1)) {
      return false
    } else {
      return true
    }
  };
  this.isMobileApp = function () {
    if (this.browser.match(/iPhone|iPod|iPad|Android|BlackBerry/i)) {
      return true
    } else {
      return false
    }
  };
  this.searchString = function (d) {
    for (var a = 0; a < d.length; a++) {
      var b = d[a].string;
      var c = d[a].prop;
      this.versionSearchString = d[a].versionSearch || d[a].identity;
      if (b) {
        if (b.indexOf(d[a].subString) != -1) {
          return d[a].identity
        }
      } else {
        if (c) {
          return d[a].identity
        }
      }
    }
  };
  this.searchVersion = function (b) {
    var a = b.indexOf(this.versionSearchString);
    if (a == -1) {
      return
    }
    return parseFloat(b.substring(a + this.versionSearchString.length + 1))
  };
  this.initializeBrowser = function () {
    this.browser = this.searchString(this.dataBrowser) || "An unknown browser";
    this.browserVersion = this.searchVersion(navigator.userAgent) || this.searchVersion(navigator.appVersion) || "an unknown version";
    this.browserOS = this.searchString(this.dataOS) || "an unknown OS";
    if (this.browser == "Explorer") {
      this.isIE = true;
      this.subBrowser = true;
      this.mobileApp = false;
      if (this.browserVersion >= 9) {
        this.isVML = false
      } else {
        if (navigator.userAgent.toLowerCase().indexOf("chromeframe") < 0) {
          this.isVML = true
        } else {
          this.isVML = false
        }
      }
    } else {
      this.isIE = false;
      this.subBrowser = this.isSubstandardBrowser();
      this.mobileApp = this.isMobileApp();
      this.isVML = false
    }
  };
  this.initializeBrowser()
};
CanvasXpress.prototype.initConfig = function (a) {
  this.config = [];
  this.graphType = "Bar";
  this.config.push("graphType");
  this.graphOrientation = "horizontal";
  this.config.push("graphOrientation");
  this.foreground = "rgb(0,0,0)";
  this.config.push("foreground");
  this.foregroundWindow = "rgb(0,0,0)";
  this.config.push("foregroundWindow");
  this.background = "rgb(255,255,255)";
  this.config.push("background");
  this.backgroundWindow = "rgb(255,255,255)";
  this.config.push("backgroundWindow");
  this.backgroundType = "solid";
  this.config.push("backgroundType");
  this.backgroundImage = false;
  this.config.push("backgroundImage");
  this.backgroundWindowGradientOrientation = "vertical";
  this.config.push("backgroundWindowGradientOrientation");
  this.backgroundGradient1Color = "rgb(0,0,200)";
  this.config.push("backgroundGradient1Color");
  this.backgroundGradient2Color = "rgb(0,0,36)";
  this.config.push("backgroundGradient2Color");
  this.backgroundWindowGradient1Color = "rgb(0,0,200)";
  this.config.push("backgroundWindowGradient1Color");
  this.backgroundWindowGradient2Color = "rgb(0,0,36)";
  this.config.push("backgroundWindowGradient2Color");
  this.gradient = false;
  this.config.push("gradient");
  this.gradientType = "radial";
  this.config.push("gradientType");
  this.gradientRatio = 1.3;
  this.config.push("gradientRatio");
  this.transparency = null;
  this.config.push("transparency");
  this.margin = 5;
  this.config.push("margin");
  this.marginLeft = 20;
  this.config.push("marginLeft");
  this.marginTop = 20;
  this.config.push("marginTop");
  this.marginBottom = 20;
  this.config.push("marginBottom");
  this.marginRight = 20;
  this.config.push("marginRight");
  this.showShadow = false;
  this.config.push("showShadow");
  this.shadowOffsetX = 3;
  this.config.push("shadowOffsetX");
  this.shadowOffsetY = 3;
  this.config.push("shadowOffsetY");
  this.shadowBlur = 2;
  this.config.push("shadowBlur");
  this.shadowColor = "rgba(0,0,0,0.5)";
  this.config.push("shadowColor");
  this.canvasBox = true;
  this.config.push("canvasBox");
  this.canvasBoxColor = "rgb(0,0,0)";
  this.config.push("canvasBoxColor");
  this.autoAdjust = true;
  this.config.push("autoAdjust");
  this.functions = [];
  this.fonts = this.isVML || this.subBrowser ? ["Gentilis", "Helvetiker", "Optimer"] : ["Gentilis", "Helvetiker", "Optimer", "Verdana, sans-serif"];
  this.fontName = this.isVML ? "Gentilis" : this.subBrowser ? "Optimer" : "Verdana, sans-serif";
  this.config.push("fontName");
  this.fontStyle = false;
  this.config.push("fontStyle");
  this.fontSize = 12;
  this.config.push("fontSize");
  this.font = this.fontStyle ? this.fontStyle + this.fontSize + "pt " + this.fontName : this.fontSize + "pt " + this.fontName;
  this.align = "center";
  this.config.push("align");
  this.baseline = "middle";
  this.config.push("baseline");
  this.maxTextSize = 14;
  this.config.push("maxTextSize");
  this.minTextSize = 4;
  this.config.push("minTextSize");
  this.autoScaleFont = true;
  this.config.push("autoScaleFont");
  this.scaleTextConstantMult = 40;
  this.scaleTextConstantAdd = 2.5;
  this.title = false;
  this.config.push("title");
  this.titleHeight = 30;
  this.config.push("titleHeight");
  this.subtitle = false;
  this.config.push("subtitle");
  this.subtitleHeight = 16;
  this.config.push("subtitleHeight");
  this.dashLength = 8;
  this.config.push("dashLength");
  this.dotLength = 5;
  this.config.push("dotLength");
  this.arrowPointSize = 10;
  this.config.push("arrowPointSize");
  this.capType = "butt";
  this.config.push("capType");
  this.outlineWidth = 1;
  this.config.push("outlineWidth");
  this.lines = [];
  this.colorScheme = "user";
  this.config.push("colorScheme");
  this.colors = ["rgb(255,0,0)", "rgb(0,0,255)", "rgb(0,255,0)", "rgb(255,255,0)", "rgb(0,255,255)", "rgb(255,0,255)", "rgb(153,51,0)", "rgb(0,0,128)", "rgb(128,128,0)", "rgb(255,102,0)", "rgb(0,128,128)", "rgb(102,0,102)", "rgb(255,153,204)", "rgb(102,102,153)", "rgb(51,51,0)", "rgb(255,204,153)", "rgb(204,255,255)", "rgb(153,153,255)", "rgb(255,128,128)", "rgb(51,102,255)", "rgb(51,153,102)", "rgb(255,204,0)", "rgb(0,204,255)", "rgb(208,32,144)", "rgb(153,51,102)", "rgb(0,102,204)", "rgb(153,204,0)", "rgb(255,153,0)", "rgb(51,204,204)", "rgb(204,153,255)", "rgb(128,0,0)", "rgb(51,51,153)", "rgb(0,128,0)", "rgb(255,255,204)", "rgb(153,204,255)", "rgb(128,0,128)"];
  this.colors0 = ["rgb(0,0,255)", "rgb(255,0,77)", "rgb(255,191,0)", "rgb(116,255,0)", "rgb(164,0,255)", "rgb(255,116,0)", "rgb(255,255,0)", "rgb(0,255,255)", "rgb(255,0,0)", "rgb(255,211,0)", "rgb(61,0,255)", "rgb(0,255,0)", "rgb(255,0,255)", "rgb(255,146,0)", "rgb(211,255,0)", "rgb(0,144,255)", "rgb(109,0,255)", "rgb(255,73,0)", "rgb(255,232,0)", "rgb(0,255,146)", "rgb(255,0,144)", "rgb(255,170,0)", "rgb(170,255,0)", "rgb(0,77,255)"];
  this.colors1 = ["rgb(60,60,157)", "rgb(157,60,90)", "rgb(157,133,60)", "rgb(104,157,60)", "rgb(122,60,157)", "rgb(157,104,60)", "rgb(157,157,60)", "rgb(60,157,157)", "rgb(157,60,60)", "rgb(157,140,60)", "rgb(83,60,157)", "rgb(60,157,60)", "rgb(157,60,157)", "rgb(157,115,60)", "rgb(140,157,60)", "rgb(60,115,157)", "rgb(102,60,157)", "rgb(157,88,60)", "rgb(157,148,60)", "rgb(60,157,115)", "rgb(157,60,115)", "rgb(157,125,60)", "rgb(125,157,60)", "rgb(60,90,157)"];
  this.colors2 = ["rgb(0,0,117)", "rgb(117,0,36)", "rgb(117,88,0)", "rgb(53,117,0)", "rgb(76,0,117)", "rgb(117,53,0)", "rgb(117,117,0)", "rgb(0,117,117)", "rgb(117,0,0)", "rgb(117,97,0)", "rgb(28,0,117)", "rgb(0,117,0)", "rgb(117,0,117)", "rgb(117,67,0)", "rgb(97,117,0)", "rgb(0,66,117)", "rgb(50,0,117)", "rgb(117,33,0)", "rgb(117,107,0)", "rgb(0,117,67)", "rgb(117,0,66)", "rgb(117,78,0)", "rgb(78,117,0)", "rgb(0,36,117)"];
  this.colors3 = ["rgb(101,101,255)", "rgb(255,101,147)", "rgb(255,216,101)", "rgb(171,255,101)", "rgb(200,101,255)", "rgb(255,171,101)", "rgb(255,255,101)", "rgb(101,255,255)", "rgb(255,101,101)", "rgb(255,229,101)", "rgb(137,101,255)", "rgb(101,255,101)", "rgb(255,101,255)", "rgb(255,189,101)", "rgb(229,255,101)", "rgb(101,188,255)", "rgb(167,101,255)", "rgb(255,145,101)", "rgb(255,241,101)", "rgb(101,255,189)", "rgb(255,101,188)", "rgb(255,204,101)", "rgb(204,255,101)", "rgb(101,147,255)"];
  this.colors4 = ["rgb(181,181,255)", "rgb(255,181,203)", "rgb(255,237,181)", "rgb(215,255,181)", "rgb(229,181,255)", "rgb(255,215,181)", "rgb(255,255,181)", "rgb(181,255,255)", "rgb(255,181,181)", "rgb(255,242,181)", "rgb(199,181,255)", "rgb(181,255,181)", "rgb(255,181,255)", "rgb(255,223,181)", "rgb(242,255,181)", "rgb(181,223,255)", "rgb(213,181,255)", "rgb(255,202,181)", "rgb(255,248,181)", "rgb(181,255,223)", "rgb(255,181,223)", "rgb(255,230,181)", "rgb(230,255,181)", "rgb(181,203,255)"];
  this.config.push("colors");
  this.shapes = ["sphere", "square", "triangle", "star", "rhombus", "octagon", "oval", "plus", "minus", "pacman", "mdavid", "rect2", "rect3", "rectangle", "image"];
  this.sizes = [4, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30, 32, 34];
  this.images = [];
  this.config.push("images");
  this.timeFormat = "isoDate";
  this.config.push("timeFormat");
  this.maxSmpStringLen = 50;
  this.config.push("maxSmpStringLen");
  this.maxVarStringLen = 30;
  this.config.push("maxVarStringLen");
  this.smpLabelDescription = false;
  this.config.push("smpLabelDescription");
  this.smpTitle = false;
  this.config.push("smpTitle");
  this.smpTitleFontSize = 10;
  this.config.push("smpTitleFontSize");
  this.smpTitleFont = (this.smpTitleFontSize) + "Pt " + this.fontName;
  this.smpTitleColor = this.foreground;
  this.config.push("smpTitleColor");
  this.smpTitleScaleFontFactor = 1;
  this.config.push("smpTitleScaleFontFactor");
  this.smpLabelFontSize = 10;
  this.config.push("smpLabelFontSize");
  this.smpLabelFont = (this.smpLabelFontSize) + "Pt " + this.fontName;
  this.smpLabelColor = this.foreground;
  this.config.push("smpLabelColor");
  this.smpLabelScaleFontFactor = 1;
  this.config.push("smpLabelScaleFontFactor");
  this.smpLabelRotate = 0;
  this.config.push("smpLabelRotate");
  this.varLabelDescription = false;
  this.config.push("varLabelDescription");
  this.varTitle = false;
  this.config.push("varTitle");
  this.varTitleFontSize = 10;
  this.config.push("varTitleFontSize");
  this.varTitleFont = (this.varTitleFontSize) + "Pt " + this.fontName;
  this.varTitleColor = this.foreground;
  this.config.push("varTitleColor");
  this.varTitleScaleFontFactor = 1;
  this.config.push("varTitleScaleFontFactor");
  this.varLabelFontSize = 10;
  this.config.push("varLabelFontSize");
  this.varLabelFont = (this.varLabelFontSize) + "Pt " + this.fontName;
  this.varLabelColor = this.foreground;
  this.config.push("varLabelColor");
  this.varLabelScaleFontFactor = 1;
  this.config.push("varLabelScaleFontFactor");
  this.varLabelRotate = 0;
  this.config.push("varLabelRotate");
  this.varTitleLabelOverlayPosition = "bottom";
  this.config.push("varTitleLabelOverlayPosition");
  this.showSampleNames = true;
  this.config.push("showSampleNames");
  this.showVariableNames = true;
  this.config.push("showVariableNames");
  this.highlightVar = [];
  this.config.push("highlightVar");
  this.highlightSmp = [];
  this.config.push("highlightSmp");
  this.highlightNode = [];
  this.config.push("highlightNode");
  this.smpHighlightColor = "rgb(255,0,0)";
  this.config.push("smpHighlightColor");
  this.varHighlightColor = "rgb(255,0,0)";
  this.config.push("varHighlightColor");
  this.nodeHighlightColor = "rgb(255,0,0)";
  this.config.push("nodeHighlightColor");
  this.selectNode = {};
  this.isSelectNodes = 0;
  this.overlaysWidth = 30;
  this.config.push("overlaysWidth");
  this.overlays = [];
  this.smpOverlays = [];
  this.config.push("smpOverlays");
  this.varOverlays = [];
  this.config.push("varOverlays");
  this.showOverlays = true;
  this.config.push("showOverlays");
  this.overlayFontSize = 6;
  this.config.push("overlayFontSize");
  this.overlayFont = (this.overlayFontSize) + "Pt " + this.fontName;
  this.overlayFontColor = this.foreground;
  this.config.push("overlayFontColor");
  this.overlayScaleFontFactor = 1;
  this.config.push("overlayScaleFontFactor");
  this.showLegend = true;
  this.config.push("showLegend");
  this.legendPosition = "right";
  this.config.push("legendPosition");
  this.legendColor = this.foreground;
  this.config.push("legendColor");
  this.legendBox = true;
  this.config.push("legendBox");
  this.legendBoxColor = "rgb(0,0,0)";
  this.config.push("legendBoxColor");
  this.legendBackgroundColor = this.background;
  this.config.push("legendBackgroundColor");
  this.legendFontSize = 10;
  this.config.push("legendFontSize");
  this.legendFont = (this.legendFontSize) + "Pt " + this.fontName;
  this.legendScaleFontFactor = 1;
  this.config.push("legendScaleFontFactor");
  this.showIndicators = true;
  this.config.push("showIndicators");
  this.indicatorsPosition = "bottom";
  this.config.push("indicatorsPosition");
  this.showDecorations = false;
  this.config.push("showDecorations");
  this.decorationsPosition = "bottom";
  this.config.push("decorationsPosition");
  this.decorationsColor = this.foreground;
  this.config.push("decorationsColor");
  this.decorationFontSize = 10;
  this.config.push("decorationFontSize");
  this.decorationFont = (this.decorationFontSize) + "Pt " + this.fontName;
  this.decorationScaleFontFactor = 1;
  this.config.push("decorationScaleFontFactor");
  this.decorationsWidth = 10;
  this.config.push("decorationsWidth");
  this.decorationsHeight = 20;
  this.config.push("decorationsHeight");
  this.decorationsType = "bar";
  this.config.push("decorationsType");
  this.decorations = [];
  this.config.push("decorations");
  this.decorationsColors = [];
  this.config.push("decorationsColors");
  this.axisTickColor = this.foreground;
  this.config.push("axisTickColor");
  this.axisTickFontSize = 10;
  this.config.push("axisTickFontSize");
  this.axisTickFont = (this.axisTickFontSize) + "Pt " + this.fontName;
  this.tickScaleFontFactor = 1;
  this.config.push("tickScaleFontFactor");
  this.tickRotate = 0;
  this.config.push("tickRotate");
  this.axisTitleColor = this.foreground;
  this.config.push("axisTitleColor");
  this.axisTitleFontSize = 10;
  this.config.push("axisTitleFontSize");
  this.axisTitleFont = (this.axisTitleFontSize) + "Pt " + this.fontName;
  this.axisTitleScaleFontFactor = 1;
  this.config.push("axisTitleScaleFontFactor");
  this.isGroupedData = false;
  this.isTransformedData = false;
  this.isBoxPlotCalc = false;
  this.isMarketDataFormated = false;
  this.isMarketSwitched = false;
  this.isGraphTime = false;
  this.config.push("isGraphTime");
  this.groupingFactors = [];
  this.config.push("groupingFactors");
  this.segregateSamplesBy = false;
  this.config.push("segregateSamplesBy");
  this.segregateVariablesBy = false;
  this.config.push("segregateVariablesBy");
  this.isLogData = false;
  this.config.push("isLogData");
  this.smpSort = -1;
  this.varSort = -1;
  this.tmpAsciiArray = [];
  this.sortDir = "ascending";
  this.config.push("sortDir");
  this.transformBase = 2;
  this.config.push("transformBase");
  this.transformType = false;
  this.config.push("transformType");
  this.ratioReference = 0;
  this.config.push("ratioReference");
  this.zscoreAxis = "samples";
  this.config.push("zscoreAxis");
  this.showErrorBars = true;
  this.config.push("showErrorBars");
  this.randomData = false;
  this.config.push("randomData");
  this.randomDataMean = 100;
  this.config.push("randomDataMean");
  this.randomDataSigma = 50;
  this.config.push("randomDataSigma");
  this.randomDataVariables = 6;
  this.config.push("randomDataVariables");
  this.randomDataSamples = 12;
  this.config.push("randomDataSamples");
  this.randomDataVariableAnnotations = 2;
  this.config.push("randomDataVariableAnnotations");
  this.randomDataSampleAnnotations = 3;
  this.config.push("randomDataSampleAnnotations");
  this.randomDataVariableAnnotationRatio = 2;
  this.config.push("randomDataVariableAnnotationRatio");
  this.randomDataSampleAnnotationRatio = 2;
  this.config.push("randomDataSampleAnnotationRatio");
  this.randomMissingDataPercentage = 0;
  this.config.push("randomMissingDataPercentage");
  this.missingDataColor = "rgb(255,255,0)";
  this.config.push("missingDataColor");
  this.zoom = 1;
  this.config.push("zoom");
  this.zoomStep = 0.1;
  this.config.push("zoomStep");
  this.panningX = 0;
  this.config.push("panningX");
  this.panningY = 0;
  this.config.push("panningY");
  this.panningGlobalX = 0;
  this.panningGlobalY = 0;
  this.panningStep = 0.1;
  this.config.push("panningStep");
  this.distance = "euclidian";
  this.config.push("distance");
  this.linkage = "single";
  this.config.push("linkage");
  this.clusterAxis = "samples";
  this.config.push("clusterAxis");
  this.kmeansClusters = 3;
  this.config.push("kmeansClusters");
  this.maxIterations = 10;
  this.config.push("maxIterations");
  this.imputeMethod = "mean";
  this.config.push("imputeMethod");
  this.centerData = false;
  this.config.push("centerData");
  this.useSmpVarIndexOnClustering = false;
  this.config.push("useSmpVarIndexOnClustering");
  this.xAxisAbsMax = null;
  this.xAxisAbsMin = null;
  this.xAxis2AbsMax = null;
  this.xAxis2AbsMin = null;
  this.yAxisAbsMax = null;
  this.yAxisAbsMin = null;
  this.zAxisAbsMax = null;
  this.zAxisAbsMin = null;
  this.xAxisMaxStrLength = false;
  this.xAxis2MaxStrLength = false;
  this.yAxisMaxStrLength = false;
  this.zAxisMaxStrLength = false;
  this.setMax = null;
  this.config.push("setMax");
  this.setMin = null;
  this.config.push("setMin");
  this.setMax2 = null;
  this.config.push("setMax2");
  this.setMin2 = null;
  this.config.push("setMin2");
  this.setMaxX = null;
  this.config.push("setMaxX");
  this.setMinX = null;
  this.config.push("setMinX");
  this.setMaxY = null;
  this.config.push("setMaxY");
  this.setMinY = null;
  this.config.push("setMinY");
  this.setMaxZ = null;
  this.config.push("setMaxZ");
  this.setMinZ = null;
  this.config.push("setMinZ");
  this.xAxisCurrent = 0;
  this.config.push("xAxisCurrent");
  this.yAxisCurrent = 0;
  this.config.push("yAxisCurrent");
  this.zAxisCurrent = 0;
  this.config.push("zAxisCurrent");
  this.xAxisTitle = false;
  this.config.push("xAxisTitle");
  this.yAxisTitle = false;
  this.config.push("yAxisTitle");
  this.zAxisTitle = false;
  this.config.push("zAxisTitle");
  this.xAxis = [];
  this.config.push("xAxis");
  this.xAxis2 = [];
  this.config.push("xAxis2");
  this.yAxis = [];
  this.config.push("yAxis");
  this.zAxis = [];
  this.config.push("zAxis");
  this.xAxisTickStyle = "solid";
  this.config.push("xAxisTickStyle");
  this.yAxisTickStyle = "solid";
  this.config.push("yAxisTickStyle");
  this.zAxisTickStyle = "solid";
  this.config.push("zAxisTickStyle");
  this.xAxisTickFormat = false;
  this.config.push("xAxisTickFormat");
  this.xAxis2TickFormat = false;
  this.config.push("xAxis2TickFormat");
  this.yAxisTickFormat = false;
  this.config.push("yAxisTickFormat");
  this.zAxisTickFormat = false;
  this.config.push("zAxisTickFormat");
  this.xAxisTickColor = "rgb(220,220,220)";
  this.config.push("xAxisTickColor");
  this.yAxisTickColor = "rgb(220,220,220)";
  this.config.push("yAxisTickColor");
  this.zAxisTickColor = "rgb(220,220,220)";
  this.config.push("zAxisTickColor");
  this.axisExtension = 0.1;
  this.config.push("axisExtension");
  this.axisExact = false;
  this.config.push("axisExact");
  this.xAxisExact = false;
  this.config.push("xAxisExact");
  this.xAxis2Exact = false;
  this.config.push("xAxis2Exact");
  this.yAxisExact = false;
  this.config.push("yAxisExact");
  this.zAxisExact = false;
  this.config.push("zAxisExact");
  this.timeValues = [];
  this.config.push("timeValues");
  this.timeValueIndices = [];
  this.config.push("timeValueIndices");
  this.xAxisValues = [];
  this.config.push("xAxisValues");
  this.xAxis2Values = [];
  this.config.push("xAxis2Values");
  this.yAxisValues = [];
  this.config.push("yAxisValues");
  this.zAxisValues = [];
  this.config.push("zAxisValues");
  this.xAxisMinorValues = [];
  this.config.push("xAxisMinorValues");
  this.xAxis2MinorValues = [];
  this.config.push("xAxis2MinorValues");
  this.yAxisMinorValues = [];
  this.config.push("yAxisMinorValues");
  this.zAxisMinorValues = [];
  this.config.push("zAxisMinorValues");
  this.xAxisTransform = false;
  this.config.push("xAxisTransform");
  this.yAxisTransform = false;
  this.config.push("yAxisTransform");
  this.zAxisTransform = false;
  this.config.push("zAxisTransform");
  this.xAxisTransformTicks = true;
  this.config.push("xAxisTransformTicks");
  this.yAxisTransformTicks = true;
  this.config.push("yAxisTransformTicks");
  this.zAxisTransformTicks = true;
  this.config.push("zAxisTransformTicks");
  this.xAxisShow = true;
  this.config.push("xAxisShow");
  this.yAxisShow = true;
  this.config.push("yAxisShow");
  this.zAxisShow = true;
  this.config.push("zAxisShow");
  this.ticks = 50;
  this.config.push("ticks");
  this.timeTicksFirst = false;
  this.config.push("timeTicksFirst");
  this.xAxisTicks = 5;
  this.config.push("xAxisTicks");
  this.xAxis2Ticks = 5;
  this.config.push("xAxis2Ticks");
  this.yAxisTicks = 5;
  this.config.push("yAxisTicks");
  this.zAxisTicks = 5;
  this.config.push("zAxisTicks");
  this.xAxisMinorTicks = true;
  this.config.push("xAxisMinorTicks");
  this.xAxis2MinorTicks = true;
  this.config.push("xAxis2MinorTicks");
  this.yAxisMinorTicks = true;
  this.config.push("yAxisMinorTicks");
  this.zAxisMinorTicks = false;
  this.config.push("zAxisMinorTicks");
  this.smpHairline = "dotted";
  this.config.push("smpHairline");
  this.smpHairlineColor = "rgb(220,220,220)";
  this.config.push("smpHairlineColor");
  this.blockContrast = false;
  this.config.push("blockContrast");
  this.blockContrastOddColor = this.background;
  this.config.push("blockContrastOddColor");
  this.blockContrastEvenColor = "rgb(240,240,240)";
  this.config.push("blockContrastEvenColor");
  this.seriesSeparationFactor = 2;
  this.config.push("seriesSeparationFactor");
  this.blockSeparationFactor = 1;
  this.config.push("blockSeparationFactor");
  this.blockFactor = 1;
  this.config.push("blockFactor");
  this.showDataValues = false;
  this.config.push("showDataValues");
  this.lineDecoration = "symbol";
  this.config.push("lineDecoration");
  this.coordinateLineColor = true;
  this.config.push("coordinateLineColor");
  this.lineThickness = 1;
  this.config.push("lineThickness");
  this.lineType = "rect";
  this.config.push("lineType");
  this.areaIsCumulative = false;
  this.config.push("areaIsCumulative");
  this.colorBy = false;
  this.config.push("colorBy");
  this.shapeBy = false;
  this.config.push("shapeBy");
  this.sizeBy = false;
  this.config.push("sizeBy");
  this.scatterType = false;
  this.config.push("scatterType");
  this.selectDataPoint = [];
  this.hideDataPoint = [];
  this.isSelectDataPoints = 0;
  this.isHistogram = false;
  this.config.push("isHistogram");
  this.histogramBarWidth = 0.5;
  this.config.push("histogramBarWidth");
  this.histogramBins = 20;
  this.config.push("histogramBins");
  this.isCreateHistogram = false;
  this.allVsAll = false;
  this.config.push("allVsAll");
  this.allVsAllType = "both";
  this.config.push("allVsAllType");
  this.functionIntervals = 20;
  this.config.push("functionIntervals");
  this.xRotate = 45;
  this.config.push("xRotate");
  this.yRotate = 0;
  this.config.push("yRotate");
  this.zRotate = 0;
  this.config.push("zRotate");
  this.minRotate = null;
  this.config.push("minRotate");
  this.maxRotate = null;
  this.config.push("maxRotate");
  this.show3DGrid = true;
  this.config.push("show3DGrid");
  this.rotationDelay = 100;
  this.config.push("rotationDelay");
  this.rotationSensitivity = 45;
  this.config.push("rotationSensitivity");
  this.rotationStep = 2;
  this.config.push("rotationStep");
  this.indicatorCenter = "white";
  this.config.push("indicatorCenter");
  this.indicatorHeight = 8;
  this.config.push("indicatorHeight");
  this.indicatorWidth = 1;
  this.config.push("indicatorWidth");
  this.heatmapType = "blue-red";
  this.config.push("heatmapType");
  this.varIndicesStart = -1;
  this.smpIndicesStart = -1;
  this.correlationAxis = "samples";
  this.config.push("correlationAxis");
  this.correlationAnchorLegend = false;
  this.config.push("correlationAnchorLegend");
  this.correlationAnchorLegendAlignWidth = 40;
  this.config.push("correlationAnchorLegendAlignWidth");
  this.showSmpDendrogram = false;
  this.config.push("showSmpDendrogram");
  this.showVarDendrogram = false;
  this.config.push("showVarDendrogram");
  this.varDendrogramPosition = "top";
  this.config.push("varDendrogramPosition");
  this.smpDendrogramPosition = "left";
  this.config.push("smpDendrogramPosition");
  this.dendrogramSpace = 6;
  this.config.push("dendrogramSpace");
  this.dendrogramHang = false;
  this.config.push("dendrogramHang");
  this.dendrogramHeight = false;
  this.config.push("dendrogramHeight");
  this.vennGroups = 4;
  this.config.push("vennGroups");
  this.vennCompartments = ["A", "B", "AB", "C", "AC", "BC", "ABC", "D", "AD", "BD", "CD", "ABD", "ACD", "BCD", "ABCD"];
  this.vennColors = false;
  this.config.push("vennColors");
  this.pieType = "separated";
  this.config.push("pieType");
  this.pieSegmentPrecision = 0;
  this.config.push("pieSegmentPrecision");
  this.pieSegmentSeparation = 3;
  this.config.push("pieSegmentSeparation");
  this.pieSegmentLabels = "inside";
  this.config.push("pieSegmentLabels");
  this.maxPieSectors = 20;
  this.config.push("maxPieSectors");
  this.showPieGrid = true;
  this.config.push("showPieGrid");
  this.showPieValues = true;
  this.config.push("showPieValues");
  this.showPieSampleLabel = true;
  this.config.push("showPieSampleLabel");
  this.startPieSectors = 0;
  this.config.push("startPieSectors");
  this.showVolume = true;
  this.config.push("showVolume");
  this.stockIndicators = ["Sma5", "Sma10", "Sma25"];
  this.config.push("stockIndicators");
  this.sma5Color = "rgb(255,0,0)";
  this.config.push("sma5Color");
  this.sma10Color = "rgb(0,255,0)";
  this.config.push("sma1Color");
  this.sma20Color = "rgb(0,0,255)";
  this.config.push("sma20Color");
  this.sma25Color = "rgb(255,0,255)";
  this.config.push("sma25Color");
  this.sma50Color = "rgb(0,0,0)";
  this.config.push("sma50Color");
  this.calculateLayout = true;
  this.config.push("calculateLayout");
  this.networkFreeze = false;
  this.config.push("networkFreeze");
  this.networkFreezeOnLoad = false;
  this.config.push("networkFreezeOnLoad");
  this.networkStack = [];
  this.networkStack[0] = [];
  this.networkStack[1] = [];
  this.networkStackStates = 5;
  this.config.push("networkStackStates");
  this.approximateNodePositions = false;
  this.config.push("approximateNodePositions");
  this.networkLayoutType = "forceDirected";
  this.config.push("networkLayoutType");
  this.networkRoot = false;
  this.config.push("networkRoot");
  this.showNetworkRadialLayout = false;
  this.config.push("showNetworkRadialLayout");
  this.networkDepth = 0;
  this.networkDivisions = 0;
  this.preScaleNetwork = false;
  this.config.push("preScaleNetwork");
  this.showNetworkNodesLegend = true;
  this.config.push("showNetworkNodesLegend");
  this.showNetworkEdgesLegend = true;
  this.config.push("showNetworkEdgesLegend");
  this.showNetworkTextLegend = true;
  this.config.push("showNetworkTextLegend");
  this.showNetworkDecorationsLegend = true;
  this.config.push("showNetworkDecorationsLegend");
  this.showNodeNameThreshold = 50;
  this.config.push("showNodeNameThreshold");
  this.showHiddenChildEdges = true;
  this.config.push("showHiddenChildEdges");
  this.nodeSize = this.preScaleNetwork ? 10 : 20;
  this.config.push("nodeSize");
  this.edgeWidth = 1;
  this.config.push("edgeWidth");
  this.layoutTime = 15;
  this.config.push("layoutTime");
  this.nodeFontSize = this.nodeSize;
  this.config.push("nodeFontSize");
  this.nodeFont = this.nodeFontSize + "Pt " + this.fontName;
  this.nodeFontColor = this.foreground;
  this.config.push("nodeFontColor");
  this.nodeScaleFontFactor = 1;
  this.config.push("nodeScaleFontFactor");
  this.randomNetwork = false;
  this.config.push("randomNetwork");
  this.reduceRandomNetwork = false;
  this.config.push("reduceRandomNetwork");
  this.randomNetworkNodes = 50;
  this.config.push("randomNetworkNodes");
  this.randomNetworkNodeEdgesMax = 5;
  this.config.push("randomNetworkNodeEdgesMax");
  this.colorNodeBy = false;
  this.config.push("colorNodeBy");
  this.shapeNodeBy = false;
  this.config.push("shapeNodeBy");
  this.sizeNodeBy = false;
  this.config.push("sizeNodeBy");
  this.colorEdgeBy = false;
  this.config.push("colorEdgeBy");
  this.shapeEdgeBy = false;
  this.config.push("shapeEdgeBy");
  this.sizeEdgeBy = false;
  this.config.push("sizeEdgeBy");
  this.sizeDecorationBy = false;
  this.config.push("sizeDecorationBy");
  this.is3DNetwork = false;
  this.config.push("is3DNetwork");
  this.network2DRotate = 0;
  this.skipClick = false;
  this.autoHideOnDecorationsCenter = false;
  this.config.push("autoHideOnDecorationsCenter");
  this.networkNodesOnTop = true;
  this.config.push("networkNodesOnTop");
  this.trackNameFontSize = 10;
  this.config.push("trackNameFontSize");
  this.trackNameFont = (this.trackNameFontSize) + "Pt " + this.fontName;
  this.trackNameFontColor = this.background;
  this.config.push("trackNameFontColor");
  this.showFeatureNameThereshold = 20;
  this.config.push("showFeatureNameThereshold ");
  this.featureNameFontSize = 8;
  this.config.push("featureNameFontSize");
  this.featureNameFont = (this.featureNameFontSize) + "Pt " + this.fontName;
  this.featureNameFontColor = this.background;
  this.config.push("featureNameFontColor");
  this.featureWidthDefault = 3;
  this.config.push("featureWidthDefault");
  this.featureHeightDefault = 8;
  this.config.push("featureHeightDefault");
  this.featureTypeDefault = "line";
  this.config.push("featureTypeDefault");
  this.sequenceFontSize = 10;
  this.config.push("sequenceFontSize");
  this.sequenceFont = (this.sequenceFontSize) + "Pt " + this.fontName;
  this.sequenceAColor = "rgb(255,255,0)";
  this.config.push("sequenceAColor");
  this.sequenceCColor = "rgb(0,255,255)";
  this.config.push("sequenceCColor");
  this.sequenceGColor = "rgb(0,255,0)";
  this.config.push("sequenceGColor");
  this.sequenceTColor = "rgb(255,0,0)";
  this.config.push("sequenceTColor");
  this.sequenceMColor = "rgb(255,0,255)";
  this.config.push("sequenceMColor");
  this.wireColor = this.isVML ? "rgb(220,220,220)" : "rgba(220,220,220,0.1)";
  this.config.push("wireColor");
  this.periodTicksLabels = 5;
  this.config.push("periodTicksLabels");
  this.tension = 0.3;
  this.config.push("tension");
  this.debug = false;
  this.config.push("debug");
  this.showVersion = true;
  this.config.push("showVersion");
  this.showCode = false;
  this.config.push("showCode");
  this.codeType = "pretty";
  this.config.push("codeType");
  this.checkGoogleChromeFrame = true;
  this.config.push("checkGoogleChromeFrame");
  this.disableConfigurator = false;
  this.config.push("disableConfigurator");
  this.disableEvents = false;
  this.config.push("disableEvents");
  this.eventKeys = true;
  this.config.push("eventKeys");
  this.eventArrowKeys = true;
  this.config.push("eventArrowKeys");
  this.eventPlusMinusKeys = false;
  this.config.push("eventPlusMinusKeys");
  this.helpKeyEvents = true;
  this.config.push("helpKeyEvents");
  this.dragAreaColor = "rgb(208,222,240)";
  this.config.push("dragAreaColor");
  this.selectedBackgroundColor = "rgb(255,0,0)";
  this.config.push("selectedBackgroundColor");
  this.infoAreaColor = "rgb(208,222,240)";
  this.config.push("infoAreaColor");
  this.infoTimeOut = 3000;
  this.config.push("infoTimeOut");
  this.modelEvent = "global";
  this.config.push("modelEvent");
  this.dataEvent = [];
  this.broadcast = true;
  this.config.push("broadcast");
  this.broadcastType = "var";
  this.config.push("broadcastType");
  this.resizerAreaColor = this.isIE ? "rgb(210,210,210)" : "rgba(210,210,210,0.7)";
  this.config.push("resizerAreaColor");
  this.resizerAreaColorCurrent = "rgb(237,247,255)";
  this.config.push("resizerAreaColorCurrent");
  this.resizerAreaColorOutlineCurrent = "rgb(59,138,229)";
  this.config.push("resizerAreaColorOutlineCurrent");
  this.resizerTransparency = true;
  this.config.push("resizerTransparency");
  this.resizerType = false;
  this.config.push("resizerType");
  this.resizerDraw = false;
  this.config.push("resizerDraw");
  this.resizerPosition = "bottom";
  this.config.push("resizerPosition");
  this.resizerWidth = 50;
  this.config.push("resizerWidth");
  this.resizerBackgroundImage = false;
  this.config.push("resizerBackgroundImage");
  this.resizerDataIndex = 0;
  this.config.push("resizerDataIndex");
  this.resizerEventData = {};
  this.showAnimation = false;
  this.config.push("showAnimation");
  this.showAnimationFontSize = 10;
  this.config.push("showAnimationFontSize");
  this.showAnimationFont = (this.showAnimationFontSize) + "Pt " + this.fontName;
  this.showAnimationFontColor = this.background;
  this.config.push("showAnimationFontColor");
  this.animationTime = 30;
  this.config.push("animationTime");
  this.animationType = "grow";
  this.config.push("animationType");
  this.animationCycles = 20;
  this.config.push("animationCycles");
  this.snapshotCopyChangeOnly = true;
  this.config.push("snapshotCopyChangeOnly");
  this.snapshots = [];
  this.isAnimation = false;
  this.weight = 1;
  this.scaleX = 1;
  this.scaleY = 1;
  this.translateX = 0;
  this.translateY = 0;
  this.offsetX = 0;
  this.offsetY = 0;
  this.layoutWidth = 0;
  this.layoutHeight = 0;
  this.layout = "1X1";
  this.config.push("layout");
  this.layoutComb = false;
  this.config.push("layoutComb");
  this.layoutAdjust = false;
  this.config.push("layoutAdjust");
  this.layoutCurrent = 0;
  this.layoutMaxVarLegend = 0;
  this.layoutMulticolor = true;
  this.config.push("layoutMulticolor");
  this.layoutAxis = 3;
  this.imageDir = "http://canvasxpress.org/images/";
  this.config.push("imageDir");
  this.playButton = "play.png";
  this.config.push("playButton");
  this.stopButton = "stop.png";
  this.config.push("stopButton");
  this.previousButton = "rewind.png";
  this.config.push("previousButton");
  this.nextButton = "fast_forward.png";
  this.config.push("nextButton");
  this.remoteService = false;
  this.config.push("remoteService");
  this.remoteParams = {};
  this.config.push("remoteParams");
  this.remoteDirection = "next";
  this.config.push("remoteDirection");
  this.remoteDataIndex = 0;
  this.remoteWindow = 1;
  this.config.push("remoteWindow");
  this.remoteAutoPlay = false;
  this.config.push("remoteAutoPlay");
  this.remoteAutoPlayDelay = 3000;
  this.config.push("remoteAutoPlayDelay");
  this.remoteUpdating = false;
  this.showDataTable = false;
  this.config.push("showDataTable");
  this.dataTableTransposed = true;
  this.config.push("dataTableTransposed");
  this.maxRows = 5;
  this.config.push("maxRows");
  this.maxCols = 8;
  this.config.push("maxCols");
  this.colWidth = 50;
  this.config.push("colWidth");
  this.rowHeight = 15;
  this.config.push("rowHeight");
  this.freezeColLeft = 0;
  this.config.push("freezeColLeft");
  this.freezeColRight = 0;
  this.config.push("freezeColRight");
  this.freezeRowTop = 0;
  this.config.push("freezeRowTop");
  this.freezeRowBottom = 0;
  this.config.push("freezeRowBottom");
  this.refresehDataTableOnDraw = false;
  this.config.push("refresehDataTableOnDraw");
  this.showDataTableOnSelect = true;
  this.config.push("showDataTableOnSelect");
  this.networkShowDataTable = "nodes";
  this.config.push("networkShowDataTable");
  this.startCol = 0;
  this.startRow = 0;
  if (!a) {
    this.setConfig(this.config);
    this.initDimensions();
    this.initUtils();
    this.initPlotUtils();
    this.initConfigUtils();
    this.initLegendUtils();
    this.init3dUtils();
    this.initApi();
    this.initText();
    this.initTime();
    this.initColor();
    this.initAttributes()
  }
};
CanvasXpress.prototype.initDimensions = function () {
  this.resizeCanvas = function (c, a, b) {
    if (this.ctx) {
      this.disableGradientTransparency();
      if (c || a || b) {
        if (a || b) {
          if (a) {
            this.width = a;
            this.originalWidth = a
          }
          if (b) {
            this.height = b;
            this.originalHeight = b
          }
        } else {
          if (c) {
            this.width = this.originalWidth;
            this.height = this.originalHeight
          }
        }
        this.top = this.margin;
        if (this.title) {
          this.top += this.titleHeight + this.margin;
          if (this.subtitle) {
            this.top += this.subtitleHeight + this.margin
          }
        }
        this.left = this.margin;
        this.right = this.margin;
        this.bottom = this.margin;
        this.resetMap()
      }
      this.canvas.width = this.width;
      this.canvas.height = this.height;
      if (this.backgroundType == "image" || this.backgroundType == "windowImage") {
        this.ctx.save();
        this.ctx.translate(0, 0);
        this.ctx.drawImage(CanvasXpress.cacheImages[this.backgroundImage], 0, 0, this.width, this.height);
        this.ctx.restore()
      } else {
        if (this.backgroundType == "solid" || this.backgroundType == "window" || this.backgroundType == "windowSolidGradient") {
          this.rectangle(0, 0, this.width, this.height, this.background, this.background)
        } else {
          this.setLinearGradient(0, 0, 0, this.height, this.backgroundGradient1Color, this.backgroundGradient2Color);
          this.ctx.fillRect(0, 0, this.width, this.height)
        }
      }
      this.enableGradientTransparency()
    }
  };
  this.setWidth = function (a) {
    if (!a) {
      a = 1000
    }
    this.draw(a)
  };
  this.setHeight = function (a) {
    if (!a) {
      a = 8000
    }
    this.draw(false, a)
  };
  this.setDimensions = function (a, b) {
    if (!a) {
      a = 1000
    }
    if (!b) {
      b = 1000
    }
    this.draw(a, b)
  };
  this.initializeDimensions = function () {
    if (this.canvas.width) {
      this.originalWidth = this.canvas.width;
      this.width = this.canvas.width
    } else {
      if (this.width) {
        this.originalWidth = this.width;
        this.canvas.width = this.width
      } else {
        this.width = 600;
        this.originalWidth = this.width;
        this.canvas.width = this.width
      }
    }
    if (this.canvas.height) {
      this.originalHeight = this.canvas.height;
      this.height = this.canvas.height
    } else {
      if (this.height) {
        this.originalHeight = this.height;
        this.canvas.height = this.height
      } else {
        this.height = 600;
        this.originalHeight = this.height;
        this.canvas.height = this.height
      }
    }
    this.top = this.margin;
    if (this.title) {
      this.top += this.titleHeight + this.margin;
      if (this.subtitle) {
        this.top += this.subtitleHeight + this.margin
      }
    }
    this.left = this.margin;
    this.right = this.margin;
    this.bottom = this.margin;
    this.x = this.width - (this.leftMargin + this.left + this.rightMargin + this.right);
    this.y = this.height - (this.topMargin + this.top + this.bottomMargin + this.bottom)
  };
  this.initializeDimensions()
};
CanvasXpress.prototype.initPlotUtils = function () {
  this.drawCanvasBox = function () {
    if (this.canvasBox && this.graphType != "Network") {
      this.rectangle(0, 0, this.width - 1, this.height - 1, false, this.canvasBoxColor, "open")
    }
  };
  this.drawPlotWindow = function () {
    if (this.backgroundType.match("window")) {
      this.disableGradientTransparency();
      var a = this.marginLeft + this.offsetX;
      var b = this.marginTop + this.offsetY;
      if (this.backgroundType == "windowGradient2" || this.backgroundType == "windowSolidGradient") {
        this.ctx.save();
        if (this.backgroundWindowGradientOrientation == "horizontal") {
          this.setLinearGradient(a + this.left, b, a + this.left + this.x, b, this.backgroundWindowGradient1Color, this.backgroundWindowGradient2Color)
        } else {
          this.setLinearGradient(a, b + this.top, a, b + this.top + this.y, this.backgroundWindowGradient1Color, this.backgroundWindowGradient2Color)
        }
        this.ctx.fillRect(a + this.left, b + this.top, this.x, this.y)
      } else {
        this.rectangle(a + this.left, b + this.top, this.x, this.y, this.backgroundWindow, this.backgroundWindow)
      }
      this.enableGradientTransparency()
    }
  };
  this.maskArea = function (q) {
    this.disableGradientTransparency();
    var n = function (i, e, t, l) {
        this.ctx.fillRect(i, e, t, l)
      };
    var d = function (l, i, w, v, t) {
        this.ctx.drawImage(l, i, w, v, t)
      };
    var r;
    var g = this.marginLeft + this.offsetX;
    var u = this.marginTop + this.offsetY;
    var o = this.outlineWidth + 1;
    var a = [0, 0, this.width, this.marginTop];
    var j = [0, 0, this.marginLeft, this.height];
    var s = [0, this.height - this.marginBottom, this.width, this.marginBottom];
    var c = [this.width - this.marginRight, 0, this.marginRight, this.height];
    var h = q ? [g, u, this.left, this.top - o] : [g, u, (this.left + this.x + this.right), this.top - o];
    var p = q ? [g, u + this.top + this.y, this.left - o, this.bottom] : [g, u, this.left - o, (this.top + this.y + this.bottom)];
    var b = q ? [g + this.left + this.x, u + this.top + this.y + o, this.right, this.bottom] : [g, u + this.top + this.y + o, (this.left + this.x + this.right), this.bottom];
    var m = q ? [g + this.left + this.x + o, u, this.right, this.top] : [g + this.left + this.x + o, u, this.right, (this.top + this.y + this.bottom)];
    var f = [a, j, s, c, h, p, b, m];
    if (this.backgroundType == "image" || this.backgroundType == "windowImage") {
      this.ctx.save();
      this.ctx.translate(0, 0);
      for (var k = 0; k < f.length; k++) {
        f[k].unshift(CanvasXpress.cacheImages[this.backgroundImage])
      }
      r = d
    } else {
      if (this.backgroundType == "solid" || this.backgroundType == "window" || this.backgroundType == "windowSolidGradient") {
        for (var k = 0; k < f.length; k++) {
          f[k].push(this.background);
          f[k].push(this.background)
        }
        r = this.rectangle
      } else {
        this.setLinearGradient(0, 0, 0, this.height, this.backgroundGradient1Color, this.backgroundGradient2Color);
        r = n
      }
    }
    if (this.marginTop) {
      r.apply(this, a)
    }
    if (this.marginLeft) {
      r.apply(this, j)
    }
    if (this.marginBottom) {
      r.apply(this, s)
    }
    if (this.marginRight) {
      r.apply(this, c)
    }
    r.apply(this, h);
    r.apply(this, p);
    r.apply(this, b);
    r.apply(this, m);
    if (this.backgroundType == "image" || this.backgroundType == "windowImage") {
      this.ctx.restore()
    }
    this.enableGradientTransparency()
  };
  this.maskPlotArea = function () {
    this.maskArea()
  };
  this.maskTreeArea = function () {
    this.maskArea(true)
  }
};
CanvasXpress.prototype.initColor = function () {
  this.validateColor = function (b, a) {
    if (b.match(/^rgba?\((\d{1,3},\d{1,3},\d{1,3})(?:,([0-9\.]+))?/i)) {
      if (RegExp.$2) {
        b = "rgba(" + RegExp.$1 + "," + RegExp.$2 + ")"
      } else {
        b = "rgb(" + RegExp.$1 + ")"
      }
    } else {
      if (b.match(/^#?[0-9abcdef]/i)) {
        b = this.hexToRgb(b)
      } else {
        b = false
      }
    }
    return b && a != null ? this.addColorTransparency(b, a) : b
  };
  this.darkenLightenColor = function (j, e) {
    j = this.rgbToHex(j).substring(1, 7);
    var i = parseInt(j, 16);
    var h = (i >> 16) + e;
    h = h > 255 ? 255 : h < 0 ? 0 : h;
    var f = ((i >> 8) & 255) + e;
    f = f > 255 ? 255 : f < 0 ? 0 : f;
    var d = (i & 255) + e;
    d = d > 255 ? 255 : d < 0 ? 0 : d;
    j = d | (f << 8) | (h << 16);
    return this.hexToRgb(j.toString(16))
  };
  this.rgbToHex = function (i) {
    if (i.substr(0, 1) === "#") {
      return i
    }
    if (i.match(/^rgb\((\d{1,3}),(\d{1,3}),(\d{1,3})\)/)) {
      var f = parseInt(RegExp.$1);
      var e = parseInt(RegExp.$2);
      var a = parseInt(RegExp.$3);
      var d = a | (e << 8) | (f << 16);
      return "#" + d.toString(16)
    } else {
      return i
    }
  };
  this.hexToRgb = function (c) {
    var b = function () {
        return parseInt(c.substring(0, 2), 16)
      };
    var a = function () {
        return parseInt(c.substring(2, 4), 16)
      };
    var d = function () {
        return parseInt(c.substring(4, 6), 16)
      };
    c = c.charAt(0) == "#" ? c.substring(1, 7) : c;
    return "rgb(" + b() + "," + a() + "," + d() + ")"
  };
  this.addColorTransparency = function (b, a) {
    if (b.match(/^rgba?\((\d{1,3},\d{1,3},\d{1,3})(?:,([0-9\.]+))?/i)) {
      if (RegExp.$2) {
        return "rgba(" + RegExp.$1 + "," + a + ")"
      } else {
        return "rgba(" + RegExp.$1 + "," + a + ")"
      }
    } else {
      return false
    }
  };
  this.disableGradientTransparency = function () {
    this.gradientTemp = this.gradient;
    this.transparencyTemp = this.transparency;
    this.gradient = false;
    this.transparency = null
  };
  this.enableGradientTransparency = function () {
    this.gradient = this.gradientTemp;
    this.transparency = this.transparencyTemp
  };
  this.getGradientColor = function (h) {
    if (h.match(/^rgba?\((\d{1,3}),(\d{1,3}),(\d{1,3})(?:,([0-9\.]+))?/i)) {
      var f = Math.floor(parseInt(RegExp.$1) / this.gradientRatio);
      var e = Math.floor(parseInt(RegExp.$2) / this.gradientRatio);
      var a = Math.floor(parseInt(RegExp.$3) / this.gradientRatio);
      var d = RegExp.$4 ? parseFloat(RegExp.$4) : false;
      if (d) {
        return "rgba(" + f + "," + e + "," + a + "," + d + ")"
      } else {
        return "rgb(" + f + "," + e + "," + a + ")"
      }
    } else {
      return false
    }
  };
  this.setGradient = function (e, d, c, b, a) {
    if (this.gradientType == "radial") {
      this.setRadialGradient(e, d, c, b, false, false, true)
    } else {
      this.setLinearGradient(e, d, c, b, a, false, true)
    }
  };
  this.setLinearGradient = function (c, i, a, f, e, b, d) {
    if (!d) {
      this.ctx.save()
    }
    var h = this.ctx.createLinearGradient(c, i, a, f);
    if (!b) {
      b = this.getGradientColor(e);
      h.addColorStop(0, b);
      h.addColorStop(0.6, e);
      h.addColorStop(1, e)
    } else {
      h.addColorStop(0, e);
      h.addColorStop(1, b)
    }
    this.ctx.fillStyle = h
  };
  this.setRadialGradient = function (j, h, a, e, d, c, i) {
    if (!i) {
      this.ctx.save()
    }
    var b = a / 5.5;
    var f = this.ctx.createRadialGradient(j - b, h - b, 1, j, h, a);
    if (!d) {
      if (this.transparency != null) {
        d = "rgba(0,0,0," + this.transparency + ")"
      } else {
        d = "rgb(0,0,0)"
      }
    }
    if (!c) {
      c = "rgba(0,0,0,0)"
    }
    f.addColorStop(0, e);
    f.addColorStop(1, d);
    f.addColorStop(1, c);
    this.ctx.fillStyle = f
  }
};
CanvasXpress.prototype.initUtils = function () {
  this.modifyObjectArray = function (f, b, g) {
    var a = this[g];
    if (b) {
      var d = {};
      var c = [];
      if (this.isArray(f)) {
        for (var e = 0; e < f.length; e++) {
          d[f[e]] = 1
        }
      } else {
        d[f] = 1
      }
      for (var e = 0; e < a.length; e++) {
        if (!d.hasOwnProperty(a[e])) {
          c.push(a[e])
        }
      }
      this[g] = c
    } else {
      if (this.isArray(f)) {
        for (var e = 0; e < f.length; e++) {
          a.push(f[e])
        }
      } else {
        a.push(f)
      }
    }
  };
  this.getObjectArray = function (a) {
    var b = {};
    for (var c = 0; c < a.length; c++) {
      b[a[c]] = 1
    }
    return b
  };
  this.isInArray = function (d, b) {
    if (b && !(b.propertyIsEnumerable("length")) && typeof b === "object" && typeof b.length === "number") {
      for (var c = 0; c < b.length; c++) {
        if (b[c] == d) {
          return true
        }
      }
    }
    return false
  };
  this.isArray = function (a) {
    if (a && !(a.propertyIsEnumerable("length")) && typeof a === "object" && typeof a.length === "number") {
      return true
    } else {
      return false
    }
  };
  this.isObject = function (a) {
    return a instanceof Object
  };
  this.toArray = function (a) {
    return this.isArray(a) ? a : [a]
  };
  this.toNumber = function (b, e) {
    var g = [];
    var c = this.isArray(b);
    b = this.toArray(b);
    for (var d = 0; d < b.length; d++) {
      if (e) {
        g.push(parseFloat(b[d]))
      } else {
        g.push(parseInt(b[d]))
      }
    }
    return c ? g : g[0]
  };
  this.cloneObject = function (c) {
    if (c == null || typeof (c) != "object") {
      return c
    }
    var a = new c.constructor();
    for (var b in c) {
      a[b] = this.cloneObject(c[b])
    }
    return a
  };
  this.isSameObject = function (d, c) {
    if (d === c) {
      return true
    }
    if (typeof (d) != "object" || typeof (c) != "object") {
      return false
    }
    for (var e in d) {
      if (d[e] === c[e]) {
        continue
      }
      if (c[e] === undefined || typeof (d[e]) != typeof (c[e])) {
        return false
      }
      if (!this.isSameObject(d[e], c[e])) {
        return false
      }
    }
    for (var e in c) {
      if (d[e] === undefined) {
        return false
      }
    }
    return true
  }
};
CanvasXpress.prototype.initConfigUtils = function () {
  this.getPixelImage = function () {
    if (navigator.onLine) {
      return this.imageDir + "canvasXpress.gif"
    } else {
      if (this.path) {
        return this.path + "canvasXpress.gif"
      } else {
        alert("Dude! I couldn't find canvasXpress.js")
      }
    }
  };
  this.drawTitle = function () {
    var c;
    var b;
    var a;
    if (this.title) {
      b = this.titleHeight / 2;
      c = this.scaleTextToFont(this.title, this.titleHeight - b, this.x);
      this.drawText(this.title, this.width / 2, this.marginTop + (this.titleHeight / 2), c, false, "center", "middle");
      if (this.subtitle) {
        a = this.subtitleHeight / 4;
        c = this.scaleTextToFont(this.subtitle, this.subtitleHeight - a, this.x);
        this.drawText(this.subtitle, this.width / 2, (this.marginTop + this.titleHeight) + (this.subtitleHeight / 2), c, false, "center", "middle")
      }
    }
  };
  this.drawSelectedBackground = function (f, i, j, b, d, e, c, a, g) {
    if (this.isInSelectedDataPoints(f, i)) {
      this.drawShape(j, b, d, e + 12, c + 12, this.selectedBackgroundColor, this.foreground, "closed", a, 1, g)
    }
  };
  this.isInSelectedDataPoints = function (a, b) {
    if (this.isSelectDataPoints) {
      if (this.selectDataPoint[a]) {
        if (this.broadcastType == "var") {
          return true
        } else {
          if (this.selectDataPoint[a].hasOwnProperty(b)) {
            return true
          }
        }
      }
    }
    return false
  };
  this.isVisibleSelectedDataPoint = function (a, b) {
    if (this.isSelectDataPoints) {
      if (this.hideDataPoint.length > 0 && this.hideDataPoint[a]) {
        if (this.hideDataPoint[a].hasOwnProperty(b)) {
          return false
        }
      }
    }
    return true
  };
  this.getVarColor = function (a) {
    if (this.isArray(this.highlightVar)) {
      for (var b = 0; b < this.highlightVar.length; b++) {
        if (this.highlightVar[b] == a) {
          return this.varHighlightColor
        }
      }
    } else {
      if (this.highlightVar == a) {
        return this.varHighlightColor
      }
    }
    return this.varLabelColor
  };
  this.getSmpColor = function (b) {
    if (this.isArray(this.highlightSmp)) {
      for (var a = 0; a < this.highlightSmp.length; a++) {
        if (this.highlightSmp[a] == b) {
          return this.smpHighlightColor
        }
      }
    } else {
      if (this.highlightSmp == b) {
        return this.smpHighlightColor
      }
    }
    return this.smpLabelColor
  };
  this.getNodeColor = function (b) {
    if (this.isArray(this.highlightNode)) {
      for (var a = 0; a < this.highlightNode.length; a++) {
        if (this.highlightNode[a] == b) {
          return this.nodeHighlightColor
        }
      }
    } else {
      if (this.highlightNode == b) {
        return this.nodeHighlightColor
      }
    }
    return this.selectNode.hasOwnProperty(b) ? this.nodeHighlightColor : this.nodeFontColor
  };
  this.getTechnicalIndicatorColor = function (a) {
    return this[a.toLowerCase() + "Color"] ? this[a.toLowerCase() + "Color"] : "rgb(255,255,0)"
  };
  this.setPieData = function (g, a) {
    var f = {
      no: 0,
      idso: [],
      ids: [],
      data: {},
      oprc: 0,
      tot: this.sum(g)
    };
    for (var c = 0; c < g.length; c++) {
      var b = a ? this.varIndices[c] : c;
      var e = g[b] / f.tot;
      f.data[b] = e;
      if (c >= this.maxPieSectors - 1) {
        f.oprc += e;
        f.idso.push(b);
        f.no++
      } else {
        f.ids.push(b)
      }
    }
    if (f.no) {
      f.ids.push(-1)
    }
    return f
  };
  this.drawPie = function (t, h, g, q, I) {
    var o = this.startPieSectors ? (this.startPieSectors * Math.PI / 180) % 360 : 0;
    var z = this.pieSegmentLabels == "inside" ? q - (q / 3) : q + (q / 10);
    var G = 0;
    var C = this.layoutComb ? this.smpIndices[0] : this.xAxisIndices ? this.xAxisIndices[0] : 0;
    var m = this.ctx.lineWidth;
    var v, e;
    for (var D = 0; D < t.ids.length; D++) {
      var n = t.ids[D];
      var u = n > -1 ? t.data[n] : t.oprc;
      var f = D % this.colors.length;
      v = e != null ? e : (Math.PI * (2 * G)) - o;
      if (v > (Math.PI * 2)) {
        v = v % (Math.PI * 2)
      }
      e = v + (Math.PI * (2 * (u)));
      if (e > (Math.PI * 2)) {
        e = e % (Math.PI * 2)
      }
      var H = (v + e) / 2;
      var B = (e - v) / 32;
      var F = this.transparency != null ? this.validateColor(this.colors[f], this.transparency) : this.colors[f];
      if (this.gradient) {
        this.setRadialGradient(h, g, q * 1.3, F)
      } else {
        this.ctx.fillStyle = F
      }
      this.ctx.beginPath();
      this.ctx.lineWidth = this.pieSegmentSeparation;
      this.ctx.strokeStyle = this.pieType == "separated" ? this.background : this.pieType == "solid" ? this.foreground : this.foreground;
      if (this.showShadow) {
        this.ctx.shadowOffsetX = this.shadowOffsetX;
        this.ctx.shadowOffsetY = this.shadowOffsetY;
        this.ctx.shadowBlur = this.shadowBlur;
        this.ctx.shadowColor = this.shadowColor
      }
      this.ctx.moveTo(h, g);
      this.ctx.arc(h, g, q, v, e, false);
      this.ctx.lineTo(h, g);
      this.ctx.closePath();
      this.ctx.fill();
      if (this.graphType != "Network") {
        this.ctx.stroke()
      }
      if (this.showShadow) {
        this.ctx.restore();
        this.ctx.shadowOffsetX = 0;
        this.ctx.shadowOffsetY = 0;
        this.ctx.shadowBlur = 0;
        this.ctx.shadowColor = this.background
      }
      var E = h - Math.sin(H - (Math.PI / 2)) * z;
      var d = g + Math.cos(H - (Math.PI / 2)) * z;
      if (this.graphType != "Network" && this.showPieValues) {
        this.drawText(sprintf("%." + this.pieSegmentPrecision + "f", u * 100) + "%", E, d, this.axisTickFont, this.axisTickColor, "center", "middle")
      }
      if (I) {
        E = h + q * Math.cos(v);
        d = g + q * Math.sin(v);
        var b = ["poly", h, g, E, d];
        var o = v;
        for (var A = 0; A < 32; A++) {
          o += B;
          E = h + q * Math.cos(o);
          d = g + q * Math.sin(o);
          b.push(E);
          b.push(d)
        }
        b.push(h);
        b.push(g);
        if (n > -1) {
          this.addArea(b, [n, C])
        } else {
          t.idso.push(C);
          this.addArea(b, t.idso)
        }
      }
      G += u
    }
    this.ctx.lineWidth = m
  };
  this.drawBarPlot = function (K, J, F, x, k, C, p, g, E, D, L, N) {
    var n, j, I, B;
    var M = this.backgroundType.match(/window/i) ? this.foregroundWindow : this.foreground;
    var A = this.seriesSeparationFactor / 2;
    var H = k / 2;
    if (D == "vertical") {
      for (var G = 0; G < K.length; G++) {
        n = ((K[G] - E) * p) - g;
        if (!isNaN(n)) {
          this.addArea(this.rectangle(F + A, x, k - this.seriesSeparationFactor, -n, L[G], L[G], "closed", false, false, true), N[G]);
          I = x - n;
          B = F + H;
          if (J[G] && !isNaN(J[G])) {
            j = (((J[G] - E) / 2) * p) - (g / 2);
            this.errorBar(B, I, B, I + j, -H, M);
            this.errorBar(B, I, B, I - j, -H, M)
          }
          if (this.showDataValues) {
            this.drawText(K[G], B, x - (j + this.overlayFontSize), this.overlayFont, L[G], "center", "bottom")
          }
        }
        F += C
      }
    } else {
      for (var G = 0; G < K.length; G++) {
        n = ((K[G] - E) * p) - g;
        if (!isNaN(n)) {
          this.addArea(this.rectangle(F, x + A, n, k - this.seriesSeparationFactor, L[G], L[G], "closed", false, false, true), N[G]);
          I = F + n;
          B = x + H;
          if (J[G] && !isNaN(J[G])) {
            j = (((J[G] - E) / 2) * p) - (g / 2);
            this.errorBar(I, B, I + j, B, H, M);
            this.errorBar(I, B, I - j, B, H, M)
          }
          if (this.showDataValues) {
            this.drawText(K[G], K + j + this.overlayFontSize, B, this.overlayFont, L[G], "left", "middle")
          }
        }
        x += C
      }
    }
  };
  this.drawLinePlot = function (M, L, F, y, v, H, x, E, C, N, A, P) {
    var B, K, J, I, j, g, f;
    var r = v / 2;
    var q = v / 3;
    var O = this.backgroundType.match(/window/i) ? this.foregroundWindow : this.foreground;
    var D = M.length - 1;
    var k = [
      [],
      []
    ];
    if (C == "vertical") {
      K = F;
      J = F + H;
      for (var G = 1; G < M.length; G++) {
        B = G - 1;
        j = y - ((M[B] - E) * x);
        g = y - ((M[G] - E) * x);
        if (!isNaN(j) && !isNaN(g)) {
          if (this.lineType == "spline") {
            k[0].push(K);
            k[1].push(j)
          } else {
            this.drawLine("line", K, j, J, g, N, this.lineThickness, "butt", false, false, true)
          }
          if (this.lineDecoration && A) {
            this.addArea(this.drawShape(A, K, j, q, q, N, O, "closed", false, false, true), P[B])
          }
          if (L[B] && !isNaN(L[B])) {
            f = ((L[B] - E) / 2) * x;
            this.errorBar(K, j, K, j + f, -r, N);
            this.errorBar(K, j, K, j - f, -r, N)
          }
        }
        K += H;
        J += H
      }
      if (!isNaN(g)) {
        if (this.lineDecoration && A) {
          this.addArea(this.drawShape(A, K, g, q, q, N, O, "closed", false, false, true), P[D])
        }
        if (L[D] && !isNaN(L[D])) {
          f = ((L[D] - E) / 2) * x;
          this.errorBar(K, g, K, g + f, -r, N);
          this.errorBar(K, g, K, g - f, -r, N)
        }
        if (this.lineType == "spline") {
          k[0].push(K);
          k[1].push(g);
          this.drawLine("spline", k[0], k[1], false, false, N, this.lineThickness)
        }
      }
    } else {
      j = y;
      g = y + H;
      for (var G = 1; G < M.length; G++) {
        B = G - 1;
        K = F + ((M[B] - E) * x);
        J = F + ((M[G] - E) * x);
        if (!isNaN(K) && !isNaN(J)) {
          if (this.lineType == "spline") {
            k[0].push(K);
            k[1].push(j)
          } else {
            this.drawLine("line", J, g, K, j, N, this.lineThickness, "butt", false, false, true)
          }
          if (this.lineDecoration && A) {
            this.addArea(this.drawShape(A, K, j, q, q, N, O, "closed", false, false, true), P[B])
          }
          if (L[B] && !isNaN(L[B])) {
            I = ((L[B] - E) / 2) * x;
            this.errorBar(K, j, K + I, j, r, N);
            this.errorBar(K, j, K - I, j, r, N)
          }
        }
        j += H;
        g += H
      }
      if (!isNaN(J)) {
        if (this.lineDecoration && A) {
          this.addArea(this.drawShape(A, J, j, q, q, N, O, "closed", false, false, true), P[D])
        }
        if (L[D] && !isNaN(L[D])) {
          I = ((L[D] - E) / 2) * x;
          this.errorBar(J, j, J + I, j, -r, N);
          this.errorBar(J, j, J - I, j, -r, N)
        }
        if (this.lineType == "spline") {
          k[0].push(J);
          k[1].push(j);
          this.drawLine("spline", k[0], k[1], false, false, N, this.lineThickness)
        }
      }
    }
  };
  this.drawStackedPlot = function (k, f, x, q, b, j, s, e, m, n) {
    if (e == "vertical") {
      for (var h = 0; h < k.length; h++) {
        var r = b ? (k[h] * 100 / j) * s : k[h] * s;
        this.addArea(this.rectangle(f, x, q, -r, m[h]), n[h]);
        x -= r
      }
    } else {
      for (var h = 0; h < k.length; h++) {
        var r = b ? (k[h] * 100 / j) * s : k[h] * s;
        this.addArea(this.rectangle(f, x, r, q, m[h]), n[h]);
        f += r
      }
    }
  };
  this.drawAreaPlot = function (T, M, E, B, Q, F, D, L, J, S, U, V) {
    var I = S ? "open" : "close";
    var K = T.length - 1;
    var N = S && this.lineType == "rect" ? "path" : this.lineType == "rect" ? "polygon" : "spline";
    var R = [];
    var G = [];
    if (this.graphOrientation == "vertical") {
      for (var P = 0; P < T[0].length; P++) {
        R[P] = E
      }
      for (var P = 0; P < T.length; P++) {
        var H = M;
        var A = S ? [] : [M];
        var e = S ? [] : [E];
        for (var O = 0; O < T[0].length; O++) {
          var C = !isNaN(T[P][O]) ? (T[P][O] - L) * D : 0;
          A.push(H);
          if (this.areaIsCumulative) {
            e.push(R[O] - C);
            R[O] -= C
          } else {
            e.push(E - C)
          }
          H += F
        }
        if (!S) {
          A.push(M + B);
          e.push(E)
        }
        G.push([A, e])
      }
    } else {
      for (var P = 0; P < T[0].length; P++) {
        R[P] = M
      }
      for (var P = 0; P < T.length; P++) {
        var H = E;
        var A = S ? [] : [M];
        var e = S ? [] : [E];
        for (var O = 0; O < T[0].length; O++) {
          var C = !isNaN(T[P][O]) ? (T[P][O] - L) * D : 0;
          e.push(H);
          if (this.areaIsCumulative) {
            A.push(R[O] + C);
            R[O] += C
          } else {
            A.push(M + C)
          }
          H += F
        }
        if (!S) {
          A.push(M);
          e.push(E + Q)
        }
        G.push([A, e])
      }
    }
    while (K >= 0) {
      var b = G.pop();
      if (S && this.lineType == "spline") {
        this.addArea(this.drawLine(N, b[0], b[1], false, false, U[K], this.lineThickness), V[K])
      } else {
        this.addArea(this.drawShape(N, b[0], b[1], false, false, U[K], U[K], I), V[K])
      }
      K--
    }
  };
  this.drawBoxPlot = function (G, C, n, f, y, k, B, A, H, J) {
    var e = function (a) {
        return A == "vertical" ? n - ((a - B) * k) : C + ((a - B) * k)
      };
    var I = this.backgroundType.match(/window/i) ? this.foregroundWindow : this.foreground;
    var p = this.seriesSeparationFactor / 2;
    var F = f / 2;
    if (A == "vertical") {
      for (var E = 0; E < G.length; E++) {
        var x = C + F;
        var g = [];
        for (var D = 0; D < 5; D++) {
          g.push(e(G[E][D]))
        }
        this.addArea(this.rectangle(C + p, g[2], f - this.seriesSeparationFactor, g[3] - g[2], H, H, "closed", false, false, true), J[E]);
        this.drawLine("line", C + p, g[4], C + (f - p), g[4], I, false, "butt", false, false, true);
        this.errorBar(x, g[2], x, g[0], -F, I);
        this.errorBar(x, g[3], x, g[1], -F, I);
        if (G[E][5]) {
          for (var D = 0; D < G[E][5].length; D++) {
            this.drawShape("sphere", x, e(G[E][5][D]), F, F, H, I, "closed", false, false, true)
          }
        }
        C += y
      }
    } else {
      for (var E = 0; E < G.length; E++) {
        var x = n + F;
        var g = [];
        for (var D = 0; D < 5; D++) {
          g.push(e(G[E][D]))
        }
        this.addArea(this.rectangle(g[2], n + p, g[3] - g[2], f - this.seriesSeparationFactor, H, H, "closed", false, false, true), J[E]);
        this.drawLine("line", g[4], n + p, g[4], n + (f - p), I, false, "butt", false, false, true);
        this.errorBar(g[2], x, g[0], x, F, I);
        this.errorBar(g[3], x, g[1], x, F, I);
        if (G[E][5]) {
          for (var D = 0; D < G[E][5].length; D++) {
            this.drawShape("sphere", e(G[E][5][D]), x, F, F, H, I, "closed", false, false, true)
          }
        }
        n += y
      }
    }
  };
  this.drawCandlestickPlot = function (G, B, q, f, F, n, A, x, y, H, J) {
    var e = function (a) {
        return y == "vertical" ? q - ((a - A) * n) : B + ((a - A) * n)
      };
    var I = this.backgroundType.match(/window/i) ? this.foregroundWindow : this.foreground;
    var E = f / 2;
    if (y == "vertical") {
      for (var D = 0; D < G.length; D++) {
        var s = B + E;
        var k = [];
        for (var C = 0; C < G[D].length; C++) {
          k.push(e(G[D][C]))
        }
        if (x) {
          if (k[0] > k[1]) {
            this.addArea(this.rectangle(B, k[0], f, k[1] - k[0], false, H, "open", false, false, true), J[D]);
            this.drawLine("line", s, k[1], s, k[2], H, false, "butt", false, false, true);
            this.drawLine("line", s, k[3], s, k[0], H, false, "butt", false, false, true)
          } else {
            this.addArea(this.rectangle(B, k[0], f, k[1] - k[0], H, H, "closed", false, false, true), J[D]);
            this.drawLine("line", s, k[3], s, k[1], H, false, "butt", false, false, true);
            this.drawLine("line", s, k[0], s, k[2], H, false, "butt", false, false, true)
          }
        } else {
          this.addArea(this.rectangle(B, k[0], f, q - k[0], H, H, "closed", false, false, true), J[D])
        }
        B += F
      }
    } else {
      for (var D = 0; D < G.length; D++) {
        var s = q + E;
        var k = [];
        for (var C = 0; C < G[D].length; C++) {
          k.push(e(G[D][C]))
        }
        if (x) {
          if (k[1] > k[0]) {
            this.addArea(this.rectangle(k[0], q, k[1] - k[0], f, false, H, "open", false, false, true), J[D]);
            this.drawLine("line", k[1], s, k[2], s, H, false, "butt", false, false, true);
            this.drawLine("line", k[3], s, k[0], s, H, false, "butt", false, false, true)
          } else {
            this.addArea(this.rectangle(k[0], q, k[1] - k[0], f, H, H, "closed", false, false, true), J[D]);
            this.drawLine("line", k[3], s, k[1], s, H, false, "butt", false, false, true);
            this.drawLine("line", k[0], s, k[2], s, H, false, "butt", false, false, true)
          }
        } else {
          this.addArea(this.rectangle(B, q, k[0] - B, f, H, H, "closed", false, false, true), J[D])
        }
        q += F
      }
    }
  };
  this.drawDotPlot = function (H, G, C, r, g, F, y, n, B, x, A, I, z, K) {
    var k, f;
    var J = this.backgroundType.match(/window/i) ? this.foregroundWindow : this.foreground;
    if (A == "vertical") {
      for (var E = 0; E < H.length; E++) {
        for (var D = 0; D < H[E].length; D++) {
          k = r - ((H[E][D] - B) * n);
          if (!isNaN(k)) {
            this.addArea(this.drawShape(z, C, k, x, x, I, J, "closed", false, false, true), K[E]);
            if (G[E] && !isNaN(G[E])) {
              f = ((G[E] - B) / 2) * n;
              this.errorBar(C, k, C, k + f, -F, I);
              this.errorBar(C, k, C, k - f, -F, I)
            }
          }
        }
        C += y
      }
    } else {
      for (var E = 0; E < H.length; E++) {
        for (var D = 0; D < H[E].length; D++) {
          k = C + ((H[E][D] - B) * n);
          if (!isNaN(k)) {
            this.addArea(this.drawShape(z, k, r, x, x, I, J, "closed", false, false, true), K[E]);
            if (G[E] && !isNaN(G[E])) {
              f = ((G[E] - B) / 2) * n;
              this.errorBar(k, r, k + f, r, F, I);
              this.errorBar(k, r, k - f, r, F, I)
            }
          }
        }
        r += y
      }
    }
  };
  this.drawHeatmap = function (u, k, F, D, q, j, C, s, B, r) {
    var g = this.backgroundType.match(/window/i) ? this.foregroundWindow : this.foreground;
    var A = "rgb(255,255, 0)";
    var b = this.graphType == "Network" ? false : true;
    for (var m = 0; m < u.length; m++) {
      var E = u[m];
      var z = !isNaN(E) ? this.getHeatmapColor(j, C, E) : A;
      g = s ? z : g;
      this.addArea(this.rectangleHM(k, F, D, q, z, g, "closed", false, false, b), B[m], r);
      if (this.showDataValues) {
        this.drawText(E, (k + D) - 2, (F + q) - 2, this.overlayFont, this.foreground, "right", "bottom")
      }
      k += D
    }
  }
};
CanvasXpress.prototype.init3dUtils = function () {
  this.set3DRotation = function () {
    var a;
    var c = 0;
    var b = 0;
    if (this.graphType == "Network") {
      c = this.maxRotate != null ? this.maxRotate : 360;
      b = this.minRotate != null ? this.minRotate : 360
    } else {
      if (this.graphType == "Scatter3D") {
        c = this.maxRotate != null ? this.maxRotate : 90;
        b = this.minRotate != null ? this.minRotate : 0
      }
    }
    if (this.xRotate > c) {
      this.xRotate = c
    }
    if (this.xRotate < 0) {
      this.xRotate = b
    }
    if (this.yRotate > c) {
      this.yRotate = c
    }
    if (this.yRotate < 0) {
      this.yRotate = b
    }
    if (this.zRotate > c) {
      this.zRotate = c
    }
    if (this.zRotate < 0) {
      this.zRotate = b
    }
    if (this.xRotate > 0) {
      a = 180 / this.xRotate;
      this.ry = Math.PI / a
    } else {
      this.ry = 0
    }
    if (this.yRotate > 0) {
      a = 180 / this.yRotate;
      this.rx = Math.PI / a
    } else {
      this.rx = 0
    }
    if (this.zRotate > 0) {
      a = 180 / this.zRotate;
      this.rz = Math.PI / a
    } else {
      this.rz = 0
    }
  };
  this.set3DParams = function () {
    this.perspective = this.x * 1.5;
    this.len = this.x / 4 * this.zoom;
    this.pad = this.x / 2
  };
  this.get3DTransfrom = function (e, f, g) {
    var d, c, b;
    var a = [];
    c = f;
    b = g;
    f = c * Math.cos(this.rx) - b * Math.sin(this.rx);
    g = c * Math.sin(this.rx) + b * Math.cos(this.rx);
    d = e;
    b = g;
    e = b * Math.sin(this.ry) + d * Math.cos(this.ry);
    g = b * Math.cos(this.ry) - d * Math.sin(this.ry);
    d = e;
    c = f;
    e = d * Math.cos(this.rz) - c * Math.sin(this.rz);
    f = d * Math.sin(this.rz) + c * Math.cos(this.rz);
    e = e * (this.perspective / (g + this.perspective));
    f = f * (this.perspective / (g + this.perspective));
    g = g * (this.perspective / (g + this.perspective));
    a.push(e + this.pad);
    a.push(f + this.pad);
    a.push(g + this.pad);
    return a
  };
  this.backfaceCulling = function (s) {
    var y = [];
    var w = [];
    var u = [];
    var r = [];
    var q = [];
    var p = [];
    for (var o = 0; o < s.length; o++) {
      var k = s[o][1][0] - s[o][0][0];
      var x = s[o][1][1] - s[o][0][1];
      var n = s[o][1][2] - s[o][0][2];
      var j = s[o][2][0] - s[o][0][0];
      var v = s[o][2][1] - s[o][0][1];
      var m = s[o][2][2] - s[o][0][2];
      var h = s[o][3][0] - s[o][0][0];
      var t = s[o][3][1] - s[o][0][1];
      var l = s[o][3][2] - s[o][0][2];
      y[o] = x * m - v * n;
      w[o] = j * n - k * m;
      u[o] = k * v - j * x;
      r[o] = t * m - v * l;
      q[o] = j * l - h * m;
      p[o] = h * v - j * t
    }
    return [y, w, u, r, q, p]
  }
};
CanvasXpress.prototype.initLegendUtils = function () {
  this.drawDecorationData = function () {
    var T = this.marginLeft + this.offsetX + this.left;
    var K = this.marginTop + this.offsetY + this.top;
    var aa = this.data.d;
    if (this.showDecorations && aa) {
      if (aa.nlfit) {
        var X, w, Z;
        var h = this.smpIndices;
        var s, a;
        for (var W = 0; W < aa.nlfit.length; W++) {
          var E = [];
          var D = [];
          var m = W % this.colors.length;
          var R = aa.nlfit[W].param[0];
          var P = aa.nlfit[W].param[1];
          var N = aa.nlfit[W].param[2];
          var L = aa.nlfit[W].param[3];
          var F = this.xAxisIndices[W];
          if (this.xAxisTransform == "percentile") {
            X = this.getAxisRangeBySample(F);
            Z = X[0];
            w = X[1]
          }
          var G = this.data.y.data[F];
          h.sort(function (i, d) {
            return G[i] - G[d]
          });
          m = aa.nlfit[W].color ? aa.nlfit[W].color : this.colors[m];
          for (var V = 0; V < h.length; V++) {
            var o = h[V];
            var J = this.getDataAtPos(F, o, false, this.xAxisTransform, Z, w);
            var I;
            var p = [];
            if (J > 0) {
              if (V > 0) {
                var b = h[V - 1];
                var H = this.getDataAtPos(F, b, false, this.xAxisTransform, Z, w);
                if (H > 0) {
                  var c = (J - H) / this.functionIntervals;
                  for (var U = 0; U < this.functionIntervals; U++) {
                    p.push(H);
                    H += c
                  }
                }
              } else {
                p = [J]
              }
              for (var U = 0; U < p.length; U++) {
                if (aa.nlfit[W].type == "reg") {
                  I = R + ((P - R) / (1 + N / (Math.pow(p[U], L))))
                } else {
                  I = R + ((P - R) / (1 + (Math.pow((p[U] / N), L))))
                }
                s = T + ((p[U] - this.xAxisMin) * this.xAxisUnit);
                a = (K + this.y) - ((I - this.yAxisMin) * this.yAxisUnit);
                E.push(s);
                D.push(a)
              }
            }
          }
          if (E.length > 0) {
            this.drawShape("path", E, D, false, false, m, m, "open");
            if (aa.nlfit[W].type == "reg") {
              I = R + ((P - R) / (1 + N / (Math.pow(N, L))))
            } else {
              I = R + ((P - R) / (1 + (Math.pow((N / N), L))))
            }
            s = T + ((N - this.xAxisMin) * this.xAxisUnit);
            a = (K + this.y) - ((I - this.yAxisMin) * this.yAxisUnit);
            this.drawLine("dashedLine", T, a, s, a, m);
            this.drawLine("dashedLine", s, a, s, (K + this.y), m)
          }
        }
      }
      if (aa.line) {
        for (var W = 0; W < aa.line.length; W++) {
          if (aa.line[W].x || aa.line[W].y) {
            var m = aa.line[W].color ? aa.line[W].color : this.colors[W % this.colors.length];
            var ac = aa.line[W].type ? aa.line[W].type : "line";
            var J = aa.line[W].x ? aa.line[W].x : false;
            var I = aa.line[W].y ? aa.line[W].y : false;
            var s, a, q, ad;
            if (J) {
              s = T + (J * this.xAxisUnit) - (this.xAxisMin * this.xAxisUnit);
              a = K;
              q = s;
              ad = a + this.y;
              this.drawLine(ac, s, a, q, ad, m)
            }
            if (I) {
              s = T;
              a = K + (I * this.yAxisUnit) + (this.yAxisMin * this.yAxisUnit);
              q = s + this.x;
              ad = a;
              this.drawLine(ac, s, a, q, ad, m)
            }
          }
        }
      }
      if (aa.reg) {
        for (var W = 0; W < aa.reg.length; W++) {
          if (aa.reg[W].slope && aa.reg[W].intercept) {
            var m = W % this.colors.length;
            var ac = aa.reg[W].type ? aa.reg[W].type : "line";
            var v = aa.reg[W].slope;
            var Q = aa.reg[W].intercept;
            var f = aa.reg[W].cor;
            var s = aa.reg[W].start ? aa.reg[W].start : this.xAxisMin;
            var a = (v * s) + Q;
            var q = aa.reg[W].end ? aa.reg[W].end : this.xAxisMax;
            var ad = (v * q) + Q;
            s = T + (s * this.xAxisUnit) - (this.xAxisMin * this.xAxisUnit);
            a = (K + this.y) - (a * this.yAxisUnit) + (this.yAxisMin * this.yAxisUnit);
            q = T + (q * this.xAxisUnit) - (this.xAxisMin * this.xAxisUnit);
            ad = (K + this.y) - (ad * this.yAxisUnit) + (this.yAxisMin * this.yAxisUnit);
            m = aa.reg[W].color ? aa.reg[W].color : this.colors[m];
            this.drawLine(ac, s, a, q, ad, m)
          }
        }
      }
      if (aa.nor) {
        for (var W = 0; W < aa.nor.length; W++) {
          if (aa.nor[W].mu && aa.nor[W].sigma) {
            var S = [];
            var O = [];
            var C = [];
            var s, a;
            var m = W % this.colors.length;
            var n = aa.nor[W].mu;
            var e = aa.nor[W].sigma;
            var A = aa.nor[W].start ? aa.nor[W].start : aa.nor[W].yAxis ? this.yAxisMin : this.xAxisMin;
            var u = aa.nor[W].end ? aa.nor[W].end : aa.nor[W].yAxis ? this.yAxisMax : this.xAxisMax;
            var g = 120;
            if (A < n && n < u) {
              var H, r, z, Y;
              var c = (n - A) / g;
              var ab = e * e;
              var M = ab * 2;
              var B = 1 / Math.sqrt(2 * Math.PI * ab);
              if (aa.nor[W].yAxis) {
                z = aa.nor[W].max ? aa.nor[W].max * this.xAxisUnit : this.x - (this.x * this.axisExtension);
                Y = z / B;
                H = A;
                for (var V = 0; V < g - 1; V++) {
                  r = (B * Math.pow(Math.E, -Math.pow(H - n, 2) / ab));
                  C.push([H, r]);
                  s = T + (r * Y);
                  a = (K + this.y) - (H * this.yAxisUnit) + (this.yAxisMin * this.yAxisUnit);
                  S.push(s);
                  O.push(a);
                  H += c
                }
                c = (u - n) / g;
                H = n;
                for (var V = 0; V < g; V++) {
                  r = (B * Math.pow(Math.E, -Math.pow(H - n, 2) / ab));
                  C.push([H, r]);
                  s = T + (r * Y);
                  a = (K + this.y) - (H * this.yAxisUnit) + (this.yAxisMin * this.yAxisUnit);
                  S.push(s);
                  O.push(a);
                  H += c
                }
                m = aa.nor[W].color ? aa.nor[W].color : this.colors[m];
                this.drawShape("path", S, O, false, false, m, m, "open")
              } else {
                z = aa.nor[W].max ? aa.nor[W].max * this.yAxisUnit : this.y - (this.y * this.axisExtension);
                Y = z / B;
                H = A;
                for (var V = 0; V < g - 1; V++) {
                  r = (B * Math.pow(Math.E, -Math.pow(H - n, 2) / ab));
                  C.push([H, r]);
                  s = T + (H * this.xAxisUnit) - (this.xAxisMin * this.xAxisUnit);
                  a = (K + this.y) - (r * Y);
                  S.push(s);
                  O.push(a);
                  H += c
                }
                c = (u - n) / g;
                H = n;
                for (var V = 0; V < g; V++) {
                  r = (B * Math.pow(Math.E, -Math.pow(H - n, 2) / ab));
                  C.push([H, r]);
                  s = T + (H * this.xAxisUnit) - (this.xAxisMin * this.xAxisUnit);
                  a = (K + this.y) - (r * Y);
                  S.push(s);
                  O.push(a);
                  H += c
                }
                m = aa.nor[W].color ? aa.nor[W].color : this.colors[m];
                this.drawShape("path", S, O, false, false, m, m, "open")
              }
            }
          }
        }
      }
      if (aa.area) {
        for (var W = 0; W < aa.area.length; W++) {
          if (aa.area[W].type && aa.area[W].x && aa.area[W].y && aa.area[W].size) {
            if (this.isValidShape(aa.area[W].type)) {
              var s = T + (aa.area[W].x * this.xAxisUnit) - (this.xAxisMin * this.xAxisUnit);
              var a = (K + this.y) - (aa.area[W].y * this.yAxisUnit) + (this.yAxisMin * this.yAxisUnit);
              var m = aa.area[W].color ? aa.area[W].color : this.colors[W % this.colors.length];
              this[aa.area[W].type](s, a, m, aa.area[W].size)
            }
          }
        }
      }
    }
  };
  this.setDecorationLegendDimension = function () {
    this.legendDecorationWidth = 0;
    this.legendDecorationHeight = 0;
    var f = this.data.d;
    if (this.showDecorations && f) {
      var k = this.getFontPt(this.decorationFont);
      var j = 0;
      var g = 0;
      var j = 0;
      var b = 0;
      if (f.nlfit) {
        var a = this.measureText(sprintf("%75s", "X"), this.decorationFont) + (this.margin * 4);
        var h = 0;
        for (var e = 0; e < f.nlfit.length; e++) {
          var c = this.measureText(f.nlfit[e].label, this.decorationFont);
          h = Math.max(c, h)
        }
        j = this.margin + h + this.margin + a + this.margin;
        this.legendDecorationHeight = (f.nlfit.length * k * 2) + k + this.margin
      }
      if (f.line) {
        var h = 0;
        for (var e = 0; e < f.line.length; e++) {
          var c = this.measureText(f.line[e].label, this.decorationFont);
          h = Math.max(c, h)
        }
        b = this.margin + h + this.margin;
        this.legendDecorationHeight += (f.line.length * k * 2) + this.margin
      }
      if (f.reg) {
        var a = this.measureText(sprintf("%45s", "X"), this.decorationFont) + (this.margin * 2);
        var h = 0;
        for (var e = 0; e < f.reg.length; e++) {
          var c = this.measureText(f.reg[e].label, this.decorationFont);
          h = Math.max(c, h)
        }
        g = this.margin + h + this.margin + a + this.margin;
        this.legendDecorationHeight += (f.reg.length * k * 2) + k + this.margin
      }
      if (f.nor) {
        var a = this.measureText(sprintf("%30s", "X"), this.decorationFont) + (this.margin * 1);
        var h = 0;
        for (var e = 0; e < f.nor.length; e++) {
          var c = this.measureText(f.nor[e].label, this.decorationFont);
          h = Math.max(c, h)
        }
        j = this.margin + h + this.margin + a + this.margin;
        this.legendDecorationHeight += (f.nor.length * k * 2) + k + this.margin
      }
      if (f.area) {
        var h = 0;
        for (var e = 0; e < f.area.length; e++) {
          var c = this.measureText(f.area[e].label, this.decorationFont);
          h = Math.max(c, h)
        }
        b = this.margin + h + this.margin;
        this.legendDecorationHeight += (f.area.length * k * 2) + this.margin
      }
      this.legendDecorationWidth = Math.max(Math.max(Math.max(j, g), j), b)
    }
  };
  this.drawDecorationLegend = function () {
    var e = this.marginLeft + this.offsetX + this.left;
    var p = this.marginTop + this.offsetY + this.top;
    var k = this.data.d;
    if (this.showDecorations && k) {
      var q = this.getFontPt(this.decorationFont);
      var c = this.measureText(sprintf("%15s", "X"), this.decorationFont);
      var a, o, m, n;
      if (this.decorationsPosition == "right") {
        o = p + (q / 2) + ((this.y - this.legendDecorationHeight) / 2)
      } else {
        o = (p + this.y + this.bottom) - this.legendDecorationHeight
      }
      if (k.nlfit) {
        n = (this.margin * 5) + (5 * c);
        if (this.decorationsPosition == "right") {
          m = (e + this.x + this.right + c) - n
        } else {
          m = e + ((this.x - this.legendDecorationWidth) / 2) + (this.legendDecorationWidth - n) + c
        }
        a = m;
        this.drawText("Min", a, o, this.decorationFont, this.decorationsColor, "right", "middle");
        a += c + this.margin;
        this.drawText("Max", a, o, this.decorationFont, this.decorationsColor, "right", "middle");
        a += c + this.margin;
        this.drawText("EC50", a, o, this.decorationFont, this.decorationsColor, "right", "middle");
        a += c + this.margin;
        this.drawText("Slope", a, o, this.decorationFont, this.decorationsColor, "right", "middle");
        a += c + this.margin;
        this.drawText("p-Val", a, o, this.decorationFont, this.decorationsColor, "right", "middle");
        o += q + q;
        for (var h = 0; h < k.nlfit.length; h++) {
          var f = h % this.colors.length;
          f = k.nlfit[h].color ? k.nlfit[h].color : this.colors[f];
          a = (m + this.margin) - c;
          if (this.graphType == "ScatterBubble2D" && this.decorationsPosition == "right" && this.zAxisShow) {
            a += this.margin + this.getFontPt(this.axisTitleFont) + this.margin
          }
          this.drawText(k.nlfit[h].label, a, o, this.decorationFont, f, "right", "middle");
          a = m;
          for (var g = 0; g < 5; g++) {
            var b = k.nlfit[h].param[g];
            this.drawText(this.formatNumber(b), a, o, this.decorationFont, this.decorationsColor, "right", "middle");
            a += c + this.margin
          }
          o += q + q
        }
      }
      if (k.line) {
        for (var h = 0; h < k.line.length; h++) {
          if (k.line[h].label) {
            if (this.decorationsPosition == "right") {
              a = e + this.x + this.margin
            } else {
              a = e + ((this.x - this.legendDecorationWidth) / 2) + this.margin
            }
            for (var h = 0; h < k.line.length; h++) {
              var f = k.line[h].color ? k.line[h].color : this.colors[h % this.colors.length];
              if (this.graphType == "ScatterBubble2D" && this.decorationsPosition == "right" && this.zAxisShow) {
                a += this.margin + this.getFontPt(this.axisTitleFont) + this.margin
              }
              this.drawText(k.line[h].label, a, o, this.decorationFont, f, "left", "middle");
              o += q + q
            }
          }
        }
      }
      if (k.reg) {
        n = (this.margin * 3) + (3 * c);
        if (this.decorationsPosition == "right") {
          m = (e + this.x + this.right) - n
        } else {
          m = e + ((this.x - this.legendDecorationWidth) / 2) + (this.legendDecorationWidth - n) + c
        }
        a = m;
        this.drawText("Slope", a, o, this.decorationFont, this.decorationsColor, "right", "middle");
        a += c + this.margin;
        this.drawText("Int", a, o, this.decorationFont, this.decorationsColor, "right", "middle");
        a += c + this.margin;
        this.drawText("Cor", a, o, this.decorationFont, this.decorationsColor, "right", "middle");
        o += q + q;
        for (var h = 0; h < k.reg.length; h++) {
          var f = h % this.colors.length;
          f = k.reg[h].color ? k.reg[h].color : this.colors[f];
          a = (m + this.margin) - c;
          if (this.graphType == "ScatterBubble2D" && this.decorationsPosition == "right" && this.zAxisShow) {
            a += this.margin + this.getFontPt(this.axisTitleFont) + this.margin
          }
          this.drawText(k.reg[h].label, a, o, this.decorationFont, f, "right", "middle");
          a = m;
          this.drawText(this.formatNumber(k.reg[h].slope), a, o, this.decorationFont, this.decorationsColor, "right", "middle");
          a += c + this.margin;
          this.drawText(this.formatNumber(k.reg[h].intercept), a, o, this.decorationFont, this.decorationsColor, "right", "middle");
          a += c + this.margin;
          this.drawText(this.formatNumber(k.reg[h].cor), a, o, this.decorationFont, this.decorationsColor, "right", "middle");
          a += c + this.margin;
          o += q + q
        }
      }
      if (k.nor) {
        n = (this.margin * 2) + (2 * c);
        if (this.decorationsPosition == "right") {
          m = (e + this.x + this.right) - n
        } else {
          m = e + ((this.x - this.legendDecorationWidth) / 2) + (this.legendDecorationWidth - n) + c
        }
        a = m;
        this.drawText("Mu", a, o, this.decorationFont, this.decorationsColor, "right", "middle");
        a += c + this.margin;
        this.drawText("Sigma", a, o, this.decorationFont, this.decorationsColor, "right", "middle");
        o += q + q;
        for (var h = 0; h < k.nor.length; h++) {
          var f = h % this.colors.length;
          f = k.nor[h].color ? k.nor[h].color : this.colors[f];
          a = (m + this.margin) - c;
          if (this.graphType == "ScatterBubble2D" && this.decorationsPosition == "right" && this.zAxisShow) {
            a += this.margin + this.getFontPt(this.axisTitleFont) + this.margin
          }
          this.drawText(k.nor[h].label, a, o, this.decorationFont, f, "right", "middle");
          a = m;
          this.drawText(this.formatNumber(k.nor[h].mu), a, o, this.decorationFont, this.decorationsColor, "right", "middle");
          a += c + this.margin;
          this.drawText(this.formatNumber(k.nor[h].sigma), a, o, this.decorationFont, this.decorationsColor, "right", "middle");
          a += c + this.margin;
          o += q + q
        }
      }
      if (k.area) {
        for (var h = 0; h < k.area.length; h++) {
          if (k.area[h].label) {
            if (this.decorationsPosition == "right") {
              a = e + this.x + this.margin
            } else {
              a = e + ((this.x - this.legendDecorationWidth) / 2) + this.margin
            }
            for (var h = 0; h < k.area.length; h++) {
              var f = k.area[h].color ? k.area[h].color : this.colors[h % this.colors.length];
              if (this.graphType == "ScatterBubble2D" && this.decorationsPosition == "right" && this.zAxisShow) {
                a += this.margin + this.getFontPt(this.axisTitleFont) + this.margin
              }
              this.drawText(k.area[h].label, a, o, this.decorationFont, f, "left", "middle");
              o += q + q
            }
          }
        }
      }
    }
  };
  this.setLegendFont = function () {
    if (this.autoScaleFont) {
      if (this.layoutComb) {
        this.legendFont = (parseInt(Math.max(this.minTextSize, this.legendFontSize - Math.min(this.layoutRows, this.layoutCols)) * this.legendScaleFontFactor))
      } else {
        this.legendFont = (parseInt(this.legendFontSize * this.legendScaleFontFactor))
      }
      this.legendFont = this.legendFont + "Pt " + this.fontName
    } else {
      this.legendFont = parseInt(this.legendFontSize) + "Pt " + this.fontName
    }
  };
  this.setSampleLegendDimension = function () {
    this.legendSampleWidth = 0;
    this.legendSampleHeight = 0;
    if (this.xAxisIndices.length != this.yAxisIndices.length || this.xAxisIndices.length > 1) {
      var d = this.getFontPt(this.legendFont);
      var b = 0;
      if (this.xAxisIndices.length == this.yAxisIndices.length && this.xAxisIndices.length > 1) {
        for (var c = 0; c < this.xAxisIndices.length; c++) {
          var a = this.measureText(this.shortenText(this.xAxis[c], this.maxSmpStringLen) + " vs " + this.shortenText(this.yAxis[c], this.maxSmpStringLen), this.legendFont);
          b = Math.max(a, b)
        }
      } else {
        if (this.xAxisIndices.length > this.yAxisIndices.length) {
          b = this.measureText(this.shortenText(this.getMaxText(this.xAxis), this.maxSmpStringLen) + " vs " + this.shortenText(this.yAxis[0], this.maxSmpStringLen), this.legendFont)
        } else {
          if (this.yAxisIndices.length > this.xAxisIndices.length) {
            b = this.measureText(this.shortenText(this.getMaxText(this.yAxis), this.maxSmpStringLen) + " vs " + this.shortenText(this.xAxis[0], this.maxSmpStringLen), this.legendFont)
          }
        }
      }
      this.legendSampleWidth = (d - 1) + b + (this.margin * 3);
      this.legendSampleHeight = (Math.max(this.xAxisIndices.length, this.yAxisIndices.length) * (d + this.margin)) + this.margin
    }
  };
  this.drawSampleLegend = function (b, j) {
    var h = b;
    var d = j;
    var o = this.getFontPt(this.legendFont);
    var n = (o - 1) / 2;
    j += this.margin + n;
    if (this.legendBox) {
      this.disableGradientTransparency();
      this.rectangle(h, d, this.legendSampleWidth, this.legendSampleHeight, this.legendBackgroundColor, this.legendBoxColor);
      this.enableGradientTransparency()
    }
    var l = Math.max(this.xAxisIndices.length, this.yAxisIndices.length);
    for (var f = 0; f < l; f++) {
      var c = f % this.colors.length;
      var m = 0;
      var g = this.xAxisIndices.length == this.yAxisIndices.length ? this.yAxis[f] : this.xAxisIndices.length > this.yAxisIndices.length ? this.yAxis[0] : this.xAxis[0];
      var k = this.shortenText(this.xAxisIndices.length < this.yAxisIndices.length ? this.yAxis[f] : this.xAxis[f], this.maxSmpStringLen) + " vs " + this.shortenText(g, this.maxSmpStringLen);
      var e = b + n + this.margin;
      var a = j + n + this.margin;
      if (this.colorBy) {
        this.drawShape(this.shapes[m], e, j, o - 1, o - 1, this.background, this.colors[c], "open")
      } else {
        this.drawShape(this.shapes[m], e, j, o - 1, o - 1, this.colors[c], this.foreground, "closed")
      }
      this.drawText(k, e + n + this.margin, j, this.legendFont, this.legendColor, "left", "middle");
      j += o + this.margin
    }
    if (this.legendBox) {
      this.rectangle(h, d, this.legendSampleWidth, this.legendSampleHeight, false, this.legendBoxColor, "open")
    }
    this.addArea(["rect", h, d, h + this.legendSampleWidth, d + this.legendSampleHeight], [-1], "-legend-sample")
  };
  this.setVariableLegendDimension = function () {
    this.legendVariableWidth = 0;
    this.legendVariableHeight = 0;
    if ((this.colorBy && this.colorBy == "variable") || (this.shapeBy && this.shapeBy == "variable") || (this.sizeBy && this.sizeBy == "variable")) {
      var a = this.sizeBy && this.sizeBy == "variable" ? Math.max(this.sizes[this.varIndices.length - 1], this.getFontPt(this.legendFont)) : this.getFontPt(this.legendFont);
      var b = this.shortenText(this.maxVarNameStr, this.maxVarStringLen);
      this.legendVariableWidth = (a - 1) + this.measureText(b, this.legendFont) + (this.margin * 3);
      this.legendVariableHeight = (this.varIndices.length * (a + this.margin)) + this.margin
    }
  };
  this.drawVariableLegend = function (c, l) {
    if ((this.colorBy && this.colorBy == "variable") || (this.shapeBy && this.shapeBy == "variable") || (this.sizeBy && this.sizeBy == "variable")) {
      var j = c;
      var e = l;
      var p = this.getFontPt(this.legendFont);
      var k = this.sizeBy && this.sizeBy == "variable" ? Math.max(this.sizes[this.varIndices.length - 1], p) : p;
      l += this.margin + (k / 2);
      if (this.legendBox && this.legendBackgroundColor) {
        this.disableGradientTransparency();
        this.rectangle(j, e, this.legendVariableWidth, this.legendVariableHeight, this.legendBackgroundColor, this.legendBoxColor);
        this.enableGradientTransparency()
      }
      for (var g = 0; g < this.varIndices.length; g++) {
        var m = this.varIndices[g];
        var d = this.colorBy && this.colorBy == "variable" ? g % this.colors.length : 0;
        var o = this.shapeBy && this.shapeBy == "variable" ? g % this.shapes.length : 0;
        var b = this.sizeBy && this.sizeBy == "variable" ? this.sizes[g % this.sizes.length] : k;
        var n = k / 2;
        var f = c + n + this.margin;
        var a = l + n + this.margin;
        var h = this.shortenText(this.data.y.vars[m], this.maxVarStringLen);
        if (this.colorBy && this.colorBy != "variable") {
          this.drawShape(this.shapes[o], f, l, b, b, this.background, this.foreground, "open")
        } else {
          this.drawShape(this.shapes[o], f, l, b, b, this.colors[d], this.foreground, "closed")
        }
        this.drawText(h, f + n + this.margin, l, this.legendFont, this.legendColor, "left", "middle");
        l += k + this.margin
      }
      if (this.legendBox) {
        this.rectangle(j, e, this.legendVariableWidth, this.legendVariableHeight, false, this.legendBoxColor, "open")
      }
      this.addArea(["rect", j, e, j + this.legendVariableWidth, e + this.legendVariableHeight], [-1], "-legend-variable")
    }
  };
  this.setColorLegendIndicatorDimensions = function () {
    var a = this.heatmapType.split("-");
    if (this.heatmapType.split("-").length > 1) {
      this.legendColorIndicatorWidth = parseInt(2 * 256 * this.indicatorWidth / this.indicatorBins) + (this.margin * 4) + 1
    } else {
      this.legendColorIndicatorWidth = parseInt(1 * 256 * this.indicatorWidth / this.indicatorBins) + (this.margin * 4) + 1
    }
    this.legendColorIndicatorHeight = this.indicatorHeight + (this.margin * 3) + this.getFontPt(this.legendFont);
    if (this.colorBy) {
      this.legendColorIndicatorHeight += this.getFontPt(this.legendFont) + this.margin
    }
  };
  this.getHeatmapColor = function (d, c, e) {
    var b = Math.abs(d);
    d += b;
    c += b;
    e += b;
    var a = (e * this.heatmapColors.length / c).toFixed() - 1;
    return this.heatmapColors[Math.max(0, Math.min(this.heatmapColors.length - 1, a))]
  };
  this.drawColorIndicator = function (b, m, e, k, g) {
    var j = b;
    var d = m;
    var a = this.indicatorHeight;
    var l = this.indicatorHeight;
    if (this.indicatorsPosition == "bottom") {
      for (var f = 0; f < this.heatmapColors.length; f++) {
        var h = this.transparency != null ? this.addColorTransparency(this.heatmapColors[f], this.transparency) : this.heatmapColors[f];
        this.drawLine("line", b, m, b, m + l, h, this.indicatorWidth + 0.5);
        b += this.indicatorWidth
      }
      b = j - (this.indicatorWidth / 2);
      m += l + this.margin;
      this.drawText(sprintf("%." + g + "f", e), b, m, this.legendFont, this.legendColor, "center", "top");
      b += (this.heatmapColors.length * this.indicatorWidth / 2) - (this.indicatorWidth / 2);
      if (this.heatmapColors.length >= 10) {
        this.drawText(sprintf("%." + g + "f", (e + k) / 2), b, m, this.legendFont, this.legendColor, "center", "top")
      }
      if (this.colorBy) {
        this.drawText(this.colorBy, b, m + this.margin + this.getFontPt(this.legendFont), this.legendFont, this.legendColor, "center", "top")
      }
      b += (this.heatmapColors.length * this.indicatorWidth / 2) - (this.indicatorWidth / 2);
      this.drawText(sprintf("%." + g + "f", k), b, m, this.legendFont, this.legendColor, "center", "top");
      this.addArea(["rect", j - (this.margin * 2), d - this.margin, b + (this.margin * 2), m + this.getFontPt(this.legendFont) + this.margin], [-1], "-legend-indicator-color")
    } else {
      for (var f = 0; f < this.heatmapColors.length; f++) {
        var h = this.transparency != null ? this.addColorTransparency(this.heatmapColors[f], this.transparency) : this.heatmapColors[f];
        this.drawLine("line", b, m, b + a, m, h, this.indicatorWidth + 0.5);
        m -= this.indicatorWidth
      }
      b += a + this.margin;
      m = d - (this.indicatorWidth / 2);
      this.drawText(sprintf("%." + g + "f", e), b, m, this.legendFont, this.legendColor, "center", "top", -Math.PI / 2);
      m -= (this.heatmapColors.length * this.indicatorWidth / 2) - (this.indicatorWidth / 2);
      if (this.heatmapColors.length >= 10) {
        this.drawText(sprintf("%." + g + "f", (e + k) / 2), b, m, this.legendFont, this.legendColor, "center", "top", -Math.PI / 2)
      }
      if (this.colorBy) {
        this.drawText(this.colorBy, b + this.margin + this.getFontPt(this.legendFont), m, this.legendFont, this.legendColor, "center", "top", -Math.PI / 2)
      }
      m -= (this.heatmapColors.length * this.indicatorWidth / 2) - (this.indicatorWidth / 2);
      this.drawText(sprintf("%." + g + "f", k), b, m, this.legendFont, this.legendColor, "center", "top", -Math.PI / 2);
      this.addArea(["rect", j - this.margin, m - (this.margin * 2), b + this.margin + this.getFontPt(this.legendFont), d + (this.margin * 2)], [-1], "-legend-indicator-color")
    }
  };
  this.setShapeLegendIndicatorDimensions = function () {
    this.legendShapeIndicatorWidth = (10 * (this.indicatorHeight + this.margin)) + (this.margin * 2);
    this.legendShapeIndicatorHeight = this.indicatorHeight + (this.margin * 3) + this.getFontPt(this.legendFont);
    if (this.shapeBy) {
      this.legendShapeIndicatorHeight += this.getFontPt(this.legendFont) + this.margin
    }
  };
  this.drawShapeIndicator = function (b, m, e, l, g) {
    var k = b;
    var c = m;
    var j = this.indicatorHeight / 2;
    var d = b;
    var a = m;
    if (this.indicatorsPosition == "bottom") {
      for (var f = 0; f < 10; f++) {
        this.drawShape("pie" + f, d, a + this.margin, this.indicatorHeight, this.indicatorHeight, this.background, this.foreground, "open");
        d += this.indicatorHeight + j
      }
      b = k;
      m += this.indicatorHeight + this.margin;
      this.drawText(sprintf("%." + g + "f", e), b, m, this.legendFont, this.legendColor, "center", "top");
      b += this.indicatorHeight * 7;
      this.drawText(sprintf("%." + g + "f", (e + l) / 2), b, m, this.legendFont, this.legendColor, "center", "top");
      if (this.shapeBy) {
        this.drawText(this.shapeBy, b, m + this.margin + this.getFontPt(this.legendFont), this.legendFont, this.legendColor, "center", "top")
      }
      b += this.indicatorHeight * 7;
      this.drawText(sprintf("%." + g + "f", l), b, m, this.legendFont, this.legendColor, "center", "top");
      this.addArea(["rect", k - (this.margin * 2), c - this.margin, b + (this.margin * 2), m + this.getFontPt(this.legendFont) + this.margin], [-1], "-legend-indicator-shape")
    } else {
      for (var f = 0; f < 10; f++) {
        this.drawShape("pie" + f, d + this.margin, a, this.indicatorHeight, this.indicatorHeight, this.background, this.foreground, "open");
        a -= this.indicatorHeight + j
      }
      b += this.indicatorHeight + this.margin;
      m = c;
      this.drawText(sprintf("%." + g + "f", e), b, m, this.legendFont, this.legendColor, "center", "top", -Math.PI / 2);
      m -= this.indicatorHeight * 7;
      if (this.heatmapColors.length >= 10) {
        this.drawText(sprintf("%." + g + "f", (e + l) / 2), b, m, this.legendFont, this.legendColor, "center", "top", -Math.PI / 2)
      }
      if (this.shapeBy) {
        this.drawText(this.shapeBy, b + this.margin + this.getFontPt(this.legendFont), m, this.legendFont, this.legendColor, "center", "top", -Math.PI / 2)
      }
      m -= this.indicatorHeight * 7;
      this.drawText(sprintf("%." + g + "f", l), b, m, this.legendFont, this.legendColor, "center", "top", -Math.PI / 2);
      this.addArea(["rect", k - this.margin, m - (this.margin * 2), b + this.margin + this.getFontPt(this.legendFont), c + (this.margin * 2)], [-1], "-legend-indicator-shape")
    }
  };
  this.setSizeLegendIndicatorDimensions = function () {
    var a = 0;
    var c = this.indicatorHeight / 2;
    for (var b = 0; b < 10; b++) {
      a += this.sizes[b] + c
    }
    this.legendSizeIndicatorWidth = (a - (this.sizes[0] + this.sizes[9])) + (this.margin * 4);
    this.legendSizeIndicatorHeight = this.sizes[9] + (this.margin * 3) + this.getFontPt(this.legendFont);
    if (this.sizeBy) {
      this.legendSizeIndicatorHeight += this.getFontPt(this.legendFont) + this.margin
    }
  };
  this.drawSizeIndicator = function (d, p, g, o, k) {
    var n = d;
    var e = p;
    var l = this.indicatorHeight / 2;
    var q = this.sizes[9];
    var f = d;
    var a = p;
    var m = d;
    var c = p;
    if (this.indicatorsPosition == "bottom") {
      for (var j = 0; j < 10; j++) {
        var b = this.sizes[j];
        this.drawShape("sphere", f, a + (q / 2), b, b, this.background, this.foreground, "open");
        f += b + l;
        if (j < 4) {
          m += b + l
        } else {
          if (j == 5) {
            m += (b + l) / 2
          }
        }
      }
      p += q + this.margin;
      this.drawText(sprintf("%." + k + "f", g), n, p, this.legendFont, this.legendColor, "center", "top");
      this.drawText(sprintf("%." + k + "f", (g + o) / 2), m, p, this.legendFont, this.legendColor, "center", "top");
      if (this.sizeBy) {
        this.drawText(this.sizeBy, n + (this.legendColorIndicatorWidth / 2), p + this.margin + this.getFontPt(this.legendFont), this.legendFont, this.legendColor, "center", "top")
      }
      this.drawText(sprintf("%." + k + "f", o), f - (q + l), p, this.legendFont, this.legendColor, "center", "top");
      this.addArea(["rect", n - (this.margin * 2), e - this.margin, (f - q) + (this.margin * 2), p + this.getFontPt(this.legendFont) + this.margin], [-1], "-legend-indicator-size")
    } else {
      for (var j = 0; j < 10; j++) {
        var b = this.sizes[j];
        this.drawShape("sphere", f + (q / 2), a, b, b, this.background, this.foreground, "open");
        a -= (b + l);
        if (j < 4) {
          c -= (b + l)
        } else {
          if (j == 5) {
            c -= ((b + l) / 2)
          }
        }
      }
      d += q + this.margin;
      this.drawText(sprintf("%." + k + "f", g), d, p, this.legendFont, this.legendColor, "center", "top", -Math.PI / 2);
      this.drawText(sprintf("%." + k + "f", (g + o) / 2), d, c, this.legendFont, this.legendColor, "center", "top", -Math.PI / 2);
      if (this.sizeBy) {
        this.drawText(this.sizeBy, d + this.margin + this.getFontPt(this.legendFont), e - (this.legendColorIndicatorWidth / 2), this.legendFont, this.legendColor, "center", "top", -Math.PI / 2)
      }
      this.drawText(sprintf("%." + k + "f", o), d, a + (q + l), this.legendFont, this.legendColor, "center", "top", -Math.PI / 2);
      this.addArea(["rect", n - this.margin, (a + (q + l)) - (this.margin * 2), d + this.margin + this.getFontPt(this.legendFont), e + (this.margin * 2)], [-1], "-legend-indicator-size")
    }
  };
  this.setDataColorShapeSizeIndicatorDimensions = function () {
    this.setColorLegendIndicatorDimensions();
    this.setShapeLegendIndicatorDimensions();
    this.setSizeLegendIndicatorDimensions()
  };
  this.getDiameterLegend = function () {
    return this.x >= 600 ? 10 : this.x >= 400 ? 8 : this.x >= 200 ? 6 : 4
  };
  this.setColorLegendDimensions = function (a) {
    this.legendColorWidth = 0;
    this.legendColorHeight = 0;
    if (this.colorBy) {
      var f, e;
      var i = this.getDiameterLegend();
      if (a && this.data.x.hasOwnProperty(this.colorBy)) {
        f = this.setMaxSmpStringAnnt(this.colorBy);
        e = this.setMaxSmpStringLabel(this.colorBy)
      } else {
        if (this.data.z.hasOwnProperty(this.colorBy)) {
          f = this.setMaxVarStringAnnt(this.colorBy);
          e = this.setMaxVarStringLabel(this.colorBy)
        }
      }
      var b = this.measureText(f, this.legendFont);
      var c = this.measureText(e, this.legendFont) + i + this.margin;
      this.legendColorWidth = Math.max(b, c) + (this.margin * 2);
      var h = this.getFontPt(this.legendFont);
      var g = Math.max(i, h);
      this.legendColorHeight = (this.legendColorsN * (g + this.margin)) + h + (this.margin * 3)
    }
  };
  this.drawColorLegend = function (c, e) {
    if (this.colorBy && ((this.data.x && this.data.x.hasOwnProperty(this.colorBy)) || (this.data.z && this.data.z.hasOwnProperty(this.colorBy)))) {
      var h = this.getDiameterLegend();
      var f = this.getFontPt(this.legendFont);
      var i = f / 2;
      var g = e + this.margin + i;
      var a = Math.max(h, f);
      this.disableGradientTransparency();
      if (this.legendBackgroundColor) {
        this.rectangle(c, e, this.legendColorWidth, this.legendColorHeight, this.legendBackgroundColor)
      } else {
        this.rectangle(c, e, this.legendColorWidth, this.legendColorHeight, false, this.foreground, "open")
      }
      this.enableGradientTransparency();
      this.drawText(this.colorBy, c + (this.legendColorWidth / 2), g, this.legendFont, this.colorLegend, "center", "middle");
      g += i + this.margin;
      this.drawLine("line", c, g, c + this.legendColorWidth, g, this.foreground);
      g += this.margin + (a / 2);
      for (var b in this.legendColors) {
        this.drawShape("circle", c + this.margin + (h / 2), g, h, h, this.legendColors[b], this.foreground, "closed");
        this.drawText(b, c + (this.margin * 2) + h, g, this.legendFont, this.colorLegend, "left", "middle");
        g += this.margin + a
      }
      this.addArea(["rect", c, e, c + this.legendColorWidth, e + this.legendColorHeight], [-1], "-legend-color")
    }
  };
  this.setShapeLegendDimensions = function () {
    this.legendShapeWidth = 0;
    this.legendShapeHeight = 0;
    if (this.shapeBy && this.data.z.hasOwnProperty(this.shapeBy)) {
      var h = this.getDiameterLegend();
      var e = this.setMaxVarStringAnnt(this.shapeBy);
      var c = this.setMaxVarStringLabel(this.shapeBy);
      var a = this.measureText(e, this.legendFont);
      var b = this.measureText(c, this.legendFont) + h + this.margin;
      this.legendShapeWidth = Math.max(a, b) + (this.margin * 2);
      var g = this.getFontPt(this.legendFont);
      var f = Math.max(h, g);
      this.legendShapeHeight = (this.legendShapesN * (f + this.margin)) + g + (this.margin * 3)
    }
  };
  this.drawShapeLegend = function (b, c) {
    if (this.shapeBy && this.data.z.hasOwnProperty(this.shapeBy)) {
      var h = this.getDiameterLegend();
      var e = this.getFontPt(this.legendFont);
      var i = e / 2;
      var g = c + this.margin + i;
      var a = Math.max(h, e);
      this.disableGradientTransparency();
      if (this.legendBackgroundColor) {
        this.rectangle(b, c, this.legendShapeWidth, this.legendShapeHeight, this.legendBackgroundColor)
      } else {
        this.rectangle(b, c, this.legendShapeWidth, this.legendShapeHeight, false, this.foreground, "open")
      }
      this.enableGradientTransparency();
      this.drawText(this.shapeBy, b + (this.legendShapeWidth / 2), g, this.legendFont, this.colorLegend, "center", "middle");
      g += i + this.margin;
      this.drawLine("line", b, g, b + this.legendShapeWidth, g, this.foreground);
      g += this.margin + (a / 2);
      for (var f in this.legendShapes) {
        this.drawShape(this.legendShapes[f], b + this.margin + (h / 2), g, h, h, this.background, this.foreground, "closed");
        this.drawText(f, b + (this.margin * 2) + h, g, this.legendFont, this.colorLegend, "left", "middle");
        g += this.margin + a
      }
      this.addArea(["rect", b, c, b + this.legendShapeWidth, c + this.legendShapeHeight], [-1], "-legend-shape")
    }
  };
  this.setSizeLegendDimensions = function () {
    this.legendSizeWidth = 0;
    this.legendSizeHeight = 0;
    if (this.sizeBy && this.data.z.hasOwnProperty(this.sizeBy)) {
      var h = this.sizes[this.legendSizesN - 1];
      var e = this.setMaxVarStringAnnt(this.sizeBy);
      var c = this.setMaxVarStringLabel(this.sizeBy);
      var a = this.measureText(e, this.legendFont);
      var b = this.measureText(c, this.legendFont) + h + this.margin;
      this.legendSizeWidth = Math.max(a, b) + (this.margin * 2);
      var g = this.getFontPt(this.legendFont);
      var f = Math.max(h, g);
      this.legendSizeHeight = (this.legendSizesN * (f + this.margin)) + g + (this.margin * 3)
    }
  };
  this.drawSizeLegend = function (c, e) {
    if (this.sizeBy && this.data.z.hasOwnProperty(this.sizeBy)) {
      var h = this.sizes[this.legendSizesN - 1];
      var f = this.getFontPt(this.legendFont);
      var i = f / 2;
      var g = e + this.margin + i;
      var a = Math.max(h, f);
      this.disableGradientTransparency();
      if (this.legendBackgroundColor) {
        this.rectangle(c, e, this.legendSizeWidth, this.legendSizeHeight, this.legendBackgroundColor)
      } else {
        this.rectangle(c, e, this.legendSizeWidth, this.legendSizeHeight, false, this.foreground, "open")
      }
      this.enableGradientTransparency();
      this.drawText(this.sizeBy, c + (this.legendSizeWidth / 2), g, this.legendFont, this.colorLegend, "center", "middle");
      g += i + this.margin;
      this.drawLine("line", c, g, c + this.legendSizeWidth, g, this.foreground);
      g += this.margin + (a / 2);
      for (var b in this.legendSizes) {
        this.drawShape("circle", c + this.margin + (h / 2), g, this.legendSizes[b], this.legendSizes[b], this.background, this.foreground, "closed");
        this.drawText(b, c + (this.margin * 2) + h, g, this.legendFont, this.colorLegend, "left", "middle");
        g += this.margin + a
      }
      this.addArea(["rect", c, e, c + this.legendSizeWidth, e + this.legendSizeHeight], [-1], "-legend-size")
    }
  };
  this.setDataColorShapeSizeDimensions = function () {
    this.setColorLegendDimensions();
    this.setShapeLegendDimensions();
    this.setSizeLegendDimensions()
  };
  this.setDataColor = function (d) {
    if (this.colorBy) {
      var a = 0;
      var b = {};
      this.dataColors = [];
      this.legendColors = {};
      this.legendColorsN = 0;
      if (d) {
        if (!this.isGroupedData && this.data.x && this.data.x.hasOwnProperty(this.colorBy)) {
          if (this.isNumeric(this.data.x[this.colorBy], true)) {
            var c = this.range(this.data.x[this.colorBy], true);
            for (var f = 0; f < this.smpIndices.length; f++) {
              var g = this.smpIndices[f];
              var h = this.data.x[this.colorBy][g];
              var e = !isNaN(h) ? this.getHeatmapColor(c[0], c[1], h) : this.missingDataColor;
              this.dataColors.push(e)
            }
            this.legendColorsN = -1;
            this.legendColorRange = c;
            this.legendColorDecs = this.getAxisDecimals(this.getAxisIncrements(c[0], c[1], this.heatmapColors.length));
            this.setColorLegendIndicatorDimensions()
          } else {
            for (var f = 0; f < this.smpIndices.length; f++) {
              var g = this.smpIndices[f];
              var e = this.data.x[this.colorBy][g];
              if (!b.hasOwnProperty(e)) {
                b[e] = a % this.colors.length;
                this.legendColors[e] = this.colors[b[e]];
                a++
              }
              this.dataColors.push(this.colors[b[e]])
            }
            this.legendColorsN = a;
            this.setColorLegendDimensions(d)
          }
        }
      } else {
        if (this.data.z && this.data.z.hasOwnProperty(this.colorBy)) {
          if (this.isNumeric(this.data.z[this.colorBy], false, true)) {
            var c = this.range(this.data.z[this.colorBy], false, true);
            for (var f = 0; f < this.varIndices.length; f++) {
              var g = this.varIndices[f];
              var h = this.data.z[this.colorBy][g];
              if (!isNaN(h)) {
                this.dataColors.push(this.getHeatmapColor(c[0], c[1], h))
              } else {
                this.dataColors.push(this.missingDataColor)
              }
            }
            this.legendColorRange = c;
            this.legendColorDecs = this.getAxisDecimals(this.getAxisIncrements(c[0], c[1], this.heatmapColors.length));
            this.setColorLegendIndicatorDimensions()
          } else {
            for (var f = 0; f < this.varIndices.length; f++) {
              var g = this.varIndices[f];
              var e = this.data.z[this.colorBy][g];
              if (!b.hasOwnProperty(e)) {
                b[e] = a % this.colors.length;
                this.legendColors[e] = this.colors[b[e]];
                a++
              }
              this.dataColors.push(this.colors[b[e]])
            }
            this.legendColorsN = a;
            this.setColorLegendDimensions()
          }
        } else {
          if (this.getSampleIndices(this.colorBy) > -1) {
            a = this.getSampleIndices(this.colorBy);
            var c = this.getAxisRangeBySample(a);
            if (!this.scatterType || this.scatterType != "function") {
              for (var f = 0; f < this.varIndices.length; f++) {
                var g = this.varIndices[f];
                var h = this.getDataAtPos(g, a);
                if (!isNaN(h)) {
                  this.dataColors.push(this.getHeatmapColor(c[0], c[1], h))
                } else {
                  this.dataColors.push(this.missingDataColor)
                }
              }
            }
            this.legendColorRange = c;
            this.legendColorDecs = this.getAxisDecimals(this.getAxisIncrements(c[0], c[1], this.heatmapColors.length));
            this.setColorLegendIndicatorDimensions()
          } else {
            if (this.colorBy == "variable") {
              for (var f = 0; f < this.varIndices.length; f++) {
                var e = f % this.colors.length;
                this.dataColors.push(this.colors[e])
              }
              this.setVariableLegendDimension()
            } else {
              for (var f = 0; f < this.varIndices.length; f++) {
                this.dataColors.push(this.colors[0])
              }
            }
          }
        }
      }
    }
  };
  this.setDataShape = function () {
    if (this.shapeBy) {
      var f = 0;
      var a = {};
      var j = "square";
      this.dataShapes = [];
      this.legendShapes = {};
      this.legendShapesN = 0;
      if (this.data.z && this.data.z.hasOwnProperty(this.shapeBy)) {
        if (this.isNumeric(this.data.z[this.shapeBy], false, true)) {
          var e = this.range(this.data.z[this.shapeBy], false, true);
          for (var d = 0; d < this.varIndices.length; d++) {
            var g = this.varIndices[d];
            var b = this.data.z[this.shapeBy][g];
            if (!isNaN(b)) {
              var c = this.percentile(e[0], e[1], b);
              var h = parseInt(c / 10);
              if (h > 0) {
                h--
              }
              this.dataShapes.push("pie" + h)
            } else {
              this.dataShapes.push(j)
            }
          }
          this.legendShapeRange = e;
          this.legendShapeDecs = this.getAxisDecimals(this.getAxisIncrements(e[0], e[1], 10));
          this.setShapeLegendIndicatorDimensions()
        } else {
          for (var d = 0; d < this.varIndices.length; d++) {
            var g = this.varIndices[d];
            var h = this.data.z[this.shapeBy][g];
            if (!a.hasOwnProperty(h)) {
              a[h] = f % this.shapes.length;
              this.legendShapes[h] = this.shapes[a[h]];
              f++
            }
            this.dataShapes.push(this.shapes[a[h]])
          }
          this.legendShapesN = f;
          this.setShapeLegendDimensions()
        }
      } else {
        if (this.getSampleIndices(this.shapeBy) > -1) {
          f = this.getSampleIndices(this.shapeBy);
          var e = this.getAxisRangeBySample(f);
          for (var d = 0; d < this.varIndices.length; d++) {
            var g = this.varIndices[d];
            var b = this.getDataAtPos(g, f);
            if (!isNaN(b)) {
              var c = this.percentile(e[0], e[1], b);
              var h = parseInt(c / 10);
              if (h > 0) {
                h--
              }
              this.dataShapes.push("pie" + h)
            } else {
              this.dataShapes.push(j)
            }
          }
          this.legendShapeRange = e;
          this.legendShapeDecs = this.getAxisDecimals(this.getAxisIncrements(e[0], e[1], 10));
          this.setShapeLegendIndicatorDimensions()
        } else {
          if (this.shapeBy == "variable") {
            for (va