package org.intermine.plugin.dbmodel

import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.file.FileTree
import org.gradle.api.tasks.SourceSetContainer
import org.gradle.api.tasks.util.PatternSet

class BioSourceDBModelPlugin implements Plugin<Project>{

    BioSourceDBModelConfig config

    @Override
    void apply(Project project) {
        //only buildDB if there are tests to run
        if ((new File(project.projectDir.absolutePath + "/src/test")).exists()) {
            project.tasks.test.dependsOn 'buildUnitTestDB'
        }

        config = project.extensions.create('bioSourceDBModelConfig', BioSourceDBModelConfig)

        project.task('createTestPrioritiesFile') {
            description "Creates a blank genomic_priorities file for unit tests. When building, the mine's priority file will be used."
            dependsOn 'processResources'

            doLast {
                SourceSetContainer sourceSets = (SourceSetContainer) project.getProperties().get("sourceSets")
                String buildResourcesMainDir = sourceSets.getByName("main").getOutput().resourcesDir
                def fileName = buildResourcesMainDir + File.separator + "genomic_priorities.properties"
                def file = new File(fileName)
                file.newWriter().withWriter { w ->
                    w << "# dummy file for unit tests"
                }
            }
        }

        project.task('copyBioGenomicModel') {
            description "Copies the genomic_model.xml file (generated by biomodel adding genomic_additions and so.obo) into the output resources"
            dependsOn 'processResources'

            doLast {
                SourceSetContainer sourceSets = (SourceSetContainer) project.getProperties().get("sourceSets")
                String buildResourcesMainDir = sourceSets.getByName("main").getOutput().resourcesDir
                FileTree fileTree = project.zipTree(project.configurations.getByName("bioModel").singleFile)
                PatternSet patternSet = new PatternSet()
                patternSet.include("genomic_model.xml")
                File file = fileTree.matching(patternSet).singleFile
                String modelFilePath = buildResourcesMainDir + File.separator + "genomic_model.xml"
                file.createNewFile()
                file.renameTo(modelFilePath)
            }
        }

        project.task('copyGlobalAdditionsFile') {
            description "Copies the extra global additions file (if specified) onto the classpath"
            doLast {
                 if ((new File(project.rootDir.absolutePath + "/" + config.globalAdditionsFile)).exists()) {
                    SourceSetContainer sourceSets = (SourceSetContainer) project.getProperties().get("sourceSets")
                    String buildResourcesMainDir = sourceSets.getByName("main").getOutput().resourcesDir
                    project.copy {
                        from project.rootDir.absolutePath + "/" + config.globalAdditionsFile
                        into buildResourcesMainDir
                        rename { 'global_additions.xml' }
                    }
                }
            }
        }

        project.task('mergeModels') {
            description "Merges the bio-source specific additions.xml into an intermine XML model"
            dependsOn 'copyBioGenomicModel', 'copyGlobalAdditionsFile'
            doLast {
                SourceSetContainer sourceSets = (SourceSetContainer) project.getProperties().get("sourceSets")
                String buildResourcesMainDir = sourceSets.getByName("main").getOutput().resourcesDir
                String bioSourceAdditionsFile = project.name.minus("bio-source-") + "_additions.xml"
                String inputModelFilePath = buildResourcesMainDir + File.separator + "genomic_model.xml"

                if ((new File(buildResourcesMainDir + File.separator + bioSourceAdditionsFile)).exists()) {
                    def ant = new AntBuilder()
                    ant.taskdef(name: "mergeBioSourceModel", classname: "org.intermine.task.ModelMergerTask") {
                        classpath {
                            pathelement(path: project.configurations.mergeModel.asPath)
                            dirset(dir: project.buildDir.absolutePath)
                        }
                    }
                    String extraAdditionsFile = config.globalAdditionsFile
                    if (extraAdditionsFile != null) {
                        ant.mergeBioSourceModel(inputModelFile: inputModelFilePath, additionsFile: "global_additions.xml",
                                outputFile: inputModelFilePath)
                    }
                    ant.mergeBioSourceModel(inputModelFile: inputModelFilePath, additionsFile: bioSourceAdditionsFile,
                            outputFile: inputModelFilePath)
                }
            }
        }

        project.tasks.test.dependsOn 'createTestPrioritiesFile'
    }
}