/*******************************************************************************
* Copyright (c) 2009, 2011 Overture Team and others.
*
* Overture is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Overture is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Overture.  If not, see <http://www.gnu.org/licenses/>.
*
* The Overture Tool web-site: http://overturetool.org/
*******************************************************************************/

package org.intocps.maestro.ast.node;

import java.util.Map;

import org.intocps.maestro.ast.analysis.intf.IAnalysis;
import org.intocps.maestro.ast.analysis.intf.IAnswer;
import org.intocps.maestro.ast.analysis.intf.IQuestion;
import org.intocps.maestro.ast.analysis.intf.IQuestionAnswer;
import org.intocps.maestro.ast.analysis.AnalysisException;

public interface INode
{

	public abstract Object clone();

	public abstract INode clone(Map<INode, INode> oldToNewMap);

	/**
	 * Returns the parent node of this node.
	 * @return the parent node of this node
	 */
	public abstract INode parent();

	/**
	 * Sets the parent node of this node.
	 * @param parent the new parent node of this node
	 */
	public abstract void parent(INode parent);

	/**
	 * Removes the {@link INode} {@code child} as a child of this node.
	 * @param child the child node to be removed from this node
	 * @throws RuntimeException if {@code child} is not a child of this node
	 */
	public abstract void removeChild(INode child);

	/**
	 * If {%code oldChild} pointed to by any field of this {@link INode} instance
	 * replace it with {@code newChild}.
	 * 
	 * @param oldChild 
	 * @param newChild
	 * @throws RuntimeException in particular if {@code newChild} has an improper type 
	 * for the field pointing to {@code oldChild}.
	 */
	public void replaceChild(INode oldChild, INode newChild);
	
	/**
	 * Returns the nearest ancestor of this node (including itself)
	 * which is a subclass of {@code classType}.
	 * @param classType the superclass used
	 * @return the nearest ancestor of this node
	 */
	public abstract <T extends INode> T getAncestor(Class<T> classType);

	/**
	 * Applies this node to the {@link IAnalysis} visitor {@code analysis}.
	 * @param analysis the {@link IAnalysis} to which this node is applied
	 */
	public abstract void apply(IAnalysis analysis) throws AnalysisException;

	/**
	 * Returns the answer for {@code caller} by applying this node to the
	 * {@link IAnswer} visitor.
	 * @param caller the {@link IAnswer} to which this node is applied
	 * @return the answer as returned from {@code caller}
	 */
	public abstract <A> A apply(IAnswer<A> caller) throws AnalysisException;

	/**
	 * Applies this node to the {@link IQuestion} visitor {@code caller}.
	 * @param caller the {@link IQuestion} to which this node is applied
	 * @param question the question provided to {@code caller}
	 */
	public abstract <Q> void apply(IQuestion<Q> caller, Q question) throws AnalysisException;

	/**
	 * Returns the answer for {@code answer} by applying this node with the
	 * {@code question} to the {@link IQuestionAnswer} visitor.
	 * @param caller the {@link IQuestionAnswer} to which this node is applied
	 * @param question the question provided to {@code answer}
	 * @return the answer as returned from {@code answer}
	 */
	public abstract <Q, A> A apply(IQuestionAnswer<Q, A> caller, Q question) throws AnalysisException;

	
	public abstract Map<String,Object> getChildren(Boolean includeInheritedFields);
}
