package org.intocps.maestro.framework.fmi2.api.mabl;

import org.intocps.maestro.ast.AVariableDeclaration;
import org.intocps.maestro.ast.LexIdentifier;
import org.intocps.maestro.ast.node.*;
import org.intocps.maestro.framework.core.IRelation;
import org.intocps.maestro.framework.core.ISimulationEnvironment;
import org.intocps.maestro.framework.core.IVariable;
import org.intocps.maestro.framework.fmi2.ComponentInfo;
import org.intocps.maestro.framework.fmi2.Fmi2SimulationEnvironment;
import org.intocps.maestro.framework.fmi2.RelationVariable;
import org.intocps.maestro.framework.fmi2.api.Fmi2Builder;
import org.intocps.maestro.framework.fmi2.api.mabl.variables.ComponentVariableFmi2Api;
import org.intocps.maestro.framework.fmi2.api.mabl.variables.FmuVariableFmi2Api;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.xml.xpath.XPathExpressionException;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import static org.intocps.maestro.ast.MableAstFactory.*;

public class FromMaBLToMaBLAPI {
    final static Logger logger = LoggerFactory.getLogger(FromMaBLToMaBLAPI.class);

    public static Map.Entry<String, ComponentVariableFmi2Api> getComponentVariableFrom(MablApiBuilder builder, PExp exp,
            Fmi2SimulationEnvironment env) throws XPathExpressionException, InvocationTargetException, IllegalAccessException {
        if (exp instanceof AIdentifierExp) {
            return getComponentVariableFrom(builder, exp, env, ((AIdentifierExp) exp).getName().getText());
        } else {
            throw new RuntimeException("exp is not of type AIdentifierExp, but of type: " + exp.getClass());
        }
    }

    public static Map.Entry<String, ComponentVariableFmi2Api> getComponentVariableFrom(MablApiBuilder builder, PExp exp,
            Fmi2SimulationEnvironment env,
            String environmentComponentName) throws IllegalAccessException, XPathExpressionException, InvocationTargetException {
        if (exp instanceof AIdentifierExp) {
            String componentName = ((AIdentifierExp) exp).getName().getText();

            ComponentInfo instance = env.getInstanceByLexName(environmentComponentName);
            ModelDescriptionContext modelDescriptionContext = new ModelDescriptionContext(instance.modelDescription);

            //This dummy statement is removed later. It ensures that the share variables are added to the root scope.
            PStm dummyStm = newABlockStm();
            builder.getDynamicScope().add(dummyStm);

            FmuVariableFmi2Api fmu = new FmuVariableFmi2Api(instance.fmuIdentifier, builder, modelDescriptionContext, dummyStm, newANameType("FMI2"),
                    builder.getDynamicScope().getActiveScope(), builder.getDynamicScope(), null,
                    new AIdentifierExp(new LexIdentifier(instance.fmuIdentifier.replace("{", "").replace("}", ""), null)));

            ComponentVariableFmi2Api a;
            if (environmentComponentName == null) {
                a = new ComponentVariableFmi2Api(dummyStm, fmu, componentName, modelDescriptionContext, builder,
                        builder.getDynamicScope().getActiveScope(), null, newAIdentifierExp(componentName));
            } else {
                a = new ComponentVariableFmi2Api(dummyStm, fmu, componentName, modelDescriptionContext, builder,
                        builder.getDynamicScope().getActiveScope(), null, newAIdentifierExp(componentName), environmentComponentName);
            }
            List<RelationVariable> variablesToLog = env.getVariablesToLog(componentName);
            a.setVariablesToLog(variablesToLog);

            return Map.entry(componentName, a);
        } else {
            throw new RuntimeException("exp is not of type AIdentifierExp, but of type: " + exp.getClass());
        }
    }

    public static void createBindings(Map<String, ComponentVariableFmi2Api> instances,
            ISimulationEnvironment env) throws Fmi2Builder.Port.PortLinkException {
        for (Map.Entry<String, ComponentVariableFmi2Api> entry : instances.entrySet()) {
            for (IRelation relation : env.getRelations(entry.getKey()).stream()
                    .filter(x -> x.getDirection() == Fmi2SimulationEnvironment.Relation.Direction.OutputToInput &&
                            x.getOrigin() == Fmi2SimulationEnvironment.Relation.InternalOrExternal.External).collect(Collectors.toList())) {

                for (IVariable targetVar : relation.getTargets().values()) {
                    String targetName = targetVar.getScalarVariable().getInstance().getText();
                    if (instances.containsKey(targetName)) {
                        ComponentVariableFmi2Api instance = instances.get(targetName);

                        PortFmi2Api targetPort = instance.getPort(targetVar.getScalarVariable().getScalarVariable().getName());

                        String sourcePortName = relation.getSource().getScalarVariable().getScalarVariable().getName();
                        if (targetPort != null) {
                            entry.getValue().getPort(sourcePortName).linkTo(targetPort);
                        } else {
                            //error port not found in target var
                            logger.warn("Failed to find port '{}' on instance '{}' required by relational '{}' ", sourcePortName, targetName,
                                    relation);
                        }
                    } else {
                        logger.warn(
                                "Failed to find instance required by relational information from simulation env. Missing '{}' in relation " + "{}",
                                targetName, relation);
                    }
                }
            }
        }

    }


    public static Map<String, ComponentVariableFmi2Api> getComponentVariablesFrom(MablApiBuilder builder, PExp exp,
            Fmi2SimulationEnvironment env) throws IllegalAccessException, XPathExpressionException, InvocationTargetException {
        LexIdentifier componentsArrayName = ((AIdentifierExp) exp).getName();
        SBlockStm containingBlock = exp.getAncestor(SBlockStm.class);
        Optional<AVariableDeclaration> componentDeclaration =
                containingBlock.getBody().stream().filter(ALocalVariableStm.class::isInstance).map(ALocalVariableStm.class::cast)
                        .map(ALocalVariableStm::getDeclaration)
                        .filter(decl -> decl.getName().equals(componentsArrayName) && !decl.getSize().isEmpty() && decl.getInitializer() != null)
                        .findFirst();

        if (!componentDeclaration.isPresent()) {
            throw new RuntimeException("Could not find names for components");
        }

        AArrayInitializer initializer = (AArrayInitializer) componentDeclaration.get().getInitializer();


        List<PExp> componentIdentifiers =
                initializer.getExp().stream().filter(AIdentifierExp.class::isInstance).map(AIdentifierExp.class::cast).collect(Collectors.toList());

        HashMap<String, ComponentVariableFmi2Api> fmuInstances = new HashMap<>();

        for (PExp componentName : componentIdentifiers) {
            Map.Entry<String, ComponentVariableFmi2Api> component = getComponentVariableFrom(builder, componentName, env);
            fmuInstances.put(component.getKey(), component.getValue());
        }


        return fmuInstances;
    }
}
