/*
 * Copyright 2016 Panagiotis Minos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ioperm.morphology.el;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import morfologik.stemming.Dictionary;
import morfologik.stemming.DictionaryLookup;
import morfologik.stemming.WordData;

/**
 * A morphological analyzer for the Greek language.
 *
 * @author Panagiotis Minos
 * @since 1.0.0
 */
public class GreekAnalyzer {

    private final Dictionary dictionary;

    /**
     * Default constructor.
     */
    public GreekAnalyzer() {
        try {
            InputStream data = this.getClass().getResourceAsStream("analysis.dict");
            InputStream meta = this.getClass().getResourceAsStream("analysis.info");
            dictionary = Dictionary.read(data, meta);
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Find the lemma of the word.
     *
     * @param word the inflected surface form
     * @param capitalized if true, assume that the first char of the word may be
     * capitalized.
     * @return a (possibly empty) list of {@link Lemma} entries.
     */
    public List<Lemma> getLemma(CharSequence word, boolean capitalized) {
        List<Lemma> l = new ArrayList<>();
        DictionaryLookup lookup = new DictionaryLookup(dictionary);
        List<WordData> wl = lookup.lookup(word);
        for (WordData w : wl) {
            l.add(new Lemma(w));
        }
        if (capitalized) {
            StringBuilder builder = new StringBuilder(word);
            builder.setCharAt(0, CharUtils.toLower(builder.charAt(0), false));
            wl = lookup.lookup(builder);
            for (WordData w : wl) {
                Lemma newLemma = new Lemma(w);
                if (!l.contains(newLemma)) {
                    l.add(newLemma);
                }
            }
        }
        return l;
    }

}
