/*
 * Copyright 2016 Panagiotis Minos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ioperm.morphology.el;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import morfologik.stemming.Dictionary;
import morfologik.stemming.DictionaryLookup;
import morfologik.stemming.WordData;

/**
 * A morphological generator for the Greek language.
 *
 * @author Panagiotis Minos
 * @since 1.0.0
 */
public class GreekGenerator {

    private final Dictionary dictionary;

    /**
     * Default constructor.
     */
    public GreekGenerator() {
        try {
            InputStream data = this.getClass().getResourceAsStream("generation.dict");
            InputStream meta = this.getClass().getResourceAsStream("generation.info");
            dictionary = Dictionary.read(data, meta);
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Generate the inflected forms of the lemma.
     *
     * @param lemma the lemma
     * @param tag the morphological attributes.
     * @return a list of inflected word forms.
     */
    public List<String> generate(CharSequence lemma, CharSequence tag) {
        List<String> wfs = new ArrayList<>();
        StringBuilder builder = new StringBuilder(32);
        builder.append(lemma).append('|').append(tag);
        DictionaryLookup lookup = new DictionaryLookup(dictionary);
        List<WordData> result = lookup.lookup(builder);
        for (WordData wd : result) {
            wfs.add(wd.getStem().toString());
        }
        return wfs;
    }

}
