/*
 * Copyright 2016 Panagiotis Minos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ioperm.morphology.el;

import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Set;
import morfologik.stemming.WordData;

/**
 * Class for storing the result of the morphological analysis.
 *
 * @author Panagiotis Minos
 * @since 1.0.0
 */
public class Lemma {

    private final String wordForm;
    private final String lemma;
    private final String tag;
    private final Set<String> attributes = new LinkedHashSet<>();

    Lemma(String wordForm, String lemma, String tag) {
        this.wordForm = wordForm;
        this.lemma = lemma;
        this.tag = tag;
        attributes.addAll(Arrays.asList(tag.split("\\:")));
    }

    Lemma(WordData w) {
        this.wordForm = (w.getWord() == null) ? null : w.getWord().toString();
        this.lemma = (w.getStem() == null) ? null : w.getStem().toString();
        this.tag = (w.getTag() == null) ? null : w.getTag().toString();
        if (w.getTag() != null) {
            attributes.addAll(Arrays.asList(w.getTag().toString().split("\\:")));
        }
    }

    /**
     * Get the base form.
     *
     * @return the inflected word form.
     */
    public String getWordForm() {
        return wordForm;
    }

    /**
     * Get the lemma.
     *
     * @return the lemma.
     */
    public String getLemma() {
        return lemma;
    }

    /**
     * Get the morphological attributes.
     *
     * @return a string representation of the morphological attributes.
     */
    public String getTag() {
        return tag;
    }

    /**
     * Get the morphological attributes.
     *
     * @return an unmodifiable set containing the morphological attributes.
     */
    public Set<String> getAttributes() {
        return Collections.unmodifiableSet(attributes);
    }

    @Override
    public int hashCode() {
        int hash = 3;
        hash = 79 * hash + Objects.hashCode(this.wordForm);
        hash = 79 * hash + Objects.hashCode(this.lemma);
        hash = 79 * hash + Objects.hashCode(this.attributes);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final Lemma other = (Lemma) obj;
        if (!Objects.equals(this.wordForm, other.wordForm)) {
            return false;
        }
        if (!Objects.equals(this.lemma, other.lemma)) {
            return false;
        }
        if (!Objects.equals(this.tag, other.tag)) {
            return false;
        }
        return true;
    }

}
