(ns hoot.ontology
  (:require [cemerick.url :as url])
  (:import [com.hp.hpl.jena.ontology OntDocumentManager OntModel OntModelSpec]))

(defn ontology-manager
  []
  (OntDocumentManager/getInstance))

(defn ontology
  [ontman path model]
  (.addModel ontman path model true)
  (.getOntology ontman path OntModelSpec/OWL_DL_MEM))

(defn imported-uris
  "Lists the imported ontology URIs. If transitive? is false (the default)
   then the transitive imports (the imports of the imports) will not be listed."
  ([ont]
    (imported-uris ont false))
  ([ont transitive?]
    (seq (.listImportedOntologyURIs ont transitive?))))

(defn class->map
  "Turns a class from an ontology into a map. Note that :uri may be nil."
  [cls]
  {:namespace (.getNameSpace cls)
   :localname (.getLocalName cls)
   :uri       (.getURI cls)})

(defn map->class
  "Turns a map of the following structure into a ontology class:
     {
       :namespace \"The namespace of the class\"
       :localname \"The localname of the class\"
     }
   The returned class is not associated with an ontological model."
  [ont cls-map]
  (let [namespace (:namespace cls-map)
        localname (:localname cls-map)]
    (.createClass ont (str (assoc (url/url namespace) :anchor localname)))))

(defn uri->class
  "Turns a URI into a OntClass object."
  [ont uri]
  (.getOntClass ont uri))

(defn classes
  "Lists all of the classes in the ontology. Classes are the types of the things
   defined in the ontology."
  [ont]
  (filter #(and (.getLocalName %1) (.getNameSpace %1)) 
          (seq (.toList (.listClasses ont)))))

(defn sclass-seq
  [cls]
  (or (seq (.toList (.listSuperClasses cls))) []))

(defn superclasses
  "Lists all of the superclasses of a particular ontology class"
  [cls]
  (loop [sclasses    (sclass-seq cls)
         accumulator []]
    (if (every? #(not (.hasSuperClass %)) sclasses)
      (concat accumulator sclasses)
      (let [new-sclasses (flatten (map sclass-seq sclasses))]
        (recur new-sclasses (concat accumulator sclasses))))))

(defn properties
  "Lists all of the properties defined in the ontology."
  [ont]
  (filter #(and (.getLocalName %1) (.getNameSpace %1)) 
          (seq (.toList (.listAllOntProperties ont)))))

(defn listize
  [iter]
  (seq (.toList iter)))

(defn prop-applies?
  "Returns true if one of the classes is a domain class for the property 'prp'."
  [classes prp]
  (if-not (listize (.listDomain prp))
    true
    (some #(.hasDomain prp %) classes)))

(defn possible-class-properties
  "Lists all of the applicable properties (a.k.a. predicates) that are available
   for the given class."
  [ont cls]
  (filter 
    (partial prop-applies? (conj (superclasses cls) cls)) 
    (properties ont)))

