/*
 * Decompiled with CFR 0.152.
 */
package org.iplass.mtp.impl.entity.csv;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.BufferedInputStream;
import java.io.BufferedWriter;
import java.io.Flushable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.math.BigDecimal;
import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import org.apache.commons.text.StringEscapeUtils;
import org.iplass.mtp.ManagerLocator;
import org.iplass.mtp.auth.AuthContext;
import org.iplass.mtp.entity.BinaryReference;
import org.iplass.mtp.entity.Entity;
import org.iplass.mtp.entity.EntityManager;
import org.iplass.mtp.entity.SelectValue;
import org.iplass.mtp.entity.definition.EntityDefinition;
import org.iplass.mtp.entity.definition.EntityDefinitionManager;
import org.iplass.mtp.entity.definition.PropertyDefinition;
import org.iplass.mtp.entity.definition.PropertyDefinitionType;
import org.iplass.mtp.entity.definition.VersionControlType;
import org.iplass.mtp.entity.definition.properties.BinaryProperty;
import org.iplass.mtp.entity.definition.properties.ExpressionProperty;
import org.iplass.mtp.entity.definition.properties.ReferenceProperty;
import org.iplass.mtp.entity.definition.properties.SelectProperty;
import org.iplass.mtp.entity.permission.EntityPropertyPermission;
import org.iplass.mtp.impl.core.ExecuteContext;
import org.iplass.mtp.impl.entity.csv.EntityCsvException;
import org.iplass.mtp.impl.entity.csv.EntityWriteOption;
import org.iplass.mtp.util.DateUtil;
import org.iplass.mtp.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EntityCsvWriter
implements AutoCloseable,
Flushable {
    private static final Logger logger = LoggerFactory.getLogger(EntityCsvWriter.class);
    private static final String DOUBLE_QUOT = "\"";
    private static final int BOM = 65279;
    private static final String CR = "\n";
    private EntityDefinition definition;
    private Writer writer;
    private EntityWriteOption option;
    private ZipOutputStream binaryStore;
    private AuthContext auth;
    private EntityDefinitionManager edm;
    private EntityManager em;
    private boolean isInit;
    private List<PropertyDefinition> properties;
    private String dateFormat;
    private String dateTimeFormat;
    private String timeFormat;
    private ObjectMapper mapper;

    public EntityCsvWriter(EntityDefinition definition, OutputStream out) throws IOException {
        this(definition, out, new EntityWriteOption());
    }

    public EntityCsvWriter(EntityDefinition definition, OutputStream out, EntityWriteOption option) throws IOException {
        this(definition, out, option, null);
    }

    public EntityCsvWriter(EntityDefinition definition, OutputStream out, EntityWriteOption option, ZipOutputStream binaryStore) throws IOException {
        this.definition = definition;
        this.option = option;
        this.writer = new BufferedWriter(new OutputStreamWriter(out, option.getCharset()));
        this.binaryStore = binaryStore;
        this.auth = AuthContext.getCurrentContext();
        this.edm = ManagerLocator.getInstance().getManager(EntityDefinitionManager.class);
        this.em = ManagerLocator.getInstance().getManager(EntityManager.class);
    }

    @Override
    public void flush() throws IOException {
        this.writer.flush();
    }

    @Override
    public void close() {
        try {
            this.writer.close();
        }
        catch (IOException e) {
            logger.warn("fail to close EntityCsvWriter resource. check whether resource is leak or not.", (Throwable)e);
        }
    }

    public List<PropertyDefinition> getProperties() {
        this.init();
        return this.properties;
    }

    public void writeHeader() {
        this.init();
        IntStream.range(0, this.properties.size()).forEach(i -> {
            PropertyDefinition property = this.properties.get(i);
            if (!(property instanceof ReferenceProperty) && property.getMultiplicity() != 1) {
                IntStream.range(0, property.getMultiplicity()).forEach(j -> {
                    if (j != 0) {
                        this.writeComma();
                    }
                    this.writeText(property.getName() + "[" + j + "]" + this.option.getColumnDisplayName().apply(property));
                });
            } else {
                this.writeText(property.getName() + this.option.getColumnDisplayName().apply(property));
            }
            if (i < this.properties.size() - 1) {
                this.writeComma();
            }
        });
        this.newLine();
    }

    public void writeEntity(Entity entity) {
        this.init();
        Iterator<PropertyDefinition> it = this.properties.iterator();
        while (it.hasNext()) {
            PropertyDefinition property = it.next();
            if (!(property instanceof ReferenceProperty) && property.getMultiplicity() != 1) {
                Object[] values = (Object[])entity.getValue(property.getName());
                if (property instanceof SelectProperty && this.option.getSortSelectValue().apply((SelectProperty)property).booleanValue()) {
                    SelectProperty sp = (SelectProperty)property;
                    List<SelectValue> selectableValues = sp.getSelectValueList();
                    IntStream.range(0, property.getMultiplicity()).forEach(i -> {
                        if (i != 0) {
                            this.writeComma();
                        }
                        if (values == null || i > selectableValues.size() - 1) {
                            this.writeValue(null, property, this.binaryStore);
                        } else {
                            SelectValue targetValue = (SelectValue)selectableValues.get(i);
                            boolean exist = Arrays.stream(values).anyMatch(value -> value != null && ((SelectValue)value).getValue().equals(targetValue.getValue()));
                            if (exist) {
                                this.writeValue(targetValue, property, this.binaryStore);
                            } else {
                                this.writeValue(null, property, this.binaryStore);
                            }
                        }
                    });
                } else {
                    IntStream.range(0, property.getMultiplicity()).forEach(i -> {
                        if (i != 0) {
                            this.writeComma();
                        }
                        if (values == null || i >= values.length) {
                            this.writeValue(null, property, this.binaryStore);
                        } else {
                            this.writeValue(values[i], property, this.binaryStore);
                        }
                    });
                }
            } else {
                Object val = entity.getValue(property.getName());
                this.writeValue(val, property, this.binaryStore);
            }
            if (it.hasNext()) {
                this.writeComma();
                continue;
            }
            this.newLine();
        }
    }

    protected void init() {
        if (this.isInit) {
            return;
        }
        if ("UTF-8".equalsIgnoreCase(this.option.getCharset())) {
            try {
                this.writer.write(65279);
            }
            catch (IOException e) {
                throw new EntityCsvException(e);
            }
        }
        this.properties = this.definition.getPropertyList().stream().filter(property -> this.auth.checkPermission(new EntityPropertyPermission(this.definition.getName(), property.getName(), EntityPropertyPermission.Action.REFERENCE))).filter(property -> !"version".equals(property.getName()) || this.definition.getVersionControlType() != VersionControlType.NONE).filter(property -> !(property instanceof ReferenceProperty) || ((ReferenceProperty)property).getMappedBy() == null).filter(property -> this.option.isWithBinary() || !(property instanceof BinaryProperty)).collect(Collectors.toList());
        this.isInit = true;
    }

    private void writeText(String text) {
        try {
            if (StringUtil.isEmpty(text)) {
                return;
            }
            String outText = StringEscapeUtils.escapeCsv((String)text);
            if (this.option.isQuoteAll()) {
                if (outText.startsWith(DOUBLE_QUOT) && outText.endsWith(DOUBLE_QUOT)) {
                    this.writer.write(outText);
                } else {
                    this.writer.write(DOUBLE_QUOT);
                    this.writer.write(outText);
                    this.writer.write(DOUBLE_QUOT);
                }
            } else {
                this.writer.write(outText);
            }
        }
        catch (IOException e) {
            throw new EntityCsvException(e);
        }
    }

    private void writeComma() {
        try {
            this.writer.write(",");
        }
        catch (IOException e) {
            throw new EntityCsvException(e);
        }
    }

    private void newLine() {
        try {
            this.writer.write(CR);
        }
        catch (IOException e) {
            throw new EntityCsvException(e);
        }
    }

    private void writeValue(Object val, PropertyDefinition pd, ZipOutputStream binaryStore) {
        if (val == null) {
            return;
        }
        switch (pd.getType()) {
            case EXPRESSION: {
                ExpressionProperty ep = (ExpressionProperty)pd;
                if (ep.getResultType() != null) {
                    this.writeValue(val, ep.getResultType());
                    break;
                }
                this.writeValue(val, ep.getType());
                break;
            }
            case REFERENCE: {
                ReferenceProperty rpd = (ReferenceProperty)pd;
                EntityDefinition ed = this.edm.get(rpd.getObjectDefinitionName());
                if (rpd.getMultiplicity() == 1) {
                    Entity entity = (Entity)val;
                    if (ed.getVersionControlType().equals((Object)VersionControlType.NONE) && !this.option.isWithReferenceVersion()) {
                        this.writeText(entity.getOid());
                        break;
                    }
                    this.writeText(entity.getOid() + "." + entity.getVersion());
                    break;
                }
                Entity[] eList = (Entity[])val;
                StringBuilder sb = new StringBuilder();
                for (int i = 0; i < eList.length; ++i) {
                    if (i != 0) {
                        sb.append(",");
                    }
                    if (eList[i] == null) continue;
                    if (ed.getVersionControlType().equals((Object)VersionControlType.NONE) && !this.option.isWithReferenceVersion()) {
                        sb.append(eList[i].getOid());
                        continue;
                    }
                    sb.append(eList[i].getOid() + "." + eList[i].getVersion());
                }
                this.writeText(sb.toString());
                break;
            }
            case BINARY: {
                BinaryReference br = (BinaryReference)val;
                LinkedHashMap<String, String> valueMap = new LinkedHashMap<String, String>();
                valueMap.put("lobid", String.valueOf(br.getLobId()));
                valueMap.put("name", br.getName());
                valueMap.put("type", br.getType());
                this.writeText(this.toJsonString(valueMap));
                this.writeBinaryData(br, binaryStore);
                break;
            }
            default: {
                this.writeValue(val, pd.getType());
            }
        }
    }

    private void writeValue(Object val, PropertyDefinitionType type) {
        switch (type) {
            case BOOLEAN: {
                Boolean b = (Boolean)val;
                this.writeText(b != false ? "1" : "0");
                break;
            }
            case DATE: {
                this.writeText(DateUtil.getSimpleDateFormat(this.getDateFormat(), false).format((Date)val));
                break;
            }
            case DATETIME: {
                this.writeText(DateUtil.getSimpleDateFormat(this.getDateTimeFormat(), false).format((Timestamp)val));
                break;
            }
            case TIME: {
                this.writeText(DateUtil.getSimpleDateFormat(this.getTimeFormat(), false).format((Time)val));
                break;
            }
            case SELECT: {
                SelectValue sv = (SelectValue)val;
                this.writeText(sv.getValue());
                break;
            }
            case DECIMAL: {
                this.writeText(((BigDecimal)val).toPlainString());
                break;
            }
            case FLOAT: {
                this.writeText(BigDecimal.valueOf((Double)val).toPlainString());
                break;
            }
            case EXPRESSION: 
            case INTEGER: 
            case LONGTEXT: 
            case STRING: 
            case AUTONUMBER: {
                this.writeText(val.toString());
                break;
            }
            case REFERENCE: 
            case BINARY: {
                break;
            }
            default: {
                throw new EntityCsvException("can not convert from " + (Object)((Object)type) + ":" + val);
            }
        }
    }

    private void writeBinaryData(BinaryReference br, ZipOutputStream binaryStore) {
        block17: {
            if (binaryStore != null) {
                try {
                    String entryName = "lobs/" + this.definition.getName() + "." + br.getLobId();
                    ZipEntry zentry = new ZipEntry(entryName);
                    binaryStore.putNextEntry(zentry);
                    InputStream is = this.em.getInputStream(br);
                    if (is == null) {
                        logger.warn("cannot output binary data. entity = " + br.getDefinitionName() + ", lobid = " + br.getLobId());
                        break block17;
                    }
                    try (BufferedInputStream bis = new BufferedInputStream(is);){
                        byte[] buf = new byte[1024];
                        int len = 0;
                        while ((len = ((InputStream)bis).read(buf)) >= 0) {
                            binaryStore.write(buf, 0, len);
                        }
                        binaryStore.closeEntry();
                    }
                }
                catch (IOException e) {
                    throw new EntityCsvException(e);
                }
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private String toJsonString(Object value) {
        if (this.mapper == null) {
            this.mapper = new ObjectMapper();
        }
        try (StringWriter writer = new StringWriter();){
            this.mapper.writeValue((Writer)writer, value);
            String string = writer.toString();
            return string;
        }
        catch (JsonProcessingException e) {
            throw new EntityCsvException(e);
        }
        catch (IOException e) {
            throw new EntityCsvException(e);
        }
    }

    private String getDateFormat() {
        if (this.dateFormat != null) {
            return this.dateFormat;
        }
        this.dateFormat = this.option.getDateFormat() != null ? this.option.getDateFormat() : ExecuteContext.getCurrentContext().getLocaleFormat().getOutputDateFormat();
        return this.dateFormat;
    }

    private String getDateTimeFormat() {
        if (this.dateTimeFormat != null) {
            return this.dateTimeFormat;
        }
        this.dateTimeFormat = this.option.getDatetimeSecFormat() != null ? this.option.getDatetimeSecFormat() : ExecuteContext.getCurrentContext().getLocaleFormat().getOutputDatetimeSecFormat();
        return this.dateTimeFormat;
    }

    private String getTimeFormat() {
        if (this.timeFormat != null) {
            return this.timeFormat;
        }
        this.timeFormat = this.option.getTimeSecFormat() != null ? this.option.getTimeSecFormat() : ExecuteContext.getCurrentContext().getLocaleFormat().getOutputTimeSecFormat();
        return this.timeFormat;
    }
}

