/*
 * Copyright (C) 2011 INFORMATION SERVICES INTERNATIONAL - DENTSU, LTD. All Rights Reserved.
 *
 * Unless you have purchased a commercial license,
 * the following license terms apply:
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package org.iplass.mtp.webapi.definition;

import javax.xml.bind.annotation.XmlRootElement;

import org.iplass.mtp.command.definition.config.CommandConfig;
import org.iplass.mtp.definition.Definition;

@XmlRootElement
public class WebApiDefinition implements Definition {

	private static final long serialVersionUID = 7903913056944225448L;

	/**
	 * 共通項目。
	 */
	private String name;
	private String displayName;
	private String description;
	
	/**
	 * WebAPIキャッシュ指定。
	 * 未指定の場合はキャッシュをしない。
	 */
	private CacheControlType cacheControlType;

	/** WebAPIキャッシュのmax-age（秒）*/
	private long cacheControlMaxAge = -1;

	/**
	 * このWebApiが呼び出されたときに実行するCommand。
	 */
	private CommandConfig commandConfig;

	/** このWebApiで処理されるCommandを特権（セキュリティ制約を受けない）にて処理するかどうか。デフォルトはfalse。 */
	private boolean isPrivilaged;

	/** このWebApiの呼び出しをセキュリティ設定によらず呼び出し可能にする場合は、trueを設定。 isPrivilagedとの違いは、Entityの操作などにおいては、セキュリティ制約を受ける。デフォルトはfalse。*/
	private boolean isPublicWebApi;

	/** XMLHttpRequestがセットされていることを確認するか。 */
	private boolean isCheckXRequestedWithHeader = true;

	/**
	 * Commandの実行結果と、表示処理の対応の定義。
	 */
	private String[] results;

	/**
	 * WebApiの受付種別。
	 */
	private RequestType[] accepts;

	/**
	 * WebApiのメソッド種別。
	 */
	private MethodType[] methods;

	/**
	 * REST JSON受付時のパラメータ名
	 */
	private String restJsonParameterName;

	/**
	 * REST JSON受付時のパラメータタイプ（クラスパス）
	 */
	private String restJsonParameterType;

	/**
	 * REST XML受付時のパラメータ名
	 */
	private String restXmlParameterName;

	/** Tokenチェックの実行設定。
	 * 未指定可(Tokenチェックは実行されない)。
	 */
	private WebApiTokenCheck tokenCheck;

	/** sessionにて同期を行うか否か */
	private boolean synchronizeOnSession;

	private String responseType;

	private String accessControlAllowOrigin;

	private boolean accessControlAllowCredentials;

	private boolean needTrustedAuthenticate;

	private StateType state = StateType.STATEFUL;

	private boolean supportBearerToken;
	
	private String[] oauthScopes;

	private String[] allowRequestContentTypes;

	private Long maxRequestBodySize;
	private Long maxFileSize;

	public Long getMaxFileSize() {
		return maxFileSize;
	}

	/**
	 * multipart/form-data時のアップロードファイルの最大サイズ。-1の場合は無制限。
	 * １つのファイルに対する最大サイズなので、複数のファイルの合計サイズを制限したい場合は、
	 * maxRequestBodySizeを設定します。
	 * 
	 * @param maxFileSize
	 */
	public void setMaxFileSize(Long maxFileSize) {
		this.maxFileSize = maxFileSize;
	}

	public Long getMaxRequestBodySize() {
		return maxRequestBodySize;
	}

	/**
	 * リクエストボディの最大サイズ。-1の場合は無制限。
	 * 
	 * @param maxRequestBodySize
	 */
	public void setMaxRequestBodySize(Long maxRequestBodySize) {
		this.maxRequestBodySize = maxRequestBodySize;
	}

	public String[] getAllowRequestContentTypes() {
		return allowRequestContentTypes;
	}

	/**
	 * 許可するリクエストボディのContentTypeを指定。未指定の場合はすべて許可。<br>
	 * accepts指定より、allowRequestContentTypesの指定による制限が優先されます。<br>
	 * 例えば、
	 * accepts指定によりJSON形式の処理が有効化されている場合において、
	 * allowRequestContentTypesに"application/json"が含まれない場合は、
	 * JSON形式によるリクエストは処理されません。
	 * 
	 * @param allowRequestContentTypes
	 */
	public void setAllowRequestContentTypes(String[] allowRequestContentTypes) {
		this.allowRequestContentTypes = allowRequestContentTypes;
	}
	
	public String[] getOauthScopes() {
		return oauthScopes;
	}

	public void setOauthScopes(String[] oauthScopes) {
		this.oauthScopes = oauthScopes;
	}
	
	public boolean isSupportBearerToken() {
		return supportBearerToken;
	}

	public void setSupportBearerToken(boolean supportBearerToken) {
		this.supportBearerToken = supportBearerToken;
	}

	public StateType getState() {
		return state;
	}

	/**
	 * WebApiをStatelessとして呼び出すか否かを設定します。
	 * デフォルトは、StateType.STATEFULです。
	 *
	 * @param state
	 */
	public void setState(StateType state) {
		this.state = state;
	}

	public boolean isNeedTrustedAuthenticate() {
		return needTrustedAuthenticate;
	}

	public void setNeedTrustedAuthenticate(boolean needTrustedAuthenticate) {
		this.needTrustedAuthenticate = needTrustedAuthenticate;
	}

	public String getAccessControlAllowOrigin() {
		return accessControlAllowOrigin;
	}

	public void setAccessControlAllowOrigin(String accessControlAllowOrigin) {
		this.accessControlAllowOrigin = accessControlAllowOrigin;
	}

	public boolean isAccessControlAllowCredentials() {
		return accessControlAllowCredentials;
	}

	public void setAccessControlAllowCredentials(boolean accessControlAllowCredentials) {
		this.accessControlAllowCredentials = accessControlAllowCredentials;
	}

	public String getResponseType() {
		return responseType;
	}

	public void setResponseType(String responseType) {
		this.responseType = responseType;
	}

	public boolean isSynchronizeOnSession() {
		return synchronizeOnSession;
	}

	public void setSynchronizeOnSession(boolean synchronizeOnSession) {
		this.synchronizeOnSession = synchronizeOnSession;
	}

	public WebApiTokenCheck getTokenCheck() {
		return tokenCheck;
	}

	public void setTokenCheck(WebApiTokenCheck tokenCheck) {
		this.tokenCheck = tokenCheck;
	}

	public String getRestJsonParameterType() {
		return restJsonParameterType;
	}

	public void setRestJsonParameterType(String restJsonParameterType) {
		this.restJsonParameterType = restJsonParameterType;
	}

	public String getRestJsonParameterName() {
		return restJsonParameterName;
	}

	public void setRestJsonParameterName(String restJsonParameterName) {
		this.restJsonParameterName = restJsonParameterName;
	}

	public String getRestXmlParameterName() {
		return restXmlParameterName;
	}

	public void setRestXmlParameterName(String restXmlParameterName) {
		this.restXmlParameterName = restXmlParameterName;
	}

	public RequestType[] getAccepts() {
		return accepts;
	}

	public void setAccepts(RequestType[] accepts) {
		this.accepts = accepts;
	}

	public MethodType[] getMethods() {
		return methods;
	}

	public void setMethods(MethodType[] methods) {
		this.methods = methods;
	}

	/**
	 * @return name
	 */
	public String getName() {
		return name;
	}

	/**
	 * @param name
	 *            セットする name
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * @return displayName
	 */
	public String getDisplayName() {
		return displayName;
	}

	/**
	 * @param displayName
	 *            セットする displayName
	 */
	public void setDisplayName(String displayName) {
		this.displayName = displayName;
	}

	/**
	 * @return description
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * @param description
	 *            セットする description
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * @see #setCacheControlType(cacheControlType)
	 * @return cacheControlType
	 */
	public CacheControlType getCacheControlType() {
		return cacheControlType;
	}

	/**
	 * WebAPIキャッシュ指定（Cache-Controlヘッダの制御）。
	 * 未指定の場合はキャッシュをしない。
	 * ブラウザ種別によらず、キャッシュを有効化するためには、合わせてCacheControlMaxAgeの設定も必要。
	 * 
	 * @see #setCacheControlMaxAge(long)
	 * @param cacheControlType
	 */
	public void setCacheControlType(CacheControlType cacheControlType) {
		this.cacheControlType = cacheControlType;
	}

	/**
	 * @see #setCacheControlMaxAge(long)
	 * @return
	 */
	public long getCacheControlMaxAge() {
		return cacheControlMaxAge;
	}

	/**
	 * cacheControlMaxAge=CacheControlType.CACHEを指定した場合の
	 * WebAPIキャッシュのmax-age（秒）を指定。
	 * デフォルト値は-1でこの場合はmax-ageは未指定となる。<br>
	 * <b>注意：max-age未指定の場合、FF、Chromeでは実際はキャッシュが利用されない</b>
	 * 
	 * @param cacheControlMaxAge
	 */
	public void setCacheControlMaxAge(long cacheControlMaxAge) {
		this.cacheControlMaxAge = cacheControlMaxAge;
	}

	/**
	 * @return commandConfig
	 */
	public CommandConfig getCommandConfig() {
		return commandConfig;
	}

	/**
	 * @param commandConfig
	 *            セットする commandConfig
	 */
	public void setCommandConfig(CommandConfig commandConfig) {
		this.commandConfig = commandConfig;
	}

	/**
	 * @return result
	 */
	public String[] getResults() {
		return results;
	}

	/**
	 * @param result
	 *            セットする result
	 */
	public void setResults(String[] results) {
		this.results = results;
	}

	public boolean isPublicWebApi() {
		return isPublicWebApi;
	}

	/**
	 * このWebApiの呼び出しをセキュリティ設定によらず呼び出し可能にする場合は、trueを設定します。
	 * isPrivilagedとの違いは、Entityの操作などにおいては、セキュリティ制約を受けます。
	 * デフォルトはfalseです。
	 *
	 * @param isPublicWebApi
	 */
	public void setPublicWebApi(boolean isPublicWebApi) {
		this.isPublicWebApi = isPublicWebApi;
	}

	/**
	 *
	 * @return
	 */
	public boolean isPrivilaged() {
		return isPrivilaged;
	}

	/**
	 *
	 * @param isPrivilaged
	 */
	public void setPrivilaged(boolean isPrivilaged) {
		this.isPrivilaged = isPrivilaged;
	}

	/**
	 * XMLHttpRequestがセットされていることを確認するかを取得します。
	 * @return XMLHttpRequestチェック可否
	 */
	public boolean isCheckXRequestedWithHeader() {
		return isCheckXRequestedWithHeader;
	}

	/**
	 * XMLHttpRequestチェック可否を設定します。
	 * @param isCheckXRequestedWithHeader
	 */
	public void setCheckXRequestedWithHeader(boolean isCheckXRequestedWithHeader) {
		this.isCheckXRequestedWithHeader = isCheckXRequestedWithHeader;
	}
}
