/**
 * 
 */
package org.isuper.common.utils;

import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;

import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;

import org.apache.commons.codec.binary.Base64;

/**
 * @author Super Wang
 *
 */
public final class Secure {
	
	private static final String CHARACTERS = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890";
	private static final int CHARACTERS_LENGTH = CHARACTERS.length();
	private static final int DEFAULT_PASSWORD_LENGTH = 16;
	
	public static String generate() {
		return generate(DEFAULT_PASSWORD_LENGTH);
	}
	
	public static String generate(final int length) {
		int len = Secure.DEFAULT_PASSWORD_LENGTH;
		if (len < 0) {
			len = DEFAULT_PASSWORD_LENGTH;
		}
		StringBuilder builder = new StringBuilder();
		for (int i = 0; i < len; i++) {
			double index = Math.random() * CHARACTERS_LENGTH;
			builder.append(CHARACTERS.charAt((int) index));
		}
		return builder.toString();
	}

	private static final int ITERATION_COUNT = 8192;
	private static final int KEY_SIZE = 160;

	public static byte[] hashPassword(final char[] password, final byte[] salt) throws GeneralSecurityException {
		return hashPassword(password, salt, ITERATION_COUNT, KEY_SIZE);
	}

	public static byte[] hashPassword(final char[] password, final byte[] salt, final int iterationCount, final int keySize) throws GeneralSecurityException {
		PBEKeySpec spec = new PBEKeySpec(password, salt, iterationCount, keySize);
		SecretKeyFactory factory = SecretKeyFactory.getInstance("PBKDF2WithHmacSHA1");
		return factory.generateSecret(spec).getEncoded();
	}

	public static boolean matches(final byte[] passwordHash, final char[] password, final byte[] salt) throws GeneralSecurityException {
		return matches(passwordHash, password, salt, ITERATION_COUNT, KEY_SIZE);
	}

	public static boolean matches(final byte[] passwordHash, final char[] password, final byte[] salt, final int iterationCount, final int keySize) throws GeneralSecurityException {
		return Arrays.equals(passwordHash, hashPassword(password, salt, iterationCount, keySize));
	}

	public static byte[] nextSalt() {
		byte[] salt = new byte[16];
		SecureRandom sr;
		try {
			sr = SecureRandom.getInstance("SHA1PRNG");
			sr.nextBytes(salt);
		} catch (NoSuchAlgorithmException e) {
			e.printStackTrace();
		}
		return salt;
	}
	
	public static String state() {
		return new BigInteger(130, new SecureRandom()).toString(32);
	}
	
	private static final byte[] EMPTY = new byte[0];
	
	public static byte[] empty() {
		return EMPTY;
	}
	
	private static final String UTF_8_ENCODING = "UTF-8";
	
	public static String base64Encode(String source) {
		if (Preconditions.isEmptyString(source)) {
			return null;
		}
		try {
			return Base64.encodeBase64URLSafeString(source.getBytes(UTF_8_ENCODING));
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
		return null;
	}

	public static String base64Decode(String source) {
		if (Preconditions.isEmptyString(source)) {
			return null;
		}
		try {
			return new String(Base64.decodeBase64(source), UTF_8_ENCODING);
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
		return null;
	}

	public static String md5(String source) {
		if (Preconditions.isEmptyString(source)) {
			return null;
		}
		try {
			MessageDigest md = MessageDigest.getInstance("MD5");
			StringBuilder result = new StringBuilder();
			try {
				for (byte b : md.digest(source.getBytes(UTF_8_ENCODING))) {
					result.append(Integer.toHexString((b & 0xf0) >>> 4));
					result.append(Integer.toHexString(b & 0x0f));
				}
			} catch (UnsupportedEncodingException e) {
				for (byte b : md.digest(source.getBytes())) {
					result.append(Integer.toHexString((b & 0xf0) >>> 4));
					result.append(Integer.toHexString(b & 0x0f));
				}
			}
			return result.toString();
		} catch (NoSuchAlgorithmException ex) {
			ex.printStackTrace();
		}
		return null;
	}

}
