/**
 * 
 */
package org.isuper.common.utils;

import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;

import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;

import org.apache.commons.codec.binary.Base64;

/**
 * @author Super Wang
 *
 */
public final class Secure {
	
	private static final String CHARACTERS = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890";
	private static final int CHARACTERS_LENGTH = CHARACTERS.length();
	private static final int DEFAULT_PASSWORD_LENGTH = 16;
	
	/**
	 * Generate a random string with default length 16.
	 * 
	 * @return
	 * 			The generated random string
	 */
	public static String generate() {
		return generate(DEFAULT_PASSWORD_LENGTH);
	}
	
	/**
	 * Generate a random string with specified length.
	 * 
	 * @param length
	 * 			The length of the random string
	 * @return
	 * 			The generated random string
	 */
	public static String generate(final int length) {
		int len = Secure.DEFAULT_PASSWORD_LENGTH;
		if (len < 0) {
			len = DEFAULT_PASSWORD_LENGTH;
		}
		StringBuilder builder = new StringBuilder();
		for (int i = 0; i < len; i++) {
			double index = Math.random() * CHARACTERS_LENGTH;
			builder.append(CHARACTERS.charAt((int) index));
		}
		return builder.toString();
	}

	private static final int ITERATION_COUNT = 8192;
	private static final int KEY_SIZE = 160;

	/**
	 * Generate a password hash with specified salt
	 * 
	 * @param password
	 * 			The password to hash
	 * @param salt
	 * 			The salt
	 * @return
	 * 			The password hash
	 * @throws GeneralSecurityException
	 * 			You might receive this exception because missing SecretKeyFactory in your system
	 */
	public static byte[] hashPassword(final char[] password, final byte[] salt) throws GeneralSecurityException {
		return hashPassword(password, salt, ITERATION_COUNT, KEY_SIZE);
	}

	/**
	 * Generate a password hash with specified salt, iteration count and key size
	 * 
	 * @param password
	 * 			The password to hash
	 * @param salt
	 * 			The salt
	 * @param iterationCount
	 * 			The iteration count
	 * @param keySize
	 * 			Size of the key
	 * @return
	 * 			The password hash
	 * @throws GeneralSecurityException
	 * 			You might receive this exception because missing SecretKeyFactory in your system or invalid iteration count and key size
	 */
	public static byte[] hashPassword(final char[] password, final byte[] salt, final int iterationCount, final int keySize) throws GeneralSecurityException {
		PBEKeySpec spec = new PBEKeySpec(password, salt, iterationCount, keySize);
		SecretKeyFactory factory = SecretKeyFactory.getInstance("PBKDF2WithHmacSHA1");
		return factory.generateSecret(spec).getEncoded();
	}

	/**
	 * Check if the password matches the hash with specified salt
	 * 
	 * @param passwordHash
	 * 			The expected password hash
	 * @param password
	 * 			The password to check
	 * @param salt
	 * 			The salt
	 * @return
	 * 			Matches or not
	 * @throws GeneralSecurityException
	 * 			You might receive this exception because missing SecretKeyFactory in your system
	 */
	public static boolean matches(final byte[] passwordHash, final char[] password, final byte[] salt) throws GeneralSecurityException {
		return matches(passwordHash, password, salt, ITERATION_COUNT, KEY_SIZE);
	}

	/**
	 * Check if the password matches the hash with specified salt, iteration count and key size
	 * 
	 * @param passwordHash
	 * 			The expected password hash
	 * @param password
	 * 			The password to check
	 * @param salt
	 * 			The salt
	 * @param iterationCount
	 * 			The iteration count
	 * @param keySize
	 * 			Size of the key
	 * @return
	 * 			Matches or not
	 * @throws GeneralSecurityException
	 * 			You might receive this exception because missing SecretKeyFactory in your system
	 */
	public static boolean matches(final byte[] passwordHash, final char[] password, final byte[] salt, final int iterationCount, final int keySize) throws GeneralSecurityException {
		return Arrays.equals(passwordHash, hashPassword(password, salt, iterationCount, keySize));
	}

	/**
	 * Generate salt
	 * 
	 * @return
	 * 			The generated salt
	 */
	public static byte[] nextSalt() {
		byte[] salt = new byte[16];
		SecureRandom sr;
		try {
			sr = SecureRandom.getInstance("SHA1PRNG");
			sr.nextBytes(salt);
		} catch (NoSuchAlgorithmException e) {
			e.printStackTrace();
		}
		return salt;
	}
	
	/**
	 * Generate a random state string
	 * 
	 * @return
	 * 		A random state string
	 */
	public static String state() {
		return new BigInteger(130, new SecureRandom()).toString(32);
	}
	
	private static final String UTF_8_ENCODING = "UTF-8";
	
	/**
	 * Encode a string with base64
	 * 
	 * @param source
	 * 			The source string to encode
	 * @return
	 * 			The encoded string
	 */
	public static String base64Encode(String source) {
		if (Preconditions.isEmptyString(source)) {
			return null;
		}
		try {
			return Base64.encodeBase64URLSafeString(source.getBytes(UTF_8_ENCODING));
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * Decode a base64 encrypted string
	 * 
	 * @param source
	 * 			The encrypted string
	 * @return
	 * 			The string after decode
	 */
	public static String base64Decode(String source) {
		if (Preconditions.isEmptyString(source)) {
			return null;
		}
		try {
			return new String(Base64.decodeBase64(source), UTF_8_ENCODING);
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * Generate MD5 hash of give string
	 * 
	 * @param source
	 * 			The source string
	 * @return
	 * 			MD5 hash
	 */
	public static String md5(String source) {
		if (Preconditions.isEmptyString(source)) {
			return null;
		}
		try {
			MessageDigest md = MessageDigest.getInstance("MD5");
			StringBuilder result = new StringBuilder();
			try {
				for (byte b : md.digest(source.getBytes(UTF_8_ENCODING))) {
					result.append(Integer.toHexString((b & 0xf0) >>> 4));
					result.append(Integer.toHexString(b & 0x0f));
				}
			} catch (UnsupportedEncodingException e) {
				for (byte b : md.digest(source.getBytes())) {
					result.append(Integer.toHexString((b & 0xf0) >>> 4));
					result.append(Integer.toHexString(b & 0x0f));
				}
			}
			return result.toString();
		} catch (NoSuchAlgorithmException ex) {
			ex.printStackTrace();
		}
		return null;
	}

}
