/**
 * 
 */
package org.isuper.geometry;

import java.io.IOException;
import java.io.Serializable;

import org.isuper.geometry.utils.GeometryUtils;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * @author Super Wayne
 *
 */
@JsonIgnoreProperties(ignoreUnknown=true)
public class Coordinate implements Serializable {

	/**
	 * 
	 */
	private static final long serialVersionUID = 447283454620410061L;
	
	private final double lat;
	private final double lng;
	
	@JsonCreator
	public Coordinate(@JsonProperty("lat") double lat, @JsonProperty("lng") double lng) {
		if (!GeometryUtils.isValidCoordinate(lat, lng)) {
			throw new IllegalArgumentException(String.format("Invalid coordinate: { lat: %f, lng: %f }", lat, lng));
		}
		this.lat = lat;
		this.lng = lng;
	}

	public static Coordinate parseFromCommaSeparatedString(String rawCoordinate) {
		if (rawCoordinate == null || rawCoordinate.trim().length() < 1) {
			throw new IllegalArgumentException(String.format("Invalid comma separated coordinate string: %s", rawCoordinate));
		}
		String[] array = rawCoordinate.split(",");
		if (array.length != 2) {
			throw new IllegalArgumentException(String.format("Invalid comma separated coordinate string: %s", rawCoordinate));
		}
		try {
			float lat = Float.parseFloat(array[0]);
			float lng = Float.parseFloat(array[1]);
			return new Coordinate(lat, lng);
		} catch (Exception e) {
			throw new IllegalArgumentException(String.format("Invalid comma separated coordinate string: %s", rawCoordinate));
		}
	}
	
	public String toJSON() {
		try {
			return GeometryUtils.getObjectMapper().writeValueAsString(this);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return "{}";
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return toJSON();
	}
	
	/**
	 * @return the lat
	 */
	public double getLat() {
		return this.lat;
	}

	/**
	 * @return the lng
	 */
	public double getLng() {
		return this.lng;
	}

	/**
	 * @return the latE6
	 */
	@JsonIgnore
	public int getLatE6() {
		return (int) Math.floor(this.lat * 1e6);
	}

	/**
	 * @return the lngE6
	 */
	@JsonIgnore
	public int getLngE6() {
		return (int) Math.floor(this.lng * 1e6);
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		long temp;
		temp = Double.doubleToLongBits(this.lat);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(this.lng);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		return result;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Coordinate other = (Coordinate) obj;
		if (Double.doubleToLongBits(this.lat) != Double.doubleToLongBits(other.lat))
			return false;
		if (Double.doubleToLongBits(this.lng) != Double.doubleToLongBits(other.lng))
			return false;
		return true;
	}

}
