/**
 * 
 */
package org.isuper.geometry;

import java.io.Serializable;
import java.util.LinkedList;
import java.util.List;
import java.util.StringTokenizer;

import org.isuper.geometry.utils.GeometryUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * @author Super Wang
 *
 */
@JsonIgnoreProperties(ignoreUnknown=true)
public class GeometryPolygon implements Serializable {
	
	/**
	 * 
	 */
	private static final long serialVersionUID = 1467646286428675843L;
	
	public final String type;
	public final List<List<double[]>> coordinates;
	
	public GeometryPolygon(@JsonProperty("type") String type, @JsonProperty("coordinates") List<List<double[]>> coordinates) {
		this.type = type;
		this.coordinates = coordinates;
	}
	
	@JsonIgnore
	public String toGeometryText() {
		StringBuilder builder = new StringBuilder();
		builder.append(this.type.toUpperCase());
		builder.append("(");
		for (List<double[]> subPolygon : this.coordinates) {
			if (subPolygon.isEmpty()) {
				continue;
			}
			builder.append("(");
			for (double[] point : subPolygon) {
				if (point == null || point.length != 2) {
					continue;
				}
				builder.append(String.format("%f %f", point[1], point[0]));
				builder.append(", ");
			}
			builder.delete(builder.length() - 2, builder.length());
			builder.append(")");
			builder.append(", ");
		}
		builder.delete(builder.length() - 2, builder.length());
		builder.append(")");
		return builder.toString();
	}

	public static GeometryPolygon fromGeometryText(String geometry) {
		GeometryPolygon polygon = null;
		if (geometry == null) {
			return polygon;
		}
		StringTokenizer tokenizer = new StringTokenizer(geometry, "(,)", true);
		while (tokenizer.hasMoreElements()) {
			String token = tokenizer.nextToken();
			if (!"POLYGON".equals(token.toUpperCase())) {
				return null;
			}
			token = tokenizer.nextToken();
			List<List<double[]>> coordinates = parsePolygon(tokenizer, token);
			if (coordinates != null) {
				polygon = new GeometryPolygon("POLYGON", coordinates);
				break;
			}
		}
		return polygon;
	}
	
	private static List<List<double[]>> parsePolygon(StringTokenizer tokenizer, String token) {
		if (!"(".equals(token)) {
			return null;
		}
		List<List<double[]>> items = new LinkedList<>();
		token = tokenizer.nextToken();
		while(!")".equals(token)) {
			if (",".equals(token) || " ".equals(token)) {
				token = tokenizer.nextToken();
				continue;
			}
			List<double[]> item = parsePolygonItems(tokenizer, token);
			if (item != null) {
				items.add(item);
			}
			token = tokenizer.nextToken();
		}
		return items;
	}
	
	private static List<double[]> parsePolygonItems(StringTokenizer tokenizer, String token) {
		if (!"(".equals(token)) {
			return null;
		}
		List<double[]> coordinates = new LinkedList<>();
		token = tokenizer.nextToken();
		while(!")".equals(token)) {
			if (token.startsWith(" ")) {
				token = token.substring(1);
			}
			String[] tmp = token.split(" ");
			if (tmp.length == 2) {
				double[] point = new double[] { Double.parseDouble(tmp[1]), Double.parseDouble(tmp[0])};
				if (!GeometryUtils.isValidCoordinate(point[1], point[0])) {
					throw new IllegalArgumentException(String.format("Invalid coordinates (lat:%f, lng:%f)", point[1], point[0]));
				}
				coordinates.add(point);
			}
			token = tokenizer.nextToken();
		}
		return coordinates;
	}
	
	public static void main(String... args) {
		GeometryPolygon polygon = GeometryPolygon.fromGeometryText("POLYGON((11.223344 22.334455, 33.445566 44.556677), (55.667788 66.778899, 77.889900 88.990011))");
		System.out.println(polygon.toGeometryText());
		polygon = GeometryPolygon.fromGeometryText("POLYGON((11.223344 22.334455,33.445566 44.556677),(55.667788 66.778899,77.889900 88.990011))");
		System.out.println(polygon.toGeometryText());
	}

}
