/**
 * 
 */
package org.isuper.geometry.utils;

import org.isuper.geometry.Coordinate;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * @author Super Wang
 *
 */
public class GeometryUtils {

	public static final int LNG_MAX = 179999999;
	public static final int LNG_MIN = -180000000;
	public static final int LAT_MAX = 89999999;
	public static final int LAT_MIN = -90000000;

	private static final JsonFactory JSON_FACTORY = new JsonFactory();
	private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper(JSON_FACTORY);
	static {
		//		OBJECT_MAPPER.setPropertyNamingStrategy(PropertyNamingStrategy.CAMEL_CASE_TO_LOWER_CASE_WITH_UNDERSCORES);
	}

	/**
	 * @return
	 */
	public static JsonFactory getJsonFactory() {
		return JSON_FACTORY;
	}

	/**
	 * @return
	 */
	public static ObjectMapper getObjectMapper() {
		return OBJECT_MAPPER;
	}

	public static final boolean latInRange(double lat) {
		return lat > (LAT_MIN / 1e6f) && lat < (LAT_MAX / 1e6f);
	}

	public static final boolean lngInRange(final double lng) {
		return lng > (LNG_MIN / 1e6f) && lng < (LNG_MAX / 1e6f);
	}

	public static final boolean isValidCoordinate(final double lat, final double lng) {
		return latInRange(lat) && lngInRange(lng);
	}

	/**
	 * Transfer from earth(WGS-84) to mars(GCJ-02).
	 * 
	 * @param wgs
	 * 			The WGS coordinate
	 * @return
	 * 			An {@link Coordinate} mars coordinate
	 */
	public static Coordinate wgs2gcj(Coordinate wgs) {
		double wgsLat = wgs.getLat();
		double wgsLng = wgs.getLng();
		if (isOutOfChina(wgsLat, wgsLng)) {
			return new Coordinate(wgsLat, wgsLng);
		}
		Coordinate d = delta(wgsLat, wgsLng);
		return new Coordinate(wgsLat + d.getLat(), wgsLng + d.getLng());
	}

	/**
	 * Transfer from mars(GCJ-02) to earth(WGS-84).
	 * 
	 * @param gcj
	 * 			The GCJ coordinate
	 * @return
	 * 			An {@link Coordinate} earth coordinate
	 */
	public static Coordinate gcj2wgs(final Coordinate gcj) {
		double gcjLat = gcj.getLat();
		double gcjLng = gcj.getLng();
		if (isOutOfChina(gcjLat, gcjLng)) {
			return new Coordinate(gcjLat, gcjLng);
		}
		Coordinate d = delta(gcjLat, gcjLng);
		return new Coordinate(gcjLat - d.getLat(), gcjLng - d.getLng());
	}

	private static boolean isOutOfChina(double lat, double lng) {
		return (lat < 25.401950 && lng < 125.502319 && lat > 21.675348 && lng > 119.827835) || (lng < 72.004) || (lng > 137.8347) || (lat < 0.8293) || (lat > 55.8271);
	}

	private static Coordinate delta(double lat, double lng) {
		double a = 6378245.0;
		double ee = 0.00669342162296594323;
		double dLat = transformLat(lng - 105.0, lat - 35.0);
		double dLng = transformLng(lng - 105.0, lat - 35.0);
		double radLat = lat / 180.0 * Math.PI;
		double magic = Math.sin(radLat);
		magic = 1 - ee * magic * magic;
		double sqrtMagic = Math.sqrt(magic);
		dLat = (dLat * 180.0) / ((a * (1 - ee)) / (magic * sqrtMagic) * Math.PI);
		dLng = (dLng * 180.0) / (a / sqrtMagic * Math.cos(radLat) * Math.PI);
		return new Coordinate(dLat, dLng);
	}

	private static double transformLat(double x, double y) {
		double ret = -100.0 + 2.0 * x + 3.0 * y + 0.2 * y * y + 0.1 * x * y + 0.2 * Math.sqrt(Math.abs(x));
		ret += (20.0 * Math.sin(6.0 * x * Math.PI) + 20.0 * Math.sin(2.0 * x * Math.PI)) * 2.0 / 3.0;
		ret += (20.0 * Math.sin(y * Math.PI) + 40.0 * Math.sin(y / 3.0 * Math.PI)) * 2.0 / 3.0;
		ret += (160.0 * Math.sin(y / 12.0 * Math.PI) + 320 * Math.sin(y * Math.PI / 30.0)) * 2.0 / 3.0;
		return ret;
	}

	private static double transformLng(double x, double y) {
		double ret = 300.0 + x + 2.0 * y + 0.1 * x * x + 0.1 * x * y + 0.1 * Math.sqrt(Math.abs(x));
		ret += (20.0 * Math.sin(6.0 * x * Math.PI) + 20.0 * Math.sin(2.0 * x * Math.PI)) * 2.0 / 3.0;
		ret += (20.0 * Math.sin(x * Math.PI) + 40.0 * Math.sin(x / 3.0 * Math.PI)) * 2.0 / 3.0;
		ret += (150.0 * Math.sin(x / 12.0 * Math.PI) + 300.0 * Math.sin(x / 30.0 * Math.PI)) * 2.0 / 3.0;
		return ret;
	}

	public static Coordinate gcj2wgsAccurate(final Coordinate gcj) {
		double gcjLat = gcj.getLat();
		double gcjLng = gcj.getLng();
		double initDelta = 0.01;
		double threshold = 0.000001;
		double dLat = initDelta, dLng = initDelta;
		double mLat = gcjLat - dLat, mLng = gcjLng - dLng;
		double pLat = gcjLat + dLat, pLng = gcjLng + dLng;
		Coordinate wgs = new Coordinate(mLat, mLng);
		Coordinate tmp;
		double wgsLat, wgsLng;
		for (int i = 0; i < 30; i++) {
			wgsLat = (mLat + pLat) / 2;
			wgsLng = (mLng + pLng) / 2;
			wgs = new Coordinate(wgsLat, wgsLng);
			tmp = wgs2gcj(wgs);
			dLat = tmp.getLat() - gcjLat;
			dLng = tmp.getLng() - gcjLng;
			if ((Math.abs(dLat) < threshold) && (Math.abs(dLng) < threshold)) {
				return wgs;
			}
			if (dLat > 0) {
				pLat = wgsLat;
			} else {
				mLat = wgsLat;
			}
			if (dLng > 0) {
				pLng = wgsLng;
			} else {
				mLng = wgsLng;
			}
		}
		return wgs;
	}

}
