/**
 * 
 */
package org.isuper.social.web.utils;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.isuper.common.utils.Preconditions;
import org.isuper.common.web.utils.CookieUtils;
import org.isuper.common.web.utils.ServletUtils;
import org.isuper.oauth.core.OAuthCredential;
import org.isuper.social.core.BaseSocialAccount;
import org.isuper.social.core.UserAccount;

/**
 * @author Super Wang
 *
 */
public final class SocialUtils {

	public static final String PATH_VERIFY = "/social/verify";
	public static final String PATH_ASSOCIATE = "/social/associate";
	
	private static final Logger DEFAULT_LOGGER = LogManager.getLogger(SocialUtils.class.getPackage().getName());
	
	private static int DEFAULT_COOKIE_EXPIRES_IN_SEC_ID_TOKEN = (int) TimeUnit.HOURS.toSeconds(1);
	private static int DEFAULT_COOKIE_EXPIRES_IN_SEC_SOCIAL_LAST_URL = (int) TimeUnit.DAYS.toSeconds(30);
	
	/**
	 * @param prefix
	 * 			The prefix of key in storage which used to load the id token
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 * @return
	 * 			The ID token
	 */
	public static final String loadIdToken(String prefix, HttpServletRequest req, HttpServletResponse resp) {
		HttpSession session = req.getSession();
		Object sessionIdToken = session.getAttribute(prefix + ".id_token");
		String idToken;
		if (sessionIdToken != null && !Preconditions.isEmptyString(sessionIdToken.toString())) {
			idToken = sessionIdToken.toString();
			DEFAULT_LOGGER.debug(String.format("Found ID token %s in session %s for user %s", idToken, session.getId(), req.getRemoteAddr()));
			return idToken;
		}
		idToken = req.getHeader(prefix + ".id_token");
		if (!Preconditions.isEmptyString(idToken)) {
			DEFAULT_LOGGER.debug(String.format("Found ID token %s in request header for user %s", idToken, req.getRemoteAddr()));
			return idToken;
		}
		idToken = CookieUtils.getCookieValue(prefix + ".id_token", req);
		if (!Preconditions.isEmptyString(idToken)) {
			DEFAULT_LOGGER.debug(String.format("Found ID token %s in cookies for user %s", idToken, req.getRemoteAddr()));
			return idToken;
		}
		return null;
	}

	/**
	 * @param prefix
	 * 			The prefix of key in storage which used to save the id token
	 * @param idToken
	 * 			The id token
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 */
	public static final void saveIdToken(String prefix, String idToken, HttpServletRequest req, HttpServletResponse resp) {
		HttpSession session = req.getSession();
		DEFAULT_LOGGER.debug(String.format("Saving ID token %s in session %s and cookies for user %s", idToken, session.getId(), req.getRemoteAddr()));
		if (Preconditions.isEmptyString(idToken)) {
			session.removeAttribute(prefix + ".id_token");
			CookieUtils.removeCookie(prefix + ".id_token", req, resp);
		}
		session.setAttribute(prefix + ".id_token", idToken);
		CookieUtils.putCookie(prefix + ".id_token", idToken, DEFAULT_COOKIE_EXPIRES_IN_SEC_ID_TOKEN, req, resp);
	}
	
	/**
	 * @param prefix
	 * 			The prefix of key in storage which used to remove the id token
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 */
	public static final void removeIdToken(String prefix, HttpServletRequest req, HttpServletResponse resp) {
		saveIdToken(prefix, null, req, resp);
	}
	
	/**
	 * @param prefix
	 * 			The prefix of key in storage which used to load the refresh token
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 * @return
	 * 			The refresh token
	 */
	public static final String loadRefreshToken(String prefix, HttpServletRequest req, HttpServletResponse resp) {
		HttpSession session = req.getSession();
		Object sessionRefreshToken = session.getAttribute(prefix + ".credential.refresh_token");
		String refreshToken;
		if (sessionRefreshToken != null && !Preconditions.isEmptyString(sessionRefreshToken.toString())) {
			refreshToken = sessionRefreshToken.toString();
			DEFAULT_LOGGER.debug(String.format("Found refresh token %s in session %s for user %s", refreshToken, session.getId(), req.getRemoteAddr()));
			return refreshToken;
		}
		refreshToken = req.getHeader(prefix + ".credential.refresh_token");
		if (!Preconditions.isEmptyString(refreshToken)) {
			DEFAULT_LOGGER.debug(String.format("Found refresh token %s in request header for user %s", refreshToken, req.getRemoteAddr()));
			return refreshToken;
		}
		return null;
	}

	/**
	 * @param prefix
	 * 			The prefix of key in storage which used to save the refresh token
	 * @param refreshToken
	 * 			The refresh token
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 */
	public static final void saveRefreshToken(String prefix, String refreshToken, HttpServletRequest req, HttpServletResponse resp) {
		HttpSession session = req.getSession();
		DEFAULT_LOGGER.debug(String.format("Saving refresh token %s in session %s for user %s", refreshToken, session.getId(), req.getRemoteAddr()));
		if (Preconditions.isEmptyString(refreshToken)) {
			session.removeAttribute(prefix + ".credential.refresh_token");
		}
		session.setAttribute(prefix + ".credential.refresh_token", refreshToken);
	}
	
	/**
	 * @param prefix
	 * 			The prefix of key in storage which used to remove the refresh token
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 */
	public static final void removeRefreshToken(String prefix, HttpServletRequest req, HttpServletResponse resp) {
		saveRefreshToken(prefix, null, req, resp);
	}
	
	/**
	 * @param prefix
	 * 			The prefix of identification to load credential
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 * @return
	 * 			The OAuth credential
	 */
	public static final OAuthCredential loadOAuthCredential(String prefix, HttpServletRequest req, HttpServletResponse resp) {
		String idToken = loadIdToken(prefix, req, resp);
		String refreshToken = loadRefreshToken(prefix, req, resp);
		
		OAuthCredential credential = new OAuthCredential(null, null);
		if (!Preconditions.isEmptyString(refreshToken)) {
			credential.setRefreshToken(refreshToken);
		}
		Map<String, String> extraParams = null;
		if (!Preconditions.isEmptyString(idToken)) {
			extraParams = new HashMap<>();
			extraParams.put("id_token", idToken);
			credential.setExtraParams(extraParams);
		}
		credential.setAccessTokenExpiration(System.currentTimeMillis() / 1000 + OAuthCredential.DEFAULT_ACCESS_TOKEN_EXPIRES_IN_SEC);
		DEFAULT_LOGGER.debug(String.format("Loaded OAuth credential %s for user %s", credential, req.getRemoteAddr()));
		return credential;
	}

	/**
	 * @param prefix
	 * 			The prefix of identification which used to save the credential
	 * @param credential
	 * 			The OAuth credential
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 */
	public static final void saveOAuthCredential(String prefix, OAuthCredential credential, HttpServletRequest req, HttpServletResponse resp) {
		if (credential == null) {
			removeIdToken(prefix, req, resp);
			removeRefreshToken(prefix, req, resp);
			return;
		}
		if (credential.getExtraParams() != null && !credential.getExtraParams().isEmpty()) {
			saveIdToken(prefix, credential.getExtraParams().get("id_token"), req, resp);
		}
		if (!Preconditions.isEmptyString(credential.getRefreshToken())) {
			saveRefreshToken(prefix, credential.getRefreshToken(), req, resp);
		}
	}

	/**
	 * @param prefix
	 * 			The prefix of identification which used to remove the credential
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 */
	public static final void removeOAuthCredential(String prefix, HttpServletRequest req, HttpServletResponse resp) {
		saveOAuthCredential(prefix, null, req, resp);
	}
	
	/**
	 * @param request
	 * 			HttpServletRequest
	 * @param response
	 * 			HttpServletResponse
	 * @return
	 * 			The user account
	 */
	public static final UserAccount<?> loadUserAccount(HttpServletRequest request, HttpServletResponse response) {
		HttpSession session = request.getSession();
		Object curUser = session.getAttribute("social.currentUser");
		UserAccount<?> userAccount;
		if (curUser instanceof UserAccount) {
			userAccount = (UserAccount<?>) curUser;
		} else {
			userAccount = null;
		}
		return userAccount;
	}

	/**
	 * @param userAccount
	 * 			The user account
	 * @param request
	 * 			HttpServletRequest
	 * @param response
	 * 			HttpServletResponse
	 */
	public static final void saveUserAccount(UserAccount<?> userAccount, HttpServletRequest request, HttpServletResponse response) {
		HttpSession session = request.getSession();
		DEFAULT_LOGGER.debug(String.format("Saving user account %s in session for user %s.", userAccount, request.getRemoteAddr()));
		if (userAccount == null || !BaseSocialAccount.isValid(userAccount.getSocialAccount())) {
			session.removeAttribute("social.currentUser");
			return;
		}
		session.setAttribute("social.currentUser", userAccount);
		saveOAuthCredential(userAccount.getSocialAccount().getProvider(), userAccount.getSocialAccount().getCredential(), request, response);
	}
	
	/**
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 * @return
	 * 			The last URL
	 */
	public static final String loadLastUrl(HttpServletRequest req, HttpServletResponse resp) {
		String url;
		HttpSession session = req.getSession();
		Object lastUrl = session.getAttribute("social.last_url");
		if (lastUrl != null && !Preconditions.isEmptyString(lastUrl.toString())) {
			url = lastUrl.toString();
			DEFAULT_LOGGER.debug(String.format("Found last url %s in session %s for user %s", url, session.getId(), req.getRemoteAddr()));
			return url;
		}
		url = CookieUtils.getCookieValue("social.last_url", req);
		if (url != null && !Preconditions.isEmptyString(url)) {
			DEFAULT_LOGGER.debug(String.format("Found last url %s in cookies for user %s", url, req.getRemoteAddr()));
			return url;
		}
		url = req.getContextPath();
		if (Preconditions.isEmptyString(url.toString())) {
			url = "/";
		}
		DEFAULT_LOGGER.debug(String.format("No valid last url in session %s for user %s, using calculated context path %s", session.getId(), req.getRemoteAddr(), url));
		return url;
	}
	
	/**
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 * @return
	 * 			The full URL of the request
	 */
	public static final String saveLastUrl(HttpServletRequest req, HttpServletResponse resp) {
		HttpSession session = req.getSession();
		String completeURL = ServletUtils.getRequestFullUrl(req);
		DEFAULT_LOGGER.debug(String.format("Saving last url %s in session %s for user %s.", completeURL, session.getId(), req.getRemoteAddr()));
		session.setAttribute("social.last_url", completeURL);
		DEFAULT_LOGGER.debug(String.format("Saving last url %s in cookies for user %s.", completeURL, req.getRemoteAddr()));
		CookieUtils.putCookie("social.last_url", completeURL, DEFAULT_COOKIE_EXPIRES_IN_SEC_SOCIAL_LAST_URL, req, resp);
		return completeURL;
	}

}
