/**
 * Copyright 2014-2016 Super Wayne
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.isuper.social.web.utils;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.isuper.common.utils.Preconditions;
import org.isuper.common.web.utils.CookieUtils;
import org.isuper.oauth.core.OAuthCredential;
import org.isuper.social.core.SocialAccount;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * @author Super Wayne
 *
 */
public final class SocialUtils {

	public static final String PATH_USER_VERIFY = "/user/verify";
	public static final String PATH_SIGNOUT = "/signout";
	
	private static final Logger DEFAULT_LOGGER = LogManager.getLogger(SocialUtils.class.getName());
	private static final JsonFactory JSON_FACTORY = new JsonFactory();
	private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper(JSON_FACTORY);
	
	private static int DEFAULT_COOKIE_EXPIRES_IN_SEC_ID_TOKEN = (int) TimeUnit.HOURS.toSeconds(1);
	
	/**
	 * @param socialAccount
	 * 			The social account
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 */
	public static final void saveSocialAccount(SocialAccount socialAccount, HttpServletRequest req, HttpServletResponse resp) {
		HttpSession session = req.getSession();
		if (socialAccount == null || !socialAccount.isValid()) {
			DEFAULT_LOGGER.trace(String.format("Invalid social account %s, skip saving in session %s for user %s.", socialAccount, session.getId(), req.getRemoteAddr()));
			return;
		}
		DEFAULT_LOGGER.trace(String.format("Saving social account %s in session %s for user %s.", socialAccount, session.getId(), req.getRemoteAddr()));
		session.setAttribute("social.account", socialAccount);
		saveOAuthCredential(socialAccount.getProvider(), socialAccount.getCredential(), req, resp);
	}

	/**
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 */
	public static final void removeSocialAccount(HttpServletRequest req, HttpServletResponse resp) {
		SocialAccount socialAcc = loadSocialAccount(req, resp);
		if (socialAcc != null && socialAcc.isValid()) {
			removeOAuthCredential(socialAcc.getProvider(), req, resp);
		}
		HttpSession session = req.getSession();
		DEFAULT_LOGGER.trace(String.format("Removing social account in session %s for user %s.", session.getId(), req.getRemoteAddr()));
		session.removeAttribute("social.account");
	}

	/**
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 * @return
	 * 			The user account
	 */
	public static final SocialAccount loadSocialAccount(HttpServletRequest req, HttpServletResponse resp) {
		HttpSession session = req.getSession();
		Object curUser = session.getAttribute("social.account");
		SocialAccount userAccount;
		if (curUser instanceof SocialAccount) {
			userAccount = (SocialAccount) curUser;
			DEFAULT_LOGGER.trace(String.format("Loaded social account %s in session  %s for user %s.", userAccount, session.getId(), req.getRemoteAddr()));
		} else {
			userAccount = null;
			DEFAULT_LOGGER.trace(String.format("No social account found in session %s for user %s.", session.getId(), req.getRemoteAddr()));
		}
		return userAccount;
	}

	public static final void saveOAuthCredential(String prefix, OAuthCredential credential, HttpServletRequest req, HttpServletResponse resp) {
		if (Preconditions.isEmptyString(prefix)) {
			return;
		}
		HttpSession session = req.getSession();
		if (credential == null) {
			DEFAULT_LOGGER.trace(String.format("Invalid OAuthCredential %s, skip saving in session %s for user %s.", credential, session.getId(), req.getRemoteAddr()));
			return;
		}
		DEFAULT_LOGGER.trace(String.format("Saving OAuthCredential %s in session %s for user %s.", credential, session.getId(), req.getRemoteAddr()));
		session.setAttribute(prefix + ".credential", credential);
		if (credential.getExtraParams() != null && !credential.getExtraParams().isEmpty()) {
			saveIdToken(prefix, credential.getExtraParams().get("id_token"), req, resp);
		}
	}

	public static final void removeOAuthCredential(String prefix, HttpServletRequest req, HttpServletResponse resp) {
		HttpSession session = req.getSession();
		DEFAULT_LOGGER.trace(String.format("Removing OAuthCredential in session %s for user %s.", session.getId(), req.getRemoteAddr()));
		session.removeAttribute(prefix + ".credential");
		DEFAULT_LOGGER.trace(String.format("Removing ID token in cookies for user %s.", req.getRemoteAddr()));
		CookieUtils.removeCookie(prefix + ".credential.id_token", req, resp);
	}

	/**
	 * @param prefix
	 * 			The prefix of identification to load credential
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 * @return
	 * 			The OAuth credential
	 */
	public static final OAuthCredential loadOAuthCredential(String prefix, HttpServletRequest req, HttpServletResponse resp) {
		HttpSession session = req.getSession();
		OAuthCredential credential = null;
		Object sessionCredential = session.getAttribute(prefix + ".credential");
		if (sessionCredential instanceof OAuthCredential) {
			credential = (OAuthCredential) sessionCredential;
			DEFAULT_LOGGER.trace(String.format("Loaded OAuth credential %s in session %s for user %s", credential, session.getId(), req.getRemoteAddr()));
			return credential;
		}
		
		DEFAULT_LOGGER.trace(String.format("No OAuth credential found in session %s for user %s", session.getId(), req.getRemoteAddr()));
		String refreshToken = loadRefreshToken(prefix, req, resp);
		String idToken = loadIdToken(prefix, req, resp);
		if (Preconditions.isEmptyString(refreshToken) && Preconditions.isEmptyString(idToken)) {
			DEFAULT_LOGGER.trace(String.format("No valid refresh token and ID token found for user %s", req.getRemoteAddr()));
			return credential;
		}
		
		credential = new OAuthCredential(null, null);
		credential.setAccessTokenExpiration(System.currentTimeMillis() / 1000 + OAuthCredential.DEFAULT_ACCESS_TOKEN_EXPIRES_IN_SEC);
		if (!Preconditions.isEmptyString(refreshToken)) {
			credential.setRefreshToken(refreshToken);
		}
		if (!Preconditions.isEmptyString(idToken)) {
			Map<String, String> extraParams = new HashMap<>();
			extraParams.put("id_token", idToken);
			credential.setExtraParams(extraParams);
		}
		DEFAULT_LOGGER.trace(String.format("Loaded OAuth credential %s for user %s", credential, req.getRemoteAddr()));
		return credential;
	}

	private static final void saveIdToken(String prefix, String idToken, HttpServletRequest req, HttpServletResponse resp) {
		if (Preconditions.isEmptyString(prefix) || Preconditions.isEmptyString(idToken)) {
			return;
		}
		DEFAULT_LOGGER.trace(String.format("Saving ID token %s in cookies for user %s.", idToken, req.getRemoteAddr()));
		CookieUtils.putCookie(prefix + ".credential.id_token", idToken, DEFAULT_COOKIE_EXPIRES_IN_SEC_ID_TOKEN, req, resp);
	}

	/**
	 * @param prefix
	 * 			The prefix of key in storage which used to load the id token
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 * @return
	 * 			The ID token
	 */
	private static final String loadIdToken(String prefix, HttpServletRequest req, HttpServletResponse resp) {
		String idToken = req.getHeader(prefix + ".credential.id_token");
		if (!Preconditions.isEmptyString(idToken)) {
			DEFAULT_LOGGER.trace(String.format("Found ID token %s in request header for user %s", idToken, req.getRemoteAddr()));
			return idToken;
		}
		idToken = req.getParameter(prefix + ".credential.id_token");
		if (!Preconditions.isEmptyString(idToken)) {
			DEFAULT_LOGGER.trace(String.format("Found ID token %s in request body for user %s", idToken, req.getRemoteAddr()));
			return idToken;
		}
		idToken = CookieUtils.getCookieValue(prefix + ".credential.id_token", req);
		if (!Preconditions.isEmptyString(idToken)) {
			DEFAULT_LOGGER.trace(String.format("Found ID token %s in cookies for user %s", idToken, req.getRemoteAddr()));
			return idToken;
		}
		DEFAULT_LOGGER.trace(String.format("No ID token found in both request header and body even cookies for user %s", req.getRemoteAddr()));
		return null;
	}

	/**
	 * @param prefix
	 * 			The prefix of key in storage which used to load the refresh token
	 * @param req
	 * 			HttpServletRequest
	 * @param resp
	 * 			HttpServletResponse
	 * @return
	 * 			The refresh token
	 */
	private static final String loadRefreshToken(String prefix, HttpServletRequest req, HttpServletResponse resp) {
		String refreshToken = req.getHeader(prefix + ".credential.refresh_token");
		if (!Preconditions.isEmptyString(refreshToken)) {
			DEFAULT_LOGGER.trace(String.format("Found refresh token %s in request header for user %s", refreshToken, req.getRemoteAddr()));
			return refreshToken;
		}
		refreshToken = req.getParameter(prefix + ".credential.refresh_token");
		if (!Preconditions.isEmptyString(refreshToken)) {
			DEFAULT_LOGGER.trace(String.format("Found refresh token %s in request body for user %s", refreshToken, req.getRemoteAddr()));
			return refreshToken;
		}
		DEFAULT_LOGGER.trace(String.format("No refresh token found in both request header and body for user %s", req.getRemoteAddr()));
		return null;
	}

	/**
	 * @return
	 * 			The JSON factory
	 */
	public static JsonFactory getJsonFactory() {
		return JSON_FACTORY;
	}
	
	/**
	 * @return
	 * 			The object mapper
	 */
	public static ObjectMapper getObjectMapper() {
		return OBJECT_MAPPER;
	}

	/**
	 * @return
	 *			The logger
	 */
	public static Logger getLogger() {
		return DEFAULT_LOGGER;
	}

	/**
	 * @param name
	 * 			The name of logger
	 * @return
	 * 			The logger with specified name
	 */
	public static Logger getLogger(String name) {
		return LogManager.getLogger(name);
	}
	
}
