/**
 * Copyright 2014-2017 Super Wayne
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.isuper.telegram.bot;

import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.isuper.telegram.api.models.Update;
import org.isuper.telegram.utils.TelegramUtils;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;

/**
 * @author Super Wayne
 *
 */
public class MemoryBasedBotSessionStore implements BotSessionStore {

	private static final Logger LOGGER = LogManager.getLogger(MemoryBasedBotSessionStore.class.getName());
	private static final ReentrantReadWriteLock LOCK = new ReentrantReadWriteLock();
	private final Lock rLock = LOCK.readLock();
	private final Lock wLock = LOCK.writeLock();

	private final BlockingQueue<Update> ongoingUpdates = new ArrayBlockingQueue<>(200);
	private final List<Update> cachedUpdates = new LinkedList<>();
	
	private long lastReceivedUpdateId = -1;
	private Update lastUnprocessedUpdate = null;
	
	@Override
	public void load() {
		
	}

	@Override
	public void save() {
		try {
			LOGGER.info(String.format("Data: %s", TelegramUtils.getObjectMapper().writeValueAsString(this)));
		} catch (JsonProcessingException e) {
			LOGGER.error(e.getMessage(), e);
		}
	}

	@Override
	public boolean hasUpdates() {
		this.rLock.lock();
		try {
			return !this.ongoingUpdates.isEmpty();
		} finally {
			this.rLock.unlock();
		}
	}

	@Override
	public void addUpdates(List<Update> received) {
		this.flushCachedUpdates();
		this.wLock.lock();
		try {
			int breakPoint = -1;
			for (int i = 0; i < received.size(); i++) {
				boolean accepted = this.ongoingUpdates.offer(received.get(i));
				if (accepted) {
					continue;
				}
				breakPoint = i;
				break;
			}
			if (breakPoint >= 1) {
				received.subList(0, breakPoint - 1).clear();
				this.cachedUpdates.addAll(received);
			}
		} finally {
			this.wLock.unlock();
		}
	}

	@Override
	public Update retrieveUpdate(long timeout, TimeUnit unit) throws IOException {
		this.wLock.lock();
		try {
			Update update;
			try {
				update = this.ongoingUpdates.poll(timeout, unit);
			} catch (InterruptedException e) {
				throw new IOException(e);
			}
			return update;
		} finally {
			this.wLock.unlock();
		}
	}
	
	private void flushCachedUpdates() {
		this.wLock.lock();
		try {
			if (!this.cachedUpdates.isEmpty()) {
				int breakPoint = -1;
				for (int i = 0; i < this.cachedUpdates.size(); i++) {
					boolean accepted = this.ongoingUpdates.offer(this.cachedUpdates.get(i));
					if (accepted) {
						continue;
					}
					breakPoint = i;
					break;
				}
				if (breakPoint >= 1) {
					this.cachedUpdates.subList(0, breakPoint - 1).clear();
				}
			}
		} finally {
			this.wLock.unlock();
		}
	}
	
	/**
	 * @return the ongoingUpdates
	 */
	@JsonProperty("ongoingUpdates")
	public BlockingQueue<Update> getOngoingUpdates() {
		this.rLock.lock();
		try {
			return this.ongoingUpdates;
		} finally {
			this.rLock.unlock();
		}
	}

	/**
	 * @return the cachedUpdates
	 */
	@JsonProperty("cachedUpdates")
	public List<Update> getCachedUpdates() {
		this.rLock.lock();
		try {
			return this.cachedUpdates;
		} finally {
			this.rLock.unlock();
		}
	}

	@Override
	@JsonProperty("lastReceivedUpdateId")
	public long getLastReceivedUpdateId() {
		this.rLock.lock();
		try {
			return this.lastReceivedUpdateId;
		} finally {
			this.rLock.unlock();
		}
	}
	
	@Override
	public void setLastReceivedUpdateId(long lastReceivedUpdateId) {
		this.wLock.lock();
		try {
			this.lastReceivedUpdateId = lastReceivedUpdateId;
		} finally {
			this.wLock.unlock();
		}
	}

	@Override
	@JsonProperty("lastUnprocessedUpdate")
	public Update getLastUnprocessedUpdate() {
		this.rLock.lock();
		try {
			return this.lastUnprocessedUpdate;
		} finally {
			this.rLock.unlock();
		}
	}

	@Override
	public void setLastUnprocessedUpdate(Update lastUnprocessedUpdate) {
		this.wLock.lock();
		try {
			this.lastUnprocessedUpdate = lastUnprocessedUpdate;
		} finally {
			this.wLock.unlock();
		}
	}

}
