/**
 * Copyright 2014-2017 Super Wayne
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.isuper.telegram.api.models;

import java.util.List;

import org.isuper.common.utils.Preconditions;
import org.isuper.telegram.utils.TelegramUtils;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;

/**
 * This object represents a message.
 * 
 * @author Super Wayne
 *
 */
@JsonIgnoreProperties(ignoreUnknown=true)
@JsonInclude(JsonInclude.Include.NON_NULL)
public class Message {

	@JsonProperty("message_id")
	public final long id;
	@JsonProperty("from")
	public final User from;
	@JsonProperty("date")
	public final long date;
	@JsonProperty("chat")
	public final Chat chat;
	@JsonProperty("forward_from")
	public final User forwardFrom;
	@JsonProperty("forward_from_chat")
	public final Chat forwardFromChat;
	@JsonProperty("forward_from_message_id")
	public final Long forwardFromMessageId;
	@JsonProperty("forward_date")
	public final Long forwardDate;
	@JsonProperty("reply_to_message")
	public final Message replyTo;
	@JsonProperty("edit_date")
	public final Long editDate;
	@JsonProperty("text")
	public final String text;
	@JsonProperty("entities")
	public final List<MessageEntity> entities;
	@JsonProperty("contact")
	public final Contact contact;
	@JsonProperty("location")
	public final Location location;
	@JsonProperty("new_chat_member")
	public final User newChatMember;
	@JsonProperty("left_chat_member")
	public final User leftChatMember;
	@JsonProperty("new_chat_title")
	public final String newChatTitle;
	@JsonProperty("new_chat_photo")
	public final PhotoSize[] newChatPhoto;
	@JsonProperty("delete_chat_photo")
	public final Boolean deleteChatPhoto;
	@JsonProperty("group_chat_created")
	public final Boolean groupChatCreated;
	@JsonProperty("supergroup_chat_created")
	public final Boolean supergroupChatCreated;
	@JsonProperty("channel_chat_created")
	public final Boolean channelChatCreated;
	@JsonProperty("migrate_to_chat_id")
	public final Long migrateToChatID;
	@JsonProperty("migrate_from_chat_id")
	public final Long migrateFromChatID;
	@JsonProperty("pinned_message")
	public final Message pinned;
	
	/**
	 * @param id
	 * 					Unique message identifier
	 * @param from
	 * 					Optional. Sender, can be empty for messages sent to channels
	 * @param date
	 * 					Date the message was sent in Unix time
	 * @param chat
	 * 					Conversation the message belongs to
	 * @param forwardFrom
	 * 					Optional. For forwarded messages, sender of the original message
	 * @param forwardFromChat
	 * 					Optional. For messages forwarded from a channel, information about the original channel
	 * @param forwardFromMessageId
	 * 					Optional. For forwarded channel posts, identifier of the original message in the channel
	 * @param forwardDate
	 * 					Optional. For forwarded messages, date the original message was sent in Unix time
	 * @param replyTo
	 * 					Optional. For replies, the original message. Note that the Message object in this field will not contain further reply_to_message fields even if it itself is a reply.
	 * @param editDate
	 * 					Optional. Date the message was last edited in Unix time
	 * @param text
	 * 					Optional. For text messages, the actual UTF-8 text of the message
	 * @param entities
	 * 					Optional. Optional. For text messages, special entities like usernames, URLs, bot commands, etc. that appear in the text
	 * @param contact
	 * 					Optional. Message is a shared contact, information about the contact
	 * @param location
	 * 					Optional. Message is a shared location, information about the location
	 * @param newChatMember
	 * 					Optional. A new member was added to the group, information about them (this member may be the bot itself)
	 * @param leftChatMember
	 * 					Optional. A member was removed from the group, information about them (this member may be the bot itself)
	 * @param newChatTitle
	 * 					Optional. A chat title was changed to this value
	 * @param newChatPhoto
	 * 					Optional. A chat photo was change to this value
	 * @param deleteChatPhoto
	 * 					Optional. Optional. Service message: the chat photo was deleted
	 * @param groupChatCreated
	 * 					Optional. Optional. Service message: the group has been created
	 * @param supergroupChatCreated
	 * 					Optional. Optional. Service message: the supergroup has been created. This field can‘t be received in a message coming through updates, because bot can’t be a member of a supergroup when it is created. It can only be found in reply_to_message if someone replies to a very first message in a directly created supergroup.
	 * @param channelChatCreated
	 * 					Optional. Optional. Service message: the channel has been created. This field can‘t be received in a message coming through updates, because bot can’t be a member of a channel when it is created. It can only be found in reply_to_message if someone replies to a very first message in a channel.
	 * @param migrateToChatID
	 * 					Optional. The group has been migrated to a supergroup with the specified identifier. This number may be greater than 32 bits and some programming languages may have difficulty/silent defects in interpreting it. But it smaller than 52 bits, so a signed 64 bit integer or double-precision float type are safe for storing this identifier.
	 * @param migrateFromChatID
	 * 					Optional. The supergroup has been migrated from a group with the specified identifier. This number may be greater than 32 bits and some programming languages may have difficulty/silent defects in interpreting it. But it smaller than 52 bits, so a signed 64 bit integer or double-precision float type are safe for storing this identifier.
	 * @param pinned
	 * 					Optional. Specified message was pinned. Note that the Message object in this field will not contain further reply_to_message fields even if it is itself a reply.
	 */
	public Message(
			@JsonProperty("message_id") Long id,
			@JsonProperty("from") User from,
			@JsonProperty("date") Long date,
			@JsonProperty("chat") Chat chat,
			@JsonProperty("forward_from") User forwardFrom,
			@JsonProperty("forward_from_chat") Chat forwardFromChat,
			@JsonProperty("forward_from_message_id") Long forwardFromMessageId,
			@JsonProperty("forward_date") Long forwardDate,
			@JsonProperty("reply_to_message") Message replyTo,
			@JsonProperty("edit_date") Long editDate,
			@JsonProperty("text") String text,
			@JsonProperty("entities") List<MessageEntity> entities,
			@JsonProperty("contact") Contact contact,
			@JsonProperty("location") Location location,
			@JsonProperty("new_chat_member") User newChatMember,
			@JsonProperty("left_chat_member") User leftChatMember,
			@JsonProperty("new_chat_title") String newChatTitle,
			@JsonProperty("new_chat_photo") PhotoSize[] newChatPhoto,
			@JsonProperty("delete_chat_photo") Boolean deleteChatPhoto,
			@JsonProperty("group_chat_created") Boolean groupChatCreated,
			@JsonProperty("supergroup_chat_created") Boolean supergroupChatCreated,
			@JsonProperty("channel_chat_created") Boolean channelChatCreated,
			@JsonProperty("migrate_to_chat_id") Long migrateToChatID,
			@JsonProperty("migrate_from_chat_id") Long migrateFromChatID,
			@JsonProperty("pinned_message") Message pinned
			) {
		Preconditions.notNull(id, "Message ID should be provided.");
		this.id = id;
		this.from = from;
		Preconditions.notNull(date, "Message sent date should be provided.");
		this.date = date;
		this.chat = chat;
		this.forwardFrom = forwardFrom;
		this.forwardFromChat = forwardFromChat;
		this.forwardFromMessageId = forwardFromMessageId;
		this.forwardDate = forwardDate;
		this.replyTo = replyTo;
		this.editDate = editDate;
		this.text = text;
		this.entities = entities;
		this.contact = contact;
		this.location = location;
		this.newChatMember = newChatMember;
		this.leftChatMember = leftChatMember;
		this.newChatTitle = newChatTitle;
		this.newChatPhoto = newChatPhoto;
		this.deleteChatPhoto = deleteChatPhoto;
		this.groupChatCreated = groupChatCreated;
		this.supergroupChatCreated = supergroupChatCreated;
		this.channelChatCreated = channelChatCreated;
		this.migrateToChatID = migrateToChatID;
		this.migrateFromChatID = migrateFromChatID;
		this.pinned = pinned;
	}

	public String toJSON() {
		try {
			return TelegramUtils.getObjectMapper().writeValueAsString(this);
		} catch (JsonProcessingException e) {
			e.printStackTrace();
		}
		return "{}";
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return this.toJSON();
	}
	
}
