/**
 * Copyright 2014-2017 Super Wayne
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.isuper.telegram.api.models.inline;

import org.isuper.common.utils.Preconditions;
import org.isuper.telegram.utils.TelegramUtils;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;

/**
 * Represents the content of a contact message to be sent as the result of an inline query.
 * 
 * @author Super Wayne
 *
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
public class InputContactMessageContent implements InputMessageContent {
	
	@JsonProperty("phone_number")
	public final String phoneNumber;
	@JsonProperty("first_name")
	public final String firstName;
	@JsonProperty("last_name")
	public final String lastName;
	
	/**
	 * @param phoneNumber
	 * 					Contact's phone number
	 * @param firstName
	 * 					Contact's first name
	 * @param lastName
	 * 					Optional. Contact's last name
	 */
	public InputContactMessageContent(String phoneNumber, String firstName, String lastName) {
		Preconditions.notEmptyString(phoneNumber, "Contact's phone number should be provided.");
		this.phoneNumber = phoneNumber;
		Preconditions.notEmptyString(firstName, "Contact's first name should be provided.");
		this.firstName = firstName;
		this.lastName = lastName;
	}

	public String toJSON() {
		try {
			return TelegramUtils.getObjectMapper().writeValueAsString(this);
		} catch (JsonProcessingException e) {
			e.printStackTrace();
		}
		return "{}";
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return this.toJSON();
	}
	
}
