/**
 * Copyright 2014-2017 Super Wayne
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.isuper.telegram.bot;

import java.util.ResourceBundle;

import org.apache.logging.log4j.Logger;
import org.isuper.common.utils.Preconditions;
import org.isuper.telegram.api.models.Command;
import org.isuper.telegram.api.models.Message;
import org.isuper.telegram.api.models.Update;
import org.isuper.telegram.api.models.inline.ChosenInlineResult;
import org.isuper.telegram.api.models.inline.InlineQuery;
import org.isuper.telegram.utils.TelegramUtils;

import com.fasterxml.jackson.core.JsonProcessingException;

/**
 * @author Super Wayne
 *
 */
public abstract class DefaultUpdateHandler implements UpdateHandler {

	@Override
	public boolean handleUpdate(Update update) {
		if (update == null) {
			return false;
		}
		ResourceBundle bundle = this.getResourceBundle(update);
		if (update.inlineQuery != null) {
			this.handleInlineQuery(bundle, update.inlineQuery);
		} else if (update.chosenInlineResult != null) {
			this.handleChosenInlineResult(bundle, update.chosenInlineResult);
		} else if (update.message != null) {
			if (update.message.newChatMember != null) {
				this.handleNewChatMember(bundle, update.message);
			} else if (update.message.leftChatMember != null) {
				this.handleLeftChatMember(bundle, update.message);
			} else if (update.message.location != null) {
				this.handleLocationMessage(bundle, update.message);
			} else if (!Preconditions.isEmptyString(update.message.text)) {
				Command cmd = TelegramUtils.parseCommandFromText(update.message.text);
				if (cmd == null) {
					this.handleOrdinaryTextMessage(bundle, update.message);
				} else if (Preconditions.isEmptyString(cmd.toBot) || getRobotUsername().equalsIgnoreCase(cmd.toBot)) {
					this.handleCommands(bundle, cmd, update.message);
				}
			}
		} else if (update.editedMessage != null) {
			this.handleEditedTextMessage(bundle, update.editedMessage);
		} else {
			try {
				getLogger().warn(TelegramUtils.getObjectMapper().writeValueAsString(update));
			} catch (JsonProcessingException e) {
				getLogger().error(e.getMessage(), e);
			}
		}
		return true;
	}

	protected abstract Logger getLogger();
	protected abstract ResourceBundle getResourceBundle(Update update);
	protected abstract String getRobotUsername();

	protected abstract void handleNewChatMember(ResourceBundle bundle, Message message);
	protected abstract void handleLeftChatMember(ResourceBundle bundle, Message message);

	protected abstract void handleLocationMessage(ResourceBundle bundle, Message message);
	protected abstract void handleOrdinaryTextMessage(ResourceBundle bundle, Message message);
	protected abstract void handleEditedTextMessage(ResourceBundle bundle, Message editedMessage);

	protected abstract void handleCommands(ResourceBundle bundle, Command cmd, Message message);
	
	protected abstract void handleInlineQuery(ResourceBundle bundle, InlineQuery inlineQuery);
	protected abstract void handleChosenInlineResult(ResourceBundle bundle, ChosenInlineResult chosenInlineResult);
	
}
